﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#ifdef NW_SND_SPY_ENABLE

#include <nw/ut/ut_String.h>
#include <nw/snd/spy/fnd/basis/sndspyfnd_Memory.h>
#include <nw/snd/spy/fnd/string/sndspyfnd_String.h>
#include <nw/snd/spy/protocol/sndspy_Packet.h>

#include <nw/snd/spy/sndspy_SpyDataFileChannel.h>

#if defined(NW_DEBUG)
#define STATE_DEBUG_ENABLED
#define BUFFER_DEBUG_ENABLED
#endif

#if defined(NW_PLATFORM_CAFE)
#include <nn/config.h>
#endif

namespace {

#if defined(NW_PLATFORM_CAFE)
static u32 DEFAULT_DATA_FILE_SIZE_LIMIT = 32 * 1024 * 1024;
#endif

}

namespace nw {
namespace snd {
namespace spy {

//----------------------------------------------------------
SpyDataFileChannelParam::SpyDataFileChannelParam()
{
#if defined(NW_PLATFORM_CAFE)
    fsClient = NULL;
    dataFileSizeLimit = DEFAULT_DATA_FILE_SIZE_LIMIT;
    hfioMountPath = NULL;
#endif
}

namespace internal {

//----------------------------------------------------------
SpyDataFileChannel::SpyDataFileChannel()
{
#if defined(NW_PLATFORM_CAFE)
    m_FsClient = NULL;
    m_FsCmdBlock = NULL;
    m_DataOutputFileOpen = false;
#endif
}

//----------------------------------------------------------
size_t
SpyDataFileChannel::GetRequiredMemorySize()
{
#if !defined(NW_PLATFORM_CAFE)

    return 0;

#else

    return sizeof(FSCmdBlock);

#endif
}

//----------------------------------------------------------
void
SpyDataFileChannel::Initialize(void* buffer)
{
#if !defined(NW_PLATFORM_CAFE)

    NW_UNUSED_VARIABLE(buffer);

#else

    NW_ASSERT_NOT_NULL(buffer);

    m_FsClient = NULL;
    m_HfioMountPath[0] = '\0';
    m_OutputDirMountPath[0] = '\0';
    m_DataOutputFileOpen = false;

    m_FsCmdBlock = reinterpret_cast<FSCmdBlock*>(buffer);
    FSInitCmdBlock(m_FsCmdBlock);

#endif
}

//----------------------------------------------------------
void
SpyDataFileChannel::Finalize()
{
#if defined(NW_PLATFORM_CAFE)

    if (this->IsOpen())
    {
        this->Close();
    }

    m_FsCmdBlock = NULL;

#endif
}

//----------------------------------------------------------
bool
SpyDataFileChannel::Open(const SpyDataFileChannelParam& param)
{
#if !defined(NW_PLATFORM_CAFE)

    NW_UNUSED_VARIABLE(param);
    return false;

#else

    NW_ASSERT(!this->IsOpen());

    m_FsClient = NULL;

    if (param.fsClient != NULL)
    {
        if (param.hfioMountPath == NULL)
        {
            NW_ASSERT_NOT_NULL(param.hfioMountPath);
            return false;
        }

        if (!m_FsCmdBlock)
        {
            NW_ASSERT_NOT_NULL(m_FsCmdBlock);
            return false;
        }

        m_FsClient = param.fsClient;
        m_MaxWriteLength = param.dataFileSizeLimit;
        nw::snd::spy::internal::fnd::String::Copy(m_HfioMountPath, sizeof(m_HfioMountPath), param.hfioMountPath);
    }

    return true;
#endif
}

//----------------------------------------------------------
bool
SpyDataFileChannel::IsOpen() const
{
#if !defined(NW_PLATFORM_CAFE)

    return false;

#else

    return m_FsClient != NULL;

#endif
}

//----------------------------------------------------------
void
SpyDataFileChannel::Close()
{
#if defined(NW_PLATFORM_CAFE)
    if (this->IsOpen())
    {
        if (this->IsActive())
        {
            this->EndSession();
        }

        m_FsClient = NULL;
    }
#endif
}

//----------------------------------------------------------
bool
SpyDataFileChannel::BeginSession(const char* outputDir)
{
#if !defined(NW_PLATFORM_CAFE)

    NW_UNUSED_VARIABLE(outputDir);
    return false;

#else

    if (this->IsOpen())
    {
        return this->SetOutputDirPath(outputDir);
    }

    return false;

#endif
}

//----------------------------------------------------------
void
SpyDataFileChannel::EndSession()
{
#if defined(NW_PLATFORM_CAFE)
    if (this->IsOpen())
    {
        if (this->IsActive())
        {
            this->CloseFiles();
        }
    }
#endif
}

//----------------------------------------------------------
bool
SpyDataFileChannel::IsActive() const
{
#if !defined(NW_PLATFORM_CAFE)

    return false;

#else

    return m_DataOutputFileOpen;

#endif
}

//----------------------------------------------------------
bool
SpyDataFileChannel::WriteData(void* buffer, u32 length)
{
#if !defined(NW_PLATFORM_CAFE)

    NW_UNUSED_VARIABLE(buffer);
    NW_UNUSED_VARIABLE(length);
    return false;

#else

    nw::snd::spy::internal::fnd::ScopedCriticalSection lock(m_DataFileLock);

    if (!m_DataOutputFileOpen)
    {
        return false;
    }

    FSFileHandle fileHandle = m_DataOutputFile[m_CurrentWriteFile];
    FSStatus result;

    result = FSWriteFile(
        m_FsClient,
        m_FsCmdBlock,
        buffer,
        1, // size
        length,
        fileHandle,
        0, // flag
        FS_RET_ALL_ERROR);

    if (result != length)
    {
        if (result >= 0)
        {
            NW_FATAL_ERROR("FSWriteFile partial (%d)\n", result);
        }
        else
        {
            NW_FATAL_ERROR("FSWriteFile failed (FS_STATUS_ERROR_BASE %d)\n", result - FS_STATUS_ERROR_BASE);
        }
    }

    m_WriteLength += result;

    // ファイルサイズが上限に達したら、出力ファイルを切り替える。
    if (m_WriteLength >= m_MaxWriteLength)
    {
        // SoundSpyのデータ読み込みを追い越さないようにするため、
        // SoundSpyが現在のファイルを読み込み始めてから出力ファイルを切り替える。
        // それまでは、上限をオーバーしても現在のファイルに追記を続ける。
        if (m_CurrentWriteFile == m_CurrentReadFile)
        {
            m_CurrentWriteFile = (m_CurrentWriteFile + 1) & 1;
            m_WriteLength = 0;

            // 次の出力ファイルを空にする。
            FSFileHandle newFileHandle = m_DataOutputFile[m_CurrentWriteFile];
            result = FSSetPosFile(m_FsClient, m_FsCmdBlock, newFileHandle, 0, FS_RET_ALL_ERROR);
            NW_ASSERTMSG(result == FS_STATUS_OK, "FSStatus FS_STATUS_ERROR_BASE %d\n", result - FS_STATUS_ERROR_BASE);
            result = FSTruncateFile(m_FsClient, m_FsCmdBlock, newFileHandle, FS_RET_ALL_ERROR);
            NW_ASSERTMSG(result == FS_STATUS_OK, "FSStatus FS_STATUS_ERROR_BASE %d\n", result - FS_STATUS_ERROR_BASE);

            // 出力ファイル切り替えパケットを送信。
#ifdef STATE_DEBUG_ENABLED
            NW_LOG("[nw::snd::spy::SpyController] WriteData: add DataEndPacket\n");
#endif
            internal::DataEndPacket* packet = new(buffer) internal::DataEndPacket();
            result = FSWriteFile(
                m_FsClient,
                m_FsCmdBlock,
                buffer,
                1, // size
                sizeof(internal::DataEndPacket),
                fileHandle,
                0, // flag
                FS_RET_ALL_ERROR);

            if (result != sizeof(internal::DataEndPacket))
            {
                if (result >= 0)
                {
                    NW_FATAL_ERROR("FSWriteFile partial (%d)\n", result);
                }
                else
                {
                    NW_FATAL_ERROR("FSWriteFile failed (FS_STATUS_ERROR_BASE %d)\n", result - FS_STATUS_ERROR_BASE);
                }
            }
        }
    }

    return true;

#endif
}

//----------------------------------------------------------
bool
SpyDataFileChannel::SetOutputDirPath(const char* path)
{
#if !defined(NW_PLATFORM_CAFE)

    NW_UNUSED_VARIABLE(path);
    return false;

#else

    NW_LOG("[nw::snd::spy::internal::SpyDataFileChannel] SetOutputDirPath(path = %s)\n", path? path : "<null>");

    FSStatus result;

    // 未初期化状態に。
    m_OutputDirMountPath[0] = '\0';
    this->CloseFiles();

    if (!m_FsClient)
    {
        return false;
    }

    if (path != NULL)
    {
        // FSにおける出力ディレクトリのパスを生成する。
        m_OutputDirMountPath[sizeof(m_OutputDirMountPath) - 1] = '\0';
        nw::ut::snprintf(m_OutputDirMountPath, sizeof(m_OutputDirMountPath), "%s/%s", m_HfioMountPath, path);
        int i = 0;
        for (; i < sizeof(m_OutputDirMountPath); ++i)
        {
            char c = m_OutputDirMountPath[i];
            if (c == '\0')
            {
                break;
            }
            else if (c == '\\' || c == ':')
            {
                m_OutputDirMountPath[i] = '/';
            }
        }

        // ヌル終端されていない。
        if (i == sizeof(m_OutputDirMountPath))
        {
            NW_LOG("[nw::snd::spy::internal::SpyDataFileChannel] SetOutputDirPath: path too long.\n");
            m_OutputDirMountPath[0] = '\0';
            return false;
        }

#ifdef STATE_DEBUG_ENABLED
        NW_LOG("[nw::snd::spy::internal::SpyDataFileChannel] SetOutputDirPath: m_OutputDirMountPath %s\n", m_OutputDirMountPath);
#endif

        // 出力ディレクトリが存在するかチェックする。
        FSDirHandle handle;
        result = FSOpenDir(m_FsClient, m_FsCmdBlock, m_OutputDirMountPath, &handle, FS_RET_NOT_DIR | FS_RET_NOT_FOUND);
        if (result != FS_STATUS_OK)
        {
            NW_LOG("[nw::snd::spy::internal::SpyDataFileChannel] SetOutputDirPath: dir not found (FS_STATUS_ERROR_BASE %d)\n", result - FS_STATUS_ERROR_BASE);
            m_OutputDirMountPath[0] = '\0';
            return false;
        }

        result = FSCloseDir(m_FsClient, m_FsCmdBlock, handle, FS_RET_ALL_ERROR);
        NW_ASSERTMSG(result == FS_STATUS_OK, "FSStatus FS_STATUS_ERROR_BASE %d\n", result - FS_STATUS_ERROR_BASE);

        return this->OpenFiles();
    }

    return false;

#endif
}

//----------------------------------------------------------
bool
SpyDataFileChannel::SetCurrentReadFile(u32 fileNo)
{
#if !defined(NW_PLATFORM_CAFE)

    NW_UNUSED_VARIABLE(fileNo);
    return false;

#else

    NW_ASSERT_MAX(fileNo, UCHAR_MAX); // 負の値は不可。
    m_CurrentReadFile = static_cast<u8>(fileNo);
    return true;

#endif
}

#if defined(NW_PLATFORM_CAFE)

//----------------------------------------------------------
bool
SpyDataFileChannel::OpenFiles()
{
    if (!m_FsClient)
    {
        return false;
    }

    if (m_DataOutputFileOpen)
    {
        return true;
    }

    FSStatus result;
    char fileName[FS_MAX_FULLPATH_SIZE];

    nw::snd::spy::internal::fnd::ScopedCriticalSection dataLock(m_DataFileLock);

    nw::ut::snprintf(fileName, sizeof(fileName), "%s/SpyData0.spychan", m_OutputDirMountPath);
    result = FSOpenFile(m_FsClient, m_FsCmdBlock, fileName, "w", &(m_DataOutputFile[0]), FS_RET_ALL_ERROR);
    if (result != FS_STATUS_OK)
    {
        NW_LOG("[nw::snd::spy::internal::SpyDataFileChannel] OpenFile: open file0 failed (FS_STATUS_ERROR_BASE %d)\n", result - FS_STATUS_ERROR_BASE);
        return false;
    }

    nw::ut::snprintf(fileName, sizeof(fileName), "%s/SpyData1.spychan", m_OutputDirMountPath);
    result = FSOpenFile(m_FsClient, m_FsCmdBlock, fileName, "w", &(m_DataOutputFile[1]), FS_RET_ALL_ERROR);
    if (result != FS_STATUS_OK)
    {
        NW_LOG("[nw::snd::spy::internal::SpyDataFileChannel] OpenFile: open file1 failed (FS_STATUS_ERROR_BASE %d)\n", result - FS_STATUS_ERROR_BASE);
        FSCloseFile(m_FsClient, m_FsCmdBlock, m_DataOutputFile[0], FS_RET_ALL_ERROR);
        return false;
    }

    m_DataOutputFileOpen = true;
    m_CurrentWriteFile = 0;
    m_CurrentReadFile = static_cast<u8>(-1);
    m_WriteLength = 0;

    return true;
}

//----------------------------------------------------------
void
SpyDataFileChannel::CloseFiles()
{
    if (!m_DataOutputFileOpen)
    {
        return;
    }

    nw::snd::spy::internal::fnd::ScopedCriticalSection dataLock(m_DataFileLock);

    FSStatus result;
    result = FSCloseFile(m_FsClient, m_FsCmdBlock, m_DataOutputFile[0], FS_RET_ALL_ERROR);
    NW_ASSERTMSG(result == FS_STATUS_OK, "FSStatus FS_STATUS_ERROR_BASE %d\n", result - FS_STATUS_ERROR_BASE);
    result = FSCloseFile(m_FsClient, m_FsCmdBlock, m_DataOutputFile[1], FS_RET_ALL_ERROR);
    NW_ASSERTMSG(result == FS_STATUS_OK, "FSStatus FS_STATUS_ERROR_BASE %d\n", result - FS_STATUS_ERROR_BASE);

    m_DataOutputFileOpen = false;
}

#endif

} // namespace nw::snd::spy::internal
} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE
