﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../../precompiled.h"

#include <nw/snd/spy/fnd/os/sndspyfnd_Thread.h>

#include <Windows.h>
#include <process.h>

namespace nw {
namespace snd {
namespace spy {
namespace internal {
namespace fnd {

namespace {

//---------------------------------------------------------------------------
//! @brief  Win32 スレッドのプライオリティに変換します。
s32 ToWin32Priority(u32 value)
{
    static const u32 Range      = Thread::MAX_THREAD_PRIORITY - Thread::MIN_THREAD_PRIORITY + 1;
    static const u32 RangeWin32 = THREAD_PRIORITY_HIGHEST - THREAD_PRIORITY_LOWEST + 1;
    return THREAD_PRIORITY_LOWEST + (value * RangeWin32 / Range);
}

}

//---------------------------------------------------------------------------
//! @brief  スレッドのメイン関数オブジェクトです。
//---------------------------------------------------------------------------
class Thread::ThreadMain
{
public:
    static uint __stdcall Run(void* param)
    {
        Thread* owner = reinterpret_cast<Thread*>(param);

        NW_ASSERT_NOT_NULL(owner);
        NW_ASSERT_NOT_NULL(owner->m_Handler);

        owner->OnRun();

        u32 result = owner->m_Handler->Run(owner->m_Param);

        owner->OnExit();

        return result;
    }
};

//---------------------------------------------------------------------------
Thread::Thread() :
m_State(STATE_NOT_RUN),
m_ID(INVALID_ID),
m_Priority(DEFAULT_THREAD_PRIORITY),
m_Name(""),
m_Handler(NULL)
{
}

//---------------------------------------------------------------------------
bool
Thread::RunArgs::IsValid() const
{
    if(priority > MAX_THREAD_PRIORITY) { return false; }
    if(handler == NULL) { return false; }
    return true;
}

//---------------------------------------------------------------------------
void
Thread::SetPriority(s32 value)
{
    if(value == 16)
    {
        ::SetThreadPriority(m_Handle, THREAD_PRIORITY_NORMAL);
        return;
    }

    if(value == 0)
    {
        ::SetThreadPriority(m_Handle, THREAD_PRIORITY_IDLE);
        return;
    }

    if(value < 8)
    {
        ::SetThreadPriority(m_Handle, THREAD_PRIORITY_LOWEST);
        return;
    }

    if(value < 16)
    {
        ::SetThreadPriority(m_Handle, THREAD_PRIORITY_BELOW_NORMAL);
        return;
    }

    if(value < 24)
    {
        ::SetThreadPriority(m_Handle, THREAD_PRIORITY_ABOVE_NORMAL);
        return;
    }

    if(value < 31)
    {
        ::SetThreadPriority(m_Handle, THREAD_PRIORITY_HIGHEST);
        return;
    }

    ::SetThreadPriority(m_Handle, THREAD_PRIORITY_NORMAL);
}

//---------------------------------------------------------------------------
void
Thread::Sleep(const TimeSpan& timeSpan)
{
    ::Sleep(static_cast<u32>(timeSpan.ToMilliSeconds()));
}

//---------------------------------------------------------------------------
bool
Thread::Create(Handle& handle, u32& id, const RunArgs& args)
{
    Handle threadHandle = reinterpret_cast<HANDLE>(::_beginthreadex(
        NULL,
        args.stackSize,
        ThreadMain::Run,
        this,
        CREATE_SUSPENDED,
        &id));

    if(threadHandle == NULL)
    {
        id = INVALID_ID;
        return false;
    }

    handle = threadHandle;
    NW_ASSERT(id != INVALID_ID);

    SetThreadPriority(handle, ToWin32Priority(args.priority));

    return true;
}

//---------------------------------------------------------------------------
void
Thread::Detach()
{
    if ( m_Handle != NULL )
    {
        ::CloseHandle( m_Handle );
        m_Handle = NULL;
    }
}

//---------------------------------------------------------------------------
void
Thread::SetName(const char* name)
{
    (void)name;
    // 対応APIがないので処理しません。
}

//---------------------------------------------------------------------------
void
Thread::SetAffinityMask(AFFINITY_MASK value)
{
    ::SetThreadAffinityMask(m_Handle, value);
}

//---------------------------------------------------------------------------
void
Thread::Resume()
{
    ::ResumeThread(m_Handle);
}

//---------------------------------------------------------------------------
void
Thread::Join()
{
    // TODO : ★Cafe版で結果を返せるか検討
    ::WaitForSingleObject(m_Handle, INFINITE);
}

//---------------------------------------------------------------------------
bool
Thread::IsTerminated() const
{
    if(m_Handle == NULL)
    {
        return true;
    }

    return ::WaitForSingleObject(m_Handle, 0) == WAIT_OBJECT_0;
}

} // namespace nw::snd::spy::internal::fnd
} // namespace nw::snd::spy::internal
} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw
