﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#ifdef NW_SND_SPY_ENABLE

#include <nw/snd/spy/sndspy_NwSndSpy.h>
#include <nw/snd/spy/sndspy_SpyController.h>
#include <nw/snd/spy/modules/sndspy_LogModule.h>

#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
using namespace nw::internal::winext;
#endif


namespace nw {
namespace snd {
namespace spy {

NwSndSpy::NwSndSpy()
    : m_pController( NULL )
    , m_FrameCallback( *this )
    , m_pSoundArchive( NULL )
    , m_IsEnableLog( true )
    , m_IsEnableProfile( true )
    , m_IsEnableVoiceInfo( true )
    , m_IsLogFuncHooked( false )
    , m_IsProfileReaderRegistered ( false )
    , m_IsDataInfoInitialized( false )
{
}

void NwSndSpy::Initialize( SpyController* controller )
{
    NW_ASSERT( m_pController == NULL );

    for(int i=0;i<VoiceInfoPacketData::VOICE_COUNT_MAX;i++)
    {
        m_VoiceInfoCondition[i].dirty = true;
    }

    nw::snd::internal::driver::SoundThread::GetInstance().RegisterSoundFrameCallback( &m_FrameCallback );

    m_pController = controller;

    controller->RegisterConnectionChangedCallback( ConnectionChangedCallbackFunc, this );

    InitializeDataInfo();

    if(m_IsEnableLog)
    {
        InitializeLog();
    }

    if(m_IsEnableProfile)
    {
        InitializeProfile();
    }

    AXRegisterAppFrameCallback(AxFrameCallbackFunc);
}

void NwSndSpy::Finalize()
{
    FinalizeLog();
    FinalizeProfile();
    FinalizeDataInfo();

    AXDeregisterAppFrameCallback(AxFrameCallbackFunc);

    if(m_pController != NULL)
    {
        m_pController->UnregisterConnectionChangedCallback( ConnectionChangedCallbackFunc );
        m_pController = NULL;

        nw::snd::internal::driver::SoundThread::GetInstance().UnregisterSoundFrameCallback( &m_FrameCallback );
    }
}

void NwSndSpy::AttachSoundArchive( const nw::snd::SoundArchive& soundArchive )
{
    m_pSoundArchive = &soundArchive;
    this->SendSoundDataInfo();
}

void NwSndSpy::DetachSoundArchive( const nw::snd::SoundArchive& soundArchive )
{
    NW_ASSERT( m_pSoundArchive == &soundArchive );
    m_pSoundArchive = NULL;
}

// NOTE: sndspy_Controllerの内部スレッドから呼ばれる
void NwSndSpy::ConnectionChangedCallbackFunc( void* arg )
{
    NwSndSpy* spy = reinterpret_cast<NwSndSpy*>(arg);
    spy->ConnectionChangedCallback();
}

void NwSndSpy::ConnectionChangedCallback()
{
    if ( m_pController == NULL ) return;
    if ( m_pSoundArchive == NULL ) return;
    if ( ! m_pSoundArchive->IsAvailable() ) return;
    if ( ! m_pController->IsConnected() ) return;

    if ( m_NwSoundDataInfoModule.IsDataRequested() )
    {
        this->SendSoundDataInfo();
    }
}

void NwSndSpy::SpyLog( const char* message )
{
    SpyController* pController = NwSndSpy::GetInstance().m_pController;
    if ( pController == NULL ) return;

    nw::snd::spy::LogModule::WriteFormat( *pController, message );
}

void NwSndSpy::FrameCallback::OnBeginSoundFrame()
{
    GetInstance().SetCurrentAudioFrame();
}

void NwSndSpy::FrameCallback::OnEndSoundFrame()
{
    SpyController* pController = NwSndSpy::GetInstance().m_pController;
    if ( pController == NULL ) return;

    if ( m_Spy.m_IsEnableVoiceInfo && m_Spy.m_AxVoiceInfoModule.IsRequested() )
    {
        VoiceInfoPacketData packet;
        packet.audioFrame = nw::snd::internal::driver::SoundThread::GetInstance().GetAxCallbackCounter();
        packet.arraySize = 0;

        for( u32 ch=0; ch < nw::snd::internal::AxVoice::VOICE_COUNT_MAX; ch++ )
        {
            const nw::snd::internal::AxVoice* voice =
                nw::snd::internal::AxVoice::detail_GetVoiceById( ch );
            if ( voice != NULL )
            {
                u32 status = voice->IsAvailable() ? voice->GetState() : nw::snd::internal::VOICE_STATE_STOP;;
                const nw::snd::internal::VoiceParam& voiceParam = voice->GetVoiceParam();
                f32 volume = voiceParam.m_Volume;
                f32 pitch = voiceParam.m_Pitch;
                f32 mainMixLeft = voiceParam.m_TvMix.mainBus[nw::snd::CHANNEL_INDEX_FRONT_LEFT];
                f32 mainMixRight = voiceParam.m_TvMix.mainBus[nw::snd::CHANNEL_INDEX_FRONT_RIGHT];
                f32 drcMixLeft = voiceParam.m_DrcMix[0].mainBus[nw::snd::CHANNEL_INDEX_FRONT_LEFT];
                f32 drcMixRight = voiceParam.m_DrcMix[0].mainBus[nw::snd::CHANNEL_INDEX_FRONT_RIGHT];

                if ( ch >= VoiceInfoPacketData::VOICE_COUNT_MAX ) continue;

                VoiceInfoCondition* cond = &m_Spy.m_VoiceInfoCondition[ch];
                if ( cond->dirty ||
                     cond->packet.status != status ||
                     cond->packet.volume != volume ||
                     cond->packet.pitch != pitch ||
                     cond->packet.mainMixLeft != mainMixLeft ||
                     cond->packet.mainMixRight != mainMixRight ||
                     cond->packet.drcMixLeft != drcMixLeft ||
                     cond->packet.drcMixRight != drcMixRight )
                {
                    cond->packet.index = static_cast<u8>(ch);
                    cond->packet.status = static_cast<u8>(status);
                    cond->packet.volume = volume;
                    cond->packet.pitch = pitch;
                    cond->packet.mainMixLeft = mainMixLeft;
                    cond->packet.mainMixRight = mainMixRight;
                    cond->packet.drcMixLeft = drcMixLeft;
                    cond->packet.drcMixRight = drcMixRight;
                    cond->packet.padding1[0] = 0;

                    cond->dirty = false;

                    VoiceInfoPacketData::VoiceInfo* dest = &packet.voiceInfo[packet.arraySize];
                    std::memcpy( dest, &cond->packet, sizeof(VoiceInfoPacketData::VoiceInfo) );
                    packet.arraySize ++;
                }
            }
        }

        if ( packet.arraySize > 0 )
        {
            // TODO : PushData まわりのコードを AxVoiceInfoModule に切り出す
            pController->PushData(
                m_Spy.m_AxVoiceInfoModule,
                &packet,
                sizeof(packet) - sizeof(VoiceInfoPacketData::VoiceInfo) * ( VoiceInfoPacketData::VOICE_COUNT_MAX - packet.arraySize ));
        }
    }

    if ( m_Spy.m_IsEnableProfile && m_Spy.m_AxProfileModule.IsRequested() )
    {
        nw::snd::SoundProfile profile;
        while ( m_Spy.m_ProfileReader.Read( &profile, 1 ) > 0 )
        {
            ProfilePacketData packet;

            packet.audioFrame = nw::snd::internal::driver::SoundThread::GetInstance().GetAxCallbackCounter();
            packet.voiceCount = profile.totalVoiceCount;
            packet.nwFrameProcessBegin = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.nwFrameProcess.begin)).GetNanoSeconds();
            packet.nwFrameProcessEnd = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.nwFrameProcess.end)).GetNanoSeconds();
            packet.nwVoiceParamUpdateBegin = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.nwVoiceParamUpdate.begin)).GetNanoSeconds();
            packet.nwVoiceParamUpdateEnd = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.nwVoiceParamUpdate.end)).GetNanoSeconds();
            packet.ppcVoiceRenderingBegin = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.voiceRendering.begin)).GetNanoSeconds();
            packet.ppcVoiceRenderingEnd = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.voiceRendering.end)).GetNanoSeconds();
            packet.auxProcessBegin = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.auxProcess.begin)).GetNanoSeconds();
            packet.auxProcessEnd = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.auxProcess.end)).GetNanoSeconds();
            packet.dspFrameProcessBegin = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.dspFrameProcess.begin)).GetNanoSeconds();
            packet.dspFrameProcessEnd = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.dspFrameProcess.end)).GetNanoSeconds();
            packet.syncVoiceParamBegin = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.syncVoiceParam.begin)).GetNanoSeconds();
            packet.syncVoiceParamEnd = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.syncVoiceParam.end)).GetNanoSeconds();
            packet.outputFormatProcessBegin = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.outputFormatProcess.begin)).GetNanoSeconds();
            packet.outputFormatProcessEnd = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.outputFormatProcess.end)).GetNanoSeconds();
            packet.mainMixProcessBegin = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.mainMixProcess.begin)).GetNanoSeconds();
            packet.mainMixProcessEnd = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.mainMixProcess.end)).GetNanoSeconds();
            packet.finalMixProcessBegin = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.finalMixProcess.begin)).GetNanoSeconds();
            packet.finalMixProcessEnd = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.finalMixProcess.end)).GetNanoSeconds();
            packet.axIntrBegin = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.axFrameProcess.begin)).GetNanoSeconds();
            packet.axIntrEnd = static_cast<nw::ut::TimeSpan>(nw::ut::Tick(profile.axFrameProcess.end)).GetNanoSeconds();

            // TODO : PushData まわりのコードを AxProfileModule に切り出す
            pController->PushData(
                m_Spy.m_AxProfileModule,
                &packet,
                sizeof(packet) );
        }
    }
}

void NwSndSpy::AxFrameCallbackFunc()
{
    GetInstance().SendFinalOutPacket();
}

void NwSndSpy::SetCurrentAudioFrame()
{
    SpyController* pController = NwSndSpy::GetInstance().m_pController;
    if ( pController == NULL ) return;

    pController->SetCurrentAudioFrame(
      nw::snd::internal::driver::SoundThread::GetInstance().GetAxCallbackCounter()
    );
}

bool NwSndSpy::SendFinalOutPacket()
{
    if ( !m_AxFinalOutModule.IsRequested()) return false;

    FinalOutPacketData packet;
    AXDEVICEFINALOUTPUTSTRUCT finalInfoTv;
    AXDEVICEFINALOUTPUTSTRUCT finalInfoDrc;

    AXPB_ERROR_CODE error = AXGetDeviceFinalOutput(AX_DEVICE_TV, &packet.finalOutput[0], sizeof(packet.finalOutput), &finalInfoTv);
    NW_ASSERT(error == AXPB_ERROR_NONE);

    u32 samplePerFrame = finalInfoTv.length / finalInfoTv.numChannels / sizeof(s16);
    u32 numChannelsTv = finalInfoTv.numChannels;
    u32 channelBitFlag = 0;

    if ( numChannelsTv == 6 )
    {
        // Center & LFE と Rear の有効サンプルがあるかどうかのチェック
        bool center_lfe_flag = false;
        bool rear_flag = false;
        const u32* srcp = reinterpret_cast<u32*>(&packet.finalOutput[0]);
        for( unsigned int index=0; index < samplePerFrame; index++ )
        {
            srcp++; // Front LR

            if (*srcp++ != 0 ) center_lfe_flag = true; // Center LFE
            if (*srcp++ != 0 ) rear_flag = true; // Rear LR
        }

        srcp = reinterpret_cast<u32*>(&packet.finalOutput[0]);
        u32* dstp = reinterpret_cast<u32*>(&packet.finalOutput[0]);
        if ( center_lfe_flag )
        {
            if ( rear_flag )
            {
                // 全ての要素が有効なのでそのまま
                channelBitFlag |=
                    FinalOutPacketData::CHANNEL_BIT_FRONT_LEFT |
                    FinalOutPacketData::CHANNEL_BIT_FRONT_RIGHT |
                    FinalOutPacketData::CHANNEL_BIT_REAR_LEFT |
                    FinalOutPacketData::CHANNEL_BIT_REAR_RIGHT |
                    FinalOutPacketData::CHANNEL_BIT_FRONT_CENTER |
                    FinalOutPacketData::CHANNEL_BIT_LFE;
                numChannelsTv = 6;
            }
            else
            {
                // Rearのみ無し
                for( unsigned int index=0; index < samplePerFrame; index++ )
                {
                    *dstp++ = *srcp++; // Front LR
                    *dstp++ = *srcp++; // Center LFE
                    srcp++; // Rear LR
                }
                channelBitFlag |=
                    FinalOutPacketData::CHANNEL_BIT_FRONT_LEFT |
                    FinalOutPacketData::CHANNEL_BIT_FRONT_RIGHT |
                    FinalOutPacketData::CHANNEL_BIT_FRONT_CENTER |
                    FinalOutPacketData::CHANNEL_BIT_LFE;
                numChannelsTv = 4;
            }
        }
        else
        {
            if ( rear_flag )
            {
                // Center/LFEのみ無し
                for( unsigned int index=0; index < samplePerFrame; index++ )
                {
                    *dstp++ = *srcp++; // Front LR
                    srcp++; // Center LFE
                    *dstp++ = *srcp++; // Rear LR
                }
                channelBitFlag |=
                    FinalOutPacketData::CHANNEL_BIT_FRONT_LEFT |
                    FinalOutPacketData::CHANNEL_BIT_FRONT_RIGHT |
                    FinalOutPacketData::CHANNEL_BIT_REAR_LEFT |
                    FinalOutPacketData::CHANNEL_BIT_REAR_RIGHT;
                numChannelsTv = 4;
            }
            else
            {
                // Center/LFE Rear 無し
                for( unsigned int index=0; index < samplePerFrame; index++ )
                {
                    *dstp++ = *srcp++; // Front LR
                    srcp++; // Center LFE
                    srcp++; // Rear LR
                }
                channelBitFlag |=
                    FinalOutPacketData::CHANNEL_BIT_FRONT_LEFT |
                    FinalOutPacketData::CHANNEL_BIT_FRONT_RIGHT;
                numChannelsTv = 2;
            }
        }
    }
    else if ( numChannelsTv == 4 )
    {
        // Center & LFE と Rear の有効サンプルがあるかどうかのチェック
        bool rear_flag = false;
        const u32* srcp = reinterpret_cast<u32*>(&packet.finalOutput[0]);
        for( unsigned int index=0; index < samplePerFrame; index++ )
        {
            srcp++; // Front LR
            if (*srcp++ != 0 )
            {
#if 1   // 強制無効化
#else
                rear_flag = true;
#endif
                break;
            }
        }

#if 0   // 強制出力
        rear_flag = true;
#endif

        if ( rear_flag )
        {
            // 全ての要素が有効なのでそのまま
            channelBitFlag |=
                FinalOutPacketData::CHANNEL_BIT_FRONT_LEFT |
                FinalOutPacketData::CHANNEL_BIT_FRONT_RIGHT |
                FinalOutPacketData::CHANNEL_BIT_REAR_LEFT |
                FinalOutPacketData::CHANNEL_BIT_REAR_RIGHT;
            numChannelsTv = 4;
        }
        else
        {
            // Rear無し
            srcp = reinterpret_cast<u32*>(&packet.finalOutput[0]);
            u32* dstp = reinterpret_cast<u32*>(&packet.finalOutput[0]);
            for( unsigned int index=0; index < samplePerFrame; index++ )
            {
                *dstp++ = *srcp++; // Front LR
                srcp++; // Rear LR
            }
            channelBitFlag |=
                FinalOutPacketData::CHANNEL_BIT_FRONT_LEFT |
                FinalOutPacketData::CHANNEL_BIT_FRONT_RIGHT;
            numChannelsTv = 2;
        }
    }
    else if ( numChannelsTv == 2 )
    {
        channelBitFlag |=
            FinalOutPacketData::CHANNEL_BIT_FRONT_LEFT |
            FinalOutPacketData::CHANNEL_BIT_FRONT_RIGHT;
        numChannelsTv = 2;
    }
    else
    {
        NW_ERR("Unknown TV channels %d", numChannelsTv);
        return false;
    }

    u32 sampleOffsetForDrc= samplePerFrame * numChannelsTv;
    error = AXGetDeviceFinalOutput(AX_DEVICE_DRC, &packet.finalOutput[sampleOffsetForDrc], sizeof(packet.finalOutput)-sampleOffsetForDrc*sizeof(s16), &finalInfoDrc);
    NW_ASSERT(error == AXPB_ERROR_NONE);
    NW_ASSERT( samplePerFrame == finalInfoDrc.length / finalInfoDrc.numChannels / sizeof(s16) );

    u32 numChannelsDrc = finalInfoDrc.numChannels;
    if ( numChannelsDrc == 2 )
    {
        bool drc_flag = false;
        const u32* srcp = reinterpret_cast<u32*>(&packet.finalOutput[sampleOffsetForDrc]);
        for( unsigned int index=0; index < samplePerFrame; index++ )
        {
            if (*srcp++ != 0)
            {
                drc_flag = true;
                break;
            }
        }

        if ( drc_flag )
        {
            channelBitFlag |=
                FinalOutPacketData::CHANNEL_BIT_DRC_LEFT |
                FinalOutPacketData::CHANNEL_BIT_DRC_RIGHT;
            numChannelsDrc = 2;
        }
        else
        {
            numChannelsDrc = 0;
        }
    }
    else
    {
        NW_ERR("Unknown DRC channels %d", numChannelsDrc);
        return false;
    }

    packet.audioFrame = nw::snd::internal::driver::SoundThread::GetInstance().GetAxCallbackCounter();
    packet.channelBitFlag = channelBitFlag;
    packet.samplePerFrame = samplePerFrame;

    // TODO : PushData まわりのコードを AxFinalOutModule に切り出す
    return m_pController->PushData(
        m_AxFinalOutModule,
        &packet,
        sizeof(packet) - sizeof(packet.finalOutput) + samplePerFrame * ( numChannelsTv + numChannelsDrc ) * sizeof(s16)
    );
}

void NwSndSpy::WriteSoundInfoToPacket(
    SoundStatusInfoPacketData* packet,
    const nw::snd::internal::BasicSound* sound)
{
    if ( packet->soundCount >= SoundStatusInfoPacketData::MAX_SOUND_INFO ) return;

    SoundStatusInfoPacketData::SoundInfo* soundInfo = &packet->soundInfoArray[packet->soundCount];
    soundInfo->instanceId = sound->GetInstanceId();
    soundInfo->soundId = sound->GetId();
    u8 statusBitFlag = 0;
    if ( sound->IsPause() )  {  statusBitFlag |= SoundStatusInfoPacketData::STATUS_PAUSE;  }
    if ( sound->IsStarted() )  {  statusBitFlag |= SoundStatusInfoPacketData::STATUS_START;  }
    if ( sound->IsPrepared() )  {  statusBitFlag |= SoundStatusInfoPacketData::STATUS_PREPARED;  }
    soundInfo->statusBitFlag = statusBitFlag;
    soundInfo->padding1[0] = 0;
    soundInfo->padding1[1] = 0;
    soundInfo->padding1[2] = 0;

    packet->soundCount++;
}

bool NwSndSpy::SendSoundArchivePlayerInfo( const nw::snd::SoundArchivePlayer& soundArchivePlayer )
{
    SpyController* pController = NwSndSpy::GetInstance().m_pController;
    if ( pController == NULL ) return false;

    if ( m_NwSoundDataInfoModule.IsDataRequested() )
    {
        this->SendSoundDataInfo();
    }

    if ( m_NwSoundStatusInfoModule.IsRequested() )
    {
        SoundStatusInfoPacketData packet;
        packet.soundCount = 0;

        {
            const nw::snd::internal::SequenceSoundInstanceManager& instanceManager =
                soundArchivePlayer.detail_GetSequenceSoundInstanceManager();
            const nw::snd::internal::SequenceSoundInstanceManager::PriorityList& soundList = instanceManager.GetSoundList();
            nw::snd::internal::SequenceSoundInstanceManager::PriorityList::const_iterator itr;
            for ( itr = soundList.GetBeginIter(); itr != soundList.GetEndIter(); ++itr )
            {
                WriteSoundInfoToPacket(&packet, &*itr);
            }
        }
        {
            const nw::snd::internal::WaveSoundInstanceManager& instanceManager =
                soundArchivePlayer.detail_GetWaveSoundInstanceManager();
            const nw::snd::internal::WaveSoundInstanceManager::PriorityList& soundList = instanceManager.GetSoundList();
            nw::snd::internal::WaveSoundInstanceManager::PriorityList::const_iterator itr;
            for ( itr = soundList.GetBeginIter(); itr != soundList.GetEndIter(); ++itr )
            {
                WriteSoundInfoToPacket(&packet, &*itr);
            }
        }
        {
            const nw::snd::internal::StreamSoundInstanceManager& instanceManager =
                soundArchivePlayer.detail_GetStreamSoundInstanceManager();
            const nw::snd::internal::StreamSoundInstanceManager::PriorityList& soundList = instanceManager.GetSoundList();
            nw::snd::internal::StreamSoundInstanceManager::PriorityList::const_iterator itr;
            for ( itr = soundList.GetBeginIter(); itr != soundList.GetEndIter(); ++itr )
            {
                WriteSoundInfoToPacket(&packet, &*itr);
            }
        }

//    if ( packet.soundCount > 0 )
        {
            // TODO : PushData まわりのコードを NwSoundStatusInfoModule に切り出す
            bool result = pController->PushData(
                m_NwSoundStatusInfoModule,
                &packet,
                sizeof(packet) - sizeof(packet.soundInfoArray) + sizeof(packet.soundInfoArray[0]) * packet.soundCount);
            if ( ! result  ) return false;
        }
    }

    return true;
}

bool NwSndSpy::SendSoundDataInfo()
{
    if (m_pController == NULL) return false;
    if (m_pSoundArchive == NULL) return false;
    if (!m_pSoundArchive->IsAvailable()) return false;
    if (!m_pController->IsConnected()) return false;
    if (!m_NwSoundDataInfoModule.IsRequested()) return false;

    return SendSoundDataSoundInfo( *m_pSoundArchive ) &&
        SendSoundDataPlayerInfo( *m_pSoundArchive );
}

// NOTE: 異なるスレッドから呼ばれる
bool NwSndSpy::SendSoundDataSoundInfo( const nw::snd::SoundArchive& soundArchive )
{
    SoundDataInfoPacketData packet;
    packet.itemType = SoundDataInfoPacketData::ITEM_TYPE_SOUND;
    packet.itemCount = 0;
    u32 itemAreaSize = 0;

    u32 soundCount = soundArchive.GetSoundCount();
    for ( u32 index=0; index<soundCount; index++)
    {
        if ( itemAreaSize + sizeof(SoundDataInfoPacketData::SoundInfo) > SoundDataInfoPacketData::ITEM_AREA_SIZE )
        {
            bool result = m_pController->PushData(
                m_NwSoundDataInfoModule,
                &packet,
                sizeof(packet) - sizeof(packet.itemArea) + itemAreaSize);
            if ( ! result ) return false;

            packet.itemCount = 0;
            itemAreaSize = 0;
        }

        nw::snd::SoundArchive::ItemId soundId = soundArchive.GetSoundIdFromIndex(index);
        nw::snd::SoundArchive::SoundInfo soundInfo;
        if ( ! soundArchive.ReadSoundInfo( soundId, &soundInfo ) )
        {
            continue;
        }

        SoundDataInfoPacketData::SoundInfo* pItem = reinterpret_cast<SoundDataInfoPacketData::SoundInfo*>(&packet.itemArea[itemAreaSize/sizeof(u32)]);
        pItem->soundId = soundId;
        pItem->playerId = soundInfo.playerId;
        switch( soundArchive.GetSoundType(soundId) )
        {
        case nw::snd::SoundArchive::SOUND_TYPE_SEQ:
            pItem->soundType = SoundDataInfoPacketData::SOUND_TYPE_SEQUENCE;
            break;
        case nw::snd::SoundArchive::SOUND_TYPE_STRM:
            pItem->soundType = SoundDataInfoPacketData::SOUND_TYPE_STREAM;
            break;
        case nw::snd::SoundArchive::SOUND_TYPE_WAVE:
            pItem->soundType = SoundDataInfoPacketData::SOUND_TYPE_WAVE;
            break;
        default:
            break;
        }
        pItem->volume = soundInfo.volume;
        pItem->playerPriority = soundInfo.playerPriority;
        pItem->padding1[0] = 0;

        const char* soundLabel = soundArchive.GetItemLabel(soundId);
        if ( soundLabel == NULL )
        {
            soundLabel = "";
        }
        pItem->label.Assign(soundLabel);
        itemAreaSize += sizeof(*pItem) - sizeof(pItem->label) + nw::snd::spy::internal::fnd::RoundUp(pItem->label.len+sizeof(u8),sizeof(u32));
        packet.itemCount++;
    }

    if ( packet.itemCount > 0 )
    {
        // TODO : PushData まわりのコードを NwSoundDataInfoModule に切り出す
        bool result = m_pController->PushData(
            m_NwSoundDataInfoModule,
            &packet,
            sizeof(packet) - sizeof(packet.itemArea) + itemAreaSize);
        if ( ! result ) return false;
    }

    return true;
}

bool NwSndSpy::SendSoundDataPlayerInfo( const nw::snd::SoundArchive& soundArchive )
{
    SoundDataInfoPacketData packet;
    packet.itemType = SoundDataInfoPacketData::ITEM_TYPE_PLAYER;
    packet.itemCount = 0;
    u32 itemAreaSize = 0;

    u32 playerCount = soundArchive.GetPlayerCount();
    for ( u32 index=0; index<playerCount; index++)
    {
        if ( itemAreaSize + sizeof(SoundDataInfoPacketData::PlayerInfo) > SoundDataInfoPacketData::ITEM_AREA_SIZE )
        {
            // TODO : PushData まわりのコードを NwSoundDataInfoModule に切り出す
            bool result = m_pController->PushData(
                m_NwSoundDataInfoModule,
                &packet,
                sizeof(packet) - sizeof(packet.itemArea) + itemAreaSize);
            if ( ! result ) return false;

            packet.itemCount = 0;
            itemAreaSize = 0;
        }

        nw::snd::SoundArchive::ItemId playerId = soundArchive.GetPlayerIdFromIndex(index);
        nw::snd::SoundArchive::PlayerInfo playerInfo;
        if ( ! soundArchive.ReadPlayerInfo( playerId, &playerInfo ) )
        {
            continue;
        }

        SoundDataInfoPacketData::PlayerInfo* pItem = reinterpret_cast<SoundDataInfoPacketData::PlayerInfo*>(&packet.itemArea[itemAreaSize/sizeof(u32)]);
        pItem->playerId = playerId;

        const char* label = soundArchive.GetItemLabel(playerId);
        if ( label == NULL )
        {
            label = "";
        }
        pItem->label.Assign(label);
        itemAreaSize += sizeof(*pItem) - sizeof(pItem->label) + nw::snd::spy::internal::fnd::RoundUp(pItem->label.len+sizeof(u8),sizeof(u32));
        packet.itemCount++;
    }

    if ( packet.itemCount > 0 )
    {
        // TODO : PushData まわりのコードを NwSoundDataInfoModule に切り出す
        bool result = m_pController->PushData(
            m_NwSoundDataInfoModule,
            &packet,
            sizeof(packet) - sizeof(packet.itemArea) + itemAreaSize);
        if ( ! result ) return false;
    }

    return true;
}

void NwSndSpy::InitializeLog()
{
    if(!IsInitialized())
    {
        return;
    }

    if( m_IsLogFuncHooked )
    {
        return;
    }

    nw::snd::internal::Debug_HookLogFunc( SpyLog );
    m_IsLogFuncHooked = true;
}

void NwSndSpy::FinalizeLog()
{
    if(!IsInitialized())
    {
        return;
    }

    if( !m_IsLogFuncHooked )
    {
        return;
    }

    nw::snd::internal::Debug_HookLogFunc( NULL );
    m_IsLogFuncHooked = false;
}

void NwSndSpy::InitializeProfile()
{
    if(!IsInitialized())
    {
        return;
    }

    if( m_IsProfileReaderRegistered )
    {
        return;
    }

    nw::snd::internal::driver::SoundThread::GetInstance().RegisterProfileReader( m_ProfileReader );
    m_IsProfileReaderRegistered = true;
}

void NwSndSpy::FinalizeProfile()
{
    if(!IsInitialized())
    {
        return;
    }

    if( !m_IsProfileReaderRegistered )
    {
        return;
    }

    nw::snd::internal::driver::SoundThread::GetInstance().UnregisterProfileReader( m_ProfileReader );
    m_IsProfileReaderRegistered = false;
}

void NwSndSpy::InitializeDataInfo()
{
    if (!IsInitialized())
    {
        return;
    }

    if (m_IsDataInfoInitialized)
    {
        return;
    }

    m_pController->InstallModule(m_AxFinalOutModule);
    m_pController->InstallModule(m_AxProfileModule);
    m_pController->InstallModule(m_AxVoiceInfoModule);
    m_pController->InstallModule(m_NwSoundDataInfoModule);
    m_pController->InstallModule(m_NwSoundStatusInfoModule);

    m_IsDataInfoInitialized = true;
}

void NwSndSpy::FinalizeDataInfo()
{
    if (!IsInitialized())
    {
        return;
    }

    if (!m_IsDataInfoInitialized)
    {
        return;
    }

    m_pController->UninstallModule(m_AxFinalOutModule);
    m_pController->UninstallModule(m_AxProfileModule);
    m_pController->UninstallModule(m_AxVoiceInfoModule);
    m_pController->UninstallModule(m_NwSoundDataInfoModule);
    m_pController->UninstallModule(m_NwSoundStatusInfoModule);

    m_IsDataInfoInitialized = false;
}

} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE
