﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/edit/sndedit_SoundEditConnection.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/fnd/string/sndfnd_String.h>
#include <nw/snd/fnd/string/sndfnd_Path.h>
#include <nw/snd/edit/sndedit_SoundArchiveEditor.h>
#include <nw/snd/edit/hio/sndedit_HioSyncChannel.h>
#include <nw/snd/edit/hio/sndedit_HioAsyncChannel.h>
#include <nw/snd/edit/protocol/sndedit_ItemCachePacket.h>
#include <nw/snd/edit/protocol/sndedit_QueryItemsPacket.h>
#include <nw/snd/edit/protocol/sndedit_QueryItemInfoPacket.h>
#include <nw/snd/edit/res/sndedit_ResItemInfo.h>

namespace nw {
namespace snd {
namespace edit {
namespace internal {

//----------------------------------------------------------
void
SoundEditConnection::Finalize()
{
    Close();
    UnsetupSyncChannel();

    m_SendPacketBuffer = NULL;
    m_SendPacketBufferSize = 0;

    m_CacheSyncInterval = 0;
    m_SendTimeout = 0;
    m_HioManager = NULL;
    m_SyncChannel = NULL;
    m_FuncChannel = NULL;
}

//----------------------------------------------------------
Result
SoundEditConnection::Open(SoundArchiveEditor& soundArchiveEditor)
{
    if(!IsInitialized())
    {
        NW_FATAL_ERROR("SoundEditConnection is not initialized.\n");
        return Result(SNDEDIT_RESULT_NOT_INITIALIZED);
    }

    if(IsOpened() && m_SoundArchiveEditor->IsStarted())
    {
        return Result(SNDEDIT_RESULT_TRUE);
    }

    Result result = soundArchiveEditor.detail_Start(*this);

    if(!result.IsTrue())
    {
        Close();
        return result;
    }

    m_CacheSyncStopWatch.Start();

    m_SoundArchiveEditor = &soundArchiveEditor;

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundEditConnection::Close()
{
    if(!IsInitialized())
    {
        NW_FATAL_ERROR("SoundEditConnection is not initialized.\n");
        return;
    }

    m_CacheSyncStopWatch.Stop();

    if(m_SoundArchiveEditor != NULL)
    {
        m_SoundArchiveEditor->detail_Stop();
        m_SoundArchiveEditor = NULL;
    }
}

//----------------------------------------------------------
u32
SoundEditConnection::GetRequiredMemorySize(u32 maxItemName) const
{
    // QueryItemInfoPacket 送信用のバッファサイズ
    u32 result = internal::QueryItemInfoPacket::GetRequiredSize(maxItemName);

    // BeginItemCachePacket 送信用のバッファサイズ
    result = ut::Max(result, internal::BeginItemCachePacket::GetRequiredSize(maxItemName));

    // EndItemCachePacket 送信用のバッファサイズ
    result = ut::Max(result, internal::EndItemCachePacket::GetRequiredSize(maxItemName));

    return result;
}

//----------------------------------------------------------
u32
SoundEditConnection::GetRequiredMemorySizeForOpenFile()
{
    return HioManager::GetRequiredMemorySizeForOpenFile();
}

//----------------------------------------------------------
void
SoundEditConnection::Update()
{
    if(!IsInitialized())
    {
        NW_FATAL_ERROR("SoundEditConnection is not initialized.\n");
        return;
    }

    if(!IsOpened())
    {
        return;
    }

    // 無効アイテムチェック間隔を経過したら...
    // 無効アイテム一覧を要求します。
    if(m_CacheSyncStopWatch.GetElapsedTime().ToMilliSeconds() >= m_CacheSyncInterval)
    {
        BeginCacheSync();
    }
}

//----------------------------------------------------------
Result
SoundEditConnection::BeginUpdateItemCache(const char* itemName)
{
    NW_ASSERT_NOT_NULL(itemName);
    NW_ASSERT(*itemName != '\0');

    NW_ASSERT_NOT_NULL(m_SendPacketBuffer);

    internal::BeginItemCachePacket* packet =
        internal::BeginItemCachePacket::CreateInstance(m_SendPacketBuffer, itemName);

    packet->GetBody().SetAction(internal::CACHE_ACTION_UPDATE);

    const internal::HioPacket* replyPacket = NULL;

    internal::HioResult result =
        m_FuncChannel->Send(replyPacket, *packet, m_SendTimeout);

    if(!result.IsTrue())
    {
        replyPacket = &internal::HioPacket::Null();
        return Result(static_cast<ResultType>(result));
    }

    return Result(static_cast<ResultType>(result));
}

//----------------------------------------------------------
Result
SoundEditConnection::BeginRemoveItemCache(const char* itemName)
{
    NW_ASSERT_NOT_NULL(itemName);
    NW_ASSERT(*itemName != '\0');

    NW_ASSERT_NOT_NULL(m_SendPacketBuffer);

    internal::BeginItemCachePacket* packet =
        internal::BeginItemCachePacket::CreateInstance(m_SendPacketBuffer, itemName);

    packet->GetBody().SetAction(internal::CACHE_ACTION_REMOVE);

    const internal::HioPacket* replyPacket = NULL;

    internal::HioResult result =
        m_FuncChannel->Send(replyPacket, *m_SendPacketBuffer, m_SendTimeout);

    if(!result.IsTrue())
    {
        replyPacket = &internal::HioPacket::Null();
        return Result(static_cast<ResultType>(result));
    }

    return Result(static_cast<ResultType>(result));
}

//----------------------------------------------------------
Result
SoundEditConnection::BeginRemoveAllItemCaches()
{
    NW_ASSERT_NOT_NULL(m_SendPacketBuffer);

    internal::BeginItemCachePacket* packet =
        internal::BeginItemCachePacket::CreateInstance(m_SendPacketBuffer, NULL);

    packet->GetBody().SetIsTargetAll(true);
    packet->GetBody().SetAction(internal::CACHE_ACTION_REMOVE);

    const internal::HioPacket* replyPacket = NULL;

    internal::HioResult result =
        m_FuncChannel->Send(replyPacket, *m_SendPacketBuffer, m_SendTimeout);

    if(!result.IsTrue())
    {
        replyPacket = &internal::HioPacket::Null();
        return Result(static_cast<ResultType>(result));
    }

    return Result(static_cast<ResultType>(result));
}

//----------------------------------------------------------
Result
SoundEditConnection::EndUpdateItemCache(Result itemCacheResult, const char* itemName)
{
    NW_ASSERT_NOT_NULL(itemName);
    NW_ASSERT(*itemName != '\0');

    NW_ASSERT_NOT_NULL(m_SendPacketBuffer);

    internal::EndItemCachePacket* packet =
        internal::EndItemCachePacket::CreateInstance(m_SendPacketBuffer, itemName, itemCacheResult);

    packet->GetBody().SetAction(internal::CACHE_ACTION_UPDATE);

    const internal::HioPacket* replyPacket = NULL;

    internal::HioResult result =
        m_FuncChannel->Send(replyPacket, *packet, m_SendTimeout);

    if(!result.IsTrue())
    {
        replyPacket = &internal::HioPacket::Null();
        return Result(static_cast<ResultType>(result));
    }

    return Result(static_cast<ResultType>(result));
}

//----------------------------------------------------------
Result
SoundEditConnection::EndRemoveItemCache(Result itemCacheResult, const char* itemName)
{
    NW_ASSERT_NOT_NULL(itemName);
    NW_ASSERT(*itemName != '\0');

    NW_ASSERT_NOT_NULL(m_SendPacketBuffer);

    internal::EndItemCachePacket* packet =
        internal::EndItemCachePacket::CreateInstance(m_SendPacketBuffer, itemName, itemCacheResult);

    packet->GetBody().SetAction(internal::CACHE_ACTION_REMOVE);

    const internal::HioPacket* replyPacket = NULL;

    internal::HioResult result =
        m_FuncChannel->Send(replyPacket, *packet, m_SendTimeout);

    if(!result.IsTrue())
    {
        replyPacket = &internal::HioPacket::Null();
        return Result(static_cast<ResultType>(result));
    }

    return Result(static_cast<ResultType>(result));
}

//----------------------------------------------------------
Result
SoundEditConnection::EndRemoveAllItemCaches(Result itemCacheResult)
{
    NW_ASSERT_NOT_NULL(m_SendPacketBuffer);

    internal::EndItemCachePacket* packet =
        internal::EndItemCachePacket::CreateInstance(m_SendPacketBuffer, NULL, itemCacheResult);

    packet->GetBody().SetIsTargetAll(true);

    const internal::HioPacket* replyPacket = NULL;

    internal::HioResult result =
        m_FuncChannel->Send(replyPacket, *packet, m_SendTimeout);

    if(!result.IsTrue())
    {
        replyPacket = &internal::HioPacket::Null();
        return Result(static_cast<ResultType>(result));
    }

    return Result(static_cast<ResultType>(result));
}

//----------------------------------------------------------
Result
SoundEditConnection::QueryItemInfo(const internal::HioPacket*& replyPacket, const char* itemName)
{
    if(!IsInitialized())
    {
        NW_FATAL_ERROR("SoundEditConnection is not initialized.\n");
        return Result(SNDEDIT_RESULT_NOT_INITIALIZED);
    }

    if(!IsOpened())
    {
        replyPacket = &internal::HioPacket::Null();
        return Result(SNDEDIT_RESULT_NOT_CONNECTED);
    }

    if(itemName == NULL || *itemName == '\0')
    {
        NW_FATAL_ERROR("invalid arguments.\n");
        replyPacket = &internal::HioPacket::Null();
        return Result(SNDEDIT_RESULT_FAILED);
    }

    NW_ASSERT_NOT_NULL(m_SendPacketBuffer);
    internal::QueryItemInfoPacket::CreateInstance(m_SendPacketBuffer, itemName);

    internal::HioResult result =
        m_FuncChannel->Send(replyPacket, *m_SendPacketBuffer, m_SendTimeout);

    if(!result.IsTrue())
    {
        replyPacket = &internal::HioPacket::Null();
        return Result(static_cast<ResultType>(result));
    }

    HioResult packetResult = replyPacket->GetHeader().GetResult();

    if(!packetResult.IsTrue())
    {
        replyPacket = &internal::HioPacket::Null();
        return Result(static_cast<ResultType>(packetResult));
    }

    return Result(static_cast<ResultType>(result));
}

//----------------------------------------------------------
void
SoundEditConnection::SetupSyncChannel(u32 maxItemName)
{
    NW_ASSERT_NOT_NULL(m_SyncChannel);

    m_QueryItemsReplyHandler.Initialize(*this, maxItemName);
    m_SyncChannel->RegisterMessageHandler(m_QueryItemsReplyHandler);
}

//----------------------------------------------------------
void
SoundEditConnection::UnsetupSyncChannel()
{
    if(m_SyncChannel != NULL)
    {
        m_SyncChannel->UnregisterMessageHandler(m_QueryItemsReplyHandler);
    }

    m_QueryItemsReplyHandler.Finalize();
}

//----------------------------------------------------------
bool
SoundEditConnection::BeginCacheSync()
{
    m_CacheSyncStopWatch.Reset();

    if(!IsInitialized() || !IsOpened())
    {
        return false;
    }

    return m_SyncChannel->Send(internal::QueryItemsPacket()).IsSucceeded();
}

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
