﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/edit/sndedit_SoundEdit.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

namespace nw {
namespace snd {
namespace edit {

//----------------------------------------------------------
SoundEdit::SoundEdit()
: m_IsInitialized(false)
, m_IsStarted(false)
, m_pMemoryForUpdateCachesThreadStack(NULL)
, m_UpdateCachesThreadStackSize(0)
, m_pMemoryForSoundArchiveEditor(NULL)
, m_pMemoryForSoundEditSession(NULL)
, m_pMemoryForSoundControlSession(NULL)
{
}

//----------------------------------------------------------
SoundEdit::~SoundEdit()
{
    // Finalize() 済みであることを確認します。
    NW_ASSERTMSG(!IsInitialized(), "SoundEdit is not finalized.\n");
}

//----------------------------------------------------------
void
SoundEdit::Initialize(const InitializeArg& args, const Option& param)
{
    NW_NULL_ASSERT(args.buffer);
    NW_ASSERT(args.bufferSize > 0);
    NW_NULL_ASSERT(args.soundArchive);
    NW_NULL_ASSERT(args.soundArchivePlayer);
#if defined( NW_PLATFORM_CAFE )
    NW_NULL_ASSERT(args.fsClient);
    NW_NULL_ASSERT(args.fsHfioMountPath);
#endif

    NW_ASSERT(param.threadStackSize > 0);
    NW_ASSERT(param.sndEditHeapSize > 0);

    if (m_IsInitialized)
    {
        return;
    }

    // アロケータの初期化
    m_Allocator.Initialize( args.buffer, args.bufferSize );

    // キャッシュヘルパー用のスタックメモリの確保
    m_UpdateCachesThreadStackSize = param.threadStackSize;
#if defined( NW_USE_NINTENDO_SDK )
    m_pMemoryForUpdateCachesThreadStack = static_cast<u8*>( m_Allocator.Alloc(m_UpdateCachesThreadStackSize, nn::os::StackRegionAlignment) );
#else
    m_pMemoryForUpdateCachesThreadStack = static_cast<u8*>( m_Allocator.Alloc(m_UpdateCachesThreadStackSize) );
#endif

    // サウンドアーカイブエディタの初期化
    {
        nw::snd::edit::SoundArchiveEditor::Configs configs;

        m_pMemoryForSoundArchiveEditor = m_Allocator.Alloc( param.sndEditHeapSize );
        NW_NULL_ASSERT( m_pMemoryForSoundArchiveEditor );

        nw::snd::edit::Result result = m_SoundArchiveEditor.Initialize(
            m_pMemoryForSoundArchiveEditor,
            param.sndEditHeapSize,
            args.soundArchive,
            args.soundArchivePlayer,
            configs );
        NW_ASSERT( result.IsTrue() );
    }

    // サウンド編集セッションの初期化
    {
        nw::snd::edit::SoundEditSession::Configs configs;
        configs.port0 = args.port0;
        configs.port1 = args.port1;
        u32 memorySize = m_SoundEditSession.GetRequiredMemorySize(configs);

        m_pMemoryForSoundEditSession = m_Allocator.Alloc( memorySize );
        NW_NULL_ASSERT( m_pMemoryForSoundEditSession );

        nw::snd::edit::Result result = m_SoundEditSession.Initialize(
            m_pMemoryForSoundEditSession,
            memorySize,
#if defined( NW_PLATFORM_CAFE )
            args.fsClient,
            args.fsHfioMountPath,
#endif
            configs );
        if (!result.IsTrue())
        {
            NW_WARNING(false, "SoundEditSession::Initialize is failed (%s)\n", result.ToString());
            return;
        }
    }

    // サウンドオブジェクトコントローラの初期化
    {
        nw::snd::edit::Result result = m_SoundObjectController.Initialize(args.soundArchivePlayer);
        NW_ASSERT( result.IsTrue() );
    }

    // サウンド制御セッションの初期化
    {
        nw::snd::ctrl::SoundControlSession::Configs configs;
        configs.port = args.port2;
        u32 memorySize = m_SoundControlSession.GetRequiredMemorySize(configs);

        m_pMemoryForSoundControlSession = m_Allocator.Alloc( memorySize );
        NW_NULL_ASSERT( m_pMemoryForSoundControlSession );

        nw::snd::edit::Result result = m_SoundControlSession.Initialize(
            m_pMemoryForSoundControlSession,
            memorySize,
            configs );
        if (!result.IsTrue())
        {
            NW_WARNING(false, "SoundControlSession::Initialize is failed (%s)\n", result.ToString());
            return;
        }
    }

    m_IsInitialized = true;
}

//----------------------------------------------------------
void
SoundEdit::Finalize()
{
    if (m_SoundControlSession.IsInitialized())
    {
        m_SoundControlSession.Finalize();
    }
    m_SoundObjectController.Finalize();
    if (m_SoundEditSession.IsInitialized())
    {
        m_SoundEditSession.Finalize();
    }

    // 各種オブジェクトの終了処理
    if (m_SoundArchiveEditor.IsInitialized())
    {
        m_SoundArchiveEditor.Finalize();
    }

    // アロケータ終了処理
    m_Allocator.Clear();
    m_Allocator.Finalize();

    m_IsInitialized = false;
}

//----------------------------------------------------------
u32
SoundEdit::GetRequiredMemorySize(const Option& param) const
{
    u32 memorySize = 0;

#if defined( NW_USE_NINTENDO_SDK )
    memorySize += param.threadStackSize + nn::os::StackRegionAlignment;
#else
    memorySize += param.threadStackSize;
#endif
    memorySize += param.sndEditHeapSize;
    {
        nw::snd::edit::SoundEditSession::Configs configs;
        memorySize += m_SoundEditSession.GetRequiredMemorySize(configs);
    }
    {
        nw::snd::ctrl::SoundControlSession::Configs configs;
        memorySize += m_SoundControlSession.GetRequiredMemorySize(configs);
    }

    return memorySize;
}

//----------------------------------------------------------
void
SoundEdit::Start()
{
    if (!m_IsInitialized)
    {
        return;
    }

    if (IsStarted())
    {
        return;
    }

    nw::snd::edit::Result result;

    result = m_SoundEditSession.RegisterSoundArchiveEditor(&m_SoundArchiveEditor);
    NW_ASSERT( result.IsTrue() );

    result = m_SoundControlSession.RegisterSoundObjectController(&m_SoundObjectController);
    NW_ASSERT( result.IsTrue() );

    m_SoundEditSession.Open();
    m_SoundControlSession.Open();

    result = m_UpdateCachesHelper.Start(
        &m_SoundArchiveEditor,
        m_pMemoryForUpdateCachesThreadStack,
        m_UpdateCachesThreadStackSize);
    NW_ASSERT(result.IsSucceeded());

    m_IsStarted = true;
}

//----------------------------------------------------------
void
SoundEdit::Stop()
{
    if (!m_IsInitialized)
    {
        return;
    }

    if (!IsStarted())
    {
        return;
    }

    m_UpdateCachesHelper.Stop();

    m_SoundControlSession.Close();
    m_SoundEditSession.Close();

    m_SoundControlSession.UnregisterSoundObjectController(&m_SoundObjectController );
    m_SoundEditSession.UnregisterSoundArchiveEditor(&m_SoundArchiveEditor );

    m_IsStarted = false;
}

//----------------------------------------------------------
void
SoundEdit::Open()
{
    Start();
}

//----------------------------------------------------------
void
SoundEdit::Close()
{
    Stop();
}

//----------------------------------------------------------
void
SoundEdit::Update()
{
    if (!m_IsInitialized)
    {
        return;
    }

    m_SoundEditSession.Update();
    m_SoundObjectController.Update();
    m_SoundControlSession.Update();
}

} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
