﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/edit/res/sndedit_ResourceManager.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/ut/ut_Inlines.h>
#include <nw/snd/snd_DisposeCallbackManager.h>
#include <nw/snd/snd_SoundSystem.h>
#include <nw/snd/snd_Util.h>
#include <nw/snd/edit/sndedit_Config.h>

#if !defined(NW_RELEASE)
//#define NW_ENABLE_CACHE_DEBUG
#endif

namespace nw {
namespace snd {
namespace edit {
namespace internal {

namespace
{

//----------------------------------------------------------
// データバッファを無効化します。
void InvalidateDataBuffer(CacheManager::Item& item)
{
    snd::SoundSystem::LockSoundThread();

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG("[sndedit] InvalidateDataBuffer : + Dispose() Name=%s\n", item.GetName().GetName());
#endif

    // アイテムデータを利用中かもしれないので、解放する前に再生を強制停止しておきます。
    snd::internal::driver::DisposeCallbackManager::GetInstance().Dispose(
        item.GetData(),
        item.GetDataSize()
        );

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG("[sndedit] InvalidateDataBuffer : - Dispose() Name=%s\n", item.GetName().GetName());
#endif

    snd::SoundSystem::UnlockSoundThread();
}

//----------------------------------------------------------
// キャッシュを削除します。
void SafeRemoveCache(CacheManager& cacheManager, const char* name)
{
    cacheManager.RemoveItem(name, InvalidateDataBuffer);
}

//----------------------------------------------------------
// キャッシュの参照カウントを安全にデクリメントします。
void SafeDecrementReferenceCount(CacheManager& cacheManager, const char* name)
{
    CacheManager::Item* cache = cacheManager.GetItem(name);

    if(cache == NULL)
    {
        return;
    }

    cache->DecrementReferenceCount();

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG("[sndedit] SafeDecrementReferenceCount : Count=%d, Name=%s\n", cache->GetReferenceCount(), name);
#endif

    if(cache->GetReferenceCount() == 0)
    {
        cacheManager.RemoveItem(name, InvalidateDataBuffer);
    }
}

//----------------------------------------------------------
// 削除中のキャッシュアイテムかどうか調べますします。
bool IsRemovingCacheItem(CacheManager::Item& item)
{
    return item.GetUserParameter() == ResourceManager::RESOURCE_STATE_REMOVING;
}

//----------------------------------------------------------
//! @brief  キャッシュのフリーズを解除する関数オブジェクトです。
class CacheUnfreezer
{
public:
    CacheUnfreezer(ResourceManager::RESOURCE_STATE resourceState) :
      m_ResourceState(resourceState) { }

public:
    void operator()(CacheManager::Item& item)
    {
        item.SetUserParameter(m_ResourceState);
        item.Unfreeze();
    }

private:
    ResourceManager::RESOURCE_STATE m_ResourceState;
};

}

//----------------------------------------------------------
// ResourceManager::ErrorItem
//----------------------------------------------------------
ResourceManager::ErrorItem::ErrorItem()
: m_ItemName(NULL)
, m_ItemNameLength(0)
, m_Result()
, m_Reference()
{
    m_Reference.value = this;
}

ResourceManager::ErrorItem::~ErrorItem()
{
    m_ItemNameLength = 0;
    m_ItemName = NULL;
}

void ResourceManager::ErrorItem::SetItemName(const char* itemName)
{
    if (itemName == NULL)
    {
        return;
    }

    std::memcpy(m_ItemName, itemName, m_ItemNameLength);
}

char* ResourceManager::ErrorItem::AttachItemNameBuffer(char* itenNameBuffer, u32 itemNameLength)
{
    char* lastBuffer = m_ItemName;
    m_ItemName = itenNameBuffer;
    m_ItemNameLength = itemNameLength;

    return lastBuffer;
}

//----------------------------------------------------------
ResourceManager::ResourceManager() :
m_pMemoryForItemInfoManager(NULL),
m_pMemoryForFileManager(NULL),
m_pMemoryForErrotItemList(NULL),
m_AllItemsState(RESOURCE_STATE_UNKNOWN),
m_MaxName(0),
m_IsItemInfoDictionaryExpandable(false),
m_IsFileDictionaryExpandable(false)
{
}

//----------------------------------------------------------
Result
ResourceManager::Initialize(
    snd::internal::fnd::FrameHeap& heap,
    u32 baseID,
    u32 maxItems,
    u32 maxFiles,
    u32 maxName,
    bool isMaxItemsExpandable,
    bool isMaxFilesExpandable)
{
    u32 dataBufferLength = heap.GetFreeLength();
    void* dataBuffer = heap.Alloc(dataBufferLength);

    if(dataBuffer == NULL)
    {
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

#ifndef NW_PLATFORM_CTR
    NW_WARNING(
        snd::internal::Util::IsValidMemoryForDsp(dataBuffer, dataBufferLength),
        "the memory area (0x%08x - 0x%08x %dbyte) provided cross a 512 MB segment.",
        dataBuffer,
        ut::AddOffsetToPtr(dataBuffer, dataBufferLength),
        dataBufferLength );
#endif

    if(!m_DataCacheHeap.Initialize(dataBuffer, dataBufferLength))
    {
        // 拡張ヒープを利用可能な最小メモリサイズに満たない
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

    // 断片化を防ぐためにエラーアイテム用のヒープを用意
    u32 errorItemHeapSize = GetErrorItemHeapRequiredMemorySize(maxName);
    m_pMemoryForErrotItemList = m_DataCacheHeap.Alloc(errorItemHeapSize);
    if(m_pMemoryForErrotItemList == NULL)
    {
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }
    if(!m_ErrorItemHeap.Initialize(m_pMemoryForErrotItemList, errorItemHeapSize))
    {
        // 拡張ヒープを利用可能な最小メモリサイズに満たない
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

    // メモリの断片化を軽減するために、リソースサイズに応じて、
    // 拡張ヒープの割り当て先（ヒープの前から or 後ろから）を切り替える
    // アロケータを利用します。
    m_ResourceAllocator.Initialize(&m_DataCacheHeap, RESOURCE_ALLOCATION_THRESHOLD_SIZE);

    Result result = InitializeItemInfoManager(baseID, maxItems, maxName);

    if(result.IsFailed())
    {
        Finalize();
        return result;
    }

    result = InitializeFileManager(maxFiles);

    if(result.IsFailed())
    {
        Finalize();
        return result;
    }

    m_IsItemInfoDictionaryExpandable = isMaxItemsExpandable;
    m_IsFileDictionaryExpandable = isMaxFilesExpandable;

    return result;
}

//----------------------------------------------------------
void
ResourceManager::Finalize()
{
    if(m_FileManager.IsInitialized())
    {
        // 再生中のバッファを安全に解放する(InvalidateDataBuffer() する)ために、
        // 明示的に RemoveAllItemFiles() を呼び出す。
        RemoveAllItemFiles();
    }

    m_ItemInfoManager.Finalize();
    m_FileManager.Finalize();

    m_ResourceAllocator.Free(m_pMemoryForFileManager);
    m_pMemoryForFileManager = NULL;
    m_ResourceAllocator.Free(m_pMemoryForItemInfoManager);
    m_pMemoryForItemInfoManager = NULL;

    DestroyAllErrorItems();
    m_ErrorItemHeap.Finalize();
    m_DataCacheHeap.Free(m_pMemoryForErrotItemList);
    m_pMemoryForErrotItemList = NULL;

    m_ResourceAllocator.Finalize();
    m_DataCacheHeap.Finalize();

    m_IsItemInfoDictionaryExpandable = false;
    m_IsFileDictionaryExpandable = false;
}

//----------------------------------------------------------
u32
ResourceManager::GetMemoryUsage() const
{
    if(!IsInitialized())
    {
        return 0;
    }

    // アイテム情報キャッシュ管理領域のサイズ（固定長）
    u32 result = m_ItemInfoManager.GetRequiredMemorySize(m_ItemInfoManager.GetMaxItemCount());

    // アイテム情報キャッシュデータのサイズ（可変長）
    result += m_ItemInfoManager.GetMemoryUsage();

    // ファイルキャッシュ管理領域のサイズ（固定長）
    result += m_FileManager.GetRequiredMemorySize(m_FileManager.GetMaxItemCount());

    // ファイルキャッシュデータのサイズ（可変長）
    result += m_FileManager.GetMemoryUsage();

    return result;
}

//----------------------------------------------------------
bool
ResourceManager::TryGetEditItemInfo(u32 index, EditItemInfo* pItemInfo) const
{
    NW_ASSERT_NOT_NULL(pItemInfo);
    const CacheManager::Item* pItem = m_ItemInfoManager.GetItem(index);

    if(pItem == NULL)
    {
        return false;
    }

    pItemInfo->id = pItem->GetID();
    pItemInfo->name = pItem->GetName().GetName();
    pItemInfo->isOverrideOriginal = pItem->IsOverrideOriginal();

    switch(pItem->GetDataType())
    {
    case RES_DATA_TYPE_STREAM_SOUND:
        pItemInfo->itemType = EDIT_ITEM_TYPE_STREAM_SOUND;
        break;

    case RES_DATA_TYPE_WAVE_SOUND:
        pItemInfo->itemType = EDIT_ITEM_TYPE_WAVE_SOUND;
        break;

    case RES_DATA_TYPE_SEQUENCE_SOUND:
        pItemInfo->itemType = EDIT_ITEM_TYPE_SEQUENCE_SOUND;
        break;

    case RES_DATA_TYPE_BANK:
        pItemInfo->itemType = EDIT_ITEM_TYPE_BANK;
        break;

    default:
        pItemInfo->itemType = EDIT_ITEM_TYPE_UNKNOWN;
        break;
    }

    return true;
}

//----------------------------------------------------------
bool
ResourceManager::TryGetEditItemInfo(const char* name, EditItemInfo* pItemInfo) const
{
    NW_ASSERT_NOT_NULL(name);
    NW_ASSERT_NOT_NULL(pItemInfo);
    const CacheManager::Item* pItem = m_ItemInfoManager.GetItem(name);

    if(pItem == NULL)
    {
        return false;
    }

    pItemInfo->id = pItem->GetID();
    pItemInfo->name = pItem->GetName().GetName();
    pItemInfo->isOverrideOriginal = pItem->IsOverrideOriginal();

    switch(pItem->GetDataType())
    {
    case RES_DATA_TYPE_STREAM_SOUND:
        pItemInfo->itemType = EDIT_ITEM_TYPE_STREAM_SOUND;
        break;

    case RES_DATA_TYPE_WAVE_SOUND:
        pItemInfo->itemType = EDIT_ITEM_TYPE_WAVE_SOUND;
        break;

    case RES_DATA_TYPE_SEQUENCE_SOUND:
        pItemInfo->itemType = EDIT_ITEM_TYPE_SEQUENCE_SOUND;
        break;

    case RES_DATA_TYPE_BANK:
        pItemInfo->itemType = EDIT_ITEM_TYPE_BANK;
        break;

    default:
        pItemInfo->itemType = EDIT_ITEM_TYPE_UNKNOWN;
        break;
    }

    return true;
}

//----------------------------------------------------------
bool
ResourceManager::TryGetEditItemInfoFromID(u32 id, EditItemInfo* pItemInfo) const
{
    NW_ASSERT_NOT_NULL(pItemInfo);
    const CacheManager::Item* pItem = m_ItemInfoManager.GetItemFromID(id);

    if(pItem == NULL)
    {
        return false;
    }

    pItemInfo->id = pItem->GetID();
    pItemInfo->name = pItem->GetName().GetName();
    pItemInfo->isOverrideOriginal = pItem->IsOverrideOriginal();

    switch(pItem->GetDataType())
    {
    case RES_DATA_TYPE_STREAM_SOUND:
        pItemInfo->itemType = EDIT_ITEM_TYPE_STREAM_SOUND;
        break;

    case RES_DATA_TYPE_WAVE_SOUND:
        pItemInfo->itemType = EDIT_ITEM_TYPE_WAVE_SOUND;
        break;

    case RES_DATA_TYPE_SEQUENCE_SOUND:
        pItemInfo->itemType = EDIT_ITEM_TYPE_SEQUENCE_SOUND;
        break;

    case RES_DATA_TYPE_BANK:
        pItemInfo->itemType = EDIT_ITEM_TYPE_BANK;
        break;

    default:
        pItemInfo->itemType = EDIT_ITEM_TYPE_UNKNOWN;
        break;
    }

    return true;
}

//----------------------------------------------------------
bool
ResourceManager::ContainsItemInfo(const char* name) const
{
    NW_ASSERT_NOT_NULL(name);
    NW_ASSERT(*name != '\0');

    return m_ItemInfoManager.GetItem(name) != NULL;
}

//----------------------------------------------------------
bool
ResourceManager::TryGetItemID(const char* name, u32* id) const
{
    NW_ASSERT_NOT_NULL(name);
    NW_ASSERT(*name != '\0');

    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    const CacheManager::Item* item =m_ItemInfoManager.GetItem(name);

    if(item == NULL)
    {
        return false;
    }

    if(id != NULL)
    {
        *id = item->GetID();
    }

    return true;
}

//----------------------------------------------------------
CacheState
ResourceManager::GetItemCacheState(const char* name) const
{
    NW_ASSERT_NOT_NULL(name);
    NW_ASSERT(*name != '\0');

    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    const CacheManager::Item* item =m_ItemInfoManager.GetItem(name);

    if(item == NULL)
    {
        return CACHE_STATE_NONE;
    }

    return item->IsFreezed() ? CACHE_STATE_CACHED : CACHE_STATE_CACHING;
}

//----------------------------------------------------------
const void*
ResourceManager::GetItemInfo(
    const char* soundName,
    ResDataType* itemInfoDataType /*= NULL*/,
    bool allowUnfreezedData /*= false*/) const
{
    NW_ASSERT_NOT_NULL(soundName);
    NW_ASSERT(*soundName != '\0');

    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    u32 dataType = RES_DATA_TYPE_UNKNOWN;
    const void* result = m_ItemInfoManager.GetData(soundName, &dataType, allowUnfreezedData);

    if(itemInfoDataType != NULL)
    {
        *itemInfoDataType = static_cast<ResDataType>(dataType);
    }

    return result;
}

//----------------------------------------------------------
ResDataType
ResourceManager::GetItemInfoDataType(const char* itemName, Result* result) const
{
    NW_ASSERT_NOT_NULL(itemName);
    NW_ASSERT(*itemName != '\0');

    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    const CacheManager::Item* item = m_ItemInfoManager.GetItem(itemName);

    if(item == NULL)
    {
        if(result != NULL)
        {
            *result = Result(SNDEDIT_RESULT_NAME_NOT_FOUND);
        }

        return RES_DATA_TYPE_UNKNOWN;
    }

    if(result != NULL)
    {
        *result = Result(SNDEDIT_RESULT_TRUE);
    }

    return static_cast<ResDataType>(item->GetDataType());
}

//----------------------------------------------------------
Result
ResourceManager::SetItemInfo(
    const char* itemName,
    u32 originalId,
    ResDataType itemType,
    const void* itemInfo)
{
    NW_ASSERT_NOT_NULL(itemName);
    NW_ASSERT_NOT_NULL(itemInfo);

    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    Result result = Result(SNDEDIT_RESULT_FAILED);
    u32 dataSize = 0;

    switch(itemType)
    {
    case RES_DATA_TYPE_STREAM_SOUND:
        dataSize = ResStreamSoundInfo::GetRequiredSize(m_MaxName, Limits::MaxFilePath);
        break;

    case RES_DATA_TYPE_WAVE_SOUND:
        dataSize = ResWaveSoundInfo::GetRequiredSize(m_MaxName, Limits::MaxFilePath);
        break;

    case RES_DATA_TYPE_SEQUENCE_SOUND:
        dataSize = ResSequenceSoundInfo::GetRequiredSize(m_MaxName, Limits::MaxFilePath);
        break;

    case RES_DATA_TYPE_BANK:
        dataSize = ResBankInfo::GetRequiredSize(Limits::MaxFilePath);
        break;

    default:
        return Result(SNDEDIT_RESULT_FAILED);
    }

    CacheManager::Item* item = m_ItemInfoManager.GetItem(itemName);

    // 既存のキャッシュがある場合は、データを上書きします。
    if(item != NULL)
    {
        u32 cacheItemType = item->GetDataType();

        // 違う種類のデータをキャッシュすることはないようにする
        if(cacheItemType != RES_DATA_TYPE_UNKNOWN &&
            cacheItemType != static_cast<u32>(itemType))
        {
            NW_FATAL_ERROR("[sndedit] invalid item data type.\n");
            return Result(SNDEDIT_RESULT_FAILED);
        }

        void* itemData = item->GetData();
        NW_NULL_ASSERT(itemData);
        std::memcpy(itemData, itemInfo, dataSize);
        item->SetDataType(itemType);

        return Result(SNDEDIT_RESULT_TRUE);
    }

    bool isOverrideOriginal = originalId != SoundArchive::INVALID_ID;
    u32 id;
    if (isOverrideOriginal)
    {
        id = originalId;
    }
    else
    {
        if(!m_ItemInfoManager.TryGetFreeID(&id))
        {
            return Result(SNDEDIT_RESULT_CACHE_OVER_FLOW);
        }
    }

    return m_ItemInfoManager.AddItem(id, itemName, isOverrideOriginal, itemType, itemInfo, dataSize);
}

//----------------------------------------------------------
const void*
ResourceManager::GetFile(
    const char* filePath,
    ResDataType fileDataType,
    const Hash32** hashCode /*= NULL*/) const
{
    NW_ASSERT_NOT_NULL(filePath);
    NW_ASSERT(*filePath != '\0');

    NW_ASSERTMSG(IsInitialized(), "SoundArchiveFileEditor is not initialized.\n");

    // 該当キャッシュがあれば、それを返します。
    const CacheManager::Item* cache = m_FileManager.GetItem(filePath);

    if(cache == NULL)
    {
        return NULL;
    }

    if(cache->GetDataType() != static_cast<u32>(fileDataType))
    {
        return NULL;
    }

    if(hashCode != NULL)
    {
        *hashCode = &cache->GetHashCode();
    }

    // TODO : ★リファクタリング候補 : キャッシュクラスの機能とする
    // ファイルデータはアライメント調整されてキャッシュされているので、
    // ここで位置を合わせて返します。
    return ut::RoundUp(cache->GetData(), Alignments::FILE_IO_BUFFER);
}

//----------------------------------------------------------
Result
ResourceManager::IncrementFileReferenceCount(const char* filePath)
{
    NW_ASSERT_NOT_NULL(filePath);
    NW_ASSERT(*filePath != '\0');

    CacheManager::Item* cache = m_FileManager.GetItem(filePath);

    if(cache == NULL)
    {
        return Result(SNDEDIT_RESULT_NAME_NOT_FOUND);
    }

    cache->IncrementReferenceCount();

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG("[sndedit] ResourceManager::IncrementFileReferenceCount : Count=%d, Name=%s\n", cache->GetReferenceCount(), filePath);
#endif

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
Result
ResourceManager::DecrementFileReferenceCount(const char* filePath)
{
    NW_ASSERT_NOT_NULL(filePath);
    NW_ASSERT(*filePath != '\0');

    CacheManager::Item* cache = m_FileManager.GetItem(filePath);

    if(cache == NULL)
    {
        return Result(SNDEDIT_RESULT_NAME_NOT_FOUND);
    }

    cache->DecrementReferenceCount();

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG("[sndedit] ResourceManager::DecrementReferenceCount : Count=%d, Name=%s\n", cache->GetReferenceCount(), filePath);
#endif

    if(cache->GetReferenceCount() == 0)
    {
        RemoveFile(filePath);
    }

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
Result
ResourceManager::NewItemInfoEntry(const char* name, u32 originalId)
{
    NW_ASSERT_NOT_NULL(name);

    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    // アイテム数が上限に達した場合は、辞書を拡張します。
    if(m_ItemInfoManager.GetItemCount() == m_ItemInfoManager.GetMaxItemCount())
    {
        u32 newMaxItems = m_ItemInfoManager.GetItemCount() * 2;

        if(newMaxItems == 0)
        {
            newMaxItems = 32;
        }

        Result result = RebuildItemInfoDictionary(newMaxItems);

        if(result.IsFailed())
        {
            return result;
        }
        else if(result.IsFalse())
        {
            return Result(SNDEDIT_RESULT_CACHE_OVER_FLOW);
        }
    }

    bool isOverrideOriginal = originalId != SoundArchive::INVALID_ID;
    u32 id;
    if (isOverrideOriginal)
    {
        id = originalId;
    }
    else
    {
        if(!m_ItemInfoManager.TryGetFreeID(&id))
        {
            return Result(SNDEDIT_RESULT_CACHE_OVER_FLOW);
        }
    }

    CacheManager::Item* newCache = NULL;

    Result result = m_ItemInfoManager.CreateItemEntry(
        newCache,
        id,
        name,
        isOverrideOriginal,
        ResItemInfoUtility::GetMaxItemInfoSize(m_MaxName, Limits::MaxFilePath));

    if(result.IsFailed())
    {
        return result;
    }

    result = m_ItemInfoManager.AddItem(*newCache);

    if(result.IsFailed())
    {
        newCache->Dispose();
        return result;
    }

    newCache->SetUserParameter(RESOURCE_STATE_UPDATING);

    return result;
}

//----------------------------------------------------------
Result
ResourceManager::NewFileEntry(
    const char* filePath,
    u32 fileSize,
    ResDataType fileDataType,
    void** filebuffer,
    Hash32* hashCode /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(filePath);
    NW_ASSERT(*filePath != '\0');

    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    // ファイル数が上限に達した場合は、辞書を拡張します。
    if(m_FileManager.GetItemCount() == m_FileManager.GetMaxItemCount())
    {
        u32 newMaxFiles = m_FileManager.GetItemCount() * 2;

        if(newMaxFiles == 0)
        {
            newMaxFiles = 32;
        }

        Result result = RebuildFileDictionary(newMaxFiles);

        if(result.IsFailed())
        {
            return result;
        }
        else if(result.IsFalse())
        {
            return Result(SNDEDIT_RESULT_CACHE_OVER_FLOW);
        }
    }

    CacheManager::Item* newCache = NULL;

    Result result = m_FileManager.CreateItemEntry(newCache, 0, filePath, false, fileSize);

    // メモリ不足なら、未使用ファイルを削除（ガベージコレクション）して再試行します。
    if(result == SNDEDIT_RESULT_OUT_OF_MEMORY)
    {
        RemoveAllGarbageItemFiles();

        result = m_FileManager.CreateItemEntry(newCache, 0, filePath, false, fileSize);
    }

    if(result.IsFailed())
    {
        return result;
    }

    newCache->SetDataType(fileDataType);

    if(hashCode != NULL)
    {
        newCache->SetHashCode(*hashCode);
    }

    result = m_FileManager.AddItem(*newCache);

    if(result.IsFailed())
    {
        newCache->Dispose();
        return result;
    }

    if(filebuffer != NULL)
    {
        *filebuffer = newCache->GetData();
    }

    return result;
}

//----------------------------------------------------------
void
ResourceManager::RemoveAllItemInfos()
{
    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    m_ItemInfoManager.RemoveAllItems();

    if(m_IsItemInfoDictionaryExpandable)
    {
        RebuildItemInfoDictionary(0);
    }
    else
    {
        UpdateItemInfoDictionary();
    }
}

//----------------------------------------------------------
void
ResourceManager::RemoveAllItemFiles()
{
    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    snd::SoundSystem::LockSoundThread();
    m_FileManager.RemoveAllItems(InvalidateDataBuffer);
    snd::SoundSystem::UnlockSoundThread();

    if(m_IsItemInfoDictionaryExpandable)
    {
        RebuildFileDictionary(0);
    }
    else
    {
        UpdateFileDictionary();
    }
}

//----------------------------------------------------------
void
ResourceManager::RemoveAllGarbageItemFiles()
{
    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    snd::SoundSystem::LockSoundThread();
    m_FileManager.RemoveGarbages();
    snd::SoundSystem::UnlockSoundThread();

    if(m_IsItemInfoDictionaryExpandable)
    {
        RebuildFileDictionary(0);
    }
    else
    {
        UpdateFileDictionary();
    }
}

//----------------------------------------------------------
void
ResourceManager::ClearAllItemFileReferenceCounts()
{
    m_FileManager.ClearAllItemReferenceCounts();
}

//----------------------------------------------------------
Result
ResourceManager::RemoveItemData(const char* name)
{
    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    if(name == NULL || name[0] == '\0')
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    // 関連するファイルキャッシュの参照カウントをデクリメントします。
    DecrementReferenceCountForItemFiles(name);

    // アイテム情報キャッシュを削除します。
    Result result = m_ItemInfoManager.RemoveItem(name);

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG(
        "[sndedit] ResourceManager::RemoveItemData() : result=%s, name=%s\n",
        result.ToString(),
        name);
#endif

    return result;
}

//----------------------------------------------------------
Result
ResourceManager::RemoveItemFiles(const char* name)
{
    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    if(name == NULL || name[0] == '\0')
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    // 関連するファイルキャッシュを削除します。
    Result result = ForeachFileCache(name, SafeRemoveCache);

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG(
        "[sndedit] ResourceManager::RemoveItemFiles() : result=%s, name=%s\n",
        result.ToString(),
        name);
#endif

    return result;
}

//----------------------------------------------------------
Result
ResourceManager::DecrementReferenceCountForItemFiles(const char* name)
{
    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    if(name == NULL || name[0] == '\0')
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    // 関連するファイルキャッシュの参照カウントをデクリメントします。
    Result result = ForeachFileCache(name, SafeDecrementReferenceCount);

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG(
        "[sndedit] ResourceManager::DecrementReferenceCountForItemFiles() : result=%s, name=%s\n",
        result.ToString(),
        name);
#endif

    return result;
}

//----------------------------------------------------------
void
ResourceManager::RemoveFile(const char* filePath)
{
    NW_ASSERT_NOT_NULL(filePath);
    NW_ASSERT(*filePath != '\0');

    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    m_FileManager.RemoveItem(filePath, InvalidateDataBuffer);
}

//----------------------------------------------------------
const char*
ResourceManager::GetFirstUnfreezedItemName(RESOURCE_STATE* state /*= NULL*/) const
{
    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    // メモリ確保を優先するために削除中のアイテムを先に探します。
    const CacheManager::Item* item = m_ItemInfoManager.GetFirstUnfreezedItem(IsRemovingCacheItem);

    if(item != NULL)
    {
        if(state != NULL)
        {
            *state = static_cast<RESOURCE_STATE>(item->GetUserParameter());
        }

        return item->GetName().GetName();
    }

    item = m_ItemInfoManager.GetFirstUnfreezedItem();

    if(item == NULL)
    {
        if(state != NULL)
        {
            *state = RESOURCE_STATE_UNKNOWN;
        }

        return NULL;
    }

    if(state != NULL)
    {
        *state = static_cast<RESOURCE_STATE>(item->GetUserParameter());
    }

    return item->GetName().GetName();
}

//----------------------------------------------------------
Result
ResourceManager::FreezeItemInfo(const char* itemName, RESOURCE_STATE* state /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(itemName);
    NW_ASSERT(*itemName != '\0');

    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    CacheManager::Item* item = m_ItemInfoManager.GetItem(itemName);

    if(item == NULL)
    {
        if(state != NULL)
        {
            *state = RESOURCE_STATE_UNKNOWN;
        }

        return Result(SNDEDIT_RESULT_NAME_NOT_FOUND);
    }

    item->Freeze();

    if(state != NULL)
    {
        *state = static_cast<RESOURCE_STATE>(item->GetUserParameter());
    }

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
Result
ResourceManager::UnfreezeAllItemInfos(RESOURCE_STATE state)
{
    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    CacheUnfreezer unfreezer(state);
    m_ItemInfoManager.ForEach(unfreezer);

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
Result
ResourceManager::UnfreezeItemInfo(const char* itemName, RESOURCE_STATE state)
{
    NW_ASSERT_NOT_NULL(itemName);
    NW_ASSERT(*itemName != '\0');

    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    CacheManager::Item* item = m_ItemInfoManager.GetItem(itemName);

    if(item == NULL)
    {
        return Result(SNDEDIT_RESULT_NAME_NOT_FOUND);
    }

    item->SetUserParameter(state);
    item->Unfreeze();

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
ResourceManager::UpdateItemInfoDictionary()
{
    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    m_ItemInfoManager.UpdateDictionary();
}

//----------------------------------------------------------
void
ResourceManager::UpdateFileDictionary()
{
    NW_ASSERTMSG(IsInitialized(), "ResourceManager is not initialized.\n");

    m_FileManager.UpdateDictionary();
}

ResourceManager::ErrorItem* ResourceManager::CreateErrorItem(const char* itemName, nw::snd::edit::Result result)
{
    u32 itemLength = sizeof(char) * (std::strlen(itemName) + 1);

    void* itemNameBuffer = m_ErrorItemHeap.Alloc(itemLength);
    void* memory = m_ErrorItemHeap.Alloc(sizeof(ErrorItem));
    if (itemNameBuffer == NULL || memory == NULL)
    {
        return NULL;
    }

    ErrorItem* item = new (memory) ErrorItem();
    char* lastBuffer = item->AttachItemNameBuffer(static_cast<char*>(itemNameBuffer), itemLength);
    if (lastBuffer != NULL)
    {
        m_ErrorItemHeap.Free(lastBuffer);
    }

    item->SetResult(result);
    item->SetItemName(itemName);

    return item;
}

void ResourceManager::DestroyErrorItem(ErrorItem* item)
{
    char* lastBuffer = item->AttachItemNameBuffer(NULL, 0);
    if (lastBuffer != NULL)
    {
        m_ErrorItemHeap.Free(lastBuffer);
    }
    item->~ErrorItem();
    m_ErrorItemHeap.Free(item);
    item = NULL;
}

void ResourceManager::DestroyAllErrorItems()
{
    ErrorItem* errorItem = PopErrorItem();
    while (errorItem != NULL)
    {
        DestroyErrorItem(errorItem);
        errorItem = PopErrorItem();
    }
}

void ResourceManager::PushErrorItem(ErrorItem* item)
{
    m_ErrorItemList.PushBack(&item->GetReference());
}

ResourceManager::ErrorItem* ResourceManager::PopErrorItem()
{
    if (m_ErrorItemList.empty())
    {
        return NULL;
    }

    ErrorItem* item = m_ErrorItemList.begin()->value;
    m_ErrorItemList.PopFront();
    return item;
}

u32 ResourceManager::GetErrorItemHeapRequiredMemorySize(u32 maxName) const
{
    // ErrorItemのサイズ + 文字列バッファのサイズ + 1 ブロックの管理領域サイズ
    u32 oneItemSize = ut::RoundUp(
        sizeof(ErrorItem) + ( sizeof(char) * ( maxName + 1 ) ) + snd::internal::fnd::ExpandedHeap::BLOCK_MD_SIZE,
        snd::internal::fnd::MemoryTraits::DEFAULT_ALIGNMENT
    );

    // ヒープ全体の管理領域のサイズ + 1 ErrorItem が使用する最大メモリサイズ * 同時に処理可能なエラー数
    return snd::internal::fnd::ExpandedHeap::HEAP_MD_SIZE + oneItemSize * 64;
}

//----------------------------------------------------------
u32
ResourceManager::GetItemInfoManagerRequiredMemorySize(u32 maxItems, u32 maxFiles, u32 maxName) const
{
    (void)maxFiles;
    u32 result = m_ItemInfoManager.GetRequiredMemorySize(maxItems);

    result += GetItemInfoDataBufferLength(
        maxItems,
        maxName,
        ResItemInfoUtility::GetMaxItemInfoSize(maxName, Limits::MaxFilePath));

    return result;
}

//----------------------------------------------------------
Result
ResourceManager::InitializeItemInfoManager(u32 baseID, u32 maxEntries, u32 maxName)
{
    u32 bufferLength = m_ItemInfoManager.GetRequiredMemorySize(maxEntries);
    m_pMemoryForItemInfoManager = bufferLength == 0 ? NULL : m_ResourceAllocator.Alloc(bufferLength);

    if(bufferLength > 0 && m_pMemoryForItemInfoManager == NULL)
    {
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

    CacheManager::InitializeArgs args;
    args.buffer = m_pMemoryForItemInfoManager;
    args.bufferLength = bufferLength;
    args.dataBufferAllocator = &m_ResourceAllocator;
    args.baseID = baseID;
    args.maxEntries = maxEntries;
    args.maxName = maxName;

    return m_ItemInfoManager.Initialize(args);
}

//----------------------------------------------------------
Result
ResourceManager::InitializeFileManager(u32 maxEntries)
{
    u32   bufferLength = m_FileManager.GetRequiredMemorySize(maxEntries);
    m_pMemoryForFileManager = bufferLength == 0 ? NULL : m_ResourceAllocator.Alloc(bufferLength);

    if(bufferLength > 0 && m_pMemoryForFileManager == NULL)
    {
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

    CacheManager::InitializeArgs args;
    args.buffer = m_pMemoryForFileManager;
    args.bufferLength = bufferLength;
    args.dataBufferAllocator = &m_ResourceAllocator;
    args.maxEntries = maxEntries;
    args.maxName = Limits::MaxFilePath;

    return m_FileManager.Initialize(args);
}

//----------------------------------------------------------
Result
ResourceManager::RebuildItemInfoDictionary(u32 maxEntries)
{
    if(!m_IsItemInfoDictionaryExpandable)
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    if(!m_ItemInfoManager.IsInitialized())
    {
        NW_FATAL_ERROR("ItemInfoManager is not initialized.\n");
        return Result(SNDEDIT_RESULT_NOT_INITIALIZED);
    }

    if (maxEntries > m_ItemInfoManager.GetMaxItemCount())
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    u32   bufferLength = m_ItemInfoManager.GetRequiredMemorySize(maxEntries);
    void* buffer = m_ResourceAllocator.Alloc(bufferLength);

    if(buffer == NULL)
    {
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

    void* oldBuffer = NULL;
    Result result = m_ItemInfoManager.RebuildDictionary(buffer, bufferLength, maxEntries, &oldBuffer);

    if(result.IsSucceeded() && oldBuffer != NULL)
    {
        m_ResourceAllocator.Free(oldBuffer);
    }

    return result;
}

//----------------------------------------------------------
Result
ResourceManager::RebuildFileDictionary(u32 maxEntries)
{
    if(!m_IsFileDictionaryExpandable)
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    if(!m_FileManager.IsInitialized())
    {
        NW_FATAL_ERROR("FileManager is not initialized.\n");
        return Result(SNDEDIT_RESULT_NOT_INITIALIZED);
    }

    if (maxEntries > m_FileManager.GetMaxItemCount())
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    u32   bufferLength = m_FileManager.GetRequiredMemorySize(maxEntries);
    void* buffer = m_ResourceAllocator.Alloc(bufferLength);

    if(buffer == NULL)
    {
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

    void* oldBuffer = NULL;
    Result result = m_FileManager.RebuildDictionary(buffer, bufferLength, maxEntries, &oldBuffer);

    if(result.IsSucceeded() && oldBuffer != NULL)
    {
        m_ResourceAllocator.Free(oldBuffer);
    }

    return result;
}

//----------------------------------------------------------
u32
ResourceManager::GetItemInfoDataBufferLength(u32 maxEntries, u32 maxName, u32 maxDataLength) const
{
    // キャッシュサイズ × エントリー数
    return
        m_ItemInfoManager.GetRequiredMemorySizeForItem(maxName, maxDataLength) * maxEntries;
}

//----------------------------------------------------------
Result
ResourceManager::ForeachFileCache(const char* name, CACHE_ACTION action)
{
    NW_ASSERT_NOT_NULL(name);
    NW_ASSERT_NOT_NULL(action);

    u32 cacheDataType = RES_DATA_TYPE_UNKNOWN;

    const void* itemInfo = m_ItemInfoManager.GetData(name, &cacheDataType, true);

    if(itemInfo == NULL)
    {
#if defined(NW_ENABLE_CACHE_DEBUG)
        NW_LOG(
            "[sndedit] ResourceManager::ForeachFileCache() : itemInfo not found : name=%s\n",
            name);
#endif
        return Result(SNDEDIT_RESULT_FALSE);
    }

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG("[sndedit] ResourceManager::ForeachFileCache() : name=%s\n", name);
#endif

    ResDataType itemDataType = static_cast<ResDataType>(cacheDataType);

    switch(itemDataType)
    {
    case RES_DATA_TYPE_WAVE_SOUND:
        {
            const ResSoundInfo* soundInfo = reinterpret_cast<const ResSoundInfo*>(itemInfo);

            const ResName* wsdFilePath =
                ResItemInfoUtility::GetFilePath(*soundInfo, itemDataType, RES_DATA_TYPE_WAVE_SOUND_FILE);

            const ResName* warcFilePath =
                ResItemInfoUtility::GetFilePath(*soundInfo, itemDataType, RES_DATA_TYPE_WAVE_ARCHIVE_FILE);

            if(wsdFilePath != NULL)
            {
                action(m_FileManager, wsdFilePath->GetName());
            }

            if(warcFilePath != NULL)
            {
                action(m_FileManager, warcFilePath->GetName());
            }
        }
        break;

    case RES_DATA_TYPE_SEQUENCE_SOUND:
        {
            const ResSoundInfo* soundInfo = reinterpret_cast<const ResSoundInfo*>(itemInfo);

            const ResName* seqFilePath =
                ResItemInfoUtility::GetFilePath(*soundInfo, itemDataType, RES_DATA_TYPE_SEQUENCE_FILE);

            if(seqFilePath != NULL)
            {
                action(m_FileManager, seqFilePath->GetName());
            }
        }
        break;

    case RES_DATA_TYPE_BANK:
        {
            const ResBankInfo* bankInfo = reinterpret_cast<const ResBankInfo*>(itemInfo);

            const ResName* bankFilePath =
                ResItemInfoUtility::GetFilePath(*bankInfo, RES_DATA_TYPE_BANK_FILE);

            const ResName* warcFilePath =
                ResItemInfoUtility::GetFilePath(*bankInfo, RES_DATA_TYPE_WAVE_ARCHIVE_FILE);

            if(bankFilePath != NULL)
            {
                action(m_FileManager, bankFilePath->GetName());
            }

            if(warcFilePath != NULL)
            {
                action(m_FileManager, warcFilePath->GetName());
            }
        }
        break;

#if defined(NW_ENABLE_CACHE_DEBUG)
    case RES_DATA_TYPE_STREAM_SOUND:
        NW_LOG(
            "[sndedit] ResourceManager::ForeachFileCache() : skip stream sound : name=%s\n",
            name);
        break;
#endif

    default:
        NW_WARNING(
            "[sndedit] ResourceManager::ForeachFileCache() : unknown sound type : name=%s\n",
            name);
    }

    return Result(SNDEDIT_RESULT_TRUE);
}

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
