﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/edit/hio/sndedit_HioChannel.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/edit/hio/sndedit_HioManager.h>
#include <nw/snd/edit/hio/sndedit_HioProtocol.h>

namespace nw {
namespace snd {
namespace edit {
namespace internal {

//----------------------------------------------------------
HioChannel::HioChannel() :
m_RecvStreamBufferLength(0),
m_RecvPacketBufferLength(0)
{
    m_Reference.value = this;
}

//----------------------------------------------------------
HioChannel::~HioChannel()
{
    Finalize();
}

//----------------------------------------------------------
#if defined(NW_SND_EDIT_USE_MCS)
HioResult
HioChannel::Initialize(
    void* recvStreamBuffer,
    u32 recvStreamBufferLength,
    void* recvPacketBuffer,
    u32 recvPacketBufferLength)
#else
HioResult
HioChannel::Initialize(
    void* recvStreamBuffer,
    u32 recvStreamBufferLength,
    void* recvPacketBuffer,
    u32 recvPacketBufferLength,
    void* workBuffer,
    u32 workBufferLength
)
#endif
{
    if(IsInitialized())
    {
        NW_FATAL_ERROR("HioChannel is already initialized.\n");
        return HioResult(HIO_RESULT_FAILED);
    }

    if(recvStreamBuffer == NULL ||
        recvPacketBuffer == NULL ||
        recvStreamBufferLength == 0 ||
        recvPacketBufferLength == 0)
    {
        NW_FATAL_ERROR("invalid arguments.\n");
        return HioResult(HIO_RESULT_FAILED);
    }

    m_RecvStreamBuffer = recvStreamBuffer;
    m_RecvPacketBuffer = reinterpret_cast<HioPacket*>(recvPacketBuffer);
    m_RecvStreamBufferLength = recvStreamBufferLength;
    m_RecvPacketBufferLength = recvPacketBufferLength;

#if !defined(NW_SND_EDIT_USE_MCS)
    m_Stream.Initialize(workBuffer, workBufferLength);
#endif

    HioResult result = m_PacketStream.Initialize(m_Stream, m_RecvPacketBuffer, m_RecvPacketBufferLength);

    if(!result.IsTrue())
    {
        Finalize();
        return result;
    }

    return HioResult(HIO_RESULT_TRUE);
}

//----------------------------------------------------------
void
HioChannel::Finalize()
{
    Close();

    m_PacketStream.Finalize();

#if !defined(NW_SND_EDIT_USE_MCS)
    m_Stream.Finalize();
#endif

    m_RecvStreamBuffer = NULL;
    m_RecvPacketBuffer = NULL;
    m_RecvStreamBufferLength = 0;
    m_RecvPacketBufferLength = 0;
}

//----------------------------------------------------------
bool
HioChannel::Open(HioStream::ChannelType channel)
{
    if(!IsInitialized())
    {
        return false;
    }

    return m_Stream.Open(channel, m_RecvStreamBuffer, m_RecvStreamBufferLength);
}

//----------------------------------------------------------
void
HioChannel::Close()
{
    if(!IsInitialized())
    {
        return;
    }

    if(m_Stream.IsAvailable())
    {
        m_Stream.Close();
    }
}

//----------------------------------------------------------
void
HioChannel::ClearBuffer()
{
    if(!IsInitialized() || !m_Stream.IsAvailable())
    {
        return;
    }

    m_Stream.Skip(m_Stream.GetReadableBytes());
    m_PacketStream.ClearBuffer();
}

//----------------------------------------------------------
bool
HioChannel::Polling()
{
    if (!this->IsAsync())
    {
        // HioSyncChannelが切断を検知するタイミングがないので、バッファ読み込みを実行するようにしておく
        m_Stream.GetReadableBytes();
    }

    return m_Stream.Polling();
}

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
