﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/edit/hio/sndedit_HioAsyncChannel.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/edit/hio/sndedit_HioProtocol.h>

#if !defined(NW_RELEASE)
//#define NW_ENABLE_COM_DEBUG
#endif

namespace nw {
namespace snd {
namespace edit {
namespace internal {

//----------------------------------------------------------
void
HioAsyncChannel::RegisterMessageHandler(HioMessageHandler& handler)
{
    m_MessageHandlerRefs.push_back(&handler.GetReference());
}

//----------------------------------------------------------
void
HioAsyncChannel::UnregisterMessageHandler(HioMessageHandler& handler)
{
    // HACK: 多重 erase しないようにするための保護処理
    if(handler.GetReference().node.GetNext() == NULL &&
        handler.GetReference().node.GetPrev() == NULL)
    {
        return;
    }

    m_MessageHandlerRefs.erase(&handler.GetReference());
}

//----------------------------------------------------------
HioResult
HioAsyncChannel::Update()
{
    if(!IsInitialized())
    {
        return HioResult(HIO_RESULT_TARGET_NOT_INITIALIZED);
    }

    if(!GetStream().IsAvailable())
    {
        return HioResult(HIO_RESULT_TARGET_RECIEVE_ERROR);
    }

    // メッセージヘッダを読み込みます。
    if(GetStream().CanReadNewPacket())
    {
        HioResult result = GetStream().TryReadHeader();

        if(!result.IsTrue())
        {
            return result;
        }
    }

    HioPacketHeader& header = GetRecvPacketBuffer()->GetHeader();

    if(!header.IsValid())
    {
        NW_LOG("[nw::snd::edit::internal::HioAsyncChannel] internal error : receive invalid packet.\n");
        return HioResult(HIO_RESULT_TARGET_RECIEVE_ERROR);
    }

    HioMessageHandler* handler = GetMessageHandler(header);

    if(handler == NULL)
    {
        NW_WARNING(false, "unknown HIO message. : ID=0x%08x, Size=%d", u32(header.message), u32(header.bodySize));

        // 処理できないメッセージをスキップします。
        if(!GetStream().SkipBody().IsTrue())
        {
            NW_WARNING(false, "failed to Skip().");
            return HioResult(HIO_RESULT_TARGET_RECIEVE_ERROR);
        }

        return HioResult(HIO_RESULT_TRUE);
    }

#if defined(NW_ENABLE_COM_DEBUG)
    NW_LOG(
        "[nw::snd::edit::internal::HioAsyncChannel] receive packet : message=%s(0x%08x), Size=%d, BodySize=%d.\n",
        header.message.ToString(),
        u32(header.message),
        sizeof(HioPacketHeader) + header.bodySize,
        static_cast<u32>(header.bodySize));
#endif

    return handler->Invoke(GetStream());
}

//----------------------------------------------------------
HioResult
HioAsyncChannel::Send(const HioPacket& message)
{
    if(!IsInitialized())
    {
        return HioResult(HIO_RESULT_TARGET_NOT_INITIALIZED);
    }

    // メッセージを送信します。
    return GetStream().WritePacket(message);
}

//----------------------------------------------------------
HioMessageHandler*
HioAsyncChannel::GetMessageHandler(const HioPacketHeader& header) const
{
    for(HioMessageHandlerRefList::const_iterator it = m_MessageHandlerRefs.begin();
        it != m_MessageHandlerRefs.end();
        ++it)
    {
        NW_ASSERT_NOT_NULL(it->value);

        if(it->value->CanRespond(header))
        {
            return it->value;
        }
    }

    return NULL;
}

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
