﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/ctrl/sndctrl_SoundControlSession.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/ut/ut_Inlines.h>
#include <nw/snd/fnd/basis/sndfnd_Memory.h>
#include <nw/snd/edit/sndedit_IErrorProvider.h>
#include <nw/snd/edit/hio/sndedit_HioAsyncChannel.h>
#include <nw/snd/ctrl/sndctrl_SoundObjectController.h>
#include <nw/snd/ctrl/protocol/sndctrl_PlaySoundPacket.h>
#include <nw/snd/ctrl/protocol/sndctrl_StopSoundPacket.h>
#include <nw/snd/ctrl/protocol/sndctrl_PauseSoundPacket.h>
#include <nw/snd/ctrl/protocol/sndctrl_UpdateSoundInfoPacket.h>

#if !defined(NW_RELEASE)
//#define NW_ENABLE_COM_DEBUG
#endif

namespace nw {
namespace snd {
namespace ctrl {

//----------------------------------------------------------
edit::Result
SoundControlSession::Initialize(
    void* buffer,
    u32 bufferLength,
    const Configs& configs)
{
    if(buffer == NULL ||
        bufferLength == 0 ||
        configs.channelStreamBufferSize == 0 ||
        configs.syncTimeout == 0 ||
        configs.maxItemName == 0)
    {
        NW_FATAL_ERROR("invalid arguments.\n");
        return edit::Result(edit::SNDEDIT_RESULT_FAILED);
    }

    //----------------------------------------------------------
#if !defined(NW_RELEASE)
    // バッファサイズをチェックします。
    if(bufferLength < GetRequiredMemorySize(configs))
    {
        NW_FATAL_ERROR("bufferLength too small.\n");
        return edit::Result(edit::SNDEDIT_RESULT_FAILED);
    }
#endif // !defined(NW_RELEASE)
    //----------------------------------------------------------

    if(IsInitialized())
    {
        return edit::Result(edit::SNDEDIT_RESULT_FALSE);
    }

    m_Port = configs.port;

    if(m_Channel.IsInitialized() ||
        m_Channel.IsOpened())
    {
        return edit::Result(edit::SNDEDIT_RESULT_FAILED);
    }

    if(!m_HioManager.Initialize())
    {
        return edit::Result(edit::SNDEDIT_RESULT_FAILED);
    }

    snd::internal::fnd::FrameHeap allocator;
    allocator.Initialize(buffer, bufferLength);

    edit::Result result = InitializeChannel(
        allocator,
        configs.channelStreamBufferSize,
        GetChannelRecvPacketBufferSize(configs.maxItemName));

    if(result.IsFailed())
    {
        Finalize();
        return result;
    }

    m_SyncTimeout = configs.syncTimeout;
    m_State = STATE_INITIALIZED;

    return edit::Result(edit::SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
u32
SoundControlSession::GetRequiredMemorySize(const Configs& configs) const
{
    // チャンネルの受信用ストリームバッファサイズ
    u32 result = ut::RoundUp(
        configs.channelStreamBufferSize,
        snd::internal::fnd::MemoryTraits::DEFAULT_ALIGNMENT
        );

    // チャンネルの受信用パケットバッファサイズ
    result += GetChannelRecvPacketBufferSize(configs.maxItemName);

#if !defined(NW_SND_EDIT_USE_MCS)
    result += edit::internal::HioChannel::DEFAULT_WORK_BUFFER_LENGTH;
#endif

    return result;
}

//----------------------------------------------------------
u32
SoundControlSession::GetRequiredWorkBufferSize() const
{
#if defined(NW_SND_EDIT_USE_MCS)
    return 0;
#else
    return edit::internal::HioChannel::DEFAULT_WORK_BUFFER_LENGTH;
#endif
}

//----------------------------------------------------------
edit::internal::HioStream::ChannelType
SoundControlSession::GetChannelInfo(edit::internal::HioChannelType channel) const
{
#if defined(NW_SND_EDIT_USE_MCS)
    return static_cast<edit::internal::HioStream::ChannelType>(channel);
#else
    switch(channel)
    {
    case edit::internal::HIO_SNDEDIT_CTRL_CHANNEL:
            return static_cast<edit::internal::HioStream::ChannelType>(m_Port);
        default:
            NW_ASSERT(false);
            return 0;
    }
#endif
}

} // namespace nw::snd::ctrl
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
