﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_WaveSoundFile.h>

#include <nw/snd/snd_ElementType.h>
#include <nw/ut/ut_Inlines.h>               // ut::AddOffsetToPtr

namespace nw {
namespace snd {
namespace internal {

namespace {

const u8 WSD_DEFAULT_PAN            = 64;   // 中央
const s8 WSD_DEFAULT_SURROUND_PAN   = 0;    // 中央
const f32 WSD_DEFAULT_PITCH         = 1.0f;
const u8 WSD_DEFAULT_MAIN_SEND      = 127;  // 最大
const u8 WSD_DEFAULT_FX_SEND        = 0;    // 最小
const AdshrCurve WSD_DEFAULT_ADSHR_CURVE(
    127,    // u8 attack
    127,    // u8 decay
    127,    // u8 sustain
    127,    // u8 hold
    127     // u8 release
);
const u8 WSD_DEFAULT_LPF_FREQ       = 64;   // フィルタがかかっていない
const u8 WSD_DEFAULT_BIQUAD_TYPE    = 0;    // フィルタ未使用
const u8 WSD_DEFAULT_BIQUAD_VALUE   = 0;    // フィルタがかかっていない
const u8 WSD_DEFAULT_KEY            = 64;   // TODO: おそらく中央
const u8 WSD_DEFAULT_VOLUME         = 96;

enum WaveSoundInfoBitFlagWsd
{
    WAVE_SOUND_INFO_PAN = 0x00,
    WAVE_SOUND_INFO_PITCH,
    WAVE_SOUND_INFO_FILTER,
    WAVE_SOUND_INFO_SEND = 0x08,
    WAVE_SOUND_INFO_ENVELOPE,
    WAVE_SOUND_INFO_RANDOMIZER  // 未実装
};

enum NoteInfoBitFlag
{
    NOTE_INFO_KEY = 0x00,
    NOTE_INFO_VOLUME,
    NOTE_INFO_PAN,
    NOTE_INFO_PITCH,
    NOTE_INFO_SEND = 0x08,
    NOTE_INFO_ENVELOPE,
    NOTE_INFO_RANDOMIZER,   // 未実装
    NOTE_INFO_LFO           // 未実装
};

struct SendValueWsd
{
    u8 mainSend;
    Util::Table<u8,u8> fxSend;
};

} // anonymous namespace


//
// WaveSoundFile::FileHeader
//
const WaveSoundFile::InfoBlock* WaveSoundFile::FileHeader::GetInfoBlock() const
{
    return reinterpret_cast<const InfoBlock*>( GetBlock( ElementType_WaveSoundFile_InfoBlock ) );
}


//
// WaveSoundFile::InfoBlockBody
//

const WaveSoundFile::WaveSoundData&
WaveSoundFile::InfoBlockBody::GetWaveSoundData( u32 index ) const
{
    NW_ASSERT( index < GetWaveSoundCount() );
    const void* pWaveSoundData =
        GetWaveSoundDataReferenceTable().GetReferedItem(
                index,
                ElementType_WaveSoundFile_WaveSoundMetaData );
    NW_NULL_ASSERT( pWaveSoundData );
    return *reinterpret_cast<const WaveSoundData*>( pWaveSoundData );
}
const Util::ReferenceTable&
WaveSoundFile::InfoBlockBody::GetWaveSoundDataReferenceTable() const
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            ut::AddOffsetToPtr( this, toWaveSoundDataReferenceTable.offset ) );
}
const Util::WaveIdTable&
WaveSoundFile::InfoBlockBody::GetWaveIdTable() const
{
    return *reinterpret_cast<const Util::WaveIdTable*>(
            ut::AddOffsetToPtr( this, toWaveIdTable.offset ) );
}


//
// WaveSoundFile::WaveSoundData
//
const WaveSoundFile::WaveSoundInfo&
WaveSoundFile::WaveSoundData::GetWaveSoundInfo() const
{
    NW_ASSERT( toWaveSoundInfo.IsValidTypeId(
                ElementType_WaveSoundFile_WaveSoundInfo ) );

    return *reinterpret_cast<const WaveSoundInfo*>(
            ut::AddOffsetToPtr( this, toWaveSoundInfo.offset ) );
}

const Util::ReferenceTable&
WaveSoundFile::WaveSoundData::GetTrackInfoReferenceTable() const
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            ut::AddOffsetToPtr( this, toTrackInfoReferenceTable.offset ) );
}

const Util::ReferenceTable&
WaveSoundFile::WaveSoundData::GetNoteInfoReferenceTable() const
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            ut::AddOffsetToPtr( this, toNoteInfoReferenceTable.offset ) );
}

const WaveSoundFile::TrackInfo&
WaveSoundFile::WaveSoundData::GetTrackInfo( u32 index ) const
{
    NW_ASSERT( index < GetTrackCount() );

    const void* pTrackInfo =
        GetTrackInfoReferenceTable().GetReferedItem(
                index,
                ElementType_WaveSoundFile_TrackInfo );
    NW_NULL_ASSERT( pTrackInfo );

    return *reinterpret_cast<const TrackInfo*>( pTrackInfo );
}

const WaveSoundFile::NoteInfo&
WaveSoundFile::WaveSoundData::GetNoteInfo( u32 index ) const
{
    NW_ASSERT( index < GetNoteCount() );

    const void* pNoteInfo =
        GetNoteInfoReferenceTable().GetReferedItem(
                index,
                ElementType_WaveSoundFile_NoteInfo );
    NW_NULL_ASSERT( pNoteInfo );

    return *reinterpret_cast<const NoteInfo*>( pNoteInfo );
}


//
// WaveSoundFile::WaveSoundInfo
//
u8 WaveSoundFile::WaveSoundInfo::GetPan() const
{
    u32 value;
    bool result = optionParameter.GetValue( &value, WAVE_SOUND_INFO_PAN );
    if ( result == false ) return WSD_DEFAULT_PAN;
    return Util::DevideBy8bit( value, 0 );
}
s8 WaveSoundFile::WaveSoundInfo::GetSurroundPan() const
{
    u32 value;
    bool result = optionParameter.GetValue( &value, WAVE_SOUND_INFO_PAN );
    if ( result == false ) return WSD_DEFAULT_SURROUND_PAN;
    return static_cast<s8>( Util::DevideBy8bit( value, 1 ) );
}
f32 WaveSoundFile::WaveSoundInfo::GetPitch() const
{
    f32 value;
    bool result = optionParameter.GetValueF32( &value, WAVE_SOUND_INFO_PITCH );
    if ( result == false ) return WSD_DEFAULT_PITCH;
    return value;
}

void WaveSoundFile::WaveSoundInfo::GetSendValue(
        u8* mainSend, u8* fxSend, u8 fxSendCount ) const
{
    u32 value;
    bool result = optionParameter.GetValue( &value, WAVE_SOUND_INFO_SEND );
    if ( result == false )
    {
        *mainSend = WSD_DEFAULT_MAIN_SEND;
        for ( int i = 0; i < fxSendCount; i++ )
        {
            fxSend[ i ] = WSD_DEFAULT_FX_SEND;
        }
        return;
    }

    const SendValueWsd& sendValue = *reinterpret_cast<const SendValueWsd*>(
            ut::AddOffsetToPtr( this, value ) );

    *mainSend = sendValue.mainSend;
    int countSize = sendValue.fxSend.count > AUX_BUS_NUM ?
        AUX_BUS_NUM : sendValue.fxSend.count;
    for ( int i = 0; i < countSize; i++ )
    {
        fxSend[ i ] = sendValue.fxSend.item[ i ];
    }
}

const AdshrCurve& WaveSoundFile::WaveSoundInfo::GetAdshrCurve() const
{
    u32 offsetToReference;
    bool result = optionParameter.GetValue( &offsetToReference, WAVE_SOUND_INFO_ENVELOPE );
    if ( result == false ) return WSD_DEFAULT_ADSHR_CURVE;

    const Util::Reference& ref = *reinterpret_cast<const Util::Reference*>(
            ut::AddOffsetToPtr( this, offsetToReference ) );
    return *reinterpret_cast<const AdshrCurve*>(
            ut::AddOffsetToPtr( &ref, ref.offset ) );
}

u8 WaveSoundFile::WaveSoundInfo::GetLpfFreq() const
{
    u32 value;
    bool result = optionParameter.GetValue( &value, WAVE_SOUND_INFO_FILTER );
    if ( result == false ) return WSD_DEFAULT_LPF_FREQ;
    return Util::DevideBy8bit( value, 0 );
}

u8 WaveSoundFile::WaveSoundInfo::GetBiquadType() const
{
    u32 value;
    bool result = optionParameter.GetValue( &value, WAVE_SOUND_INFO_FILTER );
    if ( result == false ) return WSD_DEFAULT_BIQUAD_TYPE;
    return Util::DevideBy8bit( value, 1 );
}

u8 WaveSoundFile::WaveSoundInfo::GetBiquadValue() const
{
    u32 value;
    bool result = optionParameter.GetValue( &value, WAVE_SOUND_INFO_FILTER );
    if ( result == false ) return WSD_DEFAULT_BIQUAD_VALUE;
    return Util::DevideBy8bit( value, 2 );
}


//
// WaveSoundFile::TrackInfo
//
const Util::ReferenceTable&
WaveSoundFile::TrackInfo::GetNoteEventReferenceTable() const
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            ut::AddOffsetToPtr( this, toNoteEventReferenceTable.offset ) );
}

const WaveSoundFile::NoteEvent&
WaveSoundFile::TrackInfo::GetNoteEvent( u32 index ) const
{
    NW_ASSERT( index < GetNoteEventCount() );

    const void* pNoteEvent =
        GetNoteEventReferenceTable().GetReferedItem(
                index,
                ElementType_WaveSoundFile_NoteEvent );
    NW_NULL_ASSERT( pNoteEvent );

    return *reinterpret_cast<const NoteEvent*>( pNoteEvent );
}


//
// WaveSoundFile::NoteInfo
//
u8 WaveSoundFile::NoteInfo::GetOriginalKey() const
{
    u32 value;
    bool result = optionParameter.GetValue( &value, NOTE_INFO_KEY );
    if ( result == false ) return WSD_DEFAULT_KEY;
    return Util::DevideBy8bit( value, 0 );
}

u8 WaveSoundFile::NoteInfo::GetVolume() const
{
    u32 value;
    bool result = optionParameter.GetValue( &value, NOTE_INFO_VOLUME );
    if ( result == false ) return WSD_DEFAULT_VOLUME;
    return Util::DevideBy8bit( value, 0 );
}

u8 WaveSoundFile::NoteInfo::GetPan() const
{
    u32 value;
    bool result = optionParameter.GetValue( &value, NOTE_INFO_PAN );
    if ( result == false ) return WSD_DEFAULT_PAN;
    return Util::DevideBy8bit( value, 0 );
}
u8 WaveSoundFile::NoteInfo::GetSurroundPan() const
{
    u32 value;
    bool result = optionParameter.GetValue( &value, NOTE_INFO_PAN );
    if ( result == false ) return WSD_DEFAULT_SURROUND_PAN;
    return static_cast<s8>( Util::DevideBy8bit( value, 1 ) );
}
f32 WaveSoundFile::NoteInfo::GetPitch() const
{
    f32 value;
    bool result = optionParameter.GetValueF32( &value, NOTE_INFO_PITCH );
    if ( result == false ) return WSD_DEFAULT_PITCH;
    return value;
}

void WaveSoundFile::NoteInfo::GetSendValue( u8* mainSend, u8* fxSend[], u8 fxSendCount ) const
{
    u32 value;
    bool result = optionParameter.GetValue( &value, NOTE_INFO_SEND );
    if ( result == false )
    {
        *mainSend = WSD_DEFAULT_MAIN_SEND;
        for ( int i = 0; i < fxSendCount; i++ )
        {
            *fxSend[i] = WSD_DEFAULT_FX_SEND;
        }
        return;
    }

    const SendValueWsd& sendValue = *reinterpret_cast<const SendValueWsd*>(
            ut::AddOffsetToPtr( this, value ) );

    NW_ASSERT( fxSendCount <= sendValue.fxSend.count );
    *mainSend = sendValue.mainSend;
    int countSize = sendValue.fxSend.count > AUX_BUS_NUM ?
        AUX_BUS_NUM : sendValue.fxSend.count;
    for ( int i = 0; i < countSize; i++ )
    {
        *fxSend[ i ] = sendValue.fxSend.item[ i ];
    }
}

const AdshrCurve& WaveSoundFile::NoteInfo::GetAdshrCurve() const
{
    u32 offsetToReference;
    bool result = optionParameter.GetValue( &offsetToReference, NOTE_INFO_ENVELOPE );
    if ( result == false ) return WSD_DEFAULT_ADSHR_CURVE;

    const Util::Reference& ref = *reinterpret_cast<const Util::Reference*>(
            ut::AddOffsetToPtr( this, offsetToReference ) );
    return *reinterpret_cast<const AdshrCurve*>(
            ut::AddOffsetToPtr( &ref, ref.offset ) );
}


} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
