﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_WaveFileReader.h>

namespace nw {
namespace snd {
namespace internal {

namespace {

const u32 SIGNATURE_INFO_BLOCK_WAV = NW_UT_MAKE_SIGWORD( 'I', 'N', 'F', 'O' );
const u32 SIGNATURE_DATA_BLOCK_WAV = NW_UT_MAKE_SIGWORD( 'D', 'A', 'T', 'A' );

const u32 SUPPORTED_FILE_VERSION_WAV       = NW_UT_MAKE_VERSION( 0, 1, 0, 0 );  // ライブラリがサポートする最低バージョン
const u32 CURRENT_FILE_VERSION_WAV         = NW_UT_MAKE_VERSION( 0, 1, 2, 0 );  // ライブラリがサポートする最新バージョン
const u32 INCLUDE_ORIGINALLOOP_VERSION_WAV = NW_UT_MAKE_VERSION( 0, 1, 2, 0 );

bool IsValidFileHeaderWav( const void* waveFile )
{
#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader& header =
        *reinterpret_cast<const ut::BinaryFileHeader*>( waveFile );
#else
    const BinaryFileHeader& header =
        *reinterpret_cast<const BinaryFileHeader*>( waveFile );
#endif

    // シグニチャ確認
    NW_ASSERTMSG( header.signature == WaveFileReader::SIGNATURE_FILE,
            "invalid file signature." );
    if ( header.signature != WaveFileReader::SIGNATURE_FILE )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SUPPORTED_FILE_VERSION_WAV <= header.version) &&
                                   (header.version <= CURRENT_FILE_VERSION_WAV) )
    {
        isSupportedVersion = true;

    }
    NW_ASSERTMSG( isSupportedVersion,
            "bfwav file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SUPPORTED_FILE_VERSION_WAV, CURRENT_FILE_VERSION_WAV, header.version
    );
    return isSupportedVersion;
}

} // anonymous namespace


SampleFormat WaveFileReader::GetSampleFormat( u8 format )
{
    switch ( format )
    {
    case WaveFile::PCM8:        return SAMPLE_FORMAT_PCM_S8;
    case WaveFile::PCM16:       return SAMPLE_FORMAT_PCM_S16;
    case WaveFile::DSP_ADPCM:   return SAMPLE_FORMAT_DSP_ADPCM;
    default:
        NW_ASSERTMSG( false, "Unknown wave data format(%d)", format );
        return SAMPLE_FORMAT_DSP_ADPCM;
    }
}

WaveFileReader::WaveFileReader( const void* waveFile, s8 waveType )
: m_pHeader(NULL), m_pInfoBlockBody( NULL ), m_pDataBlockBody( NULL ), m_WaveType(waveType)
{
    switch (m_WaveType)
    {
    case WAVE_TYPE_NWWAV:
        {
            if ( ! IsValidFileHeaderWav( waveFile ) ) return;

            m_pHeader =
                reinterpret_cast<const WaveFile::FileHeader*>(waveFile);

            const WaveFile::InfoBlock* infoBlock = m_pHeader->GetInfoBlock();
            const WaveFile::DataBlock* dataBlock = m_pHeader->GetDataBlock();

            if ( infoBlock == NULL ) return;
            if ( dataBlock == NULL ) return;

            NW_ASSERT( infoBlock->header.kind == SIGNATURE_INFO_BLOCK_WAV );
            NW_ASSERT( dataBlock->header.kind == SIGNATURE_DATA_BLOCK_WAV );

            if ( infoBlock->header.kind != SIGNATURE_INFO_BLOCK_WAV )
            {
                return;
            }
            if ( dataBlock->header.kind != SIGNATURE_DATA_BLOCK_WAV )
            {
                return;
            }

            m_pInfoBlockBody = &infoBlock->body;
            m_pDataBlockBody = &dataBlock->byte;
        }
        break;
    case WAVE_TYPE_DSPADPCM:
        m_DspadpcmReader.Initialize(waveFile);
        break;
    }
}

// MEMO: 本当に必要？！
// WaveFileReader::WaveFileReader( const WaveFile::InfoBlockBody* infoBlockBody )
// : m_pInfoBlockBody( infoBlockBody )
// {
// }

bool WaveFileReader::IsOriginalLoopAvailable() const
{
#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader& header =
        *reinterpret_cast<const ut::BinaryFileHeader*>( m_pHeader );
#else
    const BinaryFileHeader& header =
        *reinterpret_cast<const BinaryFileHeader*>( m_pHeader );
#endif
    if (header.version >= INCLUDE_ORIGINALLOOP_VERSION_WAV)
    {
        return true;
    }
    return false;
}

bool WaveFileReader::ReadWaveInfo(
    WaveInfo* info,
    const void* waveDataOffsetOrigin ) const
{
    switch (m_WaveType)
    {
    case WAVE_TYPE_NWWAV:
        {
            NW_ASSERT( m_pInfoBlockBody );

            const SampleFormat format = GetSampleFormat( m_pInfoBlockBody->encoding );
            info->sampleFormat   = format;
            info->channelCount   = m_pInfoBlockBody->GetChannelCount();
            info->sampleRate     = m_pInfoBlockBody->sampleRate;
            info->loopFlag       = ( m_pInfoBlockBody->isLoop == 1 );
            info->loopStartFrame = m_pInfoBlockBody->loopStartFrame;
            info->loopEndFrame   = m_pInfoBlockBody->loopEndFrame;

            if ( IsOriginalLoopAvailable() )
            {
                info->originalLoopStartFrame = m_pInfoBlockBody->originalLoopStartFrame;
            }
            else
            {
                info->originalLoopStartFrame = m_pInfoBlockBody->loopStartFrame;
            }

            for ( s32 i = 0; i < m_pInfoBlockBody->GetChannelCount(); i++ )
            {
                if ( i >= WAVE_CHANNEL_MAX ) continue;

                WaveInfo::ChannelParam& channelParam = info->channelParam[ i ];

                const WaveFile::ChannelInfo& channelInfo = m_pInfoBlockBody->GetChannelInfo( i );
                // if ( channelInfo.offsetToAdpcmInfo != 0 )
                if ( channelInfo.referToAdpcmInfo.offset != 0 )
                {
                    const WaveFile::DspAdpcmInfo& adpcmInfo = channelInfo.GetDspAdpcmInfo();
                    channelParam.adpcmParam = adpcmInfo.adpcmParam;
                    channelParam.adpcmLoopParam = adpcmInfo.adpcmLoopParam;
                }

                channelParam.dataAddress = GetWaveDataAddress( &channelInfo, waveDataOffsetOrigin );
            }
        }
        break;
    case WAVE_TYPE_DSPADPCM:
        {
            m_DspadpcmReader.ReadWaveInfo(info);
        }
        break;
    }

    return true;
}

const void* WaveFileReader::GetWaveDataAddress(
        const WaveFile::ChannelInfo* info,
        const void* waveDataOffsetOrigin ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( info );

    NW_UNUSED_VARIABLE( waveDataOffsetOrigin );

#if 0
    const void* waveDataAddress = NULL;

    // dataLocationがデータブロックの先頭を指しているか（WaveFileの場合）
    // 波形データの先頭を指しているか（BankFile中のWaveInfoの場合）
    bool offsetIsDataBlock = ( waveDataOffsetOrigin == NULL );

    if ( waveDataOffsetOrigin == NULL ) {
        waveDataOffsetOrigin = m_pInfoBlockBody;
    }

    // MEMO: バンクの波形差し替えで WAVE_DATA_LOCATION_ADDRESS を使っていた？模様。
    //       今回、バンクの波形差し替えは、インストの warcID と waveIndex を
    //       差し替えられるようにして、対応する。
    switch( m_pInfoBlockBody->dataLocationType ) {
    case WaveFile::WAVE_DATA_LOCATION_OFFSET:
        waveDataAddress = ut::AddOffsetToPtr( waveDataOffsetOrigin, m_pInfoBlockBody->dataLocation );
        if ( offsetIsDataBlock ) {
            waveDataAddress = ut::AddOffsetToPtr( waveDataAddress, 8 );
        }
        break;
    case WaveFile::WAVE_DATA_LOCATION_ADDRESS:
        waveDataAddress = reinterpret_cast<const void*>( m_pInfoBlockBody->dataLocation );
        break;
    default:
        return NULL;
    }

    waveDataAddress = ut::AddOffsetToPtr( waveDataAddress, waveChannelInfo->channelDataOffset );

    return waveDataAddress;
#endif

    return info->GetSamplesAddress( m_pDataBlockBody );
}

// -----------------------------------------------------------------------------
// CalcOffsetContextInfo
//  指定したサンプル (offsetSample) に応じて、ADPCM のコンテキスト情報を計算します。
//  計算結果は info に格納されます。
// -----------------------------------------------------------------------------
bool WaveFileReader::CalcOffsetContextInfo(u32 offsetSample, OffsetContextInfo* info) const
{
    NW_NULL_ASSERT(info);

    // 圧縮フォーマット違い
    const SampleFormat format = GetSampleFormat( m_pInfoBlockBody->encoding );
    if (format != SAMPLE_FORMAT_DSP_ADPCM)
    {
        return false;
    }

    // データフォマット違い
    if (m_WaveType != WAVE_TYPE_NWWAV)
    {
        return false;
    }

    // NOTE:
    // 以下、Channel::AppendWaveBuffer のほぼコピペ

    // 8 バイト境界に切り下げる (DSP ADPCM は 8 バイト 14 サンプル単位のエンコードのため)
    const int SAMPLES_PER_ADPCMFRAME = 14;
    u32 offset = offsetSample;
    if (offset > 0)
    {
        offset = (offset / SAMPLES_PER_ADPCMFRAME) * SAMPLES_PER_ADPCMFRAME;
    }
    info->offsetSample = offset;

    // ADPCM 係数
    nw::snd::internal::WaveInfo waveInfo;
    {
        bool result = ReadWaveInfo(&waveInfo);
        if (result != true)
        {
            return false;
        }
    }

    // チャンネルごとの ADPCM コンテキストの生成
    const int CH_COUNT = m_pInfoBlockBody->GetChannelCount();
    for (int ch = 0; ch < CH_COUNT; ch++)
    {
        nw::snd::AdpcmParam param;
        nw::snd::AdpcmContext* context = &(info->contexts[ch]);
        const DspAdpcmParam* pParam = &waveInfo.channelParam[ch].adpcmParam;
        for (int i = 0; i < 8; i++)
        {
            param.coef[i][0] = pParam->coef[i][0];
            param.coef[i][1] = pParam->coef[i][1];
        }

        // 初期化 (0サンプル目のコンテキスト)
        context->pred_scale = pParam->predScale;
        context->yn1 = pParam->yn1;
        context->yn2 = pParam->yn2;
        if (offset == 0)
        {
            continue;
        }

        // offset > 0 のとき
        driver::MultiVoice::CalcOffsetAdpcmParam(
            context, param, offset, waveInfo.channelParam[ch].dataAddress);
    }
    return true;
}


} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

