﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_SequenceSoundPlayer.h>

#include <nw/snd/snd_SequenceTrack.h>
#include <nw/snd/snd_SequenceTrackAllocator.h>
#include <nw/snd/snd_DisposeCallbackManager.h>
#include <nw/snd/snd_HardwareManager.h>
#include <nw/snd/snd_NoteOnCallback.h>
#include <nw/snd/snd_SoundThread.h>
#include <nw/snd/snd_PlayerHeap.h>
#include <nw/snd/snd_TaskManager.h>
#include <nw/snd/snd_SoundPlayer.h>
#include <nw/snd/snd_SoundDataManager.h>
#include <nw/snd/snd_SequenceSoundFileReader.h>

namespace
{
// DSP プロセスの割り込み間隔の分母・分子。
// NUMERATOR / DENOMINATOR [msec] (3) が実際の割り込み間隔。
const u32 INTERVAL_MSEC_DENOMINATOR = 0x10000;
const u32 INTERVAL_MSEC_NUMERATOR   = AX_MS_PER_FRAME * INTERVAL_MSEC_DENOMINATOR;
}

namespace nw {
namespace snd {
namespace internal {
namespace driver {

/* ========================================================================
        static variable
   ======================================================================== */

vs16 SequenceSoundPlayer::m_GlobalVariable[ GLOBAL_VARIABLE_NUM ];
vs32 SequenceSoundPlayer::m_SkipIntervalTickPerFrame = DEFAULT_SKIP_INTERVAL_TICK;

/* ========================================================================
        public function
   ======================================================================== */

void SequenceSoundPlayer::InitSequenceSoundPlayer()
{
    for( int variableNo = 0 ; variableNo < GLOBAL_VARIABLE_NUM ; variableNo++ )
    {
        m_GlobalVariable[ variableNo ] = VARIABLE_DEFAULT_VALUE;
    }
}

/*---------------------------------------------------------------------------*
  Name:         SequenceSoundPlayer

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
SequenceSoundPlayer::SequenceSoundPlayer()
: m_ReleasePriorityFixFlag(false),
  m_TempoRatio(1.0f),
  m_TickFraction(0.0f),
  m_SkipTimeCounter(0.0f),
  m_DelayCount(0),
  m_PanRange(1.0f),
  m_TickCounter(0),
  m_SequenceUserprocCallback(NULL),
  m_pSequenceUserprocCallbackArg(NULL),
  m_IsInitialized(false),
  m_IsRegisterPlayerCallback(false),
  m_pLoaderManager(NULL),
  m_pLoader(NULL)
{
    for ( int varNo = 0; varNo < PLAYER_VARIABLE_NUM ; varNo++ )
    {
        m_LocalVariable[ varNo ] = VARIABLE_DEFAULT_VALUE;
    }

    for ( int trackNo = 0; trackNo < TRACK_NUM_PER_PLAYER ; trackNo++ )
    {
        m_pTracks[trackNo] = NULL;
    }
}

/*---------------------------------------------------------------------------*
  Name:         ~SequenceSoundPlayer

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
SequenceSoundPlayer::~SequenceSoundPlayer()
{
    Finalize();
}

/*---------------------------------------------------------------------------*
  Name:         Initialize

  Description:  パラメータを初期化します

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::Initialize()
{
    BasicSoundPlayer::Initialize();

    m_StartedFlag                  = false;
    m_PauseFlag                    = false;
    m_ReleasePriorityFixFlag       = false;
    m_ActiveFlag                   = false;

    m_TempoRatio                   = 1.0f;
    m_TickFraction                 = 0.0f;
    m_SkipTickCounter              = 0;
    m_SkipTimeCounter              = 0.0f;
    m_DelayCount                   = 0;
    m_PanRange                     = 1.0f;
    m_TickCounter                  = 0;

    m_SequenceUserprocCallback     = NULL;
    m_pSequenceUserprocCallbackArg = NULL;

    m_ParserParam.tempo            = DEFAULT_TEMPO;
    m_ParserParam.timebase         = DEFAULT_TIMEBASE;
    m_ParserParam.volume.InitValue(127);
    m_ParserParam.priority         = 64;
    m_ParserParam.callback         = NULL;

    for ( int varNo = 0; varNo < PLAYER_VARIABLE_NUM ; varNo++ )
    {
        m_LocalVariable[ varNo ] = VARIABLE_DEFAULT_VALUE;
    }

    for ( int trackNo = 0; trackNo < TRACK_NUM_PER_PLAYER ; trackNo++ )
    {
        m_pTracks[trackNo] = NULL;
    }
    m_IsRegisterPlayerCallback = false;
    m_IsInitialized = true;
}

/*---------------------------------------------------------------------------*
  Name:         Finalize

  Description:  シーケンスプレイヤーのシャットダウン

  Arguments:    None.

  Returns:      None.
  *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::Finalize()
{
    m_FinishFlag = true;

    FinishPlayer();

    // 無効化リストから削除
    if ( m_ActiveFlag )
    {
        DisposeCallbackManager::GetInstance().UnregisterDisposeCallback( this );
        m_ActiveFlag = false;
    }

    for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
    {
        m_BankFileReader[i].Finalize();
        m_WarcFileReader[i].Finalize();
    }

    if (m_IsInitialized)
    {
        BasicSoundPlayer::Finalize();
        m_IsInitialized = false;
    }

    FreeLoader();
}

/*---------------------------------------------------------------------------*
  Name:         Setup

  Description:  シーケンスの初期化を行います

  Arguments:

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::Setup( const SetupArg& arg )
{
    //-----------------------------------------------------------------------------
    // プレイヤー初期化
    m_ParserParam.callback = arg.callback;

    {
        //-----------------------------------------------------------------------------
        // 全トラックを確保することができるかどうか
        {
            int trackCount = 0;
            for( u32 trackBitMask = arg.allocTracks; trackBitMask != 0; trackBitMask >>= 1 )
            {
                if ( trackBitMask & 0x01 ) trackCount++;
            }
            int allocatableCount = arg.trackAllocator->GetAllocatableTrackCount();
            if (trackCount > allocatableCount)
            {
                NW_WARNING(false, "Failed to start sequence sound for not enough sequence track instance." );
                Finalize();
                return;
            }
        }

        //-----------------------------------------------------------------------------
        // トラック確保
        {
            u32 trackBitMask = arg.allocTracks;
            for( int trackNo = 0; trackBitMask != 0; trackNo++, trackBitMask >>= 1 )
            {
                if ( ( trackBitMask & 0x01 ) == 0 ) continue;

                SequenceTrack* track = arg.trackAllocator->AllocTrack( this );
                NW_NULL_ASSERT( track );
                SetPlayerTrack( trackNo, track );
            }
        }
    }

    // NN_LOG( "track rest %d\n",trackAllocator->GetAllocatableTrackCount());

    m_pSequenceTrackAllocator = arg.trackAllocator;

#ifdef NW_PLATFORM_CTR
  #if 0   // TODO: perf
    for ( int i = 0; i < PERF_NUM; i++ )
    {
        s_PerfParam[ i ].Reset();
        s_PerfParam[ i ].isCalcTick = true;
    }
  #endif
#endif
    // SoundThread::GetInstance().SetSoundPlay( true );

//    return SETUP_SUCCESS;
}

/*---------------------------------------------------------------------------*
  Name:         Start

  Description:  準備完了したシーケンスの再生を開始します

  Arguments:    playerNo - プレイヤー番号

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::Start()
{
    // プレイヤーリストに追加
    m_StartedFlag = true;
}

/*---------------------------------------------------------------------------*
  Name:         Stop

  Description:  シーケンスを止めます

  Arguments:    playerNo - プレイヤー番号

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::Stop()
{
    FinishPlayer();
}

/*---------------------------------------------------------------------------*
  Name:         Pause

  Description:  シーケンスを一時停止または再開します

  Arguments:    playerNo - プレイヤー番号
                flag - trueで一時停止、falseで再開します

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::Pause( bool flag )
{
    m_PauseFlag = flag;

    // 全トラックのチャンネルを停止
    SequenceTrack* track;
    for( int trackNo = 0; trackNo < TRACK_NUM_PER_PLAYER ; trackNo++ )
    {
        track = GetPlayerTrack( trackNo );
        if ( track == NULL ) continue;

        track->PauseAllChannel( flag );
    }
}

/*---------------------------------------------------------------------------*
  Name:         Skip

  Description:  シーケンスをスキップします
                注意：サウンドフレームを生成するタイマーを一時的に止めています

  Arguments:    playerNo - プレイヤー番号
                tick - スキップ量をティックで指定します

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::Skip( StartOffsetType offsetType, int offset )
{
    if ( !m_ActiveFlag )
    {
        return;
    }

    switch ( offsetType )
    {
    case START_OFFSET_TYPE_TICK:
        m_SkipTickCounter += offset;
        break;
    case START_OFFSET_TYPE_MILLISEC:
        m_SkipTimeCounter += static_cast<f32>( offset );
        break;
    }
}

void SequenceSoundPlayer::SetTempoRatio( f32 tempoRatio )
{
    NW_ASSERT( tempoRatio >= 0.0f );
    m_TempoRatio = tempoRatio;
}

void SequenceSoundPlayer::SetPanRange( f32 panRange )
{
    m_PanRange = panRange;
}

void SequenceSoundPlayer::SetChannelPriority( int priority )
{
    NW_MINMAX_ASSERT( priority, 0, 127 );
    m_ParserParam.priority = static_cast<u8>( priority );
}

void SequenceSoundPlayer::SetReleasePriorityFix( bool fix )
{
    m_ReleasePriorityFixFlag = fix;
}

void SequenceSoundPlayer::SetSequenceUserprocCallback( SequenceUserprocCallback callback, void* arg )
{
    m_SequenceUserprocCallback = callback;
    m_pSequenceUserprocCallbackArg = arg;
}

void SequenceSoundPlayer::CallSequenceUserprocCallback( u16 procId, SequenceTrack* track )
{
    if ( m_SequenceUserprocCallback == NULL )
    {
        return;
    }

    NW_NULL_ASSERT( track );

    SequenceTrack::ParserTrackParam& trackParam = track->GetParserTrackParam();

    SequenceUserprocCallbackParam param;
    param.localVariable  = GetVariablePtr( 0 );
    param.globalVariable = GetVariablePtr( 16 );
    param.trackVariable  = track->GetVariablePtr( 0 );
    param.cmpFlag = trackParam.cmpFlag;

    m_SequenceUserprocCallback(
        procId,
        &param,
        m_pSequenceUserprocCallbackArg
    );

    trackParam.cmpFlag = param.cmpFlag;
}


/*---------------------------------------------------------------------------*
  Name:         GetLocalVariable

  Description:  シーケンスローカル変数を取得します。

  Arguments:    player - プレイヤー
                varNo - 変数番号

  Returns:      変数の値
 *---------------------------------------------------------------------------*/
s16 SequenceSoundPlayer::GetLocalVariable( int varNo ) const
{
    NW_MINMAXLT_ASSERT( varNo, 0, PLAYER_VARIABLE_NUM );

    return m_LocalVariable[ varNo ];
}

/*---------------------------------------------------------------------------*
  Name:         GetGlobalVariable

  Description:  シーケンスグローバル変数を取得します。

  Arguments:    varNo - 変数番号

  Returns:      変数の値
 *---------------------------------------------------------------------------*/
s16 SequenceSoundPlayer::GetGlobalVariable( int varNo )
{
    NW_MINMAXLT_ASSERT( varNo, 0, GLOBAL_VARIABLE_NUM );

    return m_GlobalVariable[ varNo ];
}

/*---------------------------------------------------------------------------*
  Name:         SetLocalVariable

  Description:  シーケンスローカル変数を設定します。

  Arguments:    player - プレイヤー
                varNo - 変数番号
                var - 変数値

  Returns:      変数の値
 *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::SetLocalVariable( int varNo, s16 var )
{
    NW_MINMAXLT_ASSERT( varNo, 0, PLAYER_VARIABLE_NUM );

    m_LocalVariable[ varNo ] = var;
}

/*---------------------------------------------------------------------------*
  Name:         SetGlobalVariable

  Description:  シーケンスグローバル変数を設定します。

  Arguments:    varNo - 変数番号
                var - 変数値

  Returns:      変数の値
 *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::SetGlobalVariable( int varNo, s16 var )
{
    NW_MINMAXLT_ASSERT( varNo, 0, GLOBAL_VARIABLE_NUM );

    m_GlobalVariable[ varNo ] = var;
}

/*---------------------------------------------------------------------------*
  Name:         SetTrackMute

  Description:  トラックをミュートまたはミュート解除します

  Arguments:    trackBitFlag - ミュート設定するトラックのビットマスク
                flag         - trueでミュート、falseでミュート解除します

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::SetTrackMute( u32 trackBitFlag, SeqMute mute )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetMute, mute );
}

void SequenceSoundPlayer::SetTrackSilence( unsigned long trackBitFlag, bool silenceFlag, int fadeTimes )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetSilence, silenceFlag, fadeTimes );
}

void SequenceSoundPlayer::SetTrackVolume( u32 trackBitFlag, f32 volume )
{
    NW_ASSERT( volume >= 0.0f );
    SetTrackParam( trackBitFlag, &SequenceTrack::SetVolume, volume );
}

void SequenceSoundPlayer::SetTrackPitch( u32 trackBitFlag, f32 pitch )
{
    NW_ASSERT( pitch >= 0.0f );
    SetTrackParam( trackBitFlag, &SequenceTrack::SetPitch, pitch );
}

void SequenceSoundPlayer::SetTrackLpfFreq( u32 trackBitFlag, f32 lpfFreq )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetLpfFreq, lpfFreq );
}

void SequenceSoundPlayer::SetTrackBiquadFilter( u32 trackBitFlag, int type, f32 value )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetBiquadFilter, type, value );
}

bool SequenceSoundPlayer::SetTrackBankIndex( u32 trackBitFlag, int bankIndex )
{
    // SequenceSoundHandle::SetTrackBankIndex でも範囲チェックしているが、念のため。
    NW_MINMAXLT_ASSERT( bankIndex, 0, SoundArchive::SEQ_BANK_MAX );
    if ( m_BankFileReader[ bankIndex ].IsInitialized() == false )
    {
        return false;
    }

    SetTrackParam( trackBitFlag, &SequenceTrack::SetBankIndex, bankIndex );
    return true;
}

void SequenceSoundPlayer::SetTrackTranspose( u32 trackBitFlag, s8 transpose )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetTranspose, transpose );
}

void SequenceSoundPlayer::SetTrackVelocityRange( u32 trackBitFlag, u8 range )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetVelocityRange, range );
}

void SequenceSoundPlayer::SetTrackOutputLine( u32 trackBitFlag, u32 outputLine )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetOutputLine, static_cast<s32>(outputLine) );
}

void SequenceSoundPlayer::ResetTrackOutputLine( u32 trackBitFlag )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetOutputLine, -1 );
}



void SequenceSoundPlayer::SetTrackTvVolume( u32 trackBitFlag, f32 volume )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetTvVolume, volume );
}
void SequenceSoundPlayer::SetTrackChannelTvMixParameter( u32 trackBitFlag, u32 srcChNo, const MixParameter& param )
{
    for (s32 ch = 0; ch < CHANNEL_INDEX_NUM; ch++ )
    {
        SetTrackParam( trackBitFlag, &SequenceTrack::SetTvMixParameter, srcChNo, ch, param.ch[ch] );
    }
}
void SequenceSoundPlayer::SetTrackTvPan( u32 trackBitFlag, f32 pan )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetTvPan, pan );
}
void SequenceSoundPlayer::SetTrackTvSurroundPan( u32 trackBitFlag, f32 surroundPan )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetTvSurroundPan, surroundPan );
}
void SequenceSoundPlayer::SetTrackTvMainSend( u32 trackBitFlag, f32 send )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetTvMainSend, send );
}
void SequenceSoundPlayer::SetTrackTvFxSend( u32 trackBitFlag, AuxBus bus, f32 send )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetTvFxSend, bus, send );
}



void SequenceSoundPlayer::SetTrackDrcVolume( u32 drcIndex, u32 trackBitFlag, f32 volume )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetDrcVolume, drcIndex, volume );
}
void SequenceSoundPlayer::SetTrackChannelDrcMixParameter( u32 drcIndex, u32 trackBitFlag, u32 srcChNo, const MixParameter& param )
{
    for (s32 ch = 0; ch < CHANNEL_INDEX_NUM; ch++ )
    {
        SetTrackParam( trackBitFlag, &SequenceTrack::SetDrcMixParameter, drcIndex, srcChNo, ch, param.ch[ch] );
    }
}
void SequenceSoundPlayer::SetTrackDrcPan( u32 drcIndex, u32 trackBitFlag, f32 pan )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetDrcPan, drcIndex, pan );
}
void SequenceSoundPlayer::SetTrackDrcSurroundPan( u32 drcIndex, u32 trackBitFlag, f32 span )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetDrcSurroundPan, drcIndex, span );
}
void SequenceSoundPlayer::SetTrackDrcMainSend( u32 drcIndex, u32 trackBitFlag, f32 send )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetDrcMainSend, drcIndex, send );
}
void SequenceSoundPlayer::SetTrackDrcFxSend( u32 drcIndex, u32 trackBitFlag, AuxBus bus, f32 send )
{
    SetTrackParam( trackBitFlag, &SequenceTrack::SetDrcFxSend, drcIndex, bus, send );
}

/*---------------------------------------------------------------------------*
  Name:         InvalidateData

  Description:  指定シーケンスデータを使用しているシーケンスを無効化します

  Arguments:    start - シーケンスデータ開始アドレス
                end   - シーケンスデータ終了アドレス

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::InvalidateData( const void* start, const void* end )
{
    if ( m_ActiveFlag )
    {
        for( int trackNo = 0; trackNo < TRACK_NUM_PER_PLAYER ; trackNo++ )
        {
            SequenceTrack* track = GetPlayerTrack( trackNo );
            if ( track == NULL ) continue;

            const u8* cur = track->GetParserTrackParam().baseAddr;
            if ( start <= cur && cur <= end )
            {
                Finalize();
                break;
            }
        }
        for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
        {
            const void* cur = m_BankFileReader[i].GetBankFileAddress();
            if ( start <= cur && cur <= end )
            {
                m_BankFileReader[i].Finalize();
            }
        }
    }
}

/*---------------------------------------------------------------------------*
  Name:         GetPlayerTrack

  Description:  プレイヤーが保持しているトラックを取得します

  Arguments:    trackNo  - トラック番号

  Returns:      トラックポインタ
 *---------------------------------------------------------------------------*/
SequenceTrack* SequenceSoundPlayer::GetPlayerTrack( int trackNo )
{
    if ( trackNo > TRACK_NUM_PER_PLAYER - 1 )
    {
        return NULL;
    }

    return m_pTracks[ trackNo ];
}

/*---------------------------------------------------------------------------*
  Name:         GetPlayerTrack

  Description:  プレイヤーが保持しているトラックを取得します

  Arguments:    trackNo  - トラック番号

  Returns:      トラックポインタ
 *---------------------------------------------------------------------------*/
const SequenceTrack* SequenceSoundPlayer::GetPlayerTrack( int trackNo ) const
{
    if ( trackNo > TRACK_NUM_PER_PLAYER - 1 )
    {
        return NULL;
    }

    return m_pTracks[ trackNo ];
}

/*---------------------------------------------------------------------------*
  Name:         CloseTrack

  Description:  プレイヤーのトラックを閉じます。
                トラックシーケンス完了時、シーケンス２重オープン時、
                プレイヤーの完了時に呼びだされます。

  Arguments:    player : プレイヤー

  Returns:      None
 *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::CloseTrack( int trackNo )
{
    NW_MINMAXLT_ASSERT( trackNo, 0, TRACK_NUM_PER_PLAYER );

    SequenceTrack* track = GetPlayerTrack( trackNo );
    if ( track == NULL )
    {
        return;
    }

    track->Close();
    m_pSequenceTrackAllocator->FreeTrack( m_pTracks[ trackNo ] );
    m_pTracks[ trackNo ] = NULL;
}

/*---------------------------------------------------------------------------*
  Name:         SetPlayerTrack

  Description:  プレイヤーが管理するトラックを追加します

  Arguments:    trackNo  - トラック番号
                trackID  - トラックＩＤ

  Returns:      なし
 *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::SetPlayerTrack( int trackNo, SequenceTrack* track )
{
    if ( trackNo > TRACK_NUM_PER_PLAYER - 1 )
    {
        return;
    }
    m_pTracks[ trackNo ] = track;
    track->SetPlayerTrackNo( trackNo );

    // トラックパラメータのフロントバイパスフラグに「サウンド」のフロントバイパスフラグを反映
    // (その後、frontbypass_on/_off コマンドで上書きされ得ます)
    track->GetParserTrackParam().frontBypassFlag = IsFrontBypass();
}


/*---------------------------------------------------------------------------*
  Name:         FinishPlayer

  Description:  プレイヤーの完了処理を行います。

  Arguments:    player : プレイヤーポインタ

  Returns:      None
 *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::FinishPlayer()
{
    if (m_IsRegisterPlayerCallback)
    {
        SoundThread::GetInstance().UnregisterPlayerCallback( this );
        m_IsRegisterPlayerCallback = false;
    }

    if (m_StartedFlag)
    {
        m_StartedFlag = false;
    }

    // 全トラック解放
    for( int trackNo = 0; trackNo < TRACK_NUM_PER_PLAYER ; trackNo++ )
    {
        CloseTrack( trackNo );
    }
}

/*---------------------------------------------------------------------------*
  Name:         UpdateChannelParam

  Description:  プレイヤーの全トラックのチャンネルパラメータを更新します

  Arguments:    player - プレイヤーポインタ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::UpdateChannelParam()
{
    SequenceTrack* track;
    for( int trackNo = 0; trackNo < TRACK_NUM_PER_PLAYER ; trackNo++ )
    {
        track = GetPlayerTrack( trackNo );
        if ( track == NULL ) continue;
        track->UpdateChannelParam();
    }
}

/*---------------------------------------------------------------------------*
  Name:         ParseNextTick

  Description:  プレイヤーのシーケンス処理を行います

  Arguments:    player - プレイヤーポインタ
                doNoteOn - ノートオンするかどうか

  Returns:      継続時には０を、完了時には１を返します
 *---------------------------------------------------------------------------*/
int SequenceSoundPlayer::ParseNextTick( bool doNoteOn )
{
    m_ParserParam.volume.Update();

    bool activeFlag = false;

    for ( int trackNo = 0; trackNo < TRACK_NUM_PER_PLAYER ; trackNo++ )
    {
        SequenceTrack* track = GetPlayerTrack( trackNo );
        if ( track == NULL )
        {
            continue;
        }

        track->UpdateChannelLength();

        if ( track->ParseNextTick( doNoteOn ) < 0 )
        {
            // トラックのシーケンス終了
            CloseTrack( trackNo );
        }

        if ( track->IsOpened() )
        {
            activeFlag = true;
        }
    }

    if ( ! activeFlag )
    {
        return 1;
    }

    return 0;
}

/*---------------------------------------------------------------------------*
  Name:         GetVariablePtr

  Description:  変数のポインタを取得します

  Arguments:    player - プレイヤーポインタ
                varNo    - 変数番号

  Returns:      変数のポインタ
 *---------------------------------------------------------------------------*/
vs16* SequenceSoundPlayer::GetVariablePtr( int varNo )
{
    NW_MINMAX_ASSERT( varNo, 0, PLAYER_VARIABLE_NUM + GLOBAL_VARIABLE_NUM );

    if ( varNo < PLAYER_VARIABLE_NUM )
    {
        return & m_LocalVariable[ varNo ] ;
    }
    else if ( varNo < PLAYER_VARIABLE_NUM + GLOBAL_VARIABLE_NUM )
    {
        return & m_GlobalVariable[ varNo - PLAYER_VARIABLE_NUM ];
    }
    else
    {
        return NULL;
    }
}

/*---------------------------------------------------------------------------*
  Name:         Update

  Description:  プレイヤーのフレーム処理を行います

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::Update()
{
    // データロード状態の管理
    {
        switch (m_ResState)
        {
        case RES_STATE_ASSIGNED:
            break;
        case RES_STATE_RECV_LOAD_REQ:
            if (TryAllocLoader())
            {
                m_pLoader->Initialize(m_LoaderArg);
            }
            else
            {
                return; // リトライする
            }
            // /* FALLTHROUGH */
        case RES_STATE_APPEND_LOAD_TASK:
            NW_ASSERT_NOT_NULL(m_pLoader);
            if (m_pLoader->TryWait())
            {
                if (m_pLoader->IsLoadSuccess())
                {
                    // タスク完了したので、プリペア
                    const SequenceSoundLoader::Data& data = m_pLoader->GetData();
                    PrepareArg arg;
                    arg.seqFile = data.seqFile;
                    arg.seqOffset = m_StartInfo.seqOffset;
                    arg.delayTime = m_StartInfo.delayTime;
                    for (int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++)
                    {
                        arg.bankFiles[i] = data.bankFiles[i];
                        arg.warcFiles[i] = data.warcFiles[i];
                        arg.warcIsIndividuals[i] = data.warcIsIndividuals[i];
                    }
                    PrepareForPlayerHeap(arg);
                    Skip(m_StartInfo.startOffsetType, m_StartInfo.startOffset);
                }
                else
                {
                    // データロードに失敗したので終了
                    m_FinishFlag = true;
                    FinishPlayer();
                    return;
                }
            }
            else
            {
                return; // リトライする
            }
            break;
        default:
            NW_ASSERT(false);
            break;
        }
    }

    if (m_DelayCount > 0)
    {
        --m_DelayCount;
        return;
    }

    // フラグチェック
    {
        if ( ! m_ActiveFlag )
        {
            return;
        }
        if ( ! m_StartedFlag )
        {
            return;
        }
    }

    if ( ( m_SkipTickCounter > 0 ) || ( m_SkipTimeCounter > 0.0f ) )
    {
        SkipTick();
    }
    else if ( ! m_PauseFlag )
    {
        UpdateTick();
    }

    UpdateChannelParam();
}

bool SequenceSoundPlayer::TryAllocLoader()
{
    if (m_pLoaderManager == NULL)
    {
        return false;
    }

    SequenceSoundLoader* loader = m_pLoaderManager->Alloc();
    if (loader == NULL)
    {
        return false;
    }

    m_pLoader = loader;
    m_ResState = RES_STATE_APPEND_LOAD_TASK;
    return true;
}

void SequenceSoundPlayer::FreeLoader()
{
    if (m_pLoader)
    {
        m_pLoaderManager->Free(m_pLoader);
        m_pLoader = NULL;
    }
}


/*---------------------------------------------------------------------------*
  Name:         UpdateTick

  Description:  プレイヤーのティック処理を行います

  Returns:      None.
  *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::UpdateTick()
{
    f32 tickPerMsec = CalcTickPerMsec();
    if ( tickPerMsec == 0.0f )
    {
        return;
    }

    u64 restMsec = INTERVAL_MSEC_NUMERATOR;
    u64 nextMsec = static_cast<u64>(INTERVAL_MSEC_DENOMINATOR * m_TickFraction / tickPerMsec);

    while ( nextMsec < restMsec )
    {
        restMsec -= nextMsec;

        bool result = ( ParseNextTick( true ) != 0 );

        if ( result )
        {
            FinishPlayer();
            m_FinishFlag = true;
            return;
        }
        ++m_TickCounter;

        tickPerMsec = CalcTickPerMsec();
        if ( tickPerMsec == 0.0f )
        {
            return;
        }
        nextMsec = static_cast<u64>(INTERVAL_MSEC_DENOMINATOR / tickPerMsec);
    }
    nextMsec -= restMsec;
    m_TickFraction = nextMsec * tickPerMsec / INTERVAL_MSEC_DENOMINATOR;
}

/*---------------------------------------------------------------------------*
  Name:         SkipTick

  Description:  ティックのスキップ処理を行います

  Arguments:    None.

  Returns:      None.
  *---------------------------------------------------------------------------*/
void SequenceSoundPlayer::SkipTick()
{
    // 全トラックの全チャンネルをリリース
    for( int trackNo = 0; trackNo < TRACK_NUM_PER_PLAYER ; trackNo++ )
    {
        SequenceTrack* track = GetPlayerTrack( trackNo );
        if ( track == NULL ) continue;

        track->ReleaseAllChannel( SequenceTrack::PAUSE_RELEASE_VALUE );
        track->FreeAllChannel();
    }

    // スキップ処理
    int skipCount = 0;
    while ( m_SkipTickCounter > 0 || m_SkipTimeCounter * CalcTickPerMsec() >= 1.0f )
    {
        if ( skipCount >= m_SkipIntervalTickPerFrame )
        {
            return;
        }

        if ( m_SkipTickCounter > 0 )
        {
            --m_SkipTickCounter;
        }
        else
        {
            f32 tickPerMsec = CalcTickPerMsec();
            NW_ASSERT( tickPerMsec > 0.0f ); // 0.0fの時、スキップは行われない
            f32 msecPerTick = 1.0f / tickPerMsec;
            m_SkipTimeCounter -= msecPerTick;
        }

        if ( ParseNextTick( false ) != 0 )
        {
            FinishPlayer();
            m_FinishFlag = true;
            return;
        }
        ++skipCount;
        ++m_TickCounter;
    }

    m_SkipTimeCounter = 0.0f;
}

Channel* SequenceSoundPlayer::NoteOn(
    u8 bankIndex,
    const NoteOnInfo& noteOnInfo
)
{
    Channel* channel = m_ParserParam.callback->NoteOn(
        this,
        bankIndex,
        noteOnInfo
    );

    return channel;
}

void SequenceSoundPlayer::Prepare(const PrepareArg& arg)
{
    if (m_ActiveFlag)
    {
        FinishPlayer();
    }

    SequenceTrack* seqTrack = GetPlayerTrack( 0 );
    if ( seqTrack == NULL )
    {
        Finalize();
        return;
    }

    // bXseq
    {
        NW_ASSERT_NOT_NULL( arg.seqFile );
        SequenceSoundFileReader reader(arg.seqFile);
        const void* seqData = reader.GetSequenceData(); // 'DATA' ブロックボディ
        seqTrack->SetSeqData( seqData, arg.seqOffset );
        seqTrack->Open();
    }

    // bXbnk, bXwar
    for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
    {
        m_BankFileReader[i].Initialize( arg.bankFiles[i] );
        m_WarcFileReader[i].Initialize( arg.warcFiles[i], arg.warcIsIndividuals[i] );
    }

    m_ResState = RES_STATE_ASSIGNED;
    m_DelayCount = ToDelayCount(arg.delayTime);
    m_ActiveFlag = true;
    DisposeCallbackManager::GetInstance().RegisterDisposeCallback( this );
    SoundThread::GetInstance().RegisterPlayerCallback( this );
    m_IsRegisterPlayerCallback = true;
}

void SequenceSoundPlayer::PrepareForPlayerHeap(const PrepareArg& arg)
{
    if (m_ActiveFlag)
    {
        FinishPlayer();
    }

    SequenceTrack* seqTrack = GetPlayerTrack( 0 );
    if ( seqTrack == NULL )
    {
        Finalize();
        return;
    }

    // bXseq
    {
        NW_ASSERT_NOT_NULL( arg.seqFile );
        SequenceSoundFileReader reader(arg.seqFile);
        const void* seqData = reader.GetSequenceData(); // 'DATA' ブロックボディ
        seqTrack->SetSeqData( seqData, arg.seqOffset );
        seqTrack->Open();
    }

    // bXbnk, bXwar
    for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
    {
        m_BankFileReader[i].Initialize( arg.bankFiles[i] );
        m_WarcFileReader[i].Initialize( arg.warcFiles[i], arg.warcIsIndividuals[i] );
    }

    m_ResState = RES_STATE_ASSIGNED;
    m_DelayCount = ToDelayCount(arg.delayTime);
    m_ActiveFlag = true;
    DisposeCallbackManager::GetInstance().RegisterDisposeCallback(this);
    m_IsRegisterPlayerCallback = true;
}

void SequenceSoundPlayer::RequestLoad(
        const StartInfo& info,
        const SequenceSoundLoader::Arg& arg)
{
    m_StartInfo = info; // 構造体コピー
    m_LoaderArg = arg; // 構造体コピー
    m_ResState = RES_STATE_RECV_LOAD_REQ;
    m_DelayCount = 0; // Prepareが済んでからカウントを開始するように実装していますが、念のためにここでは0にしておく

    SoundThread::GetInstance().RegisterPlayerCallback(this);
    m_IsRegisterPlayerCallback = true;
}

void SequenceSoundPlayer::PrepareForMidi(
        const void* banks[],
        const void* warcs[],
        bool warcIsIndividuals[] )
{
    for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
    {
        m_BankFileReader[i].Initialize( banks[i] );
        m_WarcFileReader[i].Initialize( warcs[i], warcIsIndividuals[i] );
    }
    m_ResState = RES_STATE_ASSIGNED;

    if (m_ActiveFlag == false)
    {
        DisposeCallbackManager::GetInstance().RegisterDisposeCallback( this );
        m_ActiveFlag = true;
        SoundThread::GetInstance().RegisterPlayerCallback( this );
    }
}

void SequenceSoundPlayer::SetSkipIntervalTick( s32 intervalTick )
{
    m_SkipIntervalTickPerFrame = intervalTick;
}

s32 SequenceSoundPlayer::GetSkipIntervalTick()
{
    return m_SkipIntervalTickPerFrame;
}







SequenceSoundLoader::LoadInfo::LoadInfo(
    const SoundArchive* arc,
    const SoundDataManager* mgr,
    LoadItemInfo* seq,
    LoadItemInfo* banks,
    SoundPlayer* player)
: soundArchive(arc)
, soundDataManager(mgr)
, loadInfoSeq(seq)
, soundPlayer(player)
{
    for (int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++)
    {
        loadInfoBanks[i] = &banks[i];
    }
}

void SequenceSoundLoader::Initialize(const Arg& arg)
{
    m_Task.Initialize();
    m_Task.m_Arg = arg;
    m_Task.m_pPlayerHeap = NULL;
}

void SequenceSoundLoader::Finalize()
{
    m_Task.FreePlayerHeap();
}

bool SequenceSoundLoader::TryWait()
{
    // プレイヤーヒープの確保
    if (m_Task.TryAllocPlayerHeap() == false)
    {
        return false;
    }

    // データのアペンド
    Task::Status status = m_Task.GetStatus();
    switch (status)
    {
    case Task::STATUS_FREE:
        TaskManager::GetInstance().AppendTask(&m_Task, TaskManager::PRIORITY_MIDDLE);
        return false;
    case Task::STATUS_DONE:
    case Task::STATUS_CANCEL:
        return true;
    default:
        return false;
    }
}

bool SequenceSoundLoader::IsInUse()
{
    if (m_Task.TryWait())
    {
        return false;
    }
    return true;
}







void SequenceSoundLoader::DataLoadTask::Initialize()
{
    Task::InitializeStatus();
    m_Data.Initialize();
    m_IsLoadSuccess = false;
}

bool SequenceSoundLoader::DataLoadTask::TryAllocPlayerHeap()
{
    if (m_pPlayerHeap == NULL)
    {
        NW_ASSERT_NOT_NULL(m_Arg.soundPlayer);
        m_pPlayerHeap = m_Arg.soundPlayer->detail_AllocPlayerHeap();
        if (m_pPlayerHeap == NULL)
        {
            return false;
        }
        m_pPlayerHeap->Clear();
    }
    return true;
}

void SequenceSoundLoader::DataLoadTask::FreePlayerHeap()
{
    if (m_pPlayerHeap)
    {
        m_Arg.soundPlayer->detail_FreePlayerHeap(m_pPlayerHeap);
    }
}

void SequenceSoundLoader::DataLoadTask::Execute()
{
    NW_ASSERT_NOT_NULL(m_pPlayerHeap);

    PlayerHeapDataManager playerHeapDataManager;
    playerHeapDataManager.Initialize(m_Arg.soundArchive);

    if ( ( m_Arg.loadInfoSeq.address == NULL ) &&
         ( m_Arg.loadInfoSeq.itemId != SoundArchive::INVALID_ID ) )
    {
        SoundArchive::ItemId soundId = m_Arg.loadInfoSeq.itemId;

        if ( ! playerHeapDataManager.LoadData(
                    soundId, m_pPlayerHeap, SoundArchiveLoader::LOAD_SEQ ) )
        {
            NW_WARNING( false, "failed to load SEQ(%08x) to PlayerHeap", soundId );
            m_pPlayerHeap->SetState(PlayerHeap::STATE_TASK_FINISHED);
            m_IsLoadSuccess = false;
            return;
        }
        m_Arg.loadInfoSeq.address =
            playerHeapDataManager.detail_GetFileAddressByItemId( soundId );
    }

    NW_ASSERT_NOT_NULL(m_Arg.loadInfoSeq.address);

    LoadItemInfo warcInfos[SEQ_BANK_MAX];
    bool isLoadIndividuals[SEQ_BANK_MAX] = {false};

    for ( int i = 0; i < SEQ_BANK_MAX; i++ )
    {
        if ( ( m_Arg.loadInfoBanks[i].address == NULL ) &&
             ( m_Arg.loadInfoBanks[i].itemId != SoundArchive::INVALID_ID ) )
        {
            SoundArchive::ItemId bankId = m_Arg.loadInfoBanks[i].itemId;
            if ( ! playerHeapDataManager.LoadData(
                        bankId, m_pPlayerHeap, SoundArchiveLoader::LOAD_BANK ) )
            {
                // ロードリクエストがあるのにロードできなかった場合
                NW_WARNING( false, "failed to load BANK(%08x) to PlayerHeap", bankId );
                m_pPlayerHeap->SetState(PlayerHeap::STATE_TASK_FINISHED);
                m_IsLoadSuccess = false;
                return;
            }
            m_Arg.loadInfoBanks[i].address =
                playerHeapDataManager.detail_GetFileAddressByItemId( bankId );
        }

        if ( m_Arg.loadInfoBanks[i].itemId != SoundArchive::INVALID_ID &&
             m_Arg.loadInfoBanks[i].address != NULL )
        {
            // SoundDataManager でロード済みか？
            internal::Util::WaveArchiveLoadStatus status = Util::GetWaveArchiveOfBank(
                    warcInfos[i],
                    isLoadIndividuals[i],
                    m_Arg.loadInfoBanks[i].address,
                    *m_Arg.soundArchive,
                    *m_Arg.soundDataManager );

            switch ( status )
            {
            case internal::Util::WARC_LOAD_OK:
            case internal::Util::WARC_LOAD_NONEED:
                // なにもしない
                break;
            case internal::Util::WARC_LOAD_PARTLY:
            case internal::Util::WARC_LOAD_NOT_YET:
                // ロードを試みる
                if ( ! playerHeapDataManager.detail_LoadWaveArchiveByBankFile(
                            m_Arg.loadInfoBanks[i].address, m_pPlayerHeap ) )
                {
                    NW_WARNING( false,
                        "failed to load WARC of BANK(%08x) to PlayerHeap",
                            m_Arg.loadInfoBanks[i].itemId );
                    m_pPlayerHeap->SetState(PlayerHeap::STATE_TASK_FINISHED);
                    m_IsLoadSuccess = false;
                    return;
                }
                // ロード成功時、warcInfos と isLoadIndividuals を埋める
                {
                    status = Util::GetWaveArchiveOfBank(
                        warcInfos[i],
                        isLoadIndividuals[i],
                        m_Arg.loadInfoBanks[i].address,
                        *m_Arg.soundArchive,
                        playerHeapDataManager );
                }
                break;
            case internal::Util::WARC_LOAD_ERROR:
            default:
                NW_ASSERT( false ); // ここに来ることは無いはず
                break;
            }
        }
    }

    m_Data.seqFile = m_Arg.loadInfoSeq.address;
    for ( int i = 0; i < SEQ_BANK_MAX; i++ )
    {
        m_Data.bankFiles[i] = m_Arg.loadInfoBanks[i].address;
        m_Data.warcFiles[i] = warcInfos[i].address;
        m_Data.warcIsIndividuals[i] = isLoadIndividuals[i];
    }

    m_pPlayerHeap->SetState(PlayerHeap::STATE_TASK_FINISHED);
    playerHeapDataManager.Finalize();
    m_IsLoadSuccess = true;
}



} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

