﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_RemoteSpeaker.h>
#include <nw/snd/snd_RemoteSpeakerManager.h>
#include <nw/assert.h>
#include <nw/ut/ut_Inlines.h>

namespace nw {
namespace snd {

/*---------------------------------------------------------------------------*
  Name:         RemoteSpeaker

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
RemoteSpeaker::RemoteSpeaker()
: m_State( STATE_INVALID )
#if defined( NW_PLATFORM_CAFE )
, m_InitFlag( false )
, m_PlayFlag( false )
, m_EnableFlag( false )
, m_WpadCallback( NULL )
, m_FirstEncodeFlag( false )
, m_CommandBusyFlag( false )
, m_ValidCallbackFlag( false )
, m_UserCommand( COMMAND_NONE )
, m_InternalCommand( COMMAND_NONE )
#endif // defined( NW_PLATFORM_CAFE )
{
#if defined( NW_PLATFORM_CAFE )
    OSCreateAlarm( &m_ContinueAlarm );
    OSSetAlarmUserData( &m_ContinueAlarm, this );

    OSCreateAlarm( &m_InvervalAlarm );
    OSSetAlarmUserData( &m_InvervalAlarm, this );
#endif // defined( NW_PLATFORM_CAFE )
}

#if defined( NW_PLATFORM_CAFE )
/*---------------------------------------------------------------------------*
  Name:         InitParam

  Description:  パラメータの初期化を行う

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void RemoteSpeaker::InitParam()
{
    // アラーム等をクリア
    ClearParam();

    // パラメータ初期化
    m_PlayFlag = false;
    m_EnableFlag = true;
    m_ContinueFlag = false;
    m_IntervalFlag = false;
}

/*---------------------------------------------------------------------------*
  Name:         ClearParam

  Description:  パラメータをクリアし、アラームをキャンセル

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void RemoteSpeaker::ClearParam()
{
    m_PlayFlag = false;
    m_EnableFlag = false;

    OSCancelAlarm( &m_ContinueAlarm );
    m_ContinueFlag = false;
    OSCancelAlarm( &m_InvervalAlarm );
    m_IntervalFlag = false;
}

/*---------------------------------------------------------------------------*
  Name:         Setup

  Description:  リモコンのセットアップを行う

  Arguments:    callback - 完了コールバック

  Returns:      None.
 *---------------------------------------------------------------------------*/
bool RemoteSpeaker::Initialize( WpadCallback callback )
{
    // ut::AutoInterruptLock lock;  // TODO:必須？

    InitParam();

    // 現在実行中のコマンドのCallbackが設定されている場合、
    // そのコマンドはキャンセルされるので、Callbackをここで返す。
    if ( m_WpadCallback )
    {
        m_WpadCallback( m_ChannelIndex, WPAD_ERR_NONE );
        m_ValidCallbackFlag = false;
    }

    // コマンドを登録
    m_UserCommand = COMMAND_SPEAKER_ON;
    m_WpadCallback = callback;

    m_InitFlag = true;
    return true;
}

/*---------------------------------------------------------------------------*
  Name:         Shutdown

  Description:  リモコンのシャットダウンを行う

  Arguments:    callback - 完了コールバック

  Returns:      None.
 *---------------------------------------------------------------------------*/
void RemoteSpeaker::Finalize( WpadCallback callback )
{
    // ut::AutoInterruptLock lock;  // TODO:必須？

    ClearParam();

    // 現在実行中のコマンドのCallbackが設定されている場合、
    // そのコマンドはキャンセルされるので、Callbackをここで返す。
    if ( m_WpadCallback )
    {
        m_WpadCallback( m_ChannelIndex, WPAD_ERR_NONE );
        m_ValidCallbackFlag = false;
    }

    m_UserCommand = COMMAND_SPEAKER_OFF;
    m_WpadCallback = callback;

    m_InitFlag = false;
}

/*---------------------------------------------------------------------------*
  Name:         EnableOutput

  Description:  リモコンのスピーカーにサウンドデータを送るかどうか設定する

  Arguments:    enable - データを送るなら true

  Returns:      None.
 *---------------------------------------------------------------------------*/
bool RemoteSpeaker::EnableOutput( bool enable )
{
    if ( ! m_InitFlag ) return false;
    m_EnableFlag = enable;
    return true;
}

/*---------------------------------------------------------------------------*
  Name:         IsEnabledOutput

  Description:  リモコンのスピーカーにデータを送る状態かどうかを取得する

  Arguments:    None.

  Returns:      データを送る状態なら true
 *---------------------------------------------------------------------------*/
bool RemoteSpeaker::IsEnabledOutput() const
{
    if ( ! m_InitFlag ) return false;
    return m_EnableFlag;
}

/*---------------------------------------------------------------------------*
  Name:         IsPlaying

  Description:  リモコンのスピーカーで再生しているデータがあるかどうかを取得する

  Arguments:    None.

  Returns:      再生中なら true
 *---------------------------------------------------------------------------*/
bool RemoteSpeaker::IsPlaying() const
{
    if ( ! IsAvailable() ) return false;
    return m_PlayFlag;
}

/*---------------------------------------------------------------------------*
  Name:         GetContinuousPlayTime

  Description:  連続して再生している時間を取得する

  Arguments:    None.

  Returns:      連続再生時間（ミリ秒）
 *---------------------------------------------------------------------------*/
u32 RemoteSpeaker::GetContinuousPlayTime() const
{
    if ( ! IsAvailable() ) return 0;
    if ( ! m_ContinueFlag ) return 0;
    return static_cast<u32>(
        OSTicksToMilliseconds( OSGetTime() - m_ContinueBeginTime )
    );
}

/*---------------------------------------------------------------------------*
  Name:         Update

  Description:  フレーム処理(フレーム間隔は6.6667ms)

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void RemoteSpeaker::Update()
{
    // コマンド処理
    if ( ! m_CommandBusyFlag )
    {
        // ユーザーコマンドがある場合は内部コマンドより優先する
        SpeakerCommand cmd = ( m_UserCommand != COMMAND_NONE )? m_UserCommand: m_InternalCommand;

        // コマンドクリア
        m_UserCommand = COMMAND_NONE;
        m_InternalCommand = COMMAND_NONE;

        // コマンド実行
        ExecCommand( cmd );
    }
}

/*---------------------------------------------------------------------------*
  Name:         ExecCommand

  Description:  コマンド処理を行う

  Arguments:    cmd - コマンド

  Returns:      None.
 *---------------------------------------------------------------------------*/
void RemoteSpeaker::ExecCommand( SpeakerCommand cmd )
{
    switch ( cmd )
    {
    case COMMAND_NONE:
        // nothing to do.
        break;

    case COMMAND_SPEAKER_ON:
        m_ValidCallbackFlag = true;
        m_CommandBusyFlag = true;
        m_State = STATE_EXEC_SPEAKER_ON;
        WPADControlSpeaker( m_ChannelIndex, WPAD_SPEAKER_ON, SpeakerOnCallback );
        break;

    case COMMAND_SPEAKER_PLAY:
        m_ValidCallbackFlag = true;
        m_CommandBusyFlag = true;
        m_State = STATE_EXEC_SPEAKER_PLAY;
        WPADControlSpeaker( m_ChannelIndex, WPAD_SPEAKER_PLAY, SpeakerPlayCallback );
        break;

    case COMMAND_SPEAKER_OFF:
        m_ValidCallbackFlag = true;
        m_CommandBusyFlag = true;
        m_State = STATE_EXEC_SPEAKER_OFF;
        WPADControlSpeaker( m_ChannelIndex, WPAD_SPEAKER_OFF, SpeakerOffCallback );
        break;

    default:
        NW_ASSERTMSG( false, "invalid command." );
        break;
    }
}

/*---------------------------------------------------------------------------*
  Name:         UpdateStreamData

  Description:  リモコンに波形データを送る

  Arguments:    axRemoteSamples - 波形バッファ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void RemoteSpeaker::UpdateStreamData( const s16* axRemoteSamples )
{
    NW_ASSERT( IsAvailable() );
    if ( ! IsAvailable() ) return;

    // このフレームで出力するかどうか
    bool playFlag = true;
    bool silentFlag = m_EnableFlag? IsAllSampleZero( axRemoteSamples ): true;
    if ( silentFlag /* || mForceResumeFlag */ )
    {
        playFlag = false;
    }

    // WENCでエンコードするタイプを決定する
    bool firstFlag = ( ! m_PlayFlag ) && playFlag;
    bool lastFlag = m_PlayFlag && ( ! playFlag );

    // リモコンにデータを出力するための処理
    if ( playFlag )
    {
        // ut::AutoInterruptLock lock;  // TODO:必須？

        if ( ! WPADCanSendStreamData( m_ChannelIndex ) )
        {
            return;
        }

        u8 adpcmBuffer[ WENCGetEncodeBufferSize( SAMPLES_PER_AUDIO_PACKET ) ];
        static const u16 AUDIO_PACKET_SIZE = 20;

        // エンコードモード設定
        u32 wencMode;
        if ( m_FirstEncodeFlag ) wencMode =  WENC_FLAG_FIRST;
        else wencMode = WENC_FLAG_CONT;
        m_FirstEncodeFlag = false;

        // データをエンコードする
        int encSampleCount = WENCGetEncodeData(
            &m_EncodeInfo,
            wencMode,
            axRemoteSamples,
            SAMPLES_PER_AUDIO_PACKET,
            adpcmBuffer
        );

        // リモコンにストリームデータを送る
        s32 result = WPADSendStreamData( m_ChannelIndex, adpcmBuffer, AUDIO_PACKET_SIZE );

        // WPADCanSendStreamData でチェックしてるので WPAD_ERR_NONE 以外は返らないはず
        // 帰ってきた場合はスピーカー再起動
        NW_WARNING( result == WPAD_ERR_NONE, "WPADSendStreamData failed. %d", result );
        if ( result != WPAD_ERR_NONE )
        {
            m_InternalCommand = COMMAND_SPEAKER_ON;
            m_State = STATE_INVALID;
            InitParam();
            return;
        }
    }

    // FIRST の時の処理
    if ( firstFlag )
    {
        // ut::AutoInterruptLock lock;  // TODO:必須？
        if ( ! m_ContinueFlag )
        {
            OSSetAlarm(
                &m_ContinueAlarm,
                OSSecondsToTicks( static_cast<s64>( CONTINUOUS_PLAY_MINUTES ) ),
                ContinueAlarmHandler
            );
            m_ContinueBeginTime = OSGetTime();
            m_ContinueFlag = true;
        }

        OSCancelAlarm( &m_InvervalAlarm );
        m_IntervalFlag = false;
    }

    // LAST の時の処理
    if ( lastFlag )
    {
        // ut::AutoInterruptLock lock;  // TODO:必須？

        m_IntervalFlag = true;
        OSCancelAlarm( &m_InvervalAlarm );
        OSSetAlarm(
            &m_InvervalAlarm,
            OSSecondsToTicks( static_cast<s64>( CONTINUOUS_PLAY_INTERVAL_MINUTES ) ),
            IntervalAlarmHandler
        );
    }

    // 再生状態更新
    m_PlayFlag = playFlag;
}

/*---------------------------------------------------------------------------*
  Name:         IsAllSampleZero

  Description:  バッファの中身が全部ゼロかどうか調べる

  Arguments:    axRemoteSamples - 波形バッファ

  Returns:      全部ゼロなら true
 *---------------------------------------------------------------------------*/
bool RemoteSpeaker::IsAllSampleZero( const s16* axRemoteSamples )
{
    bool zeroDataFlag = true;
    const u32* buf = reinterpret_cast<const u32*>( axRemoteSamples );
    const u32 n = SAMPLES_PER_AUDIO_PACKET * sizeof ( s16 ) / sizeof( u32 );
    for ( int i=0; i<n; i++ )
    {
        if ( buf[i] != 0 )
        {
            zeroDataFlag = false;
            break;
        }
    }
    return zeroDataFlag;
}

/*---------------------------------------------------------------------------*
  Name:         SpeakerOnCallback

  Description:  リモコンのスピーカーをONにしたときのコールバック。

  Arguments:    channel:    リモコンのチャンネル番号
                result:     コールバックの結果

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void RemoteSpeaker::SpeakerOnCallback( s32 channel, s32 result )
{
    RemoteSpeaker& remote = internal::RemoteSpeakerManager::GetInstance().GetRemoteSpeaker( channel );

    switch ( result )
    {
    case WPAD_ERR_NONE:
        // エンコーダ初期化
        remote.m_FirstEncodeFlag = true;
        ut::internal::Memset( &remote.m_EncodeInfo, 0, sizeof( WENCInfo ) );

        // 引き続きPLAY状態へ移行
        remote.m_State = STATE_SPEAKER_ON;
        remote.m_InternalCommand = COMMAND_SPEAKER_PLAY;
        break;
    case WPAD_ERR_BUSY:
        // retry
        remote.m_InternalCommand = COMMAND_SPEAKER_ON;
        break;
    case WPAD_ERR_TRANSFER:
        NW_WARNING( false, "failed WPAD_SPEAKER_ON");
        remote.m_State = STATE_INVALID;
        break;
    case WPAD_ERR_NO_CONTROLLER:
        remote.m_State = STATE_INVALID;
        break;
    default:
        NW_ASSERTMSG( false, "unknown WPADCallback result code." );
        remote.m_State = STATE_INVALID;
        break;
    }

    // コールバック通知
    if ( ( result != WPAD_ERR_NONE ) && ( result != WPAD_ERR_BUSY ) )
    {
        remote.NotifyCallback( channel, result );
    }

    remote.m_CommandBusyFlag = false;
}

/*---------------------------------------------------------------------------*
  Name:         SpeakerPlayCallback

  Description:  リモコンのスピーカーをPLAY状態にしたときのコールバック。

  Arguments:    channel:    リモコンのチャンネル番号
                result:     コールバックの結果

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void RemoteSpeaker::SpeakerPlayCallback( s32 channel, s32 result )
{
    RemoteSpeaker& remote = internal::RemoteSpeakerManager::GetInstance().GetRemoteSpeaker( channel );

    switch ( result )
    {
    case WPAD_ERR_NONE:
        remote.m_State = STATE_SPEAKER_PLAY;
        break;
    case WPAD_ERR_BUSY:
        // retry
        remote.m_InternalCommand = COMMAND_SPEAKER_PLAY;
        break;
    case WPAD_ERR_TRANSFER:
        NW_WARNING( false, "failed WPAD_SPEAKER_PLAY");
        remote.m_State = STATE_INVALID;
        break;
    case WPAD_ERR_NO_CONTROLLER:
        remote.m_State = STATE_INVALID;
        break;
    default:
        NW_ASSERTMSG( false, "unknown WPADCallback result code." );
        remote.m_State = STATE_INVALID;
        break;
    }

    // コールバック通知
    if ( result != WPAD_ERR_BUSY )
    {
        remote.NotifyCallback( channel, result );
    }

    remote.m_CommandBusyFlag = false;
}

/*---------------------------------------------------------------------------*
  Name:         SpeakerOffCallback

  Description:  リモコンのスピーカーをONにしたときのコールバック。

  Arguments:    channel:    リモコンのチャンネル番号
                result:     コールバックの結果

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void RemoteSpeaker::SpeakerOffCallback( s32 channel, s32 result )
{
    RemoteSpeaker& remote = internal::RemoteSpeakerManager::GetInstance().GetRemoteSpeaker( channel );

    switch ( result )
    {
    case WPAD_ERR_NONE:
        remote.m_State = STATE_SPEAKER_OFF;
        break;
    case WPAD_ERR_BUSY:
        // retry
        remote.m_InternalCommand = COMMAND_SPEAKER_OFF;
        break;
    case WPAD_ERR_TRANSFER:
        NW_WARNING( false, "failed WPAD_SPEAKER_OFF");
        remote.m_State = STATE_INVALID;
        break;
    case WPAD_ERR_NO_CONTROLLER:
        remote.m_State = STATE_INVALID;
        break;
    default:
        NW_ASSERTMSG( false, "unknown WPADCallback result code." );
        remote.m_State = STATE_INVALID;
        break;
    }

    // コールバック通知
    if ( result != WPAD_ERR_BUSY )
    {
        remote.NotifyCallback( channel, result );
    }

    remote.m_CommandBusyFlag = false;
}

/*---------------------------------------------------------------------------*
  Name:         NotifyCallback

  Description:  コールバックを呼ぶ

  Arguments:    channel:    リモコンのチャンネル番号
                result:     コールバックの結果

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void RemoteSpeaker::NotifyCallback( s32 channel, s32 result )
{
    if ( m_ValidCallbackFlag && ( m_WpadCallback != NULL ) )
    {
        m_WpadCallback( channel, result );
        m_WpadCallback = NULL;
    }
}

/*---------------------------------------------------------------------------*
  Name:         ContinueAlarmHandler

  Description:  連続再生時間計測用のアラームハンドラ

  Arguments:    alarm - アラーム

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void RemoteSpeaker::ContinueAlarmHandler( OSAlarm* alarm, OSContext* context )
{
    (void)context;

    // ut::AutoInterruptLock lock;  // TODO:必須？
    RemoteSpeaker* remote = reinterpret_cast<RemoteSpeaker*>( OSGetAlarmUserData( alarm ) );

    // 制限時間になったので強制的に停止する
//    NW_WARNING( false, "Playing time of remote speaker(%d) is over 8 minutes.", remote->m_ChannelIndex );
//    remote->mForceResumeFlag = true;
//    remote->m_ContinueFlag = false;
}

/*---------------------------------------------------------------------------*
  Name:         IntervalAlarmHandler

  Description:  間隔時間計測用のアラームハンドラ

  Arguments:    alarm - アラーム

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void RemoteSpeaker::IntervalAlarmHandler( OSAlarm* alarm, OSContext* context )
{
    (void)context;

    // ut::AutoInterruptLock lock;  // TODO:必須？
    RemoteSpeaker* remote = reinterpret_cast<RemoteSpeaker*>( OSGetAlarmUserData( alarm ) );

    if ( remote->m_IntervalFlag )
    {
        // インターバルを完了した
        OSCancelAlarm( &remote->m_ContinueAlarm );
//        remote->mForceResumeFlag = false;
        remote->m_ContinueFlag = false;
    }
    remote->m_IntervalFlag = false;
}
#endif // defined( NW_PLATFORM_CAFE )

} // namespace nw::snd
} // namespace nw

