﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_NwVoice.h>
#include <nw/snd/snd_Config.h>
#include <nw/types.h>
#include <nw/assert.h>
#include <nw/ut.h>
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_HardwareManager.h>
#include <nw/snd/snd_DecodeAdpcm.h>
#include <nw/snd/snd_VoiceCommand.h>

#if defined( NW_PLATFORM_CAFE )
  #include <cafe.h>
#elif defined( NW_PLATFORM_WIN32 )
  #include <winext/cafe.h>
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
  #include <winext/cafe.h>
#elif defined( NW_USE_NINTENDO_SDK )
  // TODO: nn_audio
  #include <winext/cafe.h>
#else
  #error
#endif

#if defined( NW_PLATFORM_WIN32 )
using namespace nw::internal::winext;
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
using namespace nw::internal::winext;
#elif defined( NW_USE_NINTENDO_SDK )
// TODO: nn_audio
using namespace nw::internal::winext;
#endif



namespace nw {
namespace snd {
namespace internal {

namespace {

u16 GetVolumeU16( f32 vol )
{
    const u16 BASE_VOLUME = 0x8000;
    const u16 VOLUME_MIN = 0;
    const u16 VOLUME_MAX = 0xffff;

    return static_cast<u16>( nw::ut::Clamp<f32>(vol * BASE_VOLUME, VOLUME_MIN, VOLUME_MAX) );
}

} // anonymous namespace

//==================================================================
//
// NwVoice
//
//==================================================================

namespace
{
ut::CriticalSection s_CriticalSection;
}

//==========================================================================================
//
// NwVoiceSynthesizeBuffer
//
//==========================================================================================

NwVoiceSynthesizeBuffer::NwVoiceSynthesizeBuffer()
    : flag(0)
    , sampleCount(0)
    , eventTv(true,true)
    , eventDrc0(true,true)
    , eventDrc1(true,true)
{
    tvMix.frontLeft = NULL;
    tvMix.frontRight = NULL;
    tvMix.rearLeft = NULL;
    tvMix.rearRight = NULL;
    tvMix.frontCenter = NULL;
    tvMix.lfe = NULL;

    tvMix.sampleCount = 0;
    tvMix.channelCount = 0;

    drc0Mix.frontLeft = NULL;
    drc0Mix.frontRight = NULL;
    drc0Mix.rearLeft = NULL;
    drc0Mix.rearRight = NULL;
    drc0Mix.frontCenter = NULL;
    drc0Mix.lfe = NULL;

    drc0Mix.sampleCount = 0;
    drc0Mix.channelCount = 0;

    drc1Mix.frontLeft = NULL;
    drc1Mix.frontRight = NULL;
    drc1Mix.rearLeft = NULL;
    drc1Mix.rearRight = NULL;
    drc1Mix.frontCenter = NULL;
    drc1Mix.lfe = NULL;

    drc1Mix.sampleCount = 0;
    drc1Mix.channelCount = 0;
}

void NwVoiceSynthesizeBuffer::Initialize( void* synthesizeBuffer, u32 synthesizeBufferSize )
{
    NW_ASSERT_ALIGN( synthesizeBuffer, CACHE_BLOCK_SIZE );
    NW_ASSERT( synthesizeBufferSize >= GetSynthesizeBufferSize() );

    f32* bufferp = reinterpret_cast<f32*>(synthesizeBuffer);

    tvMix.frontLeft = bufferp; bufferp += SAMPLE_PER_FRAME;
    tvMix.frontRight = bufferp; bufferp += SAMPLE_PER_FRAME;
    tvMix.rearLeft = bufferp; bufferp += SAMPLE_PER_FRAME;
    tvMix.rearRight = bufferp; bufferp += SAMPLE_PER_FRAME;
    tvMix.frontCenter = bufferp; bufferp += SAMPLE_PER_FRAME;
    tvMix.lfe = bufferp; bufferp += SAMPLE_PER_FRAME;

    drc0Mix.frontLeft = bufferp; bufferp += SAMPLE_PER_FRAME;
    drc0Mix.frontRight = bufferp; bufferp += SAMPLE_PER_FRAME;
    drc0Mix.rearLeft = bufferp; bufferp += SAMPLE_PER_FRAME;
    drc0Mix.rearRight = bufferp; bufferp += SAMPLE_PER_FRAME;

    drc1Mix.frontLeft = bufferp; bufferp += SAMPLE_PER_FRAME;
    drc1Mix.frontRight = bufferp; bufferp += SAMPLE_PER_FRAME;
    drc1Mix.rearLeft = bufferp; bufferp += SAMPLE_PER_FRAME;
    drc1Mix.rearRight = bufferp; bufferp += SAMPLE_PER_FRAME;

    NW_ASSERT( bufferp <= ut::AddOffsetToPtr(synthesizeBuffer, synthesizeBufferSize) );

    tvMix.sampleCount = SAMPLE_PER_FRAME;
    tvMix.channelCount = CHANNEL_COUNT_TV;

    drc0Mix.sampleCount = SAMPLE_PER_FRAME;
    drc0Mix.channelCount = CHANNEL_COUNT_DRC;

    drc1Mix.sampleCount = SAMPLE_PER_FRAME;
    drc1Mix.channelCount = CHANNEL_COUNT_DRC;

    eventTv.Signal();
    eventDrc0.Signal();
    eventDrc1.Signal();

    sampleCount = 0;
}

void NwVoiceSynthesizeBuffer::Finalize()
{
    eventTv.Wait();
    eventDrc0.Wait();
    eventDrc1.Wait();
}

//==========================================================================================
//
// NwVoice
//
//==========================================================================================


s16 NwVoice::s_DecodeWaveBuffer[ DECODE_WAVE_SAMPLES + HISTORY_WAVE_SAMPLES ];

const s16 NwVoice::s_SrcCoef[SRC_TYPE_COUNT][SRC_COEF_TAP_COUNT * SRC_COEF_ELEMENT_COUNT] =
{
    // SRC_TYPE_2_4
    {
        6600,   19426,    6722,       3,
        6479,   19424,    6845,       9,
        6359,   19419,    6968,      15,
        6239,   19412,    7093,      22,
        6121,   19403,    7219,      28,
        6004,   19391,    7345,      34,
        5888,   19377,    7472,      41,
        5773,   19361,    7600,      48,
        5659,   19342,    7728,      55,
        5546,   19321,    7857,      62,
        5434,   19298,    7987,      69,
        5323,   19273,    8118,      77,
        5213,   19245,    8249,      84,
        5104,   19215,    8381,      92,
        4997,   19183,    8513,     101,
        4890,   19148,    8646,     109,
        4785,   19112,    8780,     118,
        4681,   19073,    8914,     127,
        4579,   19031,    9048,     137,
        4477,   18988,    9183,     147,
        4377,   18942,    9318,     157,
        4277,   18895,    9454,     168,
        4179,   18845,    9590,     179,
        4083,   18793,    9726,     190,
        3987,   18738,    9863,     202,
        3893,   18682,   10000,     215,
        3800,   18624,   10137,     228,
        3709,   18563,   10274,     241,
        3618,   18500,   10411,     255,
        3529,   18436,   10549,     270,
        3441,   18369,   10687,     285,
        3355,   18300,   10824,     300,
        3269,   18230,   10962,     317,
        3186,   18157,   11100,     334,
        3103,   18082,   11238,     351,
        3022,   18006,   11375,     369,
        2942,   17927,   11513,     388,
        2863,   17847,   11650,     408,
        2785,   17765,   11788,     428,
        2709,   17681,   11925,     449,
        2635,   17595,   12062,     471,
        2561,   17507,   12198,     494,
        2489,   17418,   12334,     517,
        2418,   17327,   12470,     541,
        2348,   17234,   12606,     566,
        2280,   17140,   12741,     592,
        2213,   17044,   12876,     619,
        2147,   16946,   13010,     647,
        2083,   16846,   13144,     675,
        2020,   16745,   13277,     704,
        1958,   16643,   13409,     735,
        1897,   16539,   13541,     766,
        1838,   16434,   13673,     798,
        1780,   16327,   13803,     832,
        1723,   16218,   13933,     866,
        1667,   16109,   14062,     901,
        1613,   15998,   14191,     937,
        1560,   15885,   14318,     975,
        1508,   15772,   14445,    1013,
        1457,   15657,   14571,    1052,
        1407,   15540,   14695,    1093,
        1359,   15423,   14819,    1134,
        1312,   15304,   14942,    1177,
        1266,   15185,   15064,    1221,
        1221,   15064,   15185,    1266,
        1177,   14942,   15304,    1312,
        1134,   14819,   15423,    1359,
        1093,   14695,   15540,    1407,
        1052,   14571,   15657,    1457,
        1013,   14445,   15772,    1508,
        975,   14318,   15885,    1560,
        937,   14191,   15998,    1613,
        901,   14062,   16109,    1667,
        866,   13933,   16218,    1723,
        832,   13803,   16327,    1780,
        798,   13673,   16434,    1838,
        766,   13541,   16539,    1897,
        735,   13409,   16643,    1958,
        704,   13277,   16745,    2020,
        675,   13144,   16846,    2083,
        647,   13010,   16946,    2147,
        619,   12876,   17044,    2213,
        592,   12741,   17140,    2280,
        566,   12606,   17234,    2348,
        541,   12470,   17327,    2418,
        517,   12334,   17418,    2489,
        494,   12198,   17507,    2561,
        471,   12062,   17595,    2635,
        449,   11925,   17681,    2709,
        428,   11788,   17765,    2785,
        408,   11650,   17847,    2863,
        388,   11513,   17927,    2942,
        369,   11375,   18006,    3022,
        351,   11238,   18082,    3103,
        334,   11100,   18157,    3186,
        317,   10962,   18230,    3269,
        300,   10824,   18300,    3355,
        285,   10687,   18369,    3441,
        270,   10549,   18436,    3529,
        255,   10411,   18500,    3618,
        241,   10274,   18563,    3709,
        228,   10137,   18624,    3800,
        215,   10000,   18682,    3893,
        202,    9863,   18738,    3987,
        190,    9726,   18793,    4083,
        179,    9590,   18845,    4179,
        168,    9454,   18895,    4277,
        157,    9318,   18942,    4377,
        147,    9183,   18988,    4477,
        137,    9048,   19031,    4579,
        127,    8914,   19073,    4681,
        118,    8780,   19112,    4785,
        109,    8646,   19148,    4890,
        101,    8513,   19183,    4997,
        92,    8381,   19215,    5104,
        84,    8249,   19245,    5213,
        77,    8118,   19273,    5323,
        69,    7987,   19298,    5434,
        62,    7857,   19321,    5546,
        55,    7728,   19342,    5659,
        48,    7600,   19361,    5773,
        41,    7472,   19377,    5888,
        34,    7345,   19391,    6004,
        28,    7219,   19403,    6121,
        22,    7093,   19412,    6239,
        15,    6968,   19419,    6359,
        9,    6845,   19424,    6479,
        3,    6722,   19426,    6600
    },

    // SRC_TYPE_3_4
    {
        3195,   26287,    3329,     -32,
        3064,   26281,    3467,     -34,
        2936,   26270,    3608,     -38,
        2811,   26253,    3751,     -42,
        2688,   26230,    3897,     -46,
        2568,   26202,    4046,     -50,
        2451,   26169,    4199,     -54,
        2338,   26130,    4354,     -58,
        2227,   26085,    4512,     -63,
        2120,   26035,    4673,     -67,
        2015,   25980,    4837,     -72,
        1912,   25919,    5004,     -76,
        1813,   25852,    5174,     -81,
        1716,   25780,    5347,     -87,
        1622,   25704,    5522,     -92,
        1531,   25621,    5701,     -98,
        1442,   25533,    5882,    -103,
        1357,   25440,    6066,    -109,
        1274,   25342,    6253,    -115,
        1193,   25239,    6442,    -121,
        1115,   25131,    6635,    -127,
        1040,   25018,    6830,    -133,
        967,   24899,    7027,    -140,
        897,   24776,    7227,    -146,
        829,   24648,    7430,    -153,
        764,   24516,    7635,    -159,
        701,   24379,    7842,    -166,
        641,   24237,    8052,    -174,
        583,   24091,    8264,    -181,
        526,   23940,    8478,    -187,
        472,   23785,    8695,    -194,
        420,   23626,    8914,    -202,
        371,   23462,    9135,    -209,
        324,   23295,    9358,    -215,
        279,   23123,    9583,    -222,
        236,   22948,    9809,    -230,
        194,   22769,   10038,    -237,
        154,   22586,   10269,    -243,
        117,   22399,   10501,    -250,
        81,   22208,   10735,    -258,
        47,   22015,   10970,    -265,
        15,   21818,   11206,    -271,
        -16,   21618,   11444,    -277,
        -44,   21415,   11684,    -283,
        -71,   21208,   11924,    -290,
        -97,   20999,   12166,    -296,
        -121,   20786,   12409,    -302,
        -143,   20571,   12653,    -306,
        -163,   20354,   12898,    -311,
        -183,   20134,   13143,    -316,
        -201,   19911,   13389,    -321,
        -218,   19686,   13635,    -325,
        -234,   19459,   13882,    -328,
        -248,   19230,   14130,    -332,
        -261,   18998,   14377,    -335,
        -273,   18765,   14625,    -337,
        -284,   18531,   14873,    -339,
        -294,   18295,   15121,    -341,
        -302,   18057,   15369,    -341,
        -310,   17817,   15617,    -341,
        -317,   17577,   15864,    -340,
        -323,   17335,   16111,    -340,
        -328,   17092,   16357,    -338,
        -332,   16848,   16603,    -336,
        -336,   16603,   16848,    -332,
        -338,   16357,   17092,    -328,
        -340,   16111,   17335,    -323,
        -340,   15864,   17577,    -317,
        -341,   15617,   17817,    -310,
        -341,   15369,   18057,    -302,
        -341,   15121,   18295,    -294,
        -339,   14873,   18531,    -284,
        -337,   14625,   18765,    -273,
        -335,   14377,   18998,    -261,
        -332,   14130,   19230,    -248,
        -328,   13882,   19459,    -234,
        -325,   13635,   19686,    -218,
        -321,   13389,   19911,    -201,
        -316,   13143,   20134,    -183,
        -311,   12898,   20354,    -163,
        -306,   12653,   20571,    -143,
        -302,   12409,   20786,    -121,
        -296,   12166,   20999,     -97,
        -290,   11924,   21208,     -71,
        -283,   11684,   21415,     -44,
        -277,   11444,   21618,     -16,
        -271,   11206,   21818,      15,
        -265,   10970,   22015,      47,
        -258,   10735,   22208,      81,
        -250,   10501,   22399,     117,
        -243,   10269,   22586,     154,
        -237,   10038,   22769,     194,
        -230,    9809,   22948,     236,
        -222,    9583,   23123,     279,
        -215,    9358,   23295,     324,
        -209,    9135,   23462,     371,
        -202,    8914,   23626,     420,
        -194,    8695,   23785,     472,
        -187,    8478,   23940,     526,
        -181,    8264,   24091,     583,
        -174,    8052,   24237,     641,
        -166,    7842,   24379,     701,
        -159,    7635,   24516,     764,
        -153,    7430,   24648,     829,
        -146,    7227,   24776,     897,
        -140,    7027,   24899,     967,
        -133,    6830,   25018,    1040,
        -127,    6635,   25131,    1115,
        -121,    6442,   25239,    1193,
        -115,    6253,   25342,    1274,
        -109,    6066,   25440,    1357,
        -103,    5882,   25533,    1442,
        -98,    5701,   25621,    1531,
        -92,    5522,   25704,    1622,
        -87,    5347,   25780,    1716,
        -81,    5174,   25852,    1813,
        -76,    5004,   25919,    1912,
        -72,    4837,   25980,    2015,
        -67,    4673,   26035,    2120,
        -63,    4512,   26085,    2227,
        -58,    4354,   26130,    2338,
        -54,    4199,   26169,    2451,
        -50,    4046,   26202,    2568,
        -46,    3897,   26230,    2688,
        -42,    3751,   26253,    2811,
        -38,    3608,   26270,    2936,
        -34,    3467,   26281,    3064,
        -32,    3329,   26287,    3195
    },

    // SRC_TYPE_4_4
    {
        -68,   32639,      69,      -5,
        -200,   32630,     212,     -15,
        -328,   32613,     359,     -26,
        -450,   32586,     512,     -36,
        -568,   32551,     669,     -47,
        -680,   32507,     832,     -58,
        -788,   32454,    1000,     -69,
        -891,   32393,    1174,     -80,
        -990,   32323,    1352,     -92,
        -1084,   32244,    1536,    -103,
        -1173,   32157,    1724,    -115,
        -1258,   32061,    1919,    -128,
        -1338,   31956,    2118,    -140,
        -1414,   31844,    2322,    -153,
        -1486,   31723,    2532,    -167,
        -1554,   31593,    2747,    -180,
        -1617,   31456,    2967,    -194,
        -1676,   31310,    3192,    -209,
        -1732,   31157,    3422,    -224,
        -1783,   30995,    3657,    -240,
        -1830,   30826,    3897,    -256,
        -1874,   30649,    4143,    -272,
        -1914,   30464,    4393,    -289,
        -1951,   30272,    4648,    -307,
        -1984,   30072,    4908,    -325,
        -2014,   29866,    5172,    -343,
        -2040,   29652,    5442,    -362,
        -2063,   29431,    5716,    -382,
        -2083,   29203,    5994,    -403,
        -2100,   28968,    6277,    -424,
        -2114,   28727,    6565,    -445,
        -2125,   28480,    6857,    -468,
        -2133,   28226,    7153,    -490,
        -2139,   27966,    7453,    -514,
        -2142,   27700,    7758,    -538,
        -2142,   27428,    8066,    -563,
        -2141,   27151,    8378,    -588,
        -2136,   26867,    8694,    -614,
        -2130,   26579,    9013,    -641,
        -2121,   26285,    9336,    -668,
        -2111,   25987,    9663,    -696,
        -2098,   25683,    9993,    -724,
        -2084,   25375,   10326,    -753,
        -2067,   25063,   10662,    -783,
        -2049,   24746,   11000,    -813,
        -2030,   24425,   11342,    -844,
        -2009,   24100,   11686,    -875,
        -1986,   23771,   12033,    -907,
        -1962,   23438,   12382,    -939,
        -1937,   23103,   12733,    -972,
        -1911,   22764,   13086,   -1005,
        -1883,   22422,   13441,   -1039,
        -1855,   22077,   13798,   -1072,
        -1825,   21729,   14156,   -1107,
        -1795,   21380,   14516,   -1141,
        -1764,   21027,   14877,   -1176,
        -1732,   20673,   15239,   -1211,
        -1700,   20317,   15602,   -1246,
        -1667,   19959,   15965,   -1282,
        -1633,   19600,   16329,   -1317,
        -1599,   19239,   16694,   -1353,
        -1564,   18878,   17058,   -1388,
        -1530,   18515,   17423,   -1424,
        -1495,   18151,   17787,   -1459,
        -1459,   17787,   18151,   -1495,
        -1424,   17423,   18515,   -1530,
        -1388,   17058,   18878,   -1564,
        -1353,   16694,   19239,   -1599,
        -1317,   16329,   19600,   -1633,
        -1282,   15965,   19959,   -1667,
        -1246,   15602,   20317,   -1700,
        -1211,   15239,   20673,   -1732,
        -1176,   14877,   21027,   -1764,
        -1141,   14516,   21380,   -1795,
        -1107,   14156,   21729,   -1825,
        -1072,   13798,   22077,   -1855,
        -1039,   13441,   22422,   -1883,
        -1005,   13086,   22764,   -1911,
        -972,   12733,   23103,   -1937,
        -939,   12382,   23438,   -1962,
        -907,   12033,   23771,   -1986,
        -875,   11686,   24100,   -2009,
        -844,   11342,   24425,   -2030,
        -813,   11000,   24746,   -2049,
        -783,   10662,   25063,   -2067,
        -753,   10326,   25375,   -2084,
        -724,    9993,   25683,   -2098,
        -696,    9663,   25987,   -2111,
        -668,    9336,   26285,   -2121,
        -641,    9013,   26579,   -2130,
        -614,    8694,   26867,   -2136,
        -588,    8378,   27151,   -2141,
        -563,    8066,   27428,   -2142,
        -538,    7758,   27700,   -2142,
        -514,    7453,   27966,   -2139,
        -490,    7153,   28226,   -2133,
        -468,    6857,   28480,   -2125,
        -445,    6565,   28727,   -2114,
        -424,    6277,   28968,   -2100,
        -403,    5994,   29203,   -2083,
        -382,    5716,   29431,   -2063,
        -362,    5442,   29652,   -2040,
        -343,    5172,   29866,   -2014,
        -325,    4908,   30072,   -1984,
        -307,    4648,   30272,   -1951,
        -289,    4393,   30464,   -1914,
        -272,    4143,   30649,   -1874,
        -256,    3897,   30826,   -1830,
        -240,    3657,   30995,   -1783,
        -224,    3422,   31157,   -1732,
        -209,    3192,   31310,   -1676,
        -194,    2967,   31456,   -1617,
        -180,    2747,   31593,   -1554,
        -167,    2532,   31723,   -1486,
        -153,    2322,   31844,   -1414,
        -140,    2118,   31956,   -1338,
        -128,    1919,   32061,   -1258,
        -115,    1724,   32157,   -1173,
        -103,    1536,   32244,   -1084,
        -92,    1352,   32323,    -990,
        -80,    1174,   32393,    -891,
        -69,    1000,   32454,    -788,
        -58,     832,   32507,    -680,
        -47,     669,   32551,    -568,
        -36,     512,   32586,    -450,
        -26,     359,   32613,    -328,
        -15,     212,   32630,    -200,
        -5,      69,   32639,     -68
    }};

NwVoice::NwVoice()
  : m_IsAppendedToList( false )
  , m_DisposeCallbackFunc( NULL )
  , m_DisposeCallbackArg( NULL )
  , m_pVoiceManager( NULL )
  , m_pVoice(NULL)
{
}

void NwVoice::Initialize( DisposeCallbackFunc func, void* arg )
{
    NW_ASSERT( !m_IsAppendedToList );

    m_State = VOICE_STATE_STOP;

    m_WaveBufferListBegin = NULL;
    m_WaveBufferListEnd = NULL;

    m_LoopCount = 0;
    m_PlayPosition = 0;
    m_PauseFlag = false;

    m_SampleRate = 32000;
    m_SampleFormat = SAMPLE_FORMAT_PCM_S16;

    m_DisposeCallbackFunc = func;
    m_DisposeCallbackArg = arg;
}

void NwVoice::Finalize()
{
    if ( m_pVoiceManager != NULL )
    {
        m_pVoiceManager->FreeVoice( this );
    }

    FreeAllWaveBuffer();

    if ( m_DisposeCallbackFunc != NULL )
    {
        m_DisposeCallbackFunc( this, m_DisposeCallbackArg );
        m_DisposeCallbackFunc = NULL;
    }
    m_pVoice = NULL;
}

void NwVoice::Free()
{
//    NW_LOG("NwVoice::Free %08x\n", this);
    Finalize();
}

void NwVoice::AppendWaveBuffer( WaveBuffer* waveBuffer )
{
    ut::ScopedLock<ut::CriticalSection> lock(s_CriticalSection);

    waveBuffer->next = NULL;
    waveBuffer->status = WaveBuffer::STATUS_WAIT;

    if ( m_WaveBufferListEnd == NULL )
    {
        m_WaveBufferListBegin = waveBuffer;
    }
    else
    {
        m_WaveBufferListEnd->next = waveBuffer;
    }
    m_WaveBufferListEnd = waveBuffer;
}

void NwVoice::FreeAllWaveBuffer()
{
    ut::ScopedLock<ut::CriticalSection> lock(s_CriticalSection);

    WaveBuffer* waveBuffer = m_WaveBufferListBegin;
    while( waveBuffer != NULL ) {
        waveBuffer->status = WaveBuffer::STATUS_DONE;
        waveBuffer = waveBuffer->next;
    }
    m_WaveBufferListBegin = NULL;
    m_WaveBufferListEnd = NULL;
}

void NwVoice::SetPriority(u32 priority)
{
    m_Priority = priority;

    if ( m_pVoiceManager != NULL )
    {
        m_pVoiceManager->UpdatePriorityOrder( this );
    }
}

void NwVoice::UpdateState( const OutputMode* outputMode )
{
    bool isRun = ( m_WaveBufferListBegin != NULL &&
                   m_WaveBufferListBegin->status == WaveBuffer::STATUS_PLAY );

    switch( m_State ) {
    case VOICE_STATE_PLAY:
        if ( isRun )
        {
            UpdateVoice( m_VoiceParam, outputMode );
        }
        else
        {
            SetupVoice( m_VoiceParam, outputMode );
        }
        break;
    case VOICE_STATE_STOP:
        m_PauseFlag = false;
        break;
    case VOICE_STATE_PAUSE:
        m_PauseFlag = true;
        break;
    default:
        NW_ASSERT(false);
        break;
    }
}

void NwVoice::SetupVoice( const VoiceParam& voiceParam, const OutputMode* outputMode )
{
    m_CurrentAddressFraction = 0;
    for ( int i=0; i<HISTORY_WAVE_SAMPLES; i++ )
    {
        m_LastSamples[i] = 0;
    }
    m_PlayPosition = 0;

    // VE
    UpdateVe( voiceParam, true );

    // TV, DRC, RMT のミックス設定
    SetupTvMix( voiceParam, outputMode[OUTPUT_DEVICE_MAIN] );
    SetupDrcMix( voiceParam, outputMode[OUTPUT_DEVICE_DRC] );
}

void NwVoice::UpdateVoice( const VoiceParam& voiceParam, const OutputMode* outputMode )
{
    // VE
    UpdateVe( voiceParam, false);

    // ミックス
    UpdateTvMix(voiceParam,outputMode[OUTPUT_DEVICE_MAIN]);
    UpdateDrcMix(voiceParam,outputMode[OUTPUT_DEVICE_DRC]);
}

bool NwVoice::UpdateDelta( s32 target, VeParam* ve )
{
    s32 current = ve->vol;
    s32 delta = ve->volDelta;

    if ( (delta == 0) && (target == current) )
    {
         return false;
    }

    current += delta * SAMPLE_PER_FRAME;

    s32 diff = target - current;
    if ( diff > 0 ) {
        delta = diff / SAMPLE_PER_FRAME;
    }
    else {
        delta = - ( (-diff) / static_cast<s32>(SAMPLE_PER_FRAME ));
    }
    if ( delta != 0 ) {
        ve->vol = static_cast<u16>(current);
        ve->volDelta = static_cast<s16>(nw::ut::Clamp<int>(delta,-32768, 32767));
    }
    else {
        ve->vol = static_cast<u16>(target);
        ve->volDelta = 0;
    }

    return true;
}

void NwVoice::CalcSetupTvMix( VeParam mix[], const OutputMix& src, bool surroundFlag )
{
    mix[CHANNEL_INDEX_FRONT_LEFT].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_LEFT]);
    mix[CHANNEL_INDEX_FRONT_LEFT].volDelta = 0;
    mix[CHANNEL_INDEX_FRONT_RIGHT].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_RIGHT]);
    mix[CHANNEL_INDEX_FRONT_RIGHT].volDelta = 0;

    if (surroundFlag)
    {
        mix[CHANNEL_INDEX_REAR_LEFT].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_REAR_LEFT]);
        mix[CHANNEL_INDEX_REAR_LEFT].volDelta = 0;
        mix[CHANNEL_INDEX_REAR_RIGHT].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_REAR_RIGHT]);
        mix[CHANNEL_INDEX_REAR_RIGHT].volDelta = 0;
    }
}

bool NwVoice::CalcUpdateTvMix( VeParam mix[], const OutputMix& src, bool surroundFlag )
{
    u32 isUpdate = 0;
    isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_LEFT] ),
            &mix[CHANNEL_INDEX_FRONT_LEFT] ) ? 1 : 0;
    isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_RIGHT] ),
            &mix[CHANNEL_INDEX_FRONT_RIGHT] ) ? 1 : 0;

    if ( surroundFlag )
    {
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_REAR_LEFT] ),
            &mix[CHANNEL_INDEX_REAR_LEFT] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_REAR_RIGHT] ),
            &mix[CHANNEL_INDEX_REAR_RIGHT] ) ? 1 : 0;
    }

    return (isUpdate != 0);
}

void NwVoice::CalcSetupDrcMix( VeParam mix0[], VeParam mix1[], const OutputMix& src, bool surroundFlag, bool frontBypass )
{
    if (surroundFlag)
    {
        // 出力モードがサラウンドの場合、RL,RR にも配分を行う
        mix0[CHANNEL_INDEX_REAR_LEFT].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_REAR_LEFT]);
        mix0[CHANNEL_INDEX_REAR_LEFT].volDelta = 0;
        mix0[CHANNEL_INDEX_REAR_RIGHT].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_REAR_RIGHT]);
        mix0[CHANNEL_INDEX_REAR_RIGHT].volDelta = 0;

        if (frontBypass == false
            || driver::HardwareManager::GetInstance().GetEndUserOutputMode(OUTPUT_DEVICE_DRC) != OUTPUT_MODE_SURROUND)
        {
            mix0[CHANNEL_INDEX_FRONT_LEFT].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_LEFT]);
            mix0[CHANNEL_INDEX_FRONT_LEFT].volDelta = 0;
            mix0[CHANNEL_INDEX_FRONT_RIGHT].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_RIGHT]);
            mix0[CHANNEL_INDEX_FRONT_RIGHT].volDelta = 0;
        }
        else
        {
            // フロントバイパス設定の場合、FL,FR は DRC1 に配分する
            mix1[CHANNEL_INDEX_FRONT_LEFT].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_LEFT]);
            mix1[CHANNEL_INDEX_FRONT_LEFT].volDelta = 0;
            mix1[CHANNEL_INDEX_FRONT_RIGHT].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_RIGHT]);
            mix1[CHANNEL_INDEX_FRONT_RIGHT].volDelta = 0;
        }
    }
    else
    {
        mix0[CHANNEL_INDEX_FRONT_LEFT].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_LEFT]);
        mix0[CHANNEL_INDEX_FRONT_LEFT].volDelta = 0;
        mix0[CHANNEL_INDEX_FRONT_RIGHT].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_RIGHT]);
        mix0[CHANNEL_INDEX_FRONT_RIGHT].volDelta = 0;
    }
}

bool NwVoice::CalcUpdateDrcMix( VeParam mix0[], VeParam mix1[], const OutputMix& src, bool surroundFlag, bool frontBypass )
{
    u32 isUpdate = 0;

    if ( surroundFlag )
    {
        // 出力モードがサラウンドの場合、RL,RR にも配分を行う
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_REAR_LEFT] ),
            &mix0[CHANNEL_INDEX_REAR_LEFT] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_REAR_RIGHT] ),
            &mix0[CHANNEL_INDEX_REAR_RIGHT] ) ? 1 : 0;
        if (frontBypass == false
            || driver::HardwareManager::GetInstance().GetEndUserOutputMode(OUTPUT_DEVICE_DRC) != OUTPUT_MODE_SURROUND)
        {
            isUpdate |= UpdateDelta(
                GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_LEFT] ),
                &mix0[CHANNEL_INDEX_FRONT_LEFT] ) ? 1 : 0;
            isUpdate |= UpdateDelta(
                GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_RIGHT] ),
                &mix0[CHANNEL_INDEX_FRONT_RIGHT] ) ? 1 : 0;
        }
        else
        {
            // フロントバイパス設定の場合、FL,FR は DRC1 に配分する
            isUpdate |= UpdateDelta(
                GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_LEFT] ),
                &mix1[CHANNEL_INDEX_FRONT_LEFT] ) ? 1 : 0;
            isUpdate |= UpdateDelta(
                GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_RIGHT] ),
                &mix1[CHANNEL_INDEX_FRONT_RIGHT] ) ? 1 : 0;
        }
    }
    else
    {
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_LEFT] ),
            &mix0[CHANNEL_INDEX_FRONT_LEFT] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_RIGHT] ),
            &mix0[CHANNEL_INDEX_FRONT_RIGHT] ) ? 1 : 0;
    }

    return (isUpdate != 0);
}

void NwVoice::SetupTvMix( const VoiceParam& voiceParam, OutputMode outputMode )
{
    CalcSetupTvMix( m_MixTv, voiceParam.m_TvMix, outputMode == nw::snd::OUTPUT_MODE_SURROUND);
}

void NwVoice::UpdateTvMix( const VoiceParam& voiceParam, OutputMode outputMode )
{
    CalcUpdateTvMix( m_MixTv, voiceParam.m_TvMix, outputMode == nw::snd::OUTPUT_MODE_SURROUND);
}

void NwVoice::SetupDrcMix( const VoiceParam& voiceParam, OutputMode outputMode )
{
    CalcSetupDrcMix( m_MixDrc[AX_DRC_ID0], m_MixDrc[AX_DRC_ID1], voiceParam.m_DrcMix[AX_DRC_ID0], outputMode == nw::snd::OUTPUT_MODE_SURROUND, voiceParam.m_DrcFrontBypass );
}

void NwVoice::UpdateDrcMix( const VoiceParam& voiceParam, OutputMode outputMode )
{
    CalcUpdateDrcMix( m_MixDrc[AX_DRC_ID0], m_MixDrc[AX_DRC_ID1], voiceParam.m_DrcMix[AX_DRC_ID0], outputMode == nw::snd::OUTPUT_MODE_SURROUND, voiceParam.m_DrcFrontBypass );
}

void NwVoice::UpdateVe( const VoiceParam& voiceParam, bool initialFlag )
{
    u16 targetVolume = GetVolumeU16( voiceParam.m_Volume );

    if ( initialFlag )
    {
        m_Ve.vol = targetVolume;
        m_Ve.volDelta = 0;
    }
    else
    {
        UpdateDelta( targetVolume, &m_Ve );
    }
}

void NwVoice::Synthesize( NwVoiceSynthesizeBuffer* buffer, f32* workBuffer, u32 samples, f32 sampleRate )
{
    if ( m_WaveBufferListBegin == NULL ) return;

    if ( m_WaveBufferListBegin->status != WaveBuffer::STATUS_PLAY )
    {
        m_WaveBufferListBegin->status = WaveBuffer::STATUS_PLAY;
    }

    {
        // Parameter
        f32 srcRatio = m_VoiceParam.m_Pitch * m_SampleRate / sampleRate;
        u32 srcType;
        if ( srcRatio > (4.0f/3.0f) ) {
            // SRC_TYPE_3_4でカットしきれない周波数成分があるため
            // 1/2カットフィルタを使用
            srcType = SRC_TYPE_2_4;
        }
        else if ( srcRatio > 1.0f ) {
            // 高周波成分をカットする必要がある
            srcType = SRC_TYPE_3_4;
        }
        else {
            // 周波数成分のカット不要
            srcType = SRC_TYPE_4_4;
        }
        const s16* srcCoef = s_SrcCoef[srcType];

        u32 acc = m_CurrentAddressFraction;
        u32 pitch = static_cast<u32>(0x00010000 * srcRatio);
        u32 total_newsamples = ( ( acc + pitch * samples ) >> 16 );
        if ( total_newsamples > DECODE_WAVE_SAMPLES )
        {
            pitch = ( ( DECODE_WAVE_SAMPLES << 16 ) - acc ) / samples;
            total_newsamples = ( ( acc + pitch * samples ) >> 16 );
        }

        // Decode
        {
            s16* dest = s_DecodeWaveBuffer;
            for (int i = 0; i < HISTORY_WAVE_SAMPLES; i++)
            {
                *(dest++) = m_LastSamples[i];
            }

            switch (m_SampleFormat)
            {
            case SAMPLE_FORMAT_DSP_ADPCM:
                DecodeAdpcm( dest, total_newsamples );
                break;
            case SAMPLE_FORMAT_PCM_S16:
                DecodePcm16( dest, total_newsamples );
                break;
            case SAMPLE_FORMAT_PCM_S8:
                DecodePcm8( dest, total_newsamples );
                break;
            default:
                return;
            }
        }

        // Resampling
        const s16* refpos = s_DecodeWaveBuffer;
        u16 vol = m_Ve.vol;

#if defined( NW_PLATFORM_CAFE ) && defined( NW_COMPILER_GHS )

#if 1 // 2サンプル一括処理
        f32* destp = workBuffer;
        NW_ASSERT_ALIGN( samples, 2 );
        static const f32 ratio = 1.0f / (32768.0f * 32768.0f);
        for(unsigned long i = 0; i < samples/2; i++)
        {
            // sample1
            acc += pitch;
            u32 news = ( acc >> 16 );
            acc &= 0xffff;

            refpos += news;

            const s16* coef1 = &srcCoef[(acc >> 9) * 4];
            u16 vol1 = vol;
            vol += m_Ve.volDelta;

            f32x2 x1 = __PS_MUL( __PSQ_L( coef1, 0, OS_FASTCAST_S16 ), __PSQ_L( refpos, 0, OS_FASTCAST_S16 ) );
            x1 = __PS_MADD( __PSQ_LX( coef1, 4, 0, OS_FASTCAST_S16 ), __PSQ_LX( refpos, 4, 0, OS_FASTCAST_S16 ), x1 );

            // sample2
            acc += pitch;
            news = ( acc >> 16 );
            acc &= 0xffff;

            refpos += news;

            const s16* coef2 = &srcCoef[(acc >> 9) * 4];
            u16 vol2 = vol;
            vol += m_Ve.volDelta;

            f32x2 x2 = __PS_MUL( __PSQ_L( coef2, 0, OS_FASTCAST_S16 ), __PSQ_L( refpos, 0, OS_FASTCAST_S16 ) );
            x2 = __PS_MADD( __PSQ_LX( coef2, 4, 0, OS_FASTCAST_S16 ), __PSQ_LX( refpos, 4, 0, OS_FASTCAST_S16 ), x2 );

            // pair
            f32x2 z = { x1[0] + x1[1], x2[0] + x2[1] };
            f32x2 volx2 = {(f32)vol1 * ratio, (f32)vol2 * ratio };
            z = __PS_MUL( z, volx2 );

            __PSQ_ST( destp, z, 0, 0 );
            destp += 2;
        }
#else // not 2サンプル一括処理
        for(unsigned long i = 0; i < samples; i++)
        {
            acc += pitch;
            u32 news = ( acc >> 16 );
            acc &= 0xffff;

            refpos += news;

            const u32 phase = (acc >> 9);
            const s16* coef = &srcCoef[phase * 4];

#if 1
            s32 smp  = coef[0] * refpos[0];
            smp += coef[1] * refpos[1];
            smp += coef[2] * refpos[2];
            smp += coef[3] * refpos[3];
            smp >>= 15;
            f32 smp_f32 = static_cast<f32>(smp);
            smp_f32 *= vol;
            smp_f32 /= 32768.0f;
#else
            f32x2 z = __PS_MUL( __PSQ_L( coef, 0, OS_FASTCAST_S16 ), __PSQ_L( refpos, 0, OS_FASTCAST_S16 ) );
            z = __PS_MADD( __PSQ_LX( coef, 4, 0, OS_FASTCAST_S16 ), __PSQ_LX( refpos, 4, 0, OS_FASTCAST_S16 ), z );
            f32 smp_f32 = z[0] + z[1];
            smp_f32 *= vol;
            smp_f32 /= 32768.0f * 32768.0f;
#endif

            workBuffer[i] = smp_f32;

            vol += m_Ve.volDelta;
        }
#endif // end 2サンプル一括処理

#else // not defined (NW_PLATFORM_CAFE)
        for(unsigned long i = 0; i < samples; i++)
        {
            acc += pitch;
            u32 news = ( acc >> 16 );
            acc &= 0xffff;

            refpos += news;

            const u32 phase = (acc >> 9);
            const s16* coef = &srcCoef[phase * 4];

            s32 smp  = coef[0] * refpos[0];
            smp += coef[1] * refpos[1];
            smp += coef[2] * refpos[2];
            smp += coef[3] * refpos[3];
            smp >>= 15;
            f32 smp_f32 = static_cast<f32>(smp);
            smp_f32 *= vol;
            smp_f32 /= 32768.0f;

            workBuffer[i] = smp_f32;

            vol += m_Ve.volDelta;
        }
#endif // end defined (NW_PLATFORM_CAFE)

        m_CurrentAddressFraction = acc;
        for (int i = 0; i < HISTORY_WAVE_SAMPLES; i++)
        {
            m_LastSamples[i] = refpos[i];
        }
    }

    // Mixing
    {
        buffer->flag |= MixSample( workBuffer, buffer->tvMix.frontLeft, samples, m_MixTv[CHANNEL_INDEX_FRONT_LEFT] ) ? NwVoiceSynthesizeBuffer::FLAG_TV_L : 0;
        buffer->flag |= MixSample( workBuffer, buffer->tvMix.frontRight, samples, m_MixTv[CHANNEL_INDEX_FRONT_RIGHT] ) ? NwVoiceSynthesizeBuffer::FLAG_TV_R : 0;
        buffer->flag |= MixSample( workBuffer, buffer->tvMix.rearLeft, samples, m_MixTv[CHANNEL_INDEX_REAR_LEFT] ) ? NwVoiceSynthesizeBuffer::FLAG_TV_SL : 0;
        buffer->flag |= MixSample( workBuffer, buffer->tvMix.rearRight, samples, m_MixTv[CHANNEL_INDEX_REAR_RIGHT] ) ? NwVoiceSynthesizeBuffer::FLAG_TV_SR : 0;
        buffer->flag |= MixSample( workBuffer, buffer->tvMix.frontCenter, samples, m_MixTv[CHANNEL_INDEX_FRONT_CENTER] ) ? NwVoiceSynthesizeBuffer::FLAG_TV_FC : 0;
        buffer->flag |= MixSample( workBuffer, buffer->tvMix.lfe, samples, m_MixTv[CHANNEL_INDEX_LFE] ) ? NwVoiceSynthesizeBuffer::FLAG_TV_LFE : 0;

        buffer->flag |= MixSample( workBuffer, buffer->drc0Mix.frontLeft, samples, m_MixDrc[AX_DRC_ID0][CHANNEL_INDEX_FRONT_LEFT] ) ? NwVoiceSynthesizeBuffer::FLAG_DRC0_L : 0;
        buffer->flag |= MixSample( workBuffer, buffer->drc0Mix.frontRight, samples, m_MixDrc[AX_DRC_ID0][CHANNEL_INDEX_FRONT_RIGHT] ) ? NwVoiceSynthesizeBuffer::FLAG_DRC0_R : 0;
        buffer->flag |= MixSample( workBuffer, buffer->drc0Mix.rearLeft, samples, m_MixDrc[AX_DRC_ID0][CHANNEL_INDEX_REAR_LEFT] ) ? NwVoiceSynthesizeBuffer::FLAG_DRC0_SL : 0;
        buffer->flag |= MixSample( workBuffer, buffer->drc0Mix.rearRight, samples, m_MixDrc[AX_DRC_ID0][CHANNEL_INDEX_REAR_RIGHT] ) ? NwVoiceSynthesizeBuffer::FLAG_DRC0_SR : 0;

        buffer->flag |= MixSample( workBuffer, buffer->drc1Mix.frontLeft, samples, m_MixDrc[AX_DRC_ID1][CHANNEL_INDEX_FRONT_LEFT] ) ? NwVoiceSynthesizeBuffer::FLAG_DRC1_L : 0;
        buffer->flag |= MixSample( workBuffer, buffer->drc1Mix.frontRight, samples, m_MixDrc[AX_DRC_ID1][CHANNEL_INDEX_FRONT_RIGHT] ) ? NwVoiceSynthesizeBuffer::FLAG_DRC1_R : 0;
        buffer->flag |= MixSample( workBuffer, buffer->drc1Mix.rearLeft, samples, m_MixDrc[AX_DRC_ID1][CHANNEL_INDEX_REAR_LEFT] ) ? NwVoiceSynthesizeBuffer::FLAG_DRC1_SL : 0;
        buffer->flag |= MixSample( workBuffer, buffer->drc1Mix.rearRight, samples, m_MixDrc[AX_DRC_ID1][CHANNEL_INDEX_REAR_RIGHT] ) ? NwVoiceSynthesizeBuffer::FLAG_DRC1_SR : 0;

        buffer->sampleCount = samples;
    }
}

bool NwVoice::MixSample( const f32* srcData, f32* mixBus, u32 samples, const VeParam& ve )
{
    if ( ve.vol == 0 && ve.volDelta == 0 )
    {
        // Mix不要
        return false;
    }

    f32* dest = mixBus;
    const f32* src = srcData;

    if ( ve.volDelta == 0 )
    {
#if defined(NW_PLATFORM_CAFE) && defined( NW_COMPILER_GHS )
        f32x2 vol = __PS_FDUP( static_cast<f32>(ve.vol) / 32768.0f );

        NW_ASSERT_ALIGN( samples, 2*8 );
        for(unsigned long i = 0; i < samples/(2*8); i++)
        {
            __PSQ_STX( dest,  0, __PS_MADD( __PSQ_LX( src,  0, 0, 0 ), vol, __PSQ_LX( dest,  0, 0, 0 ) ), 0, 0 );
            __PSQ_STX( dest,  8, __PS_MADD( __PSQ_LX( src,  8, 0, 0 ), vol, __PSQ_LX( dest,  8, 0, 0 ) ), 0, 0 );
            __PSQ_STX( dest, 16, __PS_MADD( __PSQ_LX( src, 16, 0, 0 ), vol, __PSQ_LX( dest, 16, 0, 0 ) ), 0, 0 );
            __PSQ_STX( dest, 24, __PS_MADD( __PSQ_LX( src, 24, 0, 0 ), vol, __PSQ_LX( dest, 24, 0, 0 ) ), 0, 0 );

            __PSQ_STX( dest, 32, __PS_MADD( __PSQ_LX( src, 32, 0, 0 ), vol, __PSQ_LX( dest, 32, 0, 0 ) ), 0, 0 );
            __PSQ_STX( dest, 40, __PS_MADD( __PSQ_LX( src, 40, 0, 0 ), vol, __PSQ_LX( dest, 40, 0, 0 ) ), 0, 0 );
            __PSQ_STX( dest, 48, __PS_MADD( __PSQ_LX( src, 48, 0, 0 ), vol, __PSQ_LX( dest, 48, 0, 0 ) ), 0, 0 );
            __PSQ_STX( dest, 56, __PS_MADD( __PSQ_LX( src, 56, 0, 0 ), vol, __PSQ_LX( dest, 56, 0, 0 ) ), 0, 0 );

            dest += 2*8;
            src += 2*8;
        }
#else
        const f32 vol = static_cast<f32>(ve.vol) / 32768.0f;

        for(unsigned long i = 0; i < samples; i++)
        {
            (*dest++) += (*src++) * vol;
        }
#endif
    }
    else
    {
#if defined(NW_PLATFORM_CAFE) && defined( NW_COMPILER_GHS )
        f32 vol_f = static_cast<f32>(ve.vol) / 32768.0f;
        f32 delta_f = static_cast<f32>(ve.volDelta) / 32768.0f;
        f32x2 vol = { vol_f, vol_f + delta_f };
        f32x2 delta = __PS_FDUP( delta_f * 2 );

        NW_ASSERT_ALIGN( samples, 2*8 );
        for(unsigned long i = 0; i < samples/(2*8); i++)
        {
            __PSQ_STX( dest,  0, __PS_MADD( __PSQ_LX( src,  0, 0, 0 ), vol, __PSQ_LX( dest,  0, 0, 0 ) ), 0, 0 );
            vol = __PS_ADD( vol, delta );
            __PSQ_STX( dest,  8, __PS_MADD( __PSQ_LX( src,  8, 0, 0 ), vol, __PSQ_LX( dest,  8, 0, 0 ) ), 0, 0 );
            vol = __PS_ADD( vol, delta );
            __PSQ_STX( dest, 16, __PS_MADD( __PSQ_LX( src, 16, 0, 0 ), vol, __PSQ_LX( dest, 16, 0, 0 ) ), 0, 0 );
            vol = __PS_ADD( vol, delta );
            __PSQ_STX( dest, 24, __PS_MADD( __PSQ_LX( src, 24, 0, 0 ), vol, __PSQ_LX( dest, 24, 0, 0 ) ), 0, 0 );
            vol = __PS_ADD( vol, delta );

            __PSQ_STX( dest, 32, __PS_MADD( __PSQ_LX( src, 32, 0, 0 ), vol, __PSQ_LX( dest, 32, 0, 0 ) ), 0, 0 );
            vol = __PS_ADD( vol, delta );
            __PSQ_STX( dest, 40, __PS_MADD( __PSQ_LX( src, 40, 0, 0 ), vol, __PSQ_LX( dest, 40, 0, 0 ) ), 0, 0 );
            vol = __PS_ADD( vol, delta );
            __PSQ_STX( dest, 48, __PS_MADD( __PSQ_LX( src, 48, 0, 0 ), vol, __PSQ_LX( dest, 48, 0, 0 ) ), 0, 0 );
            vol = __PS_ADD( vol, delta );
            __PSQ_STX( dest, 56, __PS_MADD( __PSQ_LX( src, 56, 0, 0 ), vol, __PSQ_LX( dest, 56, 0, 0 ) ), 0, 0 );
            vol = __PS_ADD( vol, delta );

            dest += 2*8;
            src += 2*8;
        }
#else
        f32 vol = static_cast<f32>(ve.vol) / 32768.0f;
        f32 delta = static_cast<f32>(ve.volDelta) / 32768.0f;

        for(unsigned long i = 0; i < samples; i++)
        {
            (*dest++) += static_cast<s32>((*src++) * vol);
            vol += delta;
        }
#endif
    }

    return true;
}

bool NwVoice::UpdateNextWaveBuffer()
{
    if ( ! m_WaveBufferListBegin->loopFlag )
    {
        WaveBuffer* waveBuffer = m_WaveBufferListBegin;
        m_WaveBufferListBegin = m_WaveBufferListBegin->next;
        waveBuffer->status = WaveBuffer::STATUS_DONE;
        if ( m_WaveBufferListBegin == NULL ) {
            m_WaveBufferListEnd = NULL;
            return false;
        }
        m_WaveBufferListBegin->status = WaveBuffer::STATUS_PLAY;
    }
    m_PlayPosition = m_WaveBufferListBegin->sampleOffset;

    return true;
}

u32 NwVoice::DecodePcm16( s16* buffer, u32 samples )
{
    u32 restDecodeSamples = samples;
    s16* dest = buffer;

    while( restDecodeSamples > 0 )
    {
        const s16* data = reinterpret_cast<const s16*>(m_WaveBufferListBegin->bufferAddress);
        const s16* src = &data[ m_PlayPosition ];

        const u32 restBufferSamples = m_WaveBufferListBegin->sampleLength - m_PlayPosition;
        u32 decodeSamples = restDecodeSamples;
        if ( decodeSamples > restBufferSamples )
        {
            decodeSamples = restBufferSamples;
        }

    #if defined( NW_RUNTIME_CAFE ) || defined( NW_PLATFORM_CAFE )
        // SoundMaker NwRuntimeCafe.lib と 実機ランタイムで有効になる
        std::memcpy(dest, src, decodeSamples * sizeof(s16) );
        dest += decodeSamples;
    #elif defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK) // PC ランタイムで有効
        for( unsigned int i=0; i<decodeSamples; i++)
        {
            *dest = ut::ReverseEndian(*src);
            ++dest;
            ++src;
        }
    #elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
        for( unsigned int i=0; i<decodeSamples; i++)
        {
            *dest = ut::ReverseEndian(*src);
            ++dest;
            ++src;
        }
    #else
        #error "Unknown platform"
    #endif

        m_PlayPosition += decodeSamples;
        restDecodeSamples -= decodeSamples;

        if ( m_PlayPosition >= m_WaveBufferListBegin->sampleLength )
        {
            if ( ! UpdateNextWaveBuffer() ) break;
        }
    }

    if ( restDecodeSamples > 0 )
    {
        ut::internal::Memset( dest, 0, restDecodeSamples * sizeof(s16) );
    }

    return samples - restDecodeSamples;
}

u32 NwVoice::DecodePcm8( s16* buffer, u32 samples )
{
    u32 restDecodeSamples = samples;
    s16* dest = buffer;

    while( restDecodeSamples > 0 )
    {
        const s8* data = reinterpret_cast<const s8*>(m_WaveBufferListBegin->bufferAddress);
        const s8* src = &data[ m_PlayPosition ];

        const u32 restBufferSamples = m_WaveBufferListBegin->sampleLength - m_PlayPosition;
        u32 decodeSamples = restDecodeSamples;
        if ( decodeSamples > restBufferSamples )
        {
            decodeSamples = restBufferSamples;
        }
        for( unsigned int i=0; i<decodeSamples; i++)
        {
            *dest = (*src << 8);
            ++dest;
            ++src;
        }

        m_PlayPosition += decodeSamples;
        restDecodeSamples -= decodeSamples;

        if ( m_PlayPosition >= m_WaveBufferListBegin->sampleLength )
        {
            if ( ! UpdateNextWaveBuffer() ) break;
        }
    }

    if ( restDecodeSamples > 0 )
    {
        ut::internal::Memset( dest, 0, restDecodeSamples * sizeof(s16) );
    }

    return samples - restDecodeSamples;
}

u32 NwVoice::DecodeAdpcm( s16* buffer, u32 samples )
{
    u32 restDecodeSamples = samples;
    s16* dest = buffer;

    while( restDecodeSamples > 0 )
    {
        if ( m_PlayPosition == m_WaveBufferListBegin->sampleOffset )
        {
            const AdpcmContext* pAdpcmContext = m_WaveBufferListBegin->pAdpcmContext;
            if ( pAdpcmContext != NULL )
            {
                m_AdpcmContext = *pAdpcmContext;
            }
        }

        const u32 restBufferSamples = m_WaveBufferListBegin->sampleLength - m_PlayPosition;
        u32 decodeSamples = restDecodeSamples;
        if ( decodeSamples > restBufferSamples )
        {
            decodeSamples = restBufferSamples;
        }

        const void* adpcmData = reinterpret_cast<const s16*>(m_WaveBufferListBegin->bufferAddress);

        // デコード API に切り出し
        DecodeDspAdpcm( m_PlayPosition, m_AdpcmContext, m_AdpcmParam, adpcmData, decodeSamples, dest );

        m_PlayPosition += decodeSamples;
        restDecodeSamples -= decodeSamples;
        dest += decodeSamples;

        if ( m_PlayPosition >= m_WaveBufferListBegin->sampleLength )
        {
            if ( ! UpdateNextWaveBuffer() ) break;
        }
    }

    if ( restDecodeSamples > 0 )
    {
        ut::internal::Memset( dest, 0, restDecodeSamples * sizeof(s16) );
    }

    return samples - restDecodeSamples;
}


//==========================================================================================
//
// NwVoiceManager
//
//==========================================================================================

void NwVoiceManager::Initialize()
{
    for( int i=0; i < VOICE_COUNT_MAX ; i++ )
    {
        m_FreeVoiceList.PushBack(&m_NwVoiceArray[i]);
    }
}

void NwVoiceManager::Finalize()
{
    for( int i=0; i < VOICE_COUNT_MAX ; i++ )
    {
        NwVoice& voice = m_NwVoiceArray[i];
        if ( voice.IsAvailable() )
        {
            voice.Free();
        }
    }
}

void NwVoiceManager::detail_UpdateAllVoices()
{
    ut::ScopedLock<ut::CriticalSection> lock(s_CriticalSection);

    OutputMode outputMode[OUTPUT_DEVICE_COUNT];
    for ( int i = 0; i < OUTPUT_DEVICE_COUNT; i++ )
    {
        outputMode[i] = driver::HardwareManager::GetInstance().GetOutputMode(static_cast<OutputDevice>(i));
    }
    for ( int ch = 0; ch < VOICE_COUNT_MAX; ch++ )
    {
        NwVoice& voice = m_NwVoiceArray[ch];
        if ( voice.IsAvailable() )
        {
            voice.UpdateState( outputMode );
        }
    }
}

u32 NwVoiceManager::GetActiveVoiceCount() const
{
    return m_VoiceList.GetSize();
}

void NwVoiceManager::detail_Synthesize( NwVoiceSynthesizeBuffer* buffer, u32 samples, f32 sampleRate )
{
    static const u32 WORK_BUFFER_SIZE = 144;

#if 1
    static u8 waveBufferArea[ WORK_BUFFER_SIZE * sizeof(f32) + (CACHE_BLOCK_SIZE-1) ];
    f32* waveBuffer = reinterpret_cast<f32*>(ut::RoundUp( waveBufferArea, CACHE_BLOCK_SIZE ));
#if defined( NW_PLATFORM_CAFE )
    DCZeroRange( waveBuffer, sizeof(f32) * samples );
#endif

#else
    f32 waveBuffer[WORK_BUFFER_SIZE];
#endif

    NW_ASSERT_MAX( samples, WORK_BUFFER_SIZE );

    ut::ScopedLock<ut::CriticalSection> lock(s_CriticalSection);

    NwVoiceList::Iterator itr = m_VoiceList.GetBeginIter();
    while ( itr != m_VoiceList.GetEndIter() )
    {
        if ( itr->m_State == VOICE_STATE_PLAY )
        {
            itr->Synthesize( buffer, waveBuffer, samples, sampleRate );
        }
        (void)++itr;
    }
}

const NwVoice* NwVoiceManager::detail_GetVoiceById( int id )
{
    NW_MINMAXLT_ASSERT(id,0,VOICE_COUNT_MAX);
    return &m_NwVoiceArray[id];
}

NwVoice* NwVoiceManager::AllocVoice( u32 prioriry, NwVoice::DisposeCallbackFunc func, void* arg )
{
    NwVoice* pNwVoice = NULL;

    if ( ! m_FreeVoiceList.IsEmpty() )
    {
        pNwVoice = &m_FreeVoiceList.GetBack();
        m_FreeVoiceList.PopBack();
    }
    else
    {
        if ( m_VoiceList.IsEmpty() )
        {
            return NULL;
        }
        pNwVoice = &m_VoiceList.GetBack();
        if ( prioriry < pNwVoice->m_Priority )
        {
            return NULL;
        }

        pNwVoice->Finalize();

        NW_ASSERT( ! m_FreeVoiceList.IsEmpty() );

        pNwVoice = &m_FreeVoiceList.GetBack();
        m_FreeVoiceList.PopBack();
    }

    pNwVoice->Initialize( func, arg );
    pNwVoice->m_Priority = prioriry;
    pNwVoice->m_pVoiceManager = this;

    AppendVoiceList( pNwVoice );

//    NW_LOG("NwVoice::AllocVoice %08x\n", pNwVoice);

    return pNwVoice;
}

void NwVoiceManager::FreeVoice( NwVoice* voice )
{
    if ( voice->m_IsAppendedToList )
    {
        EraseVoiceList( voice );
        m_FreeVoiceList.PushBack( voice );
    }
}

void NwVoiceManager::AppendVoiceList( NwVoice* voice )
{
    if ( voice->m_IsAppendedToList )
    {
        return;
    }

    NwVoiceList::ReverseIterator itr = m_VoiceList.GetBeginReverseIter();
    while ( itr != m_VoiceList.GetEndReverseIter() )
    {
        if ( itr->m_Priority <= voice->m_Priority ) break;
        (void)++itr;
    }
    m_VoiceList.Insert( itr.GetBase(), voice );

    voice->m_IsAppendedToList = true;
}

void NwVoiceManager::EraseVoiceList( NwVoice* voice )
{
    if ( ! voice->m_IsAppendedToList )
    {
        return;
    }

    m_VoiceList.Erase( voice );

    voice->m_IsAppendedToList = false;
}

void NwVoiceManager::UpdatePriorityOrder( NwVoice* voice )
{
    if ( voice->m_IsAppendedToList )
    {
        EraseVoiceList( voice );
        AppendVoiceList( voice );
    }
}

//==========================================================================================
//
// NwVoiceRenderer
//
//==========================================================================================

void NwVoiceRenderer::Initialize( void* synthesizeBuffer, u32 synthesizeBufferSize )
{
    NW_ASSERT_ALIGN( synthesizeBuffer, CACHE_BLOCK_SIZE );

    const u32 synthesizeBufferUnitSize = NwVoiceSynthesizeBuffer::GetSynthesizeBufferSize();
    u32 synthesizeBufferCount = synthesizeBufferSize / synthesizeBufferUnitSize;
    if ( synthesizeBufferCount > SYNTHESIZE_BUFFER_COUNT_MAX ) synthesizeBufferCount = SYNTHESIZE_BUFFER_COUNT_MAX;

    void* bufferp = synthesizeBuffer;
    for( u32 i=0; i<synthesizeBufferCount; i++ )
    {
        NwVoiceSynthesizeBuffer* buffer = &m_SynthesizeBuffer[i];
        buffer->Initialize( bufferp, synthesizeBufferUnitSize );
        bufferp = ut::AddOffsetToPtr( bufferp, synthesizeBufferUnitSize );
    }
    NW_ASSERT( bufferp <= nw::ut::AddOffsetToPtr(synthesizeBuffer, synthesizeBufferSize) );

    m_SynthesizeBufferCount = synthesizeBufferCount;

    m_SynthesizeMsgQueueTv.Initialize( m_SynthesizeMsgBufferTv, SYNTHESIZE_BUFFER_COUNT_MAX );
    m_SynthesizeMsgQueueDrc0.Initialize( m_SynthesizeMsgBufferDrc0, SYNTHESIZE_BUFFER_COUNT_MAX );
    m_SynthesizeMsgQueueDrc1.Initialize( m_SynthesizeMsgBufferDrc1, SYNTHESIZE_BUFFER_COUNT_MAX );

    m_VoiceManager.Initialize();

    SetSampleRate( SAMPLE_RATE_48000 );

    driver::HardwareManager::GetInstance().AppendFinalMixCallback( this );

    Voice::detail_SetNwVoiceManager( &m_VoiceManager );
}

void NwVoiceRenderer::Finalize()
{
    Voice::detail_SetNwVoiceManager( NULL );

    driver::HardwareManager::GetInstance().EraseFinalMixCallback( this );

    m_VoiceManager.Finalize();

    ut::MessageQueue::MessageType msg;
    while( m_SynthesizeMsgQueueTv.Recv(&msg, false) )
    {
        // 空になるまでループ
    }
    while( m_SynthesizeMsgQueueDrc0.Recv(&msg, false) )
    {
        // 空になるまでループ
    }
    while( m_SynthesizeMsgQueueDrc1.Recv(&msg, false) )
    {
        // 空になるまでループ
    }

    m_SynthesizeMsgQueueTv.Finalize();
    m_SynthesizeMsgQueueDrc0.Finalize();
    m_SynthesizeMsgQueueDrc1.Finalize();
}

void NwVoiceRenderer::UpdateAllVoices()
{
    m_VoiceManager.detail_UpdateAllVoices();
}

u32 NwVoiceRenderer::Synthesize()
{
    u32 voiceCount = m_VoiceManager.GetActiveVoiceCount();

    // 波形合成
    if ( voiceCount > 0 )
    {
        NwVoiceSynthesizeBuffer* synthesizeBuffer = NULL;
        for( u32 i=0; i<m_SynthesizeBufferCount;i++)
        {
            if ( m_SynthesizeBuffer[i].eventTv.TryWait() )
            {
                if ( m_SynthesizeBuffer[i].eventDrc0.TryWait() )
                {
                    if ( m_SynthesizeBuffer[i].eventDrc1.TryWait() )
                    {
                        synthesizeBuffer = &m_SynthesizeBuffer[i];

                        synthesizeBuffer->eventTv.Reset();
                        synthesizeBuffer->eventDrc0.Reset();
                        synthesizeBuffer->eventDrc1.Reset();
                        break;
                    }
                }
            }
        }
        if ( synthesizeBuffer == NULL )
        {
            return 0;
        }

#if defined(NW_PLATFORM_CAFE)
        DCZeroRange( synthesizeBuffer->tvMix.frontLeft, NwVoiceSynthesizeBuffer::GetSynthesizeBufferSize() );
#else
        ut::internal::Memset( synthesizeBuffer->tvMix.frontLeft, 0, NwVoiceSynthesizeBuffer::GetSynthesizeBufferSize() );
#endif
        synthesizeBuffer->flag = 0;

        m_VoiceManager.detail_Synthesize( synthesizeBuffer, m_SampleCount, m_SampleRate );

        m_SynthesizeMsgQueueTv.Send( synthesizeBuffer, true );
        m_SynthesizeMsgQueueDrc0.Send( synthesizeBuffer, true );
        m_SynthesizeMsgQueueDrc1.Send( synthesizeBuffer, true );
    }

    return voiceCount;
}

void NwVoiceRenderer::SetSampleRate( SampleRateType type )
{
    switch( type )
    {
    case SAMPLE_RATE_48000:
        m_SampleCount = 144;
        m_SampleRate = 48000.0f;
        break;
    case SAMPLE_RATE_32000:
        m_SampleCount = 96;
        m_SampleRate = 32000.0f;
        break;
    default:
        NW_ERR("Unexpected SampleRateType %d\n", type);
        break;
    }
}

void NwVoiceRenderer::MixSamples( const f32* srcp, s32* destp, u32 samples )
{
    NW_ASSERT_ALIGN( samples, 8 );
    for( unsigned int i=0; i< samples/8 ; i++ )
    {
        *destp++ += static_cast<s32>(*srcp++);
        *destp++ += static_cast<s32>(*srcp++);
        *destp++ += static_cast<s32>(*srcp++);
        *destp++ += static_cast<s32>(*srcp++);

        *destp++ += static_cast<s32>(*srcp++);
        *destp++ += static_cast<s32>(*srcp++);
        *destp++ += static_cast<s32>(*srcp++);
        *destp++ += static_cast<s32>(*srcp++);
    }
}

void NwVoiceRenderer::OnFinalMix(nw::snd::OutputDevice device, const nw::snd::FinalMixData* data)
{
    switch(device)
    {
    case OUTPUT_DEVICE_MAIN:
        OnMainOutProcess(data);
        break;
    case OUTPUT_DEVICE_DRC0:
        OnDrc0OutProcess(data);
        break;
    case OUTPUT_DEVICE_DRC1:
        OnDrc1OutProcess(data);
        break;
    default:
        break;
    }
}

void NwVoiceRenderer::OnMainOutProcess(const FinalMixData* data)
{
    ut::MessageQueue::MessageType msg;
    if ( m_SynthesizeMsgQueueTv.Recv( &msg, false ) )
    {
        NwVoiceSynthesizeBuffer* synthesizeBuffer = reinterpret_cast<NwVoiceSynthesizeBuffer*>(msg);

        const u32 samples = synthesizeBuffer->sampleCount ;
        const u32 deviceIndex = 0;

        if ( data[deviceIndex].sampleCount == samples )
        {
            u32 flag = synthesizeBuffer->flag;

            if ( data[deviceIndex].channelCount >= 6 )
            {
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_TV_L ) MixSamples( synthesizeBuffer->tvMix.frontLeft, data[deviceIndex].frontLeft, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_TV_R ) MixSamples( synthesizeBuffer->tvMix.frontRight, data[deviceIndex].frontRight, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_TV_SL ) MixSamples( synthesizeBuffer->tvMix.rearLeft, data[deviceIndex].rearLeft, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_TV_SR ) MixSamples( synthesizeBuffer->tvMix.rearRight, data[deviceIndex].rearRight, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_TV_FC ) MixSamples( synthesizeBuffer->tvMix.frontCenter, data[deviceIndex].frontCenter, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_TV_LFE ) MixSamples( synthesizeBuffer->tvMix.lfe, data[deviceIndex].lfe, samples );
            }
            else if ( data[deviceIndex].channelCount >= 4 )
            {
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_TV_L ) MixSamples( synthesizeBuffer->tvMix.frontLeft, data[deviceIndex].frontLeft, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_TV_R ) MixSamples( synthesizeBuffer->tvMix.frontRight, data[deviceIndex].frontRight, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_TV_SL ) MixSamples( synthesizeBuffer->tvMix.rearLeft, data[deviceIndex].rearLeft, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_TV_SR ) MixSamples( synthesizeBuffer->tvMix.rearRight, data[deviceIndex].rearRight, samples );
            }
            else if ( data[deviceIndex].channelCount >= 2 )
            {
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_TV_L ) MixSamples( synthesizeBuffer->tvMix.frontLeft, data[deviceIndex].frontLeft, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_TV_R ) MixSamples( synthesizeBuffer->tvMix.frontRight, data[deviceIndex].frontRight, samples );
            }
        }

        synthesizeBuffer->eventTv.Signal();
    }
}

void NwVoiceRenderer::OnDrc0OutProcess(const FinalMixData* data)
{
    ut::MessageQueue::MessageType msg;
    if ( m_SynthesizeMsgQueueDrc0.Recv( &msg, false ) )
    {
        NwVoiceSynthesizeBuffer* synthesizeBuffer = reinterpret_cast<NwVoiceSynthesizeBuffer*>(msg);

        const u32 samples = synthesizeBuffer->sampleCount;

        if ( data->sampleCount == samples )
        {
            u32 flag = synthesizeBuffer->flag;

            if ( data->channelCount >= 4 )
            {
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_DRC0_L ) MixSamples( synthesizeBuffer->drc0Mix.frontLeft, data->frontLeft, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_DRC0_R ) MixSamples( synthesizeBuffer->drc0Mix.frontRight, data->frontRight, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_DRC0_SL ) MixSamples( synthesizeBuffer->drc0Mix.rearLeft, data->rearLeft, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_DRC0_SR ) MixSamples( synthesizeBuffer->drc0Mix.rearRight, data->rearRight, samples );
            }
            else if ( data->channelCount >= 2 )
            {
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_DRC0_L ) MixSamples( synthesizeBuffer->drc0Mix.frontLeft, data->frontLeft, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_DRC0_R ) MixSamples( synthesizeBuffer->drc0Mix.frontRight, data->frontRight, samples );
            }
        }

        synthesizeBuffer->eventDrc0.Signal();
    }
}

void NwVoiceRenderer::OnDrc1OutProcess(const FinalMixData* data)
{
    ut::MessageQueue::MessageType msg;
    if ( m_SynthesizeMsgQueueDrc1.Recv( &msg, false ) )
    {
        NwVoiceSynthesizeBuffer* synthesizeBuffer = reinterpret_cast<NwVoiceSynthesizeBuffer*>(msg);

        const u32 samples = synthesizeBuffer->sampleCount;

        if ( data->sampleCount == samples )
        {
            u32 flag = synthesizeBuffer->flag;

            if ( data->channelCount >= 4 )
            {
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_DRC1_L ) MixSamples( synthesizeBuffer->drc1Mix.frontLeft, data->frontLeft, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_DRC1_R ) MixSamples( synthesizeBuffer->drc1Mix.frontRight, data->frontRight, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_DRC1_SL ) MixSamples( synthesizeBuffer->drc1Mix.rearLeft, data->rearLeft, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_DRC1_SR ) MixSamples( synthesizeBuffer->drc1Mix.rearRight, data->rearRight, samples );
            }
            else if ( data->channelCount >= 2 )
            {
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_DRC1_L ) MixSamples( synthesizeBuffer->drc1Mix.frontLeft, data->frontLeft, samples );
                if ( flag & NwVoiceSynthesizeBuffer::FLAG_DRC1_R ) MixSamples( synthesizeBuffer->drc1Mix.frontRight, data->frontRight, samples );
            }
        }

        synthesizeBuffer->eventDrc1.Signal();
    }
}

void NwVoice::UpdateVoiceInfo( VoiceInfo* voiceInfo ) const
{
    NW_NULL_ASSERT(voiceInfo);

    voiceInfo->userId = m_pVoice;
    voiceInfo->voiceState = m_State;
    volatile WaveBuffer* pListBegin = m_WaveBufferListBegin;
    if ( pListBegin != NULL )
    {
        voiceInfo->waveBufferStatus = pListBegin->status;
        voiceInfo->waveBufferTag = pListBegin->userParam;
    }
    else
    {
        voiceInfo->waveBufferStatus = WaveBuffer::STATUS_FREE;
    }
    voiceInfo->playPosition = m_PlayPosition;
}


}}}
