﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_FxMultiChDelay.h>
#include <nw/assert.h>

#if defined( NW_PLATFORM_WIN32 )
using namespace nw::internal::winext;
#elif defined( NW_USE_NINTENDO_SDK )
// TODO: nn_audio
using namespace nw::internal::winext;
#endif

#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)

namespace nw {
namespace snd {

/* ========================================================================
        constant definition
   ======================================================================== */

const f32 FxMultiChDelay::DELAY_MIN          = 0.0f;
const f32 FxMultiChDelay::DELAY_MAX          = 2000.0f;
const f32 FxMultiChDelay::FEEDBACK_MIN       = 0.1f;
const f32 FxMultiChDelay::FEEDBACK_MAX       = 1.0f;
const f32 FxMultiChDelay::LPF_MIN            = 0.1f;
const f32 FxMultiChDelay::LPF_MAX            = 1.0f;
const f32 FxMultiChDelay::OUT_GAIN_MIN       = 0.1f;
const f32 FxMultiChDelay::OUT_GAIN_MAX       = 1.0f;
const f32 FxMultiChDelay::CHANNEL_SPREAD_MIN = 0.1f;
const f32 FxMultiChDelay::CHANNEL_SPREAD_MAX = 1.0f;

/* ========================================================================
        member function
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         FxMultiChDelay

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
FxMultiChDelay::FxMultiChDelay()
: m_IsActive( false ),
  m_ChannelMode( CHANNEL_MODE_6CH ),
  m_SampleRate( SAMPLE_RATE_32000 )
{
    // パラメータ初期化
    const MultiChDelayParam param;
    SetParam( param );
}

/*---------------------------------------------------------------------------*
  Name:         GetRequiredMemSize

  Description:  エフェクトを使用するのに必用となるメモリサイズを取得する

  Arguments:    None.

  Returns:      必要となるメモリサイズ
 *---------------------------------------------------------------------------*/
u32 FxMultiChDelay::GetRequiredMemSize()
{
    AXFX_MULTI_CH_DELAY param = m_AxfxParam;

    if ( !m_IsActive )
    {
        // 初期化前にバッファサイズを知るため、一部のパラメータを設定する
        switch ( m_ChannelMode )
        {
        case CHANNEL_MODE_2CH:
            param.mode = AXFX_DELAY_MODE_2CH;
            break;
        case CHANNEL_MODE_4CH:
            param.mode = AXFX_DELAY_MODE_4CH;
            break;
        case CHANNEL_MODE_6CH:
            param.mode = AXFX_DELAY_MODE_6CH;
            break;
        default:
            param.mode = AXFX_DELAY_MODE_6CH;
        }
        param.max_delay = 2000.0;
        switch ( m_SampleRate )
        {
        case SAMPLE_RATE_32000:
            param.fs = 32000;
            break;
#if defined(NW_SND_CONFIG_ENABLE_SOUND2)
        case SAMPLE_RATE_48000:
            param.fs = 48000;
            break;
#endif
        default:
            param.fs = 32000;
        }
    }

    s32 requiredSize = AXFXMultiChDelayGetMemSize( &param );
    NW_ASSERT( requiredSize >= 0 );
    size_t size = ut::RoundUp(
        sizeof( MEMiHeapHead )
        + sizeof( MEMiFrmHeapHead )
        + requiredSize
        + 32,
        32
    );

    return static_cast<u32>( size );
}

/*---------------------------------------------------------------------------*
  Name:         AssignWorkBuffer

  Description:  エフェクトのワークバッファを割り当てる

  Arguments:    buffer - バッファのアドレス
                size - バッファのサイズ

  Returns:      割り当てに成功したらtrue
 *---------------------------------------------------------------------------*/
bool FxMultiChDelay::AssignWorkBuffer( void* buffer, u32 size )
{
    return m_Impl.CreateHeap( buffer, size );
}

/*---------------------------------------------------------------------------*
  Name:         ReleaseWorkBuffer

  Description:  エフェクトのワークバッファを解放する

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FxMultiChDelay::ReleaseWorkBuffer()
{
    m_Impl.DestroyHeap();
}

/*---------------------------------------------------------------------------*
  Name:         Initialize

  Description:  エフェクトの開始処理を行う

  Arguments:    None.

  Returns:      成功したらtrue
 *---------------------------------------------------------------------------*/
bool FxMultiChDelay::Initialize()
{
    if ( GetRequiredMemSize() > static_cast<u32>(m_Impl.GetHeapTotalSize()) ) return false;

    AXFXAlloc alloc;
    AXFXFree free;
#if defined(NW_SND_CONFIG_ENABLE_SOUND2)
    m_Impl.HookAlloc( &alloc, &free );
#else
    m_Impl.HookMultichAlloc( &alloc, &free );
#endif
    AXFX_DELAY_MODE mode;
    switch ( m_ChannelMode )
    {
    case CHANNEL_MODE_2CH:
        mode = AXFX_DELAY_MODE_2CH;
        break;
    case CHANNEL_MODE_4CH:
        mode = AXFX_DELAY_MODE_4CH;
        break;
    case CHANNEL_MODE_6CH:
        mode = AXFX_DELAY_MODE_6CH;
        break;
    default:
        mode = AXFX_DELAY_MODE_6CH;
    }

    AXFX_SAMPLE_RATE rate = GetFxSampleRate( m_SampleRate );
    int result = AXFXMultiChDelayInit( &m_AxfxParam, mode, rate );
#if defined(NW_SND_CONFIG_ENABLE_SOUND2)
    u32 allocatedSize = m_Impl.RestoreAlloc( alloc, free );
#else
    u32 allocatedSize = m_Impl.RestoreMultichAlloc( alloc, free );
#endif
    m_IsActive = true;

    // AXで実際にAllocateされたメモリと同じかどうか確認
    u32 requiredMemSize = GetRequiredMemSize();
    NW_WARNING(
        ut::RoundUp( sizeof( MEMiHeapHead ) + sizeof( MEMiFrmHeapHead ) + allocatedSize + 32, 32 ) == requiredMemSize,
        "differ between allocated buffer size(%d) and required mem size(%d).",
        ut::RoundUp( sizeof( MEMiHeapHead ) + sizeof( MEMiFrmHeapHead ) + allocatedSize + 32, 32 ),
        requiredMemSize
    );

    return result != 0;
}

/*---------------------------------------------------------------------------*
  Name:         Finalize

  Description:  エフェクトの終了処理を行う

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FxMultiChDelay::Finalize()
{
    if ( ! m_IsActive ) return;

    m_IsActive = false;

    AXFXAlloc alloc;
    AXFXFree free;
#if defined(NW_SND_CONFIG_ENABLE_SOUND2)
    m_Impl.HookAlloc( &alloc, &free );
#else
    m_Impl.HookMultichAlloc( &alloc, &free );
#endif
    AXFXMultiChDelayShutdown( &m_AxfxParam );
#if defined(NW_SND_CONFIG_ENABLE_SOUND2)
    m_Impl.RestoreAlloc( alloc, free );
#else
    m_Impl.RestoreMultichAlloc( alloc, free );
#endif
}

/*---------------------------------------------------------------------------*
  Name:         SetParam

  Description:  エフェクトのパラメータを変更する

  Arguments:    param - エフェクトパラメータ

  Returns:      None.
 *---------------------------------------------------------------------------*/
bool FxMultiChDelay::SetParam( const MultiChDelayParam& param )
{
    NW_FMINMAX_ASSERT( param.delay,         DELAY_MIN,          DELAY_MAX          );
    NW_FMINMAX_ASSERT( param.feedback,      FEEDBACK_MIN,       FEEDBACK_MAX       );
    NW_FMINMAX_ASSERT( param.lpf,           LPF_MIN,            LPF_MAX            );
    NW_FMINMAX_ASSERT( param.outGain,       OUT_GAIN_MIN,       OUT_GAIN_MAX       );
    NW_FMINMAX_ASSERT( param.channelSpread, CHANNEL_SPREAD_MIN, CHANNEL_SPREAD_MAX );

    m_Param = param; // struct copy

    m_AxfxParam.delay_time      = ut::Clamp( param.delay,         DELAY_MIN,          DELAY_MAX          );
    m_AxfxParam.feedback        = ut::Clamp( param.feedback,      FEEDBACK_MIN,       FEEDBACK_MAX       );
    m_AxfxParam.low_pass_amount = ut::Clamp( param.lpf,           LPF_MIN,            LPF_MAX            );
    m_AxfxParam.out_gain        = ut::Clamp( param.outGain,       OUT_GAIN_MIN,       OUT_GAIN_MAX       );
    m_AxfxParam.channel_spread  = ut::Clamp( param.channelSpread, CHANNEL_SPREAD_MIN, CHANNEL_SPREAD_MAX );

    if ( ! m_IsActive ) return true;

    int result = AXFXMultiChDelaySettingsUpdate( &m_AxfxParam );
    return result != 0;
}

/*---------------------------------------------------------------------------*
  Name:         UpdateBuffer

  Description:  エフェクトコールバック

  Arguments:    param - エフェクトパラメータ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FxMultiChDelay::UpdateBuffer(
    int numChannels,
    void* buffer[],
    unsigned long bufferSize,
    SampleFormat format,
    f32 sampleRate,
    OutputMode mode
)
{
    if ( ! m_IsActive ) return;

    // DPL2モードの時はバスの解釈が違うので処理しない
    if ( mode == OUTPUT_MODE_DPL2 ) return;

    (void)format;
    (void)sampleRate;

    NW_ASSERT( numChannels >= 2 );
    NW_ASSERT( format == SAMPLE_FORMAT_PCM_S32 );

    AXFX_6CH_BUFFERUPDATE axfxbuf =
    {
        static_cast<s32*>( buffer[0] ),
        static_cast<s32*>( buffer[1] ),
        static_cast<s32*>( buffer[2] ),
        static_cast<s32*>( buffer[3] ),
        static_cast<s32*>( buffer[4] ),
        static_cast<s32*>( buffer[5] )
    };
    AXAUXCBSTRUCT auxCbStruct;

    auxCbStruct.numChs = numChannels;
    auxCbStruct.numSamples = bufferSize / ( numChannels * sizeof(s32) );
    AXFXMultiChDelayCallback( &axfxbuf, &m_AxfxParam, &auxCbStruct );
}

/*---------------------------------------------------------------------------*
  Name:         IsValidChannelNum

  Description:  デバイスに対してチャンネル数が妥当かをチェックする仮想関数です。

  Arguments:    device チェック対象となるデバイスです。

  Returns:      チャンネル数が妥当であれば true を、妥当でなければ false を返します。
 *---------------------------------------------------------------------------*/
bool FxMultiChDelay::IsValidChannelNum( OutputDevice device )
{
    if ( device == OUTPUT_DEVICE_DRC0 || device == OUTPUT_DEVICE_DRC1 )
    {
        if ( m_ChannelMode == CHANNEL_MODE_6CH )
        {
            NW_ASSERTMSG( false, "CHANNEL_MODE_6CH (OUTPUT_DEVICE_DRC) is inValid. Please use lower channel mode." );
            return false;
        }
    }

    return true;
}

} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_MULTICHEFT
