﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_FsFileStream.h>
#include <nw/snd/snd_Util.h>
#include <nw/assert.h>

#if defined(NW_PLATFORM_ANDROID)
#include <android/asset_manager.h>
#endif

#if defined( NW_PLATFORM_CAFE )
#elif defined( NW_PLATFORM_WIN32 )
    #include <Windows.h>
    namespace
    {
        u32 ConvertAccessMode(nw::snd::internal::fnd::File::AccessMode accessMode)
        {
            switch(accessMode)
            {
            case nw::snd::internal::fnd::File::ACCESS_MODE_READ:
                return GENERIC_READ;

            case nw::snd::internal::fnd::File::ACCESS_MODE_WRITE:
                return GENERIC_WRITE;

            case nw::snd::internal::fnd::File::ACCESS_MODE_READ_AND_WRITE:
                return GENERIC_READ | GENERIC_WRITE;
            }

            return 0;
        }
    }
#elif defined(NW_PLATFORM_ANDROID)
namespace
{
    const char* ConvertAccessMode(nw::snd::internal::fnd::File::AccessMode accessMode)
    {
        switch (accessMode)
        {
        case nw::snd::internal::fnd::File::ACCESS_MODE_READ:
            return "rb";

        case nw::snd::internal::fnd::File::ACCESS_MODE_WRITE:
            return "wb";

        case nw::snd::internal::fnd::File::ACCESS_MODE_READ_AND_WRITE:
            return "w+b";
        }

        return NULL;
    }

    u32
        GetFileSize(void *fp, nw::snd::internal::FsFileStream::AccessMethodType accessMethodType)
    {
        if(accessMethodType == nw::snd::internal::FsFileStream::AccessMethodType_Stdio)
        {
            int result = 0;
            fpos_t currentFPos, fpos;

            // TODO: 本当はfstatを利用して取得するべき
            //       https://www.jpcert.or.jp/sc-rules/c-fio19-c.html

            // 現在位置を取得
            result = std::fgetpos(reinterpret_cast<FILE*>(fp), &currentFPos);
            if (result < 0)
            {
                return -1;
            }

            // ファイルの終端位置を取得
            result = std::fseek(reinterpret_cast<FILE*>(fp), 0, SEEK_END);
            if (result < 0)
            {
                return -1;
            }

            result = std::fgetpos(reinterpret_cast<FILE*>(fp), &fpos);
            if (result < 0)
            {
                return -1;
            }

            u32 fileSize = static_cast<u32>(fpos);

            // 元の位置に戻す
            result = std::fsetpos(reinterpret_cast<FILE*>(fp), &currentFPos);
            if (result < 0)
            {
                return -1;
            }

            return fileSize;
        }
        else if (accessMethodType == nw::snd::internal::FsFileStream::AccessMethodType_AndroidAssets)
        {
            return AAsset_getLength(reinterpret_cast<AAsset*>(fp));
        }
        else
        {
            NW_INTERNAL_ERROR("m_AccessMethodType is invalid.");
        }
    }
}
#elif defined(NW_PLATFORM_IOS)
namespace
{
    const char* ConvertAccessMode(nw::snd::internal::fnd::File::AccessMode accessMode)
    {
        switch (accessMode)
        {
        case nw::snd::internal::fnd::File::ACCESS_MODE_READ:
            return "rb";

        case nw::snd::internal::fnd::File::ACCESS_MODE_WRITE:
            return "wb";

        case nw::snd::internal::fnd::File::ACCESS_MODE_READ_AND_WRITE:
            return "w+b";
        }

        return NULL;
    }

    bool GetFileSize(u32* fileSize, void *fp)
    {
        int result = 0;
        fpos_t currentFPos, fpos;

        // TODO: 本当はfstatを利用して取得するべき
        //       https://www.jpcert.or.jp/sc-rules/c-fio19-c.html

        // 現在位置を取得
        result = std::fgetpos(reinterpret_cast<FILE*>(fp), &currentFPos);
        if (result < 0)
        {
            return false;
        }

        // ファイルの終端位置を取得
        result = std::fseek(reinterpret_cast<FILE*>(fp), 0, SEEK_END);
        if (result < 0)
        {
            return false;
        }

        result = std::fgetpos(reinterpret_cast<FILE*>(fp), &fpos);
        if (result < 0)
        {
            return false;
        }

        *fileSize = static_cast<u32>(fpos);

        // 元の位置に戻す
        result = std::fsetpos(reinterpret_cast<FILE*>(fp), &currentFPos);
        if (result < 0)
        {
            return false;
        }

        return true;
    }
}
#elif defined( NW_USE_NINTENDO_SDK )
    namespace
    {
        nn::fs::OpenMode ConvertAccessMode(nw::snd::internal::fnd::File::AccessMode accessMode)
        {
            switch(accessMode)
            {
            case nw::snd::internal::fnd::File::ACCESS_MODE_READ:
                return nn::fs::OpenMode_Read;

            case nw::snd::internal::fnd::File::ACCESS_MODE_WRITE:
                return nn::fs::OpenMode_Write;

            case nw::snd::internal::fnd::File::ACCESS_MODE_READ_AND_WRITE:
                return nn::fs::OpenMode_AllowAppend;

            default:
                break;
            }

            return nn::fs::OpenMode_Read;
        }
    }
#else
    #error
#endif

namespace nw {
namespace snd {
namespace internal {

#if defined(NW_PLATFORM_ANDROID)
AAssetManager *FsFileStream::m_pAssetManager;
const char* const FsFileStream::ANDROID_ASSETS_PATH_PREFIX = "android_assets:";
#endif

//---------------------------------------------------------------------------
FsFileStream::FsFileStream()
: m_Available( false ),
  m_DuplicateFlag( false ),
#if defined( NW_PLATFORM_CAFE )
  m_pClient( NULL ),
  m_pPool( NULL ),
  m_Priority( FS_PRIORITY_DEFAULT ),
#endif
  m_FilePos( 0 ),
  m_FileSize( 0 )
{
}

//---------------------------------------------------------------------------
FsFileStream::FsFileStream( const FsFileStream* fileStream )
: m_Available( true ),
  m_DuplicateFlag( true ),
#if defined( NW_PLATFORM_CAFE )
  m_pClient( fileStream->m_pClient ),
  m_FileHandle( fileStream->m_FileHandle ),
  m_FileStat( fileStream->m_FileStat ),
  m_pPool( fileStream->m_pPool ),
  m_Priority( fileStream->m_Priority ),
#elif defined( NW_PLATFORM_WIN32 )
  m_FileHandle(fileStream->m_FileHandle),
#elif defined(NW_PLATFORM_ANDROID)
  m_FileHandle(fileStream->m_FileHandle),
  m_AccessMethodType( fileStream->m_AccessMethodType ),
#elif defined(NW_PLATFORM_IOS)
  m_FileHandle(fileStream->m_FileHandle),
#elif defined( NW_USE_NINTENDO_SDK )
  m_FileHandle(fileStream->m_FileHandle),
#else
  #error
#endif
  m_FilePos( 0 ),
  m_FileSize( fileStream->m_FileSize )
{
    NW_ASSERT_NOT_NULL(fileStream);
}

//---------------------------------------------------------------------------
#if defined( NW_PLATFORM_CAFE )
bool
FsFileStream::Open(
        FSClient* client,
        const char* filePath,
        const char* mode,
        FsCommandBlockPool* pool )
{
    NW_ASSERT_NOT_NULL(client);
    NW_ASSERT_NOT_NULL(pool);
    NW_ASSERT_NOT_NULL(filePath);
    NW_ASSERT_NOT_NULL(mode);

    if ( m_Available )
    {
        Close();
    }

    m_pPool = pool;

    FSCmdBlock* block = m_pPool->Alloc();
    FSSetCmdPriority(block, m_Priority);
    NW_ASSERT_NOT_NULL(block);

    FSStatus status = FSOpenFile(client, block, filePath, mode, &m_FileHandle, Util::GetRetFlag());
    if ( status != FS_STATUS_OK )
    {
#if !defined(NW_RELEASE)
        NW_WARNING( status == FS_STATUS_OK, "FsFileStream::Open is Failed: %s(%s)\n", Util::FSStatusToString(status), filePath );
#endif
        m_pPool->Free(block);
        return false;
    }

    status = FSGetStatFile(client, block, m_FileHandle, &m_FileStat, Util::GetRetFlag());

    if ( status != FS_STATUS_OK )
    {
#if !defined(NW_RELEASE)
        NW_WARNING( status == FS_STATUS_OK, "FsFileStream::Open is Failed: %s\n", Util::FSStatusToString(status) );
#endif
        status = FSCloseFile( client, block, m_FileHandle, Util::GetRetFlag() );
        NW_ASSERT( status == FS_STATUS_OK );
        m_pPool->Free(block);
        return false;
    }

    m_pPool->Free(block);

    m_pClient = client;
    m_FilePos = 0;
    m_FileSize = m_FileStat.size;
    m_Available = true;
    return true;
}
#elif defined(NW_PLATFORM_ANDROID)
bool
FsFileStream::Open(
const char* filePath,
snd::internal::fnd::File::AccessMode mode)
{
    const char *desiredAccess = ConvertAccessMode(mode);
    NW_ASSERT(desiredAccess);

    void *handle;
    if(strncmp(filePath, ANDROID_ASSETS_PATH_PREFIX, strlen(ANDROID_ASSETS_PATH_PREFIX)) == 0)
    {
        m_AccessMethodType = AccessMethodType_AndroidAssets;
        filePath += strlen(ANDROID_ASSETS_PATH_PREFIX) + 1;
    }
    else
    {
        m_AccessMethodType = AccessMethodType_Stdio;
    }

    NW_ASSERT(m_pAssetManager);

    if(m_AccessMethodType == AccessMethodType_Stdio)
    {
        handle = fopen(filePath, desiredAccess);
    }
    else if (m_AccessMethodType == AccessMethodType_AndroidAssets)
    {
        handle = AAssetManager_open(m_pAssetManager, filePath, AASSET_MODE_UNKNOWN);
    }
    else
    {
        NW_INTERNAL_ERROR("m_AccessMethodType is invalid.");
    }

    if (handle == NULL)
    {
        return false;
    }

    m_FileHandle = reinterpret_cast<void*>(handle);

    u32 result = GetFileSize(handle, m_AccessMethodType);


    if (result == -1)
    {
        return false;
    }
    m_FilePos = 0;
    m_FileSize = result;
    m_Available = true;
    return true;
}
#elif defined(NW_PLATFORM_IOS)
bool
FsFileStream::Open(
        const char* filePath,
        snd::internal::fnd::File::AccessMode mode)
{
    const char *desiredAccess = ConvertAccessMode(mode);
    NW_ASSERT(desiredAccess);

    m_FileHandle = fopen(filePath, desiredAccess);

    if (m_FileHandle == NULL)
    {
        return false;
    }


    const bool result = GetFileSize(&m_FileSize, m_FileHandle);

    if (result)
    {
        m_FilePos = 0;
        m_Available = true;
    }

    return result;
}
#elif defined( NW_PLATFORM_WIN32 )
bool
FsFileStream::Open(
        const char* filePath,
        snd::internal::fnd::File::AccessMode mode)
{
    DWORD desiredAccess = ConvertAccessMode(mode);
    NW_ASSERT(desiredAccess > 0);

    HANDLE handle = CreateFileA(
        filePath,
        desiredAccess,
        FILE_SHARE_READ,
        NULL,
        OPEN_EXISTING,
        FILE_ATTRIBUTE_NORMAL,
        NULL);

    if (handle == INVALID_HANDLE_VALUE)
    {
        return false;
    }

    m_FileHandle = reinterpret_cast<void*>(handle);

    DWORD result = GetFileSize(m_FileHandle, NULL);
    if(result == -1)
    {
        return false;
    }
    m_FilePos = 0;
    m_FileSize = result;
    m_Available = true;
    return true;
}
#elif defined( NW_USE_NINTENDO_SDK )
bool
FsFileStream::Open(
        const char* filePath,
        snd::internal::fnd::File::AccessMode mode)
{
    nn::fs::OpenMode desiredAccess = ConvertAccessMode(mode);
    NW_ASSERT(desiredAccess > 0);

    nn::Result result;

    result = nn::fs::OpenFile(&m_FileHandle, filePath, desiredAccess);

    if (!result.IsSuccess())
    {
        return false;
    }

    int64_t fileSize;
    result = nn::fs::GetFileSize(&fileSize, m_FileHandle);
    if (!result.IsSuccess())
    {
        return false;
    }
    m_FilePos = 0;
    m_FileSize = static_cast<u32>(fileSize); // TODO: 64bit対応
    m_Available = true;

    return true;
}
#else
#error
#endif

//---------------------------------------------------------------------------
void
FsFileStream::Close()
{
    if ( ! m_DuplicateFlag )
    {
    #if defined( NW_PLATFORM_CAFE )
        NW_ASSERT_NOT_NULL(m_pPool);
        FSCmdBlock* block = m_pPool->Alloc();
        FSSetCmdPriority(block, m_Priority);
        NW_ASSERT_NOT_NULL(block);

        FSStatus status = FSCloseFile( m_pClient, block, m_FileHandle, Util::GetRetFlag() );
        NW_ASSERT( status == FS_STATUS_OK );

        m_pPool->Free(block);
        m_pPool = NULL;
        m_pClient = NULL;
    #elif defined(NW_PLATFORM_ANDROID)
        if(m_AccessMethodType == AccessMethodType_Stdio)
        {
            std::fclose(reinterpret_cast<FILE*>(m_FileHandle));
        }
        else if (m_AccessMethodType == AccessMethodType_AndroidAssets)
        {
            AAsset_close(reinterpret_cast<AAsset*>(m_FileHandle));
        }
        else
        {
            NW_INTERNAL_ERROR("m_AccessMethodType is invalid.");
        }
        m_FileHandle = NULL;
    #elif defined(NW_PLATFORM_IOS)
        std::fclose(reinterpret_cast<FILE*>(m_FileHandle));
        m_FileHandle = NULL;
    #elif defined( NW_PLATFORM_WIN32 )
        CloseHandle(m_FileHandle);
        m_FileHandle = NULL;
    #elif defined( NW_USE_NINTENDO_SDK )
        nn::fs::CloseFile(m_FileHandle);
    #else
        #error
    #endif

        m_FilePos = 0;
        m_FileSize = 0;
    }

    m_Available = false;
}

//---------------------------------------------------------------------------
s32
FsFileStream::Read( void* buf, u32 length )
{
    NW_ASSERT( m_Available );
    NW_ASSERTMSG( CanRead(), "Stream don't support Read function\n" );
    NW_ASSERT_ALIGN( buf, BUFFER_ALIGN_SIZE );

#if defined( NW_PLATFORM_CAFE )

    NW_ASSERT_NOT_NULL(m_pPool);
    FSCmdBlock* block = m_pPool->Alloc();
    FSSetCmdPriority(block, m_Priority);

    NW_ASSERT_NOT_NULL(block);
    FSStatus status = FSReadFile(
        m_pClient, block, buf, 1, length, m_FileHandle, 0, Util::GetRetFlag() );
    m_pPool->Free(block);

    if ( status >= FS_STATUS_OK )
    {
        m_FilePos += status;
    }
#if !defined(NW_RELEASE)
    else
    {
        NW_WARNING( status == FS_STATUS_OK, "FsFileStream::Read is Failed: %s\n", Util::FSStatusToString(status) );
    }
#endif
    return status;
#elif defined(NW_PLATFORM_ANDROID)
    u32 readFileLength = 0;

    if(m_AccessMethodType == AccessMethodType_Stdio)
    {
        readFileLength = fread(buf, 1, length, reinterpret_cast<FILE*>(m_FileHandle));
    }
    else if (m_AccessMethodType == AccessMethodType_AndroidAssets)
    {
        readFileLength = AAsset_read(reinterpret_cast<AAsset*>(m_FileHandle), buf, length);
    }
    else
    {
        NW_INTERNAL_ERROR("m_AccessMethodType is invalid.");
    }

    if (readFileLength < 0)
    {
        return -1;
    }

    m_FilePos += readFileLength;
    return readFileLength;
#elif defined(NW_PLATFORM_IOS)
    u32 readFileLength = fread(buf, 1, length, reinterpret_cast<FILE*>(m_FileHandle));

    if (readFileLength < 0)
    {
        return -1;
    }

    m_FilePos += readFileLength;
    return readFileLength;
#elif defined( NW_PLATFORM_WIN32 )
    DWORD readFileLength = 0;
    if ( ! ReadFile(m_FileHandle, buf, length, &readFileLength, NULL) )
    {
        return -1;
    }
    m_FilePos += readFileLength;
    return readFileLength;
#elif defined( NW_USE_NINTENDO_SDK )


    size_t readFileLength = 0;
    nn::Result nnResult = nn::fs::ReadFile(&readFileLength, m_FileHandle, m_FilePos, buf, length);

    if (!nnResult.IsSuccess())
    {
        return -1;
    }

    m_FilePos += readFileLength;
    return readFileLength;
#else
    #error
#endif

}

//---------------------------------------------------------------------------
bool
FsFileStream::ReadAsync(
    void*               /* buf */,
    u32                 /* length */,
    IoStreamCallback    /* callback */,
    void*               /* arg */
)
{
    NW_ASSERT( m_Available );
    NW_ASSERTMSG( CanRead(), "Stream don't support Read function\n" );
    NW_ASSERTMSG( CanAsync(), "Stream don't support Aync function\n" );

    return false;
}

//---------------------------------------------------------------------------
s32
FsFileStream::Write( const void* buf, u32 length )
{
    NW_ASSERT( m_Available );
    NW_ASSERTMSG( CanWrite(), "Stream don't support Write function\n" );

#if defined( NW_PLATFORM_CAFE )
    NW_ASSERT_NOT_NULL(m_pPool);
    FSCmdBlock* block = m_pPool->Alloc();
    FSSetCmdPriority(block, m_Priority);
    NW_ASSERT_NOT_NULL(block);
    FSStatus status = FSWriteFile(
        m_pClient, block, const_cast<void*>(buf), 1, length,
        m_FileHandle, 0, Util::GetRetFlag() );
    m_pPool->Free(block);

    if ( status >= FS_STATUS_OK )
    {
        m_FilePos += status;

        if ( m_FileSize < m_FilePos )
        {
            m_FileSize = m_FilePos;
        }
    }
#if !defined(NW_RELEASE)
    else
    {
        NW_WARNING( status == FS_STATUS_OK, "FsFileStream::Write is Failed: %s\n", Util::FSStatusToString(status) );
    }
#endif
    return status;
#elif defined(NW_PLATFORM_ANDROID)

#elif defined(NW_PLATFORM_IOS)
    return 0;
#elif defined( NW_PLATFORM_WIN32 )
    DWORD writtenFileLength = 0;
    if ( ! WriteFile(m_FileHandle, buf, length, &writtenFileLength, NULL) )
    {
        return -1;
    }
    m_FilePos += writtenFileLength;
    return writtenFileLength;
#elif defined( NW_USE_NINTENDO_SDK )


    nn::fs::WriteOption option;
    nn::Result nnResult = nn::fs::WriteFile(m_FileHandle, m_FilePos, buf, length, option);

    if (!nnResult.IsSuccess())
    {
        return -1;
    }
    m_FilePos += length;
    return length;
#else
    #error
#endif
}

//---------------------------------------------------------------------------
bool
FsFileStream::WriteAsync(
    const void*         /* buf */,
    u32                 /* length */,
    IoStreamCallback    /* callback */,
    void*               /* arg */
)
{
    NW_ASSERT( m_Available );
    NW_ASSERTMSG( CanWrite(), "Stream don't support Write function\n" );
    NW_ASSERTMSG( CanAsync(), "Stream don't support Aync function\n" );

    return false;
}

//---------------------------------------------------------------------------
s32
FsFileStream::WaitAsync()
{
    NW_ASSERT( m_Available );
    NW_ASSERTMSG( CanAsync(), "Stream don't support Aync function\n" );

    return -1;
}

//---------------------------------------------------------------------------
bool
FsFileStream::CanRead() const
{
    if ( ! m_Available ) { return false; }
#if 0
    return ( m_FileStat.permission & FS_STAT_PERM_READ ) ? true : false ;
#else
    return true;
#endif
}

//---------------------------------------------------------------------------
bool
FsFileStream::CanWrite() const
{
#if defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
    return false;
#endif

    if ( ! m_Available ) { return false; }
#if 0
    return ( m_FileStat.permission & FS_STAT_PERM_WRITE ) ? true : false ;
#else
    return true;
#endif
}

//---------------------------------------------------------------------------
u32
FsFileStream::GetSize() const
{
    NW_ASSERT( m_Available );

    return m_FileSize;
}

//---------------------------------------------------------------------------
bool
FsFileStream::Seek( s32 offset, u32 origin )
{
    NW_ASSERT( m_Available );

#if defined( NW_PLATFORM_CAFE )
    FSFilePosition pos;

    switch( origin )
    {
    case nw::ut::FILE_STREAM_SEEK_BEGIN:
        pos = offset;
        break;
    case nw::ut::FILE_STREAM_SEEK_CURRENT:
        pos = m_FilePos + offset;
        break;
    case nw::ut::FILE_STREAM_SEEK_END:
        pos = m_FileSize - offset;
        break;
    default:
        NW_ASSERTMSG( false, "FsFileStream::Seek Invalid origin parameter." );
        return false;
    }

    NW_ASSERT_NOT_NULL(m_pPool);
    FSCmdBlock* block = m_pPool->Alloc();
    FSSetCmdPriority(block, m_Priority);
    NW_ASSERT_NOT_NULL(block);
    FSStatus status = FSSetPosFile( m_pClient, block, m_FileHandle, pos, Util::GetRetFlag() );
    m_pPool->Free(block);

    if ( status != FS_STATUS_OK )
    {
#if !defined(NW_RELEASE)
        NW_WARNING( status == FS_STATUS_OK, "FSSetPosFile is Failed: status %s pos %d", Util::FSStatusToString(status), pos );
#endif
        return false;
    }
#elif defined(NW_PLATFORM_ANDROID)
    int moveMethod = SEEK_SET;
    u32 pos = 0;
    fpos_t fpos = 0;

    switch(origin)
    {
    case nw::ut::FILE_STREAM_SEEK_BEGIN:
        moveMethod = SEEK_SET;
        break;

    case nw::ut::FILE_STREAM_SEEK_END:
        moveMethod = SEEK_END;
        break;

    case nw::ut::FILE_STREAM_SEEK_CURRENT:
        moveMethod = SEEK_CUR;
        break;
    }

    int result;
    if(m_AccessMethodType == AccessMethodType_Stdio)
    {
        result = std::fseek(reinterpret_cast<FILE*>(m_FileHandle), offset, moveMethod);
        if(result < 0)
        {
            return false;
        }

        result = std::fgetpos(reinterpret_cast<FILE*>(m_FileHandle), &fpos);
        if(result < 0)
        {
            return false;
        }
        pos = fpos;
    }
    else if (m_AccessMethodType == AccessMethodType_AndroidAssets)
    {
        result = AAsset_seek(reinterpret_cast<AAsset*>(m_FileHandle), offset, moveMethod);
        if(result < 0)
        {
            return false;
        }
        pos = result;
    }
    else
    {
        NW_INTERNAL_ERROR("m_AccessMethodType is invalid.");
    }

    m_FilePos = pos;

    return true;
#elif defined(NW_PLATFORM_IOS)
    int moveMethod = SEEK_SET;
    u32 pos = 0;
    fpos_t fpos = 0;

    switch(origin)
    {
    case nw::ut::FILE_STREAM_SEEK_BEGIN:
        moveMethod = SEEK_SET;
        break;

    case nw::ut::FILE_STREAM_SEEK_END:
        moveMethod = SEEK_END;
        break;

    case nw::ut::FILE_STREAM_SEEK_CURRENT:
        moveMethod = SEEK_CUR;
        break;
    }

    int result = std::fseek(reinterpret_cast<FILE*>(m_FileHandle), offset, moveMethod);
    if(result < 0)
    {
        return false;
    }

    result = std::fgetpos(reinterpret_cast<FILE*>(m_FileHandle), &fpos);
    if(result < 0)
    {
        return false;
    }
    pos = fpos;

    m_FilePos = pos;

    return true;
#elif defined(NW_PLATFORM_WIN32)
    DWORD moveMethod = FILE_BEGIN;

    switch(origin)
    {
    case nw::ut::FILE_STREAM_SEEK_BEGIN:
        moveMethod = FILE_BEGIN;
        break;

    case nw::ut::FILE_STREAM_SEEK_END:
        moveMethod = FILE_END;
        break;

    case nw::ut::FILE_STREAM_SEEK_CURRENT:
        moveMethod = FILE_CURRENT;
        break;
    }

    u32 pos = SetFilePointer(m_FileHandle, offset, NULL, moveMethod);

    if(pos == INVALID_SET_FILE_POINTER)
    {
        return false;
    }
#elif defined( NW_USE_NINTENDO_SDK )
    switch( origin )
    {
    case nw::ut::FILE_STREAM_SEEK_BEGIN:
        m_FilePos = offset;
        break;
    case nw::ut::FILE_STREAM_SEEK_CURRENT:
        m_FilePos += offset;
        break;
    case nw::ut::FILE_STREAM_SEEK_END:
        m_FilePos = m_FileSize - offset;
        break;
    default:
        NW_ASSERTMSG( false, "FsFileStream::Seek Invalid origin parameter." );
        return false;
    }
#endif

    return true;
}

//---------------------------------------------------------------------------
void
FsFileStream::Cancel()
{
    NW_ASSERTMSG( CanCancel(), "Stream don't support Cancel function\n" );
}

//---------------------------------------------------------------------------
bool
FsFileStream::CancelAsync( IoStreamCallback /* callback */, void* /* arg */ )
{
    NW_ASSERT( m_Available );
    NW_ASSERTMSG( CanCancel(), "Stream don't support Cancel function\n" );
    NW_ASSERTMSG( CanAsync(), "Stream don't support Aync function\n" );

    return false;

}

}   /* namespace internal */
}   /* namespace snd */
}   /* namespace nw */

