﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_FrameHeap.h>

#include <nw/config.h>
#include <nw/snd/snd_BankFileReader.h>
#include <nw/snd/snd_GroupFileReader.h>
#include <nw/snd/snd_SequenceSoundFileReader.h>
#include <nw/snd/snd_WaveArchiveFileReader.h>
#include <nw/snd/snd_WaveFileReader.h>
#include <nw/snd/snd_WaveSoundFileReader.h>
#include <nw/snd/snd_Util.h>
#include <new>

namespace nw {
namespace snd {
namespace internal {

namespace
{
#ifdef NW_CONSOLE_ENABLE
u32 GetWaveArchiveId( u32 fileId, const SoundArchive& arc )
{
    for ( u32 j = 0; j < arc.GetWaveArchiveCount(); j++ )
    {
        SoundArchive::WaveArchiveInfo info;
        SoundArchive::ItemId id =
            Util::GetMaskedItemId( j, ItemType_WaveArchive );
        if ( arc.ReadWaveArchiveInfo( id, &info ) )
        {
            if ( info.fileId == fileId )
            {
                return id;
            }
        }
    }
    return SoundArchive::INVALID_ID;
}
#endif // NW_CONSOLE_ENABLE

} // anonymous namespace

/* ========================================================================
        member function
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         FrameHeap

  Description:  コンストラクタ

  Arguments:    なし

  Returns:      なし
 *---------------------------------------------------------------------------*/
FrameHeap::FrameHeap()
: m_pHeap( NULL )
{
}

/*---------------------------------------------------------------------------*
  Name:         ~FrameHeap

  Description:  デストラクタ

  Arguments:    なし

  Returns:      なし
 *---------------------------------------------------------------------------*/
FrameHeap::~FrameHeap()
{
    if ( IsValid() )
    {
        Destroy();
    }
}

/*---------------------------------------------------------------------------*
  Name:         Create

  Description:  ヒープを作成

  Arguments:    startAddress - 開始アドレス
                size         - メモリサイズ

  Returns:      ヒープハンドル
 *---------------------------------------------------------------------------*/
bool FrameHeap::Create( void* startAddress, u32 size )
{
    NW_NULL_ASSERT( startAddress );

    if ( IsValid() )
    {
        Destroy();
    }

    void* endAddress = static_cast<u8*>( startAddress ) + size;
    startAddress = ut::RoundUp( startAddress, 4 ); // Heap align

    if ( startAddress > endAddress )
    {
        return false;
    }

    size = static_cast<u32>( static_cast<u8*>( endAddress ) - static_cast<u8*>( startAddress ) );

    m_pHeap = ut::FrameHeap::Create( startAddress, size );
    if ( m_pHeap == NULL )
    {
        return false;
    }

    // ベースセクションの作成
    if ( ! NewSection() )
    {
        return false;
    }

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         Destroy

  Description:  ヒープを破棄します

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FrameHeap::Destroy()
{
    if ( IsValid() )
    {
        // セクションの破棄
        ClearSection();

        // ヒープのクリア
        m_pHeap->Free( ut::FrameHeap::FREE_ALL );

        // ヒープの破棄
        m_pHeap->Destroy();

        m_pHeap = NULL;
    }
}

/*---------------------------------------------------------------------------*
  Name:         Clear

  Description:  ヒープを作成時の状態に戻す

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FrameHeap::Clear()
{
    NW_ASSERT( IsValid() );

    // セクションの破棄
    ClearSection();

    // ヒープのクリア
    m_pHeap->Free( ut::FrameHeap::FREE_ALL );

    // ベースセクションの作成
    bool result = NewSection();
    NW_ASSERTMSG( result, "FrameHeap::Clear(): NewSection is Failed");
}

/*---------------------------------------------------------------------------*
  Name:         Alloc

  Description:  ヒープからメモリを確保

  Arguments:    size     - メモリサイズ
                callback - メモリが破棄されたときに呼びだされるコールバック関数
                callbackArg - コールバック引数

  Returns:      確保したメモリへのポインタ
 *---------------------------------------------------------------------------*/
void* FrameHeap::Alloc( u32 size, FrameHeap::DisposeCallback callback, void* callbackArg )
{
    NW_ASSERT( IsValid() );

    const u32 blockSize = ut::RoundUp( sizeof(Block), HEAP_ALIGN );
    const u32 allocSize = blockSize + ut::RoundUp( size, HEAP_ALIGN );
    void* mem = m_pHeap->Alloc( allocSize, HEAP_ALIGN );
    if ( mem == NULL )
    {
        return NULL;
    }
    void* buffer = ut::AddOffsetToPtr( mem, blockSize );
    NW_ASSERTMSG( ( reinterpret_cast<IntPtr>(buffer) & 0x1f ) == 0, "FrameHeap::Alloc: Internal Error" );

    Block* block = new ( mem ) Block( buffer, size, callback, callbackArg );

    m_SectionList.GetBack().AppendBlock( block );
    // NW_LOG("AppendBlock block(%p) allocSize(%d) callback(%p) arg(%p)\n",
    //         block, allocSize, callback, callbackArg );

    return buffer;
}

/*---------------------------------------------------------------------------*
  Name:         SaveState

  Description:  ヒープの状態を保存

  Arguments:    None.

  Returns:      保存した階層レベルを返す
                失敗時には、-1
 *---------------------------------------------------------------------------*/
int FrameHeap::SaveState()
{
    NW_ASSERT( IsValid() );

    if ( ! m_pHeap->RecordState( m_SectionList.GetSize() ) )
    {
        return -1;
    }

    if ( ! NewSection() )
    {
        int result = m_pHeap->FreeByState( 0 );
        NW_ASSERTMSG( result, "FrameHeap::SaveState(): ut::FrameHeap::FreeByState is Failed");
        return -1;
    }

    return static_cast<int>( m_SectionList.GetSize() ) - 1;
}

// 「指定レベルまでに含まれているすべての Block」で、
// 設定されているコールバック・コールバック引数が同じ場合は、
// 当該コールバックをまとめて呼び出します。
// 同じでない場合は、コールバックは呼び出さず、false を返します。
bool FrameHeap::ProcessCallback( int level )
{
    void* begin = NULL;
    void* end = NULL;
    DisposeCallback callback = NULL;
    const void* callbackArg = NULL;
    bool isSameCallback = true;
    bool useCallback = false;

    int sectionCount = m_SectionList.GetSize();
    for ( SectionList::Iterator itr = m_SectionList.GetEndIter();
            itr != m_SectionList.GetBeginIter(); )
    {
        sectionCount--;
        SectionList::Iterator curItr = --itr;

        BlockList& list = curItr->GetBlockList();
        // NW_LOG("[%d](%p) blockSize(%d) isEmpty(%d)\n",
        //         sectionCount, &*curItr, list.GetSize(), list.IsEmpty());

        // end の決定、callback* の初期化
        if ( end == NULL && list.GetSize() > 0 )
        {
            const Block& last = list.GetBack();
            end = const_cast<void*>(ut::AddOffsetToPtr(
                        last.GetBufferAddr(), last.GetBufferSize() ) );

            callback = last.GetDisposeCallback();
            callbackArg = last.GetDisposeCallbackArg();

        }

        // 各ブロックで callback* に変更が無いか？
        for ( BlockList::Iterator blockItr = list.GetBeginIter();
                blockItr != list.GetEndIter(); )
        {
            BlockList::Iterator curBlockItr = blockItr++;
            DisposeCallback cb = curBlockItr->GetDisposeCallback();
            const void* cbArg = curBlockItr->GetDisposeCallbackArg();

            if ( callback != cb || callbackArg != cbArg )
            {
                isSameCallback = false;
                break;
            }
        }

        // begin の設定
        if ( sectionCount == level )
        {
            begin = &*curItr;
            break;
        }
    }
    if ( end == NULL )
    {
        isSameCallback = false;
    }

    // NW_LOG("begin(%p) end(%p) isSameCallback(%d)\n", begin, end, isSameCallback );
    if ( isSameCallback )
    {
        if ( callback )
        {
            callback( begin, ut::GetOffsetFromPtr(begin, end),
                    const_cast<void*>(callbackArg) );
        }
        useCallback = true;
    }

    return useCallback;
}

/*---------------------------------------------------------------------------*
  Name:         LoadState

  Description:  ヒープの状態を戻す

  Arguments:    level - 階層レベル

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FrameHeap::LoadState( int level )
{
    NW_ASSERT( IsValid() );
    NW_MINMAXLT_ASSERT( level, 0, static_cast<int>( m_SectionList.GetSize() ) );

    if ( level == 0 )
    {
        Clear();
        return;
    }

    bool alreadyUseCallback = ProcessCallback( level );

    // セクションの整理
    while ( level < static_cast<int>( m_SectionList.GetSize() ) )
    {
        // NW_LOG("LoadState(%d) / size(%d)\n", level, m_SectionList.GetSize());

        // get latest section
        Section& section = m_SectionList.GetBack();

        if ( alreadyUseCallback )
        {
            section.SetUseCallback( false );
        }

        // call dispose callback
        section.~Section();

        // セクションリストからの削除
        m_SectionList.Erase( &section );
    }

    // ヒープ状態を復元
    int result = m_pHeap->FreeByState( static_cast<u32>( level ) );
    NW_UNUSED_VARIABLE( result );
    NW_ASSERTMSG( result, "FrameHeap::LoadState(): ut::FrameHeap::FreeByState is Failed");

    // 再度記録
    result =  m_pHeap->RecordState( m_SectionList.GetSize() );
    NW_ASSERTMSG( result, "FrameHeap::LoadState(): ut::FrameHea::RecordState is Failed");

    // セクションの作成
    bool result2 = NewSection();
    NW_ASSERTMSG( result2, "FrameHeap::LoadState(): NewSection is Failed");
}

/*---------------------------------------------------------------------------*
  Name:         GetCurrentLevel

  Description:  ヒープの現在の階層レベルを取得

  Arguments:    None.

  Returns:      現在の階層レベル
 *---------------------------------------------------------------------------*/
int FrameHeap::GetCurrentLevel() const
{
    NW_ASSERT( IsValid() );

    return static_cast<int>( m_SectionList.GetSize() ) - 1;
}

/*---------------------------------------------------------------------------*
  Name:         GetSize

  Description:  ヒープの容量を取得

  Arguments:    None.

  Returns:      ヒープの容量
 *---------------------------------------------------------------------------*/
u32 FrameHeap::GetSize() const
{
    NW_ASSERT( IsValid() );

    return static_cast<u32>(
        static_cast<u8*>( m_pHeap->GetHeapEndAddress() ) -
        static_cast<u8*>( m_pHeap->GetHeapStartAddress() )
    );
}

/*---------------------------------------------------------------------------*
  Name:         GetFreeSize

  Description:  ヒープの空き容量を取得

  Arguments:    None.

  Returns:      空き容量
 *---------------------------------------------------------------------------*/
u32 FrameHeap::GetFreeSize() const
{
    NW_ASSERT( IsValid() );

    u32 size = m_pHeap->GetAllocatableSize( HEAP_ALIGN );

    if ( size < sizeof( Block ) )
    {
        return 0;
    }
    size -= sizeof( Block );

    size &= ~(HEAP_ALIGN-1);

    return size;
}

/*---------------------------------------------------------------------------*
  Name:         NewSection

  Description:  新しいセクションを作成

  Arguments:    None.

  Returns:      成功したかどうか
 *---------------------------------------------------------------------------*/
bool FrameHeap::NewSection()
{
    // new HeapSection
    void* buffer = m_pHeap->Alloc( sizeof( Section ) );
    if ( buffer == NULL )
    {
        return false;
    }
    ut::internal::Memset( buffer, 0x0, sizeof(Section) );

    Section* section = new( buffer ) Section();
    m_SectionList.PushBack( section );
    return true;
}

/*---------------------------------------------------------------------------*
  Name:         ClearSection

  Description:  セクションを全て破棄する

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void FrameHeap::ClearSection()
{
    bool alreadyUseCallback = ProcessCallback( 0 );

    // セクションの破棄
    while ( !m_SectionList.IsEmpty() )
    {
        Section& section = m_SectionList.GetBack();

        if ( alreadyUseCallback )
        {
            section.SetUseCallback( false );
        }

        // コールバックの呼び出し
        section.~Section();

        // セクションリストからの削除
        m_SectionList.Erase( &section );
    }
}

void FrameHeap::Dump(
        nw::snd::SoundDataManager& mgr, nw::snd::SoundArchive& arc ) const
{
#ifdef NW_CONSOLE_ENABLE
    int i = 0;
    for ( SectionList::ConstIterator itr = m_SectionList.GetBeginIter();
            itr != m_SectionList.GetEndIter(); )
    {
        SectionList::ConstIterator curItr = itr++;
        NW_LOG("section[%d]\n", i++ );
        curItr->Dump( mgr, arc );
    }
#else // NW_CONSOLE_ENABLE
    (void)mgr;
    (void)arc;
#endif // NW_CONSOLE_ENABLE
}

/* ========================================================================
        FrameHeap::Section class member function
   ======================================================================== */

FrameHeap::Section::Section()
: m_UseCallback(true)
{
    m_BlockList.clear();
}

FrameHeap::Section::~Section()
{
    if ( m_UseCallback )
    {
        BlockList::Iterator itr = m_BlockList.GetEndIter();
        while ( itr != m_BlockList.GetBeginIter() )
        {
            (void)--itr;
            itr->~Block();
        }
     }
     m_BlockList.clear();
}

void FrameHeap::Section::AppendBlock( Block* block )
{
    m_BlockList.PushBack( block );
}

void FrameHeap::Section::Dump(
        nw::snd::SoundDataManager& mgr, nw::snd::SoundArchive& arc ) const
{
#ifdef NW_CONSOLE_ENABLE
    int i = 0;
    for ( BlockList::ConstIterator itr = m_BlockList.GetBeginIter();
            itr != m_BlockList.GetEndIter(); )
    {
        BlockList::ConstIterator curItr = itr++;
        const void* ptr = curItr->GetBufferAddr();

        u32 sign = *reinterpret_cast<const u32*>(ptr);

        char signature[5];
        signature[4] = '\0';
        std::memcpy( signature, &sign, 4 );

        u32 fileId = mgr.detail_GetFileIdFromTable( ptr );

        SoundArchive::ItemId itemId = SoundArchive::INVALID_ID;
        u32 waveIndex = 0xffffffff;
        switch ( sign )
        {
        case BankFileReader::SIGNATURE_FILE:
            for ( u32 j = 0; j < arc.GetBankCount(); j++ )
            {
                SoundArchive::BankInfo info;
                SoundArchive::ItemId id =
                    Util::GetMaskedItemId( j, ItemType_Bank );
                if ( arc.ReadBankInfo( id, &info ) )
                {
                    if ( info.fileId == fileId )
                    {
                        itemId = id;
                        break;
                    }
                }
            }
            break;
        case GroupFileReader::SIGNATURE_FILE:
            for ( u32 j = 0; j < arc.GetGroupCount(); j++ )
            {
                SoundArchive::GroupInfo info;
                SoundArchive::ItemId id =
                    Util::GetMaskedItemId( j, ItemType_Group );
                if ( arc.ReadGroupInfo( id, &info ) )
                {
                    if ( info.fileId == fileId )
                    {
                        itemId = id;
                        break;
                    }
                }
            }
            break;
        case SequenceSoundFileReader::SIGNATURE_FILE:
        case WaveSoundFileReader::SIGNATURE_FILE:
            for ( u32 j = 0; j < arc.GetSoundCount(); j++ )
            {
                SoundArchive::SoundInfo info;
                SoundArchive::ItemId id =
                    Util::GetMaskedItemId( j, ItemType_Sound );
                if ( arc.ReadSoundInfo( id, &info ) )
                {
                    if ( info.fileId == fileId )
                    {
                        itemId = id;
                        break;
                    }
                }
            }
            break;
        case WaveArchiveFileReader::SIGNATURE_FILE:
            itemId = GetWaveArchiveId( fileId, arc );
            break;
        // 個別ロードの波形
        case SoundDataManager::SIGNATURE_INDIVIDUAL_WAVE:
            {
                const u32* pU32Array = reinterpret_cast<const u32*>(ptr);
                fileId = pU32Array[1];
                waveIndex = pU32Array[2];
                itemId = GetWaveArchiveId( fileId, arc );
            }
            break;
        default:
            itemId = SoundArchive::INVALID_ID;
            break;
        }

        const char* pItemLabel = arc.GetItemLabel( itemId );
        if ( pItemLabel != NULL )
        {
            NW_LOG("  block[%d] 0x%08X [%s] fileId(%6d) itemId(%08X) [%s]",
                    i++, ptr, signature, fileId, itemId, pItemLabel );
        }
        else
        {
            NW_LOG("  block[%d] 0x%08X [%s] fileId(%6d) itemId(%08X) [(anonymous)]",
                    i++, ptr, signature, fileId, itemId );
        }

        if ( waveIndex != 0xffffffff )
        {
            NW_LOG("(%d)\n", waveIndex );
        }
        else
        {
            NW_LOG("\n");
        }
    }
#else
    (void)mgr;
    (void)arc;
#endif // NW_CONSOLE_ENABLE
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

