﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/ut/ut_FileStream.h>
#include <nw/snd/snd_SoundSystem.h>
#include <nw/snd/snd_DeviceOutRecorder.h>

namespace nw {
namespace snd {

#ifdef NW_PLATFORM_CAFE
const u32 DeviceOutRecorder::FILE_IO_BUFFER_ALIGNMENT = PPC_IO_BUFFER_ALIGN;
#else
const u32 DeviceOutRecorder::FILE_IO_BUFFER_ALIGNMENT = 1;
#endif

//---------------------------------------------------------------------------
DeviceOutRecorder::RecorderBuffer::RecorderBuffer(const char* deviceName/*=NULL*/) :
m_SampleBuffer(NULL),
m_MaxBufferSamples(0),
m_MaxSamples(0),
m_ValidSamples(0),
m_ReadPosition(0),
m_WritePosition(0),
m_ReadBlockSamples(1),
m_DeviceName(deviceName == NULL ? "" : deviceName)
{
#if defined(NW_DEBUG) || defined(NW_DEVELOP)
    m_IsWarnedDropSamples = false;
#endif

#ifdef NW_SND_DEBUG_REC_BUFFER
    m_PushCount = 0;
#endif
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::RecorderBuffer::Initialize(
    s16* sampleBuffer,
    u32 maxSamples)
{
    NW_ASSERT_NOT_NULL(sampleBuffer);
    NW_ASSERT(maxSamples > 0);

    m_SampleBuffer = sampleBuffer;
    m_MaxBufferSamples = maxSamples;

    UpdateMaxSamples();
    Clear();
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::RecorderBuffer::Finalize()
{
    m_SampleBuffer = NULL;
    m_MaxSamples = 0;
    m_ReadBlockSamples = 1;
}

//---------------------------------------------------------------------------
u32
DeviceOutRecorder::RecorderBuffer::Push(const s16* sampleBuffer, u32 samples)
{
    if(samples == 0)
    {
        return 0;
    }

#ifdef NW_SND_DEBUG_REC_BUFFER
    if(m_PushCount++ == s_PrintInterval)
    {
        NW_LOG("[DeviceOutRecorder:%4s] buffer : %8d / %8d samples\n", m_DeviceName, m_ValidSamples, m_MaxSamples);
        m_PushCount = 0;
    }
#endif // NW_SND_DEBUG_REC_BUFFER

    NW_ASSERT_NOT_NULL(sampleBuffer);

    // 書き込みポインタ⇔バッファ終端     への書き込み、または
    // 書き込みポインタ⇔読み込みポインタ への書き込み。
    u32 copyCount = ut::Min(samples, GetContiguousWritableCount());

    if(copyCount > 0)
    {
        Write(sampleBuffer, copyCount);
    }

    if(copyCount == samples)
    {
        return copyCount;
    }

    // バッファ先頭⇔書き込みポインタ への書き込み、または、バッファが一杯。
    NW_ASSERT(m_WritePosition == 0 || m_WritePosition == m_ReadPosition);

    u32 restCount = ut::Min(samples - copyCount, GetContiguousWritableCount());

    if(restCount > 0)
    {
        Write(sampleBuffer + copyCount, restCount);
    }

    if(copyCount + restCount == samples)
    {
        return samples;
    }

#if defined(NW_DEBUG) || defined(NW_DEVELOP)
    if(!m_IsWarnedDropSamples)
    {
        m_IsWarnedDropSamples = true;
        NW_WARNING(
            false,
            "[DeviceOutRecorder:%4s] buffer is full(%dsamples). dropped samples.\n", m_DeviceName, m_ValidSamples);
    }
#endif

    return copyCount + restCount;
}

//---------------------------------------------------------------------------
u32
DeviceOutRecorder::RecorderBuffer::Pop(u32 samples)
{
    if(samples == 0)
    {
        return 0;
    }

    // 読み込みポインタを後方へ移動
    u32 skipCount = ut::Min(samples, GetContiguousReadableCount());
    Skip(skipCount);

    if(skipCount == samples)
    {
        return skipCount;
    }

    // 読み込みポインタをバッファ先頭に戻して移動
    NW_ASSERT(m_ReadPosition == 0);

    u32 restCount = ut::Min(samples - skipCount, GetContiguousReadableCount());
    Skip(restCount);

    return skipCount + restCount;
}

//---------------------------------------------------------------------------
s16*
DeviceOutRecorder::RecorderBuffer::Peek()
{
    NW_ASSERT_NOT_NULL(m_SampleBuffer);
    return GetReadableCount() > 0 ? m_SampleBuffer + m_ReadPosition : NULL;
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::RecorderBuffer::SetReadBlockSamples(u32 value)
{
    m_ReadBlockSamples = ut::Max<u32>(1, value);
    UpdateMaxSamples();
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::RecorderBuffer::Clear()
{
    m_ValidSamples = 0;
    m_ReadPosition = 0;
    m_WritePosition = 0;

#if defined(NW_DEBUG) || defined(NW_DEVELOP)
    m_IsWarnedDropSamples = false;
#endif
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::RecorderBuffer::UpdateMaxSamples()
{
    if(m_ReadBlockSamples == 0)
    {
        m_MaxSamples = m_MaxBufferSamples;
    }
    else
    {
        // アライメント調整のため、ブロック単位で読み込むので、
        // ここで有効サンプル数を調整します。
        m_MaxSamples = m_MaxBufferSamples / m_ReadBlockSamples * m_ReadBlockSamples;
        NW_ASSERT(m_ReadBlockSamples <= m_MaxSamples);
    }
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::RecorderBuffer::Skip(u32 samples)
{
    NW_ASSERT(m_ValidSamples >= samples);

    m_ReadPosition = IncrementPosition(m_ReadPosition, samples);
    m_ValidSamples -= samples;
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::RecorderBuffer::Write(const s16* sampleBuffer, u32 samples)
{
    NW_ASSERT_NOT_NULL(m_SampleBuffer);
    NW_ASSERT(m_ValidSamples + samples <= m_MaxSamples);

    for(u32 i = 0; i < samples; ++i)
    {
        m_SampleBuffer[m_WritePosition + i] = sampleBuffer[i];
    }

    m_WritePosition = IncrementPosition(m_WritePosition, samples);
    m_ValidSamples += samples;
}

//---------------------------------------------------------------------------
u32
DeviceOutRecorder::RecorderBuffer::GetWritableCount() const
{
    return m_MaxSamples - m_ValidSamples;
}

//---------------------------------------------------------------------------
u32
DeviceOutRecorder::RecorderBuffer::GetContiguousReadableCount() const
{
    u32 samples = 0;

    if(m_ReadPosition <= m_WritePosition)
    {
        samples = m_ValidSamples;
    }
    else
    {
        samples = m_MaxSamples - m_ReadPosition;
    }

    if(m_ReadBlockSamples == 0)
    {
        return samples;
    }
    else
    {
        return samples / m_ReadBlockSamples * m_ReadBlockSamples;
    }
}

//---------------------------------------------------------------------------
u32
DeviceOutRecorder::RecorderBuffer::GetContiguousWritableCount() const
{
    if(m_WritePosition <= m_ReadPosition)
    {
        return m_MaxSamples - m_ValidSamples;
    }

    return m_MaxSamples - m_WritePosition;
}

//---------------------------------------------------------------------------
u32
DeviceOutRecorder::RecorderBuffer::IncrementPosition(u32 position, u32 length)
{
    NW_ASSERT(length <= m_MaxSamples);

    u32 result = position + length;
    return result >= m_MaxSamples ? result - m_MaxSamples : result;
}

//---------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32)
#pragma warning(push)
#pragma warning(disable:4355) // warning: used in base member initializer list
#endif
DeviceOutRecorder::DeviceOutRecorder(const char* deviceName/*=NULL*/) :
m_State(STATE_NOT_INITIALIZED),
m_Channels(0),
m_OutputMode(OUTPUT_MODE_STEREO),
m_IsLeadSilenceTrimming(false),
m_MaxSamples(0),
m_WrittenSamples(0),
m_Thread(this),
m_ExitThreadEvent(true, true),
m_MessageResult(0),
m_MessageDoneEvent(true, false),
m_Buffer(deviceName)
{
}
#if defined(NW_PLATFORM_WIN32)
#pragma warning(pop) // disable:4355
#endif

//---------------------------------------------------------------------------
DeviceOutRecorder::~DeviceOutRecorder()
{
    Finalize();
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::Initialize(void* buf, u32 length)
{
    NW_ASSERT_NOT_NULL(buf);

    if(IsInitialized())
    {
        return;
    }

    s16* alignedBuffer = reinterpret_cast<s16*>(
        nw::ut::RoundUp(buf, FILE_IO_BUFFER_ALIGNMENT));

    // アライメント後の各タスク用バッファを
    // MAX_FRAME_LENGTH で割り切れるサイズに丸めます。
    const u32 maxFrameLength = GetMaxFrameLength();
    const u32 bufferOffset = reinterpret_cast<u32>(alignedBuffer) - reinterpret_cast<u32>(buf);
    const u32 bufferLength = (length - bufferOffset) / maxFrameLength * maxFrameLength;

    NW_ASSERT(bufferLength >= MIN_BUFFER_LENGTH);

    m_Buffer.Initialize(alignedBuffer, bufferLength / sizeof(s16));
    m_MessageQueue.Initialize(&m_Message, 1);

    m_State = STATE_INITIALIZED;
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::Finalize()
{
    if(!IsInitialized())
    {
        return;
    }

    Stop(true);

    m_MessageQueue.Finalize();

    m_Buffer.Clear();
    m_Buffer.Finalize();

    m_State = STATE_NOT_INITIALIZED;
}

//---------------------------------------------------------------------------
bool
DeviceOutRecorder::Start(ut::FileStream& fileStream, const Options& options)
{
    if(!IsInitialized() || m_State == STATE_STOPPING)
    {
        return false;
    }

    if(m_State == STATE_RECORDING)
    {
        return true;
    }

    m_OutputMode = SoundSystem::GetOutputMode();
    m_Channels = GetValidChannels(options.channels);
    m_Stream = &fileStream;

    m_IsLeadSilenceTrimming = options.isLeadSilenceTrimmingEnabled;
    m_MaxSamples = options.maxFrames * m_Channels;

    m_Buffer.SetReadBlockSamples(GetReadBlockSamples(m_Channels));
    m_Buffer.Clear();

    if(!StartThread(options.priority))
    {
        Stop(false);
        return false;
    }

    if(Prepare() < 0)
    {
        Stop(false);
        return false;
    }

    OnStart();

    return true;
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::Stop(bool isBlocking)
{
    if(!IsInitialized())
    {
        return;
    }

    State state = m_State;

    if(state == STATE_RECORDING || state == STATE_RECORDED)
    {
        OnStop();
    }

    StopThread(isBlocking);

    m_Stream = NULL;
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::PushSamples(const s16* sampleBuffer, u32 samples)
{
    if(!IsInitialized() || m_State != STATE_RECORDING)
    {
        return;
    }

    NW_ASSERT_NOT_NULL(sampleBuffer);
    m_Buffer.Push(sampleBuffer, samples);

    PostMessage(MESSAGE_WRITE_SAMPLES);
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::ThreadHandlerProc()
{
    bool shouldExit = false;

    while(IsInitialized() && !shouldExit)
    {
        u32 message;
        if(!m_MessageQueue.Recv(reinterpret_cast<ut::MessageQueue::MessageType*>(&message), true))
        {
            NW_WARNING(false, "[DeviceOutRecorder] failed to MessageQueue::Recv().\n");
            shouldExit = true;
        }

        switch(message)
        {
        case MESSAGE_PREPARE:
            m_MessageResult = OnPrepare();

            if(m_MessageResult >= 0)
            {
                m_State = STATE_RECORDING;
            }

            m_MessageDoneEvent.Signal();
            break;

        case MESSAGE_EXIT:
            m_State = STATE_STOPPING;
            OnWriteSamples();
            shouldExit = true;
            break;

        case MESSAGE_WRITE_SAMPLES:
            if(!OnWriteSamples())
            {
                shouldExit = true;
            }
            break;

        default:
            continue;
        }
    }

    NW_ASSERT(IsInitialized());
    OnExit();

    switch(m_State)
    {
    case STATE_RECORDING:
        m_State = STATE_RECORDED;
        break;

    case STATE_STOPPING:
        m_State = STATE_INITIALIZED;
        break;

    default:
        break;
    }

    m_ExitThreadEvent.Signal();
}

//---------------------------------------------------------------------------
u32
DeviceOutRecorder::GetReadBlockSamples(u32 channels) const
{
    // フレーム長と FILE_IO_BUFFER_ALIGNMENT の最小公倍数であるべき。
    return channels * sizeof(s16) * FILE_IO_BUFFER_ALIGNMENT;
}

//---------------------------------------------------------------------------
u32
DeviceOutRecorder::GetLeadSilenceSamples(const s16* sampleBuffer, u32 samples, u32 channels) const
{
    NW_ASSERT_NOT_NULL(sampleBuffer);
    NW_ASSERT(channels > 0);

    u32 result = 0;

    for(u32 frameLeadIndex = 0; frameLeadIndex < samples; frameLeadIndex += channels)
    {
        bool isSilentFrame = true;

        // 全チャンネルが無音かどうかを調べます。
        for(u32 sampleOffset = 0; sampleOffset < channels; ++sampleOffset)
        {
            if(sampleBuffer[frameLeadIndex + sampleOffset] != 0)
            {
                isSilentFrame = false;
                break;
            }
        }

        if(!isSilentFrame)
        {
            break;
        }

        // 無音部分のサンプル数をフレーム単位で加算します。
        result += channels;
    }

    return result;
}

//---------------------------------------------------------------------------
u32
DeviceOutRecorder::GetWritableSamples(u32 samples) const
{
    if(m_MaxSamples == 0)
    {
        return samples;
    }

    if(IsNoMoreSamples())
    {
        return 0;
    }

    return ut::Min(samples, m_MaxSamples - m_WrittenSamples);
}

//---------------------------------------------------------------------------
bool
DeviceOutRecorder::IsNoMoreSamples() const
{
    return m_MaxSamples > 0 && m_MaxSamples <= m_WrittenSamples;
}

//---------------------------------------------------------------------------
bool
DeviceOutRecorder::StartThread(u32 priority)
{
    if(!IsInitialized() || m_State == STATE_STOPPING)
    {
        return false;
    }

    if(m_State == STATE_RECORDING)
    {
        return true;
    }

    NW_ASSERT(m_ExitThreadEvent.TryWait());
    m_Thread.Destroy();

    // メッセージキューを空にします。
    u32 message;
    while(m_MessageQueue.Recv(reinterpret_cast<ut::MessageQueue::MessageType*>(&message), false)) { }

    m_MessageDoneEvent.Reset();

    // スレッドを再利用可能にするために、DETACH 属性を付加します。
    ut::Thread::CreateArg arg;
    arg.priority = priority;
#if defined(NW_PLATFORM_CAFE)
    arg.stackBase = m_ThreadStack;
    arg.stackSize = THREAD_STACK_SIZE;
    arg.attribute = OS_THREAD_ATTR_DETACH;
    arg.nameString = "nw::snd::WavRecordingThread";
#endif

    if(!m_Thread.Create(arg))
    {
        NW_WARNING(false, "[DeviceOutRecorder] failed to StartThread.\n");
        return false;
    }

    m_ExitThreadEvent.Reset();
    m_Thread.Resume();

    return true;
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::StopThread(bool isBlocking)
{
    if(!IsInitialized())
    {
        return;
    }

    // 録音が完了して、STATE_INITIALIZED に移行待ちとみなし、状態を更新します。
    if(m_State == STATE_INITIALIZED || m_State == STATE_RECORDED)
    {
        m_State = STATE_INITIALIZED;
        return;
    }

    if(IsInitialized() && m_State != STATE_STOPPING)
    {
        SendMessage(MESSAGE_EXIT);
    }

    if(isBlocking)
    {
        // スレッドを再利用可能にするために、DETACH 属性をつけているので、
        // スレッドの終了待ちには、Join() ではなく、イベントを利用しています。
        m_ExitThreadEvent.Wait();
    }
}

//---------------------------------------------------------------------------
s32
DeviceOutRecorder::Prepare()
{
    if(!IsInitialized())
    {
        return -1;
    }

    m_MessageDoneEvent.Reset();

    bool result = m_MessageQueue.Send(
        reinterpret_cast<ut::MessageQueue::MessageType>(MESSAGE_PREPARE),
        true);

    if(!result)
    {
        NW_ASSERTMSG(false, "[DeviceOutRecorder] failed to send message to worker thread.\n");
        return -1;
    }

    m_MessageDoneEvent.Wait();

    return m_MessageResult;
}

//---------------------------------------------------------------------------
bool
DeviceOutRecorder::SendMessage(Message message)
{
    if(!IsInitialized())
    {
        return false;
    }

    return m_MessageQueue.Send(
        reinterpret_cast<ut::MessageQueue::MessageType>(message),
        true);
}

//---------------------------------------------------------------------------
bool
DeviceOutRecorder::PostMessage(Message message)
{
    if(!IsInitialized())
    {
        return false;
    }

    return m_MessageQueue.Send(
        reinterpret_cast<ut::MessageQueue::MessageType>(message),
        false);
}

//---------------------------------------------------------------------------
s32
DeviceOutRecorder::OnPrepare()
{
    if(m_Stream == NULL ||
        !m_Stream->IsAvailable())
    {
        return -1;
    }

    if(!m_WavOutStream.Open(*m_Stream, m_Channels, GetSamplesPerSec()))
    {
        return -1;
    }

    return m_WavOutStream.IsAvailable() ? 0 : -1;
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::OnExit()
{
    m_WavOutStream.Close();
}

//---------------------------------------------------------------------------
bool
DeviceOutRecorder::OnWriteSamples()
{
    if(!IsInitialized())
    {
        return false;
    }

    while(m_State == STATE_RECORDING || m_State == STATE_STOPPING)
    {
        u32 samples = m_Buffer.GetContiguousReadableCount();

        // IsRecording() の場合、
        // バッファ終端でなく、WRITE_BLOCK_PER_SAMPLES に満たない場合は、
        // サンプルが十分に溜まるまで書き込みません。
        if(m_State == STATE_RECORDING &&
            samples == m_Buffer.GetReadableCount() &&
            samples < WRITE_BLOCK_PER_SAMPLES)
        {
            break;
        }

        s16* sampleBuffer = m_Buffer.Peek();

        if(sampleBuffer == NULL)
        {
            break;
        }

        u32 validSamples = samples;

        // 先頭の無音部分をトリミングします。
        if(m_IsLeadSilenceTrimming)
        {
            u32 silenceSamples = GetLeadSilenceSamples(sampleBuffer, samples, m_Channels);

            sampleBuffer += silenceSamples;
            validSamples -= silenceSamples;

            // 発音を確認したら以降はトリミングを行わないようにします。
            if(silenceSamples < samples)
            {
                m_IsLeadSilenceTrimming = false;
            }
        }

        if(validSamples > 0 && !IsNoMoreSamples())
        {
            // DMA バッファの内容を波形出力用に加工します。
            validSamples = OnProcessSamples(sampleBuffer, validSamples);

            // 書き込み可能なサンプル数を求めます。
            validSamples = GetWritableSamples(validSamples);

            if(!m_WavOutStream.IsAvailable())
            {
                return false;
            }

            // 加工した波形データを出力します。
            u32 writeLength = validSamples * sizeof(s16);
            s32 result = m_WavOutStream.Write(sampleBuffer, writeLength);

            if(result != static_cast<s32>(writeLength))
            {
                NW_WARNING(false, "[DeviceOutRecorder] failed to WriteFile. : %d\n", result);
            }

            m_WrittenSamples += validSamples;
        }

        m_Buffer.Pop(samples);
    }

    return !IsNoMoreSamples();
}

} // snd
} // nw
