﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_AxVoice.h>
#include <nw/snd/snd_Config.h>
#include <nw/types.h>
#include <nw/assert.h>
#include <nw/ut.h>
#if !defined(NW_PLATFORM_ANDROID) && !defined(NW_PLATFORM_IOS)
#include <nw/dbg.h>
#endif
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_Debug.h>
#include <nw/snd/snd_HardwareManager.h>
#include <nw/snd/snd_VoiceCommand.h>

#if defined( NW_PLATFORM_CAFE )
  #include <cafe.h>
#elif defined( NW_PLATFORM_WIN32 )
  #include <winext/cafe.h>
  #include <winext/cafe/os/OSInterrupt.h>
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
  #include <winext/cafe.h>
  #include <winext/cafe/os/OSInterrupt.h>
#elif defined( NW_USE_NINTENDO_SDK )
  // TODO: nn_audio 対応時に排他制御が必要であるかを判断
  #include <winext/cafe.h>
  #include <winext/cafe/os/OSInterrupt.h>
  //#include <nn/os.h>
#else
  #error
#endif

#if defined( NW_PLATFORM_WIN32 )
using namespace nw::internal::winext;
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
using namespace nw::internal::winext;
#elif defined( NW_USE_NINTENDO_SDK )
// TODO: nn_audio
using namespace nw::internal::winext;
#endif

// #define ENABLE_BUFFER_JUMP_LOG


// #define ENABLE_DEBUG_LOG
// #define ENABLE_DEBUG_LOG_DETAIL

#ifdef ENABLE_DEBUG_LOG

#define NW_DEBUG_LOG(...)   DebugLog(false, __VA_ARGS__)
#define NW_DEBUG_LOG_BUFFER(...)   DebugLog(true, __VA_ARGS__)

namespace {

void DebugLog(bool /* isBuffering */, const char* /* format */, ...)
{
    NW_FATAL_ERROR("not implemented.");
}

} // anonymous namespace

#else
#   define NW_DEBUG_LOG(...)   ((void)0)
#   define NW_DEBUG_LOG_BUFFER(...)   ((void)0)
#endif // ENABLE_DEBUG_LOG

namespace {

u16 GetHiFromU32( u32 x )
{
    return static_cast<u16>(x >> 16);
}

u16 GetLoFromU32( u32 x )
{
    return static_cast<u16>(x & 0xffff);
}

u16 GetVolumeU16( f32 vol )
{
    const u16 BASE_VOLUME = 0x8000;
    const u16 VOLUME_MIN = 0;
    const u16 VOLUME_MAX = 0xffff;

    return static_cast<u16>( nw::ut::Clamp<f32>(vol * BASE_VOLUME, VOLUME_MIN, VOLUME_MAX) );
}

u32 GetAxRendererType(nw::snd::VoiceRendererType type)
{
    NW_ASSERT( (type == nw::snd::VOICE_RENDERER_SDK_DSP)
            || (type == nw::snd::VOICE_RENDERER_SDK_PPC) );

    u32 ret = AX_PB_RENDERER_SELECT_DSP_OR_PPC;

    switch ( type )
    {
    case nw::snd::VOICE_RENDERER_SDK_DSP:
        ret = AX_PB_RENDERER_SELECT_DSP;
        break;
    case nw::snd::VOICE_RENDERER_SDK_PPC:
        ret = AX_PB_RENDERER_SELECT_PPC;
        break;
    default:
        break;
    }

    return ret;
}

#ifndef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
#if defined(NW_PLATFORM_WIN32) || defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
class AXUserLevelProtection
{
public:
    AXUserLevelProtection() { oldState = OSDisableInterrupts(); }
    ~AXUserLevelProtection() { OSRestoreInterrupts(oldState); }
private:
    int oldState;
};
#elif defined( NW_USE_NINTENDO_SDK )
// TODO: nn_audio
class AXUserLevelProtection
{
public:
    AXUserLevelProtection() { oldState = OSDisableInterrupts(); }
    ~AXUserLevelProtection() { OSRestoreInterrupts(oldState); }
private:
    int oldState;
};
#elif defined( NW_PLATFORM_CAFE )
class AXUserLevelProtection
{
public:
    AXUserLevelProtection() { AXUserBegin(); }
    ~AXUserLevelProtection() { AXUserEnd(); }
};
#else
#error
#endif
#endif // NW_SND_CONFIG_ENABLE_VOICE_COMMAND

} // anonymous namespace


namespace nw {
namespace snd {
namespace internal {

//==================================================================
//
// AxVoice
//
//==================================================================

namespace
{

typedef ut::LinkList< AxVoice, offsetof(AxVoice,m_VoiceListNode)> AxVoiceList;
AxVoiceList s_VoiceList;
AxVoice s_AxVoiceArray[AX_MAX_VOICES];

}

ut::CriticalSection AxVoice::s_CriticalSection;
s32                 AxVoice::s_SamplesPerFrame;
s32                 AxVoice::s_SamplesPerSec;

const AXPBADPCM AxVoice::AXPBADPCM_PCM8 =
{
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, // a[8][2]
    0x0100, // gain
    0,0,0   // pred_scale,yn1,yn2
};

const AXPBADPCM AxVoice::AXPBADPCM_PCM16 =
{
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, // a[8][2]
    0x0800, // gain
    0,0,0   // pred_scale,yn1,yn2
};

AxVoice::AxVoice()
    : m_pVpb(NULL)
    , m_pVoice(NULL)
    , m_DisposeCallbackFunc(NULL)
    , m_DisposeCallbackArg(NULL)
    , m_IsAppendedToList( false )
{
}

void AxVoice::Initialize( AXVPB* axvpb )
{
    NW_ASSERT( !m_IsAppendedToList );
    NW_ASSERT( m_pVpb == NULL );

    m_State = VOICE_STATE_STOP;
    m_pVpb = axvpb;

    m_WaveBufferListBegin = NULL;
    m_WaveBufferListEnd = NULL;

    m_SetupInitialNextWaveBufferFlag = false;
    m_SetupNextWaveBufferFlag = false;
    m_LoopCount = 0;
    m_PlayPosition = 0;
    m_PauseFlag = false;

    m_SampleRate = 32000;
    m_SampleFormat = SAMPLE_FORMAT_PCM_S16;

    m_Renderer = VOICE_RENDERER_SDK;
}

void AxVoice::Finalize()
{
//    NW_LOG("AxVoice::Finalize %08x\n",this);

    if ( m_IsAppendedToList )
    {
        EraseVoiceList( this );
    }

    FreeAllWaveBuffer();

    if ( m_DisposeCallbackFunc != NULL )
    {
        m_DisposeCallbackFunc( this, m_DisposeCallbackArg );
        m_DisposeCallbackFunc = NULL;
    }
    m_pVoice = NULL;

    m_pVpb = NULL;
}

AxVoice* AxVoice::AllocVoice( u32 priority, DisposeCallbackFunc func, void* arg )
{
    // [NOTE]
    // 同一プライオリティの時、後着優先
    // AX内部のプライオリティリストの順番を適切な状態にするため
    // 一度、＋１のプライオリティ値で確保した後、元のプライオリティ値に戻している
    // (ストリームサウンド用の Voice::PRIORITY_NODROP の場合は別)

    u32 axPriority;
    if (priority == Voice::PRIORITY_NODROP)
    {
        axPriority = VOICE_PRIORITY_NODROP;
    }
    else
    {
        axPriority = VOICE_PRIORITY_ALLOC;
    }

    NW_DEBUG_LOG("+ Voice::AllocVoice() : AXAcquireVoice : VOICE_PRIORITY_ALLOC\n");
#if defined( NW_PLATFORM_CAFE )
    AXVPB* vpb = AXAcquireVoiceEx(axPriority, VoiceCallback, 0);
    NW_SND_LOG_APICALL_SDK_LOW("AXAquireVoiceEx [%08x][prio:%d] AxVoice::AllocVoice", vpb, axPriority);
#elif defined( NW_PLATFORM_WIN32 )
    AXVPB* vpb = AXAcquireVoice(axPriority, VoiceCallback, 0);
    NW_SND_LOG_APICALL_SDK_LOW("AXAquireVoice [%08x][prio:%d] AxVoice::AllocVoice", vpb, axPriority);
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
    AXVPB* vpb = AXAcquireVoice(axPriority, VoiceCallback, 0);
    NW_SND_LOG_APICALL_SDK_LOW("AXAquireVoice [%08x][prio:%d] AxVoice::AllocVoice", vpb, axPriority);
#elif defined( NW_USE_NINTENDO_SDK )
    // TODO: nn_audio
    AXVPB* vpb = AXAcquireVoice(axPriority, VoiceCallback, 0);
    NW_SND_LOG_APICALL_SDK_LOW("AXAquireVoice [%08x][prio:%d] AxVoice::AllocVoice", vpb, axPriority);
#else
    #error
#endif
    NW_DEBUG_LOG("- Voice::AllocVoice() : AXAcquireVoice\n");
    NW_SND_LOG_APICALL_SDK("AXAquireVoice [%08x][prio:%d]", vpb, axPriority);

    if ( vpb == NULL ) {
        return NULL;
    }

    vpb->callback = NULL;  // nw3_snd:2313

    NW_DEBUG_LOG("+ Voice::AllocVoice() : AXSetVoicePriority : vpb=0x%p, VOICE_PRIORITY_USE\n", vpb);
    if (priority != Voice::PRIORITY_NODROP)
    {
        AXSetVoicePriority( vpb, VOICE_PRIORITY_USE );
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoicePriority [%08x][prio:%d] AxVoice::AllocVoice", vpb, VOICE_PRIORITY_USE);
    }
    NW_DEBUG_LOG("- Voice::AllocVoice() : AXSetVoicePriority\n");

    NW_ASSERT_MAXLT( vpb->index, VOICE_COUNT_MAX );
    AxVoice* pAxVoice = &s_AxVoiceArray[vpb->index];
    pAxVoice->Finalize();

    pAxVoice->Initialize( vpb );
    pAxVoice->m_DisposeCallbackFunc = func;
    pAxVoice->m_DisposeCallbackArg = arg;
    pAxVoice->m_Priority = priority;

    AppendVoiceList( pAxVoice );

    return pAxVoice;
}

void AxVoice::Free()
{
//    NW_LOG("AxVoice::Free %08x\n", this);

    if ( m_pVpb != NULL ) {
        AXVPB* tmp = m_pVpb;
        NW_DEBUG_LOG("+ Voice::Free() : AXFreeVoice\n");
        NW_SND_LOG_APICALL_SDK( "AXFreeVoice [%08x]", m_pVpb);
        m_pVpb = NULL;
#if defined( NW_PLATFORM_CAFE )
        OSCoherencyBarrier();
#endif // defined( NW_PLATFORM_CAFE )
        AXFreeVoice( tmp );
        NW_SND_LOG_APICALL_SDK_LOW( "AXFreeVoice [%08x] AxVoice::Free", m_pVpb);
        NW_DEBUG_LOG("- Voice::Free() : AXFreeVoice\n");
    }

    Finalize();
}

void AxVoice::AppendWaveBuffer( WaveBuffer* waveBuffer )
{
//    NW_LOG("AxVoice::AppendWaveBuffer %08x %08x\n", this,waveBuffer);

    ut::ScopedLock<ut::CriticalSection> lock(s_CriticalSection);

#if 0 // StreamSoundPlayer 側の対応してから if 0 を有効にする
    if ( m_pVoiceParam->m_State == VOICE_STATE_PLAY || m_WaveBufferListBegin != NULL )
    {
        // 停止中かつ、一番目のバッファのみエラーチェックは不要
        const f32 PITCH_MAX = 8.0f;

        s32 minimumLength = static_cast<s32>(
            m_SampleRate * PITCH_MAX * (AX_MS_PER_FRAME / 1000.0f));
        if ( minimumLength > waveBuffer->sampleLength )
        {
            NW_ASSERTMSG( false,
                          "need to satisfy [minimumLength < waveBuffer->sampleLength],"
                          "but minimumLength(%d) / sampleLength(%d)\n",
                          minimumLength, waveBuffer->sampleLength );
            return;
        }
    }
#endif

    waveBuffer->next = NULL;
    waveBuffer->status = WaveBuffer::STATUS_WAIT;

    if ( m_WaveBufferListEnd == NULL )
    {
        m_WaveBufferListBegin = waveBuffer;
    }
    else
    {
        m_WaveBufferListEnd->next = waveBuffer;
    }
    m_WaveBufferListEnd = waveBuffer;

#ifdef ENABLE_DEBUG_LOG_DETAIL
    {
        NW_DEBUG_LOG("* AppendWaveBuffer\n");

        WaveBuffer* wb = m_WaveBufferListBegin;
        while(wb != NULL){
            NW_DEBUG_LOG("*  %p %d\n",wb->bufferAddress,wb->sampleLength);
            wb = wb->next;
        }
    }
#endif
}

void AxVoice::FreeAllWaveBuffer()
{
//    NW_LOG("AxVoice::FreeAllWaveBuffer %08x\n", this);

    ut::ScopedLock<ut::CriticalSection> lock(s_CriticalSection);

    WaveBuffer* waveBuffer = m_WaveBufferListBegin;
    while( waveBuffer != NULL ) {
        waveBuffer->status = WaveBuffer::STATUS_DONE;
        waveBuffer = waveBuffer->next;
    }
    m_WaveBufferListBegin = NULL;
    m_WaveBufferListEnd = NULL;
}

void AxVoice::AppendVoiceList( AxVoice* voice )
{
    if ( voice->m_IsAppendedToList )
    {
        // TODO この判定が必要か要検証
        return;
    }

    AxVoiceList::ReverseIterator itr = s_VoiceList.GetBeginReverseIter();
    while ( itr != s_VoiceList.GetEndReverseIter() )
    {
        if ( itr->m_Priority <= voice->m_Priority ) break;
        (void)++itr;
    }
    s_VoiceList.Insert( itr.GetBase(), voice );

    voice->m_IsAppendedToList = true;
}

void AxVoice::EraseVoiceList( AxVoice* voice )
{
    if ( ! voice->m_IsAppendedToList )
    {
        // TODO この判定が必要か要検証
        return;
    }

    s_VoiceList.Erase( voice );

    voice->m_IsAppendedToList = false;
}

void AxVoice::VoiceCallback(void* voiceIn
#if defined(NW_PLATFORM_CAFE)
        , u32 /*context*/, u32 /*reason*/
#endif
        )
{
    NW_DEBUG_LOG("+ Voice::VoiceCallback()\n");

    AXVPB* vpb = reinterpret_cast<AXVPB*>(voiceIn);
    AxVoice& voice = s_AxVoiceArray[vpb->index];
    NW_ASSERT( vpb == voice.m_pVpb );

    voice.m_pVpb = NULL;

    NW_DEBUG_LOG("- Voice::VoiceCallback()\n");
}

void AxVoice::SetPriority(u32 priority)
{
    m_Priority = priority;

    if ( m_IsAppendedToList )
    {
        EraseVoiceList( this );
        AppendVoiceList( this );
    }
}

void AxVoice::detail_UpdateAllVoices()
{
    ut::ScopedLock<ut::CriticalSection> lock(s_CriticalSection);

    NW_DEBUG_LOG_BUFFER("**** + Voice::detail_UpdateAllVoices() ************************************\n");

    if (AXUserIsProtected())
    {
        NW_DEBUG_LOG_BUFFER("**** ! Voice::detail_UpdateAllVoices() : AXUserIsProtected()\n");
        NW_DEBUG_LOG_BUFFER("**** - Voice::detail_UpdateAllVoices() ************************************\n");
        return;
    }

    OutputMode mode[OUTPUT_DEVICE_COUNT];
    for ( int i = 0; i < OUTPUT_DEVICE_COUNT; i++ )
    {
        mode[i] = driver::HardwareManager::GetInstance().GetOutputMode(static_cast<OutputDevice>(i));
    }
    for ( int ch = 0; ch < VOICE_COUNT_MAX; ch++ )
    {
        AxVoice* voice = &s_AxVoiceArray[ch];
        if ( voice->m_pVpb != NULL )
        {
            voice->UpdateState( mode );
        }
    }

    NW_DEBUG_LOG_BUFFER("**** - Voice::detail_UpdateAllVoices() ************************************\n");
}

void AxVoice::detail_UpdatePriorityList()
{
#ifndef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    // TODO: nn_audio 対応時に排他制御が必要であるかを判断
    AXUserLevelProtection protection;
#endif

    AxVoiceList::Iterator itr = s_VoiceList.GetBeginIter();
    while ( itr != s_VoiceList.GetEndIter() )
    {
        // DSP でのボイスドロップ (VoiceCallback) が起こると、
        // 「s_VoiceList から削除されていないが、m_pVpb が NULL である」状態が生じるため、
        // 【「s_VoiceList につながっている」かつ「m_pVpb != NULL」】ときに、
        // s_VoiceList につながっている順でプライオリティを振り直す。
        AXVPB* tmp = itr->m_pVpb;
        if ( tmp )
        {
            NW_DEBUG_LOG_BUFFER("+ Voice::detail_UpdateAllVoices() : AXSetVoicePriority : VOICE_PRIORITY_USE\n");
            AXSetVoicePriority( tmp, VOICE_PRIORITY_USE );
            NW_SND_LOG_APICALL_SDK_LOW("AXSetVoicePriority [%08x][prio:%d] AxVoice::detail_UpdatePriorityList", tmp, VOICE_PRIORITY_USE);
            NW_DEBUG_LOG_BUFFER("- Voice::detail_UpdateAllVoices() : AXSetVoicePriority\n");
        }

        (void)++itr;
    }
}

const AxVoice* AxVoice::detail_GetVoiceById( int id )
{
    NW_MINMAXLT_ASSERT(id,0,VOICE_COUNT_MAX);
    return &s_AxVoiceArray[id];
}

u32 AxVoice::CalcVoiceOffsetFromSampleCount( u32 sampleCount, SampleFormat format )
{
    if ( format == SAMPLE_FORMAT_DSP_ADPCM )
    {
        u32 block = (sampleCount / 14);
        u32 rest = (sampleCount % 14);
        return block * 16 + rest + 2; // [nibble]
    }
    else
    {
        return sampleCount;
    }
}

u32 AxVoice::AddVoiceOffsetSampleCount( u32 voiceOffset, u32 sampleCount, SampleFormat format )
{
    if ( format == SAMPLE_FORMAT_DSP_ADPCM )
    {
        u32 block = (sampleCount / 14);
        u32 rest = (sampleCount % 14);
        voiceOffset += block * 16 + rest;
    }
    else
    {
        voiceOffset += sampleCount;
    }
    return voiceOffset;
}

u32 AxVoice::CalcVoiceOffsetFromByte( u32 byte, SampleFormat format )
{
    switch( format ) {
    case SAMPLE_FORMAT_DSP_ADPCM:
        NW_ASSERT( (byte % 8) == 0 );
        return ( byte << 1 ) + 2;
    case SAMPLE_FORMAT_PCM_S16:
        return ( byte >> 1 );
    case SAMPLE_FORMAT_PCM_S8:
        return byte;
    default:
        NW_ASSERT(false);
        return 0;
    }
}

void AxVoice::SetupVoice( const VoiceParam& voiceParam, const WaveBuffer* waveBuffer, const OutputMode* mode )
{
    // レンダラ設定
    if ( m_Renderer != VOICE_RENDERER_SDK )
    {
        AXSetVoiceRenderer(m_pVpb, GetAxRendererType(m_Renderer));
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceRenderer [%08x][renderer:%d] AxVoice::SetupVoice", m_pVpb, GetAxRendererType(m_Renderer));
    }

    // サンプル位置指定
    SetupOffset( waveBuffer, voiceParam );

    // ADPCM パラメータ
    SetupAdpcm( waveBuffer, voiceParam );

    // TV, DRC, RMT のミックス設定
    SetupTvMix( voiceParam, mode[OUTPUT_DEVICE_MAIN] );
    SetupDrcMix( voiceParam, mode[OUTPUT_DEVICE_DRC] );
    SetupRmtMix( voiceParam );

    // VE
    UpdateVe( voiceParam, true );

    // SRC
    UpdateSrc( voiceParam, true );

    // LPF, biquad フィルタ
    UpdateMonoFilter( voiceParam, true );
    UpdateBiquadFilter( voiceParam, true );

    // リモコンフィルタ
    UpdateRemoteFilter( voiceParam, true );
}

void AxVoice::SetupOffset( const WaveBuffer* waveBuffer, const VoiceParam& /* m_VoiceParam */ )
{
    AXPBOFFSET ofs;
    ofs.samples = waveBuffer->bufferAddress;

    switch( m_SampleFormat ) {
    case SAMPLE_FORMAT_DSP_ADPCM:
        ofs.format = AX_PB_FORMAT_ADPCM;
        break;
    case SAMPLE_FORMAT_PCM_S16:
        ofs.format = AX_PB_FORMAT_PCM16;
        break;
    case SAMPLE_FORMAT_PCM_S8:
        ofs.format = AX_PB_FORMAT_PCM8;
        break;
    default:
        NW_ASSERT(false);
        return;
    }

    ofs.currentOffset = CalcVoiceOffsetFromSampleCount(waveBuffer->sampleOffset, m_SampleFormat);
    ofs.endOffset = CalcVoiceOffsetFromSampleCount( waveBuffer->sampleLength-1, m_SampleFormat );
    m_CurrentOffset = ofs.currentOffset;

    ofs.loopOffset = CalcVoiceOffsetFromSampleCount( 0, m_SampleFormat );
    if ( waveBuffer->next != NULL )
    {
        ofs.loopFlag = AXPBADDR_LOOP_ON;
        SetupNextWaveBuffer( waveBuffer->next );
        m_SetupNextWaveBufferFlag = true;
        m_SetupInitialNextWaveBufferFlag = true;
    }
    else
    {
        ofs.loopFlag = waveBuffer->loopFlag ? AXPBADDR_LOOP_ON: AXPBADDR_LOOP_OFF;
        m_SetupNextWaveBufferFlag = false;
        m_SetupInitialNextWaveBufferFlag = false;
    }

    NW_DEBUG_LOG_BUFFER("+ Voice::SetupVoice() : AXSetVoiceOffsets : cur=%d, end=%d, loop=%d, flag=%d\n", ofs.currentOffset, ofs.endOffset, ofs.loopOffset, ofs.loopFlag);
    // NW_LOG("+ Voice::SetupVoice() : AXSetVoiceOffsets : cur=%d, end=%d, loop=%d, flag=%d\n", ofs.currentOffset, ofs.endOffset, ofs.loopOffset, ofs.loopFlag);
    AXSetVoiceOffsets(m_pVpb, &ofs);
    NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceOffsets [%08x][fmt:%u][loopFlag:%u][loopOfs:%u][endOfs:%u][curOfs:%u][sampleAddr:%x] AxVoice::SetupOffset"
        , m_pVpb
        , ofs.format
        , ofs.loopFlag
        , ofs.loopOffset
        , ofs.endOffset
        , ofs.currentOffset
        , ofs.samples );
    NW_DEBUG_LOG_BUFFER("- Voice::SetupVoice() : AXSetVoiceOffsets\n");

#ifdef ENABLE_BUFFER_JUMP_LOG
    NW_LOG("[Voice(%p)::%s] AXSetVoiceOffsets fmt(%d) loop?(%d)\n               loop(%6d) end(%6d) cur(%6d) samples(%08x)\n",
            this, __FUNCTION__,
            ofs.format, ofs.loopFlag,
            ofs.loopOffset, ofs.endOffset,
            ofs.currentOffset, ofs.samples );
#endif
}

void AxVoice::SetupNextWaveBuffer( const WaveBuffer* nextWaveBuffer )
{
    // ADPCM -> コンテキスト (yn1, yn2, pred_scale) の設定
    if ( nextWaveBuffer->pAdpcmContext != NULL )
    {
        const AdpcmContext* adpcmContext = nextWaveBuffer->pAdpcmContext;
        AXPBADPCMLOOP adpcmLoop;
        adpcmLoop.loop_pred_scale = adpcmContext->pred_scale;
        adpcmLoop.loop_yn1 = adpcmContext->yn1;
        adpcmLoop.loop_yn2 = adpcmContext->yn2;

        NW_DEBUG_LOG_BUFFER("+ Voice::SetupNextWaveBuffer() : AXSetVoiceType : AX_PB_TYPE_NORMAL\n");
        AXSetVoiceType( m_pVpb, AX_PB_TYPE_NORMAL );
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceType [%08x][type:%d] AxVoice::SetupNextWaveBuffer", m_pVpb, AX_PB_TYPE_NORMAL);
        NW_DEBUG_LOG_BUFFER("- Voice::SetupNextWaveBuffer() : AXSetVoiceType\n");

        NW_DEBUG_LOG_BUFFER("+ Voice::SetupNextWaveBuffer() : AXSetVoiceAdpcmLoop\n");
        AXSetVoiceAdpcmLoop(m_pVpb, &adpcmLoop);
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceAdpcmLoop [%08x][predScale:%x][yn1:%x][yn2:%x] AxVoice::SetupNextWaveBuffer"
          , m_pVpb
          , adpcmLoop.loop_pred_scale
          , adpcmLoop.loop_yn1
          , adpcmLoop.loop_yn2);
        NW_DEBUG_LOG_BUFFER("- Voice::SetupNextWaveBuffer() : AXSetVoiceAdpcmLoop\n");

        // NW_LOG("AxVoice   [%p] NORMAL ps(%04x) yn1(%6d) yn2(%6d) [yn1(%6d) yn2(%6d)]\n",
        //         this, adpcmLoop.loop_pred_scale,adpcmLoop.loop_yn1, adpcmLoop.loop_yn2,
        //         adpcmContext->yn1, adpcmContext->yn2 );
    }
    else
    {
        AXPBADPCMLOOP adpcmLoop;
        adpcmLoop.loop_pred_scale = * reinterpret_cast<const u8*>(nextWaveBuffer->bufferAddress);

        NW_DEBUG_LOG_BUFFER("+ Voice::SetupNextWaveBuffer() : AXSetVoiceType : AX_PB_TYPE_STREAM\n");
        AXSetVoiceType( m_pVpb, AX_PB_TYPE_STREAM );
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceType [%08x][type:%d] AxVoice::SetupNextWaveBuffer", m_pVpb, AX_PB_TYPE_STREAM);
        NW_DEBUG_LOG_BUFFER("- Voice::SetupNextWaveBuffer() : AXSetVoiceType\n");

        NW_DEBUG_LOG_BUFFER("+ Voice::SetupNextWaveBuffer() : AXSetVoiceAdpcmLoop\n");
        AXSetVoiceAdpcmLoop(m_pVpb, &adpcmLoop);
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceAdpcmLoop [%08x][predScale:%x][yn1:%x][yn2:%x] AxVoice::SetupNextWaveBuffer"
          , m_pVpb
          , adpcmLoop.loop_pred_scale
          , adpcmLoop.loop_yn1
          , adpcmLoop.loop_yn2);
        NW_DEBUG_LOG_BUFFER("- Voice::SetupNextWaveBuffer() : AXSetVoiceAdpcmLoop\n");

        // NW_LOG("AxVoice   [%p] STREAM ps(%04x) yn1(%6d) yn2(%6d)\n",
        //         this, adpcmLoop.loop_pred_scale, adpcmLoop.loop_yn1, adpcmLoop.loop_yn2 );
    }
}

void AxVoice::SetupAdpcm( const WaveBuffer* waveBuffer, const VoiceParam& /* VoiceParam& */ )
{
    AXPBADPCM adpcm;
    switch( m_SampleFormat )
    {
    case SAMPLE_FORMAT_DSP_ADPCM:
        {
            NW_STATIC_ASSERT( sizeof(m_AdpcmParam.coef) == sizeof(u32)*8 );

            const u32* src = reinterpret_cast<const u32*>(m_AdpcmParam.coef);
            u32* dst = reinterpret_cast<u32*>(adpcm.a);
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
        }
        adpcm.gain = 0x0000;
        if ( waveBuffer->pAdpcmContext != NULL ) {
            const AdpcmContext* ac = waveBuffer->pAdpcmContext;
            adpcm.pred_scale = ac->pred_scale;
            adpcm.yn1 = ac->yn1;
            adpcm.yn2 = ac->yn2;
        }
        else {
            adpcm.pred_scale = 0;
            adpcm.yn1 = 0;
            adpcm.yn2 = 0;
        }
        break;
    case SAMPLE_FORMAT_PCM_S16:
        adpcm = AXPBADPCM_PCM16;
        break;
    case SAMPLE_FORMAT_PCM_S8:
        adpcm = AXPBADPCM_PCM8;
        break;
    default:
        NW_ASSERT(false);
        return;
    }

    NW_DEBUG_LOG_BUFFER("+ Voice::SetupVoice() : AXSetVoiceAdpcm\n");
    AXSetVoiceAdpcm( m_pVpb, &adpcm );
    NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceAdpcm [%08x][predScale:%x][yn1:%x][yn2:%x] AxVoice::SetupAdpcm"
      , m_pVpb
      , adpcm.pred_scale
      , adpcm.yn1
      , adpcm.yn2);
    NW_DEBUG_LOG_BUFFER("- Voice::SetupVoice() : AXSetVoiceAdpcm\n");
}

void AxVoice::UpdateState( const OutputMode* mode )
{
    bool isRun = ( m_pVpb->state == AX_PB_STATE_RUN );

    switch( m_State ) {
    case VOICE_STATE_PLAY:
        UpdateStatePlay( m_VoiceParam, isRun, mode );
        break;
    case VOICE_STATE_STOP:
        UpdateStateStop( m_VoiceParam, isRun );
        break;
    case VOICE_STATE_PAUSE:
        UpdateStatePause( m_VoiceParam, isRun );
        break;
    default:
        NW_ASSERT(false);
        break;
    }

    UpdatePlayPosition();
}

void AxVoice::UpdatePlayPosition()
{
    // 現在再生位置の更新
    if ( m_WaveBufferListBegin != NULL && m_WaveBufferListBegin->status == WaveBuffer::STATUS_PLAY )
    {
        NW_ASSERT_NOT_NULL( m_pVpb );
        NW_ASSERT_NOT_NULL( m_WaveBufferListBegin->bufferAddress );

        u32 currentOffset = AXGetVoiceCurrentOffsetEx( m_pVpb, m_WaveBufferListBegin->bufferAddress );

        // endOffset 直前でポーズした場合に endOffset を超える場合があるため補正する
        AXPBOFFSET ofs;
        AXGetVoiceOffsets( m_pVpb, &ofs );
        if ( currentOffset > ofs.endOffset )
        {
            currentOffset = ofs.endOffset;

            // ADPCM でかつフレームヘッダを指す場合は直前のフレームを指すようにする
            if ( ( m_SampleFormat == SAMPLE_FORMAT_DSP_ADPCM ) && ( currentOffset % 16 <= 1 ) )
            {
                currentOffset -= ( currentOffset % 16 ) + 1;
            }
        }

        if ( m_SampleFormat == SAMPLE_FORMAT_DSP_ADPCM )
        {
            u32 nibble = currentOffset;
            u32 block = nibble / 16;    // 8 バイト 1 フレーム換算のフレーム数
            u32 mod = nibble % 16;
            if ( mod >= 2 )
            {
                mod -= 2;
            }
            m_PlayPosition = block * 14 + mod;
            // NW_LOG("m_PlayPosition(%d)\n", m_PlayPosition);
        }
        else
        {
            m_PlayPosition = currentOffset;
        }
    }
}

void AxVoice::UpdateStatePlay( const VoiceParam& voiceParam, bool isRun, const OutputMode* mode )
{
    if ( ! isRun && ! m_PauseFlag )
    {
        if ( m_WaveBufferListBegin != NULL )
        {
            WaveBuffer* waveBuffer = m_WaveBufferListBegin;
            if ( waveBuffer->status == WaveBuffer::STATUS_PLAY )
            {
                NW_DEBUG_LOG_BUFFER("* Voice::UpdateState() : Play Finish\n");

                // 再生完了バッファの後始末
                m_WaveBufferListBegin = waveBuffer->next;
                if ( m_WaveBufferListEnd == waveBuffer ) {
                    m_WaveBufferListEnd = NULL;
                }
                waveBuffer->status = WaveBuffer::STATUS_DONE;

                // 次のバッファがあれば、処理継続
                waveBuffer = m_WaveBufferListBegin;
                if ( waveBuffer == NULL )
                {
                    return;
                }
            }
            if ( waveBuffer->status == WaveBuffer::STATUS_WAIT )
            {
                NW_DEBUG_LOG_BUFFER("* Voice::UpdateState() : SetupVoice\n");

                // 待ち状態のバッファ再生開始
                SetupVoice( voiceParam, waveBuffer, mode );

                NW_DEBUG_LOG_BUFFER("+ Voice::UpdateState() : AXSetVoiceState AX_PB_STATE_RUN\n");
                AXSetVoiceState( m_pVpb, AX_PB_STATE_RUN );
                NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceState [%08x][state:%d] AxVoice::UpdateStatePlay", m_pVpb, AX_PB_STATE_RUN);
                NW_DEBUG_LOG_BUFFER("- Voice::UpdateState() : AXSetVoiceState\n");

                waveBuffer->status = WaveBuffer::STATUS_PLAY;
            }
        }
    }
    else
    {
        if ( m_PauseFlag )
        {
            NW_DEBUG_LOG_BUFFER("+ Voice::UpdateState() : AXSetVoiceState AX_PB_STATE_RUN\n");
            {
                // ADPCM でオフセットがフレームヘッダを指す場合 DEBUG 版でアサートになるため
                // 直前のフレームを指すようにする
                if ( ( m_SampleFormat == SAMPLE_FORMAT_DSP_ADPCM ) && ( m_CurrentOffset % 16 <= 1 ) )
                {
                    m_CurrentOffset -= ( m_CurrentOffset % 16 ) + 1;
                }
                AXSetVoiceCurrentOffset( m_pVpb, m_CurrentOffset );
                NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceCurrentOffset [%08x][curOfs:%u] AxVoice::UpdateStatePlay", m_pVpb, m_CurrentOffset);
            }
            AXSetVoiceState( m_pVpb, AX_PB_STATE_RUN );
            NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceState [%08x][state:%d] AxVoice::UpdateStatePlay", m_pVpb, AX_PB_STATE_RUN);
            NW_DEBUG_LOG_BUFFER("- Voice::UpdateState() : AXSetVoiceState\n");

            m_PauseFlag = false;
        }

        NW_NULL_ASSERT( m_WaveBufferListBegin );
        WaveBuffer* waveBuffer = m_WaveBufferListBegin;

#ifdef ENABLE_BUFFER_JUMP_LOG
        {
            AXPBOFFSET ofs;
            AXGetVoiceOffsetsEx( m_pVpb, &ofs, waveBuffer->bufferAddress );

            // 現在再生中のバッファを起点とした ofs
            NW_LOG("ofs(%p): loop(%6d) end(%6d) cur(%6d) samples(%p)\n",
                   this, ofs.loopOffset, ofs.endOffset, ofs.currentOffset, ofs.samples );
        }
#endif

        if ( m_SetupNextWaveBufferFlag )
        {
            u32 loopCount = AXGetVoiceLoopCount( m_pVpb );

            // バッファジャンプ発生
            if ( loopCount != m_LoopCount || m_SetupInitialNextWaveBufferFlag )
            {
                m_SetupNextWaveBufferFlag = false;

#ifdef ENABLE_BUFFER_JUMP_LOG
                NW_LOG("[Voice(%p)::%s] loopCount(%d) m_LoopCount(%d)\n",
                       this, __FUNCTION__, loopCount, m_LoopCount );
#endif

                // WaveBuffer の更新
                if ( m_SetupInitialNextWaveBufferFlag == false )
                {
                    m_WaveBufferListBegin = waveBuffer->next;
                    NW_ASSERT_NOT_NULL( m_WaveBufferListBegin );
                    waveBuffer->status = WaveBuffer::STATUS_DONE;

                    waveBuffer = m_WaveBufferListBegin;
                    waveBuffer->status = WaveBuffer::STATUS_PLAY;
                }

                // EndOffset の更新 + 波形バッファ起点を変更
                if ( m_SetupInitialNextWaveBufferFlag == false )
                {
                    u32 endOffset =
                        CalcVoiceOffsetFromSampleCount( waveBuffer->sampleLength-1, m_SampleFormat );
                    AXSetVoiceEndOffsetEx( m_pVpb, endOffset, waveBuffer->bufferAddress );
                    NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceEndOffsetEx [%08x][ofs:%u][bufAddr:%08x] AxVoice::UpdateStatePlay", m_pVpb, endOffset, waveBuffer->bufferAddress);
#ifdef ENABLE_BUFFER_JUMP_LOG
                    NW_LOG("[Voice(%p)::%s] EndOffsetEx  ofs(%6d) samples(%p)\n",
                           this, __FUNCTION__, endOffset, waveBuffer->bufferAddress );
#endif
                }
                m_SetupInitialNextWaveBufferFlag = false;

                // 次のバッファ用のループオフセット (ジャンプ先) を設定
                WaveBuffer* nextBuffer = waveBuffer->next;
                if ( nextBuffer != NULL )
                {
                    u32 loopOffset = CalcVoiceOffsetFromSampleCount( nextBuffer->sampleOffset, m_SampleFormat );
                    AXSetVoiceLoopOffsetEx( m_pVpb, loopOffset, nextBuffer->bufferAddress );
                    NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceLoopOffsetEx [%08x][ofs:%u][bufAddr:%08x] AxVoice::UpdateStatePlay", m_pVpb, loopOffset, waveBuffer->bufferAddress);
                    AXSetVoiceLoop( m_pVpb, AXPBADDR_LOOP_ON );
                    NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceLoop [%08x][loop:%d] AxVoice::UpdateStatePlay", m_pVpb, AXPBADDR_LOOP_ON);
#ifdef ENABLE_BUFFER_JUMP_LOG
                    NW_LOG("[Voice(%p)::%s] LoopOffsetEx ofs(%6d) samples(%p) <next>\n",
                           this, __FUNCTION__, loopOffset, nextBuffer->bufferAddress );
#endif

                    SetupNextWaveBuffer( nextBuffer );
                    m_SetupNextWaveBufferFlag = true;
                }
                // 今のバッファ内でループさせる
                else
                {
                    u32 loopOffset = CalcVoiceOffsetFromSampleCount( 0, m_SampleFormat );
                    AXSetVoiceLoopOffsetEx( m_pVpb, loopOffset, waveBuffer->bufferAddress );
                    NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceLoopOffsetEx [%08x][ofs:%u][bufAddr:%08x] AxVoice::UpdateStatePlay", m_pVpb, loopOffset, waveBuffer->bufferAddress);
                    AXSetVoiceLoop( m_pVpb,
                                    waveBuffer->loopFlag ? AXPBADDR_LOOP_ON : AXPBADDR_LOOP_OFF );
                    NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceLoop [%08x][loop:%d] AxVoice::UpdateStatePlay", m_pVpb, waveBuffer->loopFlag ? AXPBADDR_LOOP_ON : AXPBADDR_LOOP_OFF);
#ifdef ENABLE_BUFFER_JUMP_LOG
                    NW_LOG("[Voice(%p)::%s] LoopOffsetEx ofs(%6d) samples(%p) loop?(%d)<same>\n",
                           this, __FUNCTION__, loopOffset, waveBuffer->bufferAddress,
                           waveBuffer->loopFlag );
#endif
                }

                // ループカウンタの更新
                m_LoopCount = loopCount;
            }
        }
        else
        {
            WaveBuffer* nextBuffer = waveBuffer->next;
            if ( nextBuffer != NULL )
            {
                AXSetVoiceLoopOffsetEx( m_pVpb, nextBuffer->sampleOffset, nextBuffer->bufferAddress );
                NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceLoopOffsetEx [%08x][ofs:%u][bufAddr:%08x] AxVoice::UpdateStatePlay", m_pVpb, nextBuffer->sampleOffset, nextBuffer->bufferAddress);
                AXSetVoiceLoop( m_pVpb, AXPBADDR_LOOP_ON );
                NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceLoop [%08x][loop:%d] AxVoice::UpdateStatePlay", m_pVpb, AXPBADDR_LOOP_ON);
                SetupNextWaveBuffer( nextBuffer );
                m_SetupNextWaveBufferFlag = true;
            }
        }

        //
        // パラメータ更新
        //

        // VE
        UpdateVe( voiceParam, false);

        // ピッチ
        UpdateSrc(voiceParam, false);

        // ミックス
        UpdateTvMix(voiceParam, mode[OUTPUT_DEVICE_MAIN]);
        UpdateDrcMix(voiceParam, mode[OUTPUT_DEVICE_DRC]);
        UpdateRmtMix(voiceParam);

        // LPF, biquad フィルタ
        UpdateMonoFilter(voiceParam, false);
        UpdateBiquadFilter(voiceParam, false);

        // リモコンフィルタ
        UpdateRemoteFilter(voiceParam, false);
    }
}

void AxVoice::UpdateStateStop( const VoiceParam& /* voiceParam */, bool isRun )
{
    if ( isRun )
    {
        NW_DEBUG_LOG_BUFFER("+ Voice::UpdateState() : AXSetVoiceState : AX_PB_STATE_STOP\n");
        AXSetVoiceState( m_pVpb, AX_PB_STATE_STOP );
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceState [%08x][state:%d] AxVoice::UpdateStateStop", m_pVpb, AX_PB_STATE_STOP);
        NW_DEBUG_LOG_BUFFER("- Voice::UpdateState() : AXSetVoiceState\n");

        m_PauseFlag = false;
    }
}

void AxVoice::UpdateStatePause( const VoiceParam& /* voiceParam */, bool isRun )
{
    if ( isRun )
    {
        NW_DEBUG_LOG_BUFFER("+ Voice::UpdateState() : AXSetVoiceState : AX_PB_STATE_STOP\n");
        AXSetVoiceState( m_pVpb, AX_PB_STATE_STOP );
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceState [%08x][state:%d] AxVoice::UpdateStatePause", m_pVpb, AX_PB_STATE_STOP);
        {
            AXPBOFFSET ofs;
            AXGetVoiceOffsets( m_pVpb, &ofs );
            m_CurrentOffset = ofs.currentOffset;
        }
        NW_DEBUG_LOG_BUFFER("- Voice::UpdateState() : AXSetVoiceState\n");
        m_PauseFlag = true;
    }
}

bool AxVoice::UpdateDelta( s32 target, AXPBCHVE* ve )
{
    s32 current = ve->vol;
    s32 delta = ve->volDelta;

    if ( (delta == 0) && (target == current) )
    {
         return false;
    }

    current += delta * s_SamplesPerFrame;

    s32 diff = target - current;
    if ( diff > 0 ) {
        delta = diff / s_SamplesPerFrame;
    }
    else {
        delta = - ( (-diff) / s_SamplesPerFrame );
    }
    if ( delta != 0 ) {
        ve->vol = static_cast<u16>(current);
        ve->volDelta = static_cast<s16>(nw::ut::Clamp<int>(delta,-32768, 32767));
    }
    else {
        ve->vol = static_cast<u16>(target);
        ve->volDelta = 0;
    }

    return true;
}

void AxVoice::CalcSetupTvMix( AXPBCHMIX mix[], const OutputMix& src, bool surroundFlag)
{
    if (surroundFlag)
    {
        mix[AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_REAR_LEFT]);
        mix[AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].volDelta = 0;
        mix[AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_REAR_RIGHT]);
        mix[AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].volDelta = 0;
        mix[AX_CH_FC].bus[AX_MAIN_BUS].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_CENTER]);
        mix[AX_CH_FC].bus[AX_MAIN_BUS].volDelta = 0;
        mix[AX_CH_LFE].bus[AX_MAIN_BUS].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_LFE]);
        mix[AX_CH_LFE].bus[AX_MAIN_BUS].volDelta = 0;

        mix[AX_CH_SUR_LEFT].bus[AX_AUXA_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_A][CHANNEL_INDEX_REAR_LEFT]);
        mix[AX_CH_SUR_LEFT].bus[AX_AUXA_BUS].volDelta = 0;
        mix[AX_CH_SUR_RIGHT].bus[AX_AUXA_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_A][CHANNEL_INDEX_REAR_RIGHT]);
        mix[AX_CH_SUR_RIGHT].bus[AX_AUXA_BUS].volDelta = 0;
        mix[AX_CH_FC].bus[AX_AUXA_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_CENTER]);
        mix[AX_CH_FC].bus[AX_AUXA_BUS].volDelta = 0;
        mix[AX_CH_LFE].bus[AX_AUXA_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_A][CHANNEL_INDEX_LFE]);
        mix[AX_CH_LFE].bus[AX_AUXA_BUS].volDelta = 0;

        mix[AX_CH_SUR_LEFT].bus[AX_AUXB_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_B][CHANNEL_INDEX_REAR_LEFT]);
        mix[AX_CH_SUR_LEFT].bus[AX_AUXB_BUS].volDelta = 0;
        mix[AX_CH_SUR_RIGHT].bus[AX_AUXB_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_B][CHANNEL_INDEX_REAR_RIGHT]);
        mix[AX_CH_SUR_RIGHT].bus[AX_AUXB_BUS].volDelta = 0;
        mix[AX_CH_FC].bus[AX_AUXB_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_CENTER]);
        mix[AX_CH_FC].bus[AX_AUXB_BUS].volDelta = 0;
        mix[AX_CH_LFE].bus[AX_AUXB_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_B][CHANNEL_INDEX_LFE]);
        mix[AX_CH_LFE].bus[AX_AUXB_BUS].volDelta = 0;

        mix[AX_CH_SUR_LEFT].bus[AX_AUXC_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_C][CHANNEL_INDEX_REAR_LEFT]);
        mix[AX_CH_SUR_LEFT].bus[AX_AUXC_BUS].volDelta = 0;
        mix[AX_CH_SUR_RIGHT].bus[AX_AUXC_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_C][CHANNEL_INDEX_REAR_RIGHT]);
        mix[AX_CH_SUR_RIGHT].bus[AX_AUXC_BUS].volDelta = 0;
        mix[AX_CH_FC].bus[AX_AUXC_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_CENTER]);
        mix[AX_CH_FC].bus[AX_AUXC_BUS].volDelta = 0;
        mix[AX_CH_LFE].bus[AX_AUXC_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_C][CHANNEL_INDEX_LFE]);
        mix[AX_CH_LFE].bus[AX_AUXC_BUS].volDelta = 0;
    }
    else
    {
        const size_t MEMSET_SIZE = sizeof(AXPBCHMIX);
        std::memset(&mix[AX_CH_SUR_LEFT], 0, MEMSET_SIZE);
        std::memset(&mix[AX_CH_SUR_RIGHT], 0, MEMSET_SIZE);
        std::memset(&mix[AX_CH_FC], 0, MEMSET_SIZE);
        std::memset(&mix[AX_CH_LFE], 0, MEMSET_SIZE);
    }

    mix[AX_CH_LEFT].bus[AX_MAIN_BUS].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_LEFT]);
    mix[AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta = 0;
    mix[AX_CH_RIGHT].bus[AX_MAIN_BUS].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_RIGHT]);
    mix[AX_CH_RIGHT].bus[AX_MAIN_BUS].volDelta = 0;

    mix[AX_CH_LEFT].bus[AX_AUXA_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_LEFT]);
    mix[AX_CH_LEFT].bus[AX_AUXA_BUS].volDelta = 0;
    mix[AX_CH_RIGHT].bus[AX_AUXA_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_RIGHT]);
    mix[AX_CH_RIGHT].bus[AX_AUXA_BUS].volDelta = 0;

    mix[AX_CH_LEFT].bus[AX_AUXB_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_LEFT]);
    mix[AX_CH_LEFT].bus[AX_AUXB_BUS].volDelta = 0;
    mix[AX_CH_RIGHT].bus[AX_AUXB_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_RIGHT]);
    mix[AX_CH_RIGHT].bus[AX_AUXB_BUS].volDelta = 0;

    mix[AX_CH_LEFT].bus[AX_AUXC_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_LEFT]);
    mix[AX_CH_LEFT].bus[AX_AUXC_BUS].volDelta = 0;
    mix[AX_CH_RIGHT].bus[AX_AUXC_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_RIGHT]);
    mix[AX_CH_RIGHT].bus[AX_AUXC_BUS].volDelta = 0;
}


void AxVoice::CalcSetupDrcMix(AXPBCHMIX* mix0, AXPBCHMIX* mix1,
        const OutputMix& src, bool surroundFlag, bool frontBypass)
{
    // モノ・ステレオの時 → DRC0 の fL, fR のみにルーティング
    // サラウンド (FB OFF) → DRC0 の fL, fR, rL, rR にルーティング
    // サラウンド (FB ON)  → [メインセンド] DRC1 の fL, fR にルーティング
    //                                       DRC0 の rL, rR にルーティング
    //                        [AUX センド]   DRC0 の fL, fR, rL, rR にルーティング

    NW_ASSERT_NOT_NULL(mix0);
    {
        const size_t MEMSET_SIZE = sizeof(AXPBCHMIX);
        std::memset(&mix0[AX_CH_LEFT], 0, MEMSET_SIZE);
        std::memset(&mix0[AX_CH_RIGHT], 0, MEMSET_SIZE);
        std::memset(&mix0[AX_CH_SUR_LEFT], 0, MEMSET_SIZE);
        std::memset(&mix0[AX_CH_SUR_RIGHT], 0, MEMSET_SIZE);
        std::memset(&mix0[AX_CH_FC], 0, MEMSET_SIZE);
        std::memset(&mix0[AX_CH_LFE], 0, MEMSET_SIZE);
    }

    if (surroundFlag == false)
    {
        mix0[AX_CH_LEFT].bus[AX_MAIN_BUS].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_LEFT]);
        mix0[AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta = 0;
        mix0[AX_CH_RIGHT].bus[AX_MAIN_BUS].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_RIGHT]);
        mix0[AX_CH_RIGHT].bus[AX_MAIN_BUS].volDelta = 0;

        mix0[AX_CH_LEFT].bus[AX_AUXA_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_LEFT]);
        mix0[AX_CH_LEFT].bus[AX_AUXA_BUS].volDelta = 0;
        mix0[AX_CH_RIGHT].bus[AX_AUXA_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_RIGHT]);
        mix0[AX_CH_RIGHT].bus[AX_AUXA_BUS].volDelta = 0;

        mix0[AX_CH_LEFT].bus[AX_AUXB_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_LEFT]);
        mix0[AX_CH_LEFT].bus[AX_AUXB_BUS].volDelta = 0;
        mix0[AX_CH_RIGHT].bus[AX_AUXB_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_RIGHT]);
        mix0[AX_CH_RIGHT].bus[AX_AUXB_BUS].volDelta = 0;

        mix0[AX_CH_LEFT].bus[AX_AUXC_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_LEFT]);
        mix0[AX_CH_LEFT].bus[AX_AUXC_BUS].volDelta = 0;
        mix0[AX_CH_RIGHT].bus[AX_AUXC_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_RIGHT]);
        mix0[AX_CH_RIGHT].bus[AX_AUXC_BUS].volDelta = 0;
    }
    else
    {
        mix0[AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_REAR_LEFT]);
        mix0[AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].volDelta = 0;
        mix0[AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_REAR_RIGHT]);
        mix0[AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].volDelta = 0;

        mix0[AX_CH_SUR_LEFT].bus[AX_AUXA_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_A][CHANNEL_INDEX_REAR_LEFT]);
        mix0[AX_CH_SUR_LEFT].bus[AX_AUXA_BUS].volDelta = 0;
        mix0[AX_CH_SUR_RIGHT].bus[AX_AUXA_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_A][CHANNEL_INDEX_REAR_RIGHT]);
        mix0[AX_CH_SUR_RIGHT].bus[AX_AUXA_BUS].volDelta = 0;

        mix0[AX_CH_SUR_LEFT].bus[AX_AUXB_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_B][CHANNEL_INDEX_REAR_LEFT]);
        mix0[AX_CH_SUR_LEFT].bus[AX_AUXB_BUS].volDelta = 0;
        mix0[AX_CH_SUR_RIGHT].bus[AX_AUXB_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_B][CHANNEL_INDEX_REAR_RIGHT]);
        mix0[AX_CH_SUR_RIGHT].bus[AX_AUXB_BUS].volDelta = 0;

        mix0[AX_CH_SUR_LEFT].bus[AX_AUXC_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_C][CHANNEL_INDEX_REAR_LEFT]);
        mix0[AX_CH_SUR_LEFT].bus[AX_AUXC_BUS].volDelta = 0;
        mix0[AX_CH_SUR_RIGHT].bus[AX_AUXC_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_C][CHANNEL_INDEX_REAR_RIGHT]);
        mix0[AX_CH_SUR_RIGHT].bus[AX_AUXC_BUS].volDelta = 0;

        if (frontBypass == false
            || driver::HardwareManager::GetInstance().GetEndUserOutputMode(OUTPUT_DEVICE_DRC) != OUTPUT_MODE_SURROUND)
        {
            mix0[AX_CH_LEFT].bus[AX_MAIN_BUS].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_LEFT]);
            mix0[AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta = 0;
            mix0[AX_CH_RIGHT].bus[AX_MAIN_BUS].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_RIGHT]);
            mix0[AX_CH_RIGHT].bus[AX_MAIN_BUS].volDelta = 0;

            mix0[AX_CH_LEFT].bus[AX_AUXA_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_LEFT]);
            mix0[AX_CH_LEFT].bus[AX_AUXA_BUS].volDelta = 0;
            mix0[AX_CH_RIGHT].bus[AX_AUXA_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_RIGHT]);
            mix0[AX_CH_RIGHT].bus[AX_AUXA_BUS].volDelta = 0;

            mix0[AX_CH_LEFT].bus[AX_AUXB_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_LEFT]);
            mix0[AX_CH_LEFT].bus[AX_AUXB_BUS].volDelta = 0;
            mix0[AX_CH_RIGHT].bus[AX_AUXB_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_RIGHT]);
            mix0[AX_CH_RIGHT].bus[AX_AUXB_BUS].volDelta = 0;

            mix0[AX_CH_LEFT].bus[AX_AUXC_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_LEFT]);
            mix0[AX_CH_LEFT].bus[AX_AUXC_BUS].volDelta = 0;
            mix0[AX_CH_RIGHT].bus[AX_AUXC_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_RIGHT]);
            mix0[AX_CH_RIGHT].bus[AX_AUXC_BUS].volDelta = 0;
        }
        else
        {
            NW_ASSERT_NOT_NULL(mix1);
            {
                const size_t MEMSET_SIZE = sizeof(AXPBCHMIX);
                std::memset(&mix1[AX_CH_LEFT], 0, MEMSET_SIZE);
                std::memset(&mix1[AX_CH_RIGHT], 0, MEMSET_SIZE);
                std::memset(&mix1[AX_CH_SUR_LEFT], 0, MEMSET_SIZE);
                std::memset(&mix1[AX_CH_SUR_RIGHT], 0, MEMSET_SIZE);
            }

            mix1[AX_CH_LEFT].bus[AX_MAIN_BUS].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_LEFT]);
            mix1[AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta = 0;
            mix1[AX_CH_RIGHT].bus[AX_MAIN_BUS].vol = GetVolumeU16(src.mainBus[CHANNEL_INDEX_FRONT_RIGHT]);
            mix1[AX_CH_RIGHT].bus[AX_MAIN_BUS].volDelta = 0;

            mix0[AX_CH_LEFT].bus[AX_AUXA_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_LEFT]);
            mix0[AX_CH_LEFT].bus[AX_AUXA_BUS].volDelta = 0;
            mix0[AX_CH_RIGHT].bus[AX_AUXA_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_RIGHT]);
            mix0[AX_CH_RIGHT].bus[AX_AUXA_BUS].volDelta = 0;

            mix0[AX_CH_LEFT].bus[AX_AUXB_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_LEFT]);
            mix0[AX_CH_LEFT].bus[AX_AUXB_BUS].volDelta = 0;
            mix0[AX_CH_RIGHT].bus[AX_AUXB_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_RIGHT]);
            mix0[AX_CH_RIGHT].bus[AX_AUXB_BUS].volDelta = 0;

            mix0[AX_CH_LEFT].bus[AX_AUXC_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_LEFT]);
            mix0[AX_CH_LEFT].bus[AX_AUXC_BUS].volDelta = 0;
            mix0[AX_CH_RIGHT].bus[AX_AUXC_BUS].vol = GetVolumeU16(src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_RIGHT]);
            mix0[AX_CH_RIGHT].bus[AX_AUXC_BUS].volDelta = 0;
        }
    }
}

bool AxVoice::CalcUpdateTvMix(AXPBCHMIX mix[], const OutputMix& src, bool surroundFlag)
{
    u32 isUpdate = 0;

    if ( surroundFlag )
    {
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_REAR_LEFT] ),
            &mix[AX_CH_SUR_LEFT].bus[AX_MAIN_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_REAR_RIGHT] ),
            &mix[AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_CENTER] ),
            &mix[AX_CH_FC].bus[AX_MAIN_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_LFE] ),
            &mix[AX_CH_LFE].bus[AX_MAIN_BUS] ) ? 1 : 0;

        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_A][CHANNEL_INDEX_REAR_LEFT] ),
            &mix[AX_CH_SUR_LEFT].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_A][CHANNEL_INDEX_REAR_RIGHT] ),
            &mix[AX_CH_SUR_RIGHT].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_CENTER] ),
            &mix[AX_CH_FC].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_A][CHANNEL_INDEX_LFE] ),
            &mix[AX_CH_LFE].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_B][CHANNEL_INDEX_REAR_LEFT] ),
            &mix[AX_CH_SUR_LEFT].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_B][CHANNEL_INDEX_REAR_RIGHT] ),
            &mix[AX_CH_SUR_RIGHT].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_CENTER] ),
            &mix[AX_CH_FC].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_B][CHANNEL_INDEX_LFE] ),
            &mix[AX_CH_LFE].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_C][CHANNEL_INDEX_REAR_LEFT] ),
            &mix[AX_CH_SUR_LEFT].bus[AX_AUXC_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_C][CHANNEL_INDEX_REAR_RIGHT] ),
            &mix[AX_CH_SUR_RIGHT].bus[AX_AUXC_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_CENTER] ),
            &mix[AX_CH_FC].bus[AX_AUXC_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_C][CHANNEL_INDEX_LFE] ),
            &mix[AX_CH_LFE].bus[AX_AUXC_BUS] ) ? 1 : 0;
    }
    else
    {
        isUpdate |= UpdateDelta( 0, &mix[AX_CH_SUR_LEFT].bus[AX_MAIN_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix[AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix[AX_CH_FC].bus[AX_MAIN_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix[AX_CH_LFE].bus[AX_MAIN_BUS] ) ? 1 : 0;

        isUpdate |= UpdateDelta( 0, &mix[AX_CH_SUR_LEFT].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix[AX_CH_SUR_RIGHT].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix[AX_CH_FC].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix[AX_CH_LFE].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix[AX_CH_SUR_LEFT].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix[AX_CH_SUR_RIGHT].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix[AX_CH_FC].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix[AX_CH_LFE].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix[AX_CH_SUR_LEFT].bus[AX_AUXC_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix[AX_CH_SUR_RIGHT].bus[AX_AUXC_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix[AX_CH_FC].bus[AX_AUXC_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix[AX_CH_LFE].bus[AX_AUXC_BUS] ) ? 1 : 0;
    }

    isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_LEFT] ),
            &mix[AX_CH_LEFT].bus[AX_MAIN_BUS] ) ? 1 : 0;
    isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_RIGHT] ),
            &mix[AX_CH_RIGHT].bus[AX_MAIN_BUS] ) ? 1 : 0;

    isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_LEFT] ),
            &mix[AX_CH_LEFT].bus[AX_AUXA_BUS] ) ? 1 : 0;
    isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_RIGHT] ),
            &mix[AX_CH_RIGHT].bus[AX_AUXA_BUS] ) ? 1 : 0;
    isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_LEFT] ),
            &mix[AX_CH_LEFT].bus[AX_AUXB_BUS] ) ? 1 : 0;
    isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_RIGHT] ),
            &mix[AX_CH_RIGHT].bus[AX_AUXB_BUS] ) ? 1 : 0;
    isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_LEFT] ),
            &mix[AX_CH_LEFT].bus[AX_AUXC_BUS] ) ? 1 : 0;
    isUpdate |= UpdateDelta(
        GetVolumeU16( src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_RIGHT] ),
        &mix[AX_CH_RIGHT].bus[AX_AUXC_BUS] ) ? 1 : 0;

    return (isUpdate != 0);
}

bool AxVoice::CalcUpdateDrcMix(AXPBCHMIX* mix0, AXPBCHMIX* mix1,
        const OutputMix& src, bool surroundFlag, bool frontBypass)
{
    // モノ・ステレオの時 → DRC0 の fL, fR のみにルーティング
    // サラウンド (FB OFF) → DRC0 の fL, fR, rL, rR にルーティング
    // サラウンド (FB ON)  → [メインセンド] DRC1 の fL, fR にルーティング
    //                                       DRC0 の rL, rR にルーティング
    //                        [AUX センド]   DRC0 の fL, fR, rL, rR にルーティング

    NW_ASSERT_NOT_NULL(mix0);

    u32 isUpdate = 0;

    if (surroundFlag == false)
    {
        // front L/R の MAIN / AUX_A-C
        isUpdate |= UpdateDelta(
                GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_LEFT] ),
                &mix0[AX_CH_LEFT].bus[AX_MAIN_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
                GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_RIGHT] ),
                &mix0[AX_CH_RIGHT].bus[AX_MAIN_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
                GetVolumeU16( src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_LEFT] ),
                &mix0[AX_CH_LEFT].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
                GetVolumeU16( src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_RIGHT] ),
                &mix0[AX_CH_RIGHT].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
                GetVolumeU16( src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_LEFT] ),
                &mix0[AX_CH_LEFT].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
                GetVolumeU16( src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_RIGHT] ),
                &mix0[AX_CH_RIGHT].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
                GetVolumeU16( src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_LEFT] ),
                &mix0[AX_CH_LEFT].bus[AX_AUXC_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
                GetVolumeU16( src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_RIGHT] ),
                &mix0[AX_CH_RIGHT].bus[AX_AUXC_BUS] ) ? 1 : 0;

        // rear L/R の MAIN / AUX_A-C
        isUpdate |= UpdateDelta( 0, &mix0[AX_CH_SUR_LEFT].bus[AX_MAIN_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix0[AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix0[AX_CH_SUR_LEFT].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix0[AX_CH_SUR_RIGHT].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix0[AX_CH_SUR_LEFT].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix0[AX_CH_SUR_RIGHT].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix0[AX_CH_SUR_LEFT].bus[AX_AUXC_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta( 0, &mix0[AX_CH_SUR_RIGHT].bus[AX_AUXC_BUS] ) ? 1 : 0;
    }
    else
    {
        // front L/R の AUX_A-C
        isUpdate |= UpdateDelta(
                GetVolumeU16( src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_LEFT] ),
                &mix0[AX_CH_LEFT].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
                GetVolumeU16( src.auxBus[AUX_BUS_A][CHANNEL_INDEX_FRONT_RIGHT] ),
                &mix0[AX_CH_RIGHT].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
                GetVolumeU16( src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_LEFT] ),
                &mix0[AX_CH_LEFT].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
                GetVolumeU16( src.auxBus[AUX_BUS_B][CHANNEL_INDEX_FRONT_RIGHT] ),
                &mix0[AX_CH_RIGHT].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
                GetVolumeU16( src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_LEFT] ),
                &mix0[AX_CH_LEFT].bus[AX_AUXC_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
                GetVolumeU16( src.auxBus[AUX_BUS_C][CHANNEL_INDEX_FRONT_RIGHT] ),
                &mix0[AX_CH_RIGHT].bus[AX_AUXC_BUS] ) ? 1 : 0;

        // rear L/R の MAIN / AUX_A-C
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_REAR_LEFT] ),
            &mix0[AX_CH_SUR_LEFT].bus[AX_MAIN_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.mainBus[CHANNEL_INDEX_REAR_RIGHT] ),
            &mix0[AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_A][CHANNEL_INDEX_REAR_LEFT] ),
            &mix0[AX_CH_SUR_LEFT].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_A][CHANNEL_INDEX_REAR_RIGHT] ),
            &mix0[AX_CH_SUR_RIGHT].bus[AX_AUXA_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_B][CHANNEL_INDEX_REAR_LEFT] ),
            &mix0[AX_CH_SUR_LEFT].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_B][CHANNEL_INDEX_REAR_RIGHT] ),
            &mix0[AX_CH_SUR_RIGHT].bus[AX_AUXB_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_C][CHANNEL_INDEX_REAR_LEFT] ),
            &mix0[AX_CH_SUR_LEFT].bus[AX_AUXC_BUS] ) ? 1 : 0;
        isUpdate |= UpdateDelta(
            GetVolumeU16( src.auxBus[AUX_BUS_C][CHANNEL_INDEX_REAR_RIGHT] ),
            &mix0[AX_CH_SUR_RIGHT].bus[AX_AUXC_BUS] ) ? 1 : 0;

        // front L/R の MAIN
        if (frontBypass == false
            || driver::HardwareManager::GetInstance().GetEndUserOutputMode(OUTPUT_DEVICE_DRC) != OUTPUT_MODE_SURROUND)
        {
            isUpdate |= UpdateDelta(
                    GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_LEFT] ),
                    &mix0[AX_CH_LEFT].bus[AX_MAIN_BUS] ) ? 1 : 0;
            isUpdate |= UpdateDelta(
                    GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_RIGHT] ),
                    &mix0[AX_CH_RIGHT].bus[AX_MAIN_BUS] ) ? 1 : 0;
        }
        else
        {
            isUpdate |= UpdateDelta(
                    GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_LEFT] ),
                    &mix1[AX_CH_LEFT].bus[AX_MAIN_BUS] ) ? 1 : 0;
            isUpdate |= UpdateDelta(
                    GetVolumeU16( src.mainBus[CHANNEL_INDEX_FRONT_RIGHT] ),
                    &mix1[AX_CH_RIGHT].bus[AX_MAIN_BUS] ) ? 1 : 0;
        }
    }

    return (isUpdate != 0);
}

void AxVoice::SetupTvMix( const VoiceParam& voiceParam, OutputMode mode )
{
    CalcSetupTvMix( m_AxMixTv[AX_TV_ID0], voiceParam.m_TvMix, mode == OUTPUT_MODE_SURROUND);

    NW_DEBUG_LOG_BUFFER("+ Voice::SetupVoice() : AXSetVoiceDeviceMix\n");
    AXSetVoiceDeviceMix(m_pVpb, AX_DEVICE_TV, AX_TV_ID0, m_AxMixTv[AX_TV_ID0]);
    NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceDeviceMix [%08x][device:%d][id:%u][fL:%x][fR:%x][rL:%x][rR:%x][lfe:%x][fC:%x][fLd:%d][fRd:%d][rLd:%d][rRd:%d][lfed:%d][fCd:%d] AxVoice::SetupTvMix"
        , m_pVpb
        , AX_DEVICE_TV
        , AX_TV_ID0
        , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].vol)
        , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_RIGHT].bus[AX_MAIN_BUS].vol)
        , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].vol)
        , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].vol)
        , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_LFE].bus[AX_MAIN_BUS].vol)
        , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_FC].bus[AX_MAIN_BUS].vol)
        , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta)
        , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_RIGHT].bus[AX_MAIN_BUS].volDelta)
        , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].volDelta)
        , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].volDelta)
        , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_LFE].bus[AX_MAIN_BUS].volDelta)
        , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_FC].bus[AX_MAIN_BUS].volDelta)
        );
    NW_DEBUG_LOG_BUFFER("- Voice::SetupVoice() : AXSetVoiceDeviceMix\n");
}

void AxVoice::UpdateTvMix( const VoiceParam& voiceParam, OutputMode mode )
{
    if ( CalcUpdateTvMix( m_AxMixTv[AX_TV_ID0], voiceParam.m_TvMix, mode == OUTPUT_MODE_SURROUND) )
    {
        NW_DEBUG_LOG_BUFFER("+ Voice::UpdateState() : AXSetVoiceDeviceMix\n");
        AXSetVoiceDeviceMix(m_pVpb, AX_DEVICE_TV, AX_TV_ID0, m_AxMixTv[AX_TV_ID0]);
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceDeviceMix [%08x][device:%d][id:%u][fL:%x][fR:%x][rL:%x][rR:%x][lfe:%x][fC:%x][fLd:%d][fRd:%d][rLd:%d][rRd:%d][lfed:%d][fCd:%d] AxVoice::UpdateTvMix"
            , m_pVpb
            , AX_DEVICE_TV
            , AX_TV_ID0
            , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_RIGHT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_LFE].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_FC].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta)
            , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_RIGHT].bus[AX_MAIN_BUS].volDelta)
            , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].volDelta)
            , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].volDelta)
            , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_LFE].bus[AX_MAIN_BUS].volDelta)
            , static_cast<s32>(m_AxMixTv[AX_TV_ID0][AX_CH_FC].bus[AX_MAIN_BUS].volDelta)
            );
        NW_DEBUG_LOG_BUFFER("- Voice::UpdateState() : AXSetVoiceDeviceMix\n");
        // NW_LOG("[%14s](%p) fL(%4X) fR(%4X) rL(%4X) rR(%4X)\n",
        //         __FUNCTION__, this,
        //         m_AxMixDrc[AX_DRC_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].vol,
        //         m_AxMixDrc[AX_DRC_ID0][AX_CH_RIGHT].bus[AX_MAIN_BUS].vol,
        //         m_AxMixDrc[AX_DRC_ID0][AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].vol,
        //         m_AxMixDrc[AX_DRC_ID0][AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].vol
        //         );
    }
}

void AxVoice::SetupDrcMix( const VoiceParam& voiceParam, OutputMode mode )
{
    CalcSetupDrcMix(
            m_AxMixDrc[AX_DRC_ID0],
            m_AxMixDrc[AX_DRC_ID1],
            voiceParam.m_DrcMix[AX_DRC_ID0],
            mode == OUTPUT_MODE_SURROUND,
            voiceParam.m_DrcFrontBypass);

    AXSetVoiceDeviceMix(m_pVpb, AX_DEVICE_DRC, AX_DRC_ID0, m_AxMixDrc[AX_DRC_ID0]);
    NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceDeviceMix [%08x][device:%d][id:%u][fL:%d][fR:%d][rL:%d][rR:%d][fLd:%d][fRd:%d][rLd:%d][rRd:%d] AxVoice::SetupDrcMix"
        , m_pVpb
        , AX_DEVICE_DRC
        , AX_DRC_ID0
        , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].vol)
        , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_RIGHT].bus[AX_MAIN_BUS].vol)
        , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].vol)
        , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].vol)
        , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta)
        , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_RIGHT].bus[AX_MAIN_BUS].volDelta)
        , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].volDelta)
        , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].volDelta)
        );

#if defined(NW_PLATFORM_CAFE)
    // フロントバイパスが有効な場合は、FL/FR を DRC1 に MIX します。
    if(mode == OUTPUT_MODE_SURROUND && voiceParam.m_DrcFrontBypass)
    {
        AXSetVoiceDeviceMix(m_pVpb, AX_DEVICE_DRC, AX_DRC_ID1, m_AxMixDrc[AX_DRC_ID1]);
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceDeviceMix [%08x][device:%d][id:%u][fL:%d][fR:%d][rL:%d][rR:%d][fLd:%d][fRd:%d][rLd:%d][rRd:%d] AxVoice::SetupDrcMix"
            , m_pVpb
            , AX_DEVICE_DRC
            , AX_DRC_ID1
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_RIGHT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_RIGHT].bus[AX_MAIN_BUS].volDelta)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].volDelta)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].volDelta)
            );
    }
#endif
}

void AxVoice::UpdateDrcMix( const VoiceParam& voiceParam, OutputMode mode )
{
    bool result = CalcUpdateDrcMix(
            m_AxMixDrc[AX_DRC_ID0],
            m_AxMixDrc[AX_DRC_ID1],
            voiceParam.m_DrcMix[AX_DRC_ID0],
            mode == OUTPUT_MODE_SURROUND,
            voiceParam.m_DrcFrontBypass);

    if (result)
    {
        NW_DEBUG_LOG_BUFFER("+ Voice::UpdateState() : AXSetVoiceDRCMix\n");
        AXSetVoiceDeviceMix(m_pVpb, AX_DEVICE_DRC, AX_DRC_ID0, m_AxMixDrc[AX_DRC_ID0]);
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceDeviceMix [%08x][device:%d][id:%u][fL:%x][fR:%x][rL:%x][rR:%x][fLd:%d][fRd:%d][rLd:%d][rRd:%d] AxVoice::UpdateDrcMix"
            , m_pVpb
            , AX_DEVICE_DRC
            , AX_DRC_ID0
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_RIGHT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_RIGHT].bus[AX_MAIN_BUS].volDelta)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].volDelta)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID0][AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].volDelta)
            );
        NW_DEBUG_LOG_BUFFER("- Voice::UpdateState() : AXSetVoiceDRCMix\n");
        // NW_LOG("[%14s](%p) fL(%4X) fR(%4X) rL(%4X) rR(%4X)\n",
        //         __FUNCTION__, this,
        //         m_AxMixDrc[AX_DRC_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].vol,
        //         m_AxMixDrc[AX_DRC_ID0][AX_CH_RIGHT].bus[AX_MAIN_BUS].vol,
        //         m_AxMixDrc[AX_DRC_ID0][AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].vol,
        //         m_AxMixDrc[AX_DRC_ID0][AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].vol
        //         );
    }

#if defined(NW_PLATFORM_CAFE)
    // フロントバイパスが有効な場合は、FL/FR を DRC1 に MIX します。
    if(result && mode == OUTPUT_MODE_SURROUND && voiceParam.m_DrcFrontBypass)
    {
        NW_DEBUG_LOG_BUFFER("+ Voice::UpdateState() : AXSetVoiceDRCMix for FrontBypass\n");
        AXSetVoiceDeviceMix(m_pVpb, AX_DEVICE_DRC, AX_DRC_ID1, m_AxMixDrc[AX_DRC_ID1]);
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceDeviceMix [%08x][device:%d][id:%u][fL:%x][fR:%x][rL:%x][rR:%x][fLd:%d][fRd:%d][rLd:%d][rRd:%d] AxVoice::UpdateDrcMix"
            , m_pVpb
            , AX_DEVICE_DRC
            , AX_DRC_ID1
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_RIGHT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_RIGHT].bus[AX_MAIN_BUS].volDelta)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].volDelta)
            , static_cast<s32>(m_AxMixDrc[AX_DRC_ID1][AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].volDelta)
            );
        NW_DEBUG_LOG_BUFFER("- Voice::UpdateState() : AXSetVoiceDRCMix for FrontBypass\n");
        // NW_LOG("[%14s](%p) fL(%4X) fR(%4X) rL(%4X) rR(%4X)\n",
        //         __FUNCTION__, this,
        //         m_AxMixDrc[AX_DRC_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].vol,
        //         m_AxMixDrc[AX_DRC_ID0][AX_CH_RIGHT].bus[AX_MAIN_BUS].vol,
        //         m_AxMixDrc[AX_DRC_ID0][AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].vol,
        //         m_AxMixDrc[AX_DRC_ID0][AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].vol
        //         );
    }
#endif
}

void AxVoice::SetupRmtMix( const VoiceParam& voiceParam )
{
    m_RemoteOn = voiceParam.m_RemoteMix.enable ? AX_PB_REMOTE_ON : AX_PB_REMOTE_OFF;

    AXSetVoiceRmtOn( m_pVpb, m_RemoteOn );
    NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceRmtOn [%08x][rmtOn:%x] AxVoice::SetupRmtMix", m_pVpb, m_RemoteOn);

    // REMOTE MIX
    if ( voiceParam.m_RemoteMix.enable )
    {
        m_AxMixRemote[AX_RMT_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].vol = GetVolumeU16( voiceParam.m_RemoteMix.mainBus[AX_RMT_ID0] );
        m_AxMixRemote[AX_RMT_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta = 0;

        m_AxMixRemote[AX_RMT_ID1][AX_CH_LEFT].bus[AX_MAIN_BUS].vol = GetVolumeU16( voiceParam.m_RemoteMix.mainBus[AX_RMT_ID1] );
        m_AxMixRemote[AX_RMT_ID1][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta = 0;

        m_AxMixRemote[AX_RMT_ID2][AX_CH_LEFT].bus[AX_MAIN_BUS].vol = GetVolumeU16( voiceParam.m_RemoteMix.mainBus[AX_RMT_ID2] );
        m_AxMixRemote[AX_RMT_ID2][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta = 0;

        m_AxMixRemote[AX_RMT_ID3][AX_CH_LEFT].bus[AX_MAIN_BUS].vol = GetVolumeU16( voiceParam.m_RemoteMix.mainBus[AX_RMT_ID3] );
        m_AxMixRemote[AX_RMT_ID3][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta = 0;

        AXSetVoiceDeviceMix( m_pVpb, AX_DEVICE_RMT, AX_RMT_ID0, m_AxMixRemote[AX_RMT_ID0] );
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceDeviceMix [%08x][device:%d][id:%u][vol:%x][vold:%x] AxVoice::SetupRmtMix"
            , m_pVpb
            , AX_DEVICE_RMT
            , AX_RMT_ID0
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta)
            );
        AXSetVoiceDeviceMix( m_pVpb, AX_DEVICE_RMT, AX_RMT_ID1, m_AxMixRemote[AX_RMT_ID1] );
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceDeviceMix [%08x][device:%d][id:%u][vol:%x][vold:%x] AxVoice::SetupRmtMix"
            , m_pVpb
            , AX_DEVICE_RMT
            , AX_RMT_ID1
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID1][AX_CH_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID1][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta)
            );
        AXSetVoiceDeviceMix( m_pVpb, AX_DEVICE_RMT, AX_RMT_ID2, m_AxMixRemote[AX_RMT_ID2] );
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceDeviceMix [%08x][device:%d][id:%u][vol:%x][vold:%x] AxVoice::SetupRmtMix"
            , m_pVpb
            , AX_DEVICE_RMT
            , AX_RMT_ID2
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID2][AX_CH_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID2][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta)
            );
        AXSetVoiceDeviceMix( m_pVpb, AX_DEVICE_RMT, AX_RMT_ID3, m_AxMixRemote[AX_RMT_ID3] );
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceDeviceMix [%08x][device:%d][id:%u][vol:%x][vold:%x] AxVoice::SetupRmtMix"
            , m_pVpb
            , AX_DEVICE_RMT
            , AX_RMT_ID3
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID3][AX_CH_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID3][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta)
            );
    }
}

void AxVoice::UpdateRmtMix( const VoiceParam& voiceParam )
{
    u16 remoteOn = voiceParam.m_RemoteMix.enable ? AX_PB_REMOTE_ON : AX_PB_REMOTE_OFF;
    if ( remoteOn != m_RemoteOn )
    {
        AXSetVoiceRmtOn( m_pVpb, remoteOn );
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceRmtOn [%08x][rmtOn:%x] AxVoice::UpdateRmtMix", m_pVpb, remoteOn);
        m_RemoteOn = remoteOn;
    }

    u32 isUpdate = 0;
    isUpdate |= UpdateDelta(
            GetVolumeU16( voiceParam.m_RemoteMix.mainBus[0] ),
            &m_AxMixRemote[AX_RMT_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS] ) ? 1 : 0;
    isUpdate |= UpdateDelta(
            GetVolumeU16( voiceParam.m_RemoteMix.mainBus[1] ),
            &m_AxMixRemote[AX_RMT_ID1][AX_CH_LEFT].bus[AX_MAIN_BUS] ) ? 1 : 0;
    isUpdate |= UpdateDelta(
            GetVolumeU16( voiceParam.m_RemoteMix.mainBus[2] ),
            &m_AxMixRemote[AX_RMT_ID2][AX_CH_LEFT].bus[AX_MAIN_BUS] ) ? 1 : 0;
    isUpdate |= UpdateDelta(
            GetVolumeU16( voiceParam.m_RemoteMix.mainBus[3] ),
            &m_AxMixRemote[AX_RMT_ID3][AX_CH_LEFT].bus[AX_MAIN_BUS] ) ? 1 : 0;

    if ( isUpdate )
    {
        NW_DEBUG_LOG_BUFFER("+ Voice::UpdateState() : AXSetVoiceRmtMix\n");
        AXSetVoiceDeviceMix(m_pVpb, AX_DEVICE_RMT, AX_RMT_ID0, m_AxMixRemote[AX_RMT_ID0]);
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceDeviceMix [%08x][device:%d][id:%u][vol:%x][vold:%x] AxVoice::UpdateRmtMix"
            , m_pVpb
            , AX_DEVICE_RMT
            , AX_RMT_ID0
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta)
            );
        AXSetVoiceDeviceMix(m_pVpb, AX_DEVICE_RMT, AX_RMT_ID1, m_AxMixRemote[AX_RMT_ID1]);
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceDeviceMix [%08x][device:%d][id:%u][vol:%x][vold:%x] AxVoice::UpdateRmtMix"
            , m_pVpb
            , AX_DEVICE_RMT
            , AX_RMT_ID1
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID1][AX_CH_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID1][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta)
            );
        AXSetVoiceDeviceMix(m_pVpb, AX_DEVICE_RMT, AX_RMT_ID2, m_AxMixRemote[AX_RMT_ID2]);
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceDeviceMix [%08x][device:%d][id:%u][vol:%x][vold:%x] AxVoice::UpdateRmtMix"
            , m_pVpb
            , AX_DEVICE_RMT
            , AX_RMT_ID2
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID2][AX_CH_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID2][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta)
            );
        AXSetVoiceDeviceMix(m_pVpb, AX_DEVICE_RMT, AX_RMT_ID3, m_AxMixRemote[AX_RMT_ID3]);
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceDeviceMix [%08x][device:%d][id:%u][vol:%x][vold:%x] AxVoice::UpdateRmtMix"
            , m_pVpb
            , AX_DEVICE_RMT
            , AX_RMT_ID3
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID3][AX_CH_LEFT].bus[AX_MAIN_BUS].vol)
            , static_cast<s32>(m_AxMixRemote[AX_RMT_ID3][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta)
            );
        NW_DEBUG_LOG_BUFFER("- Voice::UpdateState() : AXSetVoiceRmtMix\n");
    }
}

void AxVoice::UpdateVe( const VoiceParam& voiceParam, bool initialFlag )
{
    u16 targetVolume = GetVolumeU16( voiceParam.m_Volume );

    if ( initialFlag )
    {
        m_Ve.currentVolume = targetVolume;
        m_Ve.currentDelta = 0;

        NW_DEBUG_LOG_BUFFER("+ Voice::UpdateState() : AXSetVoiceVe\n");
        AXSetVoiceVe(m_pVpb, &m_Ve);
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceVe [%08x][curVol:%d][curDel:%d] AxVoice::UpdateVe initialFlag"
            , m_pVpb
            , static_cast<s32>(m_Ve.currentVolume)
            , static_cast<s32>(m_Ve.currentDelta));
        NW_DEBUG_LOG_BUFFER("- Voice::UpdateState() : AXSetVoiceVe\n");
    }
    else
    {
        AXPBCHVE ve;
        ve.vol = m_Ve.currentVolume;
        ve.volDelta = m_Ve.currentDelta;
        if ( UpdateDelta( targetVolume, &ve ) )
        {
            m_Ve.currentVolume = ve.vol;
            m_Ve.currentDelta = ve.volDelta;

            NW_DEBUG_LOG_BUFFER("+ Voice::UpdateState() : AXSetVoiceVe\n");
            AXSetVoiceVe(m_pVpb, &m_Ve);
            NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceVe [%08x][curVol:%d][curDel:%d] AxVoice::UpdateVe"
                , m_pVpb
                , static_cast<s32>(m_Ve.currentVolume)
                , static_cast<s32>(m_Ve.currentDelta));
            NW_DEBUG_LOG_BUFFER("- Voice::UpdateState() : AXSetVoiceVe\n");
        }
    }
}

void AxVoice::UpdateSrc( const VoiceParam& voiceParam, bool initialFlag )
{
    f32 srcRatio = voiceParam.m_Pitch * m_SampleRate / s_SamplesPerSec;

    if ( initialFlag )
    {
        u32 srcBits = static_cast<u32>( BASE_PITCH * srcRatio );
        AXPBSRC src;
        src.ratioHi = GetHiFromU32( srcBits );
        src.ratioLo = GetLoFromU32( srcBits );
        if ( src.ratioHi >= 8 )
        {
            src.ratioHi = 8;
            src.ratioLo = 0;
        }
        src.currentAddressFrac = 0;
        src.last_samples[0] = 0;
        src.last_samples[1] = 0;
        src.last_samples[2] = 0;
        src.last_samples[3] = 0;

        u32 srcType;
        switch ( voiceParam.m_InterpolationType )
        {
        case 0: // ４点補間
            if ( srcRatio > (4.0f/3.0f) ) {
                srcType = AX_SRC_TYPE_4TAP_8K;
            }
            else if ( srcRatio > 1.0f ) {
                srcType = AX_SRC_TYPE_4TAP_12K;
            }
            else {
                srcType = AX_SRC_TYPE_4TAP_16K;
            }
            break;
        case 1: // 線形補間
            srcType = AX_SRC_TYPE_LINEAR;
            break;
        case 2: // 補間なし
            srcType = AX_SRC_TYPE_NONE;
            break;
        default:
            srcType = 0;
            NW_ASSERTMSG( 0, "invalid InterpolationType(%u)\n", voiceParam.m_InterpolationType );
            break;
        }

        NW_DEBUG_LOG_BUFFER("+ Voice::SetupVoice() : AXSetVoiceSrc : ratioHi=%d, ratioLo=%d\n", src.ratioHi, src.ratioLo);
        AXSetVoiceSrc(m_pVpb, &src);
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceSrc [%08x][ratioHi:%u][ratioLo:%u] AxVoice::UpdateSrc", m_pVpb, src.ratioHi, src.ratioLo);
        NW_DEBUG_LOG_BUFFER("- Voice::SetupVoice() : AXSetVoiceSrc\n");

        NW_DEBUG_LOG_BUFFER("+ Voice::SetupVoice() : AXSetVoiceSrcType : %d\n", srcType);
        AXSetVoiceSrcType(m_pVpb, srcType);
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceSrcType [%08x][type:%d] AxVoice::UpdateSrc", m_pVpb, srcType);
        NW_DEBUG_LOG_BUFFER("- Voice::SetupVoice() : AXSetVoiceSrcType\n");

        m_SrcRatio = srcRatio;
    }
    else if ( srcRatio != m_SrcRatio ) {
        NW_DEBUG_LOG_BUFFER("+ Voice::UpdateState() : AXSetVoiceSrcRatio : m_SrcRatio=%5.2f\n", srcRatio);
        AXSetVoiceSrcRatio(m_pVpb, srcRatio);
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceSrcRatio [%08x][ratio:%f] AxVoice::UpdateSrc", m_pVpb, srcRatio);
        NW_DEBUG_LOG_BUFFER("- Voice::UpdateState() : AXSetVoiceSrcRatio\n");

        m_SrcRatio = srcRatio;
    }
}

void AxVoice::UpdateMonoFilter( const VoiceParam& voiceParam, bool initialFlag )
{
    u16 monoFilterOn = voiceParam.m_MonoFilterFlag ? AX_PB_LPF_ON : AX_PB_LPF_OFF;

    if ( initialFlag || monoFilterOn != m_MonoFilterOn )
    {
        AXPBLPF lpf;
        lpf.on = monoFilterOn;

        if ( monoFilterOn == AX_PB_LPF_ON )
        {
            lpf.yn1 = 0;
            AXComputeLpfCoefs( voiceParam.m_MonoFilterCutoff, &lpf.a0, &lpf.b0 );
            NW_SND_LOG_APICALL_SDK_LOW("AXComputeLpfCoefs [%08x][freq:%u][a0:%08x][b0:%08x] AxVoice::UpdateMonoFilter"
                , m_pVpb, voiceParam.m_MonoFilterCutoff, &lpf.a0, &lpf.b0);
        }
        AXSetVoiceLpf( m_pVpb, &lpf );
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceLpf [%08x][on:%u][yn1:%d][a0:%d][b0:%d] AxVoice::UpdateMonoFilter", m_pVpb, lpf.on, lpf.yn1, lpf.a0, lpf.b0);
        m_MonoFilterOn = monoFilterOn;
    }
    else
    {
        if ( monoFilterOn == AX_PB_LPF_ON )
        {
            u16 a0, b0;
            AXComputeLpfCoefs( voiceParam.m_MonoFilterCutoff, &a0, &b0 );
            NW_SND_LOG_APICALL_SDK_LOW("AXComputeLpfCoefs [%08x][freq:%u][a0:%08x][b0:%08x] AxVoice::UpdateMonoFilter"
                , m_pVpb, voiceParam.m_MonoFilterCutoff, &a0, &b0);
            AXSetVoiceLpfCoefs( m_pVpb, a0, b0 );
            NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceLpfCoefs [%08x][a0:%u][b0:%u] AxVoice::UpdateMonoFilter", m_pVpb, a0, b0);
        }
    }
}

void AxVoice::UpdateBiquadFilter( const VoiceParam& voiceParam, bool initialFlag )
{
    u16 biquadFilterOn = voiceParam.m_BiquadFilterFlag ? AX_PB_BIQUAD_ON : AX_PB_BIQUAD_OFF;

    if ( initialFlag || biquadFilterOn != m_BiquadFilterOn )
    {
        AXPBBIQUAD biquad;
        biquad.on = biquadFilterOn;

        if ( biquadFilterOn == AX_PB_BIQUAD_ON )
        {
            biquad.xn1 = 0;
            biquad.xn2 = 0;
            biquad.yn1 = 0;
            biquad.yn2 = 0;

            biquad.b0  = voiceParam.m_BiquadFilterCoefficients.b0;
            biquad.b1  = voiceParam.m_BiquadFilterCoefficients.b1;
            biquad.b2  = voiceParam.m_BiquadFilterCoefficients.b2;
            biquad.a1  = voiceParam.m_BiquadFilterCoefficients.a1;
            biquad.a2  = voiceParam.m_BiquadFilterCoefficients.a2;
        }

        AXSetVoiceBiquad( m_pVpb, &biquad );
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceBiquad [%08x][on:%u][xn1:%d][xn2:%d][yn1:%d][yn2:%d][b0:%d][b1:%d][b2:%d][a1:%d][a2:%d] AxVoice::UpdateBiquadFilter"
            , m_pVpb
            , biquad.on
            , biquad.xn1
            , biquad.xn2
            , biquad.yn1
            , biquad.yn2
            , biquad.b0
            , biquad.b1
            , biquad.b2
            , biquad.a1
            , biquad.a2
            );
        m_BiquadFilterOn = biquadFilterOn;
    }
    else
    {
        if ( biquadFilterOn == AX_PB_BIQUAD_ON )
        {
            AXSetVoiceBiquadCoefs(
                m_pVpb,
                voiceParam.m_BiquadFilterCoefficients.b0,
                voiceParam.m_BiquadFilterCoefficients.b1,
                voiceParam.m_BiquadFilterCoefficients.b2,
                voiceParam.m_BiquadFilterCoefficients.a1,
                voiceParam.m_BiquadFilterCoefficients.a2
            );
            NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceBiquadCoefs [%08x][b0:%u][b1:%u][b2:%u][a1:%u][a2:%u] AxVoice::UpdateBiquadFilter"
                , m_pVpb
                , voiceParam.m_BiquadFilterCoefficients.b0
                , voiceParam.m_BiquadFilterCoefficients.b1
                , voiceParam.m_BiquadFilterCoefficients.b2
                , voiceParam.m_BiquadFilterCoefficients.a1
                , voiceParam.m_BiquadFilterCoefficients.a2
                );
        }
    }
}

void AxVoice::UpdateRemoteFilter( const VoiceParam& voiceParam, bool initialFlag )
{
    u16 remoteFilterOn = voiceParam.m_RemoteFilterFlag ? AX_PB_BIQUAD_ON : AX_PB_BIQUAD_OFF;

    if ( initialFlag || remoteFilterOn != m_RemoteFilterOn )
    {
        AXPBRMTIIR iir;
        iir.biquad.on = remoteFilterOn;

        if ( remoteFilterOn == AX_PB_BIQUAD_ON )
        {
            iir.biquad.xn1 = 0;
            iir.biquad.xn2 = 0;
            iir.biquad.yn1 = 0;
            iir.biquad.yn2 = 0;

            Util::GetRemoteFilterCoefs(
                voiceParam.m_RemoteFilter,
                &iir.biquad.b0,
                &iir.biquad.b1,
                &iir.biquad.b2,
                &iir.biquad.a1,
                &iir.biquad.a2
            );
        }

        AXSetVoiceRmtIIR( m_pVpb, &iir );
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceRmtIIR [%08x][xn1:%d][xn2:%d][yn1:%d][yn2:%d][b0:%d][b1:%d][b2:%d][a1:%d][a2:%d] AxVoice::UpdateRemoteFilter"
            , m_pVpb
            , iir.biquad.xn1
            , iir.biquad.xn2
            , iir.biquad.yn1
            , iir.biquad.yn2
            , iir.biquad.b0
            , iir.biquad.b1
            , iir.biquad.b2
            , iir.biquad.a1
            , iir.biquad.a2
            );
        m_RemoteFilterOn = remoteFilterOn;
    }
    else
    {
        s16 b0, b1, b2, a1, a2;
        Util::GetRemoteFilterCoefs( voiceParam.m_RemoteFilter, &b0, &b1, &b2, &a1, &a2 );
        AXSetVoiceRmtIIRCoefs( m_pVpb, AX_PB_BIQUAD_ON, b0, b1, b2, a1, a2 );
        NW_SND_LOG_APICALL_SDK_LOW("AXSetVoiceRmtIIRCoefs [%08x][type:%d][b0:%d][b1:%d][b2:%d][a1:%d][a2:%d] AxVoice::UpdateRemoteFilter"
            , m_pVpb
            , AX_PB_BIQUAD_ON
            , b0
            , b1
            , b2
            , a1
            , a2
            );
    }
}

void AxVoice::UpdateVoiceInfo( VoiceInfo* voiceInfo ) const
{
    NW_NULL_ASSERT(voiceInfo);

    voiceInfo->userId = m_pVoice;
    voiceInfo->voiceState = m_State;
    volatile WaveBuffer* pListBegin = m_WaveBufferListBegin;
    if ( pListBegin != NULL )
    {
        voiceInfo->waveBufferStatus = pListBegin->status;
        voiceInfo->waveBufferTag = pListBegin->userParam;
    }
    else
    {
        voiceInfo->waveBufferStatus = WaveBuffer::STATUS_FREE;
    }
    voiceInfo->playPosition = m_PlayPosition;
}


}}}
