﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/string/sndfnd_Path.h>

#include <limits>
#include <string.h>
#include <cafe/env.h>
#include <nw/snd/fnd/string/sndfnd_String.h>

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//----------------------------------------------------------
void
Path::ConvertToPcFilePath(
    const char* src,
    u32 srcLength,
    char* dest,
    u32 destLength,
    const char* volume,
    u32 volumeLength)
{
    NW_ASSERT_NOT_NULL(src);
    NW_ASSERT_NOT_NULL(dest);
    NW_ASSERT_NOT_NULL(volume);
    NW_ASSERT(srcLength > 0);
    NW_ASSERT(destLength > 0);
    NW_ASSERT(volumeLength > 0);
    NW_ASSERTMSG(volumeLength + srcLength < destLength, "buffer over flow.\n");

    // HACK : フルパスのみをサポートしています。
    NW_ASSERT(srcLength > 1);
    NW_ASSERT(src[1] == ':');
    NW_ASSERT(src[2] == '\\' || src[2] == '/');

    u32  pathOffset     = volumeLength;
    bool shouldAddSlash = volume[volumeLength - 1] != '/';

    if(shouldAddSlash)
    {
        ++pathOffset;
    }

    // ドライブレターを除くパスをコピーします。
    String::Copy(
        dest + pathOffset + 1,
        destLength - pathOffset,
        src + 2,
        srcLength - 2);

    // 終端文字を設定します。
    if(src[srcLength - 1] != '\0')
    {
        dest[pathOffset + srcLength - 1] = '\0';
    }

    // ドライブレターをコピーします。
    dest[pathOffset] = src[0];

    // パス上に含まれる '\\' --> '/' に変換します。
    u32 index = pathOffset + 1;
    u32 endIndex = index + srcLength;

    for(; index < endIndex; ++index)
    {
        if(dest[index] == '\\')
        {
            dest[index] = '/';
        }
    }

    // ボリュームパスをコピーします。
    String::Copy(
        dest,
        volumeLength,
        volume,
        volumeLength);

    if(shouldAddSlash)
    {
        dest[volumeLength] = '/';
    }
}

//----------------------------------------------------------
u32
Path::ExpandEnvironmentVariables(
    char* dest,
    u32 destLength,
    const char* src,
    u32 srcLength)
{
    NW_ASSERT_NOT_NULL(src);
    NW_ASSERT_NOT_NULL(dest);
    NW_ASSERT(srcLength > 0);
    NW_ASSERT(destLength > 0);

    const u32 variableNameMaxLength = 64;
    char variableName[variableNameMaxLength] = { 0 };

    u32 variableStart = std::numeric_limits<u32>::max();
    u32 destIndex = 0;

    for(u32 srcIndex = 0; srcIndex < srcLength; ++srcIndex)
    {
        if(destIndex >= destLength)
        {
            break;
        }

        if(src[srcIndex] != '%')
        {
            // 環境変数を検出していないならコピー
            if(variableStart == std::numeric_limits<u32>::max())
            {
                dest[destIndex] = src[srcIndex];
                ++destIndex;
            }

            continue;
        }

        // 環境変数開始トークンの検出
        if(variableStart == std::numeric_limits<u32>::max())
        {
            variableStart = srcIndex + 1;
            continue;
        }
        // 環境変数終了トークンの検出
        else
        {
            u32 variableNameLength = srcIndex - variableStart;

            // 環境変数名の長さが上限を超えている場合は失敗させる
            if(variableNameMaxLength < variableNameLength)
            {
                NW_ASSERTMSG(false,
                    "[%s] environment variable name is too long. : '%s'\n",
                    __FUNCTION__, variableName);

                dest[0] = '\0';
                return 0;
            }

            // 環境変数を展開する
            String::Copy(variableName, variableNameMaxLength, src + variableStart, variableNameLength);

            if(ENVGetEnvironmentVariable(
                variableName,
                dest + destIndex,
                destLength - destIndex) != 0)
            {
                NW_WARNING(false,
                    "[%s] environment variable '%s' not found.\n",
                    __FUNCTION__, variableName);
                return 0;
            }

            // ENVGetEnvironmentVariable が変数名の長さを返してくれないので、
            // strlen() で計算する
            destIndex = strlen(dest);

            variableStart = std::numeric_limits<u32>::max();
        }
    }

    NW_ASSERT(destIndex < destLength);
    dest[destIndex] = '\0';

    // 終端文字を含めます。
    return destIndex + 1;
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
