﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/io/sndfnd_SimpleRingBuffer.h>

#include <new>
#include <cstring>

#if defined( NW_PLATFORM_CAFE )
#pragma ghs nowarning 177 // function was declared but never referenced.
#endif

#if !defined( NW_PLATFORM_CTR )
#include <nw/ut/ut_ScopedLock.h>
#endif

/* ========================================================================
    static
   ======================================================================== */

namespace
{

using namespace nw::snd::internal::fnd;

inline u32
GetMin(
    const u32 a,
    const u32 b
)
{
    return a < b ? a: b;
}

inline u32
RoundUp(
    u32    value,
    u32    alignment
)
{
    return (value + alignment -1) & ~(alignment -1);
}

inline void*
RoundUp(
    void*  value,
    u32    alignment
)
{
    return reinterpret_cast<void*>(RoundUp(reinterpret_cast<u32>(value), alignment));
}

inline u32
RoundDown(
    u32    value,
    u32    alignment
)
{
    return value & ~(alignment -1);
}

u32
GetReadableBytes(
    const SimpleRingBufferRange&  range,
    u32                           bufSize
)
{
    if (range.start <= range.end)
    {
        return range.end - range.start;
    }
    else
    {
        return range.end + bufSize - range.start;
    }
}

u32
GetSequenceReadableBytes(
    const SimpleRingBufferRange&  range,
    u32                           bufSize
)
{
    if (range.start <= range.end)
    {
        return range.end - range.start;
    }
    else
    {
        return bufSize - range.start;
    }
}

u32
GetWritableBytes(
    const SimpleRingBufferRange&  range,
    u32                           bufSize
)
{
    u32 start = (range.start + bufSize - 4) % bufSize;

    if (start >= range.end)
    {
        return start - range.end;
    }
    else
    {
        return start + bufSize - range.end;
    }
}

u32
GetSequenceWritableBytes(
    const SimpleRingBufferRange&  range,
    u32                           bufSize
)
{
    u32 start = (range.start + bufSize - 4) % bufSize;

    if (start >= range.end)
    {
        return start - range.end;
    }
    else
    {
        return bufSize - range.end;
    }
}

}   // namespace


/* ========================================================================
    外部
   ======================================================================== */

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

#if defined( NW_PLATFORM_CTR )
typedef LockObject::ScopedLock ScopedLock;
#else
typedef nw::ut::ScopedLock<LockObject> ScopedLock;
#endif

//---------------------------------------------------------------------------
void
SimpleRingBuffer::Init(
    void*   buf,
    u32     size
)
{
    u32 bufStart = reinterpret_cast<u32>(buf);
    u32 bufEnd = bufStart + size;

    bufStart = RoundUp(bufStart, 4);   // 4バイトアライン
    bufEnd = RoundDown(bufEnd, 4);     // 4バイトアライン

    NW_ASSERT(bufStart < bufEnd);
    NW_ASSERT(bufEnd - bufStart > sizeof(SimpleRingBufferHeader) + 4); // ヘッダサイズ超過であること

    m_pHeader = new( reinterpret_cast<void*>(bufStart) ) SimpleRingBufferHeader();
    m_pHeader->bufferSize = bufEnd - bufStart - sizeof(SimpleRingBufferHeader);

    m_pHeader->range.start = 0;
    m_pHeader->range.end   = 0;
    m_pHeader->lockObject.Initialize();
}


//---------------------------------------------------------------------------
void
SimpleRingBuffer::Discard()
{
    ScopedLock lock(m_pHeader->lockObject);

    m_pHeader->range.start = 0;
    m_pHeader->range.end   = 0;
}


//---------------------------------------------------------------------------
u32
SimpleRingBuffer::GetReadableBytes()
{
    SimpleRingBufferRange range;

    {
        ScopedLock lock(m_pHeader->lockObject);

        range = m_pHeader->range;
    }

    return ::GetReadableBytes(range, m_pHeader->bufferSize);
}


//---------------------------------------------------------------------------
u32
SimpleRingBuffer::GetWritableBytes()
{
    SimpleRingBufferRange range;

    {
        ScopedLock lock(m_pHeader->lockObject);

        range = m_pHeader->range;
    }

    return ::GetWritableBytes(range, m_pHeader->bufferSize);
}

//---------------------------------------------------------------------------
u32
SimpleRingBuffer::Read(
    void*       buf,
    u32         size
)
{
    SimpleRingBufferRange range;

    BeginRead(&range);

    u32 ret = ContinueRead(&range, buf, size);

    EndRead(range);

    return ret;
}

//---------------------------------------------------------------------------
u32
SimpleRingBuffer::Write(
    const void*     buf,
    u32             size
)
{
    SimpleRingBufferRange range;

    BeginWrite(&range);

    u32 ret = ContinueWrite(&range, buf, size);

    EndWrite(range);

    return ret;
}

//---------------------------------------------------------------------------
u32
SimpleRingBuffer::Skip(
    u32         size
)
{
    SimpleRingBufferRange range;

    BeginSkip(&range);

    u32 ret = ContinueSkip(&range, size);

    EndSkip(range);

    return ret;
}

//---------------------------------------------------------------------------
u32
SimpleRingBuffer::Peek(
    void*       buf,
    u32         size
)
{
    SimpleRingBufferRange range;

    range = m_pHeader->range;
    u32 ret = ContinueRead(&range, buf, size);

    return ret;
}

//---------------------------------------------------------------------------
void
SimpleRingBuffer::BeginRead(SimpleRingBufferRange* pRange)
{
    // データ開始・終了位置を取得
    ScopedLock lock(m_pHeader->lockObject);

    *pRange = m_pHeader->range;
}


//---------------------------------------------------------------------------
u32
SimpleRingBuffer::ContinueRead(
    SimpleRingBufferRange*    pRange,
    void*                     buf,
    u32                       size
)
{
    u32 readableBytes = ::GetReadableBytes(*pRange, m_pHeader->bufferSize);
    if (readableBytes == 0)
    {
        return 0;
    }

    u8* pDst = static_cast<u8*>(buf);
    const u8 *const pSrc = reinterpret_cast<u8*>(m_pHeader) + sizeof(SimpleRingBufferHeader);

    u32 restSize = GetMin(readableBytes, size);

    u32 ret = restSize;

    while (restSize > 0)
    {
        const u32 readByte = GetMin(GetSequenceReadableBytes(*pRange, m_pHeader->bufferSize), restSize);
        (void)std::memcpy(pDst, pSrc + pRange->start, readByte);
        pRange->start = (pRange->start + readByte) % m_pHeader->bufferSize;
        pDst += readByte;
        restSize -= readByte;
    }

    return ret;
}


//---------------------------------------------------------------------------
void
SimpleRingBuffer::EndRead(const SimpleRingBufferRange& range)
{
    // 読み込み開始位置を更新
    ScopedLock lock(m_pHeader->lockObject);

    m_pHeader->range.start = range.start;
}


//---------------------------------------------------------------------------
void
SimpleRingBuffer::BeginWrite(SimpleRingBufferRange* pRange)
{
    // データ開始・終了位置を取得
    ScopedLock lock(m_pHeader->lockObject);

    *pRange = m_pHeader->range;
}

/*
    end ポイントが、buf + bufferSize でないようにすること。このときは、bufを指すようにする。

    書き込むときは、end はstartと同じ値にならないようにすること。常にend は常に startに対して 4バイト以上空きを取ることになる。
*/
u32
SimpleRingBuffer::ContinueWrite(
    SimpleRingBufferRange*    pRange,
    const void*               buf,
    u32                       size
)
{
    u32 writableBytes = ::GetWritableBytes(*pRange, m_pHeader->bufferSize);
    if (writableBytes == 0)
    {
        return 0;
    }

    u8 *const pDst = reinterpret_cast<u8*>(m_pHeader) + sizeof(SimpleRingBufferHeader);
    const u8* pSrc = static_cast<const u8*>(buf);

    u32 restSize = GetMin(writableBytes, size);

    u32 ret = restSize;

    while (restSize > 0)
    {
        const u32 writeByte = GetMin(GetSequenceWritableBytes(*pRange, m_pHeader->bufferSize), restSize);
        (void)std::memcpy(pDst + pRange->end, pSrc, writeByte);
        pRange->end = (pRange->end + writeByte) % m_pHeader->bufferSize;
        pSrc += writeByte;
        restSize -= writeByte;
    }

    return ret;
}

//---------------------------------------------------------------------------
void
SimpleRingBuffer::EndWrite(const SimpleRingBufferRange& range)
{
    // 書き込み開始位置を更新
    ScopedLock lock(m_pHeader->lockObject);

    m_pHeader->range.end = range.end;
}

//---------------------------------------------------------------------------
void
SimpleRingBuffer::BeginSkip(SimpleRingBufferRange* pRange)
{
    // データ開始・終了位置を取得
    ScopedLock lock(m_pHeader->lockObject);

    *pRange = m_pHeader->range;
}


//---------------------------------------------------------------------------
u32
SimpleRingBuffer::ContinueSkip(
    SimpleRingBufferRange*    pRange,
    u32                       size
)
{
    u32 readableBytes = ::GetReadableBytes(*pRange, m_pHeader->bufferSize);
    if (readableBytes == 0)
    {
        return 0;
    }

    u32 restSize = GetMin(readableBytes, size);
    u32 ret = restSize;

    while (restSize > 0)
    {
        const u32 readByte = GetMin(GetSequenceReadableBytes(*pRange, m_pHeader->bufferSize), restSize);
        pRange->start = (pRange->start + readByte) % m_pHeader->bufferSize;
        restSize -= readByte;
    }

    return ret;
}


//---------------------------------------------------------------------------
void
SimpleRingBuffer::EndSkip(const SimpleRingBufferRange& range)
{
    // 読み込み開始位置を更新
    ScopedLock lock(m_pHeader->lockObject);

    m_pHeader->range.start = range.start;
}

}   // namespace fnd
}   // namespace internal
}   // namespace snd
}   // namespace nw
