﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/io/sndfnd_FileStreamImpl.h>

#include <Windows.h>

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

namespace {

//---------------------------------------------------------------------------
DWORD
SeekOriginToMoveMethod(Stream::SeekOrigin origin)
{
    switch(origin)
    {
        case FileStreamImpl::SEEK_ORIGIN_BEGIN:
            return FILE_BEGIN;

        case FileStreamImpl::SEEK_ORIGIN_END:
            return FILE_END;

        case FileStreamImpl::SEEK_ORIGIN_CURRENT:
            return FILE_CURRENT;
    }

    NW_FATAL_ERROR("invalid seek origin.\n");
    return FILE_BEGIN;
}

}

//---------------------------------------------------------------------------
FileStreamImpl::FileStreamImpl() :
m_FileHandle(NULL),
m_FileSize(INVALID_SIZE),
m_CurrentPosition(0)
{
    m_DirectStream.Initialize(*this);
}

//---------------------------------------------------------------------------
FileStreamImpl::FileStreamImpl(void* fileHandle) :
m_FileHandle(fileHandle),
m_FileSize(INVALID_SIZE),
m_CurrentPosition(0)
{
    NW_ASSERT_NOT_NULL(fileHandle);
    m_DirectStream.Initialize(*this);
}

//---------------------------------------------------------------------------
void
FileStreamImpl::Open(void* fileHandle)
{
    NW_ASSERT_NOT_NULL(fileHandle);
    NW_ASSERTMSG(!IsOpened(), "FileStreamImpl is already opened.\n");
    m_FileHandle = fileHandle;
    m_CurrentPosition = 0;
}

//---------------------------------------------------------------------------
void
FileStreamImpl::Close()
{
    if(m_FileHandle == NULL)
    {
        return;
    }

    CloseHandle(m_FileHandle);
    m_FileHandle = NULL;
    m_CurrentPosition = 0;
}

//---------------------------------------------------------------------------
bool
FileStreamImpl::IsOpened() const
{
    return m_FileHandle != NULL;
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::GetSize() const
{
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    if(m_FileSize == INVALID_SIZE)
    {
        DWORD result = GetFileSize(m_FileHandle, NULL);

        if(result == -1)
        {
            return 0;
        }

        m_FileSize = result;
    }

    return m_FileSize;
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::ReadDirect(void* buf, u32 length, FndResult* result /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(buf);
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    FndResult readResult(SNDFND_RESULT_TRUE);
    DWORD readFileLength = 0;

    if(!ReadFile(m_FileHandle, buf, length, &readFileLength, NULL))
    {
        switch(GetLastError())
        {
        case ERROR_ACCESS_DENIED:
            readResult = FndResult(SNDFND_RESULT_IO_INVALID_ACCESS);
            break;

        default:
            readResult = FndResult(SNDFND_RESULT_IO_ERROR);
            break;
       }
    }
    else
    {
        readResult = length == readFileLength ?
            FndResult(SNDFND_RESULT_TRUE) : FndResult(SNDFND_RESULT_FALSE);

        m_CurrentPosition += readFileLength;
    }

    if(result != NULL)
    {
        *result = readResult;
    }

    return readFileLength;
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::WriteDirect(const void* buf, u32 length, FndResult* result /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(buf);
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    FndResult writeResult(SNDFND_RESULT_TRUE);
    DWORD writtenFileLength = 0;

    if(!WriteFile(m_FileHandle, buf, length, &writtenFileLength, NULL))
    {
        switch(GetLastError())
        {
        case ERROR_ACCESS_DENIED:
            writeResult = FndResult(SNDFND_RESULT_IO_INVALID_ACCESS);
            break;

        default:
            writeResult = FndResult(SNDFND_RESULT_IO_ERROR);
            break;
        }
    }
    else
    {
        writeResult = length == writtenFileLength ?
            FndResult(SNDFND_RESULT_TRUE) : FndResult(SNDFND_RESULT_FALSE);

        m_CurrentPosition += writtenFileLength;
    }

    if(result != NULL)
    {
        *result = writeResult;
    }

    return writtenFileLength;
}

//---------------------------------------------------------------------------
FndResult
FileStreamImpl::SeekDirect(s32 offset, SeekOrigin origin)
{
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    DWORD result = SetFilePointer(m_FileHandle, offset, NULL, SeekOriginToMoveMethod(origin));

    if(result == INVALID_SET_FILE_POINTER)
    {
        return FndResult(SNDFND_RESULT_IO_ERROR);
    }

    m_CurrentPosition = result;

    return FndResult(SNDFND_RESULT_TRUE);
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
