﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/io/sndfnd_HioStream.h>

#include <sys/socket.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <cerrno>

#if !defined(NW_PLATFORM_CTR)
#include <nw/ut/ut_String.h>
#endif

#define INVALID_SOCKET -1
#define SOCKET_ERROR -1

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
void
HioStream::Initialize(void* workBuffer, u32 workBufferLength)
{
    m_WorkBuffer = workBuffer;
    m_WorkBufferLength = workBufferLength;

    m_Socket = INVALID_SOCKET;
    m_ClientSocket = INVALID_SOCKET;
}

//---------------------------------------------------------------------------
void
HioStream::Finalize()
{
    m_ClientSocket = INVALID_SOCKET;
    m_Socket = INVALID_SOCKET;

    m_WorkBuffer = NULL;
}

//---------------------------------------------------------------------------
void
HioStream::OpenImpl(ChannelType channel)
{
    m_Socket = socket(AF_INET, SOCK_STREAM, 0);

    if(m_Socket == INVALID_SOCKET)
    {
        return;
    }

    // ノンブロッキングモードに設定。
    unsigned long arg = 1;
    if (SOCKET_ERROR == ioctl(m_Socket, FIONBIO, &arg))
    {
        this->Close();
        NW_LOG("ioctlsocket() failed. (%d)\n", errno);
        return;
    }

    struct sockaddr_in addr;
    addr.sin_family = AF_INET;
    addr.sin_port = htons(static_cast<u16>(channel));
    addr.sin_addr.s_addr = INADDR_ANY;

    if(bind(m_Socket, reinterpret_cast<const struct sockaddr*>(&addr), sizeof(struct sockaddr_in)) == SOCKET_ERROR)
    {
        this->Close();
        return;
    }

    if(listen(m_Socket, 1) == SOCKET_ERROR)
    {
        this->Close();
        return;
    }

    m_IsOpened = true;
}

//---------------------------------------------------------------------------
void
HioStream::CloseImpl()
{
    m_IsOpened = false;

    Disconnect();

    if(m_Socket != INVALID_SOCKET)
    {
        close(m_Socket);
        m_Socket = INVALID_SOCKET;
    }
}

//---------------------------------------------------------------------------
s32
HioStream::WriteImpl(const void* buffer, u32 length)
{
    if(!IsAvailable())
    {
        return 0;
    }

    if(length == 0)
    {
        return 0;
    }

    const char* current = reinterpret_cast<const char*>(buffer);
    u32         writtenLength = 0;
    while(writtenLength < length)
    {
        int sendLength = send(m_ClientSocket, current, length - writtenLength, 0);

        current += sendLength;
        writtenLength += sendLength;
    }

    return writtenLength;
}

//---------------------------------------------------------------------------
bool
HioStream::Connect()
{
    if(!IsOpened())
    {
        NW_ASSERTMSG(false, "HioStream is not opened.");
        return false;
    }

    if(IsAvailable())
    {
        //NW_LOG("[%d is Available] m_ClientSocket = %d\n", this->m_Channel, m_ClientSocket);
        return true;
    }

    struct sockaddr_in client;
    s32 addrlen = sizeof(client);

    m_ClientSocket = accept(m_Socket, reinterpret_cast<struct sockaddr*>(&client), &addrlen);
    //NW_LOG("[%d] m_ClientSocket = %d\n", this->m_Channel, m_ClientSocket);

    if(m_ClientSocket == INVALID_SOCKET)
    {
        Disconnect();
        return false;
    }

    return true;
}

//---------------------------------------------------------------------------
void
HioStream::Disconnect()
{
    if(m_ClientSocket != INVALID_SOCKET)
    {
        close(m_ClientSocket);
        m_ClientSocket = INVALID_SOCKET;
    }
}

//---------------------------------------------------------------------------
bool
HioStream::IsAvailable() const
{
    return IsOpened() && m_ClientSocket != INVALID_SOCKET;
}

//---------------------------------------------------------------------------
void
HioStream::ReadBuffer()
{
    if (!IsAvailable())
    {
        return;
    }

    char* buffer = reinterpret_cast<char*>(m_WorkBuffer);
    int result = recv(m_ClientSocket, buffer, m_WorkBufferLength, 0);

    if (result == SOCKET_ERROR)
    {
        int errCode = errno;
        if (errCode != EWOULDBLOCK || errCode != EAGAIN )
        {
            Disconnect();
        }

        return;
    }

    // 通信相手がコネクションを閉じると0を返すため、この場合は切断だと判断する
    if (result == 0)
    {
        Disconnect();
        return;
    }

    m_RingBuffer.Write(m_WorkBuffer, result);
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
