﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

 #include <nw/lyt/lyt_TextBox.h>

#include <nw/lyt/lyt_DrawInfo.h>
#include <nw/lyt/lyt_GraphicsResource.h>
#include <nw/lyt/lyt_Layout.h>
#include <nw/lyt/lyt_Material.h>
#include <nw/lyt/lyt_Animation.h>
#include <nw/lyt/lyt_Common.h>
#include <nw/lyt/lyt_ResourceAccessor.h>
#include <nw/font/font_DispStringBuffer.h>
#include <nw/lyt/lyt_Util.h>

#include <nw/dev/dev_Profile.h>

namespace nw
{
namespace lyt
{
namespace internal
{
namespace
{

/*!--------------------------------------------------------------------------*
  @brief        基点が中央であるときのカーソル位置の値を求めます。

  @param[in]    value   考慮する幅/高さ
  @param[in]    isCeil  半分の値が浮動小数点にならないように整数値に
                        切り上げる場合は真。

  @return       基点が中央であるときのカーソル位置補正の値を返します。
 *---------------------------------------------------------------------------*/
NW_INLINE
f32
AdjustCenterValue(
    f32 value,
    bool isCeil
)
{
    f32 ret = value / 2;
    return isCeil ? math::FCeil(ret): ret;
}

} // namespace nw::lyt::internal::{anonymous}

} // namespace nw::lyt::internal

using namespace math;

TextBox::TextBox()
{
    Init();
    InitMaterial();
}

TextBox::TextBox(
    const res::TextBox* pBaseBlock,
    const res::TextBox* pOverrideBlock,
    const BuildArgSet& buildArgSet,
    InitStringParam* initStringParam
)
: Pane(pBaseBlock, buildArgSet)
{
    const res::TextBox* pBlock;
    const res::TextBox* pBlockForText;
    const BuildResSet* buildResSet;
    const BuildResSet* buildResSetForText;
    if (pOverrideBlock)
    {
        if (buildArgSet.overrideUsageFlag == 0 && buildArgSet.overrideMaterialUsageFlag == 0)
        {
            // 部分上書きがない、丸ごと上書き
            pBlock = pOverrideBlock;
            buildResSet = buildArgSet.pOverrideBuildResSet;
            pBlockForText = pOverrideBlock;
            buildResSetForText = buildArgSet.pOverrideBuildResSet;
        }
        else
        {
            pBlock = pBaseBlock;
            buildResSet = buildArgSet.pCurrentBuildResSet;
            if (internal::TestBit(buildArgSet.overrideUsageFlag, TEXTBOXOVERRIDEUSAGEFLAG_TEXT_ENABLED))
            {
                // テキストを部分的に上書きする
                pBlockForText = pOverrideBlock;
                buildResSetForText = buildArgSet.pOverrideBuildResSet;
            }
            else
            {
                pBlockForText = pBaseBlock;
                buildResSetForText = buildArgSet.pCurrentBuildResSet;
            }
        }
    }
    else
    {
        // 上書きがない場合
        pBlock = pBaseBlock;
        pBlockForText = pBaseBlock;
        buildResSet = buildArgSet.pCurrentBuildResSet;
        buildResSetForText = buildArgSet.pCurrentBuildResSet;
    }

    Init();

    bool shadowEnabled = pBlock->textBoxFlag & (1 << TEXTBOXFLAG_SHADOW_ENABLED);

    u32 allocStrBufLen =
        static_cast<u32>(pBlock->textBufBytes / sizeof(char16));

    if (allocStrBufLen > 0)
    {
        allocStrBufLen -= 1;
    }

    u32 resStrLen =
        static_cast<u32>(pBlockForText->textStrBytes / sizeof(char16));

    if (resStrLen > 0)
    {
        resStrLen -= 1;
    }

    for (int i = 0; i < TEXTCOLOR_MAX; ++i)
    {
        m_TextColors[i] = pBlock->textCols[i];
    }

    m_FontSize.width = pBlock->fontSize.x;
    m_FontSize.height = pBlock->fontSize.y;
    m_TextPosition = pBlock->textPosition;
    m_Bits.textAlignment = pBlock->textAlignment;
    m_CharSpace = pBlock->charSpace;
    m_LineSpace = pBlock->lineSpace;
    m_ItalicRatio = pBlock->italicRatio;

    m_Bits.shadowEnabled = shadowEnabled;
    m_ShadowOffset = pBlock->shadowOffset;
    m_ShadowScale = pBlock->shadowScale;
    m_ShadowTopColor = pBlock->shadowCols[0];
    m_ShadowBottomColor = pBlock->shadowCols[1];
    m_ShadowItalicRatio = pBlock->shadowItalicRatio;

    m_Bits.invisibleBorderEnabled = (pBlock->textBoxFlag & (1 << TEXTBOXFLAG_INVISIBLE_BORDER_ENABLED)) != 0;
    m_Bits.doubleDrawnBorderEnabled = (pBlock->textBoxFlag & (1 << TEXTBOXFLAG_DOUBLE_DRAWN_BORDER_ENABLED)) != 0;
    m_Bits.perCharacterTransformEnabled = (pBlock->textBoxFlag & (1 << TEXTBOXFLAG_PER_CHARACTER_TRANSFORM_ENABLED)) != 0;
    m_Bits.widthLimitEnabled = true;

    if (m_Bits.perCharacterTransformEnabled && pBlock->perCharacterTransformOffset != 0)
    {
        const res::PerCharacterTransform* perCharacterTransform = internal::ConvertOffsToPtr<res::PerCharacterTransform>(pBlock, pBlock->perCharacterTransformOffset);
        const res::AnimationInfo* animInfo = NULL;
        u32 allocSize = sizeof(PerCharacterTransform);
        if (perCharacterTransform->hasAnimationInfo != 0) {
            animInfo = internal::ConvertOffsToPtr<res::AnimationInfo>(perCharacterTransform, sizeof(res::PerCharacterTransform));
            if (animInfo->num > 0) {
                allocSize += sizeof(CurveInfo) * (animInfo->num - 1);
            }
        }
        m_pPerCharacterTransform = static_cast<PerCharacterTransform*>(Layout::AllocMemory(allocSize));
        m_pPerCharacterTransform->Offset = perCharacterTransform->evalTimeOffset;
        m_pPerCharacterTransform->Width = perCharacterTransform->evalTimeWidth;
        m_pPerCharacterTransform->pPerCharacterTransformInfos = NULL;
        m_pPerCharacterTransform->LoopType = perCharacterTransform->loopType;
        m_pPerCharacterTransform->OriginV = perCharacterTransform->originV;
        if (animInfo)
        {
            m_pPerCharacterTransform->CurveNumber = animInfo->num;
            InitPerCharacterTransformCurves(animInfo);
        }
        else
        {
            m_pPerCharacterTransform->CurveNumber = 0;
        }
    }

    // Fontの構築
    NW_ASSERT_NOT_NULL(buildResSet->pFontList);
    NW_ASSERT(pBlock->fontIdx < buildResSet->pFontList->fontNum);
    const res::Font *const fonts = internal::ConvertOffsToPtr<res::Font>(buildResSet->pFontList, sizeof(*buildResSet->pFontList));
    const char *const fontName = internal::ConvertOffsToPtr<char>(fonts, fonts[pBlock->fontIdx].nameStrOffset);

    m_pFont = buildResSet->pResAccessor->GetFont(fontName);

    // マテリアルの作成
    {
        const res::Material *const pResMaterial = nw::lyt::internal::GetResMaterial(buildArgSet.pCurrentBuildResSet, pBaseBlock->materialIdx);
        const res::Material *pOverrideResMaterial = NULL;
        if (pOverrideBlock) {
            pOverrideResMaterial = nw::lyt::internal::GetResMaterial(buildArgSet.pOverrideBuildResSet, pOverrideBlock->materialIdx);
        }
        m_pMaterial = Layout::NewObj<Material, const res::Material*, const res::Material*, const BuildArgSet&>(pResMaterial, pOverrideResMaterial, buildArgSet);
    }

    // テキストIDの設定
    if (pBlockForText->textIDOffset != 0)
    {
        // Initメソッドの中でNULLに初期化されているので、このif文の中に来ない場合はNULLになる
        m_TextID = internal::ConvertOffsToPtr<char>(pBlockForText, pBlockForText->textIDOffset);
    }

    initStringParam->textBoxFlag = pBlock->textBoxFlag;
    initStringParam->resourceLayout = buildResSetForText->pLayout;
    initStringParam->pBlockText = internal::ConvertOffsToPtr<char16>(pBlockForText, pBlockForText->textStrOffset);
    initStringParam->allocStrBufLen = allocStrBufLen;
    initStringParam->resStrLen = resStrLen;
}

void TextBox::InitString(const BuildArgSet& buildArgSet, const InitStringParam& initStringParam)
{
    bool isTextSetByTextSearcher = false; // TextSearcherによってテキストが設定されたか否か
    if (buildArgSet.pTextSearcher)
    {
        TextSearcher::TextInfo textInfo;
        // 「文字列の長さを強制的に指定する」のチェックが入っている場合、TextInfoに設定する
        if (initStringParam.textBoxFlag & (1 << TEXTBOXFLAG_FORCE_ASSIGN_TEXT_LENGTH))
        {
            textInfo.forceAssignedTextLength = initStringParam.allocStrBufLen;
        }

        // テキストIDから設定するテキストの情報を取得する
        buildArgSet.pTextSearcher->SearchText(&textInfo, m_TextID, buildArgSet.pLayout, this, initStringParam.resourceLayout);

        NW_ASSERTMSG(textInfo.textLength <= 0xffff, "text length[%d] must be smaller or equal than 0xffff for TextBox[%s]", textInfo.textLength, GetName());
        NW_ASSERTMSG(textInfo.bufferLength <= 0xffff, "buffer length[%d] must be smaller or equal than 0xffff for TextBox[%s]", textInfo.bufferLength, GetName());

        isTextSetByTextSearcher = InitStringWithTextSearcherInfo(buildArgSet, textInfo);
    }

    // TextSearcherによって設定されなかった場合、バッファを確保する
    if ( ! isTextSetByTextSearcher) {
        // レイアウトエディタで指定された値とテキストの長さの大きい方を取る
        AllocStringBuffer(static_cast<u16>(ut::Max(initStringParam.allocStrBufLen, initStringParam.resStrLen)), buildArgSet.isShareTextBoxCharAttributeBuffer);
    }

    // ここではAllocStringBufferでDispStringBufferにCharCountMaxが設定されているので、BodyLayoutに設定されている
    // 値より大きければ更新する
    if (buildArgSet.isShareTextBoxCharAttributeBuffer && m_pDispStringBuf && buildArgSet.pBodyLayout->GetMaxCharAttributeNum() < m_pDispStringBuf->GetCharCountMax())
    {
        buildArgSet.pBodyLayout->SetMaxCharAttributeNum(m_pDispStringBuf->GetCharCountMax());
    }

    // TextSearcherで文字列が設定されなかった場合、初期文字列のコピー
    if (initStringParam.resStrLen >= 1 && m_TextBuf && ! isTextSetByTextSearcher)
    {
        const char16 *const pBlockText = initStringParam.pBlockText;
#if defined( NW_UT_RES_SWAP_ENDIAN )
        // リソースのエンディアンが逆転している場合は、スワップしながら文字列を設定しなければならない。SetStringを使わずに特別処理する。
        m_TextLen = ut::Min(static_cast<u16>(initStringParam.resStrLen), GetStringBufferLength());
        for (int i = 0; i < m_TextLen; i++)
        {
            m_TextBuf[i] = ut::EndianSwap::BSwap(static_cast<u16>(pBlockText[i]));
        }
        m_TextBuf[m_TextLen] = 0;
        UpdatePTDirty(true);
#else
        SetString(pBlockText, 0, initStringParam.resStrLen);
#endif
    }
}

bool TextBox::InitStringWithTextSearcherInfo(const BuildArgSet& buildArgSet, const TextSearcher::TextInfo& textInfo)
{
    if (textInfo.text) {
        //---- テキストが指定されたとき
        // テキストの長さを取得する、0が指定された場合は指定されたテキストの長さを数える
        u16 textLength = static_cast<u16>(textInfo.textLength);
        if (textLength == 0) {
            textLength = static_cast<u16>(font::CalcWcs16Length(textInfo.text));
        }
        if (textInfo.bufferLength == 0) {
            // バッファの長さの指定がないときは、テキストの長さで確保する
            AllocStringBuffer(textLength, buildArgSet.isShareTextBoxCharAttributeBuffer);
        } else {
            // バッファの長さが指定されたらそれを使用する
            AllocStringBuffer(static_cast<u16>(textInfo.bufferLength), buildArgSet.isShareTextBoxCharAttributeBuffer);
        }
        if (GetStringBuffer()) {
            // AllocStringBufferの結果バッファが設定されたら文字列をセット。
            // バッファが設定されないのは、textInfo.textに空文字が設定され、textLengthとbufferLengthが両方0の場合のみ
            SetString(textInfo.text, 0, textLength);
        }
        // trueを返すと、レイアウトエディタで指定した文字列は設定されない
        return true;
    } else {
        return false;
    }
}

void TextBox::InitPerCharacterTransformCurves(const res::AnimationInfo* animInfo)
{
    const ut::ResU32* animTargetOffsets = internal::ConvertOffsToPtr<ut::ResU32>(animInfo, sizeof(*animInfo));
    for (s32 k = 0; k < animInfo->num; k++)
    {
        const res::AnimationTarget* pAnimTarget = internal::ConvertOffsToPtr<res::AnimationTarget>(animInfo, animTargetOffsets[k]);
        m_pPerCharacterTransform->CurveInfos[k].Curve = pAnimTarget;
        m_pPerCharacterTransform->CurveInfos[k].CurveType = static_cast<AnimTargetPerCharacterTransformCurve>(pAnimTarget->target);
    }
}

TextBox::TextBox(const TextBox& textBox)
 : Base(textBox)
 , m_TextBuf(NULL)
 , m_TextID(textBox.m_TextID)
 , m_pFont(textBox.m_pFont)
 , m_FontSize(textBox.m_FontSize)
 , m_LineSpace(textBox.m_LineSpace)
 , m_CharSpace(textBox.m_CharSpace)
 , m_pTagProcessor(textBox.m_pTagProcessor)
 , m_TextBufLen(0)
 , m_TextLen(0)
 , m_Bits(textBox.m_Bits)
 , m_TextPosition(textBox.m_TextPosition)
 , m_ItalicRatio(textBox.m_ItalicRatio)
 , m_ShadowOffset(textBox.m_ShadowOffset)
 , m_ShadowScale(textBox.m_ShadowScale)
 , m_ShadowTopColor(textBox.m_ShadowTopColor)
 , m_ShadowBottomColor(textBox.m_ShadowBottomColor)
 , m_ShadowItalicRatio(textBox.m_ShadowItalicRatio)
 , m_pMaterial(NULL)
 , m_pDispStringBuf(NULL)
 , m_pPerCharacterTransform(NULL)
{
    for (int i = 0; i < TEXTCOLOR_MAX; ++i)
    {
        m_TextColors[i] = textBox.m_TextColors[i];
    }

    if (textBox.m_pPerCharacterTransform != NULL)
    {
        m_pPerCharacterTransform = static_cast<PerCharacterTransform*>(Layout::AllocMemory(sizeof(PerCharacterTransform)));
        *m_pPerCharacterTransform = *textBox.m_pPerCharacterTransform;
        m_pPerCharacterTransform->pPerCharacterTransformInfos = NULL;
    }

    if (textBox.GetStringBufferLength() > 0)
    {
        // 文字列バッファを確保
        AllocStringBuffer(textBox.GetStringBufferLength(), textBox.GetDrawStringBufferLength(), textBox.IsShareCharAttributeBuffer());
        // 文字をセット
        SetString(textBox.GetString(), 0, textBox.GetStringLength());
    }
    // マテリアルの複製
    m_pMaterial = Layout::NewObj<Material, const Material&>(*textBox.m_pMaterial);
}

TextBox::TextBox(const TextBox& textBox, u16 bufferStrlen)
 : Base(textBox)
 , m_TextBuf(NULL)
 , m_TextID(textBox.m_TextID)
 , m_pFont(textBox.m_pFont)
 , m_FontSize(textBox.m_FontSize)
 , m_LineSpace(textBox.m_LineSpace)
 , m_CharSpace(textBox.m_CharSpace)
 , m_pTagProcessor(textBox.m_pTagProcessor)
 , m_TextBufLen(0)
 , m_TextLen(0)
 , m_Bits(textBox.m_Bits)
 , m_TextPosition(textBox.m_TextPosition)
 , m_ItalicRatio(textBox.m_ItalicRatio)
 , m_ShadowOffset(textBox.m_ShadowOffset)
 , m_ShadowScale(textBox.m_ShadowScale)
 , m_ShadowTopColor(textBox.m_ShadowTopColor)
 , m_ShadowBottomColor(textBox.m_ShadowBottomColor)
 , m_ShadowItalicRatio(textBox.m_ShadowItalicRatio)
 , m_pMaterial(NULL)
 , m_pDispStringBuf(NULL)
 , m_pPerCharacterTransform(NULL)
{
    for (int i = 0; i < TEXTCOLOR_MAX; ++i)
    {
        m_TextColors[i] = textBox.m_TextColors[i];
    }

    if (textBox.m_pPerCharacterTransform != NULL)
    {
        m_pPerCharacterTransform = static_cast<PerCharacterTransform*>(Layout::AllocMemory(sizeof(PerCharacterTransform)));
        *m_pPerCharacterTransform = *textBox.m_pPerCharacterTransform;
        m_pPerCharacterTransform->pPerCharacterTransformInfos = NULL;
    }

    if (bufferStrlen > 0)
    {
        // コピー元が文字属性バッファを共有する設定であり、かつ指定された文字列バッファサイズがコピー元以下である場合は
        // 文字属性バッファを共有する設定にする。二つ目の条件が必要であるのは、レイアウトが保持している共有のバッファが
        // 不足する恐れがあるため。
        bool isShareCharAttrs = (textBox.IsShareCharAttributeBuffer() && bufferStrlen <= textBox.GetStringBufferLength());
        // 文字列バッファを確保、文字列のセットは行わない
        AllocStringBuffer(bufferStrlen, isShareCharAttrs);
    }
    // マテリアルの複製
    m_pMaterial = Layout::NewObj<Material, const Material&>(*textBox.m_pMaterial);
}

void
TextBox::Init()
{
    m_TextBuf = 0;
    m_TextID = 0;
    m_TextBufLen = 0;
    m_TextLen = 0;
    m_pFont = 0;
    m_FontSize = Size(0, 0);
    std::memset(&m_Bits, 0, sizeof(m_Bits));
    m_TextPosition = 0;
    SetTextPositionH(HORIZONTALPOSITION_CENTER);
    SetTextPositionV(VERTICALPOSITION_CENTER);
    m_LineSpace = 0;
    m_CharSpace = 0;
    m_pTagProcessor = 0;
    m_pDispStringBuf = 0;
    m_ItalicRatio = 0.0f;
    m_ShadowOffset.Set(0.0f, 0.0f);
    m_ShadowScale.Set(1.0f, 1.0f);
    m_ShadowTopColor.Set(0, 0, 0, 255);
    m_ShadowBottomColor.Set(0, 0, 0, 255);
    m_ShadowItalicRatio = 0.0f;
    m_pPerCharacterTransform = NULL;
}

void
TextBox::InitMaterial()
{
    // マテリアルの作成
    m_pMaterial = Layout::NewObj<Material>();
    if (m_pMaterial)
    {
        m_pMaterial->ReserveMem(0, 0, 0);
    }
}

TextBox::~TextBox()
{
    // OSReport("TextBox::~TextBox()\n");

    if (m_pMaterial && !m_pMaterial->IsUserAllocated())
    {
        Layout::DeleteObj(m_pMaterial);
        m_pMaterial = 0;
    }

    FreeStringBuffer();

    if (m_pPerCharacterTransform != NULL)
    {
        Layout::FreeMemory(m_pPerCharacterTransform);
        m_pPerCharacterTransform = NULL;
    }
}

u8
TextBox::GetMaterialNum() const
{
    return static_cast<u8>(m_pMaterial ? 1 : 0);
}

Material*
TextBox::GetMaterial(u32 idx) const
{
    NW_WARNING(idx < GetMaterialNum(), "idx >= GetMaterialNum() : %d >= %d", idx, GetMaterialNum());

    return (idx == 0) ? m_pMaterial : NULL;
}

void TextBox::SetMaterial(Material* pMaterial)
{
    if (m_pMaterial && !m_pMaterial->IsUserAllocated())
    {
        Layout::DeleteObj(m_pMaterial);
    }
    m_pMaterial = pMaterial;
}

void TextBox::UnsetMaterial()
{
    m_pMaterial = NULL;
}

const ut::Color8
TextBox::GetVtxColor(u32 idx) const
{
    NW_ASSERTMSG(idx < VERTEXCOLOR_MAX, "out of bounds: idx[%u] < VERTEXCOLOR_MAX for TextBox[%s]", idx, GetName());

    return GetTextColor(idx / 2);
}

void
TextBox::SetVtxColor(
    u32         idx,
    ut::Color8 value
)
{
    NW_ASSERTMSG(idx < VERTEXCOLOR_MAX, "out of bounds: idx[%u] < VERTEXCOLOR_MAX for TextBox[%s]", idx, GetName());

    SetTextColor(idx / 2, value);
}

u8
TextBox::GetVtxColorElement(u32 idx) const
{
    NW_ASSERTMSG(idx < ANIMTARGET_VERTEXCOLOR_MAX, "out of bounds: idx[%u] < ANIMTARGET_VERTEXCOLOR_MAX for TextBox[%s]", idx, GetName());

    return reinterpret_cast<const u8*>(&m_TextColors[idx / (2 * sizeof(ut::Color8))])[idx % sizeof(ut::Color8)];
}

void
TextBox::SetVtxColorElement(u32 idx, u8 value)
{
    NW_ASSERTMSG(idx < ANIMTARGET_VERTEXCOLOR_MAX, "out of bounds: idx[%u] < ANIMTARGET_VERTEXCOLOR_MAX for TextBox[%s]", idx, GetName());

    u8& elm =
        reinterpret_cast<u8*>(&m_TextColors[idx / (2 * sizeof(ut::Color8))])[idx % sizeof(ut::Color8)];

    UpdatePTDirty(elm != value);
    elm = value;
}

const ut::Rect
TextBox::GetTextDrawRect() const
{
    if (m_pFont == NULL)
    {
        NW_WARNING(false, "m_pFont is NULL");
        return ut::Rect();
    }

    font::WideTextWriter writer;
    writer.SetCursor(0, 0);
    SetFontInfo(&writer);

    ut::Rect textRect;
    writer.CalcStringRect(&textRect, m_TextBuf, m_TextLen);

    const Size textSize(textRect.GetWidth(), textRect.GetHeight());

    VEC2 ltPos = GetVtxPos();

    const VEC2 curPos  = AdjustTextPos(GetSize(), false);
    const VEC2 textPos = AdjustTextPos(textSize, true);

    ltPos.x += curPos.x - textPos.x;
    ltPos.y -= curPos.y - textPos.y;

    textRect.left   = ltPos.x;
    textRect.top    = ltPos.y;
    textRect.right  = ltPos.x + textSize.width;
    textRect.bottom = ltPos.y - textSize.height;

    return textRect;
}

void
TextBox::CalculateMtx(Pane::CalculateMtxContext& context, bool isDirtyParentMtx)
{
    Pane::CalculateMtx(context, isDirtyParentMtx);

    if (m_TextLen <= 0 || !m_pFont || !m_pMaterial)
    {
        return;
    }

    if (m_Bits.isPTDirty) {
        m_pDispStringBuf->ClearCommand();
        m_Bits.isPTDirty = 0;
    }

    if (!this->m_pDispStringBuf->IsGeneratedCommand())
    {
        if (m_pDispStringBuf->IsUseSharedCharAttrs())
        {
            if (context.pLayout->GetMaxCharAttributeNum() < m_pDispStringBuf->GetCharCountMax())
            {
                NW_ERR("CharCountMax[%d] exceeds the shared buffer size", m_pDispStringBuf->GetCharCountMax());
                return;
            }
            m_pDispStringBuf->SetCharAttributes(context.pLayout->GetSharedCharAttrs());
        }

        // 描画文字列バッファの作成
        font::WideTextWriter writer;
        SetupTextWriter(&writer);
        writer.SetDispStringBuffer(this->m_pDispStringBuf);

        writer.StartPrint();
        writer.Print(m_TextBuf, m_TextLen);
        writer.EndPrint();

        if (m_Bits.perCharacterTransformEnabled)
        {
            UpdatePerCharacterTransform();
        }

        // バーテックスバッファ作成
        if (m_Bits.shadowEnabled)
        {
            font::ShadowParameter param;
            param.shadowUpperColor = m_ShadowTopColor;
            param.shadowLowerColor = m_ShadowBottomColor;
            param.shadowOffset = m_ShadowOffset;
            param.shadowScale = m_ShadowScale;
            param.shadowItalicOffset = m_ShadowItalicRatio * m_FontSize.width;
            if (m_Bits.perCharacterTransformEnabled)
            {
                context.rectDrawer->BuildVertexElements(this->m_pDispStringBuf, &param, m_pPerCharacterTransform->pPerCharacterTransformInfos, m_pPerCharacterTransform->OriginV);
            }
            else
            {
                context.rectDrawer->BuildVertexElements(this->m_pDispStringBuf, &param);
            }
        }
        else
        {
            if (m_Bits.perCharacterTransformEnabled)
            {
                context.rectDrawer->BuildVertexElements(this->m_pDispStringBuf, NULL, m_pPerCharacterTransform->pPerCharacterTransformInfos, m_pPerCharacterTransform->OriginV);
            }
            else
            {
                context.rectDrawer->BuildVertexElements(this->m_pDispStringBuf, NULL);
            }
        }
    }
}

void
TextBox::DrawSelf(DrawInfo& drawInfo)
{
    NW_PROFILE("nw::lyt::TextBox::DrawSelf");

    if (m_TextLen <= 0 || !m_pFont || !m_pMaterial)
    {
        return;
    }

    // 描画内容設定
    MTX34 mtx;
    GetTextGlobalMtx(&mtx);

    font::DrawContent drawContent;
    drawContent.dispStringBuffer = this->m_pDispStringBuf;
    drawContent.projectionMatrix = &drawInfo.GetProjectionMtx();
    drawContent.viewMatrix = &mtx;
    drawContent.localMatrix = &nw::math::MTX34::Identity();

    if (m_Bits.invisibleBorderEnabled)
    {
        drawContent.flags |= font::DrawContent::INVISIBLE_BORDER;
    }

    {
        drawContent.SetInterpolateBlackWithAlpha(m_pMaterial->GetColor(INTERPOLATECOLOR_BLACK), GetGlobalAlpha());
        drawContent.SetInterpolateWhiteWithAlpha(m_pMaterial->GetColor(INTERPOLATECOLOR_WHITE), GetGlobalAlpha());
    }
    if (m_pMaterial->IsFontShadowParameterCap())
    {
        const res::FontShadowParameter& color = m_pMaterial->GetFontShadowParameter();
        drawContent.SetShadowInterpolateBlackWithAlpha(ut::Color4u8(
            color.blackInterporateColor[0],
            color.blackInterporateColor[1],
            color.blackInterporateColor[2],
            0),
            GetGlobalAlpha());
        drawContent.SetShadowInterpolateWhiteWithAlpha(ut::Color4u8(
            color.whiteInterporateColor[0],
            color.whiteInterporateColor[1],
            color.whiteInterporateColor[2],
            color.whiteInterporateColor[3]),
            GetGlobalAlpha());
    }
    else
    {
        float alpha = static_cast<float>(GetGlobalAlpha()) / ut::Color4u8::ALPHA_MAX;
        drawContent.shadowInterpolateBlack.Set(0.f, 0.f, 0.f, alpha);
        drawContent.shadowInterpolateWhite.Set(1.f, 1.f, 1.f, alpha);
    }

    m_pMaterial->SetupSubmaterialOf_Blender();
    font::RectDrawer& drawer = drawInfo.GetGraphicsResource()->GetFontDrawer();
    if (m_Bits.doubleDrawnBorderEnabled)
    {
        DrawSelfBorder(drawer, drawContent);
        // 次の描画をフチなしにする
        drawContent.flags |= font::DrawContent::INVISIBLE_BORDER;
    }
    drawer.Draw(drawContent);
    drawInfo.ResetGlState();
}

void
TextBox::DrawSelfBorder(font::RectDrawer& drawer, const font::DrawContent& drawContent)
{
    // DrawContentのコピーを作る
    font::DrawContent borderDrawContent = drawContent;

    // 白補間のカラーを黒補間と同じにする(文字色をフチ色と同じにする)
    borderDrawContent.interpolateWhite.r = drawContent.interpolateBlack.r;
    borderDrawContent.interpolateWhite.g = drawContent.interpolateBlack.g;
    borderDrawContent.interpolateWhite.b = drawContent.interpolateBlack.b;
    borderDrawContent.shadowInterpolateWhite.r = drawContent.shadowInterpolateBlack.r;
    borderDrawContent.shadowInterpolateWhite.g = drawContent.shadowInterpolateBlack.g;
    borderDrawContent.shadowInterpolateWhite.b = drawContent.shadowInterpolateBlack.b;

    // フチなし設定は解除
    borderDrawContent.flags &= ~static_cast<u32>(font::DrawContent::INVISIBLE_BORDER);

    drawer.Draw(borderDrawContent);
}

u16
TextBox::GetStringBufferLength() const
{
    if (m_TextBufLen == 0)
    {
        return 0;
    }

    return static_cast<u16>(m_TextBufLen - 1);
}

u16
TextBox::GetDrawStringBufferLength() const
{
    if (m_pDispStringBuf)
    {
        return static_cast<u16>(m_pDispStringBuf->GetCharCountMax());
    }
    else
    {
        return 0;
    }
}

void
TextBox::AllocStringBuffer(u16 minLen, bool isShareCharAttributeBuffer /* = false */)
{
    AllocStringBuffer(minLen, minLen, isShareCharAttributeBuffer);
}

void
TextBox::AllocStringBuffer(u16 minStrLen, u16 minDrawStrLen, bool isShareCharAttributeBuffer /* = false */)
{
    if (minStrLen == 0 || minDrawStrLen == 0)
    {
        return;
    }

    u32 textBufLen = minStrLen;
    u32 dispStringCharAttrLen = minDrawStrLen;
    u32 dispStringVertexBufLen = minDrawStrLen;
    if (m_Bits.shadowEnabled)
    {
        dispStringVertexBufLen *= 2;
    }
    ++textBufLen; // 終端文字分インクリメント

    // 要求する長さのバッファを既に確保している場合は何もしない
    if (textBufLen <= m_TextBufLen && (m_pDispStringBuf != NULL && dispStringCharAttrLen <= m_pDispStringBuf->GetCharCountMax()))
    {
        return;
    }

    FreeStringBuffer();

    // 指定した文字数分VBOバッファを確保
    const u32 drawBufSize = isShareCharAttributeBuffer ?
        font::CharWriter::GetDispStringBufferSizeWithSharedCharAttrs() : font::CharWriter::GetDispStringBufferSize(dispStringCharAttrLen);
    const u32 gfxBufSize = font::CharWriter::GetGraphicsBufferSize(dispStringVertexBufLen);

    char16* textBuf     = Layout::NewArray<char16>(textBufLen);
    void* pDispStringBuf = Layout::AllocMemory(drawBufSize);
    void* pFontGraphicsBuf = Layout::AllocMemory(gfxBufSize, font::CharWriter::GRAPHICS_BUFFER_ALIGNMENT);
    if (NULL != m_pPerCharacterTransform)
    {
        m_pPerCharacterTransform->pPerCharacterTransformInfos = Layout::NewArray<nw::font::PerCharacterTransformInfo>(minDrawStrLen);
    }
    if (NULL == textBuf || NULL == pDispStringBuf || NULL == pFontGraphicsBuf || (NULL != m_pPerCharacterTransform && NULL == m_pPerCharacterTransform->pPerCharacterTransformInfos))
    {
        if (NULL != textBuf)
        {
            Layout::DeletePrimArray(textBuf);
        }
        if (NULL != pDispStringBuf)
        {
            Layout::FreeMemory(pDispStringBuf);
        }
        if (NULL != pFontGraphicsBuf)
        {
            Layout::FreeMemory(pFontGraphicsBuf);
        }
        if (NULL != m_pPerCharacterTransform && NULL != m_pPerCharacterTransform->pPerCharacterTransformInfos)
        {
            Layout::DeletePrimArray(m_pPerCharacterTransform->pPerCharacterTransformInfos);
            m_pPerCharacterTransform->pPerCharacterTransformInfos = NULL;
        }
        return;
    }

    m_TextBuf = textBuf;
    m_TextBufLen = static_cast<u16>(textBufLen);

    // 表示文字列バッファを初期化
    if (isShareCharAttributeBuffer) {
        m_pDispStringBuf = font::CharWriter::InitDispStringBufferWithSharedCharAttrs(
            pDispStringBuf,
            pFontGraphicsBuf,
            dispStringCharAttrLen,
            m_Bits.shadowEnabled);
    } else {
        m_pDispStringBuf = font::CharWriter::InitDispStringBuffer(
            pDispStringBuf,
            pFontGraphicsBuf,
            dispStringCharAttrLen,
            m_Bits.shadowEnabled);
    }
}

void
TextBox::FreeStringBuffer()
{
    if (m_TextBuf)
    {
        Layout::FreeMemory(m_pDispStringBuf->GetGraphicsBuffer());
        Layout::FreeMemory(m_pDispStringBuf);
        Layout::DeletePrimArray(m_TextBuf);
        m_pDispStringBuf = 0;
        m_TextBuf = 0;
        m_TextBufLen = 0;
        m_TextLen = 0;
    }
    if (m_pPerCharacterTransform)
    {
        if (m_pPerCharacterTransform->pPerCharacterTransformInfos != NULL)
        {
            Layout::DeletePrimArray(m_pPerCharacterTransform->pPerCharacterTransformInfos);
            m_pPerCharacterTransform->pPerCharacterTransformInfos = NULL;
        }
    }
}

u16
TextBox::SetString(
    const char16* str,
    u16 dstIdx
)
{
    return SetStringImpl(str, dstIdx, font::CalcWcs16Length(str));
}

u16
TextBox::SetString(
    const char16* str,
    u16 dstIdx,
    u16 strLen
)
{
    return SetStringImpl(str, dstIdx, strLen);
}

u16
TextBox::SetStringImpl(
    const char16*  str,
    u16             dstIdx,
    u32             strLen
)
{
    if (m_pFont == 0)
    {
        NW_WARNING(false, "m_pFont is NULL.\n");
        return 0;
    }

    if (m_TextBuf == 0)
    {
        NW_WARNING(false, "m_TextBuf is NULL.\n");
        return 0;
    }

    const u16 bufLen = GetStringBufferLength();

    if (dstIdx >= bufLen)   // バッファを超えているためコピーしない
    {
        NW_WARNING(false, "dstIdx is out of range.\n");
        return 0;
    }

    u32 cpLen = bufLen;
    cpLen -= dstIdx;        // コピー可能な文字数

    cpLen = ut::Min(strLen, cpLen);
    NW_WARNING(cpLen >= strLen, "%d character(s) droped.\n", strLen - cpLen);

    ut::MemCpy(m_TextBuf + dstIdx, str, cpLen * sizeof(char16));

    // m_TextLen にセットする値は 16bitの bufLen 以下の値。
    if (m_TextBuf[0] == 0) {
        // ライブラリユーザが指定したstrLenが誤っていた場合、0か否かは致命的なので、
        // strに空文字列が指定された場合だけは正しい値(0)をm_TextLenに入れる
        m_TextLen = 0;
    } else {
        m_TextLen = static_cast<u16>(dstIdx + cpLen);
    }
    m_TextBuf[m_TextLen] = 0;

    UpdatePTDirty(true);

    return static_cast<u16>(cpLen);
}

const font::Font*
TextBox::GetFont() const
{
    return m_pFont;
}

void
TextBox::SetFont(const font::Font* pFont)
{
    if (pFont && pFont->GetCharacterCode() != font::CHARACTER_CODE_UNICODE)
    {
        NW_WARNING(false, "pFont is not uincode encoding.");
        return;
    }

    if (UpdatePTDirty(m_pFont != pFont))
    {
        m_pFont = pFont;

        if (m_pFont)
        {
            SetFontSize(Size(static_cast<f32>(m_pFont->GetWidth()), static_cast<f32>(m_pFont->GetHeight())));
        }
        else
        {
            SetFontSize(Size(0.f, 0.f));
        }
    }
}

void
TextBox::LoadMtx(DrawInfo& drawInfo)
{
#ifdef LYT_TEST
    // 何のためにvirtual関数になっているのかわからない
    // ひとまず使わないようにする

    MTX34 mtx;

    GetTextGlobalMtx(&mtx);

    drawInfo.GetGraphicsResource()->GetFontDrawer().SetViewMtx(&mtx);
#else
    (void)drawInfo;
#endif
}

void
TextBox::SetFontInfo(font::WideTextWriter* pWriter) const
{
    pWriter->SetFont(m_pFont);
    if (m_pFont != NULL)
    {
        pWriter->SetFontSize(m_FontSize.width, m_FontSize.height);
        pWriter->SetLineSpace(m_LineSpace);
        pWriter->SetCharSpace(m_CharSpace);
        if (IsWidthLimitEnable())
        {
            pWriter->SetWidthLimit(GetSize().width);
        }
        else
        {
            pWriter->ResetWidthLimit();
        }
    }

    if (m_pTagProcessor)
    {
        pWriter->SetTagProcessor(m_pTagProcessor);
    }
}

void
TextBox::SetTextPos(font::WideTextWriter* pWriter) const
{
    u32 value = 0;

    switch (GetTextAlignment())
    {
    case TEXTALIGNMENT_SYNCHRONOUS:
    default:
        switch (GetTextPositionH())
        {
        case HORIZONTALPOSITION_LEFT:
        default:                        value = font::WideTextWriter::HORIZONTAL_ALIGN_LEFT;   break;
        case HORIZONTALPOSITION_CENTER: value = font::WideTextWriter::HORIZONTAL_ALIGN_CENTER; break;
        case HORIZONTALPOSITION_RIGHT:  value = font::WideTextWriter::HORIZONTAL_ALIGN_RIGHT;  break;
        }
        break;

    case TEXTALIGNMENT_LEFT:        value = font::WideTextWriter::HORIZONTAL_ALIGN_LEFT;    break;
    case TEXTALIGNMENT_CENTER:      value = font::WideTextWriter::HORIZONTAL_ALIGN_CENTER;  break;
    case TEXTALIGNMENT_RIGHT:       value = font::WideTextWriter::HORIZONTAL_ALIGN_RIGHT;   break;
    }

    switch (GetTextPositionH())
    {
    case HORIZONTALPOSITION_LEFT:
    default:
        // NOTE: HORIZONTAL_ORIGIN_LEFT は 0 なので何もしなくていい。
        //value |= font::WideTextWriter::HORIZONTAL_ORIGIN_LEFT;
        break;
    case HORIZONTALPOSITION_CENTER:
        value |= font::WideTextWriter::HORIZONTAL_ORIGIN_CENTER;
        break;
    case HORIZONTALPOSITION_RIGHT:
        value |= font::WideTextWriter::HORIZONTAL_ORIGIN_RIGHT;
        break;
    }

    switch (GetTextPositionV())
    {
    case VERTICALPOSITION_TOP:
    default:
        // NOTE: VERTICAL_ORIGIN_TOP は 0 なので何もしなくていい。
        //value |= font::WideTextWriter::VERTICAL_ORIGIN_TOP;
        break;
    case VERTICALPOSITION_CENTER:
        value |= font::WideTextWriter::VERTICAL_ORIGIN_MIDDLE;
        break;
    case VERTICALPOSITION_BOTTOM:
        value |= font::WideTextWriter::VERTICAL_ORIGIN_BOTTOM;
        break;
    }

    pWriter->SetDrawFlag(value);
}

VEC2
TextBox::AdjustTextPos(
    const Size&     size,
    bool            isCeil
) const
{
    VEC2 pos;

    switch (GetTextPositionH())
    {
    case HORIZONTALPOSITION_LEFT:
    default:
        pos.x = 0.f;
        break;
    case HORIZONTALPOSITION_CENTER:
        pos.x = internal::AdjustCenterValue(size.width, isCeil);
        break;
    case HORIZONTALPOSITION_RIGHT:
        pos.x = size.width;
        break;
    }

    switch (GetTextPositionV())
    {
    case VERTICALPOSITION_TOP:
    default:
        pos.y = 0.f;
        break;
    case VERTICALPOSITION_CENTER:
        pos.y = internal::AdjustCenterValue(size.height, isCeil);
        break;
    case VERTICALPOSITION_BOTTOM:
        pos.y = size.height;
        break;
    }

    return pos;
}

void
TextBox::GetTextGlobalMtx(nw::math::MTX34* pMtx) const
{
    MTX34Copy(pMtx, &GetGlobalMtx());

    // テキストの描画開始位置の調整をViewModel行列で行う。
    VEC2 pos = GetVtxPos();
    const VEC2 txtPos = AdjustTextPos(GetSize(), false);

    pos.x += txtPos.x;
    pos.y -= txtPos.y;

    pMtx->m[0][3] += pMtx->m[0][0] * pos.x + pMtx->m[0][1] * pos.y;
    pMtx->m[1][3] += pMtx->m[1][0] * pos.x + pMtx->m[1][1] * pos.y;
    pMtx->m[2][3] += pMtx->m[2][0] * pos.x + pMtx->m[2][1] * pos.y;

    // TextWriterはY軸正を下にしているので、YとZを反転させる。
    //     Yの反転。
    pMtx->m[0][1] = - pMtx->m[0][1];
    pMtx->m[1][1] = - pMtx->m[1][1];
    pMtx->m[2][1] = - pMtx->m[2][1];

    //     Zの反転。
    //     Zは影響が無いので計算を省略。
    /*
    pMtx->m[0][2] = - pMtx->m[0][2];
    pMtx->m[1][2] = - pMtx->m[1][2];
    pMtx->m[2][2] = - pMtx->m[2][2];
    */
}

bool
TextBox::IsShareCharAttributeBuffer() const
{
    if (m_pDispStringBuf) {
        return m_pDispStringBuf->IsUseSharedCharAttrs();
    } else {
        return false;
    }
}

void
TextBox::SetupTextWriter(font::WideTextWriter* pWriter)
{
    pWriter->SetCursor(0, 0);
    SetFontInfo(pWriter);
    SetTextPos(pWriter);

    pWriter->SetTextColor(m_TextColors[TEXTCOLOR_TOP], m_TextColors[TEXTCOLOR_BOTTOM]);
    pWriter->SetItalicRatio(m_ItalicRatio);
}

void TextBox::UpdatePerCharacterTransform()
{
    NW_PROFILE("nw::lyt::TextBox::UpdatePerCharacterTransform");

    NW_ASSERT_NOT_NULL(m_pPerCharacterTransform);
    u32 charCount = m_pDispStringBuf->GetCharCount();
    u32 curveNumber = m_pPerCharacterTransform->CurveNumber;
    if (charCount == 0)
    {
        return;
    }
    const f32 dividedWidth = charCount == 1 ? 0.0f : m_pPerCharacterTransform->Width / static_cast<f32>(charCount - 1);
    for (u32 i = 0; i < charCount; i++)
    {
        nw::font::PerCharacterTransformInfo* pPerCharacterTransformInfo = &m_pPerCharacterTransform->pPerCharacterTransformInfos[i];
        f32 frame = m_pPerCharacterTransform->Offset + dividedWidth * static_cast<f32>(i);
        for (u32 j = 0; j < curveNumber; j++)
        {
            const nw::lyt::res::AnimationTarget* pAnimTarget = m_pPerCharacterTransform->CurveInfos[j].Curve;
            AnimTargetPerCharacterTransformCurve curveType = m_pPerCharacterTransform->CurveInfos[j].CurveType;
            const res::HermiteKey* keys = internal::ConvertOffsToPtr<res::HermiteKey>(pAnimTarget, pAnimTarget->keysOffset);
            f32 loopFrame = (m_pPerCharacterTransform->LoopType == 1 && pAnimTarget->keyNum > 1) ? GetLoopFrame(frame, keys[0].frame, keys[pAnimTarget->keyNum - 1].frame) : frame;
            f32 value = nw::lyt::GetHermiteCurveValue(loopFrame, keys, pAnimTarget->keyNum);
            if (curveType <= ANIMTARGET_PERCHARACTERTRANSFORMCURVE_TRANSLATION_Z)
            {
                // ANIMTARGET_PERCHARACTERTRANSFORMCURVE_TRANSLATION_X から ANIMTARGET_PERCHARACTERTRANSFORMCURVE_TRANSLATION_Zまで
                u32 index = curveType - ANIMTARGET_PERCHARACTERTRANSFORMCURVE_TRANSLATION_X;
                pPerCharacterTransformInfo->Translation[index] = (curveType == ANIMTARGET_PERCHARACTERTRANSFORMCURVE_TRANSLATION_Y ? - value : value);
            }
            else if (curveType <= ANIMTARGET_PERCHARACTERTRANSFORMCURVE_ROTATION_Z)
            {
                // ANIMTARGET_PERCHARACTERTRANSFORMCURVE_ROTATION_X から ANIMTARGET_PERCHARACTERTRANSFORMCURVE_ROTATION_Zまで
                u32 index = curveType - ANIMTARGET_PERCHARACTERTRANSFORMCURVE_ROTATION_X;
                nw::math::SinCosDeg(&pPerCharacterTransformInfo->RotationSin[index], &pPerCharacterTransformInfo->RotationCos[index],
                    (curveType == ANIMTARGET_PERCHARACTERTRANSFORMCURVE_ROTATION_Y ? value : - value));
            }
            else if (curveType <= ANIMTARGET_PERCHARACTERTRANSFORMCURVE_LT_A)
            {
                // ANIMTARGET_PERCHARACTERTRANSFORMCURVE_LT_RからANIMTARGET_PERCHARACTERTRANSFORMCURVE_LT_Aまで
                u32 index = curveType - ANIMTARGET_PERCHARACTERTRANSFORMCURVE_LT_R;
                pPerCharacterTransformInfo->LT[index] = static_cast<u8>(value);
            }
            else
            {
                // ANIMTARGET_PERCHARACTERTRANSFORMCURVE_LB_RからANIMTARGET_PERCHARACTERTRANSFORMCURVE_LB_Aまで
                u32 index = curveType - ANIMTARGET_PERCHARACTERTRANSFORMCURVE_LB_R;
                pPerCharacterTransformInfo->LB[index] = static_cast<u8>(value);
            }
        }
    }
}

} // namespace nw::lyt
} // namespace nw
