﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_Picture.h>

#include <nw/lyt/lyt_Layout.h>
#include <nw/lyt/lyt_Material.h>
#include <nw/lyt/lyt_DrawInfo.h>

#include <nw/dev/dev_Profile.h>

using namespace nw::math;

namespace nw
{
namespace lyt
{

Picture::Picture(u8 texNum)
{
    NW_ASSERTMSG(texNum <= TexMapMax, "out of bounds: texNum[%u] <= TexMapMax for Picture[%s]", texNum, GetName());

    Init(texNum);

    // マテリアルの作成
    m_pMaterial = Layout::NewObj<Material>();
    if (m_pMaterial)
    {
        m_pMaterial->ReserveMem(texNum, texNum, texNum);
    }
}

Picture::Picture(const TextureInfo& textureInfo)
{
    const int texNum = 1;
    Init(texNum);

    // マテリアルの作成
    m_pMaterial = Layout::NewObj<Material>();
    if (m_pMaterial)
    {
        m_pMaterial->ReserveMem(texNum, texNum, texNum);
        Append(textureInfo);
    }
}

Picture::Picture(
    const res::Picture* pBaseBlock,
    const res::Picture* pOverrideBlock,
    const BuildArgSet& buildArgSet
)
: Base(pBaseBlock, buildArgSet)
{
    const res::Picture* pBlock;
    if (pOverrideBlock && buildArgSet.overrideMaterialUsageFlag == 0)
    {
        // 丸ごと上書きする場合
        pBlock = pOverrideBlock;
    }
    else
    {
        // 上書きがない、もしくは部分上書きの場合
        pBlock = pBaseBlock;
    }

    const u8 texCoordNum = static_cast<u8>(ut::Min((int)pBlock->texCoordNum, TexMapMax));

    Init(texCoordNum);

    // 頂点カラー
    for (int i = 0; i < VERTEXCOLOR_MAX; ++i)
    {
        m_VtxColors[i] = pBlock->vtxCols[i];
    }

    // テクスチャ座標
    if (texCoordNum > 0)
    {
        if (! m_TexCoordAry.IsEmpty())
        {
            m_TexCoordAry.Copy(reinterpret_cast<const char*>(pBlock) + sizeof(*pBlock), texCoordNum);
        }
    }

    // マテリアルの作成
    {
        const res::Material *const pResMaterial = nw::lyt::internal::GetResMaterial(buildArgSet.pCurrentBuildResSet, pBaseBlock->materialIdx);
        const res::Material *pOverrideResMaterial = NULL;
        if (pOverrideBlock) {
            pOverrideResMaterial = nw::lyt::internal::GetResMaterial(buildArgSet.pOverrideBuildResSet, pOverrideBlock->materialIdx);
        }
        m_pMaterial = Layout::NewObj<Material, const res::Material*, const res::Material*, const BuildArgSet&>(pResMaterial, pOverrideResMaterial, buildArgSet);
    }
}

Picture::Picture(const Picture& picture)
 : Pane(picture)
 , m_pMaterial(NULL)
{
    // 頂点カラー
    for (int i = 0; i < VERTEXCOLOR_MAX; ++i)
    {
        m_VtxColors[i] = picture.m_VtxColors[i];
    }
    // テクスチャ座標
    const u8 texCoordNum = picture.m_TexCoordAry.GetSize();
    if (texCoordNum > 0)
    {
        ReserveTexCoord(texCoordNum);
        m_TexCoordAry.SetSize(texCoordNum);
        for (u32 i = 0; i < texCoordNum; i++)
        {
            m_TexCoordAry.SetCoord(i, picture.m_TexCoordAry.GetArray()[i]);
        }
    }
    // マテリアルの複製
    m_pMaterial = Layout::NewObj<Material, const Material&>(*picture.m_pMaterial);
}

void
Picture::Init(u8 texNum)
{
    if (texNum > 0)
    {
        ReserveTexCoord(texNum);
    }
}

Picture::~Picture()
{
    // OSReport("Picture::~Picture()\n");

    if (m_pMaterial && ! m_pMaterial->IsUserAllocated())
    {
        Layout::DeleteObj(m_pMaterial);
        m_pMaterial = 0;
    }

    m_TexCoordAry.Free();
}

u8
Picture::GetMaterialNum() const
{
    return static_cast<u8>(m_pMaterial ? 1 : 0);
}

Material*
Picture::GetMaterial(u32 idx) const
{
    NW_WARNING(idx < GetMaterialNum(), "idx >= GetMaterialNum() : %d >= %d", idx, GetMaterialNum());

    return idx == 0 ? m_pMaterial : 0;
}

void Picture::SetMaterial(Material* pMaterial)
{
    if (m_pMaterial == pMaterial)
    {
        return;
    }

    if (m_pMaterial != NULL && !m_pMaterial->IsUserAllocated())
    {
        Layout::DeleteObj(m_pMaterial);
    }

    m_pMaterial = pMaterial;
}

void
Picture::Append(const TextureInfo& textureInfo)
{
    if (m_pMaterial->GetTexMapNum() >= m_pMaterial->GetTexMapCap() || m_pMaterial->GetTexCoordGenNum() >= m_pMaterial->GetTexCoordGenCap())
    {
        NW_WARNING(false, "m_pMaterial->GetTexMapNum(%d) is large. m_pMaterial->GetTexMapCap(%d), m_pMaterial->GetTexCoordGenCap(%d)\n", m_pMaterial->GetTexMapNum(), m_pMaterial->GetTexMapCap(), m_pMaterial->GetTexCoordGenCap());
        return;
    }

    const u8 texIdx = m_pMaterial->GetTexMapNum();
    m_pMaterial->AppendTexMap(textureInfo);
    m_pMaterial->SetTexCoordGenNum(m_pMaterial->GetTexMapNum());
    m_pMaterial->SetTexCoordGen(texIdx, res::TexCoordGen());

    SetTexCoordNum(m_pMaterial->GetTexMapNum());

    // サイズが設定されていないときは、サイズを設定する。
    if (GetSize() == Size(0.f, 0.f) && m_pMaterial->GetTexMapNum() == 1)
    {
        const TexSize& texSize = m_pMaterial->GetTexMap(0).GetSize();
        SetSize(Size(static_cast<f32>(texSize.width), static_cast<f32>(texSize.height)));
    }
}

void
Picture::ReserveTexCoord(u8 num)
{
    // TODO:
    // * Cap値を取得する方法がない。
    // * 広げることはできるがちじめることができない。
    m_TexCoordAry.Reserve(num);
}

u8
Picture::GetTexCoordNum() const
{
    return m_TexCoordAry.GetSize();
}

void
Picture::SetTexCoordNum(u8 num)
{
    m_TexCoordAry.SetSize(num);
}

void
Picture::GetTexCoord(
    u32 idx,
    TexCoordQuad coords
) const
{
    return m_TexCoordAry.GetCoord(idx, coords);
}

void
Picture::SetTexCoord(
    u32 idx,
    const TexCoordQuad coords
)
{
    m_TexCoordAry.SetCoord(idx, coords);
}

const ut::Color8
Picture::GetVtxColor(u32 idx) const
{
    NW_ASSERTMSG(idx < VERTEXCOLOR_MAX, "out of bounds: idx[%u] <= VERTEXCOLOR_MAX for Picture[%s]", idx, GetName());

    return m_VtxColors[idx];
}

void
Picture::SetVtxColor(
    u32 idx,
    ut::Color8 value
)
{
    NW_ASSERTMSG(idx < VERTEXCOLOR_MAX, "out of bounds: idx[%u] <= VERTEXCOLOR_MAX for Picture[%s]", idx, GetName());

    m_VtxColors[idx] = value;
}

u8
Picture::GetVtxColorElement(u32 idx) const
{
    return internal::GetVtxColorElement(m_VtxColors, idx);
}

void
Picture::SetVtxColorElement(u32 idx, u8 value)
{
    internal::SetVtxColorElement(m_VtxColors, idx, value);
}

void
Picture::DrawSelf(DrawInfo& drawInfo)
{
    NW_PROFILE("nw::lyt::Picture::DrawSelf");

    if (! m_pMaterial)
    {
        return;
    }

    LoadMtx(drawInfo);
    NW_GL_ASSERT();

    bool vertexColorEnabled = internal::TestVertexColorEnabled(m_VtxColors);

    if (vertexColorEnabled || GetGlobalAlpha() != 255)
    {
        m_pMaterial->SetupGraphics(drawInfo, GetGlobalAlpha(), STANDARD_SHADER_VARIATION, true, this->GetGlobalMtx());
        NW_GL_ASSERT();

        internal::SetupVertexColor(drawInfo, m_VtxColors);
    }
    else
    {
        m_pMaterial->SetupGraphics(drawInfo, GetGlobalAlpha(), WITHOUT_VERTEX_COLOR_SHADER_VARIATION, true, this->GetGlobalMtx());
        NW_GL_ASSERT();
    }

    internal::DrawQuadWithTexCoords(
        drawInfo,
        GetVtxPos(),
        GetSize(),
        m_TexCoordAry.GetSize(),
        m_TexCoordAry.GetArray());

    NW_GL_ASSERT();
}

} // namespace nw::lyt
} // namespace nw
