﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_Common.h>
#include <nw/lyt/lyt_Pane.h>
#include <nw/lyt/lyt_DrawInfo.h>
#include <nw/lyt/lyt_GraphicsResource.h>
#include <nw/lyt/lyt_Layout.h>
#include <nw/lyt/lyt_Animation.h>
#include <nw/lyt/lyt_Material.h>

using namespace nw::math;

namespace nw
{
namespace lyt
{
namespace internal
{


PaneBase::PaneBase()
{
}

PaneBase::~PaneBase()
{
}

} // namespace nw::lyt::internal

namespace local
{
namespace
{

inline void
MultMatrixSRTxST(
    NW_LYT_WRITEONLY nw::math::MTX34* __restrict pMtx,
    const nw::math::MTX34* __restrict pMtxParent,
    const nw::math::VEC2* __restrict pScale,
    const nw::math::VEC3* __restrict pTransform)
{
    //(Ma . Mc) . Mb
    //
    // = [ a00  a01  a02  a03 ] . [   1    0    0   cx ] . [ b00    0    0  b03 ]
    //   [                    ]   [                    ]   [                    ]
    //   [ a10  a11  a12  a13 ]   [   0    1    0   cy ]   [   0  b11    0  b13 ]
    //   [                    ]   [                    ]   [                    ]
    //   [ a20  a21  a22  a23 ]   [   0    0    1    0 ]   [   0    0    1  b23 ]
    //   [                    ]   [                    ]   [                    ]
    //   [   0    0    0    1 ]   [   0    0    0    1 ]   [   0    0    0    1 ]
    //
    // = [ a00*b00  a01*b11  a02  a02*b23 + a01*b13 + a00*b03 + a03 + a00 * cx + a01 * cy ]
    //   [                                                                                ]
    //   [ a10*b00  a11*b11  a12  a12*b23 + a11*b13 + a10*b03 + a13 + a10 * cx + a11 * cy ]
    //   [                                                                                ]
    //   [ a20*b00  a21*b11  a22  a22*b23 + a21*b13 + a20*b03 + a23                       ]
    //   [                                                                                ]
    //   [    0        0      0                              1                            ]

    register const f32 b00 = pScale->x;
    register const f32 b11 = pScale->y;
    register const f32 b03 = pTransform->x;
    register const f32 b13 = pTransform->y;
    register const f32 b23 = pTransform->z;

    register const f32 a00 = pMtxParent->m[0][0];
    register const f32 a01 = pMtxParent->m[0][1];
    register const f32 a02 = pMtxParent->m[0][2];
    register const f32 a03 = pMtxParent->m[0][3];

    register const f32 a10 = pMtxParent->m[1][0];
    register const f32 a11 = pMtxParent->m[1][1];
    register const f32 a12 = pMtxParent->m[1][2];
    register const f32 a13 = pMtxParent->m[1][3];

    register const f32 a20 = pMtxParent->m[2][0];
    register const f32 a21 = pMtxParent->m[2][1];
    register const f32 a22 = pMtxParent->m[2][2];
    register const f32 a23 = pMtxParent->m[2][3];

    pMtx->m[0][0] = a00 * b00;
    pMtx->m[0][1] = a01 * b11;
    pMtx->m[0][2] = a02;
    pMtx->m[0][3] = a02 * b23 + a01 * b13 + a00 * b03 + a03;

    pMtx->m[1][0] = a10 * b00;
    pMtx->m[1][1] = a11 * b11;
    pMtx->m[1][2] = a12;
    pMtx->m[1][3] = a12 * b23 + a11 * b13 + a10 * b03 + a13;

    pMtx->m[2][0] = a20 * b00;
    pMtx->m[2][1] = a21 * b11;
    pMtx->m[2][2] = a22;
    pMtx->m[2][3] = a22 * b23 + a21 * b13 + a20 * b03 + a23;
}

// Scale、Rotate、Transの設定
inline void
MakeMatrixSRT(NW_LYT_WRITEONLY nw::math::MTX34& o, const nw::math::VEC2& s, const f32 rotateZ, const nw::math::VEC3& t)
{
    register f32 sin, cos;

    nw::math::SinCosDeg(&sin, &cos, rotateZ);

    const f32 tmp0 = s.x * cos;
    const f32 tmp1 = -s.y * sin;
    const f32 tmp2 = s.x * sin;
    const f32 tmp3 = s.y * cos;

    o.m[0][0] = tmp0;
    o.m[0][1] = tmp1;
    o.m[0][2] = 0.f;
    o.m[0][3] = t.x;

    o.m[1][0] = tmp2;
    o.m[1][1] = tmp3;
    o.m[1][2] = 0.f;
    o.m[1][3] = t.y;

    o.m[2][0] = 0.f;
    o.m[2][1] = 0.f;
    o.m[2][2] = 1.f;
    o.m[2][3] = t.z;
}

inline void
MakeMatrixSRT(NW_LYT_WRITEONLY nw::math::MTX34& o, const nw::math::VEC2& s, const nw::math::VEC3& r, const nw::math::VEC3& t)
{
    register f32 sinx, cosx;
    register f32 siny, cosy;
    register f32 sinz, cosz;

    nw::math::SinCosDeg(&sinx, &cosx, r.x);
    nw::math::SinCosDeg(&siny, &cosy, r.y);
    nw::math::SinCosDeg(&sinz, &cosz, r.z);

    const f32 opt1 = cosx * cosz;
    const f32 opt2 = sinx * siny;
    const f32 opt3 = cosx * sinz;

    const f32 tmp00 = s.x * (cosy * cosz);
    const f32 tmp10 = s.x * (cosy * sinz);
    const f32 tmp20 = s.x * (-siny);

    const f32 tmp01 = s.y * ((opt2 * cosz) - (opt3));
    const f32 tmp11 = s.y * ((opt2 * sinz) + (opt1));
    const f32 tmp21 = s.y * ((sinx * cosy));

    const f32 tmp02 = (opt1 * siny) + (sinx * sinz);
    const f32 tmp12 = (opt3 * siny) - (sinx * cosz);
    const f32 tmp22 = (cosx * cosy);

    o.m[0][0] = tmp00;
    o.m[0][1] = tmp01;
    o.m[0][2] = tmp02;
    o.m[0][3] = t.x;

    o.m[1][0] = tmp10;
    o.m[1][1] = tmp11;
    o.m[1][2] = tmp12;
    o.m[1][3] = t.y;

    o.m[2][0] = tmp20;
    o.m[2][1] = tmp21;
    o.m[2][2] = tmp22;
    o.m[2][3] = t.z;
}

} // namespace nw::lyt::local::{anonymous}
} // namespace nw::lyt::local

Pane::Pane()
{
    using namespace std;

    Init();

    m_BasePosition =
          HORIZONTALPOSITION_CENTER
        + VERTICALPOSITION_CENTER * VERTICALPOSITION_MAX;
    memset(m_Name, 0, sizeof(m_Name));
    memset(m_UserData, 0, sizeof(m_UserData));

    m_Translate = VEC3(0, 0, 0);
    m_Rotate = VEC3(0, 0, 0);
    m_Scale = VEC2(1, 1);
    m_Size = Size(0.f, 0.f);
    m_Alpha = ut::Color8::ALPHA_MAX;
    m_GlbAlpha = m_Alpha;

    SetVisible(true);
    SetGlbMtxDirty();
}

Pane::Pane(const res::Pane* pBlock, const BuildArgSet& buildArgSet)
{
    using namespace std;

    Init();

    const res::Vec3* translate = &pBlock->translate;
    const res::Vec3* rotate = &pBlock->rotate;
    const res::Vec2* scale = &pBlock->scale;
    const res::Vec2* size = &pBlock->size;
    const ut::ResU8* alpha = &pBlock->alpha;
    const char* userData = pBlock->userData;

    // ペイン基本情報の上書きを行っている場合、そちらに置き換える
    if (buildArgSet.pOverridePartsPaneBasicInfo)
    {
        if (internal::TestBit(buildArgSet.overrideBasicUsageFlag, BASICOVERRIDEUSAGEFLAG_TRANSLATE_ENABLED))
        {
            translate = &buildArgSet.pOverridePartsPaneBasicInfo->translate;
        }
        if (internal::TestBit(buildArgSet.overrideBasicUsageFlag, BASICOVERRIDEUSAGEFLAG_ROTATE_ENABLED))
        {
            rotate = &buildArgSet.pOverridePartsPaneBasicInfo->rotate;
        }
        if (internal::TestBit(buildArgSet.overrideBasicUsageFlag, BASICOVERRIDEUSAGEFLAG_SCALE_ENABLED))
        {
            scale = &buildArgSet.pOverridePartsPaneBasicInfo->scale;
        }
        if (internal::TestBit(buildArgSet.overrideBasicUsageFlag, BASICOVERRIDEUSAGEFLAG_SIZE_ENABLED))
        {
            size = &buildArgSet.pOverridePartsPaneBasicInfo->size;
        }
        if (internal::TestBit(buildArgSet.overrideBasicUsageFlag, BASICOVERRIDEUSAGEFLAG_ALPHA_ENABLED))
        {
            alpha = &buildArgSet.pOverridePartsPaneBasicInfo->alpha;
        }
        if (internal::TestBit(buildArgSet.overrideBasicUsageFlag, BASICOVERRIDEUSAGEFLAG_BASIC_USERDATA_ENABLED))
        {
            userData = buildArgSet.pOverridePartsPaneBasicInfo->userData;
        }
    }

    m_BasePosition = pBlock->basePosition;
    SetName(pBlock->name);
    SetUserData(userData);
    SetExtUserDataList(buildArgSet.pExtUserDataList);

    m_Translate = *translate;
    m_Rotate = *rotate;
    m_Scale = *scale;
    if (buildArgSet.pOverrideBuildResSet && (buildArgSet.magnify.x != 1.f || buildArgSet.magnify.y != 1.f))
    {
        // 部品を構築中、部品倍率を考慮する必要がある
        if (internal::TestBit(pBlock->flagEx, PANEFLAG_EX_IGNORE_PARTS_MAGNIFY))
        {
            // 部品倍率を適用しない
            m_Size.width = size->x;
            m_Size.height = size->y;
        }
        else if (internal::TestBit(pBlock->flagEx, PANEFLAG_EX_PARTS_MAGNIFY_ADJUST_TO_PARTS_BOUND))
        {
            // 部品倍率を部品の境界からの位置が一定になるように適用する
            float compH = (GetBasePositionH() == HORIZONTALPOSITION_CENTER ? 1.0f : 0.5f );
            float compV = (GetBasePositionV() == VERTICALPOSITION_CENTER ? 1.0f : 0.5f );
            math::VEC2 global_scale;
            CalcScaleFromPartsRoot(&global_scale, buildArgSet.pParentPane);
            float expectPartsSizeX = buildArgSet.partsSize.x * buildArgSet.magnify.x;
            float expectPartsSizeY = buildArgSet.partsSize.y * buildArgSet.magnify.y;
            float expandH = (expectPartsSizeX - buildArgSet.partsSize.x) * compH;
            float expandV = (expectPartsSizeY - buildArgSet.partsSize.y) * compV;
            m_Size.width = (size->x + expandH / (global_scale.x * scale->x));
            m_Size.height = (size->y + expandV / (global_scale.y * scale->y));
        }
        else
        {
            // 部品倍率を単純にスケール倍して適用する
            m_Size.width = size->x * buildArgSet.magnify.x;
            m_Size.height = size->y * buildArgSet.magnify.y;
        }
    }
    else
    {
        m_Size.width = size->x;
        m_Size.height = size->y;
    }
    m_Alpha = *alpha;
    m_GlbAlpha = m_Alpha;

    m_Flag = pBlock->flag;
#if ! defined(NW_RELEASE)
    internal::SetBit(&m_Flag, PANEFLAG_VIEWER_INVISIBLE, IsViewerInvisible() || buildArgSet.isViewerInvisibleByParent);
#endif

    // 表示/非表示についてはここで上書きを行う
    if (buildArgSet.pOverrideBuildResSet)
    {
        if (internal::TestBit(buildArgSet.overrideBasicUsageFlag, BASICOVERRIDEUSAGEFLAG_VISIBLE_ENABLED))
        {
            SetVisible(internal::TestBit(buildArgSet.overrideBasicUsageFlag, BASICOVERRIDEUSAGEFLAG_VISIBLE_VALUE));
        }
    }
    SetGlbMtxDirty();
}

Pane::Pane(const Pane& pane)
 : m_pParent(NULL)
 , m_Translate(pane.m_Translate)
 , m_Rotate(pane.m_Rotate)
 , m_Scale(pane.m_Scale)
 , m_Size(pane.m_Size)
 , m_Flag(pane.m_Flag)
 , m_Alpha(pane.m_Alpha)
 , m_GlbAlpha(pane.m_GlbAlpha)
 , m_BasePosition(pane.m_BasePosition)
 , m_GlbMtx(pane.m_GlbMtx)
 , m_UserMtx(NULL)
 , m_pExtUserDataList(pane.m_pExtUserDataList)
{
    SetName(pane.GetName());
    SetUserData(pane.GetUserData());
    ResetMtx();
    SetGlbMtxDirty();
}

void
Pane::Init()
{
    m_pParent = 0;
    m_Flag = 0;
    m_pExtUserDataList = 0;
    MTX34Identity(&m_GlbMtx);
    m_UserMtx = NULL;
}

Pane::~Pane()
{
    // OSReport("Pane::~Pane()\n");

    // 子供Paneの後処理
    for (PaneList::Iterator it = m_ChildList.GetBeginIter(); it != m_ChildList.GetEndIter();)
    {
        PaneList::Iterator currIt = it++;
        m_ChildList.Erase(currIt);
        if (! currIt->IsUserAllocated())
        {
            Parts* parts = ut::DynamicCast<Parts*>(&(*currIt));
            if (parts && parts->GetLayout()) {
                // 部品ペインでレイアウトがある場合は、レイアウトをDeleteObjする。
                // 部品ペインは部品レイアウトのルートペインになってるので、一緒に消される
                Layout::DeleteObj(parts->GetLayout());
            } else {
                Layout::DeleteObj(&(*currIt));
            }
        }
    }
}

void
Pane::SetName(const char* name)
{
    ut::strcpy(m_Name, sizeof(m_Name), name);
}

void
Pane::SetUserData(const char* userData)
{
    ut::strcpy(m_UserData, sizeof(m_UserData), userData);
}

void
Pane::AppendChild(Pane* pChild)
{
    InsertChild(m_ChildList.GetEndIter(), pChild);
}

void
Pane::PrependChild(Pane* pChild)
{
    InsertChild(m_ChildList.GetBeginIter(), pChild);
}

void
Pane::InsertChild(
    Pane* pNext,
    Pane* pChild
)
{
    NW_ASSERTMSG(pNext, "pNext must not be NULL for Pane[%s]", GetName());
    NW_ASSERTMSG(pNext->m_pParent == this, "pNext->m_pParent must be this for Pane[%s]", GetName());

    InsertChild(m_ChildList.GetIteratorFromPointer(pNext), pChild);
}

void
Pane::InsertChild(
    PaneList::Iterator next,
    Pane* pChild
)
{
    NW_ASSERTMSG(pChild, "pChild must be NULL for Pane[%s]", GetName());
    NW_ASSERTMSG(pChild->m_pParent == 0, "pChild->m_pParent must be NULL for Pane[%s]", GetName());

    m_ChildList.Insert(next, pChild);
    pChild->m_pParent = this;
    // 追加された子ペインはグローバルマトリックスを計算し直す必要がある
    pChild->SetGlbMtxDirty();
}

void
Pane::RemoveChild(Pane* pChild)
{
    NW_ASSERTMSG(pChild, "pChild must be NULL for Pane[%s]", GetName());
    NW_ASSERTMSG(pChild->m_pParent == this, "pChild->m_pParent must be this for Pane[%s]", GetName());

    m_ChildList.Erase(pChild);
    pChild->m_pParent = NULL;
}

const ut::Rect
Pane::GetPaneRect() const
{
    ut::Rect ret;
    VEC2 basePt = GetVtxPos();

    ret.left = basePt.x;
    ret.top = basePt.y;
    ret.right = ret.left + m_Size.width;
    ret.bottom = ret.top - m_Size.height;

    return ret;
}

const ut::Color8
Pane::GetVtxColor(u32 idx) const
{
    NW_UNUSED_VARIABLE(idx)

    return ut::Color4u8(ut::Color8::WHITE);
}

void
Pane::SetVtxColor(
    u32 idx,
    ut::Color8 value
)
{
    NW_UNUSED_VARIABLE(idx)
    NW_UNUSED_VARIABLE(value)
}

u8
Pane::GetColorElement(u32 idx) const
{
    NW_ASSERTMSG(idx < ANIMTARGET_PANE_COLOR_MAX, "out of bounds: idx[%u] < ANIMTARGET_PANE_COLOR_MAX for Pane[%s]", idx, GetName());

    switch (idx)
    {
    case ANIMTARGET_PANE_ALPHA:
        return m_Alpha;
    default:
        return GetVtxColorElement(idx);
    }
}

void
Pane::SetColorElement(
    u32 idx,
    u8 value
)
{
    NW_ASSERTMSG(idx < ANIMTARGET_PANE_COLOR_MAX, "out of bounds: idx[%u] < ANIMTARGET_PANE_COLOR_MAX for Pane[%s]", idx, GetName());

    switch (idx)
    {
    case ANIMTARGET_PANE_ALPHA:
        m_Alpha = value;
        break;
    default:
        SetVtxColorElement(idx, value);
    }
}

u8
Pane::GetVtxColorElement(u32 idx) const
{
    NW_UNUSED_VARIABLE(idx)
    return ut::Color8::ELEMENT_MAX;
}

void
Pane::SetVtxColorElement(
    u32 idx,
    u8 value)
{
    NW_UNUSED_VARIABLE(idx)
    NW_UNUSED_VARIABLE(value)
}

Pane*
Pane::FindPaneByName(
    const char* findName,
    bool bRecursive
)
{
    if (internal::EqualsResName(m_Name, findName))
    {
        return this;
    }

    if (bRecursive)
    {
        // 子供に一致するものが無いか検索
        PaneList::Iterator it_end = m_ChildList.GetEndIter();
        for (PaneList::Iterator it = m_ChildList.GetBeginIter(); it != it_end; ++it)
        {
            if (Pane* pPane = it->FindPaneByNameRecursive(findName))
            {
                return pPane;
            }
        }
    }

    return 0;
}

Material*
Pane::FindMaterialByName(
    const char* findName,
    bool bRecursive
)
{
    u32 nbMaterial = GetMaterialNum();
    for (u32 idx = 0; idx < nbMaterial; ++idx)
    {
        Material* pMaterial = GetMaterial(idx);
        if (pMaterial)
        {
            if (internal::EqualsMaterialName(pMaterial->GetName(), findName))
            {
                return pMaterial;
            }
        }
    }

    if (bRecursive)
    {
        // 子供に一致するものが無いか検索
        PaneList::Iterator it_end = m_ChildList.GetEndIter();
        for (PaneList::Iterator it = m_ChildList.GetBeginIter(); it != it_end; ++it)
        {
            if (Material* pMat = it->FindMaterialByNameRecursive(findName))
            {
                return pMat;
            }
        }
    }

    return 0;
}

void
Pane::CalculateMtx(CalculateMtxContext& context, bool isDirtyParentMtx)
{
    if (! IsVisible() && ! context.isInvisiblePaneCalculateMtx)
    {
        // 非表示であることによりCalculateMtxを中断した場合、非表示が解除された
        // ときに正しく自分及び子階層のグローバル行列が計算されるように、自分自身
        // をDirty状態にしておく
        if (isDirtyParentMtx)
        {
            SetGlbMtxDirty();
        }
        return;
    }

    // アルファの計算
    if (context.isInfluenceAlpha && m_pParent)
    {
        m_GlbAlpha = u8(m_Alpha * context.influenceAlpha);
    }
    else
    {
        m_GlbAlpha = m_Alpha;
    }

    if (m_Alpha == 0 && IsInfluencedAlpha() && ! context.isAlphaZeroPaneCalculateMtx)
    {
        // 透明かつ子階層に透明度の影響を与えることによりCalculateMtxを中断した場合、
        // 透明でなくなったときに正しく自分及び子階層のグローバル行列が計算されるように、
        // 自分自身をDirty状態にしておく
        if (isDirtyParentMtx)
        {
            SetGlbMtxDirty();
        }
        return;
    }

    bool isGlbMtxDirty = (this->IsGlbMtxDirty() || isDirtyParentMtx);
    if (this->IsUserGlobalMtx())
    {
        // グローバルマトリックスは常にDirtyと見なす
        isGlbMtxDirty = true;
    }
    else
    {
        const math::MTX34* pParentMtx = NULL;
        if (m_pParent)
        {
            pParentMtx = &m_pParent->m_GlbMtx;
        }
        else
        {
            pParentMtx = context.viewMtx;
        }

        // 行列の計算
        if (this->IsUserMtx())
        {
            // グローバルマトリックスは常にDirtyと見なす
            isGlbMtxDirty = true;
            math::MTX34Mult(&m_GlbMtx, pParentMtx, m_UserMtx);
        }
        else if (isGlbMtxDirty)
        {
            // ここで計算されるので、グローバルマトリックスはクリーンになる
            CleanGlbMtx();
            math::VEC2 scale(this->GetScale());

            if (context.isLocationAdjust && this->IsLocationAdjust())
            {
                scale.x *= context.locationAdjustScale.x;
                scale.y *= context.locationAdjustScale.y;
            }

            math::VEC3 trans = m_Translate;

            // 親ペイン相対での原点位置を計算する
            switch (GetParentRelativePositionH()) {
            case HORIZONTALPOSITION_LEFT:
                trans.x += - m_pParent->GetSize().width / 2;
                break;
            case HORIZONTALPOSITION_RIGHT:
                trans.x += m_pParent->GetSize().width / 2;
                break;
            default:
                break;
            }
            switch (GetParentRelativePositionV()) {
            case VERTICALPOSITION_TOP:
                trans.y += m_pParent->GetSize().height / 2;
                break;
            case VERTICALPOSITION_BOTTOM:
                trans.y += - m_pParent->GetSize().height / 2;
                break;
            default:
                break;
            }

            if (m_Rotate.x != 0 || m_Rotate.y != 0)
            { // XY回転有り
                math::MTX34 mtx;
                local::MakeMatrixSRT(mtx, scale, m_Rotate, trans);
                math::MTX34Mult(&m_GlbMtx, pParentMtx, &mtx);
            }
            else if (m_Rotate.z != 0)
            { // Z回転有り
                math::MTX34 mtx;
                local::MakeMatrixSRT(mtx, scale, m_Rotate.z, trans);
                math::MTX34Mult(&m_GlbMtx, pParentMtx, &mtx);
            }
            else
            {
                local::MultMatrixSRTxST(&m_GlbMtx, pParentMtx, &scale, &trans);
            }
        }
    }

    /*
        PANEFLAG_INFLUENCEDALPHAが立っていて、自分の透明度が255でないなら、
        CalculateMtxContextに累積透明度(0.f - 1.f)をセットして、子供の計算に影響させる。
    */
    if (IsInfluencedAlpha() && m_Alpha != ut::Color8::ALPHA_MAX)
    {
        const f32 crInfluenceAlpha = context.influenceAlpha;
        const bool bCrInfluenced = context.isInfluenceAlpha;
        const f32 invAlpha = 1.0f / ut::Color8::ALPHA_MAX;
        context.influenceAlpha = crInfluenceAlpha * m_Alpha * invAlpha;
        context.isInfluenceAlpha = true;

        // 子供の行列計算
        PaneList::Iterator it_end = m_ChildList.GetEndIter();
        for (PaneList::Iterator it = m_ChildList.GetBeginIter(); it != it_end; ++it)
        {
            it->CalculateMtx(context, isGlbMtxDirty);
        }

        // 変更した値を元に戻す
        context.influenceAlpha = crInfluenceAlpha;
        context.isInfluenceAlpha = bCrInfluenced;
    }
    else
    {
        // 子供の行列計算
        PaneList::Iterator it_end = m_ChildList.GetEndIter();
        for (PaneList::Iterator it = m_ChildList.GetBeginIter(); it != it_end; ++it)
        {
            it->CalculateMtx(context, isGlbMtxDirty);
        }
    }
}

void
Pane::Draw(DrawInfo& drawInfo)
{
    if (! IsVisible())
    {
        return;
    }

    if (0 < GetGlobalAlpha()
#if ! defined(NW_RELEASE)
         && (m_Flag & (1 << PANEFLAG_VIEWER_INVISIBLE)) == 0
#endif
        )
    {
        DrawSelf(drawInfo);
    }

    // 子供の描画。子にアルファが影響する設定でかつ自分のアルファが0なら、子のアルファは0なので描画しない
    if ( ! IsInfluencedAlpha() || m_Alpha != 0)
    {
        PaneList::Iterator it_end = m_ChildList.GetEndIter();
        for (PaneList::Iterator it = m_ChildList.GetBeginIter(); it != it_end; ++it)
        {
            it->Draw(drawInfo);
        }
    }
}

void
Pane::DrawSelf(DrawInfo& drawInfo)
{
    NW_UNUSED_VARIABLE(drawInfo)
}

void
Pane::BindAnimation(
    AnimTransform* pAnimTrans,
    bool bRecursive,
    bool bEnable
)
{
    NW_ASSERTMSG(pAnimTrans, "pAnimTrans must not be NULL for Pane[%s]", GetName());

    pAnimTrans->BindPane(this, bRecursive);
    pAnimTrans->SetEnable(bEnable);
}

void
Pane::UnbindAnimation(
    AnimTransform* pAnimTrans,
    bool bRecursive
)
{
    UnbindAnimationSelf(pAnimTrans);

    // 再帰的にたどる
    if (bRecursive)
    {
        PaneList::Iterator it_end = m_ChildList.GetEndIter();
        for (PaneList::Iterator it = m_ChildList.GetBeginIter(); it != it_end; ++it)
        {
            it->UnbindAnimation(pAnimTrans, bRecursive);
        }
    }
}

void
Pane::UnbindAnimationSelf(AnimTransform* pAnimTrans)
{
    u32 nbMaterial = GetMaterialNum();
    for (u32 idx = 0; idx < nbMaterial; ++idx)
    {
        Material* pMaterial = GetMaterial(idx);
        if (pMaterial)
        {
            pAnimTrans->UnbindMaterial(pMaterial);
        }
    }

    pAnimTrans->UnbindPane(this);
}

void
Pane::LoadMtx(DrawInfo& drawInfo)
{
    drawInfo.SetModelViewMtx(this->GetGlobalMtx());
}

const math::VEC2
Pane::GetVtxPos() const
{
    VEC2 basePt(0, 0);

    switch (this->GetBasePositionH())
    {
    case HORIZONTALPOSITION_LEFT:
    default:                        basePt.x = 0;                   break;
    case HORIZONTALPOSITION_CENTER: basePt.x = - m_Size.width / 2;   break;
    case HORIZONTALPOSITION_RIGHT:  basePt.x = - m_Size.width;       break;
    }

    switch (this->GetBasePositionV())
    {
    case VERTICALPOSITION_TOP:
    default:                        basePt.y = 0;                   break;
    case VERTICALPOSITION_CENTER:   basePt.y =   m_Size.height / 2;  break;
    case VERTICALPOSITION_BOTTOM:   basePt.y =   m_Size.height;      break;
    }

    return basePt;
}

Material*
Pane::GetMaterial() const
{
    if (GetMaterialNum() > 0)
    {
        return GetMaterial(0);
    }
    else
    {
        return 0;
    }
}

u8
Pane::GetMaterialNum() const
{
    return 0;
}

Material*
Pane::GetMaterial(u32 idx) const
{
    NW_UNUSED_VARIABLE(idx);
    NW_WARNING(idx < GetMaterialNum(), "idx >= GetMaterialNum() : %d >= %d", idx, GetMaterialNum());

    return 0;
}

u16
Pane::GetExtUserDataNum() const
{
    if (! m_pExtUserDataList)
    {
        return 0;
    }

    return m_pExtUserDataList->num;
}

const res::ExtUserData*
Pane::GetExtUserDataArray() const
{
    if (! m_pExtUserDataList)
    {
        return 0;
    }

    return internal::ConvertOffsToPtr<const res::ExtUserData>(m_pExtUserDataList, sizeof(*m_pExtUserDataList));
}

const res::ExtUserData*
Pane::FindExtUserDataByName(const char* name) const
{
    const res::ExtUserData* pExtUserData = GetExtUserDataArray();
    if (! pExtUserData)
    {
        return 0;
    }

    for (int i = 0; i < m_pExtUserDataList->num; ++i, ++pExtUserData)
    {
        if (0 == std::strcmp(name, pExtUserData->GetName()))
        {
            return pExtUserData;
        }
    }

    return 0;
}

Pane* Pane::FindPaneByNameRecursive(const char* findName)
{
    if (internal::EqualsResName(m_Name, findName))
    {
        return this;
    }

    // 子供に一致するものが無いか検索
    PaneList::Iterator it_end = m_ChildList.GetEndIter();
    for (PaneList::Iterator it = m_ChildList.GetBeginIter(); it != it_end; ++it)
    {
        if (Pane* pPane = it->FindPaneByNameRecursive(findName))
        {
            return pPane;
        }
    }

    return NULL;
}

Material* Pane::FindMaterialByNameRecursive(const char* findName)
{
    u32 nbMaterial = GetMaterialNum();
    for (u32 idx = 0; idx < nbMaterial; ++idx)
    {
        Material* pMaterial = GetMaterial(idx);
        if (pMaterial)
        {
            if (internal::EqualsMaterialName(pMaterial->GetName(), findName))
            {
                return pMaterial;
            }
        }
    }

    // 子供に一致するものが無いか検索
    PaneList::Iterator it_end = m_ChildList.GetEndIter();
    for (PaneList::Iterator it = m_ChildList.GetBeginIter(); it != it_end; ++it)
    {
        if (Material* pMat = it->FindMaterialByNameRecursive(findName))
        {
            return pMat;
        }
    }

    return 0;
}

void Pane::CalcScaleFromPartsRoot(math::VEC2* scale, Pane* pane)
{
    scale->Set(1.f, 1.f);
    while (pane && ! nw::ut::IsTypeOf<nw::lyt::Parts>(pane)) {
        scale->Set(scale->x * pane->GetScale().x, scale->y * pane->GetScale().y);
        pane = pane->GetParent();
    }
}

Pane::CalculateMtxContext::CalculateMtxContext()
 : rectDrawer(NULL)
 , viewMtx(NULL)
 , locationAdjustScale(1.f, 1.f)
 , influenceAlpha(1.f)
 , isLocationAdjust(false)
 , isInvisiblePaneCalculateMtx(false)
 , isAlphaZeroPaneCalculateMtx(false)
 , isInfluenceAlpha(false)
 , pLayout(NULL)
{
}

Pane::CalculateMtxContext::CalculateMtxContext(const DrawInfo& drawInfo, const Layout* layout)
 : rectDrawer(&drawInfo.GetGraphicsResource()->GetFontDrawer())
 , viewMtx(&drawInfo.GetViewMtx())
 , locationAdjustScale(drawInfo.GetLocationAdjustScale())
 , influenceAlpha(1.f)
 , isLocationAdjust(drawInfo.IsLocationAdjust())
 , isInvisiblePaneCalculateMtx(drawInfo.IsInvisiblePaneCalculateMtx())
 , isAlphaZeroPaneCalculateMtx(drawInfo.IsAlphaZeroPaneCalculateMtx())
 , isInfluenceAlpha(false)
 , pLayout(layout)
{
}

} // namespace nw::lyt
} // namespace nw
