﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_DrawInfo.h>

#include <nw/lyt/lyt_GraphicsResource.h>

#if defined(NW_PLATFORM_CAFE)
#include <nw/lyt/lyt_Layout.h>
#endif

namespace nw
{
namespace lyt
{

//----------------------------------------
DrawInfo::DrawInfo()
: m_LocationAdjustScale(1.f, 1.f)
, m_pGraphicsResource(NULL)
, m_pLayout(NULL)
, m_NumTexEnv(0)
, m_NumTexMap(0)
, m_TexEnvAuto(false)
, m_ModelViewMtxLoaded(false)
, m_FirstDraw(false)
, m_AllowableTextureQuantity(0)
#if defined(NW_PLATFORM_CAFE)
, m_CurrentShader(NULL)
#endif
{
    std::memset(&m_Flag, 0, sizeof(m_Flag));

    math::MTX44Identity(&m_ProjMtx);
    math::MTX34Identity(&m_ViewMtx);
    math::MTX34Identity(&m_ModelViewMtx);
}

//----------------------------------------
DrawInfo::~DrawInfo()
{
}

//----------------------------------------
void
DrawInfo::SetProjectionMtx(const nw::math::MTX44& mtx)
{
    // 直ちにシェーダに反映させると複数の DrawInfo がある場合に
    // 誤作動するので、ここでは値を保持するのみ。
    // 実際の設定は Layout::Draw() で行う。
    math::MTX44Copy(&m_ProjMtx, &mtx);
}

//----------------------------------------
void
DrawInfo::SetupProgram()
{
    if (m_FirstDraw)
    {
        m_FirstDraw = false;
        m_pGraphicsResource->ActiveVBO();
    }
}

//########################################
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
//########################################

//----------------------------------------
void
DrawInfo::LoadProjectionMtx()
{
    NW_ASSERT(m_pGraphicsResource->Initialized());

    glUseProgram(m_pGraphicsResource->m_GlProgram);
    glUniformMatrix4fv(GetUniformId(UNIFORM_uProjection), 1, GL_FALSE, m_ProjMtx.a);
    NW_GL_ASSERT();
}

//----------------------------------------
void
DrawInfo::LoadMtxModelView()
{
    if (!m_ModelViewMtxLoaded)
    {
        m_ModelViewMtxLoaded = true;
        gfnd::ShaderUniformId id = GetUniformId(UNIFORM_uModelView);
        glUniform4fv(id, 3, m_ModelViewMtx.a);
    }
}

//----------------------------------------
void
DrawInfo::SetShader(int id)
{
    NW_UNUSED_VARIABLE(id);
    glUseProgram(m_pGraphicsResource->m_GlProgram);
    m_ModelViewMtxLoaded = false;
}

//----------------------------------------
gfnd::ShaderUniformId
DrawInfo::GetUniformId(int uniformName) const
{
    return m_pGraphicsResource->m_UniformIds[uniformName];
}

//########################################
#elif defined(NW_PLATFORM_CAFE)
//########################################

//----------------------------------------
void
DrawInfo::LoadProjectionMtx()
{
    NW_ASSERT(m_pGraphicsResource->Initialized());

    // GX2: プロジェクションマトリクス設定
    GX2SetVertexUniformReg(GetUniformId(UNIFORM_uProjection), 4 * 4, m_ProjMtx);
}

//----------------------------------------
void
DrawInfo::LoadMtxModelView()
{
    if (!m_ModelViewMtxLoaded)
    {
        m_ModelViewMtxLoaded = true;
        // GX2: モデルビューマトリクス設定
        GX2SetVertexUniformReg(GetUniformId(UNIFORM_uModelView), 3 * 4, m_ModelViewMtx);
    }
}

//----------------------------------------
void
DrawInfo::SetShader(int id)
{
    m_CurrentShader = m_pGraphicsResource->m_ShaderSlot.GetConnection(id);
    m_pGraphicsResource->m_ShaderSlot.SetToDevice(m_CurrentShader);
    m_ModelViewMtxLoaded = false;
}

//----------------------------------------
void
DrawInfo::SetArchiveShader(const ArchiveShaderInfo* pShaderInfo, s32 shaderId)
{
    m_CurrentShader = pShaderInfo->GetConnection(shaderId);
    m_pGraphicsResource->m_ShaderSlot.SetToDevice(m_CurrentShader);
    m_ModelViewMtxLoaded = false;
}

//----------------------------------------
gfnd::ShaderUniformId
DrawInfo::GetUniformId(int uniformName) const
{
    return m_pGraphicsResource->GetShaderSlot().GetUniformId(
        m_CurrentShader, static_cast<Uniform>(uniformName));
}

//########################################
#else
//########################################
#error "Not support platform."
#endif

} // namespace nw::lyt
} // namespace nw
