﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_Animator.h>
#include <nw/lyt/lyt_Util.h>
#include <nw/lyt/lyt_Group.h>
#include <nw/lyt/lyt_Pane.h>
#include <nw/lyt/lyt_Resources.h>

namespace nw
{
namespace lyt
{

Animator::Animator()
 : m_Speed(0.f)
 , m_PlayType(PLAYTYPE_ONESHOT)
{
}

Animator::~Animator()
{
}

void Animator::Play(PlayType type, f32 speed)
{
    if ((- GetFrameMax() <= speed && speed <= GetFrameMax()) || (IsWaitData() && type == PLAYTYPE_ONESHOT)) {
        m_PlayType = type;
        m_Speed = speed;
        ClearPlayStatusFlag();

        if (speed >= 0) {
            SetFrame(0.f);
        } else {
            SetFrame(GetFrameMax());
        }
    } else {
        if (IsWaitData()) {
            NW_ERR("wait animation play type must be PLAYTYPE_ONESHOT");
        } else {
            NW_ERR("play speed[%f] exceeds MaxFrame", speed);
        }
    }
}

void Animator::PlayAuto(f32 speed)
{
    Play(IsLoopData() ? PLAYTYPE_LOOP : PLAYTYPE_ONESHOT, speed);
}

void Animator::PlayFromCurrent(PlayType type, f32 speed)
{
    if ((- GetFrameMax() <= speed && speed <= GetFrameMax()) || (IsWaitData() && type == PLAYTYPE_ONESHOT)) {
        m_PlayType = type;
        m_Speed = speed;
        ClearPlayStatusFlag();
    } else {
        if (IsWaitData()) {
            NW_ERR("wait animation play type must be PLAYTYPE_ONESHOT");
        } else {
            NW_ERR("play speed[%f] exceeds MaxFrame", speed);
        }
    }
}

void Animator::Stop(f32 frame)
{
    NW_ASSERT_MINMAX(frame, 0.f, GetFrameMax());
    m_Speed = 0.f;
    SetFrame(frame);
    ClearPlayStatusFlag();
}

void Animator::StopCurrent()
{
    m_Speed = 0.f;
    ClearPlayStatusFlag();
}

void Animator::StopAtMin()
{
    m_Speed = 0.f;
    SetFrame(0.f);
    ClearPlayStatusFlag();
}

void Animator::StopAtMax()
{
    m_Speed = 0.f;
    SetFrame(GetFrameMax());
    ClearPlayStatusFlag();
}

void Animator::UpdateFrame(f32 progress_frame)
{
    ClearPlayStatusFlag();

    if (m_Speed == 0.f) {
        // 再生スピードが0の場合はここで抜ける
        return;
    }

    f32 next_frame = GetFrame() + m_Speed * progress_frame;

    if ( m_Speed > 0 ) {
        if (next_frame >= GetFrameMax()) {
            switch( m_PlayType ) {
            case PLAYTYPE_ONESHOT:
                next_frame = GetFrameMax();
                m_Speed = 0.f;
                m_Flag.SetMaskOn(MASK_PLAY_STATUS_END);
                break;
            case PLAYTYPE_LOOP:
                next_frame = (next_frame - GetFrameMax());
                m_Flag.SetMaskOn(MASK_PLAY_STATUS_TURNED_MAX);
                break;
            case PLAYTYPE_ROUNDTRIP:
                next_frame = GetFrameMax() - (next_frame - GetFrameMax());
                m_Speed *= -1;
                m_Flag.SetMaskOn(MASK_PLAY_STATUS_TURNED_MAX);
                break;
            }
        }
    } else {
        if ( next_frame <= 0.f ) {
            switch( m_PlayType ) {
            case PLAYTYPE_ONESHOT:
                next_frame = 0.f;
                m_Speed = 0.f;
                m_Flag.SetMaskOn(MASK_PLAY_STATUS_END);
                break;
            case PLAYTYPE_LOOP:
                next_frame = next_frame + GetFrameMax();
                m_Flag.SetMaskOn(MASK_PLAY_STATUS_TURNED_MIN);
                break;
            case PLAYTYPE_ROUNDTRIP:
                next_frame = - next_frame;
                m_Speed *= -1;
                m_Flag.SetMaskOn(MASK_PLAY_STATUS_TURNED_MIN);
                break;
            }
        }
    }

    SetFrame(next_frame);
}

void PaneAnimator::Setup(Pane* pane, bool enable)
{
    BindPane(pane, false);
    mPane = pane;
    SetEnable(enable);
}

void PaneAnimator::Unbind()
{
    UnbindPane(mPane);
}

void GroupAnimator::Setup(Group* group, bool enable)
{
    BindGroup(group);
    mGroup = group;
    SetEnable(enable);
}

void GroupAnimator::Unbind()
{
    UnbindGroup(mGroup);
}

void GroupAnimator::Setup(const AnimResource& animRes, GroupContainer* groupContainer, u32 index, bool enable)
{
    const u32 bindGroupNum = animRes.GetGroupNum();
    if (index < bindGroupNum) {
        Setup(groupContainer->FindGroupByName(animRes.GetGroupArray()[index].GetName()), enable);
    } else {
        // index番目のグループが存在しない
        NW_ERR("index[%d] exceeds group num[%d] of AnimResource[%s]\n", index, bindGroupNum, animRes.GetTagName());
    }
}

void GroupArrayAnimator::Setup(const AnimResource& animRes, GroupContainer* groupContainer, Group** groupBuffer, bool enable)
{
    mGroupNum = animRes.GetGroupNum();
    mGroups = groupBuffer;
    for (u32 i = 0; i < mGroupNum; i++) {
        mGroups[i] = groupContainer->FindGroupByName(animRes.GetGroupArray()[i].GetName());
        BindGroup(mGroups[i]);
    }
    SetEnable(enable);
}

void GroupArrayAnimator::Unbind()
{
    for (u32 i = 0; i < mGroupNum; i++) {
        UnbindGroup(mGroups[i]);
    }
}

} // namespace nw::lyt
} // namespace nw
