﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef __FSA_TYPES_H__
#define __FSA_TYPES_H__

#include <winext/types.h>
#include <winext/cafe/sys_io_error.h>

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------------------------*
 *
 *    Globals/Externs
 *    -- Variables/Functions --
 *
 *---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 *
 *    Constants defined for this file
 *    -- #Defines's --
 *
 *---------------------------------------------------------------------------*/

/* Default mount paths */
#define FSA_CONTENT_DIR                 "/vol/content"
#define FSA_SAVE_DIR                    "/vol/save"

/* path to device files directory */
#define FSA_DEVICEFILES_ROOT            "/dev"
#define FSA_DEVICEFILES_ROOT_NAMESIZE   4

/* max length of path under mount point
 * should be the least value among all file systems */
#define FSA_MAX_LOCALPATH_SIZE      511
#define FSA_MAX_LOCALPATH_SIZE_PAD  1

/* max length of mount point path */
#define FSA_MAX_MOUNTPATH_SIZE      128
#define FSA_MAX_MOUNTPATH_SIZE_PAD  0

/* max length of full path (mount path + FS-local path) */
#define FSA_MAX_FULLPATH_SIZE       (FSA_MAX_LOCALPATH_SIZE + FSA_MAX_MOUNTPATH_SIZE)
#define FSA_MAX_FULLPATH_SIZE_PAD   (FSA_MAX_LOCALPATH_SIZE_PAD + FSA_MAX_MOUNTPATH_SIZE_PAD)

/* max length of path strings that user passes in arguments
 * should be the least value among all file systems */
#define FSA_MAX_ARGPATH_SIZE        FSA_MAX_FULLPATH_SIZE
#define FSA_MAX_ARGPATH_SIZE_PAD    FSA_MAX_FULLPATH_SIZE_PAD

/* max length of file/dir name
 * should be the least value among all file systems */
#define FSA_MAX_ENTNAME_SIZE        256
#define FSA_MAX_ENTNAME_SIZE_PAD    0

/* mode string in open file command parameters */
#define FSA_MAX_MODE_SIZE           16
#define FSA_MAX_MODE_SIZE_PAD       0

/* device type name in FSAVolumeInfo */
#define FSA_MAX_DEVTYPE_SIZE        8
#define FSA_MAX_DEVTYPE_SIZE_PAD    0

/* max length of device path, which consists of:
 *   [device file directory path] + ['/']
 *    + [device file name (device type + 2 digit index)] + [Termination '\0'] */
#define FSA_MAX_DEVICEPATH_SIZE     (FSA_DEVICEFILES_ROOT_NAMESIZE + 1 + \
                                     FSA_MAX_DEVTYPE_SIZE + 2 + 1)
#define FSA_MAX_DEVICEPATH_SIZE_PAD (16 - FSA_MAX_DEVICEPATH_SIZE)

/* file system name in FSAVolumeInfo */
#define FSA_MAX_FS_NAME_SIZE        8
#define FSA_MAX_FS_NAME_SIZE_PAD    0

/* volume label in FSAVolumeInfo */
#define FSA_MAX_VOL_LABEL_SIZE      128
#define FSA_MAX_VOL_LABEL_SIZE_PAD  0

/* volume id in FSAVolumeInfo */
#define FSA_MAX_VOL_ID_SIZE         128
#define FSA_MAX_VOL_ID_SIZE_PAD     0

/* mount option flag */
#define FSA_MNT_FLAG_RONLY          0x80000000 /* mount volume as "read only" */
#define FSA_MNT_FLAG_EXCL           0x40000000 /* prohibit access to this volume by other clients */
#define FSA_MNT_FLAG_NODEV          0x20000000 /* prohibit access to get volume info through this mount point */
#define FSA_MNT_FLAG_DEFAULT        0x00000000 /* default mount flag set */

/* unmount option flag */
#define FSA_UMNT_FLAG_FORCE         0x80000000 /* force unmount while the device is detached */
#define FSA_UMNT_FLAG_DEFAULT       0x00000000 /* default unmount flag set */

/* read/write option flag */
#define FSA_RW_FLAG_WITH_POS        0x00000001 /* enables fpos argument (automatically set in APIs) */
#define FSA_READ_FLAG_DEFAULT       0x00000000 /* default read flag set */
#define FSA_WRITE_FLAG_DEFAULT      0x00000000 /* default write flag set */

/* media state information indicated at .media_state in FSAVolumeInfo */
#define FSA_MEDIA_READY             0
#define FSA_MEDIA_NO_MEDIA          1
#define FSA_MEDIA_INVALID_MEDIA     2
#define FSA_MEDIA_DIRTY_MEDIA       3
#define FSA_MEDIA_FATAL             4

/* flag in volume information */
#define FSA_VOL_FLAG_RONLY              0x80000000  /* read only or write protected volume */
#define FSA_VOL_FLAG_SPRT_VID           0x40000000  /* supports volume ID */

/* directory entry stat flag */
#define FSA_STAT_ATTRIBUTES_SIZE            (48)        /* size of FS-specific attributes field */
#define FSA_STAT_FLAG_IS_DIRECTORY          0x80000000  /* entry is directory */
#define FSA_STAT_FLAG_IS_QUOTA              0x40000000  /* entry is quota */

#define FSA_STAT_FLAG_SPRT_QUOTA_SIZE       0x20000000  /* FS supports .quota_size field */
#define FSA_STAT_FLAG_SPRT_ENTID            0x10000000  /* FS supports .ent_id field */
#define FSA_STAT_FLAG_SPRT_CTIME            0x08000000  /* FS supports .ctime field */
#define FSA_STAT_FLAG_SPRT_MTIME            0x04000000  /* FS supports .mtime field */
#define FSA_STAT_FLAG_SPRT_ATTRIBUTES       0x02000000  /* FS supports .attributes field */
#define FSA_STAT_FLAG_SPRT_FILE_ALLOC_SIZE  0x01000000  /* FS supports .alloc_size field */
#define FSA_STAT_FLAG_SPRT_DIR_SIZE         0x00100000  /* FS supports .size field for directory
                                                         * this field does not have meaning for file */

/* file system flag */
#define FSA_FS_FLAG_RONLY                  0x80000000   /* File system has only read feature. */
#define FSA_FS_FLAG_SPRT_QUOTA             0x40000000   /* File system support quota feature. */
#define FSA_FS_FLAG_SPRT_APPEND            0x20000000   /* File system support file append feature. */
#define FSA_FS_FLAG_SPRT_JOURNALING_VOL    0x10000000   /* File system support journaling feature for each volume. (WFS) */
#define FSA_FS_FLAG_SPRT_CACHE_FILE        0x08000000   /* File system support cache feature for each file. (ISFS) */
#define FSA_FS_FLAG_SPRT_REMOVE_NOEMP_DIR  0x04000000   /* File system support remove feature for not empty directory. */
#define FSA_FS_FLAG_SPRT_CHANGE_FILE_NAME  0x02000000   /* File system support changing file name by FSARename(). */

/* device flag */
#define FSA_DEV_FLAG_RONLY                 0x80000000   /* Read only or write protected device. */
#define FSA_DEV_FLAG_SPRT_ID               0x40000000   /* Vendor/Product ID field is valid to identify vendor and production of media. */
#define FSA_DEV_FLAG_SPRT_LOG_BLK          0x20000000   /* Logical block field is valid to know the specific of media. */
#define FSA_DEV_FLAG_SINGULAR              0x10000000   /* Device has only one physical detachable port */
#define FSA_DEV_FLAG_DETACHABLE            0x08000000   /* Device is detachable. */

/* directory entry permission flag */
#define FSA_STAT_PERM_CHANGE_SIZE           0x0001
#define FSA_STAT_PERM_WRITE                 0x0002
#define FSA_STAT_PERM_READ                  0x0004
#define FSA_STAT_PERM_ADD_OR_DELETE_FILE    0x0008
#define FSA_STAT_PERM_ADD_OR_DELETE_SUBDIR  0x0010
#define FSA_STAT_PERM_LIST                  0x0020
#define FSA_STAT_PERM_DELETE_OR_MOVE        0x0040

#define FSA_STAT_PERM_RESERVED0             0x0080
#define FSA_STAT_PERM_RESERVED1             0x0100
#define FSA_STAT_PERM_RESERVED2             0x0200
#define FSA_STAT_PERM_RESERVED3             0x0400
#define FSA_STAT_PERM_RESERVED4             0x0800

/* Permission mode flag */
#define FSA_MODE_IRUSR          0x000400 /* read bit for owner */
#define FSA_MODE_IWUSR          0x000200 /* write bit for owner */
#define FSA_MODE_IRGRP          0x000040 /* read bit for owner's group */
#define FSA_MODE_IWGRP          0x000020 /* write bit for owner's group */
#define FSA_MODE_IROTH          0x000004 /* read bit for others */
#define FSA_MODE_IWOTH          0x000002 /* write bit for others */

/* read/write bits for owner/owner's group/others */
#define FSA_MODE_IRWU           (FSA_MODE_IRUSR | FSA_MODE_IWUSR)
#define FSA_MODE_IRWG           (FSA_MODE_IRGRP | FSA_MODE_IWGRP)
#define FSA_MODE_IRWO           (FSA_MODE_IROTH | FSA_MODE_IWOTH)

#define FSA_MODE_MASK           (FSA_MODE_IRWU | FSA_MODE_IRWG | FSA_MODE_IRWO)
#define FSA_MODE_RESERVED       (~FSA_MODE_MASK)

/* device type names, used in device file name */
#define FSA_DEV_TYPE_NAME_SI                "si"        /* SI interface */
#define FSA_DEV_TYPE_NAME_ODD               "odd"       /* optical disc drive */
#define FSA_DEV_TYPE_NAME_SLC_COMPAT        "slccmpt"   /* NAND SLC (Wii compat region) */
#define FSA_DEV_TYPE_NAME_SLC               "slc"       /* NAND SLC (CAFE region) */
#define FSA_DEV_TYPE_NAME_MLC               "mlc"       /* NAND MLC */
#define FSA_DEV_TYPE_NAME_SD                "sd"        /* SD (Devkit) */
#define FSA_DEV_TYPE_NAME_SD_CARD           "sdcard"    /* SD card */
#define FSA_DEV_TYPE_NAME_USB               "usb"       /* USB */
#define FSA_DEV_TYPE_NAME_HOST_FILEIO       "hfio"      /* Host File IO */
#define FSA_DEV_TYPE_NAME_RAMDISK           "ramdisk"   /* RAMDISK */
#define FSA_DEV_TYPE_NAME_UNKNOWN           "unknown"   /* unknown bus */

/* FSAInfoQuery */
enum {
    FSA_INFO_QUERY_FREE_SPACE_SIZE, // Free space in quota
    FSA_INFO_QUERY_DIR_SIZE,        // Directory size (include sub directories)
    FSA_INFO_QUERY_ENTRY_NUM,       // Number of entries in directory
    FSA_INFO_QUERY_FILESYSTEM,      // File system information
    FSA_INFO_QUERY_DEVICE,          // Device information
    FSA_INFO_QUERY_STAT,            // File stat
    FSA_INFO_QUERY_BAD_BLOCK,       // Bad block information of device

    FSA_INFO_QUERY_MAX
};

/* max number of error reports it can remember */
#define FSA_MAX_ERROR_REPORTS               128

/* values for attach notification */
#define FSA_DETACH                          0
#define FSA_ATTACH                          1

/* invalid handle value for FSAOpenFile/FSAOpenDir */
#define FSA_INVALID_HANDLE_VALUE            -1

/* FSASetCode */
enum {
    FSA_LOCAL_CODE_ASCII = 1,       // ASCII
    FSA_LOCAL_CODE_CP932,           // CP932 -- Shift JIS
    FSA_LOCAL_CODE_CP950,           // CP950  -- BIG_5
    FSA_LOCAL_CODE_CP936,           // CP936  -- GBK
    FSA_LOCAL_CODE_CP1252,          // CP1252 -- LATIN_1
    FSA_LOCAL_CODE_CP1250,          // CP1250 -- LATIN_2
    FSA_LOCAL_CODE_CP874,           // CP874  -- THAI
    FSA_LOCAL_CODE_CP949,           // CP949  -- KOREAN
    FSA_LOCAL_CODE_CP1251,          // CP1251 -- CYRILLIC
    FSA_LOCAL_CODE_CP1256,          // CP1256 -- ARABIC
    FSA_LOCAL_CODE_MAX
};

/*---------------------------------------------------------------------------*
 *
 *    Data types defined for this file
 *    -- Struct's, Typedef's, Enum's --
 *
 *---------------------------------------------------------------------------*/

/* FSA Commands */
#define FSA_COMMAND_INVALID             0
#define FSA_COMMAND_MOUNT               1
#define FSA_COMMAND_UNMOUNT             2
#define FSA_COMMAND_GET_VOLUME_INFO     3
#define FSA_COMMAND_GET_ATTACH          4
#define FSA_COMMAND_CHANGE_DIR          5
#define FSA_COMMAND_GET_CWD             6
#define FSA_COMMAND_MAKE_DIR            7
#define FSA_COMMAND_REMOVE              8
#define FSA_COMMAND_RENAME              9
#define FSA_COMMAND_OPEN_DIR            10
#define FSA_COMMAND_READ_DIR            11
#define FSA_COMMAND_REWIND_DIR          12
#define FSA_COMMAND_CLOSE_DIR           13
#define FSA_COMMAND_OPEN_FILE           14
#define FSA_COMMAND_READ_FILE           15 /* TODO: Separate this for each derivative function */
#define FSA_COMMAND_WRITE_FILE          16 /* TODO: Separate this for each derivative function */
#define FSA_COMMAND_GET_POS_FILE        17
#define FSA_COMMAND_SET_POS_FILE        18
#define FSA_COMMAND_IS_EOF              19
#define FSA_COMMAND_STAT_FILE           20
#define FSA_COMMAND_CLOSE_FILE          21
#define FSA_COMMAND_GET_ERROR           22
#define FSA_COMMAND_FLUSH_FILE          23
#define FSA_COMMAND_GET_INFO_BY_QUERY   24 /* TODO: Separate this for each derivative function */
#define FSA_COMMAND_APPEND_FILE         25
#define FSA_COMMAND_TRUNCATE_FILE       26
#define FSA_COMMAND_FLUSH_VOLUME        27
#define FSA_COMMAND_ROLLBACK_VOLUME     28
#define FSA_COMMAND_MAKE_QUOTA          29
#define FSA_COMMAND_FLUSH_QUOTA         30
#define FSA_COMMAND_ROLLBACK_QUOTA      31
#define FSA_COMMAND_CHANGE_MODE         32
#define FSA_COMMAND_OPEN_FILE_BY_STAT   33


/* structures passed between CPUs must be packed */
PACKED_STRUCT_BEGIN

/* Common primitives */
typedef s32                 FSACommand;
typedef s32                 FSAError;
typedef s32                 FSAStatus;
typedef s32                 FSAClientHandle;
typedef s32                 FSADirHandle;
typedef s32                 FSAFileHandle;
typedef u32                 FSASize;
typedef u64                 FSABSize;
typedef u32                 FSACount;
typedef u32                 FSAFlag;
typedef u32                 FSAPermission;
typedef u32                 FSAMode;
typedef u32                 FSAFilePosition;
typedef u64                 FSATime;
typedef u32                 FSAEntryNum;
typedef u32                 FSAInfoQuery;
typedef u32                 FSAUserId;
typedef u32                 FSAGroupId;
typedef u32                 FSACharCode;

typedef FSABSize            FSAFreeSpaceSize;
typedef FSABSize            FSADirSize;

/* Volume information */
typedef struct
{
    /* bit map flags indicating properties of this volume (see FSA_VOL_FLAG_x) */
    u32         flag;

    /* media status (see FSA_MEDIA_x) */
    u32         media_state;

    /* index to identify devices of same device type */
    s32         dev_index;

    /* block size depending on block device */
    FSASize     blocksize; // *** NO LONGER BE USED ***

    /* logical block size depending on file system */
    FSASize     fs_blocksize; // *** NO LONGER BE USED ***

    /* required alignment for read buffer [bytes] */
    FSASize     buf_align_r; // *** NO LONGER BE USED ***

    /* required alignment for write buffer [bytes] */
    FSASize     buf_align_w; // *** NO LONGER BE USED ***

    /* device type name (see FSA_DEV_TYPE_NAME_x) */
    char        dev_type[FSA_MAX_DEVTYPE_SIZE + FSA_MAX_DEVTYPE_SIZE_PAD];

    /* file system name */
    char        fs_name[FSA_MAX_FS_NAME_SIZE + FSA_MAX_FS_NAME_SIZE_PAD];

    /* volume label */
    char        vol_label[FSA_MAX_VOL_LABEL_SIZE + FSA_MAX_VOL_LABEL_SIZE_PAD];

    /* volume ID */
    char        vol_id[FSA_MAX_VOL_ID_SIZE + FSA_MAX_VOL_ID_SIZE_PAD];

    /* device file path */
    char        dev_path[FSA_MAX_DEVICEPATH_SIZE + FSA_MAX_DEVICEPATH_SIZE_PAD];

    /* mount path (null string while not mounted) */
    char        mnt_path[FSA_MAX_MOUNTPATH_SIZE + FSA_MAX_MOUNTPATH_SIZE_PAD]; // *** NO LONGER BE USED ***
} PACKED_STRUCT_ATTRIBUTE FSAVolumeInfo;

/* File/Dir status */
typedef struct
{
    FSAFlag         flag;           /* flag bitmap */

    FSAPermission   permission;     /* access permissions */

    FSAUserId       owner_id;       /* owner title id */

    FSAGroupId      group_id;       /* owner group id */

    FSASize         size;           /* file size in bytes.
                                       valid when entry is file. */
    FSASize         alloc_size;     /* allocated file size in bytes.
                                       valid when entry is file and FSA_STAT_FLAG_SPRT_FILE_ALLOC_SIZE in flag is set */
    FSABSize        quota_size;     /* total size of quota/partition to which the entry belongs
                                       valid when FSA_STAT_FLAG_SPRT_DIR_SIZE in flag is set */
    u32             ent_id;         /* entry ID. valid when FSA_STAT_FLAG_SPRT_ENTID in flag is set */

    FSATime         ctime;          /* last created time
                                       valid when FSA_STAT_FLAG_SPRT_CTIME in flag is set */
    FSATime         mtime;          /* last modified time
                                       valid when FSA_STAT_FLAG_SPRT_MTIME in flag is set */

    u8              attributes[FSA_STAT_ATTRIBUTES_SIZE];
                                    /* FS-specific attribute field
                                       valid when FSA_STAT_FLAG_SPRT_ATTRIBUTES in flag is set */
} PACKED_STRUCT_ATTRIBUTE FSAStat;

typedef struct
{
    /* maximum absolute path length */
    u16         abs_path_len;

    /* maximum entry length */
    u16         entry_len;

    /* maximum path depth */
    u16         path_depth;

    /* maximum number of file handle */
    u16         fh_num;

    /* maximum number of directory handle */
    u16         dh_num;

    /* maximum file size */
    FSASize     file_size;

    /* bit map flags indicating properties of this file system (see FSA_FS_FLAG_x) */
    u32         flag;

    /* logical block size depending on file system */
    FSASize     fs_blk_size;

    /* file system name */
    char        fs_name[FSA_MAX_FS_NAME_SIZE + FSA_MAX_FS_NAME_SIZE_PAD];

} PACKED_STRUCT_ATTRIBUTE FSAFileSystemInfo;

typedef struct
{
    /* vendor id */
    u16         vendor_id;

    /* product id */
    u16         product_id;

    /* bit map flags indicating properties of this device (see FSA_DEV_FLAG_x) */
    u32         flag;

    /* number of total logical blocks */
    u64         log_blk_num;

    /* logical block size depending on device */
    FSASize     log_blk_size;

    /* required alignment for read buffer [bytes] */
    FSASize     buf_align_r;

    /* required alignment for write buffer [bytes] */
    FSASize     buf_align_w;

    /* device type name (see FSA_DEV_TYPE_NAME_x) */
    char        dev_type[FSA_MAX_DEVTYPE_SIZE + FSA_MAX_DEVTYPE_SIZE_PAD];

} PACKED_STRUCT_ATTRIBUTE FSADeviceInfo;

typedef struct
{
    /* number of total blocks depending on bad block management of device */
    u64         blk_num;

    /* number of bad blocks */
    u64         bad_blk_num;

    /* block size depending on bad block management of device */
    FSASize     blk_size;

} PACKED_STRUCT_ATTRIBUTE FSABadBlockInfo;

/* Directory entry */
typedef struct
{
    FSAStat     stat;
    char        name[FSA_MAX_ENTNAME_SIZE + FSA_MAX_ENTNAME_SIZE_PAD];
} PACKED_STRUCT_ATTRIBUTE FSADirEntry;

/* Error report */
typedef struct
{
    FSACommand  command;            /* command */
    FSAError    error;              /* result code */
} PACKED_STRUCT_ATTRIBUTE FSAErrorReport;

/* Data types for the FSA_COMMAND_MOUNT command */
typedef struct
{
    char     source[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
    char     target[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
    FSAFlag  flag;
    void     *p_fs_args;
    FSASize  fs_args_size;
} PACKED_STRUCT_ATTRIBUTE FSAMountRequestData;

/* Data types for the FSA_COMMAND_UNMOUNT command */
typedef struct
{
    char target[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
    FSAFlag  flag;
} PACKED_STRUCT_ATTRIBUTE FSAUnmountRequestData;

/* Data types for the FSA_COMMAND_GET_VOLUME_INFO command */
typedef struct
{
    char path[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
} PACKED_STRUCT_ATTRIBUTE FSAGetVolumeInfoRequestData;
typedef struct
{
    FSAVolumeInfo volume_info;
} PACKED_STRUCT_ATTRIBUTE FSAGetVolumeInfoResponseData;

/* Data types for the FSA_COMMAND_GET_ATTACH command */
typedef struct
{
    s32           attach;
    FSAVolumeInfo volume_info;
} PACKED_STRUCT_ATTRIBUTE FSAGetAttachResponseData;

/* Data types for the FSA_COMMAND_CHANGE_DIR command */
typedef struct
{
    char path[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
} PACKED_STRUCT_ATTRIBUTE FSAChangeDirRequestData;

/* Data types for the FSA_COMMAND_CHANGE_DIR command */
typedef struct
{
    char path[FSA_MAX_FULLPATH_SIZE + FSA_MAX_FULLPATH_SIZE_PAD];
} PACKED_STRUCT_ATTRIBUTE FSAGetCwdResponseData;

/* Data types for the FSA_COMMAND_MAKE_DIR command */
typedef struct
{
    char    path[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
    FSAMode mode;
} PACKED_STRUCT_ATTRIBUTE FSAMakeDirRequestData;

/* Data types for the FSA_COMMAND_OPEN_DIR command */
typedef struct
{
    char path[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
} PACKED_STRUCT_ATTRIBUTE FSAOpenDirRequestData;
typedef struct
{
    FSADirHandle dir_handle;
} PACKED_STRUCT_ATTRIBUTE FSAOpenDirResponseData;

/* Data types for the FSA_COMMAND_READ_DIR command */
typedef struct
{
    FSADirHandle dir_handle;
} PACKED_STRUCT_ATTRIBUTE FSAReadDirRequestData;
typedef struct
{
    FSADirEntry dir_entry;
} PACKED_STRUCT_ATTRIBUTE FSAReadDirResponseData;

/* Data types for the FSA_COMMAND_REWIND_DIR command */
typedef struct
{
    FSADirHandle dir_handle;
} PACKED_STRUCT_ATTRIBUTE FSARewindDirRequestData;

/* Data types for the FSA_COMMAND_CLOSE_DIR command */
typedef struct
{
    FSADirHandle dir_handle;
} PACKED_STRUCT_ATTRIBUTE FSACloseDirRequestData;

/* Data types for the FSA_COMMAND_OPEN_FILE command */
typedef struct
{
    char path[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
    char mode[FSA_MAX_MODE_SIZE + FSA_MAX_MODE_SIZE_PAD];
} PACKED_STRUCT_ATTRIBUTE FSAOpenFileRequestData;
typedef struct
{
    FSAFileHandle file_handle;
} PACKED_STRUCT_ATTRIBUTE FSAOpenFileResponseData;

/* Data types for the FSA_COMMAND_READ_FILE command */
typedef struct
{
    void            *destination_address;
    FSASize         size;
    FSACount        count;
    FSAFilePosition fpos;
    FSAFileHandle   file_handle;
    FSAFlag         flag;
} PACKED_STRUCT_ATTRIBUTE FSAReadFileRequestData;

/* Data types for the FSA_COMMAND_WRITE_FILE command */
typedef struct
{
    const void      *source_address;
    FSASize         size;
    FSACount        count;
    FSAFilePosition fpos;
    FSAFileHandle   file_handle;
    FSAFlag         flag;
} PACKED_STRUCT_ATTRIBUTE FSAWriteFileRequestData;

/* Data types for the FSA_COMMAND_GET_POS_FILE command */
typedef struct
{
    FSAFileHandle file_handle;
} PACKED_STRUCT_ATTRIBUTE FSAGetPosFileRequestData;
typedef struct
{
    FSAFilePosition fpos;
} PACKED_STRUCT_ATTRIBUTE FSAGetPosFileResponseData;

/* Data types for the FSA_COMMAND_SET_POS_FILE command */
typedef struct
{
    FSAFileHandle file_handle;
    FSAFilePosition fpos;
} PACKED_STRUCT_ATTRIBUTE FSASetPosFileRequestData;

/* Data types for the FSA_COMMAND_IS_EOF command */
typedef struct
{
    FSAFileHandle file_handle;
} PACKED_STRUCT_ATTRIBUTE FSAIsEofRequestData;

/* Data types for the FSA_COMMAND_STAT_FILE command */
typedef struct
{
    FSAFileHandle file_handle;
} PACKED_STRUCT_ATTRIBUTE FSAStatFileRequestData;
typedef struct
{
    FSAStat    stat;
} PACKED_STRUCT_ATTRIBUTE FSAStatFileResponseData;

/* Data types for the FSA_COMMAND_FLUSH_FILE command */
typedef struct
{
    FSAFileHandle file_handle;
} PACKED_STRUCT_ATTRIBUTE FSAFlushFileRequestData;

/* Data types for the FSA_COMMAND_CLOSE_FILE command */
typedef struct
{
    FSAFileHandle file_handle;
} PACKED_STRUCT_ATTRIBUTE FSACloseFileRequestData;

/* Data types for the FSA_COMMAND_REMOVE command */
typedef struct
{
    char    path[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
} PACKED_STRUCT_ATTRIBUTE FSARemoveRequestData;

/* Data types for the FSA_COMMAND_RENAME command */
typedef struct
{
    char    oldpath[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
    char    newpath[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
} PACKED_STRUCT_ATTRIBUTE FSARenameRequestData;

/* Data types for the FSA_COMMAND_GET_ERROR command */
typedef struct
{
    FSAErrorReport  *p_dest;
    FSACount        count;
} PACKED_STRUCT_ATTRIBUTE FSAGetErrorRequestData;



/* Data types for the FSA_COMMAND_GET_INFO_BY_QUERY command */
typedef struct
{
    char path[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
    FSAInfoQuery query;
} PACKED_STRUCT_ATTRIBUTE FSAGetInfoByQueryRequestData;
typedef union
{
    FSADeviceInfo     dev_info;
    FSADirSize        dir_size;
    FSAEntryNum       entry_num;
    FSAFileSystemInfo fs_info;
    FSAFreeSpaceSize  free_size;
    FSAStat           stat;
    FSABadBlockInfo   bad_block;
} PACKED_STRUCT_ATTRIBUTE FSAInfoQueryData;
typedef struct
{
    FSAInfoQueryData data;
} PACKED_STRUCT_ATTRIBUTE FSAGetInfoByQueryResponseData;

/* Data types for the FSA_COMMAND_APPEND_FILE command */
typedef struct
{
    FSASize       size;
    FSACount      count;
    FSAFileHandle file_handle;
} PACKED_STRUCT_ATTRIBUTE FSAAppendFileRequestData;

/* Data types for the FSA_COMMAND_TRUNCATE_FILE command */
typedef struct
{
    FSAFileHandle file_handle;
} PACKED_STRUCT_ATTRIBUTE FSATruncateFileRequestData;

/* Data types for the FSA_COMMAND_FLUSH_VOLUME command */
typedef struct
{
    char path[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
} PACKED_STRUCT_ATTRIBUTE FSAFlushVolumeRequestData;

/* Data types for the FSA_COMMAND_ROLLBACK_VOLUME command */
typedef struct
{
    char path[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
} PACKED_STRUCT_ATTRIBUTE FSARollbackVolumeRequestData;

/* Data types for the FSA_COMMAND_MAKE_QUOTA command */
typedef struct
{
    char     path[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
    FSAMode  mode;
    FSABSize size;
} PACKED_STRUCT_ATTRIBUTE FSAMakeQuotaRequestData;

/* Data types for the FSA_COMMAND_FLUSH_QUOTA command */
typedef struct
{
    char path[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
} PACKED_STRUCT_ATTRIBUTE FSAFlushQuotaRequestData;

/* Data types for the FSA_COMMAND_ROLLBACK_QUOTA command */
typedef struct
{
    char path[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
} PACKED_STRUCT_ATTRIBUTE FSARollbackQuotaRequestData;

/* Data types for the FSA_COMMAND_CHANGE_MODE command */
typedef struct
{
    char           path[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
    FSAMode        mode;
    FSAMode        mask;
} PACKED_STRUCT_ATTRIBUTE FSAChangeModeRequestData;

/* Data types for the FSA_COMMAND_OPEN_FILE_BY_STAT command */
typedef struct
{
    FSAStat        stat;
    char           mode[FSA_MAX_MODE_SIZE + FSA_MAX_MODE_SIZE_PAD];
    char           path[FSA_MAX_ARGPATH_SIZE + FSA_MAX_ARGPATH_SIZE_PAD];
} PACKED_STRUCT_ATTRIBUTE FSAOpenFileByStatRequestData;
typedef struct
{
    FSAFileHandle file_handle;
} PACKED_STRUCT_ATTRIBUTE FSAOpenFileByStatResponseData;

/* all possible types of transaction parameters */
typedef union
{
    FSAMountRequestData             mount;
    FSAUnmountRequestData           unmount;
    FSAGetVolumeInfoRequestData     get_volume_info;
    FSAChangeDirRequestData         change_dir;
    FSAMakeDirRequestData           make_dir;
    FSAMakeQuotaRequestData         make_quota;
    FSAOpenDirRequestData           open_dir;
    FSAReadDirRequestData           read_dir;
    FSARewindDirRequestData         rewind_dir;
    FSACloseDirRequestData          close_dir;
    FSAOpenFileRequestData          open_file;
    FSAReadFileRequestData          read_file;
    FSAWriteFileRequestData         write_file;
    FSAGetPosFileRequestData        get_pos_file;
    FSASetPosFileRequestData        set_pos_file;
    FSAIsEofRequestData             is_eof;
    FSAStatFileRequestData          stat_file;
    FSAFlushFileRequestData         flush_file;
    FSAAppendFileRequestData        append_file;
    FSATruncateFileRequestData      truncate_file;
    FSACloseFileRequestData         close_file;
    FSARemoveRequestData            remove;
    FSARenameRequestData            rename;
    FSAFlushVolumeRequestData       flush_volume;
    FSARollbackVolumeRequestData    rollback_volume;
    FSAGetErrorRequestData          get_error;
    FSAGetInfoByQueryRequestData    get_info_by_query;
    FSAFlushQuotaRequestData        flush_quota;
    FSARollbackQuotaRequestData     rollback_quota;
    FSAChangeModeRequestData        change_mode;
    FSAOpenFileByStatRequestData    open_file_by_stat;
}PACKED_STRUCT_ATTRIBUTE FSARequest;

/* all possible types of transaction parameters */
typedef union {
    FSAGetVolumeInfoResponseData    get_volume_info;
    FSAGetAttachResponseData        get_attach;
    FSAGetCwdResponseData           get_cwd;
    FSAOpenDirResponseData          open_dir;
    FSAReadDirResponseData          read_dir;
    FSAOpenFileResponseData         open_file;
    FSAGetPosFileResponseData       get_pos_file;
    FSAStatFileResponseData         stat_file;
    FSAGetInfoByQueryResponseData   get_info_by_query;
    FSAOpenFileByStatResponseData   open_file_by_stat;
}PACKED_STRUCT_ATTRIBUTE FSAResponse;


struct FSAAsyncParams;
struct FSAAttachNoticeParams;
typedef struct FSAAsyncParams FSAAsyncParams;
typedef struct FSAAttachNoticeParams FSAAttachNoticeParams;

/* All data structures must be defined before this */
PACKED_STRUCT_END

#ifdef __cplusplus
}
#endif

#endif // __FSA_TYPES_H__
