﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_WINEXT_AX_H_
#define NW_WINEXT_AX_H_

#include <winext/cafe/os.h>

namespace nw {
namespace internal {
namespace winext {

#define AX_ENABLE_REMOTE

#define AXSYNC_ERROR_IN_USE         (-1)
#define AXSYNC_ERROR_IN_CALLBACK    (-2)
#define AXSYNC_ERROR_NOT_PROTECTED     (-3)
#define AXSYNC_ERROR_TOO_MANY_PROTECTS (-4)

// CAFE Audio API version definitions
#define CAFE_AUDIO_API_VER_2_0 20
#define CAFE_AUDIO_API_VERSION CAFE_AUDIO_API_VER_2_0
/*--------------------------------------------------------------------------*
    profiling
 *--------------------------------------------------------------------------*/
typedef struct _AXPROFILE
{

    OSTime  axFrameStart;
    OSTime  axFrameEnd;
    OSTime  userAuxProcStart;
    OSTime  userAuxProcEnd;
    OSTime  userCallbackStart;
    OSTime  userCallbackEnd;
    OSTime  userFinalMixStart;
    OSTime  userFinalMixEnd;

    u32     axNumVoices;
    u32     axNumDspVoices;
    OSTime  axDSPStart;
    OSTime  axDSPEnd;
    OSTime  axPPCStart;
    OSTime  axPPCEnd;
    OSTime  axPPCPostStart;
    OSTime  axPPCPostEnd;


    OSTime  istLatency;
    OSTime  istPostLatency;
} AXPROFILE;

/*---------------------------------------------------------------------------*
    callback interface
 *---------------------------------------------------------------------------*/
typedef void    (*AXUserCallback)   (void);
typedef void    (*AXAuxCallback)    (void *data, void *context);
typedef struct _AXAUXCBSTRUCT
{
    u32 numChs;     // number of channels pointed by s32 **data in cb
    u32 numSamples; // number of samples in each of the channels
} AXAUXCBSTRUCT;
typedef void    (*AXUserAuxCallback)    (s32 **data, void *context, AXAUXCBSTRUCT *info);
typedef void    (*AXVoiceCallback)  (void *p);
typedef void    (*AXExceedCallback) (u32 cycles);

/*-----------------------------------------------------------------------------------------
// struct for the device final mix callback
//   data ----> chBufPtrs[numChnsIn*numDevices] --> chBuf[i][j][numSamples];
//                                                  (i=0:numChnsIn-1), (j=0:numDevices-1)
-----------------------------------------------------------------------------------------*/
typedef struct _AX_FINAL_MIX_CB_STRUCT
{
    s32 **data;         // pointer to an array of pointers; each pointer points to
    u16 numChnsIn;
    u16 numSamples;
    u16 numDevices;
    u16 numChnsOut;     // when AX invokes callback, numChnsOut = numChnsIn;
                        // callback can tell AX how many of outputs to pick from processed samples
} AX_FINAL_MIX_CB_STRUCT;
typedef void    (*AXUserFinalMixCallback) (AX_FINAL_MIX_CB_STRUCT *info);


/*--------------------------------------------------------------------------*
    volume envelope
 *--------------------------------------------------------------------------*/
typedef struct _AXPBVE
{
    u16 currentVolume;
    s16 currentDelta;
}AXPBVE;
#define AXPBVE_CURRENTVOLUME_OFF  AXPBVE_OFF
#define AXPBVE_CURRENTDELTA_OFF   (AXPBVE_OFF+1)
#define AXPBVE_SIZE               2
/**********************************************************/
/* The following are the new API structure, definitions   */
/* etc                                                    */
/**********************************************************/

// basic element of the mix structure
// vol - the target volume
// delta - the ramp for value
typedef struct _AXPBCHVE
{
    u16 vol;
    s16 volDelta;
}AXPBCHVE;

// define the buses in this enum
typedef enum _AXPBBusType
{
    AX_MAIN_BUS=0,
    AX_MAX_NUM_RMT_BUSES,
    AX_AUXA_BUS = 1,
    AX_AUXB_BUS,
    AX_AUXC_BUS,
    AX_MAX_NUM_BUSES
} AXPBBusType;

// basic channel mix structures for buses
// each channel has mix gains for each of the buses
typedef struct _AXPBCHMIX
{
    AXPBCHVE bus[AX_MAX_NUM_BUSES];
} AXPBCHMIX ;

// enumerate the device types we support here
typedef enum {
    AX_DEVICE_TV=0,
    AX_DEVICE_DRC,
    AX_DEVICE_RMT,
    AX_MAX_NUM_DEVICES
} AXPBDeviceType;

// enumerate the remotes supported here
typedef enum{
    AX_RMT_ID0 = 0,
    AX_RMT_ID1,
    AX_RMT_ID2,
    AX_RMT_ID3,
    AX_MAX_NUM_RMTS
} AXPB_RMT_ID;

// enumerate the DRCs supported
typedef enum
{
    AX_DRC_ID0=0,
    AX_DRC_ID1,
    AX_MAX_NUM_DRCS
} AXPB_DRC_ID;

// enumerate the channels here
typedef enum
{
    AX_CH_LEFT=0,
    AX_MAX_NUM_RMT_CHS,
    AX_CH_RIGHT = AX_MAX_NUM_RMT_CHS,
    AX_CH_SUR_LEFT,
    AX_CH_SUR_RIGHT,
    AX_MAX_NUM_DRC_CHS,
    AX_CH_FC = AX_MAX_NUM_DRC_CHS,
    AX_CH_LFE,
    AX_MAX_NUM_TV_CHS
} AXPB_CH_ID;

// enumerate the tvs supported
typedef enum
{
    AX_TV_ID0=0,
    AX_MAX_NUM_TVS
} AXPB_TV_ID;

// enumerate the aux ids here
typedef enum
{
    AX_AUX_ID_A=0,
    AX_AUX_ID_B,
    AX_AUX_ID_C,
    AX_AUX_ID_MAX_NUM
} AXPB_AUX_ID;


// enumerate the error types here.
typedef enum{
    AXPB_ERROR_NONE=0,
    AXPB_ERROR_DEVICE_TYPE=-1,
    AXPB_ERROR_DEVICE_ID=-2,
    AXPB_ERROR_NULL_MIX_PTR=-3,
    AXPB_ERROR_NULL_VOICE_PTR=-4,
    AXPB_ERROR_AUX_ID=-5,
    AXPB_ERROR_ILLEGAL_COMPRESSOR_SETTINGS=-6,
    AXPB_ERROR_REMIX_ILLEGAL_IN_CHANNELS=-7,
    AXPB_ERROR_REMIX_ILLEGAL_OUT_CHANNELS=-8,
    AXPB_ERRIR_REMIX_EXCEED_MAX_ALLOWED=-9,
    AXPB_ERRIR_REMIX_MATRIX_UNINITIALIZED=-10,
    AXPB_ERROR_FINAL_OUTSIZE = -11,
    AXPB_ERROR_DRCVS_CONFIG = -12,
    AXPB_ERROR_DRCVS_MODE = -13,
    AXPB_ERROR_FRAMECB_TRY_AGAIN = -14,
    AXPB_ERROR_FRAMECB_MAX_DEPTH_REACHED = -15,
    AXPB_ERROR_FRAMECB_NOT_FOUND = -16,
    AXPB_ERROR_FRAMECB_NULL_FUNC = -17
} AXPB_ERROR_CODE;


/*--------------------------------------------------------------------------*
    buffer addressing
 *--------------------------------------------------------------------------*/
 // bug 568: make samples const void *
 // also moved to top of struct for other compatibility reasons
typedef struct _AXPBOFFSET {
    u16   format;           // AX_PB_FORMAT_ADPCM, AX_PB_FORMAT_PCM8, or AX_PB_FORMAT_PCM16
    u16   loopFlag;
    u32   loopOffset;       // sample offsets into samples data
    u32   endOffset;
    u32   currentOffset;
    const void *samples;    // effective address of Sample Offset 0
} AXPBOFFSET;

#define AXPBADDR_LOOP_OFF     0           // States for loopFlag field
#define AXPBADDR_LOOP_ON      1

/*--------------------------------------------------------------------------*
    ADPCM decoder state
 *--------------------------------------------------------------------------*/
typedef struct _AXPBADPCM
{

    u16     a[8][2];            //  coef table a1[0],a2[0],a1[1],a2[1]....

    u16     gain;               //  gain to be applied (0 for ADPCM, 0x0800 for PCM16 and 0x0100 for PCM8)

    u16     pred_scale;         //  predictor / scale combination (nibbles, as in hardware)
    u16     yn1;                //  y[n - 1]
    u16     yn2;                //  y[n - 2]

} AXPBADPCM;

#define AXPBADPCM_A1                  AXPBADPCM_OFF
#define AXPBADPCM_A2                  (AXPBADPCM_OFF+8)
#define AXPBADPCM_GAIN                (AXPBADPCM_OFF+16
#define AXPBADPCM_PRED_SCALE          (AXPBADPCM_OFF+17)
#define AXPBADPCM_YN1                 (AXPBADPCM_OFF+18)
#define AXPBADPCM_YN2                 (AXPBADPCM_OFF+19)
#define AXPBADPCM_SIZE                20

/*--------------------------------------------------------------------------*
    sample rate converter state
 *--------------------------------------------------------------------------*/
typedef struct _AXPBSRC
{

    u16     ratioHi;            //  sampling ratio, integer
    u16     ratioLo;            //  sampling ratio, fraction

    u16     currentAddressFrac; //  current fractional sample position

    u16     last_samples[4];    //  last 4 input samples

} AXPBSRC;

#define AXPBSRC_RATIOHI_OFF               AXPBSRC_OFF
#define AXPBSRC_RATIOLO_OFF               (AXPBSRC_OFF+1)
#define AXPBSRC_CURRENTADDRESSFRAC_OFF    (AXPBSRC_OFF+2)
#define AXPBSRC_LAST_SAMPLES_OFF          (AXPBSRC_OFF+3)       // 4 words
#define AXPBSRC_SIZE                      7

/*--------------------------------------------------------------------------*
    sample rate converter state for remote speakers
 *--------------------------------------------------------------------------*/
typedef struct _AXPBRMTSRC
{

    u16     currentAddressFrac; //  current fractional sample position

    u16     last_samples[4];    //  last 4 input samples

} AXPBRMTSRC;

#define AXPBRMTSRC_SIZE                   5

/*--------------------------------------------------------------------------*
    ADPCM loop parameters
 *--------------------------------------------------------------------------*/
typedef struct _AXPBADPCMLOOP
{

    u16     loop_pred_scale;    //  predictor / scale combination (nibbles, as in hardware)
    u16     loop_yn1;           //  y[n - 1]
    u16     loop_yn2;           //  y[n - 2]

} AXPBADPCMLOOP;

#define AXPBADPCMLOOP_PRED_SCALE      AXPBADPCMLOOP_OFF
#define AXPBADPCMLOOP_YN1             (AXPBADPCMLOOP_OFF+1)
#define AXPBADPCMLOOP_YN2             (AXPBADPCMLOOP_OFF+2)
#define AXPBADPCMLOOP_SIZE            3

/*--------------------------------------------------------------------------*
    IIR filter parameters (lowpass)
 *--------------------------------------------------------------------------*/
typedef struct _AXPBLPF
{

    u16     on;
    u16     yn1;
    u16     a0;
    u16     b0;

} AXPBLPF;

#define AXPBLPF_ON                    AXPBLPF_OFF
#define AXPBLPF_YN1                   (AXPBLPF_OFF+1)
#define AXPBLPF_A0                    (AXPBLPF_OFF+2)
#define AXPBLPF_B0                    (AXPBLPF_OFF+3)
#define AXPBLPF_SIZE                  4

/*--------------------------------------------------------------------------*
    Biquad IIR filter parameters
 *--------------------------------------------------------------------------*/
typedef struct _AXPBBIQUAD
{

    u16     on;
    s16     xn1;
    s16     xn2;
    s16     yn1;
    s16     yn2;
    s16     b0;
    s16     b1;
    s16     b2;
    s16     a1;
    s16     a2;

} AXPBBIQUAD;

#define AXPBBIQUAD_ON                 AXPBBIQUAD_OFF
#define AXPBBIQUAD_XN1                (AXPBBIQUAD_OFF+1)
#define AXPBBIQUAD_XN2                (AXPBBIQUAD_OFF+2)
#define AXPBBIQUAD_YN1                (AXPBBIQUAD_OFF+3)
#define AXPBBIQUAD_YN2                (AXPBBIQUAD_OFF+4)
#define AXPBBIQUAD_B0                 (AXPBBIQUAD_OFF+5)
#define AXPBBIQUAD_B1                 (AXPBBIQUAD_OFF+6)
#define AXPBBIQUAD_B2                 (AXPBBIQUAD_OFF+7)
#define AXPBBIQUAD_A1                 (AXPBBIQUAD_OFF+8)
#define AXPBBIQUAD_A2                 (AXPBBIQUAD_OFF+9)
#define AXPBBIQUAD_SIZE               10

/*--------------------------------------------------------------------------*
    IIR filter for remote speakers
 *--------------------------------------------------------------------------*/
typedef union __AXPBRMTIIR
{
    AXPBLPF    lpf;
    AXPBBIQUAD biquad;

} AXPBRMTIIR;

#define AXPBRMTIIR_SIZE               10

//  state
#define AX_PB_STATE_STOP        0x0000
#define AX_PB_STATE_RUN         0x0001

//  type
#define AX_PB_TYPE_NORMAL       0x0000
#define AX_PB_TYPE_STREAM       0x0001  // use only pred & scale. ignore histories.

//  format
#define AX_PB_FORMAT_PCM16      0x000A  // signed 16 bit PCM mono
#define AX_PB_FORMAT_PCM8       0x0019  // signed 8 bit PCM mono
#define AX_PB_FORMAT_ADPCM      0x0000  // ADPCM encoded (both standard & extended)

//  src select
#define AX_PB_SRCSEL_POLYPHASE  0x0000  // N64 type polyphase filter (4-tap)
#define AX_PB_SRCSEL_LINEAR     0x0001  // Linear interpolator
#define AX_PB_SRCSEL_NONE       0x0002  // No SRC (1:1)

//  coef select
#define AX_PB_COEFSEL_8KHZ      0x0000  // 8KHz low pass response
#define AX_PB_COEFSEL_12KHZ     0x0001  // 12.8KHz N64 type response
#define AX_PB_COEFSEL_16KHZ     0x0002  // 16KHz response

//  mixer ctrl for main speakers
#define AX_PB_MIXCTRL_L         0x00000001  // main left mix
#define AX_PB_MIXCTRL_R         0x00000002  // main right mix
#define AX_PB_MIXCTRL_LR_RAMP   0x00000004  // main bus ramp (applies to LR only)
#define AX_PB_MIXCTRL_S         0x00000008  // main surround mix
#define AX_PB_MIXCTRL_S_RAMP    0x00000010  // main bus ramp (applies to S only)

#define AX_PB_MIXCTRL_A_L       0x00010000  // AuxA left mix
#define AX_PB_MIXCTRL_A_R       0x00020000  // AuxA rigth mix
#define AX_PB_MIXCTRL_A_LR_RAMP 0x00040000  // AuxA bus ramp (applies to LR only)
#define AX_PB_MIXCTRL_A_S       0x00080000  // AuxA surround mix
#define AX_PB_MIXCTRL_A_S_RAMP  0x00100000  // AuxA bus ramp (applies to S only)

#define AX_PB_MIXCTRL_B_L       0x00200000  // AuxB left mix
#define AX_PB_MIXCTRL_B_R       0x00400000  // AuxB rigth mix
#define AX_PB_MIXCTRL_B_LR_RAMP 0x00800000  // AuxB bus ramp (applies to LR only)
#define AX_PB_MIXCTRL_B_S       0x01000000  // AuxB surround mix
#define AX_PB_MIXCTRL_B_S_RAMP  0x02000000  // AuxB bus ramp (applies to S only)

#define AX_PB_MIXCTRL_C_L       0x04000000  // AuxCleft mix
#define AX_PB_MIXCTRL_C_R       0x08000000  // AuxC rigth mix
#define AX_PB_MIXCTRL_C_LR_RAMP 0x10000000  // AuxC bus ramp (applies to LR only)
#define AX_PB_MIXCTRL_C_S       0x20000000  // AuxC surround mix
#define AX_PB_MIXCTRL_C_S_RAMP  0x40000000  // AuxC bus ramp (applies to S only)
#define AX_PB_MIXCTRL_C_DPL2    0x80000000  // AuxC DPL2, does not apply ITD for surrounds

// Mixer Management (PPC or DSP)

#define AX_PB_RENDERER_SELECT_DSP 0
#define AX_PB_RENDERER_SELECT_PPC 1
#define AX_PB_RENDERER_SELECT_DSP_OR_PPC 2
#define AX_PB_MIXER_SELECT_DSP AX_PB_RENDERER_SELECT_DSP
#define AX_PB_MIXER_SELECT_PPC AX_PB_RENDERER_SELECT_PPC


#define AXMIX_DSP_ENABLE 1
#define AXMIX_PPC_ENABLE 2

//  IIR filter switch
#define AX_PB_LPF_OFF           0x0000  // LPF switch
#define AX_PB_LPF_ON            0x0001

#define AX_PB_BIQUAD_OFF        0x0000  // Biquad IIR filter switch
#define AX_PB_BIQUAD_ON         0x0002

// output
#define AX_PB_REMOTE_OFF        0x0000
#define AX_PB_REMOTE_ON         0x0001

//  mixer ctrl for remote speakers
#define AX_PB_MIXCTRL_MAIN0       0x0001  // main0 mix
#define AX_PB_MIXCTRL_MAIN0_RAMP  0x0002  // main0 mix with ramp
#define AX_PB_MIXCTRL_AUX0        0x0004  // aux0
#define AX_PB_MIXCTRL_AUX0_RAMP   0x0008  // aux0  mix with ramp
#define AX_PB_MIXCTRL_MAIN1       0x0010  // main1
#define AX_PB_MIXCTRL_MAIN1_RAMP  0x0020  // main1 mix with ramp
#define AX_PB_MIXCTRL_AUX1        0x0040  // aux1
#define AX_PB_MIXCTRL_AUX1_RAMP   0x0080  // aux1  mix with ramp
#define AX_PB_MIXCTRL_MAIN2       0x0100  // main2
#define AX_PB_MIXCTRL_MAIN2_RAMP  0x0200  // main2 mix with ramp
#define AX_PB_MIXCTRL_AUX2        0x0400  // aux2
#define AX_PB_MIXCTRL_AUX2_RAMP   0x0800  // aux2  mix with ramp
#define AX_PB_MIXCTRL_MAIN3       0x1000  // main3
#define AX_PB_MIXCTRL_MAIN3_RAMP  0x2000  // main3 mix with ramp
#define AX_PB_MIXCTRL_AUX3        0x4000  // aux3
#define AX_PB_MIXCTRL_AUX3_RAMP   0x8000  // aux3  mix with ramp

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
#define AX_DSP_CYCLES               (OS_BUS_CLOCK / 667)

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
#define AX_MAX_VOICES               96

#define AX_MS_PER_FRAME             3

#define AX_IN_SAMPLES_PER_MS        32
#define AX_IN_SAMPLES_PER_SEC       (AX_IN_SAMPLES_PER_MS * 1000)
#define AX_IN_SAMPLES_PER_FRAME     (AX_IN_SAMPLES_PER_MS * AX_MS_PER_FRAME)

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
#define AX_RMT_SAMPLES_PER_MS       6
#define AX_RMT_SAMPLES_PER_SEC      (AX_RMT_SAMPLES_PER_MS * 1000)
#define AX_RMT_SAMPLES_PER_FRAME    (AX_RMT_SAMPLES_PER_MS * AX_MS_PER_FRAME)

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
#define AX_MODE_STEREO              0
#define AX_MODE_SURROUND            1
#define AX_MODE_DPL2                2
// placeholder def for now, will become =3 when all systems are ready for it
#define AX_MODE_6CHAN               3

#define AX_MODE_MONO                5

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
#define AX_COMPRESSOR_OFF           0
#define AX_COMPRESSOR_ON            1

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
#define AX_PRIORITY_STACKS          32
#define AX_PRIORITY_NODROP          (AX_PRIORITY_STACKS - 1)
#define AX_PRIORITY_LOWEST          1
#define AX_PRIORITY_FREE            0

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
#define AX_SRC_TYPE_NONE            0
#define AX_SRC_TYPE_LINEAR          1
#define AX_SRC_TYPE_4TAP_8K         2
#define AX_SRC_TYPE_4TAP_12K        3
#define AX_SRC_TYPE_4TAP_16K        4

// return codes from AXSetVoiceSrcRatio()
#define AX_SRC_RATIO_OK             0
#define AX_SRC_RATIO_TOO_SMALL     -1
#define AX_SRC_RATIO_TOO_LARGE     -2

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
#define AX_ADDR_ONESHOT             0
#define AX_ADDR_LOOP                1

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
#define AX_SYNC_NONEWPARAMS         0x00000000
#define AX_SYNC_USER_SRCSELECT      0x00000001
#define AX_SYNC_USER_MIXCTRL        0x00000002
#define AX_SYNC_USER_STATE          0x00000004
#define AX_SYNC_USER_TYPE           0x00000008
#define AX_SYNC_USER_MIX            0x00000010
#define AX_SYNC_USER_ITD            0x00000020
#define AX_SYNC_USER_ITDTARGET      0x00000040
#define AX_SYNC_USER_DPOP           0x00000080
#define AX_SYNC_USER_VE             0x00000100
#define AX_SYNC_USER_VEDELTA        0x00000200
#define AX_SYNC_USER_ADDR           0x00000400
#define AX_SYNC_USER_LOOP           0x00000800
#define AX_SYNC_USER_LOOPADDR       0x00001000
#define AX_SYNC_USER_ENDADDR        0x00002000
#define AX_SYNC_USER_CURRADDR       0x00004000
#define AX_SYNC_USER_ADPCM          0x00008000
#define AX_SYNC_USER_SRC            0x00010000
#define AX_SYNC_USER_SRCRATIO       0x00020000
#define AX_SYNC_USER_ADPCMLOOP      0x00040000
#define AX_SYNC_USER_LPF            0x00080000
#define AX_SYNC_USER_LPF_COEF       0x00100000
#define AX_SYNC_USER_BIQUAD         0x00200000
#define AX_SYNC_USER_BIQUAD_COEF    0x00400000
#define AX_SYNC_USER_REMOTE         0x00800000
#define AX_SYNC_USER_RMTMIXCTRL     0x01000000
#define AX_SYNC_USER_RMTMIX         0x02000000
#define AX_SYNC_USER_RMTDPOP        0x04000000
#define AX_SYNC_USER_RMTSRC         0x08000000
#define AX_SYNC_USER_RMTIIR         0x10000000
#define AX_SYNC_USER_RMTIIR_COEF1   0x20000000
#define AX_SYNC_USER_RMTIIR_COEF2   0x40000000
#define AX_SYNC_USER_DRCDPOP        0x80000000

#define AX_SYNC_USER_ALLPARAMS      0xffffffff

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
#define AX_OUTPUT_BUFFER_DOUBLE     0
#define AX_OUTPUT_BUFFER_TRIPLE     1

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
// new Drop Voice callback API for SDK 2.0.5+
#define AXVCB_REASON_NO_RESOURCES  1
#define AXVCB_REASON_SEG_VIOLATION 2

typedef void (*AXVoiceCallbackEx)(void *p, u32 context, u32 reason);

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
typedef struct _AXVPB
{
    // useful user information
    u32             index;          // index of VPB in array
    u32             state;          // mirrored from the underlying state
    u32             currentVolume;  // mirrored from the underlying state
    u32             mixerSelect;    // DSP or PPC mixer for this voice

    // stuff the user does not need and should never touch

    // Voice stack linkages
    void            *next;          // used in priority stacks
    void            *prev;          // used in priority stacks
    void            *next1;         // used in callback stack

    // these ares are used in voice allocation
    u32             priority;       // index to stack
    AXVoiceCallback callback;       // user callback for specified voice (deprecated)
    u32             userContext;    // user assigned context for callback

    // vars & flags for updating and sync PBs
    u32             sync;           // bit mask for each PB item to sync
    u32             depop;          // should depop voice
    void            *itdBuffer;     // pointer to itd buffer

    AXPBOFFSET      offsets;        // stash AXSetVoiceOffsets() settings here for AX reference

    // new for 2.0.5
    AXVoiceCallbackEx callbackex;   // includes context and reason arguments
    u32             cb_reason;      // an AXVCB_REASON code

    f32             dspLoadEstimate;
    f32             ppcLoadEstimate;
} AXVPB;

/*---------------------------------------------------------------------------*
Header for DSPADPCM.exe header
 *---------------------------------------------------------------------------*/
typedef struct
{
    u32 num_samples;       // total number of RAW samples
    u32 num_adpcm_nibbles; // number of ADPCM nibbles (including frame headers)
    u32 sample_rate;       // Sample rate, in Hz
    u16 loop_flag;         // 1=LOOPED, 0=NOT LOOPED
    u16 format;            // Always 0x0000, for ADPCM
    u32 sa;                // Start offset address for looped samples (zero for non-looped)
    u32 ea;                // End offset address for looped samples
    u32 ca;                // always zero
    u16 coef[16];          // decode coefficients (eight pairs of 16-bit words)
    u16 gain;              // always zero for ADPCM
    u16 ps;                // predictor/scale
    u16 yn1;               // sample history
    u16 yn2;               // sample history
    u16 lps;               // predictor/scale for loop context
    u16 lyn1;              // sample history (n-1) for loop context
    u16 lyn2;              // sample history (n-2) for loop context
    u16 pad[11];           // reserved

} DSPADPCM;

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void    AXInit                      (void);
BOOL    AXIsInit                    (void);
void    AXQuit                      (void);

u32     AXGetMaxVoices              (void);

u16     AXGetMasterVolume           (void);
void    AXSetMasterVolume           (u16 volume);

AXVPB*  AXAcquireVoice              (
                                     u32                priority,
                                     AXVoiceCallback    callback,
                                     u32                userContext
                                     );

void    AXFreeVoice                 (AXVPB *p);
void    AXSetVoicePriority          (AXVPB *p, u32 priority);

void    AXSetVoiceSrcType           (AXVPB *p, u32 type);
void    AXSetVoiceState             (AXVPB *p, u16 state);
void    AXSetVoiceType              (AXVPB *p, u16 type);
void    AXSetVoiceVe                (AXVPB *p, const AXPBVE *ve);
void    AXSetVoiceVeDelta           (AXVPB *p, s16 delta);
void    AXSetVoiceLoop              (AXVPB *p, u16 loop);

// CAFE new API's
void    AXSetVoiceOffsets           (AXVPB *p, const AXPBOFFSET *addr);
void    AXSetVoiceLoopOffset        (AXVPB *p, u32 offset);
void    AXSetVoiceEndOffset         (AXVPB *p, u32 offset);
void    AXSetVoiceCurrentOffset     (AXVPB *p, u32 offset);
void    AXGetVoiceOffsets           (AXVPB *p, AXPBOFFSET *addr);
BOOL    AXIsVoiceRunning            (AXVPB *p);
BOOL    AXIsAudioOutReady           (void);

void    AXSetVoiceSamplesAddr       (AXVPB *p, const void *samples);
u32     AXGetVoiceLoopCount         (AXVPB *p);

void    AXSetVoiceAdpcm             (AXVPB *p, const AXPBADPCM *adpcm);
void    AXSetVoiceSrc               (AXVPB *p, const AXPBSRC *src);
s32     AXSetVoiceSrcRatio          (AXVPB *p, f32 ratio);
void    AXSetVoiceAdpcmLoop         (AXVPB *p, const AXPBADPCMLOOP *adpcmloop);
void    AXSetVoiceLpf               (AXVPB *p, const AXPBLPF *lpf);
void    AXSetVoiceLpfCoefs          (AXVPB *p, u16 a0, u16 b0);
void    AXSetVoiceBiquad            (AXVPB *p, const AXPBBIQUAD *biquad);
void    AXSetVoiceBiquadCoefs       (AXVPB *p, u16 b0, u16 b1, u16 b2, u16 a1, u16 a2);

void    AXInitProfile               (AXPROFILE *profile, u32 maxProfiles);
u32     AXGetProfile                (void);
u32     AXGetSwapProfile            (AXPROFILE *profile, u32 maxProfiles);

void    AXSetStepMode               (u32);

void    AXComputeLpfCoefs           (u16 freq, u16 *a0, u16 *b0);

void    AXSetVoiceRmtOn             (AXVPB *p, u16 on);
void    AXSetVoiceRmtSrc            (AXVPB *p, const AXPBRMTSRC  *src);
void    AXSetVoiceRmtIIR            (AXVPB *p, const AXPBRMTIIR  *iir);
void    AXSetVoiceRmtIIRCoefs       (AXVPB *p, u16 type, ...);

s32     AXRmtGetSamplesLeft         (void);
s32     AXRmtGetSamples             (s32 chan, s16* buffer, s32 samples);
s32     AXRmtAdvancePtr             (s32 samples);

void    AXRegisterExceedCallback    (AXExceedCallback callback);

s32     AXGetAdpcmOutputSize        (s32       samples);
s32     AXGetAdpcmWorkSize          (s32       samples);
s32     AXGetAdpcmData              (const s16*      input,
                                     s32       samples,
                                     s32       rate,
                                     s32       loop_start,
                                     s32       loop_end,
                                     u8*       output,
                                     DSPADPCM* info,
                                     void*     work);
s32     AXDecodeAdpcmData           (const u8*       input,
                                     DSPADPCM* info,
                                     s32       samples,
                                     s16*      output);

s32     AXUserBegin                 (void);
s32     AXUserEnd                   (void);
BOOL    AXUserIsProtected           (void);
s32     AXVoiceBegin                (AXVPB *p);
s32     AXVoiceEnd                  (AXVPB *p);
BOOL    AXVoiceIsProtected          (AXVPB *p);


// still keeping the old mixer select functions
#define AXSetDefaultRenderer     AXSetDefaultMixerSelect
#define AXGetDefaultRenderer     AXGetDefaultMixerSelect
#define AXSetVoiceRenderer       AXSetVoiceMixerSelect
#define AXGetVoiceRenderer       AXGetVoiceMixerSelect

u32     AXSetDefaultRenderer     (u32 mixerSelect);
u32     AXGetDefaultRenderer     (void);
u32     AXSetVoiceRenderer       (AXVPB *p, u32 mixerSelect);
u32     AXGetVoiceRenderer       (AXVPB *p);
u32     __AXGetMixMode           (void);

BOOL AXIsVoiceRunning(AXVPB *pvpb);
s16  AXGetVoiceVolume(AXVPB *pvpb);

void __AXSynthesize(
                    int channels,
                    signed short* buffer,
                    unsigned long samples,
                    int sampleRate
                    );
/******************************************************/
/******************************************************/

/*
 *      All the new API prototypes
 */


AXPB_ERROR_CODE AXSetVoiceDeviceMix(AXVPB *voice, AXPBDeviceType device, u32 Id, AXPBCHMIX *mix);
    // - AXPBDeviceType will specify for which device you are setting the mix
    // - the mode set previously will dictate what the device channels are set to.
    // - deviceId = 0 for TV, 0,1 for DRC, 0,1,2,3 for RMT ; gives finer control on the mixing
    // - we could have enums and make sure deviceId is within bounds
    // - mix is a pointer to array of AXPBCHMIx, number depending on the mode and device
    // - one consistent interface for all the devices now and that may be added in future.

// new APIs
void AXSetMaxVoices(u32 num);
    // doesnt change memory allocation, only the max number of voices usable

AXPB_ERROR_CODE AXSetDeviceMode(AXPBDeviceType device, u32 mode);
    // sets the output mode for the device (stereo, dpl2, dpl?, 6ch)

AXPB_ERROR_CODE AXGetDeviceMode(AXPBDeviceType device, u32 *mode);
    // gets the output mode for the device (stereo, dpl2, dpl?, 6ch)

AXPB_ERROR_CODE AXGetDeviceChannelCount(AXPBDeviceType device, u32 Id, u32 *count);
    // gets the number of active channels for the device

AXUserCallback  AXRegisterFrameCallback(AXUserCallback fcb);
    // callback function to control AX per frame

AXPB_ERROR_CODE AXRegisterAppFrameCallback(AXUserCallback fcb);
    // callback function to control AX per frame
AXPB_ERROR_CODE    AXDeregisterAppFrameCallback(AXUserCallback fcb);
    // callback function to control AX per frame

AXPB_ERROR_CODE AXRegisterAuxCallback(AXPBDeviceType device, u32 Id, u32 auxId, AXUserAuxCallback fcb, void *context);
    // registers a callback for aux on a device, called once each frame

AXPB_ERROR_CODE AXGetAuxCallback(AXPBDeviceType device, u32 Id, u32 auxId, AXUserAuxCallback *fcb, void **context);
    // gets the callback for aux on a device

AXPB_ERROR_CODE AXSetDeviceVolume(AXPBDeviceType device, u32 Id, u16 volume);
    // sets the device volume

AXPB_ERROR_CODE AXGetDeviceVolume(AXPBDeviceType device, u32 Id, u16 *volume);
    // gets the current volume of the device

AXPB_ERROR_CODE AXSetAuxReturnVolume(AXPBDeviceType device, u32 Id, u32 auxId, u16 volume);
    // sets the return volume of the aux specified by auxId on the device

AXPB_ERROR_CODE AXGetAuxReturnVolume(AXPBDeviceType device, u32 Id, u32 auxId, u16 *volume);
    // gets the return volume of the aux specified by auxId on the device

AXPB_ERROR_CODE AXSetDeviceCompressor(AXPBDeviceType device, u16 on_off);
    // turns ON/OFF the compressor on a particular device

AXPB_ERROR_CODE AXSetDeviceCompressorTable(AXPBDeviceType device, u16* table, u16 frames);
    // sets the tables for compressor to be used for this particular device
void    AXMakeCompressorTable       (f32  gain,  u16 frames, u16* table);
    // Function to create compressor table with user preferences

// new APIs for access to final mix
AXPB_ERROR_CODE AXRegisterDeviceFinalMixCallback(AXPBDeviceType device, AXUserFinalMixCallback fcb);
    // sets the callback function to be called after the final mix of main and aux buses is done for this device

AXPB_ERROR_CODE AXGetDeviceFinalMixCallback(AXPBDeviceType device, AXUserFinalMixCallback *fcb);
    // gets the callback function to be called after the final mix of main and aux buses is done for this device

AXPB_ERROR_CODE AXSetDeviceRemixMatrix(AXPBDeviceType device,  u32 const numInChns,  u32 const numOutChns, f32 const *matrixPtr);
    // sets the matrix for remixing numInChns to numOutChns for the device
AXPB_ERROR_CODE AXGetDeviceRemixMatrix(AXPBDeviceType device,  u32 const numInChns,  u32 const numOutChns, f32 const **matrixPtr);
    // gets the matrix for remixing numInChns to numOutChns for the device

AXPB_ERROR_CODE AXGetDeviceUpsampleStage(AXPBDeviceType device, BOOL *post);
    // gets the current setting of upsample stage for the device
AXPB_ERROR_CODE AXSetDeviceUpsampleStage(AXPBDeviceType device, BOOL post);
    // sets the current setting of upsample stage for the device
    //
typedef struct {
    u32 numChannels;
    u32 sampFreq;
    u32 numDevices;
    u32 format;
    u32 length;
}AXDEVICEFINALOUTPUTSTRUCT ;
// gets the final output samples for the device
AXPB_ERROR_CODE AXGetDeviceFinalOutput(AXPBDeviceType device, s16 *dataPtr, u32 const size, AXDEVICEFINALOUTPUTSTRUCT *axDeviceFinalOutputStruct);

AXPB_ERROR_CODE AXSetDeviceLinearUpsampler(AXPBDeviceType device, u32 deviceId, BOOL flag);

    //===== SDK V 2.0.5 new API's
    // returns 0 for a segment violation else non-zero
AXVPB*  AXAcquireVoiceEx            (
                                     u32                priority,
                                     AXVoiceCallbackEx  callback,
                                     u32                userContext
                                     );

u32 AXCheckVoiceOffsets(const AXPBOFFSET *maddr);

// Get/set DSP voice allocation limit
// set non-zero to override load calculations as reason to shed
u32 AXGetMaxDspVoices(void);
void AXSetMaxDspVoices(u32 max_voices);

// set load alculatin shed thresholds
// DSP Loasd Limit controls when to shed to PPC Renderer
// or drop if the voice is pinned to DSP but cannot be alloocated
// range is 0.0% to 100.0% (but allow margin for jitter!)
void AXSetDspLoadLimit(f32 limit);
f32 AXGetDspLoadLimit(void);
// PPC Load Limit controls when to drop the voice
void AXSetPpcLoadLimit(f32 limit);
f32 AXGetPpcLoadLimit(void);

// Check current frame load computations and allocations
// call in User Frame Callback only
f32 AXGetPpcLoad(void);
f32 AXGetDspLoad(void);
u32 AXGetNumVoices(void);
u32 AXGetNumDspVoices(void);
u32 AXGetDroppedVoiceCount(void);

// Virtualization functions for DRC
typedef enum {
    AX_DRC_VS_OFF = 0,
    AX_DRC_VS_ON_NORMAL,
    AX_DRC_VS_ON_FRONT_BYPASS
} AXDRCVSModeType;

typedef enum
{
    AX_DRC_VS_OUTPUT_SPEAKER = 0,
    AX_DRC_VS_OUTPUT_HEADPHONE,
    AX_DRC_VS_OUTPUT_NUM
} AXDRCVSOutputMode;
typedef enum
{
    AX_DRC_VS_SPEAKER_POS_WIDE = 0,
    AX_DRC_VS_SPEAKER_POS_SQUARE,
    AX_DRC_VS_SPEAKER_POS_NUM
} AXDRCVSSpeakerPosition;

typedef enum
{
    AX_DRC_VS_SLEV_1_0 = 0,
    AX_DRC_VS_SLEV_0_707,
    AX_DRC_VS_SLEV_0_5,
    AX_DRC_VS_SLEV_NUM
} AXDRCVSSurroundLevelGain;

typedef enum
{
    AX_DRC_VS_LC_NOT_USE = 0,
    AX_DRC_VS_LC_USE,
    AX_DRC_VS_LC_USE_MAX,
    AX_DRC_VS_LC_NUM
} AXDRCVSLCMode;

AXPB_ERROR_CODE AXSetDRCVSMode(AXDRCVSModeType mode);
AXPB_ERROR_CODE AXGetDRCVSMode(AXDRCVSModeType *mode);
AXPB_ERROR_CODE AXSetDRCVSLC(AXDRCVSLCMode mode);
AXPB_ERROR_CODE AXSetDRCVSSpeakerPosition(AXDRCVSOutputMode output, AXDRCVSSpeakerPosition pos);
AXPB_ERROR_CODE AXSetDRCVSSurroundDepth(AXDRCVSOutputMode output, f32 depth);
AXPB_ERROR_CODE AXSetDRCVSDownmixBalance(AXDRCVSOutputMode output, f32 balance);
AXPB_ERROR_CODE AXSetDRCVSSurroundLevelGain(AXDRCVSSurroundLevelGain gain);
AXPB_ERROR_CODE AXSetDRCVSOutputGain(AXDRCVSOutputMode output, f32 gain);
AXPB_ERROR_CODE AXSetDRCVSLimiter(BOOL enable);
AXPB_ERROR_CODE AXSetDRCVSLimiterThreshold(f32 level);

/************************************************************************
*   UNUSED STRUCTURES AND ENUMS                                         *
************************************************************************/
// AXSetMode()
// definitions compatible with original definitions in ax.h
//
typedef enum {
    AX_MODE_STEREO_32K=0,
    AX_MODE_SURROUND_32K,
    AX_MODE_4CHAN_32K,
    AX_MODE_6CHAN_32K,
    AX_MODE_STEREO_48K,
    AX_MODE_SURROUND_48K,
    AX_MODE_4CHAN_48K,
    AX_MODE_6CHAN_48K,
    AX_TV_MAX_MODES
} AXPBTvModeType;

typedef enum {
// AXSetDrcMode()
// DRC mode definitions
    AX_DRC_MODE_MONO=0,
    AX_DRC_MODE_STEREO,
    AX_DRC_MODE_DUAL_MONO,
    AX_DRC_MODE_DUAL_STEREO,
    AX_DRC_MAX_MODES
} AXPBDrcModeType;

/************************************************************************
*   DEPRECATED SECTION: DO NOT USE THESE IF YOU ARE WRITING NEW CODE    *
*   USE THE REPLACEMENT FUNCTIONS & STRUCTS PROVIDED. CHECK MAN PAGES   *
************************************************************************/
/*----------------------------------------------------------------------*
 *----------------------------------------------------------------------*/
//aliases for enum
#define AXPBMIXMOV AXPBDRCMIX
#define AXPBDPOPMOV AXPBDRCDPOP

//aliases for API
#define AXRegisterMovCallback     AXRegisterDRCCallback
#define AXGetMasterMOVVolume      AXGetMasterDRCVolume
#define AXSetMasterMOVVolume      AXSetMasterDRCVolume
#define AXRegisterAuxMOVCallback  AXRegisterAuxDRCCallback
#define AXGetAuxMOVCallback       AXGetAuxDRCCallback
#define AXGetMOVAuxReturnVolume   AXGetDRCAuxReturnVolume
#define AXSetMOVAuxReturnVolume   AXSetDRCAuxReturnVolume
#define AXSetVoiceMOVMix          AXSetVoiceDRCMix
#define AXSetVoiceMOVDpop         AXSetVoiceDRCDpop

/**************************************************************
 DSP Re-write To Be Deprecated Public AX Voice Data Definitions
 *************************************************************/
 // bug 581 - use signed deltas
typedef struct _AXPBMIX
{
    //  mixing values in .15, 0x8000 = ca. 1.0

    u16     vL;
    s16     vDeltaL;
    u16     vR;
    s16     vDeltaR;

    u16     vAuxAL;
    s16     vDeltaAuxAL;
    u16     vAuxAR;
    s16     vDeltaAuxAR;

    u16     vAuxBL;
    s16     vDeltaAuxBL;
    u16     vAuxBR;
    s16     vDeltaAuxBR;

    u16     vAuxCL;
    s16     vDeltaAuxCL;
    u16     vAuxCR;
    s16     vDeltaAuxCR;

    u16     vS;
    s16     vDeltaS;
    u16     vAuxAS;
    s16     vDeltaAuxAS;
    u16     vAuxBS;
    s16     vDeltaAuxBS;
    u16     vAuxCS;
    s16     vDeltaAuxCS;

} AXPBMIX;

typedef struct _AXPBDRCMIX
{
    //  mixing values in .15, 0x8000 = ca. 1.0

    u16     vL;
    s16     vDeltaL;
    u16     vR;
    s16     vDeltaR;

    u16     vAuxL;
    s16     vDeltaAuxL;
    u16     vAuxR;
    s16     vDeltaAuxR;

    u16     vS;
    s16     vDeltaS;
    u16     vAuxS;
    s16     vDeltaAuxS;

} AXPBDRCMIX;

#define AXPBMIX_VL_OFF            AXPBMIX_OFF
#define AXPBMIX_VDELTAL_OFF       (AXPBMIX_OFF+1)
#define AXPBMIX_VR_OFF            (AXPBMIX_OFF+2)
#define AXPBMIX_VDELTAR_OFF       (AXPBMIX_OFF+3)

#define AXPBMIX_VAUXAL_OFF        (AXPBMIX_OFF+4)
#define AXPBMIX_VDELTAAUXAL_OFF   (AXPBMIX_OFF+5)
#define AXPBMIX_VAUXAR_OFF        (AXPBMIX_OFF+6)
#define AXPBMIX_VDELTAAUXAR_OFF   (AXPBMIX_OFF+7)

#define AXPBMIX_VAUXBL_OFF        (AXPBMIX_OFF+8)
#define AXPBMIX_VDELTAAUXBL_OFF   (AXPBMIX_OFF+9)
#define AXPBMIX_VAUXBR_OFF        (AXPBMIX_OFF+10)
#define AXPBMIX_VDELTAAUXBR_OFF   (AXPBMIX_OFF+11)

#define AXPBMIX_VAUXCL_OFF        (AXPBMIX_OFF+12)
#define AXPBMIX_VDELTAAUXCL_OFF   (AXPBMIX_OFF+13)
#define AXPBMIX_VAUXCR_OFF        (AXPBMIX_OFF+14)
#define AXPBMIX_VDELTAAUXCR_OFF   (AXPBMIX_OFF+15)

#define AXPBMIX_VS_OFF            (AXPBMIX_OFF+16)
#define AXPBMIX_VDELTAS_OFF       (AXPBMIX_OFF+17)
#define AXPBMIX_VAUXAS_OFF        (AXPBMIX_OFF+18)
#define AXPBMIX_VDELTAAUXAS_OFF   (AXPBMIX_OFF+19)
#define AXPBMIX_VAUXBS_OFF        (AXPBMIX_OFF+20)
#define AXPBMIX_VDELTAAUXBS_OFF   (AXPBMIX_OFF+21)
#define AXPBMIX_VAUXCS_OFF        (AXPBMIX_OFF+22)
#define AXPBMIX_VDELTAAUXCS_OFF   (AXPBMIX_OFF+23)

#define AXPBMIX_SIZE              24

//  mixer ctrl for DRC speakers
#define AX_PB_MIXCTRL_DL         0x00000020  // DRC left mix
#define AX_PB_MIXCTRL_DR         0x00000040  // DRC right mix
#define AX_PB_MIXCTRL_DLR_RAMP   0x00000080  // DRC bus ramp (applies to LR only)
#define AX_PB_MIXCTRL_DS         0x00000100  // DRC surround mix
#define AX_PB_MIXCTRL_DS_RAMP    0x00000200  // DRC bus ramp (applies to S only)

#define AX_PB_MIXCTRL_DA_L       0x00000400  // DRC Aux left mix
#define AX_PB_MIXCTRL_DA_R       0x00000800  // DRC Aux rigth mix
#define AX_PB_MIXCTRL_DA_LR_RAMP 0x00001000  // DRC Aux bus ramp (applies to LR only)
#define AX_PB_MIXCTRL_DA_S       0x00002000  // DRC Aux surround mix
#define AX_PB_MIXCTRL_DA_S_RAMP  0x00004000  // DRC Aux bus ramp (applies to S only)

#define AX_PB_MOV_MIX_ALL        AX_PB_DRC_MIX_ALL
#define AX_PB_DRC_MIX_ALL        0x00007fe0


/******************************************************************
 End DSP Re-write To Be Deprecated Public AX Voice Data Definitions
 ******************************************************************/
/*--------------------------------------------------------------------------*
    mixing for remote speaker
 *--------------------------------------------------------------------------*/
typedef struct _AXPBRMTMIX
{

    u16     vMain0;
    s16     vDeltaMain0;
    u16     vAux0;
    s16     vDeltaAux0;

    u16     vMain1;
    s16     vDeltaMain1;
    u16     vAux1;
    s16     vDeltaAux1;

    u16     vMain2;
    s16     vDeltaMain2;
    u16     vAux2;
    s16     vDeltaAux2;

    u16     vMain3;
    s16     vDeltaMain3;
    u16     vAux3;
    s16     vDeltaAux3;

} AXPBRMTMIX;

#define AXPBRMTMIX_SIZE           16

/*--------------------------------------------------------------------------*
    initial time delay
 *--------------------------------------------------------------------------*/
typedef struct _AXPBITD
{

    u16     flag;               //  on or off for this voice
    u16     bufferHi;           //  MRAM buffer
    u16     bufferLo;           //
    u16     shiftL;             //  phase shift samples left (current)
    u16     shiftR;             //  phase shift samples right (current)
    u16     targetShiftL;       //  phase shift samples left (target)
    u16     targetShiftR;       //  phase shift samples right (target)

} AXPBITD;

//  flag
#define AX_PB_ITD_OFF      0x0000
#define AX_PB_ITD_ON       0x0001

#define AXPBITD_FLAG_OFF          AXPBITD_OFF
#define AXPBITD_BUFFERHI_OFF      (AXPBITD_OFF+1)
#define AXPBITD_BUFFERLO_OFF      (AXPBITD_OFF+2)
#define AXPBITD_SHIFTL_OFF        (AXPBITD_OFF+3)
#define AXPBITD_SHIFTR_OFF        (AXPBITD_OFF+4)
#define AXPBITD_TARGETSHIFTL_OFF  (AXPBITD_OFF+5)
#define AXPBITD_TARGETSHIFTR_OFF  (AXPBITD_OFF+6)
#define AXPBITD_SIZE              7

typedef struct _AXPBITDBUFFER
{

    s16 data[32];

} AXPBITDBUFFER;

/*--------------------------------------------------------------------------*
    depop data (last amplitudes mixed into buffers)
 *--------------------------------------------------------------------------*/
typedef struct _AXPBDPOP
{

    s16     aL;
    s16     aAuxAL;
    s16     aAuxBL;
    s16     aAuxCL;

    s16     aR;
    s16     aAuxAR;
    s16     aAuxBR;
    s16     aAuxCR;

    s16     aS;
    s16     aAuxAS;
    s16     aAuxBS;
    s16     aAuxCS;

} AXPBDPOP;

typedef struct _AXPBDRCDPOP
{

    s16     aL;
    s16     aAuxL;

    s16     aR;
    s16     aAuxR;

    s16     aS;
    s16     aAuxS;

} AXPBDRCDPOP;

#define AXPBDPOP_AL_OFF       AXPBDPOP_OFF
#define AXPBDPOP_AAUXAL_OFF   (AXPBDPOP_OFF+1)
#define AXPBDPOP_AAUXBL_OFF   (AXPBDPOP_OFF+2)
#define AXPBDPOP_AAUXCL_OFF   (AXPBDPOP_OFF+3)

#define AXPBDPOP_AR_OFF       (AXPBDPOP_OFF+4)
#define AXPBDPOP_AAUXAR_OFF   (AXPBDPOP_OFF+5)
#define AXPBDPOP_AAUXBR_OFF   (AXPBDPOP_OFF+6)
#define AXPBDPOP_AAUXCR_OFF   (AXPBDPOP_OFF+7)

#define AXPBDPOP_AS_OFF       (AXPBDPOP_OFF+8)
#define AXPBDPOP_AAUXAS_OFF   (AXPBDPOP_OFF+9)
#define AXPBDPOP_AAUXBS_OFF   (AXPBDPOP_OFF+10)
#define AXPBDPOP_AAUXCS_OFF   (AXPBDPOP_OFF+11)

#define AXPBDPOP_SIZE         12

/*--------------------------------------------------------------------------*
    depop data for remote speakers
 *--------------------------------------------------------------------------*/
typedef struct _AXPBRMTDPOP
{

    s16     aMain0;
    s16     aMain1;
    s16     aMain2;
    s16     aMain3;

    s16     aAux0;
    s16     aAux1;
    s16     aAux2;
    s16     aAux3;

} AXPBRMTDPOP;

#define AXPBRMTDPOP_SIZE      8

#define AXGetMemorySize(num)        ((sizeof(AXPBITDBUFFER) + sizeof(AXVPB)) * num)

void    AXInitEx                    (u32 outputBufferMode);
void    AXInitSpecifyMem            (u32 num, void* mem);
void    AXInitExSpecifyMem          (u32 num, void* mem, u32 outputBufferMode);
AXUserCallback AXRegisterCallback   (AXUserCallback callback);

AXUserCallback AXRegisterDRCCallback(AXUserCallback callback);
void    AXSetMode                   (u32 mode);
u32     AXGetMode                   (void);

void    AXSetMaxDspCycles           (u32 cycles);
u32     AXGetMaxDspCycles           (void);
u32     AXGetDspCycles              (void);

void    AXRegisterAuxACallback      (AXAuxCallback callback, void *context);
void    AXRegisterAuxBCallback      (AXAuxCallback callback, void *context);
void    AXRegisterAuxCCallback      (AXAuxCallback callback, void *context);

void    AXGetAuxACallback           (AXAuxCallback *callback, void **context);
void    AXGetAuxBCallback           (AXAuxCallback *callback, void **context);
void    AXGetAuxCCallback           (AXAuxCallback *callback, void **context);

u16     AXGetAuxAReturnVolume       (void);
u16     AXGetAuxBReturnVolume       (void);
u16     AXGetAuxCReturnVolume       (void);
void    AXSetAuxAReturnVolume       (u16 volume);
void    AXSetAuxBReturnVolume       (u16 volume);
void    AXSetAuxCReturnVolume       (u16 volume);

void    AXSetVoiceMix               (AXVPB *p, const AXPBMIX *mix);
void    AXSetVoiceItdOn             (AXVPB *p);
void    AXSetVoiceItdTarget         (AXVPB *p, u16 lShift, u16 rShift);

void    AXPrepareEfxData            (const void *samples, u32 size);

// deprecating these
void    AXGetVoiceOffsetsEx         (AXVPB *p, AXPBOFFSET *addr, const void *samples);
void    AXSetVoiceOffsetsEx         (AXVPB *p, const AXPBOFFSET *addr, const void *samples);
void    AXSetVoiceLoopOffsetEx      (AXVPB *p, u32 offset, const void *samples);
void    AXSetVoiceEndOffsetEx       (AXVPB *p, u32 offset, const void *samples);
void    AXSetVoiceCurrentOffsetEx   (AXVPB *p, u32 offset, const void *samples);
u32     AXGetVoiceCurrentOffsetEx   (AXVPB *p, const void *samples);
//
u16     AXGetMasterDRCVolume           (void);
void    AXSetMasterDRCVolume           (u16 volume);

void    AXRegisterAuxDRCCallback      (AXAuxCallback callback, void *context);
void    AXGetAuxDRCCallback           (AXAuxCallback *callback, void **context);

u16     AXGetDRCAuxReturnVolume       (void);
void    AXSetDRCAuxReturnVolume       (u16 volume);

void    AXSetVoiceDRCMix              (AXVPB *p, const AXPBDRCMIX *mix);
void    AXSetVoiceRmtMix            (AXVPB *p, const AXPBRMTMIX  *mix);

/*---------------------------------------------------------------------------*
  Sound 2
 *---------------------------------------------------------------------------*/
#define AX_TOTAL_NUM_OF_CHUNKS	   1

#define AX_SAMPLES_PER_AUDIO_FRAME_48 144

/* AX Initialization related enumartions */
typedef enum {
    AX_32K_RENDERER = 0,
    AX_48K_RENDERER = 1
} AX_RENDERER_FREQ_ENUM;

typedef enum {
    AX_3MS_FRAME = 0
} AX_FRAMESIZE_ENUM;

typedef enum {
    AX_SINGLE_PIPELINE_STAGE = 0,
    AX_FOUR_PIPELINE_STAGE = 1
} AX_PIPELINE_STAGES_ENUM;

typedef struct {
  AX_RENDERER_FREQ_ENUM   rendererFreq;
  AX_FRAMESIZE_ENUM frameSize;
  AX_PIPELINE_STAGES_ENUM pipelineStages; // <-- this is replacing the latency.
}AXINITPARAMS;

typedef enum {
    AXINIT_OK = 0,
    AXINIT_ALREADY_INITIALIZED = -1,
    AXINIT_INVALID_RENDERER = -2,
    AXINIT_INVALID_FRAMESIZE = -3,
    AXINIT_INVALID_PIPELINE_STAGE = -4
} AXINIT_ERROR_CODE;


AXINIT_ERROR_CODE    AXInitWithParams              (AXINITPARAMS *initParams);
void    AXGetCurrentParams    (AXINITPARAMS *currentParams);
AX_RENDERER_FREQ_ENUM AXGetRendererFreq (void);
u32  	AXGetInputSamplesPerFrame     (void);
u32  	AXGetInputSamplesPerSec       (void);
u32  	AXGetLibraryVersion		      (void);

AXPB_ERROR_CODE    AXSetVoiceInitialTimeDelay  (AXVPB *p, u16 timeDelay);

// =============================================================================
//
//  WinExt 専用 API
//
// =============================================================================
// TV および DRC をセットできる。RMT をセットしても効果が無い。
// デフォルトでは TV のみ true。
void AXSetEnableDevice(AXPBDeviceType type, bool enable);
bool AXGetEnableDevice(AXPBDeviceType type);

void AXSetEnableSwapPcmSample(bool enable);
bool AXGetEnableSwapPcmSample();

} // namespace winext
} // namespace internal
} // namespace nw

#endif // NW_WINEXT_AX_H_
