﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_VWRLYT_VIEWER_H_
#define NW_VWRLYT_VIEWER_H_

#include <nw/vwrlyt/vwrlyt_Config.h>
#if defined(NW_VWRLYT_ENABLE)

#include <nw/lyt.h>
#include <nw/font.h>
#include <nw/ctrl.h>
#include <nw/ut.h>
#include <nw/math.h>
#include <nw/dw.h>
#include <nw/vwrlyt/vwrlyt_AnimationManager.h>
#include <nw/vwrlyt/vwrlyt_Inputs.h>
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#include <nw/vwrlyt/win/vwrlyt_DirResourceAccessorWin.h>
#include <nw/vwrlyt/win/vwrlyt_ScreenShotWin.h>
#elif defined(NW_PLATFORM_CAFE)
#include <nw/vwrlyt/cafe/vwrlyt_HIOListenerCafe.h>
#include <nw/vwrlyt/cafe/vwrlyt_FindableArcResourceAccessorCafe.h>
#endif
#include <cstdarg>

namespace nw
{
namespace vwrlyt
{

class Viewer
{
public:
    enum Fps
    {
        FPS_60,
        FPS_30
    };

    Viewer();

    virtual ~Viewer();

    void Initialize(lyt::GraphicsResource* graphicsResource, ut::IAllocator* allocator, const lyt::Size& mainViewportSize);
    void Finalize();

    //------------------------------------------------------------------
    //! @brief プレビュー用バイナリリソースを読み込むパスを
    //!        デフォルトに指定します。
    //!
    //! @param[in] nwRoot   CAFE_CONTENT_DIR環境変数が指すディレクトリの
    //!                     FSにおけるパスを指定します。
    //!
    //! @return Viewerへの参照を返します。
    //!
    //! @details
    //! LayoutEditor がプレビュー用のバイナリリソースを出力するパスに
    //! 設定します。
    //!
    //!   $CAFE_CONTENT_DIR/data/content/
    //!
    //! この設定は実機の場合に使用されます。
    //!
    //! @sa SetPreviewPath
    //------------------------------------------------------------------
    Viewer& SetDefaultPreviewPath(const char* nwRoot)
    {
        m_PreviewPath.format("%s/content", nwRoot);
        return *this;
    }

    //------------------------------------------------------------------
    //! @brief プレビュー用バイナリリソースを読み込むパスを指定します。
    //!
    //! @param[in] path     FSのパスを指定します。
    //!
    //! @return Viewerへの参照を返します。
    //!
    //! @details
    //! 実機の場合に、PathCommandStr() において PreviewOption::path の
    //! 設定に利用されます。
    //!
    //! LayoutEditor からのプレビューを表示するには
    //! SetDefaultPreviewPath() を使用します。
    //!
    //! @sa SetDefaultPreviewPath
    //------------------------------------------------------------------
    Viewer& SetPreviewPath(const char* path)
    {
        m_PreviewPath.copy(path);
        return *this;
    }

    //------------------------------------------------------------------
    //! @brief プレビュー用バイナリリソースの置かれるパスの設定を取得します。
    //!
    //! @return パスへのポインタを返します。
    //------------------------------------------------------------------
    const char* GetPreviewPath() const
    {
        return m_PreviewPath.c_str();
    }

    //-------------------------------------------------------------------------
    //! @brief        ホストファイルシステムのパスをFSのパスに変換します。
    //!
    //! @param[out] outFsPath   FSにおけるパスが格納されます。
    //! @param[in] mountPoint   PC-FS(hfio01デバイス)のマウントポイントを指定します。
    //! @param[in] inHostPath   ホストファイルシステムにおける絶対パスを指定します。
    //!
    //! @return       パスが変換できた場合には true を返します。
    //-------------------------------------------------------------------------
    static bool ConvertHostToFsPath(
        ut::BufferedSafeString* outFsPath,
        const char* mountPoint,
        const char* inHostPath);

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    bool PreviewByCommandLineOption(int argc, const char** argv);
    LRESULT PreviewByMessageProcCallback(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
    bool IsScreenShotDone() const
    {
        return m_ScreenShot.GetState() == ScreenShot::cState_Done;
    }
#endif
    bool PreviewByCommandStr(const char* commandStr);

    void UpdateLayout();
    void UpdateSystem();
    void UpdateInputs(const nw::internal::dw::Inputs& inputs, const dev::Pad* vpad = NULL);
    void UpdateInputs(const IInputs& inputs)
    {
        this->UpdateInputs(nw::internal::dw::Inputs(inputs.GetPad(), inputs.GetMouse()), inputs.GetVPad());
    }
    void UpdateMenu();

    void DrawLayout();
    void DrawLayoutDRC();
    void DrawSystem();

    Fps GetFps() const
    {
        return m_Fps;
    }

    bool IsSRGBWriteEnable() const
    {
        return m_IsSRGBWriteEnable;
    }

    Viewer& SetUIRenderer(nw::internal::dw::UIRenderer* pUIRenderer)
    {
        m_pUIRenderer = pUIRenderer;
        return *this;
    }

    nw::internal::dw::UIRenderer* GetUIRenderer() const
    {
        return m_pUIRenderer;
    }

    //-------------------------------------------------------------------------
    //! @brief ButtonGroup を取得します。
    //!
    //! @return Viewer が所有する ButtonGroup への参照を返します。
    //-------------------------------------------------------------------------
    ctrl::ButtonGroup& GetButtonGroup()
    {
        return m_ButtonGroup;
    }

    //-------------------------------------------------------------------------
    //! @brief ButtonGroup を取得します。
    //!
    //! @return Viewer が所有する ButtonGroup への参照を返します。
    //-------------------------------------------------------------------------
    const ctrl::ButtonGroup& GetButtonGroup() const
    {
        return m_ButtonGroup;
    }

    //-------------------------------------------------------------------------
    //! @brief ControlCreator を設定します。
    //!
    //! @param[in] pControlCreator  ControlCreator を設定します。
    //!
    //! @return Viewer への参照を返します。
    //!
    //! @details
    //! pControlCreator に NULL を設定した場合には nw::ctrl::DefaultControlCreator が
    //! 使用されます。
    //!
    //! Viewer の初期状態では ControlCreator は NULL に設定されます。
    //!
    //! @sa GetButtonGroup
    //! @sa GetControlCreator
    //-------------------------------------------------------------------------
    Viewer& SetControlCreator(lyt::ControlCreator* pControlCreator)
    {
        m_pControlCreator = pControlCreator;
        return *this;
    }

    //-------------------------------------------------------------------------
    //! @brief ControlCreator を取得します。
    //!
    //! @return 設定されている ControlCreator へのポインタを返します。
    //!
    //! @sa SetControlCreator
    //-------------------------------------------------------------------------
    lyt::ControlCreator* GetControlCreator() const
    {
        return m_pControlCreator;
    }

    //-------------------------------------------------------------------------
    //! @brief TV画面のビューポートサイズを取得します。
    //!
    //! @return TV画面のビューポートサイズを返します。
    //!
    //! @sa Initialize
    //-------------------------------------------------------------------------
    const lyt::Size& GetMainViewportSize() const
    {
        return m_MainViewportSize;
    }

protected:
    enum
    {
        cPathMax = 512
    };

    enum PreviewMode
    {
        MODE_ANIMATION,
        MODE_CONTROL
    };

    enum ViewSize
    {
        VIEWSIZE_ORIGINAL,      // レイアウトのスクリーンサイズ
        VIEWSIZE_FULL_HD,       // フルHD
        VIEWSIZE_HD,            // HD
        VIEWSIZE_DRC,           // DRC
        VIEWSIZE_NUM_MAX
    };

    struct ViewSizeValue
    {
        f32 x;
        f32 y;
    };

    struct PreviewOption
    {
        PreviewMode previewMode;

        //! レイアウト・バイナリリソースのパスを指定します。
        //!
        //! 実機の場合は、アーカイブのFSにおけるパスを指定します。
        //!
        //! PCの場合は、バイナリリソースの出力されたディレクトリの、
        //! PCファイルシステムにおけるパスを指定します。
        ut::FixedSafeString<cPathMax> path;

        ut::FixedSafeString<cPathMax> layoutName;

        //! (未対応)
        u8 isSRGBWriteEnable;

        void reset()
        {
            previewMode = MODE_ANIMATION;
            path.clear();
            layoutName.clear();
            isSRGBWriteEnable = false;
        }
    };

    bool m_AnimateOnce; // 一回はアニメーションを行ったか
    lyt::DrawInfo m_DrawInfo;
    lyt::Layout* m_Layout;
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    DirResourceAccessor& m_ResourceAccessor;
    ScreenShot m_ScreenShot;
#elif defined(NW_PLATFORM_CAFE)
    FindableArcResourceAccessor m_ResourceAccessor;
#endif
    PreviewOption m_PreviewOption;
    ut::IAllocator* m_Allocator;
    AnimationManager m_AnimationMgr;
    ctrl::ButtonGroup m_ButtonGroup;
    lyt::ControlCreator* m_pControlCreator;
    ViewSize m_ViewSize;
    lyt::Size m_MainViewportSize;
    math::Matrix44 m_ProjectionMatrix;
    math::Matrix34 m_CameraViewMatrix;
    math::Matrix44 m_ProjectionMatrixDRC;
    math::Matrix34 m_CameraViewMatrixDRC;
    bool m_IsPerspectiveProjection;
    Fps m_Fps;
    float m_PerspectiveFovyDeg;
    float m_PerspectiveNear;
    float m_PerspectiveFar;
    bool m_IsViewMtxDirty;
    bool m_IsSRGBWriteEnable;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    struct LayoutProtocol
    {
        char Verb[ 32 ];
        char Data[ 1024 ];
    };
#elif defined(NW_PLATFORM_CAFE)
    HIOListener m_HIOListener;
#endif

    static const ViewSizeValue VIEWSIZE_VALUES[VIEWSIZE_NUM_MAX];
    static const char* VIEWSIZE_DESCRIPTIONS[VIEWSIZE_NUM_MAX];

    void StartPreview(const PreviewOption& option);
    void ParseCommandStr(PreviewOption* option, const char* commandStr);
    lyt::Size GetViewSize() const;
    void SetViewSize(ViewSize viewSize);

    nw::internal::dw::UIRenderer* m_pUIRenderer;
    ut::FixedSafeString<cPathMax> m_PreviewPath;

    ///---- デバッグメニューに関連するメンバ

    enum PageId
    {
        PAGE_ID_ANIM,
        PAGE_ID_CONTROL,
        PAGE_ID_PERSPECIVE_CONFIG
    };

    enum ItemId
    {
        ITEM_ID_ANIM_FRAME,
        ITEM_ID_ANIM_TAG,
        ITEM_ID_ANIM_TAG_NUM,
        ITEM_ID_ANIM_TARGET_MODE,
        ITEM_ID_ANIM_TARGET,
        ITEM_ID_ANIM_TARGET_NUM,
        ITEM_ID_ANIM_SIZE,
        ITEM_ID_ANIM_FPS,
        ITEM_ID_ANIM_CAMERA,

        ITEM_ID_CONTROL_FORCE_OFF,
        ITEM_ID_CONTROL_CANCEL_SELECT,
        ITEM_ID_CONTROL_DISABLE_ALL,
        ITEM_ID_CONTROL_SIZE,
        ITEM_ID_CONTROL_FPS,

        ITEM_ID_PERSPECIVE_FOVY,
        ITEM_ID_PERSPECIVE_NEAR,
        ITEM_ID_PERSPECIVE_FAR
    };

    static const int LABEL_LENGTH = 32;

    class MenuLabel : public nw::internal::dw::FixedLabel<LABEL_LENGTH>
    {
    public:
        s32 Format(const char* format_string, ...)
        {
            va_list va;
            va_start( va, format_string );

            ut::FixedSafeString<Viewer::LABEL_LENGTH> text;

            s32 ret = text.formatV(format_string, va);
            va_end( va );

            this->SetText(text.c_str());

            return ret;
        }
    };

    template <class ListBox, ItemId StartItemIndex>
    class MenuListBox : public ListBox
    {
        typedef ListBox Base;

    public:
        MenuListBox()
            : m_pViewer(NULL)
        {}

        MenuListBox& SetViewer(Viewer* pViewer)
        {
            m_pViewer = pViewer;

            return *this;
        }

    protected:
        /* override */
        bool OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
        {
            if (Base::OnUpdateFocusedInput(inputs))
            {
                return true;
            }

            if (m_pViewer == NULL)
            {
                return false;
            }

            s32 focusedItem = this->GetFocusedItemIndex();
            if (focusedItem < 0)
            {
                return false;
            }

            return m_pViewer->OnUpdateFocusedInput(
                Viewer::ItemId(focusedItem + StartItemIndex),
                inputs);
        }

        /* override */
        bool OnUpdatePointerInput(const nw::internal::dw::Inputs& inputs)
        {
            if (Base::OnUpdatePointerInput(inputs))
            {
                return true;
            }

            if (m_pViewer == NULL)
            {
                return false;
            }

            s32 focusedItem = this->GetFocusedItemIndex();
            if (focusedItem < 0)
            {
                return false;
            }

            return m_pViewer->OnUpdatePointerInput(
                Viewer::ItemId(focusedItem + StartItemIndex),
                inputs);
        }

    private:
        Viewer* m_pViewer;
    };

    enum MenuEvent
    {
        MENUEVENT_LEFT,
        MENUEVENT_RIGHT,
        MENUEVENT_INVOKE,
        MENUEVENT_CANCEL,
        MENUEVENT_X,
        MENUEVENT_NONE
    };

    MenuEvent GetMenuEvent(const nw::internal::dw::Inputs& inputs);

    MenuLabel m_AnimMenuItemFrame;
    MenuLabel m_AnimMenuItemTag;
    MenuLabel m_AnimMenuItemTagNum;
    MenuLabel m_AnimMenuItemTargetMode;
    MenuLabel m_AnimMenuItemTarget;
    MenuLabel m_AnimMenuItemTargetNum;
    MenuLabel m_AnimMenuItemSize;
    MenuLabel m_AnimMenuItemFps;
    MenuLabel m_AnimMenuItemCamera;
    MenuListBox<nw::internal::dw::FixedListBox<9>, ITEM_ID_ANIM_FRAME> m_AnimMenuListBox;

    MenuLabel m_ControlMenuItemForceOff;
    MenuLabel m_ControlMenuItemCancelSelect;
    MenuLabel m_ControlMenuItemDisableAll;
    MenuLabel m_ControlMenuItemSize;
    MenuLabel m_ControlMenuItemFps;
    MenuListBox<nw::internal::dw::FixedListBox<5>, ITEM_ID_CONTROL_FORCE_OFF> m_ControlMenuListBox;

    MenuLabel m_PerspMenuItemFovy;
    MenuLabel m_PerspMenuItemNear;
    MenuLabel m_PerspMenuItemFar;
    MenuListBox<nw::internal::dw::FixedListBox<3>, ITEM_ID_PERSPECIVE_FOVY> m_PerspMenuListBox;

    nw::internal::dw::ControlWindow m_MenuWindow;
    nw::internal::dw::WindowManager m_DwWindowManager;
    bool m_bMenuVisible;
    bool m_IsDisableAllButton;
    PageId m_MenuPage;
    PageId m_MenuPagePrev;

    void InitializeMenu();

    void UpdateMenuText();

    void SetMenuVisible(bool visible);

    bool GetMenuVisible() const
    {
        return m_bMenuVisible;
    }

    void SetMenuPage(PageId pageId);

    void SetMenuPageToPrev();

protected:
    virtual bool OnUpdateFocusedInput(ItemId itemId, const nw::internal::dw::Inputs& inputs);
    virtual bool OnUpdatePointerInput(ItemId itemId, const nw::internal::dw::Inputs& inputs);
};

} // namespace vwrlyt
} // namespace nw

#endif // NW_VWRLYT_ENABLE
#endif
