﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_RESULT_H_
#define NW_UT_RESULT_H_

#include <nw/types.h>

namespace nw {
namespace ut {

//---------------------------------------------------------------------------
//! @brief        結果を表すリザルトのベースクラスです。
//---------------------------------------------------------------------------
class Result
{
public:

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    Result() : m_Code( MASK_FAIL_BIT )
    {
    }

    //---------------------------------------------------------------------------
    //! @brief        コピーコンストラクタです。
    //!
    //! @param[in]    result  コピー
    //---------------------------------------------------------------------------
    /* ctor */ Result(const Result& result) : m_Code(result.m_Code) {}

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    code  エラーコードです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit Result(bit32 code) : m_Code(code) {}

    //---------------------------------------------------------------------------
    //! @brief        結果が失敗かどうかを返します。
    //!
    //! @return       処理の結果が失敗なら true を、成功なら false を返します。
    //---------------------------------------------------------------------------
    bool IsFailure() const
    {
        return (m_Code & MASK_FAIL_BIT) != 0;
    }

    //---------------------------------------------------------------------------
    //! @brief        結果が成功かどうかを返します。
    //!
    //! @return       処理の結果が成功なら true を、失敗なら false を返します。
    //---------------------------------------------------------------------------
    bool IsSuccess() const
    {
        return !IsFailure();
    }

    //---------------------------------------------------------------------------
    //! @brief        一致判定を行います。
    //!
    //! @param[in]    rhs     比較対象となるリザルト値です。
    //!
    //! @return       一致した場合に true を、一致しない場合に false を返します。
    //---------------------------------------------------------------------------
    bool operator ==(const Result& rhs) const { return this->m_Code == rhs.m_Code; }

    //---------------------------------------------------------------------------
    //! @brief        不一致判定を行います。
    //!
    //! @param[in]    rhs     比較対象となるリザルト値です。
    //!
    //! @return       一致しない場合に true を、一致する場合に false を返します。
    //---------------------------------------------------------------------------
    bool operator !=(const Result& rhs) const { return this->m_Code != rhs.m_Code; }

    //---------------------------------------------------------------------------
    //! @brief        エラーコードを取得します。
    //!
    //! @return       生のエラーコードを返します。
    //---------------------------------------------------------------------------
    bit32 GetCode() const
    {
        return m_Code;
    }

protected:
    static const bit32 MASK_FAIL_BIT = 0x80000000u; // 最上位ビット

    //---------------------------------------------------------------------------
    //! @brief        エラーコードの部分ビット列を取得します。
    //!
    //! @param[in]    mask    マスクです。
    //! @param[in]    shift   シフト値です。
    //!
    //! @return       エラーコードの部分ビット列を返します。
    //---------------------------------------------------------------------------
    bit32 GetCodeBits(bit32 mask, s32 shift) const
    {
        return ((m_Code & mask) >> shift);
    }

    bit32 m_Code;
};

} /* namespace ut */
} /* namespace nw */

#endif /* NW_UT_RESULT_H_ */
