﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_LOCK_OBJECT_H_
#define NW_UT_LOCK_OBJECT_H_

#include <nw/types.h>

namespace nw {
namespace ut {

//---------------------------------------------------------------------------
//! @brief        ロックオブジェクトのインターフェイスクラスです。
//---------------------------------------------------------------------------
class ILockObject
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    ILockObject() {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~ILockObject() {}


    //---------------------------------------------------------------------------
    //! @brief        ロックをおこないます。
    //---------------------------------------------------------------------------
    virtual void Lock() = 0;

    //---------------------------------------------------------------------------
    //! @brief        ロックを解除します。
    //---------------------------------------------------------------------------
    virtual void Unlock() = 0;
};



//---------------------------------------------------------------------------
//! @brief        ロックオブジェクトの実装をもつクラスです。
//---------------------------------------------------------------------------
template<typename TLockObject>
class LockObjectAdapter : public ILockObject
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[out]   lockObj 実際のロックオブジェクトです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit LockObjectAdapter( TLockObject& lockObj ) : ILockObject(), m_LockObj( lockObj ) {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~LockObjectAdapter() {}


    //---------------------------------------------------------------------------
    //! @brief        ロックをおこないます。
    //---------------------------------------------------------------------------
    virtual void Lock() { m_LockObj.Lock(); }

    //---------------------------------------------------------------------------
    //! @brief        ロックを解除します。
    //---------------------------------------------------------------------------
    virtual void Unlock() { m_LockObj.Unlock(); }

protected:
    TLockObject& m_LockObj;
};



//---------------------------------------------------------------------------
//! @brief        空のロックオブジェクトです。
//---------------------------------------------------------------------------
class NullLockObject
{
public:

    //---------------------------------------------------------------------------
    //! @brief        初期化をおこないます。
    //---------------------------------------------------------------------------
    void Initialize() {}

    //---------------------------------------------------------------------------
    //! @brief        終了処理をおこないます。
    //---------------------------------------------------------------------------
    void Finalize() {}

    //---------------------------------------------------------------------------
    //! @brief        ロックをおこないます。
    //---------------------------------------------------------------------------
    void Lock() {}

    //---------------------------------------------------------------------------
    //! @brief        ロックの取得を試みて、失敗した場合は false を返します。
    //!
    //! @return       ロックに成功した場合は true、失敗した場合は false を返します。
    //---------------------------------------------------------------------------
    bool TryLock() { return true; }

    //---------------------------------------------------------------------------
    //! @brief        ロックを解除します。
    //---------------------------------------------------------------------------
    void Unlock()  {}
};

} // namespace ut
} // namespace nw

#endif //  NW_UT_LOCK_OBJECT_H_
