﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_FORMAT_PARSER_16_H_
#define NW_UT_FORMAT_PARSER_16_H_

#include <nw/ut/ut_Inlines.h>

namespace nw {
namespace ut {
namespace detail {

/**
 * 出力用バッファ（内部クラス）
 *
 * 必要最低限のはみ出しチェックとNull終端の保証をする
 */
class OutBuffer16
{
public:
    OutBuffer16(char16* buffer, size_t length)
        : mBuffer(buffer)
        , mLength(length-1)
        , mIdx(0)
        , mIsOverflow(false)
    {
        NW_ASSERT(length > 0);
    }

    ~OutBuffer16() { close(); }

    //! @brief バッファに1文字追加します。
    //! @param[in] letter 追加する文字です。
    void append(char16 letter)
    {
        if (mIdx >= mLength)
        {
            mIsOverflow = true;
            return;
        }
        mBuffer[mIdx] = letter;
        mIdx++;
    }

    //! @brief バッファを指定文字で埋めます。
    //! @param[in] letter 埋める文字です。
    //! @param[in] letter_num 埋める文字数です。
    void fill(char16 letter, u16 letter_num)
    {
        while (letter_num > 0)
        {
            if (mIdx >= mLength)
            {
                mIsOverflow = true;
                return;
            }
            mBuffer[mIdx] = letter;
            mIdx++;
            letter_num--;
        }
    }

    //! @brief 終端文字を挟みます。
    void close()
    {
        if (mIdx <= mLength) mBuffer[mIdx] = '\0';
    }

    //! @brief 文字数を取得します。
    //! @return 文字数です。
    u32 getLetterNum() const {return (u32)mIdx;}

    //! @brief 文字列を取得します。
    //! @return 文字列です。
    const char16* getString() const {return mBuffer;}

    //! @brief バッファあふれしたかを取得します。
    //! @return バッファあふれした場合は true を返します。
    bool isOverflow() const {return mIsOverflow;}

    //! @brief 文字列を末尾にコピーします。
    //! @param[in] src コピーする文字列です。
    void copyFrom(const OutBuffer16& src)
    {
        for (int i=0; i<src.mIdx; i++)
        {
            append(src.mBuffer[i]);
        }
    }

    //! @brief 文字列を末尾に逆順でコピーします。
    //! @param[in] src コピーする文字列です。
    void copyFromInverse(const OutBuffer16& src)
    {
        for (int i=src.mIdx-1; i>=0; i--)
        {
            append(src.mBuffer[i]);
        }
    }

private:
    char16* mBuffer;
    s32     mLength; //< 実際に文字列を書き込める長さで、バッファより1文字分小さい
    s32     mIdx;
    bool    mIsOverflow;
};

//------------------------------------------------------------------------------

/**
 * フォーマット文字列を解析して引数を文字列化する（内部クラス）
 *
 * 書式指定は以下の使用に対応しています。
 *  % [flags] [width] [.precision] [size] type
 *
 * flags :    以下のフラグを順不同に指定できます。
 *     '-'    左詰めで出力します。
 *     '+'    符号付きの場合符号（+または-）を必ず出力します。
 *     '0'    最小幅になるまで0でパディングします。
 *     ' '    符号付き正数の場合' '（半角スペース）をつけます。
 *     '#'    o,x,X,p の場合値の前にそれぞれ"0","0x","0X","0x"をつけます。
 *            e,E,f の場合必ず小数点を出力します。
 *            g,G の場合必ず小数点とそれに続く0を出力します。
 *
 * width :    10進整数を指定すると、出力値がその文字数に満たない場合は' 'でパディングします。
 *            flagsで'0'を指定した場合は’'ではなく'0'が使われます。
 *            '*'を指定すると、引数リストで指定した値が使われます。
 *
 * prec :    10進整数を指定すると、typeに応じて出力の精度を制御します。
 *           '*'を指定すると、引数リストで指定した値が使われます。
 *           type毎に以下の意味を持ちます。
 *           c,p    精度指定は無効です。
 *           d,i,u,o,x,X    出力する桁数の最小値を指定します。
 *           e,E    小数点以下の出力桁数を指定します。
 *           f    小数点以下の桁数を指定します。
 *           g,G    有効桁数の最大値を指定します。
 *           s    最大文字数を指定します。
 *
 * length:    引数リストの値を解釈する型を指定します。
 *            type毎に以下の型として解釈します。
 *            hh        h     l     ll      無指定
 *            d,i       s8    s16   s32     s64     s32
 *            u,o,x,X   u8    u16   u32     u64     u32
 *            e,E,f,g,G f64   f64   f64     f64     f64
 *            c         char  char  char16  char16  char16
 *            s         char* char* char16* char16* char16*
 *            p         void* void* void*   void*   void*
 *
 * type : 引数リストの値を解釈する型、出力形式を指定します。
 *    'd' 符号付10進整数
 *    'i' 符号付10進整数
 *    'u' 符号無10進整数
 *    'o' 符号無8進整数
 *    'x' 符号無16進整数、アルファベットは小文字を使用
 *    'X' 符号無16進整数、アルファベットは大文字を使用
 *    'e' 浮動小数、[s]d.dddesddd形式（dは数字、sは符号）
 *    'E' 浮動小数、[s]d.dddEsddd形式（dは数字、sは符号）
 *    'f' 浮動小数、[s]dddd.dddd形式（dは数字）
 *    'g' 浮動小数、'e'形式と'f'形式の短い方
 *    'G' 浮動小数、'E'形式と'f'形式の短い方
 *    'c' 文字
 *    's' 文字列
 *    'p' ポインタを16進8桁で表す
 *    '%' '%'の文字をそのまま出す
 *
 * typeに'n'を指定して、書き込んだ文字数を引数リストのint*のアドレスに出力する機能は
 * セキュリティの観点から非対応です。
 */
class FormatParser16
{
public:
    explicit FormatParser16(va_list args);

    //! @brief 引数の値を文字列化する処理を実行
    //! @param[out] buffer 文字列化したものを格納するバッファです。
    //! @param[in] format 書式文字列です。
    //! @return 文字列化した長さです。
    u32 execute(OutBuffer16* buffer, const char16* format);

private:
    /// 解析結果をクリア
    void clear_();

    /// フラグの解析
    const char16* parseFlag_(const char16* format_string);

    /// 幅の解析
    const char16* parseWidth_(const char16* format_string);

    /// 精度の解析
    const char16* parsePrecision_(const char16* format_string);

    /// 引数サイズの解析
    const char16* parseSize_(const char16* format_string);

    /// パラメータの読み込み
    template <typename T>
    T getArg_() {return va_arg(m_Args, T);}


    /// 整数値の出力
    void printInteger_(OutBuffer16* buffer, u32 radix, bool is_signed, bool is_capital = false);

    /// 整数値を逆順の文字列に変換
    template <typename T>
    static void printIntegerInverseString_(OutBuffer16* buffer, u32 radix, bool is_capital, T value);

    /// 文字の出力
    void printCharacter_(OutBuffer16* buffer);

    /// 文字列の出力
    void printString_(OutBuffer16* buffer);

    /// 文字列の出力
    template <typename T>
    void printStringCore_(OutBuffer16* buffer, const T* string);

    /// 浮動小数の出力
    void printFloat_(OutBuffer16* buffer, char16 type, bool is_capital);

private:

    enum Flag
    {
        cFlag_JustifyLeft = 0x01,
        cFlag_PrintSign   = 0x02,
        cFlag_PadZero     = 0x04,
        cFlag_PrintSpace  = 0x08,
        cFlag_SharpFlag   = 0x10
    };

    enum ArgLength
    {
        cArgLength_hh,
        cArgLength_h,
        cArgLength_default,
        cArgLength_l,
        cArgLength_ll
    };

    va_list     m_Args;

    u32         m_Flags;
    u32         m_Width;
    u32         m_Precision;
    bool        m_IsPrecisionDefined;
    ArgLength   m_ArgLength;


    // 浮動小数関連

    enum
    {
        cOutputMaxPrecision = 17,                   //< 出力する最大桁数
        cDecimalPrecision = 21,                     //< f64を10進変換する際に計算する桁数
        cFractionBitNum = 52,                       //< f64の仮数部のビット数
        cExponentMin = -1022,                       //< f64指数の最小値
        cExponentMax = 1023,                        //< f64指数の最大値
        cPlaceMin = cExponentMin - cFractionBitNum, //< f64最小の桁の指数
        cPlaceMax = cExponentMax                    //< f64最大の桁の指数
    };

    // 2^n を10進で表した時の上位桁の情報
    // 2^n = 0. decimals[0] decimals[1] ... decimals[cDecimalPrecision-2] * 10 ^ shift
    struct DigitData
    {
        s32 shift;
        u8 decimals[cDecimalPrecision-1];
    };

    // i 番の要素が 2^(i-cPlaceMin) の10進表現
    static const DigitData cDigitTable[cPlaceMax - cPlaceMin + 1];

};


//------------------------------------------------------------------------------

FormatParser16::FormatParser16(va_list args)
{
#if defined(NW_PLATFORM_CAFE)
    va_copy(m_Args, args);
#else
    m_Args = args;
#endif
    clear_();
}

//------------------------------------------------------------------------------
/**
 * 引数の値を文字列化する処理を実行
 */
u32 FormatParser16::execute(OutBuffer16* buffer, const char16* format)
{
    clear_();

    const char16* format_string = format;

    format_string = parseFlag_(format_string);
    format_string = parseWidth_(format_string);
    format_string = parsePrecision_(format_string);
    format_string = parseSize_(format_string);

    switch(*format_string)
    {
    case '%': // '%'を出力
        buffer->append('%');
        break;

    case 'd': // 符号付10進整数
    case 'i':
        printInteger_(buffer, 10, true, false);
        break;

    case 'u': // 符号無10進整数
        printInteger_(buffer, 10, false, false);
        break;

    case 'o': // 符号無8進整数
        printInteger_(buffer, 8, false, false);
        break;

    case 'x': // 符号無16進整数（小文字）
        printInteger_(buffer, 16, false, false);
        break;

    case 'X': // 符号無16進整数（大文字）
        printInteger_(buffer, 16, false, true);
        break;

    case 'p':    // ポインタのアドレス
        // 符号無16進整数8ケタとして扱う
        m_Precision = 8;
        m_Flags &= ~(cFlag_PadZero);
        m_ArgLength = cArgLength_default;
        printInteger_(buffer, 16, false, false);
        break;

    case 'c': // 文字
        printCharacter_(buffer);
        break;

    case 's': // 文字列
        printString_(buffer);
        break;

    case 'f': // 浮動小数 dddd.dddd
        printFloat_(buffer, 'f', false);
        break;

    case 'e': // 浮動小数 d.ddddesddd
        printFloat_(buffer, 'e', false);
        break;

    case 'E': // 浮動小数 d.ddddEsddd
        printFloat_(buffer, 'e', true);
        break;

    case 'g': // 浮動小数 'f'形式と'e'形式の短い方
        printFloat_(buffer, 'g', false);
        break;

    case 'G': // 浮動小数 'f'形式と'E'形式の短い方
        printFloat_(buffer, 'g', true);
        break;

    default:  // 解析失敗 -> 何も出力せず打ち切り
        return 0;
    }

    format_string++;
    s32 letter_num = format_string - format;
    return letter_num;
}

//------------------------------------------------------------------------------

/**
 * 解析結果をクリア
 */
void FormatParser16::clear_()
{
    m_Flags = 0;
    m_Width = 0;
    m_Precision = 0;
    m_IsPrecisionDefined = false;
    m_ArgLength = cArgLength_default;
}

//------------------------------------------------------------------------------

/**
 * フラグの解析
 */
const char16* FormatParser16::parseFlag_(const char16* format_string)
{
    for(;;)
    {
        char16 letter = *format_string;

        switch(letter)
        {
        case '-':
            m_Flags |= cFlag_JustifyLeft;
            break;

        case '+':
            m_Flags |= cFlag_PrintSign;
            break;

        case '0':
            m_Flags |= cFlag_PadZero;
            break;

        case ' ':
            m_Flags |= cFlag_PrintSpace;
            break;

        case '#':
            m_Flags |= cFlag_SharpFlag;
            break;

        default:
            return format_string;
        }

        format_string++;
    }
}

//------------------------------------------------------------------------------

/**
 * 幅の解析
 */
const char16* FormatParser16::parseWidth_(const char16* format_string)
{
    char16 letter = *format_string;

    if (letter == '*')
    {
        // 引数で値を指定
        m_Width = getArg_<u32>();
        format_string++;
    }
    else if (letter >= '1' && letter <= '9') // '0'はparseFlagで処理されているので1文字目は'0'にはならない
    {
        // 数値文字列を数値に変換
        m_Width = (letter - '0');
        for(;;)
        {
            format_string++;
            letter = *format_string;
            if (letter < '0' || letter > '9') break;
            m_Width *= 10;
            m_Width += (letter - '0');
        }
    }

    return format_string;
}

//------------------------------------------------------------------------------

/**
 * 精度の解析
 */
const char16* FormatParser16::parsePrecision_(const char16* format_string)
{
    char16 letter = *format_string;

    if (letter == '.')
    {
        m_IsPrecisionDefined = true;

        format_string++;
        letter = *format_string;

        if (letter == '*')
        {
            // 引数で値を指定
            m_Precision = getArg_<u32>();
            format_string++;
        }
        else if (letter >= '0' && letter <= '9')
        {
            // 数値文字列を数値に変換
            m_Precision = (letter - '0');
            for(;;)
            {
                format_string++;
                letter = *format_string;
                if (letter < '0' || letter > '9') break;
                m_Precision *= 10;
                m_Precision += (letter - '0');
            }
        }
    }

    return format_string;
}

//------------------------------------------------------------------------------

/**
 * 引数サイズの解析
 */
const char16* FormatParser16::parseSize_(const char16* format_string)
{
    char16 letter = *format_string;

    switch(letter)
    {
    case 'h':
        {
            format_string++;
            letter = *format_string;
            if (letter == 'h')
            {
                format_string++;
                m_ArgLength = cArgLength_hh;
            }
            else
            {
                m_ArgLength = cArgLength_h;
            }
        }
        break;

    case 'l':
        {
            format_string++;
            letter = *format_string;
            if (letter == 'l')
            {
                format_string++;
                m_ArgLength = cArgLength_ll;
            }
            else
            {
                m_ArgLength = cArgLength_l;
            }
        }
        break;
    }

    return format_string;
}

//------------------------------------------------------------------------------

/**
 * 整数値の出力
 */
void FormatParser16::printInteger_(OutBuffer16* buffer, u32 radix, bool is_signed, bool is_capital)
{
    // 実装によっては値が0のときは特殊な出力をするようだが、
    // 一般的なドキュメントにはそのような記述は見当たらないので
    // 特に要望が出なければ普通に扱う
    // 例えば、winでは
    //  "%.0d"  => ""
    //  "%#.0o" => "0"
    //  "%#.0x" => ""
    //  "%#x"   => "0"
    // みたいなことになっている


    char16 temp_buffer_array[32];
    OutBuffer16 temp_buffer(temp_buffer_array, 32);;
    bool is_minus = false;

    // 数値を文字列化
    switch(m_ArgLength)
    {
    case cArgLength_hh:
        if (is_signed)
        {
            s8 val = (s8)getArg_<s32>();
            if (val >= 0)
            {
                printIntegerInverseString_<u32>(&temp_buffer, radix, is_capital, (u32)val);
            }
            else
            {
                printIntegerInverseString_<u32>(&temp_buffer, radix, is_capital, (u32)(-val));
                is_minus = true;
            }
        }
        else
        {
            u8 val = (u8)getArg_<u32>();
            printIntegerInverseString_<u32>(&temp_buffer, radix, is_capital, (u32)val);
        }
        break;

    case cArgLength_h:
        if (is_signed)
        {
            s16 val = (s16)getArg_<s32>();
            if (val >= 0)
            {
                printIntegerInverseString_<u32>(&temp_buffer, radix, is_capital, (u32)val);
            }
            else
            {
                printIntegerInverseString_<u32>(&temp_buffer, radix, is_capital, (u32)(-val));
                is_minus = true;
            }
        }
        else
        {
            u16 val = (u16)getArg_<u32>();
            printIntegerInverseString_<u32>(&temp_buffer, radix, is_capital, (u32)val);
        }
        break;

    case cArgLength_default:
    case cArgLength_l:
        if (is_signed)
        {
            s32 val = getArg_<s32>();
            if (val >= 0)
            {
                printIntegerInverseString_<u32>(&temp_buffer, radix, is_capital, (u32)val);
            }
            else
            {
                printIntegerInverseString_<u32>(&temp_buffer, radix, is_capital, (u32)(-val));
                is_minus = true;
            }
        }
        else
        {
            u32 val = getArg_<u32>();
            printIntegerInverseString_<u32>(&temp_buffer, radix, is_capital, val);
        }
        break;

    case cArgLength_ll:
        if (is_signed)
        {
            s64 val = getArg_<s64>();
            if (val >= 0)
            {
                printIntegerInverseString_<u64>(&temp_buffer, radix, is_capital, (u64)val);
            }
            else
            {
                printIntegerInverseString_<u64>(&temp_buffer, radix, is_capital, (u64)(-val));
                is_minus = true;
            }
        }
        else
        {
            u64 val = getArg_<u64>();
            printIntegerInverseString_<u64>(&temp_buffer, radix, is_capital, val);
        }
        break;
    }

    // パディング幅の計算

    u32 pad_zero_num = 0;
    u32 pad_left_space_num = 0;
    u32 pad_right_space_num = 0;

    {
        u32 letter_num = temp_buffer.getLetterNum();

        if (m_IsPrecisionDefined && (temp_buffer.getLetterNum() < m_Precision))
        {
            // 精度が指定されていたらその桁数まで'0'で埋める
            pad_zero_num = m_Precision - letter_num;
            letter_num = m_Precision;
        }

        if (is_signed)
        {
            if (is_minus || (m_Flags & (cFlag_PrintSign | cFlag_PrintSpace)))
            {
                // '+', '-', ' ' のいずれかの符号がつく
                letter_num++;
            }
        }
        else
        {
            if (m_Flags & cFlag_SharpFlag)
            {
                if (radix == 8)
                {
                    // '0'がつく
                    letter_num++;
                }
                else if (radix == 16)
                {
                    // "0x", "0X"のいずれかがつく
                    letter_num += 2;
                }
            }
        }

        if (letter_num < m_Width)
        {
            if (m_Flags & cFlag_JustifyLeft)
            {
                pad_right_space_num += (m_Width - letter_num);
            }
            else if ((m_Flags & cFlag_PadZero) && (m_IsPrecisionDefined == false))
            {
                pad_zero_num += (m_Width - letter_num);
            }
            else
            {
                pad_left_space_num += (m_Width - letter_num);
            }
        }
    }


    // 出力バッファに書き込む
    {
        bool is_vacant = true;

        // buffer->isOverflow()がtrueを返した時点で処理を打ち切れるが、
        // 何度もチェックするのがかえって遅そうなので最後までやりきる

        // 左のスペース
        buffer->fill(' ', pad_left_space_num);

        // 記号類
        if (is_signed)
        {
            if (is_minus)
            {
                buffer->append('-');
            }
            else if (m_Flags & cFlag_PrintSign)
            {
                buffer->append('+');
            }
            else if (m_Flags & cFlag_PrintSpace)
            {
                buffer->append(' ');
            }
        }
        else
        {
            if (m_Flags & cFlag_SharpFlag)
            {
                if (radix == 8)
                {
                    buffer->append('0');
                }
                else if (radix == 16)
                {
                    buffer->append('0');
                    buffer->append(is_capital ? 'X' : 'x');
                }
            }
        }

        // ゼロのパディング
        buffer->fill('0', pad_zero_num);

        // 数値
        buffer->copyFromInverse(temp_buffer);

        // 右のスペース
        buffer->fill(' ', pad_right_space_num);
    }
}

//------------------------------------------------------------------------------

/**
 * 整数値を逆順の文字列に変換
 */
template <typename T>
void FormatParser16::printIntegerInverseString_(OutBuffer16* buffer, u32 radix, bool is_capital, T value)
{
    if (value == 0)
    {
        buffer->append('0');
        return;
    }

    switch(radix)
    {
    case 8:
        while (value > 0)
        {
            u16 val = (u16)(value & 0x07);
            char16 letter = (char16)('0' + val);
            buffer->append(letter);
            value >>= 3;
        }
        break;

    case 10:
        while (value > 0)
        {
            T next = value / 10;
            u16 val = (u16)(value - (next * 10));
            char16 letter = (char16)('0' + val);
            buffer->append(letter);
            value = next;
        }
        break;

    case 16:
        u16 alphabet_offset = (u16)((is_capital ? 'A' : 'a') - 10);
        while (value > 0)
        {
            u16 val = (u16)(value & 0x0f);
            char16 letter = (char16)((val < 10) ? ('0' + val) : (alphabet_offset + val));
            buffer->append(letter);
            value >>= 4;
        }
        break;
    }
}


//------------------------------------------------------------------------------

/**
 * 文字の出力
 */
void FormatParser16::printCharacter_(OutBuffer16* buffer)
{
    char16 letter;

    // 文字の決定
    switch(m_ArgLength)
    {
    case cArgLength_hh:
    case cArgLength_h:
        letter = (char)getArg_<u32>();
        break;

    case cArgLength_default:
    case cArgLength_l:
    case cArgLength_ll:
        letter = (char16)getArg_<u32>();
        break;
    }

    u32 letter_num = (letter != '\0' ? 1 : 0);

    if (letter_num < m_Width)
    {
        // 幅指定の方が長いのでパディングがつく
        u32 padding_num = m_Width - letter_num;

        // 左のパディング
        if ((m_Flags & cFlag_JustifyLeft) == 0)
        {
            buffer->fill(((m_Flags & cFlag_PadZero) ? '0' : ' '), padding_num);
        }

        // 文字本体
        if (letter != '\0')
        {
            buffer->append(letter);
        }

        // 右のパディング
        if (m_Flags & cFlag_JustifyLeft)
        {
            buffer->fill(' ', padding_num);
        }
    }
    else
    {
        // 文字本体のみ
        if (letter != '\0')
        {
            buffer->append(letter);
        }
    }
}

//------------------------------------------------------------------------------

/**
 * 文字列の出力
 */
void FormatParser16::printString_(OutBuffer16* buffer)
{
    switch(m_ArgLength)
    {
    case cArgLength_hh:
    case cArgLength_h:
        // char文字列
        printStringCore_<char>(buffer, getArg_<char*>());
        break;

    case cArgLength_default:
    case cArgLength_l:
    case cArgLength_ll:
        // char16文字列
        printStringCore_<char16>(buffer, getArg_<char16*>());
        break;
    }
}

//------------------------------------------------------------------------------

/**
 * 文字列の出力
 */
template <typename T>
void FormatParser16::printStringCore_(OutBuffer16* buffer, const T* string)
{
    if (string == NULL)
    {
        // NULL文字列が与えられたとして扱う
        static const T cNullString[1] = {'\0'};
        string = cNullString;
    }

    if (m_Flags & cFlag_JustifyLeft)
    {
        // 左詰めはとりあえず文字列を出す

        u32 letter_num = 0;
        if (m_IsPrecisionDefined)
        {
            // precisionの文字数まで出す
            const T* ptr = string;
            while(*ptr != '\0' && letter_num < m_Precision)
            {
                buffer->append(*ptr);
                ptr++;
                letter_num = ptr - string;
            }
        }
        else
        {
            // 文字列を全部出す
            const T* ptr = string;
            while(*ptr != '\0')
            {
                buffer->append(*ptr);
                ptr++;
            }
            letter_num = ptr - string;
        }

        // widthに満たない部分はスペースで埋める
        if (letter_num < m_Width)
        {
            buffer->fill(' ', m_Width - letter_num);
        }
    }
    else
    {
        // 右詰めはまず文字数を数えて先にパディング

        u32 string_length = 0;
        if (m_IsPrecisionDefined)
        {
            // precisionの文字数まで
            const T* ptr = string;
            while(*ptr != '\0' && string_length < m_Precision)
            {
                ptr++;
                string_length = ptr - string;
            }
        }
        else
        {
            // 文字列全体
            const T* ptr = string;
            while(*ptr != '\0')
            {
                ptr++;
            }
            string_length = ptr - string;
        }

        // widthに満たない部分はスペースで埋める
        if (string_length < m_Width)
        {
            buffer->fill(((m_Flags & cFlag_PadZero) ? '0' : ' '), m_Width - string_length);
        }

        // 文字列を出す
        for (u32 i=0; i<string_length; i++)
        {
            buffer->append(string[i]);
        }
    }
}

//------------------------------------------------------------------------------

/**
 * 浮動小数の出力
 */
void FormatParser16::printFloat_(OutBuffer16* buffer, char16 type, bool is_capital)
{
    static const u64 cFlagMask     = 0x8000000000000000;
    static const u64 cExponentMask = 0x7ff0000000000000;
    static const u64 cFractionMask = 0x000fffffffffffff;

    bool is_minus;
    u32 exponent2;
    u64 fraction2;

    // 値を解析
    {
        // f64*からu64*へのreinterpret_castが警告が出るのでunionを使って変換する
        union
        {
            f64 mF64;
            u64 mU64;
        } value;
        value.mF64 = getArg_<f64>();
        u64 value_d = value.mU64;

        is_minus = ((value_d & cFlagMask) != 0);
        exponent2 = (u32)((value_d & cExponentMask) >> cFractionBitNum);
        fraction2 = (value_d & cFractionMask);
    }

    // NaN, Infは文字列として出力
    if (exponent2 == 0x07ff)
    {
        // 精度指定は無効にして全部出し切る
        m_IsPrecisionDefined = false;

        // cFlag_PadZero 指定の場合 "0000-NaN" みたいな出力になって変と言われれば変だが
        // 厳密に仕様が決まっているわけでもないようなので要望がなければそれでいいことにしておく

        if (fraction2 != 0)
        {
            // NaN
            if (is_minus)
                printStringCore_(buffer, "-NaN");
            else if (m_Flags & cFlag_PrintSign)
                printStringCore_(buffer, "+NaN");
            else if (m_Flags & cFlag_PrintSpace)
                printStringCore_(buffer, " NaN");
            else
                printStringCore_(buffer, "NaN");
        }
        else
        {
            // Inf
            if (is_minus)
                printStringCore_(buffer, "-Inf");
            else if (m_Flags & cFlag_PrintSign)
                printStringCore_(buffer, "+Inf");
            else if (m_Flags & cFlag_PrintSpace)
                printStringCore_(buffer, " Inf");
            else
                printStringCore_(buffer, "Inf");
        }

        return;
    }


    // 精度
    u32 precision;
    if (m_IsPrecisionDefined)
    {
        // 安全のため最大値を制限しておく
        precision = ut::Clamp<u32>( m_Precision, 0, 512 );
    }
    else
    {
        precision = 6;
    }


    // 10進数での値
    // 0 . fracrion10[0] fraction10[1] fraction10[2] ... fraction10[fraction10_size - 1] * 10 ^ exponent10
    u32* fraction10;
    u32 fraction10_size;
    s32 exponent10;

    u32 decimals[cDecimalPrecision];

    {
        static const s32 cCheckBit     = 63;
        static const u64 cCheckBitMask = 0x8000000000000000;

        // 10進変換用のビット列を作る
        // 2進の最上位桁が63番ビットに来るように正規化して
        // そのビットに相当するcDigitTableのインデックスを求める
        u32 digit_table_idx;
        u64 bits;
        {
            if (exponent2 == 0)
            {
                if (fraction2 == 0)
                {
                    // ゼロ -> 2^0 の位置
                    digit_table_idx = 0 - cPlaceMin;
                    bits = 0;
                }
                else
                {
                    // 非正規化数 -> 最下位ビットが 0 の位置
                    digit_table_idx = cFractionBitNum;
                    bits = fraction2 << (cCheckBit - cFractionBitNum);

                    // 63番ビットが1になるまでシフトする
                    do
                    {
                        bits <<= 1;
                        digit_table_idx--;
                    } while((bits & cCheckBitMask) == 0);
                }
            }
            else
            {
                // 正規化数 -> 最下位ビットが exponent-1 の位置
                digit_table_idx = exponent2 - 1 + cFractionBitNum;

                // けち表現のビットを戻す
                bits = cCheckBitMask | (fraction2 << (cCheckBit - cFractionBitNum));
            }
        }

        // 10進変換
        {
            // 変換時は1桁大きく計算する
            // decimals[0] . decimals[1] decimals[2] ... decimals[cDecimalPrecision-1] * 10 ^ shift
            // fraction10 や cDicitTable とインデックスが１つずれているので注意
            // 2進の最上位ビットの最上位桁が decimals[1] に相当し、
            // 合計した値は高々その2倍なので decimals[0] の桁まで必要

            s32 shift = cDigitTable[digit_table_idx].shift;
            std::memset( decimals, 0, sizeof(decimals) );

            while (bits != 0)
            {
                NW_TASSERTMSG(((s32)digit_table_idx) >= 0, "Inner error. Please notice sead members.");

                if (bits & cCheckBitMask)
                {
                    // cDigitTableの値をdecimalsに足しこむ

                    const DigitData& cDigitData = cDigitTable[digit_table_idx];

                    s32 offset = shift - cDigitData.shift + 1;
                    for (int i=offset; i<cDecimalPrecision; i++)
                    {
                        decimals[i] += cDigitData.decimals[i - offset];
                    }

                    // 計算する桁よりもう一桁下までデータがある場合は、四捨五入して値を加味する
                    if (offset > 0 && offset >= cDecimalPrecision)
                    {
                        if (cDigitData.decimals[cDecimalPrecision - offset] >= 5)
                        {
                            decimals[cDecimalPrecision - 1]++;
                        }
                    }
                }

                bits <<= 1;
                digit_table_idx--;
            }

            // 繰り上がりをまとめて計算
            for (int i=cDecimalPrecision-1; i>=1; i--)
            {
                u32 r = decimals[i] / 10;
                decimals[i-1] += r;
                decimals[i] -= (r * 10);
            }

            if (decimals[0] == 0)
            {
                // 最上位桁が0なのでそのまま
                fraction10 = decimals + 1;
                exponent10 = shift;
            }
            else
            {
                NW_TASSERTMSG(decimals[0] == 1, "Inner error. Please notice sead members.");

                // 最上位桁が1なので1桁ずらす
                fraction10 = decimals;
                exponent10 = shift + 1;
            }
        }

        // この段階で
        // 0 . fraction10[0] fraction10[1] ... fraction10[cDecimalPrecition-2] * 10 ^ exponent10

        // 四捨五入
        {
            // 四捨五入をする桁
            s32 round_idx;
            switch(type)
            {
            case 'f':
                round_idx = (s32)(precision + exponent10);
                break;
            case 'e':
                round_idx = (s32)(precision + 1);
                break;
            case 'g':
                if ( precision < 1 )
                {
                    round_idx = 1;
                }
                break;
            }

            if (round_idx < 0)
            {
                fraction10_size = 0;
            }
            else
            {
                if (round_idx > cOutputMaxPrecision)
                {
                    round_idx = cOutputMaxPrecision;
                }

                // 実際に四捨五入
                if (fraction10[round_idx] >= 5)
                {
                    u32* p = fraction10 + (round_idx-1);
                    (*p)++;

                    // 繰り上げの計算
                    while((*p >= 10) && (p >= decimals)) // decimals[0] はインクリメントされても高々2なので2つ目の条件は不要なはずだが念のため
                    {
                        (*p) -= 10;
                        p--;
                        (*p)++;
                    }

                    NW_TASSERTMSG(p >= fraction10 - 1, "Inner error. Please notice sead members.");
                    if (p == fraction10 - 1)
                    {
                        // 最上位桁が変わったので1桁ずらす
                        fraction10--;
                        exponent10++;
                    }
                }

                fraction10_size = round_idx;
            }
        }
    }


    {

        bool is_exponent = false;
        bool is_point = false;

        // 文字数
        u32 pad_left_space_num = 0;      // 左スペース
        u32 pad_zero_num = 0;            // ゼロパディング
        u32 integer_num = 0;             // 整数部
        u32 alignment_integer_num = 0;   // 整数部桁合わせ
        u32 alignment_decimal_num = 0;   // 小数部桁合わせ
        u32 decimal_num = 0;             // 小数部
        u32 alignment_precision_num = 0; // 精度合わせ
        u32 pad_right_space_num = 0;     // 右スペース

        // 文字数計算
        {
            u32 letter_num = 0;

            // 符号
            if (is_minus || ((m_Flags & (cFlag_PrintSign | cFlag_PrintSpace)) != 0))
            {
                letter_num += 1;
            }

            switch(type)
            {
            case 'f':
                {
                    if (exponent10 >= (s32)fraction10_size)
                    {
                        //  d d d d d d d d d d d 0 0 0 [.] 0 0 0 0 0 0 0 0
                        // |<- fraction10_size ->|
                        //                                 |<- precision ->|
                        // |<-       exponent10      ->|

                        integer_num = fraction10_size;
                        alignment_integer_num = (u32)(exponent10 - integer_num);
                        is_point = (precision > 0) || ((m_Flags & cFlag_SharpFlag) != 0);
                        alignment_precision_num = precision;
                    }
                    else if (exponent10 >= 0)
                    {
                        //  d d d d d d d d d . d d 0 0 0 0 0 0
                        // |<-  fraction10_size  ->|
                        //                     |<- precision ->|
                        // |<- exponent10  ->|

                        integer_num = (u32)exponent10;
                        is_point = true;
                        decimal_num = (u32)(fraction10_size - exponent10);
                        alignment_precision_num = (u32)(precision + exponent10 - fraction10_size);
                    }
                    else
                    {
                        alignment_integer_num = 1;
                        is_point = true;

                        if ((s32)precision <= -exponent10)
                        {
                            // 0 . 0 0 0 0 0 0 0 0
                            //                        |<- fraction10_size ->|
                            //    |<- precision ->|
                            //    |<-  -exponent10  ->|

                            alignment_decimal_num = precision;
                        }
                        else if ((s32)precision <= -exponent10 + fraction10_size)
                        {
                            // 0 . 0 0 0 0 0 0 0 0 0 0 d d d d d
                            //                        |<- fraction10_size ->|
                            //    |<-       precision         ->|
                            //    |<-  -exponent10  ->|

                            alignment_decimal_num = (u32)(-exponent10);
                            decimal_num = (u32)(precision + exponent10);
                        }
                        else
                        {
                            // 0 . 0 0 0 0 0 0 0 0 0 0 d d d d d d d d d d d 0 0
                            //                        |<- fraction10_size ->|
                            //    |<-               precision                 ->|
                            //    |<-  -exponent10  ->|

                            alignment_decimal_num = (u32)(-exponent10);
                            decimal_num = fraction10_size;
                            alignment_precision_num = (u32)(precision - fraction10_size + exponent10);
                        }
                    }
                }
                break;

            case 'e':
                {
                    // d . d d d d d d d d d 0 0 e +/- d d d
                    // |<- fraction10_size ->|
                    //    |<-     precision   ->|

                    integer_num = 1;
                    is_point = true;
                    decimal_num = fraction10_size - 1;
                    alignment_precision_num = (u32)(precision  - decimal_num);
                    is_point = (precision > 0) || ((m_Flags & cFlag_SharpFlag) != 0);
                    is_exponent = true;
                }
                break;

            case 'g':
                {
                    if ((m_Flags & cFlag_SharpFlag) == 0)
                    {
                        // fraction10の末尾の0は有効数字とみなさないのでその分を削る
                        while((fraction10_size > 0) && (fraction10[fraction10_size - 1] == 0))
                        {
                            fraction10_size--;
                        }
                    }

                    if (exponent10 >= -3 && exponent10 <= (s32)precision)
                    {
                        // 'f'形式

                        if (exponent10 >= (s32)fraction10_size)
                        {
                            integer_num = fraction10_size;
                            alignment_integer_num = (u32)(exponent10 - integer_num);
                            if (exponent10 >= (s32)precision)
                            {
                                //  d d d d d d d d d d d 0 0 0 0 [.]
                                // |<- fraction10_size ->|
                                // |<-      precision      ->|
                                // |<-        exponent10       ->|

                                is_point = ((m_Flags & cFlag_SharpFlag) != 0);
                            }
                            else
                            {
                                //  d d d d d d d d d d d 0 0 0 [. 0 0]
                                // |<- fraction10_size ->|
                                // |<-          precision            ->|
                                // |<-       exponent10      ->|

                                if (m_Flags & cFlag_SharpFlag)
                                {
                                    is_point = true;
                                    alignment_precision_num = (u32)(precision - exponent10);
                                }
                            }
                        }
                        else if (exponent10 > 0)
                        {
                            //  d d d d d d d d d . d d [0 0]
                            // |<-  fraction10_size  ->|
                            // |<-        precision       ->|
                            // |<- exponent10  ->|

                            integer_num = (u32)exponent10;
                            is_point = true;
                            decimal_num = (u32)(fraction10_size - exponent10);
                            if (m_Flags & cFlag_SharpFlag)
                            {
                                alignment_precision_num = precision - integer_num - decimal_num;
                            }
                        }
                        else
                        {
                            // 0 . 0 0 0 0 0 0 0 0 0 d d d d d d d d d d d [0 0]
                            //                      |<- fraction10_size ->|
                            //                      |<-       precision      ->|
                            //    |<- -exponent10 ->|

                            alignment_integer_num = 1;
                            is_point = true;
                            alignment_decimal_num = (u32)(-exponent10);
                            decimal_num = fraction10_size;
                            if (m_Flags & cFlag_SharpFlag)
                            {
                                alignment_precision_num = (u32)(precision - fraction10_size);
                            }
                        }
                    }
                    else
                    {
                        // 'e'形式
                        is_exponent = true;

                        // d . d d d d d d d d d 0 0 e +/- d d d
                        // |<- fraction10_size ->|
                        // |<-      precision      ->|

                        integer_num = 1;
                        is_point = (precision > 1) || ((m_Flags & cFlag_SharpFlag) != 0);
                        decimal_num = fraction10_size - 1;
                        if (m_Flags & cFlag_SharpFlag)
                        {
                            alignment_precision_num = (u32)(precision - fraction10_size);
                        }
                    }
                }
                break;
            }

            letter_num += integer_num;
            letter_num += alignment_integer_num;
            if (is_point)
                letter_num += 1;
            letter_num += alignment_decimal_num;
            letter_num += decimal_num;
            letter_num += alignment_precision_num;
            if (is_exponent)
                letter_num += 5;

            // widthに満たない部分
            if (letter_num < m_Width)
            {
                if (m_Flags & cFlag_JustifyLeft)
                {
                    pad_right_space_num += (m_Width - letter_num);
                }
                else if (m_Flags & cFlag_PadZero)
                {
                    pad_zero_num += (m_Width - letter_num);
                }
                else
                {
                    pad_left_space_num += (m_Width - letter_num);
                }
            }
        }


        // 実際に文字を出力
        {
            // 左のスペース
            buffer->fill(' ', pad_left_space_num);

            // 符号
            if (is_minus)
            {
                buffer->append('-');
            }
            else if (m_Flags & cFlag_PrintSign)
            {
                buffer->append('+');
            }
            else if (m_Flags & cFlag_PrintSpace)
            {
                buffer->append(' ');
            }

            // ゼロのパディング
            buffer->fill('0', pad_zero_num);

            // 整数部
            for (u32 i=0; i<integer_num; i++)
            {
                buffer->append('0' + fraction10[i]);
            }

            // 整数部桁合わせ
            buffer->fill('0', alignment_integer_num);

            // 小数点
            if (is_point)
            {
                buffer->append('.');
            }

            // 小数部桁合わせ
            buffer->fill('0', alignment_decimal_num);

            // 小数部
            for (u32 i=0; i<decimal_num; i++)
            {
                buffer->append('0' + fraction10[integer_num + i]);
            }

            // 精度合わせ
            buffer->fill('0', alignment_precision_num);

            // 指数部
            if (is_exponent)
            {
                buffer->append(is_capital ? 'E' : 'e');

                // exponent10は表示する指数より1大きい
                bool is_exp_plus = (exponent10 >= 1);
                u32 exp = (u32)(is_exp_plus ? (exponent10 - 1) : (1 - exponent10));
                u32 exp_figures[3];
                for (int i=0; i<3; i++)
                {
                    u32 r = exp / 10;
                    exp_figures[i] = exp - r * 10;
                    exp = r;
                }

                // 必ず符号付き3桁出力
                buffer->append(is_exp_plus ? '+' : '-');
                for (int i=2; i>=0; i--)
                {
                    buffer->append('0' + exp_figures[i]);
                }
            }

            // 右のスペース
            buffer->fill(' ', pad_right_space_num);
        }
    }

}

//------------------------------------------------------------------------------

/// 2進数を10進数に変換するテーブル
const FormatParser16::DigitData FormatParser16::cDigitTable[cPlaceMax - cPlaceMin + 1] =
{
    {    -323,    4,    9,    4,    0,    6,    5,    6,    4,    5,    8,    4,    1,    2,    4,    6,    5,    4,    4,    1,    8    },    // 2^-1074
    {    -323,    9,    8,    8,    1,    3,    1,    2,    9,    1,    6,    8,    2,    4,    9,    3,    0,    8,    8,    3,    5    },    // 2^-1073
    {    -322,    1,    9,    7,    6,    2,    6,    2,    5,    8,    3,    3,    6,    4,    9,    8,    6,    1,    7,    6,    7    },    // 2^-1072
    {    -322,    3,    9,    5,    2,    5,    2,    5,    1,    6,    6,    7,    2,    9,    9,    7,    2,    3,    5,    3,    4    },    // 2^-1071
    {    -322,    7,    9,    0,    5,    0,    5,    0,    3,    3,    3,    4,    5,    9,    9,    4,    4,    7,    0,    6,    8    },    // 2^-1070
    {    -321,    1,    5,    8,    1,    0,    1,    0,    0,    6,    6,    6,    9,    1,    9,    8,    8,    9,    4,    1,    4    },    // 2^-1069
    {    -321,    3,    1,    6,    2,    0,    2,    0,    1,    3,    3,    3,    8,    3,    9,    7,    7,    8,    8,    2,    7    },    // 2^-1068
    {    -321,    6,    3,    2,    4,    0,    4,    0,    2,    6,    6,    7,    6,    7,    9,    5,    5,    7,    6,    5,    5    },    // 2^-1067
    {    -320,    1,    2,    6,    4,    8,    0,    8,    0,    5,    3,    3,    5,    3,    5,    9,    1,    1,    5,    3,    1    },    // 2^-1066
    {    -320,    2,    5,    2,    9,    6,    1,    6,    1,    0,    6,    7,    0,    7,    1,    8,    2,    3,    0,    6,    2    },    // 2^-1065
    {    -320,    5,    0,    5,    9,    2,    3,    2,    2,    1,    3,    4,    1,    4,    3,    6,    4,    6,    1,    2,    4    },    // 2^-1064
    {    -319,    1,    0,    1,    1,    8,    4,    6,    4,    4,    2,    6,    8,    2,    8,    7,    2,    9,    2,    2,    5    },    // 2^-1063
    {    -319,    2,    0,    2,    3,    6,    9,    2,    8,    8,    5,    3,    6,    5,    7,    4,    5,    8,    4,    4,    9    },    // 2^-1062
    {    -319,    4,    0,    4,    7,    3,    8,    5,    7,    7,    0,    7,    3,    1,    4,    9,    1,    6,    8,    9,    9    },    // 2^-1061
    {    -319,    8,    0,    9,    4,    7,    7,    1,    5,    4,    1,    4,    6,    2,    9,    8,    3,    3,    7,    9,    8    },    // 2^-1060
    {    -318,    1,    6,    1,    8,    9,    5,    4,    3,    0,    8,    2,    9,    2,    5,    9,    6,    6,    7,    6,    0    },    // 2^-1059
    {    -318,    3,    2,    3,    7,    9,    0,    8,    6,    1,    6,    5,    8,    5,    1,    9,    3,    3,    5,    1,    9    },    // 2^-1058
    {    -318,    6,    4,    7,    5,    8,    1,    7,    2,    3,    3,    1,    7,    0,    3,    8,    6,    7,    0,    3,    8    },    // 2^-1057
    {    -317,    1,    2,    9,    5,    1,    6,    3,    4,    4,    6,    6,    3,    4,    0,    7,    7,    3,    4,    0,    8    },    // 2^-1056
    {    -317,    2,    5,    9,    0,    3,    2,    6,    8,    9,    3,    2,    6,    8,    1,    5,    4,    6,    8,    1,    5    },    // 2^-1055
    {    -317,    5,    1,    8,    0,    6,    5,    3,    7,    8,    6,    5,    3,    6,    3,    0,    9,    3,    6,    3,    1    },    // 2^-1054
    {    -316,    1,    0,    3,    6,    1,    3,    0,    7,    5,    7,    3,    0,    7,    2,    6,    1,    8,    7,    2,    6    },    // 2^-1053
    {    -316,    2,    0,    7,    2,    2,    6,    1,    5,    1,    4,    6,    1,    4,    5,    2,    3,    7,    4,    5,    2    },    // 2^-1052
    {    -316,    4,    1,    4,    4,    5,    2,    3,    0,    2,    9,    2,    2,    9,    0,    4,    7,    4,    9,    0,    5    },    // 2^-1051
    {    -316,    8,    2,    8,    9,    0,    4,    6,    0,    5,    8,    4,    5,    8,    0,    9,    4,    9,    8,    0,    9    },    // 2^-1050
    {    -315,    1,    6,    5,    7,    8,    0,    9,    2,    1,    1,    6,    9,    1,    6,    1,    8,    9,    9,    6,    2    },    // 2^-1049
    {    -315,    3,    3,    1,    5,    6,    1,    8,    4,    2,    3,    3,    8,    3,    2,    3,    7,    9,    9,    2,    4    },    // 2^-1048
    {    -315,    6,    6,    3,    1,    2,    3,    6,    8,    4,    6,    7,    6,    6,    4,    7,    5,    9,    8,    4,    7    },    // 2^-1047
    {    -314,    1,    3,    2,    6,    2,    4,    7,    3,    6,    9,    3,    5,    3,    2,    9,    5,    1,    9,    6,    9    },    // 2^-1046
    {    -314,    2,    6,    5,    2,    4,    9,    4,    7,    3,    8,    7,    0,    6,    5,    9,    0,    3,    9,    3,    9    },    // 2^-1045
    {    -314,    5,    3,    0,    4,    9,    8,    9,    4,    7,    7,    4,    1,    3,    1,    8,    0,    7,    8,    7,    8    },    // 2^-1044
    {    -313,    1,    0,    6,    0,    9,    9,    7,    8,    9,    5,    4,    8,    2,    6,    3,    6,    1,    5,    7,    6    },    // 2^-1043
    {    -313,    2,    1,    2,    1,    9,    9,    5,    7,    9,    0,    9,    6,    5,    2,    7,    2,    3,    1,    5,    1    },    // 2^-1042
    {    -313,    4,    2,    4,    3,    9,    9,    1,    5,    8,    1,    9,    3,    0,    5,    4,    4,    6,    3,    0,    2    },    // 2^-1041
    {    -313,    8,    4,    8,    7,    9,    8,    3,    1,    6,    3,    8,    6,    1,    0,    8,    9,    2,    6,    0,    4    },    // 2^-1040
    {    -312,    1,    6,    9,    7,    5,    9,    6,    6,    3,    2,    7,    7,    2,    2,    1,    7,    8,    5,    2,    1    },    // 2^-1039
    {    -312,    3,    3,    9,    5,    1,    9,    3,    2,    6,    5,    5,    4,    4,    4,    3,    5,    7,    0,    4,    2    },    // 2^-1038
    {    -312,    6,    7,    9,    0,    3,    8,    6,    5,    3,    1,    0,    8,    8,    8,    7,    1,    4,    0,    8,    4    },    // 2^-1037
    {    -311,    1,    3,    5,    8,    0,    7,    7,    3,    0,    6,    2,    1,    7,    7,    7,    4,    2,    8,    1,    7    },    // 2^-1036
    {    -311,    2,    7,    1,    6,    1,    5,    4,    6,    1,    2,    4,    3,    5,    5,    4,    8,    5,    6,    3,    3    },    // 2^-1035
    {    -311,    5,    4,    3,    2,    3,    0,    9,    2,    2,    4,    8,    7,    1,    0,    9,    7,    1,    2,    6,    7    },    // 2^-1034
    {    -310,    1,    0,    8,    6,    4,    6,    1,    8,    4,    4,    9,    7,    4,    2,    1,    9,    4,    2,    5,    3    },    // 2^-1033
    {    -310,    2,    1,    7,    2,    9,    2,    3,    6,    8,    9,    9,    4,    8,    4,    3,    8,    8,    5,    0,    7    },    // 2^-1032
    {    -310,    4,    3,    4,    5,    8,    4,    7,    3,    7,    9,    8,    9,    6,    8,    7,    7,    7,    0,    1,    3    },    // 2^-1031
    {    -310,    8,    6,    9,    1,    6,    9,    4,    7,    5,    9,    7,    9,    3,    7,    5,    5,    4,    0,    2,    7    },    // 2^-1030
    {    -309,    1,    7,    3,    8,    3,    3,    8,    9,    5,    1,    9,    5,    8,    7,    5,    1,    0,    8,    0,    5    },    // 2^-1029
    {    -309,    3,    4,    7,    6,    6,    7,    7,    9,    0,    3,    9,    1,    7,    5,    0,    2,    1,    6,    1,    1    },    // 2^-1028
    {    -309,    6,    9,    5,    3,    3,    5,    5,    8,    0,    7,    8,    3,    5,    0,    0,    4,    3,    2,    2,    2    },    // 2^-1027
    {    -308,    1,    3,    9,    0,    6,    7,    1,    1,    6,    1,    5,    6,    7,    0,    0,    0,    8,    6,    4,    4    },    // 2^-1026
    {    -308,    2,    7,    8,    1,    3,    4,    2,    3,    2,    3,    1,    3,    4,    0,    0,    1,    7,    2,    8,    9    },    // 2^-1025
    {    -308,    5,    5,    6,    2,    6,    8,    4,    6,    4,    6,    2,    6,    8,    0,    0,    3,    4,    5,    7,    7    },    // 2^-1024
    {    -307,    1,    1,    1,    2,    5,    3,    6,    9,    2,    9,    2,    5,    3,    6,    0,    0,    6,    9,    1,    5    },    // 2^-1023
    {    -307,    2,    2,    2,    5,    0,    7,    3,    8,    5,    8,    5,    0,    7,    2,    0,    1,    3,    8,    3,    1    },    // 2^-1022
    {    -307,    4,    4,    5,    0,    1,    4,    7,    7,    1,    7,    0,    1,    4,    4,    0,    2,    7,    6,    6,    2    },    // 2^-1021
    {    -307,    8,    9,    0,    0,    2,    9,    5,    4,    3,    4,    0,    2,    8,    8,    0,    5,    5,    3,    2,    4    },    // 2^-1020
    {    -306,    1,    7,    8,    0,    0,    5,    9,    0,    8,    6,    8,    0,    5,    7,    6,    1,    1,    0,    6,    5    },    // 2^-1019
    {    -306,    3,    5,    6,    0,    1,    1,    8,    1,    7,    3,    6,    1,    1,    5,    2,    2,    2,    1,    2,    9    },    // 2^-1018
    {    -306,    7,    1,    2,    0,    2,    3,    6,    3,    4,    7,    2,    2,    3,    0,    4,    4,    4,    2,    5,    9    },    // 2^-1017
    {    -305,    1,    4,    2,    4,    0,    4,    7,    2,    6,    9,    4,    4,    4,    6,    0,    8,    8,    8,    5,    2    },    // 2^-1016
    {    -305,    2,    8,    4,    8,    0,    9,    4,    5,    3,    8,    8,    8,    9,    2,    1,    7,    7,    7,    0,    4    },    // 2^-1015
    {    -305,    5,    6,    9,    6,    1,    8,    9,    0,    7,    7,    7,    7,    8,    4,    3,    5,    5,    4,    0,    7    },    // 2^-1014
    {    -304,    1,    1,    3,    9,    2,    3,    7,    8,    1,    5,    5,    5,    5,    6,    8,    7,    1,    0,    8,    1    },    // 2^-1013
    {    -304,    2,    2,    7,    8,    4,    7,    5,    6,    3,    1,    1,    1,    1,    3,    7,    4,    2,    1,    6,    3    },    // 2^-1012
    {    -304,    4,    5,    5,    6,    9,    5,    1,    2,    6,    2,    2,    2,    2,    7,    4,    8,    4,    3,    2,    6    },    // 2^-1011
    {    -304,    9,    1,    1,    3,    9,    0,    2,    5,    2,    4,    4,    4,    5,    4,    9,    6,    8,    6,    5,    1    },    // 2^-1010
    {    -303,    1,    8,    2,    2,    7,    8,    0,    5,    0,    4,    8,    8,    9,    0,    9,    9,    3,    7,    3,    0    },    // 2^-1009
    {    -303,    3,    6,    4,    5,    5,    6,    1,    0,    0,    9,    7,    7,    8,    1,    9,    8,    7,    4,    6,    1    },    // 2^-1008
    {    -303,    7,    2,    9,    1,    1,    2,    2,    0,    1,    9,    5,    5,    6,    3,    9,    7,    4,    9,    2,    1    },    // 2^-1007
    {    -302,    1,    4,    5,    8,    2,    2,    4,    4,    0,    3,    9,    1,    1,    2,    7,    9,    4,    9,    8,    4    },    // 2^-1006
    {    -302,    2,    9,    1,    6,    4,    4,    8,    8,    0,    7,    8,    2,    2,    5,    5,    8,    9,    9,    6,    8    },    // 2^-1005
    {    -302,    5,    8,    3,    2,    8,    9,    7,    6,    1,    5,    6,    4,    5,    1,    1,    7,    9,    9,    3,    7    },    // 2^-1004
    {    -301,    1,    1,    6,    6,    5,    7,    9,    5,    2,    3,    1,    2,    9,    0,    2,    3,    5,    9,    8,    7    },    // 2^-1003
    {    -301,    2,    3,    3,    3,    1,    5,    9,    0,    4,    6,    2,    5,    8,    0,    4,    7,    1,    9,    7,    5    },    // 2^-1002
    {    -301,    4,    6,    6,    6,    3,    1,    8,    0,    9,    2,    5,    1,    6,    0,    9,    4,    3,    9,    5,    0    },    // 2^-1001
    {    -301,    9,    3,    3,    2,    6,    3,    6,    1,    8,    5,    0,    3,    2,    1,    8,    8,    7,    8,    9,    9    },    // 2^-1000
    {    -300,    1,    8,    6,    6,    5,    2,    7,    2,    3,    7,    0,    0,    6,    4,    3,    7,    7,    5,    8,    0    },    // 2^-999
    {    -300,    3,    7,    3,    3,    0,    5,    4,    4,    7,    4,    0,    1,    2,    8,    7,    5,    5,    1,    6,    0    },    // 2^-998
    {    -300,    7,    4,    6,    6,    1,    0,    8,    9,    4,    8,    0,    2,    5,    7,    5,    1,    0,    3,    1,    9    },    // 2^-997
    {    -299,    1,    4,    9,    3,    2,    2,    1,    7,    8,    9,    6,    0,    5,    1,    5,    0,    2,    0,    6,    4    },    // 2^-996
    {    -299,    2,    9,    8,    6,    4,    4,    3,    5,    7,    9,    2,    1,    0,    3,    0,    0,    4,    1,    2,    8    },    // 2^-995
    {    -299,    5,    9,    7,    2,    8,    8,    7,    1,    5,    8,    4,    2,    0,    6,    0,    0,    8,    2,    5,    5    },    // 2^-994
    {    -298,    1,    1,    9,    4,    5,    7,    7,    4,    3,    1,    6,    8,    4,    1,    2,    0,    1,    6,    5,    1    },    // 2^-993
    {    -298,    2,    3,    8,    9,    1,    5,    4,    8,    6,    3,    3,    6,    8,    2,    4,    0,    3,    3,    0,    2    },    // 2^-992
    {    -298,    4,    7,    7,    8,    3,    0,    9,    7,    2,    6,    7,    3,    6,    4,    8,    0,    6,    6,    0,    4    },    // 2^-991
    {    -298,    9,    5,    5,    6,    6,    1,    9,    4,    5,    3,    4,    7,    2,    9,    6,    1,    3,    2,    0,    9    },    // 2^-990
    {    -297,    1,    9,    1,    1,    3,    2,    3,    8,    9,    0,    6,    9,    4,    5,    9,    2,    2,    6,    4,    2    },    // 2^-989
    {    -297,    3,    8,    2,    2,    6,    4,    7,    7,    8,    1,    3,    8,    9,    1,    8,    4,    5,    2,    8,    3    },    // 2^-988
    {    -297,    7,    6,    4,    5,    2,    9,    5,    5,    6,    2,    7,    7,    8,    3,    6,    9,    0,    5,    6,    7    },    // 2^-987
    {    -296,    1,    5,    2,    9,    0,    5,    9,    1,    1,    2,    5,    5,    5,    6,    7,    3,    8,    1,    1,    3    },    // 2^-986
    {    -296,    3,    0,    5,    8,    1,    1,    8,    2,    2,    5,    1,    1,    1,    3,    4,    7,    6,    2,    2,    7    },    // 2^-985
    {    -296,    6,    1,    1,    6,    2,    3,    6,    4,    5,    0,    2,    2,    2,    6,    9,    5,    2,    4,    5,    3    },    // 2^-984
    {    -295,    1,    2,    2,    3,    2,    4,    7,    2,    9,    0,    0,    4,    4,    5,    3,    9,    0,    4,    9,    1    },    // 2^-983
    {    -295,    2,    4,    4,    6,    4,    9,    4,    5,    8,    0,    0,    8,    9,    0,    7,    8,    0,    9,    8,    1    },    // 2^-982
    {    -295,    4,    8,    9,    2,    9,    8,    9,    1,    6,    0,    1,    7,    8,    1,    5,    6,    1,    9,    6,    3    },    // 2^-981
    {    -295,    9,    7,    8,    5,    9,    7,    8,    3,    2,    0,    3,    5,    6,    3,    1,    2,    3,    9,    2,    6    },    // 2^-980
    {    -294,    1,    9,    5,    7,    1,    9,    5,    6,    6,    4,    0,    7,    1,    2,    6,    2,    4,    7,    8,    5    },    // 2^-979
    {    -294,    3,    9,    1,    4,    3,    9,    1,    3,    2,    8,    1,    4,    2,    5,    2,    4,    9,    5,    7,    0    },    // 2^-978
    {    -294,    7,    8,    2,    8,    7,    8,    2,    6,    5,    6,    2,    8,    5,    0,    4,    9,    9,    1,    4,    0    },    // 2^-977
    {    -293,    1,    5,    6,    5,    7,    5,    6,    5,    3,    1,    2,    5,    7,    0,    0,    9,    9,    8,    2,    8    },    // 2^-976
    {    -293,    3,    1,    3,    1,    5,    1,    3,    0,    6,    2,    5,    1,    4,    0,    1,    9,    9,    6,    5,    6    },    // 2^-975
    {    -293,    6,    2,    6,    3,    0,    2,    6,    1,    2,    5,    0,    2,    8,    0,    3,    9,    9,    3,    1,    2    },    // 2^-974
    {    -292,    1,    2,    5,    2,    6,    0,    5,    2,    2,    5,    0,    0,    5,    6,    0,    7,    9,    8,    6,    2    },    // 2^-973
    {    -292,    2,    5,    0,    5,    2,    1,    0,    4,    5,    0,    0,    1,    1,    2,    1,    5,    9,    7,    2,    5    },    // 2^-972
    {    -292,    5,    0,    1,    0,    4,    2,    0,    9,    0,    0,    0,    2,    2,    4,    3,    1,    9,    4,    5,    0    },    // 2^-971
    {    -291,    1,    0,    0,    2,    0,    8,    4,    1,    8,    0,    0,    0,    4,    4,    8,    6,    3,    8,    9,    0    },    // 2^-970
    {    -291,    2,    0,    0,    4,    1,    6,    8,    3,    6,    0,    0,    0,    8,    9,    7,    2,    7,    7,    8,    0    },    // 2^-969
    {    -291,    4,    0,    0,    8,    3,    3,    6,    7,    2,    0,    0,    1,    7,    9,    4,    5,    5,    5,    6,    0    },    // 2^-968
    {    -291,    8,    0,    1,    6,    6,    7,    3,    4,    4,    0,    0,    3,    5,    8,    9,    1,    1,    1,    2,    0    },    // 2^-967
    {    -290,    1,    6,    0,    3,    3,    3,    4,    6,    8,    8,    0,    0,    7,    1,    7,    8,    2,    2,    2,    4    },    // 2^-966
    {    -290,    3,    2,    0,    6,    6,    6,    9,    3,    7,    6,    0,    1,    4,    3,    5,    6,    4,    4,    4,    8    },    // 2^-965
    {    -290,    6,    4,    1,    3,    3,    3,    8,    7,    5,    2,    0,    2,    8,    7,    1,    2,    8,    8,    9,    6    },    // 2^-964
    {    -289,    1,    2,    8,    2,    6,    6,    7,    7,    5,    0,    4,    0,    5,    7,    4,    2,    5,    7,    7,    9    },    // 2^-963
    {    -289,    2,    5,    6,    5,    3,    3,    5,    5,    0,    0,    8,    1,    1,    4,    8,    5,    1,    5,    5,    8    },    // 2^-962
    {    -289,    5,    1,    3,    0,    6,    7,    1,    0,    0,    1,    6,    2,    2,    9,    7,    0,    3,    1,    1,    7    },    // 2^-961
    {    -288,    1,    0,    2,    6,    1,    3,    4,    2,    0,    0,    3,    2,    4,    5,    9,    4,    0,    6,    2,    3    },    // 2^-960
    {    -288,    2,    0,    5,    2,    2,    6,    8,    4,    0,    0,    6,    4,    9,    1,    8,    8,    1,    2,    4,    7    },    // 2^-959
    {    -288,    4,    1,    0,    4,    5,    3,    6,    8,    0,    1,    2,    9,    8,    3,    7,    6,    2,    4,    9,    3    },    // 2^-958
    {    -288,    8,    2,    0,    9,    0,    7,    3,    6,    0,    2,    5,    9,    6,    7,    5,    2,    4,    9,    8,    7    },    // 2^-957
    {    -287,    1,    6,    4,    1,    8,    1,    4,    7,    2,    0,    5,    1,    9,    3,    5,    0,    4,    9,    9,    7    },    // 2^-956
    {    -287,    3,    2,    8,    3,    6,    2,    9,    4,    4,    1,    0,    3,    8,    7,    0,    0,    9,    9,    9,    5    },    // 2^-955
    {    -287,    6,    5,    6,    7,    2,    5,    8,    8,    8,    2,    0,    7,    7,    4,    0,    1,    9,    9,    8,    9    },    // 2^-954
    {    -286,    1,    3,    1,    3,    4,    5,    1,    7,    7,    6,    4,    1,    5,    4,    8,    0,    3,    9,    9,    8    },    // 2^-953
    {    -286,    2,    6,    2,    6,    9,    0,    3,    5,    5,    2,    8,    3,    0,    9,    6,    0,    7,    9,    9,    6    },    // 2^-952
    {    -286,    5,    2,    5,    3,    8,    0,    7,    1,    0,    5,    6,    6,    1,    9,    2,    1,    5,    9,    9,    2    },    // 2^-951
    {    -285,    1,    0,    5,    0,    7,    6,    1,    4,    2,    1,    1,    3,    2,    3,    8,    4,    3,    1,    9,    8    },    // 2^-950
    {    -285,    2,    1,    0,    1,    5,    2,    2,    8,    4,    2,    2,    6,    4,    7,    6,    8,    6,    3,    9,    7    },    // 2^-949
    {    -285,    4,    2,    0,    3,    0,    4,    5,    6,    8,    4,    5,    2,    9,    5,    3,    7,    2,    7,    9,    3    },    // 2^-948
    {    -285,    8,    4,    0,    6,    0,    9,    1,    3,    6,    9,    0,    5,    9,    0,    7,    4,    5,    5,    8,    6    },    // 2^-947
    {    -284,    1,    6,    8,    1,    2,    1,    8,    2,    7,    3,    8,    1,    1,    8,    1,    4,    9,    1,    1,    7    },    // 2^-946
    {    -284,    3,    3,    6,    2,    4,    3,    6,    5,    4,    7,    6,    2,    3,    6,    2,    9,    8,    2,    3,    5    },    // 2^-945
    {    -284,    6,    7,    2,    4,    8,    7,    3,    0,    9,    5,    2,    4,    7,    2,    5,    9,    6,    4,    6,    9    },    // 2^-944
    {    -283,    1,    3,    4,    4,    9,    7,    4,    6,    1,    9,    0,    4,    9,    4,    5,    1,    9,    2,    9,    4    },    // 2^-943
    {    -283,    2,    6,    8,    9,    9,    4,    9,    2,    3,    8,    0,    9,    8,    9,    0,    3,    8,    5,    8,    8    },    // 2^-942
    {    -283,    5,    3,    7,    9,    8,    9,    8,    4,    7,    6,    1,    9,    7,    8,    0,    7,    7,    1,    7,    5    },    // 2^-941
    {    -282,    1,    0,    7,    5,    9,    7,    9,    6,    9,    5,    2,    3,    9,    5,    6,    1,    5,    4,    3,    5    },    // 2^-940
    {    -282,    2,    1,    5,    1,    9,    5,    9,    3,    9,    0,    4,    7,    9,    1,    2,    3,    0,    8,    7,    0    },    // 2^-939
    {    -282,    4,    3,    0,    3,    9,    1,    8,    7,    8,    0,    9,    5,    8,    2,    4,    6,    1,    7,    4,    0    },    // 2^-938
    {    -282,    8,    6,    0,    7,    8,    3,    7,    5,    6,    1,    9,    1,    6,    4,    9,    2,    3,    4,    8,    0    },    // 2^-937
    {    -281,    1,    7,    2,    1,    5,    6,    7,    5,    1,    2,    3,    8,    3,    2,    9,    8,    4,    6,    9,    6    },    // 2^-936
    {    -281,    3,    4,    4,    3,    1,    3,    5,    0,    2,    4,    7,    6,    6,    5,    9,    6,    9,    3,    9,    2    },    // 2^-935
    {    -281,    6,    8,    8,    6,    2,    7,    0,    0,    4,    9,    5,    3,    3,    1,    9,    3,    8,    7,    8,    4    },    // 2^-934
    {    -280,    1,    3,    7,    7,    2,    5,    4,    0,    0,    9,    9,    0,    6,    6,    3,    8,    7,    7,    5,    7    },    // 2^-933
    {    -280,    2,    7,    5,    4,    5,    0,    8,    0,    1,    9,    8,    1,    3,    2,    7,    7,    5,    5,    1,    4    },    // 2^-932
    {    -280,    5,    5,    0,    9,    0,    1,    6,    0,    3,    9,    6,    2,    6,    5,    5,    5,    1,    0,    2,    8    },    // 2^-931
    {    -279,    1,    1,    0,    1,    8,    0,    3,    2,    0,    7,    9,    2,    5,    3,    1,    1,    0,    2,    0,    6    },    // 2^-930
    {    -279,    2,    2,    0,    3,    6,    0,    6,    4,    1,    5,    8,    5,    0,    6,    2,    2,    0,    4,    1,    1    },    // 2^-929
    {    -279,    4,    4,    0,    7,    2,    1,    2,    8,    3,    1,    7,    0,    1,    2,    4,    4,    0,    8,    2,    2    },    // 2^-928
    {    -279,    8,    8,    1,    4,    4,    2,    5,    6,    6,    3,    4,    0,    2,    4,    8,    8,    1,    6,    4,    4    },    // 2^-927
    {    -278,    1,    7,    6,    2,    8,    8,    5,    1,    3,    2,    6,    8,    0,    4,    9,    7,    6,    3,    2,    9    },    // 2^-926
    {    -278,    3,    5,    2,    5,    7,    7,    0,    2,    6,    5,    3,    6,    0,    9,    9,    5,    2,    6,    5,    8    },    // 2^-925
    {    -278,    7,    0,    5,    1,    5,    4,    0,    5,    3,    0,    7,    2,    1,    9,    9,    0,    5,    3,    1,    5    },    // 2^-924
    {    -277,    1,    4,    1,    0,    3,    0,    8,    1,    0,    6,    1,    4,    4,    3,    9,    8,    1,    0,    6,    3    },    // 2^-923
    {    -277,    2,    8,    2,    0,    6,    1,    6,    2,    1,    2,    2,    8,    8,    7,    9,    6,    2,    1,    2,    6    },    // 2^-922
    {    -277,    5,    6,    4,    1,    2,    3,    2,    4,    2,    4,    5,    7,    7,    5,    9,    2,    4,    2,    5,    2    },    // 2^-921
    {    -276,    1,    1,    2,    8,    2,    4,    6,    4,    8,    4,    9,    1,    5,    5,    1,    8,    4,    8,    5,    0    },    // 2^-920
    {    -276,    2,    2,    5,    6,    4,    9,    2,    9,    6,    9,    8,    3,    1,    0,    3,    6,    9,    7,    0,    1    },    // 2^-919
    {    -276,    4,    5,    1,    2,    9,    8,    5,    9,    3,    9,    6,    6,    2,    0,    7,    3,    9,    4,    0,    2    },    // 2^-918
    {    -276,    9,    0,    2,    5,    9,    7,    1,    8,    7,    9,    3,    2,    4,    1,    4,    7,    8,    8,    0,    3    },    // 2^-917
    {    -275,    1,    8,    0,    5,    1,    9,    4,    3,    7,    5,    8,    6,    4,    8,    2,    9,    5,    7,    6,    1    },    // 2^-916
    {    -275,    3,    6,    1,    0,    3,    8,    8,    7,    5,    1,    7,    2,    9,    6,    5,    9,    1,    5,    2,    1    },    // 2^-915
    {    -275,    7,    2,    2,    0,    7,    7,    7,    5,    0,    3,    4,    5,    9,    3,    1,    8,    3,    0,    4,    3    },    // 2^-914
    {    -274,    1,    4,    4,    4,    1,    5,    5,    5,    0,    0,    6,    9,    1,    8,    6,    3,    6,    6,    0,    9    },    // 2^-913
    {    -274,    2,    8,    8,    8,    3,    1,    1,    0,    0,    1,    3,    8,    3,    7,    2,    7,    3,    2,    1,    7    },    // 2^-912
    {    -274,    5,    7,    7,    6,    6,    2,    2,    0,    0,    2,    7,    6,    7,    4,    5,    4,    6,    4,    3,    4    },    // 2^-911
    {    -273,    1,    1,    5,    5,    3,    2,    4,    4,    0,    0,    5,    5,    3,    4,    9,    0,    9,    2,    8,    7    },    // 2^-910
    {    -273,    2,    3,    1,    0,    6,    4,    8,    8,    0,    1,    1,    0,    6,    9,    8,    1,    8,    5,    7,    4    },    // 2^-909
    {    -273,    4,    6,    2,    1,    2,    9,    7,    6,    0,    2,    2,    1,    3,    9,    6,    3,    7,    1,    4,    7    },    // 2^-908
    {    -273,    9,    2,    4,    2,    5,    9,    5,    2,    0,    4,    4,    2,    7,    9,    2,    7,    4,    2,    9,    5    },    // 2^-907
    {    -272,    1,    8,    4,    8,    5,    1,    9,    0,    4,    0,    8,    8,    5,    5,    8,    5,    4,    8,    5,    9    },    // 2^-906
    {    -272,    3,    6,    9,    7,    0,    3,    8,    0,    8,    1,    7,    7,    1,    1,    7,    0,    9,    7,    1,    8    },    // 2^-905
    {    -272,    7,    3,    9,    4,    0,    7,    6,    1,    6,    3,    5,    4,    2,    3,    4,    1,    9,    4,    3,    6    },    // 2^-904
    {    -271,    1,    4,    7,    8,    8,    1,    5,    2,    3,    2,    7,    0,    8,    4,    6,    8,    3,    8,    8,    7    },    // 2^-903
    {    -271,    2,    9,    5,    7,    6,    3,    0,    4,    6,    5,    4,    1,    6,    9,    3,    6,    7,    7,    7,    4    },    // 2^-902
    {    -271,    5,    9,    1,    5,    2,    6,    0,    9,    3,    0,    8,    3,    3,    8,    7,    3,    5,    5,    4,    9    },    // 2^-901
    {    -270,    1,    1,    8,    3,    0,    5,    2,    1,    8,    6,    1,    6,    6,    7,    7,    4,    7,    1,    1,    0    },    // 2^-900
    {    -270,    2,    3,    6,    6,    1,    0,    4,    3,    7,    2,    3,    3,    3,    5,    4,    9,    4,    2,    1,    9    },    // 2^-899
    {    -270,    4,    7,    3,    2,    2,    0,    8,    7,    4,    4,    6,    6,    7,    0,    9,    8,    8,    4,    3,    9    },    // 2^-898
    {    -270,    9,    4,    6,    4,    4,    1,    7,    4,    8,    9,    3,    3,    4,    1,    9,    7,    6,    8,    7,    8    },    // 2^-897
    {    -269,    1,    8,    9,    2,    8,    8,    3,    4,    9,    7,    8,    6,    6,    8,    3,    9,    5,    3,    7,    6    },    // 2^-896
    {    -269,    3,    7,    8,    5,    7,    6,    6,    9,    9,    5,    7,    3,    3,    6,    7,    9,    0,    7,    5,    1    },    // 2^-895
    {    -269,    7,    5,    7,    1,    5,    3,    3,    9,    9,    1,    4,    6,    7,    3,    5,    8,    1,    5,    0,    2    },    // 2^-894
    {    -268,    1,    5,    1,    4,    3,    0,    6,    7,    9,    8,    2,    9,    3,    4,    7,    1,    6,    3,    0,    0    },    // 2^-893
    {    -268,    3,    0,    2,    8,    6,    1,    3,    5,    9,    6,    5,    8,    6,    9,    4,    3,    2,    6,    0,    1    },    // 2^-892
    {    -268,    6,    0,    5,    7,    2,    2,    7,    1,    9,    3,    1,    7,    3,    8,    8,    6,    5,    2,    0,    2    },    // 2^-891
    {    -267,    1,    2,    1,    1,    4,    4,    5,    4,    3,    8,    6,    3,    4,    7,    7,    7,    3,    0,    4,    0    },    // 2^-890
    {    -267,    2,    4,    2,    2,    8,    9,    0,    8,    7,    7,    2,    6,    9,    5,    5,    4,    6,    0,    8,    1    },    // 2^-889
    {    -267,    4,    8,    4,    5,    7,    8,    1,    7,    5,    4,    5,    3,    9,    1,    0,    9,    2,    1,    6,    1    },    // 2^-888
    {    -267,    9,    6,    9,    1,    5,    6,    3,    5,    0,    9,    0,    7,    8,    2,    1,    8,    4,    3,    2,    3    },    // 2^-887
    {    -266,    1,    9,    3,    8,    3,    1,    2,    7,    0,    1,    8,    1,    5,    6,    4,    3,    6,    8,    6,    5    },    // 2^-886
    {    -266,    3,    8,    7,    6,    6,    2,    5,    4,    0,    3,    6,    3,    1,    2,    8,    7,    3,    7,    2,    9    },    // 2^-885
    {    -266,    7,    7,    5,    3,    2,    5,    0,    8,    0,    7,    2,    6,    2,    5,    7,    4,    7,    4,    5,    8    },    // 2^-884
    {    -265,    1,    5,    5,    0,    6,    5,    0,    1,    6,    1,    4,    5,    2,    5,    1,    4,    9,    4,    9,    2    },    // 2^-883
    {    -265,    3,    1,    0,    1,    3,    0,    0,    3,    2,    2,    9,    0,    5,    0,    2,    9,    8,    9,    8,    3    },    // 2^-882
    {    -265,    6,    2,    0,    2,    6,    0,    0,    6,    4,    5,    8,    1,    0,    0,    5,    9,    7,    9,    6,    7    },    // 2^-881
    {    -264,    1,    2,    4,    0,    5,    2,    0,    1,    2,    9,    1,    6,    2,    0,    1,    1,    9,    5,    9,    3    },    // 2^-880
    {    -264,    2,    4,    8,    1,    0,    4,    0,    2,    5,    8,    3,    2,    4,    0,    2,    3,    9,    1,    8,    7    },    // 2^-879
    {    -264,    4,    9,    6,    2,    0,    8,    0,    5,    1,    6,    6,    4,    8,    0,    4,    7,    8,    3,    7,    3    },    // 2^-878
    {    -264,    9,    9,    2,    4,    1,    6,    1,    0,    3,    3,    2,    9,    6,    0,    9,    5,    6,    7,    4,    7    },    // 2^-877
    {    -263,    1,    9,    8,    4,    8,    3,    2,    2,    0,    6,    6,    5,    9,    2,    1,    9,    1,    3,    4,    9    },    // 2^-876
    {    -263,    3,    9,    6,    9,    6,    6,    4,    4,    1,    3,    3,    1,    8,    4,    3,    8,    2,    6,    9,    9    },    // 2^-875
    {    -263,    7,    9,    3,    9,    3,    2,    8,    8,    2,    6,    6,    3,    6,    8,    7,    6,    5,    3,    9,    7    },    // 2^-874
    {    -262,    1,    5,    8,    7,    8,    6,    5,    7,    6,    5,    3,    2,    7,    3,    7,    5,    3,    0,    7,    9    },    // 2^-873
    {    -262,    3,    1,    7,    5,    7,    3,    1,    5,    3,    0,    6,    5,    4,    7,    5,    0,    6,    1,    5,    9    },    // 2^-872
    {    -262,    6,    3,    5,    1,    4,    6,    3,    0,    6,    1,    3,    0,    9,    5,    0,    1,    2,    3,    1,    8    },    // 2^-871
    {    -261,    1,    2,    7,    0,    2,    9,    2,    6,    1,    2,    2,    6,    1,    9,    0,    0,    2,    4,    6,    4    },    // 2^-870
    {    -261,    2,    5,    4,    0,    5,    8,    5,    2,    2,    4,    5,    2,    3,    8,    0,    0,    4,    9,    2,    7    },    // 2^-869
    {    -261,    5,    0,    8,    1,    1,    7,    0,    4,    4,    9,    0,    4,    7,    6,    0,    0,    9,    8,    5,    4    },    // 2^-868
    {    -260,    1,    0,    1,    6,    2,    3,    4,    0,    8,    9,    8,    0,    9,    5,    2,    0,    1,    9,    7,    1    },    // 2^-867
    {    -260,    2,    0,    3,    2,    4,    6,    8,    1,    7,    9,    6,    1,    9,    0,    4,    0,    3,    9,    4,    2    },    // 2^-866
    {    -260,    4,    0,    6,    4,    9,    3,    6,    3,    5,    9,    2,    3,    8,    0,    8,    0,    7,    8,    8,    3    },    // 2^-865
    {    -260,    8,    1,    2,    9,    8,    7,    2,    7,    1,    8,    4,    7,    6,    1,    6,    1,    5,    7,    6,    7    },    // 2^-864
    {    -259,    1,    6,    2,    5,    9,    7,    4,    5,    4,    3,    6,    9,    5,    2,    3,    2,    3,    1,    5,    3    },    // 2^-863
    {    -259,    3,    2,    5,    1,    9,    4,    9,    0,    8,    7,    3,    9,    0,    4,    6,    4,    6,    3,    0,    7    },    // 2^-862
    {    -259,    6,    5,    0,    3,    8,    9,    8,    1,    7,    4,    7,    8,    0,    9,    2,    9,    2,    6,    1,    3    },    // 2^-861
    {    -258,    1,    3,    0,    0,    7,    7,    9,    6,    3,    4,    9,    5,    6,    1,    8,    5,    8,    5,    2,    3    },    // 2^-860
    {    -258,    2,    6,    0,    1,    5,    5,    9,    2,    6,    9,    9,    1,    2,    3,    7,    1,    7,    0,    4,    5    },    // 2^-859
    {    -258,    5,    2,    0,    3,    1,    1,    8,    5,    3,    9,    8,    2,    4,    7,    4,    3,    4,    0,    9,    1    },    // 2^-858
    {    -257,    1,    0,    4,    0,    6,    2,    3,    7,    0,    7,    9,    6,    4,    9,    4,    8,    6,    8,    1,    8    },    // 2^-857
    {    -257,    2,    0,    8,    1,    2,    4,    7,    4,    1,    5,    9,    2,    9,    8,    9,    7,    3,    6,    3,    6    },    // 2^-856
    {    -257,    4,    1,    6,    2,    4,    9,    4,    8,    3,    1,    8,    5,    9,    7,    9,    4,    7,    2,    7,    3    },    // 2^-855
    {    -257,    8,    3,    2,    4,    9,    8,    9,    6,    6,    3,    7,    1,    9,    5,    8,    9,    4,    5,    4,    5    },    // 2^-854
    {    -256,    1,    6,    6,    4,    9,    9,    7,    9,    3,    2,    7,    4,    3,    9,    1,    7,    8,    9,    0,    9    },    // 2^-853
    {    -256,    3,    3,    2,    9,    9,    9,    5,    8,    6,    5,    4,    8,    7,    8,    3,    5,    7,    8,    1,    8    },    // 2^-852
    {    -256,    6,    6,    5,    9,    9,    9,    1,    7,    3,    0,    9,    7,    5,    6,    7,    1,    5,    6,    3,    6    },    // 2^-851
    {    -255,    1,    3,    3,    1,    9,    9,    8,    3,    4,    6,    1,    9,    5,    1,    3,    4,    3,    1,    2,    7    },    // 2^-850
    {    -255,    2,    6,    6,    3,    9,    9,    6,    6,    9,    2,    3,    9,    0,    2,    6,    8,    6,    2,    5,    4    },    // 2^-849
    {    -255,    5,    3,    2,    7,    9,    9,    3,    3,    8,    4,    7,    8,    0,    5,    3,    7,    2,    5,    0,    9    },    // 2^-848
    {    -254,    1,    0,    6,    5,    5,    9,    8,    6,    7,    6,    9,    5,    6,    1,    0,    7,    4,    5,    0,    2    },    // 2^-847
    {    -254,    2,    1,    3,    1,    1,    9,    7,    3,    5,    3,    9,    1,    2,    2,    1,    4,    9,    0,    0,    4    },    // 2^-846
    {    -254,    4,    2,    6,    2,    3,    9,    4,    7,    0,    7,    8,    2,    4,    4,    2,    9,    8,    0,    0,    7    },    // 2^-845
    {    -254,    8,    5,    2,    4,    7,    8,    9,    4,    1,    5,    6,    4,    8,    8,    5,    9,    6,    0,    1,    4    },    // 2^-844
    {    -253,    1,    7,    0,    4,    9,    5,    7,    8,    8,    3,    1,    2,    9,    7,    7,    1,    9,    2,    0,    3    },    // 2^-843
    {    -253,    3,    4,    0,    9,    9,    1,    5,    7,    6,    6,    2,    5,    9,    5,    4,    3,    8,    4,    0,    6    },    // 2^-842
    {    -253,    6,    8,    1,    9,    8,    3,    1,    5,    3,    2,    5,    1,    9,    0,    8,    7,    6,    8,    1,    1    },    // 2^-841
    {    -252,    1,    3,    6,    3,    9,    6,    6,    3,    0,    6,    5,    0,    3,    8,    1,    7,    5,    3,    6,    2    },    // 2^-840
    {    -252,    2,    7,    2,    7,    9,    3,    2,    6,    1,    3,    0,    0,    7,    6,    3,    5,    0,    7,    2,    5    },    // 2^-839
    {    -252,    5,    4,    5,    5,    8,    6,    5,    2,    2,    6,    0,    1,    5,    2,    7,    0,    1,    4,    4,    9    },    // 2^-838
    {    -251,    1,    0,    9,    1,    1,    7,    3,    0,    4,    5,    2,    0,    3,    0,    5,    4,    0,    2,    9,    0    },    // 2^-837
    {    -251,    2,    1,    8,    2,    3,    4,    6,    0,    9,    0,    4,    0,    6,    1,    0,    8,    0,    5,    8,    0    },    // 2^-836
    {    -251,    4,    3,    6,    4,    6,    9,    2,    1,    8,    0,    8,    1,    2,    2,    1,    6,    1,    1,    5,    9    },    // 2^-835
    {    -251,    8,    7,    2,    9,    3,    8,    4,    3,    6,    1,    6,    2,    4,    4,    3,    2,    2,    3,    1,    9    },    // 2^-834
    {    -250,    1,    7,    4,    5,    8,    7,    6,    8,    7,    2,    3,    2,    4,    8,    8,    6,    4,    4,    6,    4    },    // 2^-833
    {    -250,    3,    4,    9,    1,    7,    5,    3,    7,    4,    4,    6,    4,    9,    7,    7,    2,    8,    9,    2,    7    },    // 2^-832
    {    -250,    6,    9,    8,    3,    5,    0,    7,    4,    8,    9,    2,    9,    9,    5,    4,    5,    7,    8,    5,    5    },    // 2^-831
    {    -249,    1,    3,    9,    6,    7,    0,    1,    4,    9,    7,    8,    5,    9,    9,    0,    9,    1,    5,    7,    1    },    // 2^-830
    {    -249,    2,    7,    9,    3,    4,    0,    2,    9,    9,    5,    7,    1,    9,    8,    1,    8,    3,    1,    4,    2    },    // 2^-829
    {    -249,    5,    5,    8,    6,    8,    0,    5,    9,    9,    1,    4,    3,    9,    6,    3,    6,    6,    2,    8,    4    },    // 2^-828
    {    -248,    1,    1,    1,    7,    3,    6,    1,    1,    9,    8,    2,    8,    7,    9,    2,    7,    3,    2,    5,    7    },    // 2^-827
    {    -248,    2,    2,    3,    4,    7,    2,    2,    3,    9,    6,    5,    7,    5,    8,    5,    4,    6,    5,    1,    4    },    // 2^-826
    {    -248,    4,    4,    6,    9,    4,    4,    4,    7,    9,    3,    1,    5,    1,    7,    0,    9,    3,    0,    2,    7    },    // 2^-825
    {    -248,    8,    9,    3,    8,    8,    8,    9,    5,    8,    6,    3,    0,    3,    4,    1,    8,    6,    0,    5,    4    },    // 2^-824
    {    -247,    1,    7,    8,    7,    7,    7,    7,    9,    1,    7,    2,    6,    0,    6,    8,    3,    7,    2,    1,    1    },    // 2^-823
    {    -247,    3,    5,    7,    5,    5,    5,    5,    8,    3,    4,    5,    2,    1,    3,    6,    7,    4,    4,    2,    2    },    // 2^-822
    {    -247,    7,    1,    5,    1,    1,    1,    1,    6,    6,    9,    0,    4,    2,    7,    3,    4,    8,    8,    4,    3    },    // 2^-821
    {    -246,    1,    4,    3,    0,    2,    2,    2,    3,    3,    3,    8,    0,    8,    5,    4,    6,    9,    7,    6,    9    },    // 2^-820
    {    -246,    2,    8,    6,    0,    4,    4,    4,    6,    6,    7,    6,    1,    7,    0,    9,    3,    9,    5,    3,    7    },    // 2^-819
    {    -246,    5,    7,    2,    0,    8,    8,    9,    3,    3,    5,    2,    3,    4,    1,    8,    7,    9,    0,    7,    5    },    // 2^-818
    {    -245,    1,    1,    4,    4,    1,    7,    7,    8,    6,    7,    0,    4,    6,    8,    3,    7,    5,    8,    1,    5    },    // 2^-817
    {    -245,    2,    2,    8,    8,    3,    5,    5,    7,    3,    4,    0,    9,    3,    6,    7,    5,    1,    6,    3,    0    },    // 2^-816
    {    -245,    4,    5,    7,    6,    7,    1,    1,    4,    6,    8,    1,    8,    7,    3,    5,    0,    3,    2,    6,    0    },    // 2^-815
    {    -245,    9,    1,    5,    3,    4,    2,    2,    9,    3,    6,    3,    7,    4,    7,    0,    0,    6,    5,    2,    0    },    // 2^-814
    {    -244,    1,    8,    3,    0,    6,    8,    4,    5,    8,    7,    2,    7,    4,    9,    4,    0,    1,    3,    0,    4    },    // 2^-813
    {    -244,    3,    6,    6,    1,    3,    6,    9,    1,    7,    4,    5,    4,    9,    8,    8,    0,    2,    6,    0,    8    },    // 2^-812
    {    -244,    7,    3,    2,    2,    7,    3,    8,    3,    4,    9,    0,    9,    9,    7,    6,    0,    5,    2,    1,    6    },    // 2^-811
    {    -243,    1,    4,    6,    4,    5,    4,    7,    6,    6,    9,    8,    1,    9,    9,    5,    2,    1,    0,    4,    3    },    // 2^-810
    {    -243,    2,    9,    2,    9,    0,    9,    5,    3,    3,    9,    6,    3,    9,    9,    0,    4,    2,    0,    8,    6    },    // 2^-809
    {    -243,    5,    8,    5,    8,    1,    9,    0,    6,    7,    9,    2,    7,    9,    8,    0,    8,    4,    1,    7,    3    },    // 2^-808
    {    -242,    1,    1,    7,    1,    6,    3,    8,    1,    3,    5,    8,    5,    5,    9,    6,    1,    6,    8,    3,    5    },    // 2^-807
    {    -242,    2,    3,    4,    3,    2,    7,    6,    2,    7,    1,    7,    1,    1,    9,    2,    3,    3,    6,    6,    9    },    // 2^-806
    {    -242,    4,    6,    8,    6,    5,    5,    2,    5,    4,    3,    4,    2,    3,    8,    4,    6,    7,    3,    3,    8    },    // 2^-805
    {    -242,    9,    3,    7,    3,    1,    0,    5,    0,    8,    6,    8,    4,    7,    6,    9,    3,    4,    6,    7,    6    },    // 2^-804
    {    -241,    1,    8,    7,    4,    6,    2,    1,    0,    1,    7,    3,    6,    9,    5,    3,    8,    6,    9,    3,    5    },    // 2^-803
    {    -241,    3,    7,    4,    9,    2,    4,    2,    0,    3,    4,    7,    3,    9,    0,    7,    7,    3,    8,    7,    0    },    // 2^-802
    {    -241,    7,    4,    9,    8,    4,    8,    4,    0,    6,    9,    4,    7,    8,    1,    5,    4,    7,    7,    4,    1    },    // 2^-801
    {    -240,    1,    4,    9,    9,    6,    9,    6,    8,    1,    3,    8,    9,    5,    6,    3,    0,    9,    5,    4,    8    },    // 2^-800
    {    -240,    2,    9,    9,    9,    3,    9,    3,    6,    2,    7,    7,    9,    1,    2,    6,    1,    9,    0,    9,    6    },    // 2^-799
    {    -240,    5,    9,    9,    8,    7,    8,    7,    2,    5,    5,    5,    8,    2,    5,    2,    3,    8,    1,    9,    3    },    // 2^-798
    {    -239,    1,    1,    9,    9,    7,    5,    7,    4,    5,    1,    1,    1,    6,    5,    0,    4,    7,    6,    3,    9    },    // 2^-797
    {    -239,    2,    3,    9,    9,    5,    1,    4,    9,    0,    2,    2,    3,    3,    0,    0,    9,    5,    2,    7,    7    },    // 2^-796
    {    -239,    4,    7,    9,    9,    0,    2,    9,    8,    0,    4,    4,    6,    6,    0,    1,    9,    0,    5,    5,    4    },    // 2^-795
    {    -239,    9,    5,    9,    8,    0,    5,    9,    6,    0,    8,    9,    3,    2,    0,    3,    8,    1,    1,    0,    8    },    // 2^-794
    {    -238,    1,    9,    1,    9,    6,    1,    1,    9,    2,    1,    7,    8,    6,    4,    0,    7,    6,    2,    2,    2    },    // 2^-793
    {    -238,    3,    8,    3,    9,    2,    2,    3,    8,    4,    3,    5,    7,    2,    8,    1,    5,    2,    4,    4,    3    },    // 2^-792
    {    -238,    7,    6,    7,    8,    4,    4,    7,    6,    8,    7,    1,    4,    5,    6,    3,    0,    4,    8,    8,    7    },    // 2^-791
    {    -237,    1,    5,    3,    5,    6,    8,    9,    5,    3,    7,    4,    2,    9,    1,    2,    6,    0,    9,    7,    7    },    // 2^-790
    {    -237,    3,    0,    7,    1,    3,    7,    9,    0,    7,    4,    8,    5,    8,    2,    5,    2,    1,    9,    5,    5    },    // 2^-789
    {    -237,    6,    1,    4,    2,    7,    5,    8,    1,    4,    9,    7,    1,    6,    5,    0,    4,    3,    9,    0,    9    },    // 2^-788
    {    -236,    1,    2,    2,    8,    5,    5,    1,    6,    2,    9,    9,    4,    3,    3,    0,    0,    8,    7,    8,    2    },    // 2^-787
    {    -236,    2,    4,    5,    7,    1,    0,    3,    2,    5,    9,    8,    8,    6,    6,    0,    1,    7,    5,    6,    4    },    // 2^-786
    {    -236,    4,    9,    1,    4,    2,    0,    6,    5,    1,    9,    7,    7,    3,    2,    0,    3,    5,    1,    2,    7    },    // 2^-785
    {    -236,    9,    8,    2,    8,    4,    1,    3,    0,    3,    9,    5,    4,    6,    4,    0,    7,    0,    2,    5,    5    },    // 2^-784
    {    -235,    1,    9,    6,    5,    6,    8,    2,    6,    0,    7,    9,    0,    9,    2,    8,    1,    4,    0,    5,    1    },    // 2^-783
    {    -235,    3,    9,    3,    1,    3,    6,    5,    2,    1,    5,    8,    1,    8,    5,    6,    2,    8,    1,    0,    2    },    // 2^-782
    {    -235,    7,    8,    6,    2,    7,    3,    0,    4,    3,    1,    6,    3,    7,    1,    2,    5,    6,    2,    0,    4    },    // 2^-781
    {    -234,    1,    5,    7,    2,    5,    4,    6,    0,    8,    6,    3,    2,    7,    4,    2,    5,    1,    2,    4,    1    },    // 2^-780
    {    -234,    3,    1,    4,    5,    0,    9,    2,    1,    7,    2,    6,    5,    4,    8,    5,    0,    2,    4,    8,    2    },    // 2^-779
    {    -234,    6,    2,    9,    0,    1,    8,    4,    3,    4,    5,    3,    0,    9,    7,    0,    0,    4,    9,    6,    3    },    // 2^-778
    {    -233,    1,    2,    5,    8,    0,    3,    6,    8,    6,    9,    0,    6,    1,    9,    4,    0,    0,    9,    9,    3    },    // 2^-777
    {    -233,    2,    5,    1,    6,    0,    7,    3,    7,    3,    8,    1,    2,    3,    8,    8,    0,    1,    9,    8,    5    },    // 2^-776
    {    -233,    5,    0,    3,    2,    1,    4,    7,    4,    7,    6,    2,    4,    7,    7,    6,    0,    3,    9,    7,    1    },    // 2^-775
    {    -232,    1,    0,    0,    6,    4,    2,    9,    4,    9,    5,    2,    4,    9,    5,    5,    2,    0,    7,    9,    4    },    // 2^-774
    {    -232,    2,    0,    1,    2,    8,    5,    8,    9,    9,    0,    4,    9,    9,    1,    0,    4,    1,    5,    8,    8    },    // 2^-773
    {    -232,    4,    0,    2,    5,    7,    1,    7,    9,    8,    0,    9,    9,    8,    2,    0,    8,    3,    1,    7,    6    },    // 2^-772
    {    -232,    8,    0,    5,    1,    4,    3,    5,    9,    6,    1,    9,    9,    6,    4,    1,    6,    6,    3,    5,    3    },    // 2^-771
    {    -231,    1,    6,    1,    0,    2,    8,    7,    1,    9,    2,    3,    9,    9,    2,    8,    3,    3,    2,    7,    1    },    // 2^-770
    {    -231,    3,    2,    2,    0,    5,    7,    4,    3,    8,    4,    7,    9,    8,    5,    6,    6,    6,    5,    4,    1    },    // 2^-769
    {    -231,    6,    4,    4,    1,    1,    4,    8,    7,    6,    9,    5,    9,    7,    1,    3,    3,    3,    0,    8,    2    },    // 2^-768
    {    -230,    1,    2,    8,    8,    2,    2,    9,    7,    5,    3,    9,    1,    9,    4,    2,    6,    6,    6,    1,    6    },    // 2^-767
    {    -230,    2,    5,    7,    6,    4,    5,    9,    5,    0,    7,    8,    3,    8,    8,    5,    3,    3,    2,    3,    3    },    // 2^-766
    {    -230,    5,    1,    5,    2,    9,    1,    9,    0,    1,    5,    6,    7,    7,    7,    0,    6,    6,    4,    6,    6    },    // 2^-765
    {    -229,    1,    0,    3,    0,    5,    8,    3,    8,    0,    3,    1,    3,    5,    5,    4,    1,    3,    2,    9,    3    },    // 2^-764
    {    -229,    2,    0,    6,    1,    1,    6,    7,    6,    0,    6,    2,    7,    1,    0,    8,    2,    6,    5,    8,    6    },    // 2^-763
    {    -229,    4,    1,    2,    2,    3,    3,    5,    2,    1,    2,    5,    4,    2,    1,    6,    5,    3,    1,    7,    3    },    // 2^-762
    {    -229,    8,    2,    4,    4,    6,    7,    0,    4,    2,    5,    0,    8,    4,    3,    3,    0,    6,    3,    4,    5    },    // 2^-761
    {    -228,    1,    6,    4,    8,    9,    3,    4,    0,    8,    5,    0,    1,    6,    8,    6,    6,    1,    2,    6,    9    },    // 2^-760
    {    -228,    3,    2,    9,    7,    8,    6,    8,    1,    7,    0,    0,    3,    3,    7,    3,    2,    2,    5,    3,    8    },    // 2^-759
    {    -228,    6,    5,    9,    5,    7,    3,    6,    3,    4,    0,    0,    6,    7,    4,    6,    4,    5,    0,    7,    6    },    // 2^-758
    {    -227,    1,    3,    1,    9,    1,    4,    7,    2,    6,    8,    0,    1,    3,    4,    9,    2,    9,    0,    1,    5    },    // 2^-757
    {    -227,    2,    6,    3,    8,    2,    9,    4,    5,    3,    6,    0,    2,    6,    9,    8,    5,    8,    0,    3,    0    },    // 2^-756
    {    -227,    5,    2,    7,    6,    5,    8,    9,    0,    7,    2,    0,    5,    3,    9,    7,    1,    6,    0,    6,    1    },    // 2^-755
    {    -226,    1,    0,    5,    5,    3,    1,    7,    8,    1,    4,    4,    1,    0,    7,    9,    4,    3,    2,    1,    2    },    // 2^-754
    {    -226,    2,    1,    1,    0,    6,    3,    5,    6,    2,    8,    8,    2,    1,    5,    8,    8,    6,    4,    2,    4    },    // 2^-753
    {    -226,    4,    2,    2,    1,    2,    7,    1,    2,    5,    7,    6,    4,    3,    1,    7,    7,    2,    8,    4,    9    },    // 2^-752
    {    -226,    8,    4,    4,    2,    5,    4,    2,    5,    1,    5,    2,    8,    6,    3,    5,    4,    5,    6,    9,    8    },    // 2^-751
    {    -225,    1,    6,    8,    8,    5,    0,    8,    5,    0,    3,    0,    5,    7,    2,    7,    0,    9,    1,    4,    0    },    // 2^-750
    {    -225,    3,    3,    7,    7,    0,    1,    7,    0,    0,    6,    1,    1,    4,    5,    4,    1,    8,    2,    7,    9    },    // 2^-749
    {    -225,    6,    7,    5,    4,    0,    3,    4,    0,    1,    2,    2,    2,    9,    0,    8,    3,    6,    5,    5,    8    },    // 2^-748
    {    -224,    1,    3,    5,    0,    8,    0,    6,    8,    0,    2,    4,    4,    5,    8,    1,    6,    7,    3,    1,    2    },    // 2^-747
    {    -224,    2,    7,    0,    1,    6,    1,    3,    6,    0,    4,    8,    9,    1,    6,    3,    3,    4,    6,    2,    3    },    // 2^-746
    {    -224,    5,    4,    0,    3,    2,    2,    7,    2,    0,    9,    7,    8,    3,    2,    6,    6,    9,    2,    4,    6    },    // 2^-745
    {    -223,    1,    0,    8,    0,    6,    4,    5,    4,    4,    1,    9,    5,    6,    6,    5,    3,    3,    8,    4,    9    },    // 2^-744
    {    -223,    2,    1,    6,    1,    2,    9,    0,    8,    8,    3,    9,    1,    3,    3,    0,    6,    7,    6,    9,    9    },    // 2^-743
    {    -223,    4,    3,    2,    2,    5,    8,    1,    7,    6,    7,    8,    2,    6,    6,    1,    3,    5,    3,    9,    7    },    // 2^-742
    {    -223,    8,    6,    4,    5,    1,    6,    3,    5,    3,    5,    6,    5,    3,    2,    2,    7,    0,    7,    9,    4    },    // 2^-741
    {    -222,    1,    7,    2,    9,    0,    3,    2,    7,    0,    7,    1,    3,    0,    6,    4,    5,    4,    1,    5,    9    },    // 2^-740
    {    -222,    3,    4,    5,    8,    0,    6,    5,    4,    1,    4,    2,    6,    1,    2,    9,    0,    8,    3,    1,    8    },    // 2^-739
    {    -222,    6,    9,    1,    6,    1,    3,    0,    8,    2,    8,    5,    2,    2,    5,    8,    1,    6,    6,    3,    5    },    // 2^-738
    {    -221,    1,    3,    8,    3,    2,    2,    6,    1,    6,    5,    7,    0,    4,    5,    1,    6,    3,    3,    2,    7    },    // 2^-737
    {    -221,    2,    7,    6,    6,    4,    5,    2,    3,    3,    1,    4,    0,    9,    0,    3,    2,    6,    6,    5,    4    },    // 2^-736
    {    -221,    5,    5,    3,    2,    9,    0,    4,    6,    6,    2,    8,    1,    8,    0,    6,    5,    3,    3,    0,    8    },    // 2^-735
    {    -220,    1,    1,    0,    6,    5,    8,    0,    9,    3,    2,    5,    6,    3,    6,    1,    3,    0,    6,    6,    2    },    // 2^-734
    {    -220,    2,    2,    1,    3,    1,    6,    1,    8,    6,    5,    1,    2,    7,    2,    2,    6,    1,    3,    2,    3    },    // 2^-733
    {    -220,    4,    4,    2,    6,    3,    2,    3,    7,    3,    0,    2,    5,    4,    4,    5,    2,    2,    6,    4,    7    },    // 2^-732
    {    -220,    8,    8,    5,    2,    6,    4,    7,    4,    6,    0,    5,    0,    8,    9,    0,    4,    5,    2,    9,    3    },    // 2^-731
    {    -219,    1,    7,    7,    0,    5,    2,    9,    4,    9,    2,    1,    0,    1,    7,    8,    0,    9,    0,    5,    9    },    // 2^-730
    {    -219,    3,    5,    4,    1,    0,    5,    8,    9,    8,    4,    2,    0,    3,    5,    6,    1,    8,    1,    1,    7    },    // 2^-729
    {    -219,    7,    0,    8,    2,    1,    1,    7,    9,    6,    8,    4,    0,    7,    1,    2,    3,    6,    2,    3,    5    },    // 2^-728
    {    -218,    1,    4,    1,    6,    4,    2,    3,    5,    9,    3,    6,    8,    1,    4,    2,    4,    7,    2,    4,    7    },    // 2^-727
    {    -218,    2,    8,    3,    2,    8,    4,    7,    1,    8,    7,    3,    6,    2,    8,    4,    9,    4,    4,    9,    4    },    // 2^-726
    {    -218,    5,    6,    6,    5,    6,    9,    4,    3,    7,    4,    7,    2,    5,    6,    9,    8,    8,    9,    8,    8    },    // 2^-725
    {    -217,    1,    1,    3,    3,    1,    3,    8,    8,    7,    4,    9,    4,    5,    1,    3,    9,    7,    7,    9,    8    },    // 2^-724
    {    -217,    2,    2,    6,    6,    2,    7,    7,    7,    4,    9,    8,    9,    0,    2,    7,    9,    5,    5,    9,    5    },    // 2^-723
    {    -217,    4,    5,    3,    2,    5,    5,    5,    4,    9,    9,    7,    8,    0,    5,    5,    9,    1,    1,    9,    0    },    // 2^-722
    {    -217,    9,    0,    6,    5,    1,    1,    0,    9,    9,    9,    5,    6,    1,    1,    1,    8,    2,    3,    8,    0    },    // 2^-721
    {    -216,    1,    8,    1,    3,    0,    2,    2,    1,    9,    9,    9,    1,    2,    2,    2,    3,    6,    4,    7,    6    },    // 2^-720
    {    -216,    3,    6,    2,    6,    0,    4,    4,    3,    9,    9,    8,    2,    4,    4,    4,    7,    2,    9,    5,    2    },    // 2^-719
    {    -216,    7,    2,    5,    2,    0,    8,    8,    7,    9,    9,    6,    4,    8,    8,    9,    4,    5,    9,    0,    4    },    // 2^-718
    {    -215,    1,    4,    5,    0,    4,    1,    7,    7,    5,    9,    9,    2,    9,    7,    7,    8,    9,    1,    8,    1    },    // 2^-717
    {    -215,    2,    9,    0,    0,    8,    3,    5,    5,    1,    9,    8,    5,    9,    5,    5,    7,    8,    3,    6,    2    },    // 2^-716
    {    -215,    5,    8,    0,    1,    6,    7,    1,    0,    3,    9,    7,    1,    9,    1,    1,    5,    6,    7,    2,    3    },    // 2^-715
    {    -214,    1,    1,    6,    0,    3,    3,    4,    2,    0,    7,    9,    4,    3,    8,    2,    3,    1,    3,    4,    5    },    // 2^-714
    {    -214,    2,    3,    2,    0,    6,    6,    8,    4,    1,    5,    8,    8,    7,    6,    4,    6,    2,    6,    8,    9    },    // 2^-713
    {    -214,    4,    6,    4,    1,    3,    3,    6,    8,    3,    1,    7,    7,    5,    2,    9,    2,    5,    3,    7,    9    },    // 2^-712
    {    -214,    9,    2,    8,    2,    6,    7,    3,    6,    6,    3,    5,    5,    0,    5,    8,    5,    0,    7,    5,    8    },    // 2^-711
    {    -213,    1,    8,    5,    6,    5,    3,    4,    7,    3,    2,    7,    1,    0,    1,    1,    7,    0,    1,    5,    2    },    // 2^-710
    {    -213,    3,    7,    1,    3,    0,    6,    9,    4,    6,    5,    4,    2,    0,    2,    3,    4,    0,    3,    0,    3    },    // 2^-709
    {    -213,    7,    4,    2,    6,    1,    3,    8,    9,    3,    0,    8,    4,    0,    4,    6,    8,    0,    6,    0,    6    },    // 2^-708
    {    -212,    1,    4,    8,    5,    2,    2,    7,    7,    8,    6,    1,    6,    8,    0,    9,    3,    6,    1,    2,    1    },    // 2^-707
    {    -212,    2,    9,    7,    0,    4,    5,    5,    5,    7,    2,    3,    3,    6,    1,    8,    7,    2,    2,    4,    2    },    // 2^-706
    {    -212,    5,    9,    4,    0,    9,    1,    1,    1,    4,    4,    6,    7,    2,    3,    7,    4,    4,    4,    8,    5    },    // 2^-705
    {    -211,    1,    1,    8,    8,    1,    8,    2,    2,    2,    8,    9,    3,    4,    4,    7,    4,    8,    8,    9,    7    },    // 2^-704
    {    -211,    2,    3,    7,    6,    3,    6,    4,    4,    5,    7,    8,    6,    8,    9,    4,    9,    7,    7,    9,    4    },    // 2^-703
    {    -211,    4,    7,    5,    2,    7,    2,    8,    9,    1,    5,    7,    3,    7,    8,    9,    9,    5,    5,    8,    8    },    // 2^-702
    {    -211,    9,    5,    0,    5,    4,    5,    7,    8,    3,    1,    4,    7,    5,    7,    9,    9,    1,    1,    7,    6    },    // 2^-701
    {    -210,    1,    9,    0,    1,    0,    9,    1,    5,    6,    6,    2,    9,    5,    1,    5,    9,    8,    2,    3,    5    },    // 2^-700
    {    -210,    3,    8,    0,    2,    1,    8,    3,    1,    3,    2,    5,    9,    0,    3,    1,    9,    6,    4,    7,    0    },    // 2^-699
    {    -210,    7,    6,    0,    4,    3,    6,    6,    2,    6,    5,    1,    8,    0,    6,    3,    9,    2,    9,    4,    1    },    // 2^-698
    {    -209,    1,    5,    2,    0,    8,    7,    3,    2,    5,    3,    0,    3,    6,    1,    2,    7,    8,    5,    8,    8    },    // 2^-697
    {    -209,    3,    0,    4,    1,    7,    4,    6,    5,    0,    6,    0,    7,    2,    2,    5,    5,    7,    1,    7,    6    },    // 2^-696
    {    -209,    6,    0,    8,    3,    4,    9,    3,    0,    1,    2,    1,    4,    4,    5,    1,    1,    4,    3,    5,    2    },    // 2^-695
    {    -208,    1,    2,    1,    6,    6,    9,    8,    6,    0,    2,    4,    2,    8,    9,    0,    2,    2,    8,    7,    0    },    // 2^-694
    {    -208,    2,    4,    3,    3,    3,    9,    7,    2,    0,    4,    8,    5,    7,    8,    0,    4,    5,    7,    4,    1    },    // 2^-693
    {    -208,    4,    8,    6,    6,    7,    9,    4,    4,    0,    9,    7,    1,    5,    6,    0,    9,    1,    4,    8,    2    },    // 2^-692
    {    -208,    9,    7,    3,    3,    5,    8,    8,    8,    1,    9,    4,    3,    1,    2,    1,    8,    2,    9,    6,    4    },    // 2^-691
    {    -207,    1,    9,    4,    6,    7,    1,    7,    7,    6,    3,    8,    8,    6,    2,    4,    3,    6,    5,    9,    3    },    // 2^-690
    {    -207,    3,    8,    9,    3,    4,    3,    5,    5,    2,    7,    7,    7,    2,    4,    8,    7,    3,    1,    8,    6    },    // 2^-689
    {    -207,    7,    7,    8,    6,    8,    7,    1,    0,    5,    5,    5,    4,    4,    9,    7,    4,    6,    3,    7,    1    },    // 2^-688
    {    -206,    1,    5,    5,    7,    3,    7,    4,    2,    1,    1,    1,    0,    8,    9,    9,    4,    9,    2,    7,    4    },    // 2^-687
    {    -206,    3,    1,    1,    4,    7,    4,    8,    4,    2,    2,    2,    1,    7,    9,    8,    9,    8,    5,    4,    8    },    // 2^-686
    {    -206,    6,    2,    2,    9,    4,    9,    6,    8,    4,    4,    4,    3,    5,    9,    7,    9,    7,    0,    9,    7    },    // 2^-685
    {    -205,    1,    2,    4,    5,    8,    9,    9,    3,    6,    8,    8,    8,    7,    1,    9,    5,    9,    4,    1,    9    },    // 2^-684
    {    -205,    2,    4,    9,    1,    7,    9,    8,    7,    3,    7,    7,    7,    4,    3,    9,    1,    8,    8,    3,    9    },    // 2^-683
    {    -205,    4,    9,    8,    3,    5,    9,    7,    4,    7,    5,    5,    4,    8,    7,    8,    3,    7,    6,    7,    8    },    // 2^-682
    {    -205,    9,    9,    6,    7,    1,    9,    4,    9,    5,    1,    0,    9,    7,    5,    6,    7,    5,    3,    5,    5    },    // 2^-681
    {    -204,    1,    9,    9,    3,    4,    3,    8,    9,    9,    0,    2,    1,    9,    5,    1,    3,    5,    0,    7,    1    },    // 2^-680
    {    -204,    3,    9,    8,    6,    8,    7,    7,    9,    8,    0,    4,    3,    9,    0,    2,    7,    0,    1,    4,    2    },    // 2^-679
    {    -204,    7,    9,    7,    3,    7,    5,    5,    9,    6,    0,    8,    7,    8,    0,    5,    4,    0,    2,    8,    4    },    // 2^-678
    {    -203,    1,    5,    9,    4,    7,    5,    1,    1,    9,    2,    1,    7,    5,    6,    1,    0,    8,    0,    5,    7    },    // 2^-677
    {    -203,    3,    1,    8,    9,    5,    0,    2,    3,    8,    4,    3,    5,    1,    2,    2,    1,    6,    1,    1,    4    },    // 2^-676
    {    -203,    6,    3,    7,    9,    0,    0,    4,    7,    6,    8,    7,    0,    2,    4,    4,    3,    2,    2,    2,    7    },    // 2^-675
    {    -202,    1,    2,    7,    5,    8,    0,    0,    9,    5,    3,    7,    4,    0,    4,    8,    8,    6,    4,    4,    5    },    // 2^-674
    {    -202,    2,    5,    5,    1,    6,    0,    1,    9,    0,    7,    4,    8,    0,    9,    7,    7,    2,    8,    9,    1    },    // 2^-673
    {    -202,    5,    1,    0,    3,    2,    0,    3,    8,    1,    4,    9,    6,    1,    9,    5,    4,    5,    7,    8,    2    },    // 2^-672
    {    -201,    1,    0,    2,    0,    6,    4,    0,    7,    6,    2,    9,    9,    2,    3,    9,    0,    9,    1,    5,    6    },    // 2^-671
    {    -201,    2,    0,    4,    1,    2,    8,    1,    5,    2,    5,    9,    8,    4,    7,    8,    1,    8,    3,    1,    3    },    // 2^-670
    {    -201,    4,    0,    8,    2,    5,    6,    3,    0,    5,    1,    9,    6,    9,    5,    6,    3,    6,    6,    2,    5    },    // 2^-669
    {    -201,    8,    1,    6,    5,    1,    2,    6,    1,    0,    3,    9,    3,    9,    1,    2,    7,    3,    2,    5,    1    },    // 2^-668
    {    -200,    1,    6,    3,    3,    0,    2,    5,    2,    2,    0,    7,    8,    7,    8,    2,    5,    4,    6,    5,    0    },    // 2^-667
    {    -200,    3,    2,    6,    6,    0,    5,    0,    4,    4,    1,    5,    7,    5,    6,    5,    0,    9,    3,    0,    0    },    // 2^-666
    {    -200,    6,    5,    3,    2,    1,    0,    0,    8,    8,    3,    1,    5,    1,    3,    0,    1,    8,    6,    0,    1    },    // 2^-665
    {    -199,    1,    3,    0,    6,    4,    2,    0,    1,    7,    6,    6,    3,    0,    2,    6,    0,    3,    7,    2,    0    },    // 2^-664
    {    -199,    2,    6,    1,    2,    8,    4,    0,    3,    5,    3,    2,    6,    0,    5,    2,    0,    7,    4,    4,    0    },    // 2^-663
    {    -199,    5,    2,    2,    5,    6,    8,    0,    7,    0,    6,    5,    2,    1,    0,    4,    1,    4,    8,    8,    1    },    // 2^-662
    {    -198,    1,    0,    4,    5,    1,    3,    6,    1,    4,    1,    3,    0,    4,    2,    0,    8,    2,    9,    7,    6    },    // 2^-661
    {    -198,    2,    0,    9,    0,    2,    7,    2,    2,    8,    2,    6,    0,    8,    4,    1,    6,    5,    9,    5,    2    },    // 2^-660
    {    -198,    4,    1,    8,    0,    5,    4,    4,    5,    6,    5,    2,    1,    6,    8,    3,    3,    1,    9,    0,    4    },    // 2^-659
    {    -198,    8,    3,    6,    1,    0,    8,    9,    1,    3,    0,    4,    3,    3,    6,    6,    6,    3,    8,    0,    9    },    // 2^-658
    {    -197,    1,    6,    7,    2,    2,    1,    7,    8,    2,    6,    0,    8,    6,    7,    3,    3,    2,    7,    6,    2    },    // 2^-657
    {    -197,    3,    3,    4,    4,    4,    3,    5,    6,    5,    2,    1,    7,    3,    4,    6,    6,    5,    5,    2,    4    },    // 2^-656
    {    -197,    6,    6,    8,    8,    8,    7,    1,    3,    0,    4,    3,    4,    6,    9,    3,    3,    1,    0,    4,    7    },    // 2^-655
    {    -196,    1,    3,    3,    7,    7,    7,    4,    2,    6,    0,    8,    6,    9,    3,    8,    6,    6,    2,    0,    9    },    // 2^-654
    {    -196,    2,    6,    7,    5,    5,    4,    8,    5,    2,    1,    7,    3,    8,    7,    7,    3,    2,    4,    1,    9    },    // 2^-653
    {    -196,    5,    3,    5,    1,    0,    9,    7,    0,    4,    3,    4,    7,    7,    5,    4,    6,    4,    8,    3,    8    },    // 2^-652
    {    -195,    1,    0,    7,    0,    2,    1,    9,    4,    0,    8,    6,    9,    5,    5,    0,    9,    2,    9,    6,    8    },    // 2^-651
    {    -195,    2,    1,    4,    0,    4,    3,    8,    8,    1,    7,    3,    9,    1,    0,    1,    8,    5,    9,    3,    5    },    // 2^-650
    {    -195,    4,    2,    8,    0,    8,    7,    7,    6,    3,    4,    7,    8,    2,    0,    3,    7,    1,    8,    7,    0    },    // 2^-649
    {    -195,    8,    5,    6,    1,    7,    5,    5,    2,    6,    9,    5,    6,    4,    0,    7,    4,    3,    7,    4,    0    },    // 2^-648
    {    -194,    1,    7,    1,    2,    3,    5,    1,    0,    5,    3,    9,    1,    2,    8,    1,    4,    8,    7,    4,    8    },    // 2^-647
    {    -194,    3,    4,    2,    4,    7,    0,    2,    1,    0,    7,    8,    2,    5,    6,    2,    9,    7,    4,    9,    6    },    // 2^-646
    {    -194,    6,    8,    4,    9,    4,    0,    4,    2,    1,    5,    6,    5,    1,    2,    5,    9,    4,    9,    9,    2    },    // 2^-645
    {    -193,    1,    3,    6,    9,    8,    8,    0,    8,    4,    3,    1,    3,    0,    2,    5,    1,    8,    9,    9,    8    },    // 2^-644
    {    -193,    2,    7,    3,    9,    7,    6,    1,    6,    8,    6,    2,    6,    0,    5,    0,    3,    7,    9,    9,    7    },    // 2^-643
    {    -193,    5,    4,    7,    9,    5,    2,    3,    3,    7,    2,    5,    2,    1,    0,    0,    7,    5,    9,    9,    4    },    // 2^-642
    {    -192,    1,    0,    9,    5,    9,    0,    4,    6,    7,    4,    5,    0,    4,    2,    0,    1,    5,    1,    9,    9    },    // 2^-641
    {    -192,    2,    1,    9,    1,    8,    0,    9,    3,    4,    9,    0,    0,    8,    4,    0,    3,    0,    3,    9,    8    },    // 2^-640
    {    -192,    4,    3,    8,    3,    6,    1,    8,    6,    9,    8,    0,    1,    6,    8,    0,    6,    0,    7,    9,    5    },    // 2^-639
    {    -192,    8,    7,    6,    7,    2,    3,    7,    3,    9,    6,    0,    3,    3,    6,    1,    2,    1,    5,    9,    0    },    // 2^-638
    {    -191,    1,    7,    5,    3,    4,    4,    7,    4,    7,    9,    2,    0,    6,    7,    2,    2,    4,    3,    1,    8    },    // 2^-637
    {    -191,    3,    5,    0,    6,    8,    9,    4,    9,    5,    8,    4,    1,    3,    4,    4,    4,    8,    6,    3,    6    },    // 2^-636
    {    -191,    7,    0,    1,    3,    7,    8,    9,    9,    1,    6,    8,    2,    6,    8,    8,    9,    7,    2,    7,    2    },    // 2^-635
    {    -190,    1,    4,    0,    2,    7,    5,    7,    9,    8,    3,    3,    6,    5,    3,    7,    7,    9,    4,    5,    4    },    // 2^-634
    {    -190,    2,    8,    0,    5,    5,    1,    5,    9,    6,    6,    7,    3,    0,    7,    5,    5,    8,    9,    0,    9    },    // 2^-633
    {    -190,    5,    6,    1,    1,    0,    3,    1,    9,    3,    3,    4,    6,    1,    5,    1,    1,    7,    8,    1,    8    },    // 2^-632
    {    -189,    1,    1,    2,    2,    2,    0,    6,    3,    8,    6,    6,    9,    2,    3,    0,    2,    3,    5,    6,    4    },    // 2^-631
    {    -189,    2,    2,    4,    4,    4,    1,    2,    7,    7,    3,    3,    8,    4,    6,    0,    4,    7,    1,    2,    7    },    // 2^-630
    {    -189,    4,    4,    8,    8,    8,    2,    5,    5,    4,    6,    7,    6,    9,    2,    0,    9,    4,    2,    5,    4    },    // 2^-629
    {    -189,    8,    9,    7,    7,    6,    5,    1,    0,    9,    3,    5,    3,    8,    4,    1,    8,    8,    5,    0,    8    },    // 2^-628
    {    -188,    1,    7,    9,    5,    5,    3,    0,    2,    1,    8,    7,    0,    7,    6,    8,    3,    7,    7,    0,    2    },    // 2^-627
    {    -188,    3,    5,    9,    1,    0,    6,    0,    4,    3,    7,    4,    1,    5,    3,    6,    7,    5,    4,    0,    3    },    // 2^-626
    {    -188,    7,    1,    8,    2,    1,    2,    0,    8,    7,    4,    8,    3,    0,    7,    3,    5,    0,    8,    0,    7    },    // 2^-625
    {    -187,    1,    4,    3,    6,    4,    2,    4,    1,    7,    4,    9,    6,    6,    1,    4,    7,    0,    1,    6,    1    },    // 2^-624
    {    -187,    2,    8,    7,    2,    8,    4,    8,    3,    4,    9,    9,    3,    2,    2,    9,    4,    0,    3,    2,    3    },    // 2^-623
    {    -187,    5,    7,    4,    5,    6,    9,    6,    6,    9,    9,    8,    6,    4,    5,    8,    8,    0,    6,    4,    5    },    // 2^-622
    {    -186,    1,    1,    4,    9,    1,    3,    9,    3,    3,    9,    9,    7,    2,    9,    1,    7,    6,    1,    2,    9    },    // 2^-621
    {    -186,    2,    2,    9,    8,    2,    7,    8,    6,    7,    9,    9,    4,    5,    8,    3,    5,    2,    2,    5,    8    },    // 2^-620
    {    -186,    4,    5,    9,    6,    5,    5,    7,    3,    5,    9,    8,    9,    1,    6,    7,    0,    4,    5,    1,    6    },    // 2^-619
    {    -186,    9,    1,    9,    3,    1,    1,    4,    7,    1,    9,    7,    8,    3,    3,    4,    0,    9,    0,    3,    2    },    // 2^-618
    {    -185,    1,    8,    3,    8,    6,    2,    2,    9,    4,    3,    9,    5,    6,    6,    6,    8,    1,    8,    0,    6    },    // 2^-617
    {    -185,    3,    6,    7,    7,    2,    4,    5,    8,    8,    7,    9,    1,    3,    3,    3,    6,    3,    6,    1,    3    },    // 2^-616
    {    -185,    7,    3,    5,    4,    4,    9,    1,    7,    7,    5,    8,    2,    6,    6,    7,    2,    7,    2,    2,    6    },    // 2^-615
    {    -184,    1,    4,    7,    0,    8,    9,    8,    3,    5,    5,    1,    6,    5,    3,    3,    4,    5,    4,    4,    5    },    // 2^-614
    {    -184,    2,    9,    4,    1,    7,    9,    6,    7,    1,    0,    3,    3,    0,    6,    6,    9,    0,    8,    9,    0    },    // 2^-613
    {    -184,    5,    8,    8,    3,    5,    9,    3,    4,    2,    0,    6,    6,    1,    3,    3,    8,    1,    7,    8,    1    },    // 2^-612
    {    -183,    1,    1,    7,    6,    7,    1,    8,    6,    8,    4,    1,    3,    2,    2,    6,    7,    6,    3,    5,    6    },    // 2^-611
    {    -183,    2,    3,    5,    3,    4,    3,    7,    3,    6,    8,    2,    6,    4,    5,    3,    5,    2,    7,    1,    2    },    // 2^-610
    {    -183,    4,    7,    0,    6,    8,    7,    4,    7,    3,    6,    5,    2,    9,    0,    7,    0,    5,    4,    2,    5    },    // 2^-609
    {    -183,    9,    4,    1,    3,    7,    4,    9,    4,    7,    3,    0,    5,    8,    1,    4,    1,    0,    8,    4,    9    },    // 2^-608
    {    -182,    1,    8,    8,    2,    7,    4,    9,    8,    9,    4,    6,    1,    1,    6,    2,    8,    2,    1,    7,    0    },    // 2^-607
    {    -182,    3,    7,    6,    5,    4,    9,    9,    7,    8,    9,    2,    2,    3,    2,    5,    6,    4,    3,    4,    0    },    // 2^-606
    {    -182,    7,    5,    3,    0,    9,    9,    9,    5,    7,    8,    4,    4,    6,    5,    1,    2,    8,    6,    7,    9    },    // 2^-605
    {    -181,    1,    5,    0,    6,    1,    9,    9,    9,    1,    5,    6,    8,    9,    3,    0,    2,    5,    7,    3,    6    },    // 2^-604
    {    -181,    3,    0,    1,    2,    3,    9,    9,    8,    3,    1,    3,    7,    8,    6,    0,    5,    1,    4,    7,    2    },    // 2^-603
    {    -181,    6,    0,    2,    4,    7,    9,    9,    6,    6,    2,    7,    5,    7,    2,    1,    0,    2,    9,    4,    4    },    // 2^-602
    {    -180,    1,    2,    0,    4,    9,    5,    9,    9,    3,    2,    5,    5,    1,    4,    4,    2,    0,    5,    8,    9    },    // 2^-601
    {    -180,    2,    4,    0,    9,    9,    1,    9,    8,    6,    5,    1,    0,    2,    8,    8,    4,    1,    1,    7,    7    },    // 2^-600
    {    -180,    4,    8,    1,    9,    8,    3,    9,    7,    3,    0,    2,    0,    5,    7,    6,    8,    2,    3,    5,    5    },    // 2^-599
    {    -180,    9,    6,    3,    9,    6,    7,    9,    4,    6,    0,    4,    1,    1,    5,    3,    6,    4,    7,    1,    0    },    // 2^-598
    {    -179,    1,    9,    2,    7,    9,    3,    5,    8,    9,    2,    0,    8,    2,    3,    0,    7,    2,    9,    4,    2    },    // 2^-597
    {    -179,    3,    8,    5,    5,    8,    7,    1,    7,    8,    4,    1,    6,    4,    6,    1,    4,    5,    8,    8,    4    },    // 2^-596
    {    -179,    7,    7,    1,    1,    7,    4,    3,    5,    6,    8,    3,    2,    9,    2,    2,    9,    1,    7,    6,    8    },    // 2^-595
    {    -178,    1,    5,    4,    2,    3,    4,    8,    7,    1,    3,    6,    6,    5,    8,    4,    5,    8,    3,    5,    4    },    // 2^-594
    {    -178,    3,    0,    8,    4,    6,    9,    7,    4,    2,    7,    3,    3,    1,    6,    9,    1,    6,    7,    0,    7    },    // 2^-593
    {    -178,    6,    1,    6,    9,    3,    9,    4,    8,    5,    4,    6,    6,    3,    3,    8,    3,    3,    4,    1,    4    },    // 2^-592
    {    -177,    1,    2,    3,    3,    8,    7,    8,    9,    7,    0,    9,    3,    2,    6,    7,    6,    6,    6,    8,    3    },    // 2^-591
    {    -177,    2,    4,    6,    7,    7,    5,    7,    9,    4,    1,    8,    6,    5,    3,    5,    3,    3,    3,    6,    6    },    // 2^-590
    {    -177,    4,    9,    3,    5,    5,    1,    5,    8,    8,    3,    7,    3,    0,    7,    0,    6,    6,    7,    3,    1    },    // 2^-589
    {    -177,    9,    8,    7,    1,    0,    3,    1,    7,    6,    7,    4,    6,    1,    4,    1,    3,    3,    4,    6,    3    },    // 2^-588
    {    -176,    1,    9,    7,    4,    2,    0,    6,    3,    5,    3,    4,    9,    2,    2,    8,    2,    6,    6,    9,    3    },    // 2^-587
    {    -176,    3,    9,    4,    8,    4,    1,    2,    7,    0,    6,    9,    8,    4,    5,    6,    5,    3,    3,    8,    5    },    // 2^-586
    {    -176,    7,    8,    9,    6,    8,    2,    5,    4,    1,    3,    9,    6,    9,    1,    3,    0,    6,    7,    7,    0    },    // 2^-585
    {    -175,    1,    5,    7,    9,    3,    6,    5,    0,    8,    2,    7,    9,    3,    8,    2,    6,    1,    3,    5,    4    },    // 2^-584
    {    -175,    3,    1,    5,    8,    7,    3,    0,    1,    6,    5,    5,    8,    7,    6,    5,    2,    2,    7,    0,    8    },    // 2^-583
    {    -175,    6,    3,    1,    7,    4,    6,    0,    3,    3,    1,    1,    7,    5,    3,    0,    4,    5,    4,    1,    6    },    // 2^-582
    {    -174,    1,    2,    6,    3,    4,    9,    2,    0,    6,    6,    2,    3,    5,    0,    6,    0,    9,    0,    8,    3    },    // 2^-581
    {    -174,    2,    5,    2,    6,    9,    8,    4,    1,    3,    2,    4,    7,    0,    1,    2,    1,    8,    1,    6,    6    },    // 2^-580
    {    -174,    5,    0,    5,    3,    9,    6,    8,    2,    6,    4,    9,    4,    0,    2,    4,    3,    6,    3,    3,    3    },    // 2^-579
    {    -173,    1,    0,    1,    0,    7,    9,    3,    6,    5,    2,    9,    8,    8,    0,    4,    8,    7,    2,    6,    7    },    // 2^-578
    {    -173,    2,    0,    2,    1,    5,    8,    7,    3,    0,    5,    9,    7,    6,    0,    9,    7,    4,    5,    3,    3    },    // 2^-577
    {    -173,    4,    0,    4,    3,    1,    7,    4,    6,    1,    1,    9,    5,    2,    1,    9,    4,    9,    0,    6,    6    },    // 2^-576
    {    -173,    8,    0,    8,    6,    3,    4,    9,    2,    2,    3,    9,    0,    4,    3,    8,    9,    8,    1,    3,    3    },    // 2^-575
    {    -172,    1,    6,    1,    7,    2,    6,    9,    8,    4,    4,    7,    8,    0,    8,    7,    7,    9,    6,    2,    7    },    // 2^-574
    {    -172,    3,    2,    3,    4,    5,    3,    9,    6,    8,    9,    5,    6,    1,    7,    5,    5,    9,    2,    5,    3    },    // 2^-573
    {    -172,    6,    4,    6,    9,    0,    7,    9,    3,    7,    9,    1,    2,    3,    5,    1,    1,    8,    5,    0,    6    },    // 2^-572
    {    -171,    1,    2,    9,    3,    8,    1,    5,    8,    7,    5,    8,    2,    4,    7,    0,    2,    3,    7,    0,    1    },    // 2^-571
    {    -171,    2,    5,    8,    7,    6,    3,    1,    7,    5,    1,    6,    4,    9,    4,    0,    4,    7,    4,    0,    2    },    // 2^-570
    {    -171,    5,    1,    7,    5,    2,    6,    3,    5,    0,    3,    2,    9,    8,    8,    0,    9,    4,    8,    0,    5    },    // 2^-569
    {    -170,    1,    0,    3,    5,    0,    5,    2,    7,    0,    0,    6,    5,    9,    7,    6,    1,    8,    9,    6,    1    },    // 2^-568
    {    -170,    2,    0,    7,    0,    1,    0,    5,    4,    0,    1,    3,    1,    9,    5,    2,    3,    7,    9,    2,    2    },    // 2^-567
    {    -170,    4,    1,    4,    0,    2,    1,    0,    8,    0,    2,    6,    3,    9,    0,    4,    7,    5,    8,    4,    4    },    // 2^-566
    {    -170,    8,    2,    8,    0,    4,    2,    1,    6,    0,    5,    2,    7,    8,    0,    9,    5,    1,    6,    8,    8    },    // 2^-565
    {    -169,    1,    6,    5,    6,    0,    8,    4,    3,    2,    1,    0,    5,    5,    6,    1,    9,    0,    3,    3,    8    },    // 2^-564
    {    -169,    3,    3,    1,    2,    1,    6,    8,    6,    4,    2,    1,    1,    1,    2,    3,    8,    0,    6,    7,    5    },    // 2^-563
    {    -169,    6,    6,    2,    4,    3,    3,    7,    2,    8,    4,    2,    2,    2,    4,    7,    6,    1,    3,    5,    0    },    // 2^-562
    {    -168,    1,    3,    2,    4,    8,    6,    7,    4,    5,    6,    8,    4,    4,    4,    9,    5,    2,    2,    7,    0    },    // 2^-561
    {    -168,    2,    6,    4,    9,    7,    3,    4,    9,    1,    3,    6,    8,    8,    9,    9,    0,    4,    5,    4,    0    },    // 2^-560
    {    -168,    5,    2,    9,    9,    4,    6,    9,    8,    2,    7,    3,    7,    7,    9,    8,    0,    9,    0,    8,    0    },    // 2^-559
    {    -167,    1,    0,    5,    9,    8,    9,    3,    9,    6,    5,    4,    7,    5,    5,    9,    6,    1,    8,    1,    6    },    // 2^-558
    {    -167,    2,    1,    1,    9,    7,    8,    7,    9,    3,    0,    9,    5,    1,    1,    9,    2,    3,    6,    3,    2    },    // 2^-557
    {    -167,    4,    2,    3,    9,    5,    7,    5,    8,    6,    1,    9,    0,    2,    3,    8,    4,    7,    2,    6,    4    },    // 2^-556
    {    -167,    8,    4,    7,    9,    1,    5,    1,    7,    2,    3,    8,    0,    4,    7,    6,    9,    4,    5,    2,    8    },    // 2^-555
    {    -166,    1,    6,    9,    5,    8,    3,    0,    3,    4,    4,    7,    6,    0,    9,    5,    3,    8,    9,    0,    6    },    // 2^-554
    {    -166,    3,    3,    9,    1,    6,    6,    0,    6,    8,    9,    5,    2,    1,    9,    0,    7,    7,    8,    1,    1    },    // 2^-553
    {    -166,    6,    7,    8,    3,    3,    2,    1,    3,    7,    9,    0,    4,    3,    8,    1,    5,    5,    6,    2,    3    },    // 2^-552
    {    -165,    1,    3,    5,    6,    6,    6,    4,    2,    7,    5,    8,    0,    8,    7,    6,    3,    1,    1,    2,    5    },    // 2^-551
    {    -165,    2,    7,    1,    3,    3,    2,    8,    5,    5,    1,    6,    1,    7,    5,    2,    6,    2,    2,    4,    9    },    // 2^-550
    {    -165,    5,    4,    2,    6,    6,    5,    7,    1,    0,    3,    2,    3,    5,    0,    5,    2,    4,    4,    9,    8    },    // 2^-549
    {    -164,    1,    0,    8,    5,    3,    3,    1,    4,    2,    0,    6,    4,    7,    0,    1,    0,    4,    8,    0,    0    },    // 2^-548
    {    -164,    2,    1,    7,    0,    6,    6,    2,    8,    4,    1,    2,    9,    4,    0,    2,    0,    9,    7,    9,    9    },    // 2^-547
    {    -164,    4,    3,    4,    1,    3,    2,    5,    6,    8,    2,    5,    8,    8,    0,    4,    1,    9,    5,    9,    8    },    // 2^-546
    {    -164,    8,    6,    8,    2,    6,    5,    1,    3,    6,    5,    1,    7,    6,    0,    8,    3,    9,    1,    9,    7    },    // 2^-545
    {    -163,    1,    7,    3,    6,    5,    3,    0,    2,    7,    3,    0,    3,    5,    2,    1,    6,    7,    8,    3,    9    },    // 2^-544
    {    -163,    3,    4,    7,    3,    0,    6,    0,    5,    4,    6,    0,    7,    0,    4,    3,    3,    5,    6,    7,    9    },    // 2^-543
    {    -163,    6,    9,    4,    6,    1,    2,    1,    0,    9,    2,    1,    4,    0,    8,    6,    7,    1,    3,    5,    8    },    // 2^-542
    {    -162,    1,    3,    8,    9,    2,    2,    4,    2,    1,    8,    4,    2,    8,    1,    7,    3,    4,    2,    7,    2    },    // 2^-541
    {    -162,    2,    7,    7,    8,    4,    4,    8,    4,    3,    6,    8,    5,    6,    3,    4,    6,    8,    5,    4,    3    },    // 2^-540
    {    -162,    5,    5,    5,    6,    8,    9,    6,    8,    7,    3,    7,    1,    2,    6,    9,    3,    7,    0,    8,    6    },    // 2^-539
    {    -161,    1,    1,    1,    1,    3,    7,    9,    3,    7,    4,    7,    4,    2,    5,    3,    8,    7,    4,    1,    7    },    // 2^-538
    {    -161,    2,    2,    2,    2,    7,    5,    8,    7,    4,    9,    4,    8,    5,    0,    7,    7,    4,    8,    3,    4    },    // 2^-537
    {    -161,    4,    4,    4,    5,    5,    1,    7,    4,    9,    8,    9,    7,    0,    1,    5,    4,    9,    6,    6,    9    },    // 2^-536
    {    -161,    8,    8,    9,    1,    0,    3,    4,    9,    9,    7,    9,    4,    0,    3,    0,    9,    9,    3,    3,    8    },    // 2^-535
    {    -160,    1,    7,    7,    8,    2,    0,    6,    9,    9,    9,    5,    8,    8,    0,    6,    1,    9,    8,    6,    8    },    // 2^-534
    {    -160,    3,    5,    5,    6,    4,    1,    3,    9,    9,    9,    1,    7,    6,    1,    2,    3,    9,    7,    3,    5    },    // 2^-533
    {    -160,    7,    1,    1,    2,    8,    2,    7,    9,    9,    8,    3,    5,    2,    2,    4,    7,    9,    4,    7,    0    },    // 2^-532
    {    -159,    1,    4,    2,    2,    5,    6,    5,    5,    9,    9,    6,    7,    0,    4,    4,    9,    5,    8,    9,    4    },    // 2^-531
    {    -159,    2,    8,    4,    5,    1,    3,    1,    1,    9,    9,    3,    4,    0,    8,    9,    9,    1,    7,    8,    8    },    // 2^-530
    {    -159,    5,    6,    9,    0,    2,    6,    2,    3,    9,    8,    6,    8,    1,    7,    9,    8,    3,    5,    7,    6    },    // 2^-529
    {    -158,    1,    1,    3,    8,    0,    5,    2,    4,    7,    9,    7,    3,    6,    3,    5,    9,    6,    7,    1,    5    },    // 2^-528
    {    -158,    2,    2,    7,    6,    1,    0,    4,    9,    5,    9,    4,    7,    2,    7,    1,    9,    3,    4,    3,    0    },    // 2^-527
    {    -158,    4,    5,    5,    2,    2,    0,    9,    9,    1,    8,    9,    4,    5,    4,    3,    8,    6,    8,    6,    1    },    // 2^-526
    {    -158,    9,    1,    0,    4,    4,    1,    9,    8,    3,    7,    8,    9,    0,    8,    7,    7,    3,    7,    2,    2    },    // 2^-525
    {    -157,    1,    8,    2,    0,    8,    8,    3,    9,    6,    7,    5,    7,    8,    1,    7,    5,    4,    7,    4,    4    },    // 2^-524
    {    -157,    3,    6,    4,    1,    7,    6,    7,    9,    3,    5,    1,    5,    6,    3,    5,    0,    9,    4,    8,    9    },    // 2^-523
    {    -157,    7,    2,    8,    3,    5,    3,    5,    8,    7,    0,    3,    1,    2,    7,    0,    1,    8,    9,    7,    7    },    // 2^-522
    {    -156,    1,    4,    5,    6,    7,    0,    7,    1,    7,    4,    0,    6,    2,    5,    4,    0,    3,    7,    9,    5    },    // 2^-521
    {    -156,    2,    9,    1,    3,    4,    1,    4,    3,    4,    8,    1,    2,    5,    0,    8,    0,    7,    5,    9,    1    },    // 2^-520
    {    -156,    5,    8,    2,    6,    8,    2,    8,    6,    9,    6,    2,    5,    0,    1,    6,    1,    5,    1,    8,    2    },    // 2^-519
    {    -155,    1,    1,    6,    5,    3,    6,    5,    7,    3,    9,    2,    5,    0,    0,    3,    2,    3,    0,    3,    6    },    // 2^-518
    {    -155,    2,    3,    3,    0,    7,    3,    1,    4,    7,    8,    5,    0,    0,    0,    6,    4,    6,    0,    7,    3    },    // 2^-517
    {    -155,    4,    6,    6,    1,    4,    6,    2,    9,    5,    7,    0,    0,    0,    1,    2,    9,    2,    1,    4,    6    },    // 2^-516
    {    -155,    9,    3,    2,    2,    9,    2,    5,    9,    1,    4,    0,    0,    0,    2,    5,    8,    4,    2,    9,    1    },    // 2^-515
    {    -154,    1,    8,    6,    4,    5,    8,    5,    1,    8,    2,    8,    0,    0,    0,    5,    1,    6,    8,    5,    8    },    // 2^-514
    {    -154,    3,    7,    2,    9,    1,    7,    0,    3,    6,    5,    6,    0,    0,    1,    0,    3,    3,    7,    1,    6    },    // 2^-513
    {    -154,    7,    4,    5,    8,    3,    4,    0,    7,    3,    1,    2,    0,    0,    2,    0,    6,    7,    4,    3,    3    },    // 2^-512
    {    -153,    1,    4,    9,    1,    6,    6,    8,    1,    4,    6,    2,    4,    0,    0,    4,    1,    3,    4,    8,    7    },    // 2^-511
    {    -153,    2,    9,    8,    3,    3,    3,    6,    2,    9,    2,    4,    8,    0,    0,    8,    2,    6,    9,    7,    3    },    // 2^-510
    {    -153,    5,    9,    6,    6,    6,    7,    2,    5,    8,    4,    9,    6,    0,    1,    6,    5,    3,    9,    4,    6    },    // 2^-509
    {    -152,    1,    1,    9,    3,    3,    3,    4,    5,    1,    6,    9,    9,    2,    0,    3,    3,    0,    7,    8,    9    },    // 2^-508
    {    -152,    2,    3,    8,    6,    6,    6,    9,    0,    3,    3,    9,    8,    4,    0,    6,    6,    1,    5,    7,    9    },    // 2^-507
    {    -152,    4,    7,    7,    3,    3,    3,    8,    0,    6,    7,    9,    6,    8,    1,    3,    2,    3,    1,    5,    7    },    // 2^-506
    {    -152,    9,    5,    4,    6,    6,    7,    6,    1,    3,    5,    9,    3,    6,    2,    6,    4,    6,    3,    1,    4    },    // 2^-505
    {    -151,    1,    9,    0,    9,    3,    3,    5,    2,    2,    7,    1,    8,    7,    2,    5,    2,    9,    2,    6,    3    },    // 2^-504
    {    -151,    3,    8,    1,    8,    6,    7,    0,    4,    5,    4,    3,    7,    4,    5,    0,    5,    8,    5,    2,    6    },    // 2^-503
    {    -151,    7,    6,    3,    7,    3,    4,    0,    9,    0,    8,    7,    4,    9,    0,    1,    1,    7,    0,    5,    1    },    // 2^-502
    {    -150,    1,    5,    2,    7,    4,    6,    8,    1,    8,    1,    7,    4,    9,    8,    0,    2,    3,    4,    1,    0    },    // 2^-501
    {    -150,    3,    0,    5,    4,    9,    3,    6,    3,    6,    3,    4,    9,    9,    6,    0,    4,    6,    8,    2,    1    },    // 2^-500
    {    -150,    6,    1,    0,    9,    8,    7,    2,    7,    2,    6,    9,    9,    9,    2,    0,    9,    3,    6,    4,    1    },    // 2^-499
    {    -149,    1,    2,    2,    1,    9,    7,    4,    5,    4,    5,    3,    9,    9,    8,    4,    1,    8,    7,    2,    8    },    // 2^-498
    {    -149,    2,    4,    4,    3,    9,    4,    9,    0,    9,    0,    7,    9,    9,    6,    8,    3,    7,    4,    5,    6    },    // 2^-497
    {    -149,    4,    8,    8,    7,    8,    9,    8,    1,    8,    1,    5,    9,    9,    3,    6,    7,    4,    9,    1,    3    },    // 2^-496
    {    -149,    9,    7,    7,    5,    7,    9,    6,    3,    6,    3,    1,    9,    8,    7,    3,    4,    9,    8,    2,    6    },    // 2^-495
    {    -148,    1,    9,    5,    5,    1,    5,    9,    2,    7,    2,    6,    3,    9,    7,    4,    6,    9,    9,    6,    5    },    // 2^-494
    {    -148,    3,    9,    1,    0,    3,    1,    8,    5,    4,    5,    2,    7,    9,    4,    9,    3,    9,    9,    3,    0    },    // 2^-493
    {    -148,    7,    8,    2,    0,    6,    3,    7,    0,    9,    0,    5,    5,    8,    9,    8,    7,    9,    8,    6,    1    },    // 2^-492
    {    -147,    1,    5,    6,    4,    1,    2,    7,    4,    1,    8,    1,    1,    1,    7,    9,    7,    5,    9,    7,    2    },    // 2^-491
    {    -147,    3,    1,    2,    8,    2,    5,    4,    8,    3,    6,    2,    2,    3,    5,    9,    5,    1,    9,    4,    4    },    // 2^-490
    {    -147,    6,    2,    5,    6,    5,    0,    9,    6,    7,    2,    4,    4,    7,    1,    9,    0,    3,    8,    8,    8    },    // 2^-489
    {    -146,    1,    2,    5,    1,    3,    0,    1,    9,    3,    4,    4,    8,    9,    4,    3,    8,    0,    7,    7,    8    },    // 2^-488
    {    -146,    2,    5,    0,    2,    6,    0,    3,    8,    6,    8,    9,    7,    8,    8,    7,    6,    1,    5,    5,    5    },    // 2^-487
    {    -146,    5,    0,    0,    5,    2,    0,    7,    7,    3,    7,    9,    5,    7,    7,    5,    2,    3,    1,    1,    1    },    // 2^-486
    {    -145,    1,    0,    0,    1,    0,    4,    1,    5,    4,    7,    5,    9,    1,    5,    5,    0,    4,    6,    2,    2    },    // 2^-485
    {    -145,    2,    0,    0,    2,    0,    8,    3,    0,    9,    5,    1,    8,    3,    1,    0,    0,    9,    2,    4,    4    },    // 2^-484
    {    -145,    4,    0,    0,    4,    1,    6,    6,    1,    9,    0,    3,    6,    6,    2,    0,    1,    8,    4,    8,    9    },    // 2^-483
    {    -145,    8,    0,    0,    8,    3,    3,    2,    3,    8,    0,    7,    3,    2,    4,    0,    3,    6,    9,    7,    7    },    // 2^-482
    {    -144,    1,    6,    0,    1,    6,    6,    6,    4,    7,    6,    1,    4,    6,    4,    8,    0,    7,    3,    9,    5    },    // 2^-481
    {    -144,    3,    2,    0,    3,    3,    3,    2,    9,    5,    2,    2,    9,    2,    9,    6,    1,    4,    7,    9,    1    },    // 2^-480
    {    -144,    6,    4,    0,    6,    6,    6,    5,    9,    0,    4,    5,    8,    5,    9,    2,    2,    9,    5,    8,    2    },    // 2^-479
    {    -143,    1,    2,    8,    1,    3,    3,    3,    1,    8,    0,    9,    1,    7,    1,    8,    4,    5,    9,    1,    6    },    // 2^-478
    {    -143,    2,    5,    6,    2,    6,    6,    6,    3,    6,    1,    8,    3,    4,    3,    6,    9,    1,    8,    3,    3    },    // 2^-477
    {    -143,    5,    1,    2,    5,    3,    3,    2,    7,    2,    3,    6,    6,    8,    7,    3,    8,    3,    6,    6,    5    },    // 2^-476
    {    -142,    1,    0,    2,    5,    0,    6,    6,    5,    4,    4,    7,    3,    3,    7,    4,    7,    6,    7,    3,    3    },    // 2^-475
    {    -142,    2,    0,    5,    0,    1,    3,    3,    0,    8,    9,    4,    6,    7,    4,    9,    5,    3,    4,    6,    6    },    // 2^-474
    {    -142,    4,    1,    0,    0,    2,    6,    6,    1,    7,    8,    9,    3,    4,    9,    9,    0,    6,    9,    3,    2    },    // 2^-473
    {    -142,    8,    2,    0,    0,    5,    3,    2,    3,    5,    7,    8,    6,    9,    9,    8,    1,    3,    8,    6,    5    },    // 2^-472
    {    -141,    1,    6,    4,    0,    1,    0,    6,    4,    7,    1,    5,    7,    3,    9,    9,    6,    2,    7,    7,    3    },    // 2^-471
    {    -141,    3,    2,    8,    0,    2,    1,    2,    9,    4,    3,    1,    4,    7,    9,    9,    2,    5,    5,    4,    6    },    // 2^-470
    {    -141,    6,    5,    6,    0,    4,    2,    5,    8,    8,    6,    2,    9,    5,    9,    8,    5,    1,    0,    9,    2    },    // 2^-469
    {    -140,    1,    3,    1,    2,    0,    8,    5,    1,    7,    7,    2,    5,    9,    1,    9,    7,    0,    2,    1,    8    },    // 2^-468
    {    -140,    2,    6,    2,    4,    1,    7,    0,    3,    5,    4,    5,    1,    8,    3,    9,    4,    0,    4,    3,    7    },    // 2^-467
    {    -140,    5,    2,    4,    8,    3,    4,    0,    7,    0,    9,    0,    3,    6,    7,    8,    8,    0,    8,    7,    3    },    // 2^-466
    {    -139,    1,    0,    4,    9,    6,    6,    8,    1,    4,    1,    8,    0,    7,    3,    5,    7,    6,    1,    7,    5    },    // 2^-465
    {    -139,    2,    0,    9,    9,    3,    3,    6,    2,    8,    3,    6,    1,    4,    7,    1,    5,    2,    3,    4,    9    },    // 2^-464
    {    -139,    4,    1,    9,    8,    6,    7,    2,    5,    6,    7,    2,    2,    9,    4,    3,    0,    4,    6,    9,    9    },    // 2^-463
    {    -139,    8,    3,    9,    7,    3,    4,    5,    1,    3,    4,    4,    5,    8,    8,    6,    0,    9,    3,    9,    7    },    // 2^-462
    {    -138,    1,    6,    7,    9,    4,    6,    9,    0,    2,    6,    8,    9,    1,    7,    7,    2,    1,    8,    7,    9    },    // 2^-461
    {    -138,    3,    3,    5,    8,    9,    3,    8,    0,    5,    3,    7,    8,    3,    5,    4,    4,    3,    7,    5,    9    },    // 2^-460
    {    -138,    6,    7,    1,    7,    8,    7,    6,    1,    0,    7,    5,    6,    7,    0,    8,    8,    7,    5,    1,    8    },    // 2^-459
    {    -137,    1,    3,    4,    3,    5,    7,    5,    2,    2,    1,    5,    1,    3,    4,    1,    7,    7,    5,    0,    4    },    // 2^-458
    {    -137,    2,    6,    8,    7,    1,    5,    0,    4,    4,    3,    0,    2,    6,    8,    3,    5,    5,    0,    0,    7    },    // 2^-457
    {    -137,    5,    3,    7,    4,    3,    0,    0,    8,    8,    6,    0,    5,    3,    6,    7,    1,    0,    0,    1,    4    },    // 2^-456
    {    -136,    1,    0,    7,    4,    8,    6,    0,    1,    7,    7,    2,    1,    0,    7,    3,    4,    2,    0,    0,    3    },    // 2^-455
    {    -136,    2,    1,    4,    9,    7,    2,    0,    3,    5,    4,    4,    2,    1,    4,    6,    8,    4,    0,    0,    6    },    // 2^-454
    {    -136,    4,    2,    9,    9,    4,    4,    0,    7,    0,    8,    8,    4,    2,    9,    3,    6,    8,    0,    1,    1    },    // 2^-453
    {    -136,    8,    5,    9,    8,    8,    8,    1,    4,    1,    7,    6,    8,    5,    8,    7,    3,    6,    0,    2,    3    },    // 2^-452
    {    -135,    1,    7,    1,    9,    7,    7,    6,    2,    8,    3,    5,    3,    7,    1,    7,    4,    7,    2,    0,    5    },    // 2^-451
    {    -135,    3,    4,    3,    9,    5,    5,    2,    5,    6,    7,    0,    7,    4,    3,    4,    9,    4,    4,    0,    9    },    // 2^-450
    {    -135,    6,    8,    7,    9,    1,    0,    5,    1,    3,    4,    1,    4,    8,    6,    9,    8,    8,    8,    1,    8    },    // 2^-449
    {    -134,    1,    3,    7,    5,    8,    2,    1,    0,    2,    6,    8,    2,    9,    7,    3,    9,    7,    7,    6,    4    },    // 2^-448
    {    -134,    2,    7,    5,    1,    6,    4,    2,    0,    5,    3,    6,    5,    9,    4,    7,    9,    5,    5,    2,    7    },    // 2^-447
    {    -134,    5,    5,    0,    3,    2,    8,    4,    1,    0,    7,    3,    1,    8,    9,    5,    9,    1,    0,    5,    5    },    // 2^-446
    {    -133,    1,    1,    0,    0,    6,    5,    6,    8,    2,    1,    4,    6,    3,    7,    9,    1,    8,    2,    1,    1    },    // 2^-445
    {    -133,    2,    2,    0,    1,    3,    1,    3,    6,    4,    2,    9,    2,    7,    5,    8,    3,    6,    4,    2,    2    },    // 2^-444
    {    -133,    4,    4,    0,    2,    6,    2,    7,    2,    8,    5,    8,    5,    5,    1,    6,    7,    2,    8,    4,    4    },    // 2^-443
    {    -133,    8,    8,    0,    5,    2,    5,    4,    5,    7,    1,    7,    1,    0,    3,    3,    4,    5,    6,    8,    7    },    // 2^-442
    {    -132,    1,    7,    6,    1,    0,    5,    0,    9,    1,    4,    3,    4,    2,    0,    6,    6,    9,    1,    3,    7    },    // 2^-441
    {    -132,    3,    5,    2,    2,    1,    0,    1,    8,    2,    8,    6,    8,    4,    1,    3,    3,    8,    2,    7,    5    },    // 2^-440
    {    -132,    7,    0,    4,    4,    2,    0,    3,    6,    5,    7,    3,    6,    8,    2,    6,    7,    6,    5,    5,    0    },    // 2^-439
    {    -131,    1,    4,    0,    8,    8,    4,    0,    7,    3,    1,    4,    7,    3,    6,    5,    3,    5,    3,    1,    0    },    // 2^-438
    {    -131,    2,    8,    1,    7,    6,    8,    1,    4,    6,    2,    9,    4,    7,    3,    0,    7,    0,    6,    2,    0    },    // 2^-437
    {    -131,    5,    6,    3,    5,    3,    6,    2,    9,    2,    5,    8,    9,    4,    6,    1,    4,    1,    2,    4,    0    },    // 2^-436
    {    -130,    1,    1,    2,    7,    0,    7,    2,    5,    8,    5,    1,    7,    8,    9,    2,    2,    8,    2,    4,    8    },    // 2^-435
    {    -130,    2,    2,    5,    4,    1,    4,    5,    1,    7,    0,    3,    5,    7,    8,    4,    5,    6,    4,    9,    6    },    // 2^-434
    {    -130,    4,    5,    0,    8,    2,    9,    0,    3,    4,    0,    7,    1,    5,    6,    9,    1,    2,    9,    9,    2    },    // 2^-433
    {    -130,    9,    0,    1,    6,    5,    8,    0,    6,    8,    1,    4,    3,    1,    3,    8,    2,    5,    9,    8,    4    },    // 2^-432
    {    -129,    1,    8,    0,    3,    3,    1,    6,    1,    3,    6,    2,    8,    6,    2,    7,    6,    5,    1,    9,    7    },    // 2^-431
    {    -129,    3,    6,    0,    6,    6,    3,    2,    2,    7,    2,    5,    7,    2,    5,    5,    3,    0,    3,    9,    4    },    // 2^-430
    {    -129,    7,    2,    1,    3,    2,    6,    4,    5,    4,    5,    1,    4,    5,    1,    0,    6,    0,    7,    8,    7    },    // 2^-429
    {    -128,    1,    4,    4,    2,    6,    5,    2,    9,    0,    9,    0,    2,    9,    0,    2,    1,    2,    1,    5,    7    },    // 2^-428
    {    -128,    2,    8,    8,    5,    3,    0,    5,    8,    1,    8,    0,    5,    8,    0,    4,    2,    4,    3,    1,    5    },    // 2^-427
    {    -128,    5,    7,    7,    0,    6,    1,    1,    6,    3,    6,    1,    1,    6,    0,    8,    4,    8,    6,    3,    0    },    // 2^-426
    {    -127,    1,    1,    5,    4,    1,    2,    2,    3,    2,    7,    2,    2,    3,    2,    1,    6,    9,    7,    2,    6    },    // 2^-425
    {    -127,    2,    3,    0,    8,    2,    4,    4,    6,    5,    4,    4,    4,    6,    4,    3,    3,    9,    4,    5,    2    },    // 2^-424
    {    -127,    4,    6,    1,    6,    4,    8,    9,    3,    0,    8,    8,    9,    2,    8,    6,    7,    8,    9,    0,    4    },    // 2^-423
    {    -127,    9,    2,    3,    2,    9,    7,    8,    6,    1,    7,    7,    8,    5,    7,    3,    5,    7,    8,    0,    8    },    // 2^-422
    {    -126,    1,    8,    4,    6,    5,    9,    5,    7,    2,    3,    5,    5,    7,    1,    4,    7,    1,    5,    6,    2    },    // 2^-421
    {    -126,    3,    6,    9,    3,    1,    9,    1,    4,    4,    7,    1,    1,    4,    2,    9,    4,    3,    1,    2,    3    },    // 2^-420
    {    -126,    7,    3,    8,    6,    3,    8,    2,    8,    9,    4,    2,    2,    8,    5,    8,    8,    6,    2,    4,    6    },    // 2^-419
    {    -125,    1,    4,    7,    7,    2,    7,    6,    5,    7,    8,    8,    4,    5,    7,    1,    7,    7,    2,    4,    9    },    // 2^-418
    {    -125,    2,    9,    5,    4,    5,    5,    3,    1,    5,    7,    6,    9,    1,    4,    3,    5,    4,    4,    9,    8    },    // 2^-417
    {    -125,    5,    9,    0,    9,    1,    0,    6,    3,    1,    5,    3,    8,    2,    8,    7,    0,    8,    9,    9,    7    },    // 2^-416
    {    -124,    1,    1,    8,    1,    8,    2,    1,    2,    6,    3,    0,    7,    6,    5,    7,    4,    1,    7,    9,    9    },    // 2^-415
    {    -124,    2,    3,    6,    3,    6,    4,    2,    5,    2,    6,    1,    5,    3,    1,    4,    8,    3,    5,    9,    9    },    // 2^-414
    {    -124,    4,    7,    2,    7,    2,    8,    5,    0,    5,    2,    3,    0,    6,    2,    9,    6,    7,    1,    9,    7    },    // 2^-413
    {    -124,    9,    4,    5,    4,    5,    7,    0,    1,    0,    4,    6,    1,    2,    5,    9,    3,    4,    3,    9,    5    },    // 2^-412
    {    -123,    1,    8,    9,    0,    9,    1,    4,    0,    2,    0,    9,    2,    2,    5,    1,    8,    6,    8,    7,    9    },    // 2^-411
    {    -123,    3,    7,    8,    1,    8,    2,    8,    0,    4,    1,    8,    4,    5,    0,    3,    7,    3,    7,    5,    8    },    // 2^-410
    {    -123,    7,    5,    6,    3,    6,    5,    6,    0,    8,    3,    6,    9,    0,    0,    7,    4,    7,    5,    1,    6    },    // 2^-409
    {    -122,    1,    5,    1,    2,    7,    3,    1,    2,    1,    6,    7,    3,    8,    0,    1,    4,    9,    5,    0,    3    },    // 2^-408
    {    -122,    3,    0,    2,    5,    4,    6,    2,    4,    3,    3,    4,    7,    6,    0,    2,    9,    9,    0,    0,    6    },    // 2^-407
    {    -122,    6,    0,    5,    0,    9,    2,    4,    8,    6,    6,    9,    5,    2,    0,    5,    9,    8,    0,    1,    3    },    // 2^-406
    {    -121,    1,    2,    1,    0,    1,    8,    4,    9,    7,    3,    3,    9,    0,    4,    1,    1,    9,    6,    0,    3    },    // 2^-405
    {    -121,    2,    4,    2,    0,    3,    6,    9,    9,    4,    6,    7,    8,    0,    8,    2,    3,    9,    2,    0,    5    },    // 2^-404
    {    -121,    4,    8,    4,    0,    7,    3,    9,    8,    9,    3,    5,    6,    1,    6,    4,    7,    8,    4,    1,    0    },    // 2^-403
    {    -121,    9,    6,    8,    1,    4,    7,    9,    7,    8,    7,    1,    2,    3,    2,    9,    5,    6,    8,    2,    0    },    // 2^-402
    {    -120,    1,    9,    3,    6,    2,    9,    5,    9,    5,    7,    4,    2,    4,    6,    5,    9,    1,    3,    6,    4    },    // 2^-401
    {    -120,    3,    8,    7,    2,    5,    9,    1,    9,    1,    4,    8,    4,    9,    3,    1,    8,    2,    7,    2,    8    },    // 2^-400
    {    -120,    7,    7,    4,    5,    1,    8,    3,    8,    2,    9,    6,    9,    8,    6,    3,    6,    5,    4,    5,    6    },    // 2^-399
    {    -119,    1,    5,    4,    9,    0,    3,    6,    7,    6,    5,    9,    3,    9,    7,    2,    7,    3,    0,    9,    1    },    // 2^-398
    {    -119,    3,    0,    9,    8,    0,    7,    3,    5,    3,    1,    8,    7,    9,    4,    5,    4,    6,    1,    8,    3    },    // 2^-397
    {    -119,    6,    1,    9,    6,    1,    4,    7,    0,    6,    3,    7,    5,    8,    9,    0,    9,    2,    3,    6,    5    },    // 2^-396
    {    -118,    1,    2,    3,    9,    2,    2,    9,    4,    1,    2,    7,    5,    1,    7,    8,    1,    8,    4,    7,    3    },    // 2^-395
    {    -118,    2,    4,    7,    8,    4,    5,    8,    8,    2,    5,    5,    0,    3,    5,    6,    3,    6,    9,    4,    6    },    // 2^-394
    {    -118,    4,    9,    5,    6,    9,    1,    7,    6,    5,    1,    0,    0,    7,    1,    2,    7,    3,    8,    9,    2    },    // 2^-393
    {    -118,    9,    9,    1,    3,    8,    3,    5,    3,    0,    2,    0,    1,    4,    2,    5,    4,    7,    7,    8,    4    },    // 2^-392
    {    -117,    1,    9,    8,    2,    7,    6,    7,    0,    6,    0,    4,    0,    2,    8,    5,    0,    9,    5,    5,    7    },    // 2^-391
    {    -117,    3,    9,    6,    5,    5,    3,    4,    1,    2,    0,    8,    0,    5,    7,    0,    1,    9,    1,    1,    4    },    // 2^-390
    {    -117,    7,    9,    3,    1,    0,    6,    8,    2,    4,    1,    6,    1,    1,    4,    0,    3,    8,    2,    2,    7    },    // 2^-389
    {    -116,    1,    5,    8,    6,    2,    1,    3,    6,    4,    8,    3,    2,    2,    2,    8,    0,    7,    6,    4,    5    },    // 2^-388
    {    -116,    3,    1,    7,    2,    4,    2,    7,    2,    9,    6,    6,    4,    4,    5,    6,    1,    5,    2,    9,    1    },    // 2^-387
    {    -116,    6,    3,    4,    4,    8,    5,    4,    5,    9,    3,    2,    8,    9,    1,    2,    3,    0,    5,    8,    2    },    // 2^-386
    {    -115,    1,    2,    6,    8,    9,    7,    0,    9,    1,    8,    6,    5,    7,    8,    2,    4,    6,    1,    1,    6    },    // 2^-385
    {    -115,    2,    5,    3,    7,    9,    4,    1,    8,    3,    7,    3,    1,    5,    6,    4,    9,    2,    2,    3,    3    },    // 2^-384
    {    -115,    5,    0,    7,    5,    8,    8,    3,    6,    7,    4,    6,    3,    1,    2,    9,    8,    4,    4,    6,    5    },    // 2^-383
    {    -114,    1,    0,    1,    5,    1,    7,    6,    7,    3,    4,    9,    2,    6,    2,    5,    9,    6,    8,    9,    3    },    // 2^-382
    {    -114,    2,    0,    3,    0,    3,    5,    3,    4,    6,    9,    8,    5,    2,    5,    1,    9,    3,    7,    8,    6    },    // 2^-381
    {    -114,    4,    0,    6,    0,    7,    0,    6,    9,    3,    9,    7,    0,    5,    0,    3,    8,    7,    5,    7,    2    },    // 2^-380
    {    -114,    8,    1,    2,    1,    4,    1,    3,    8,    7,    9,    4,    1,    0,    0,    7,    7,    5,    1,    4,    5    },    // 2^-379
    {    -113,    1,    6,    2,    4,    2,    8,    2,    7,    7,    5,    8,    8,    2,    0,    1,    5,    5,    0,    2,    9    },    // 2^-378
    {    -113,    3,    2,    4,    8,    5,    6,    5,    5,    5,    1,    7,    6,    4,    0,    3,    1,    0,    0,    5,    8    },    // 2^-377
    {    -113,    6,    4,    9,    7,    1,    3,    1,    1,    0,    3,    5,    2,    8,    0,    6,    2,    0,    1,    1,    6    },    // 2^-376
    {    -112,    1,    2,    9,    9,    4,    2,    6,    2,    2,    0,    7,    0,    5,    6,    1,    2,    4,    0,    2,    3    },    // 2^-375
    {    -112,    2,    5,    9,    8,    8,    5,    2,    4,    4,    1,    4,    1,    1,    2,    2,    4,    8,    0,    4,    6    },    // 2^-374
    {    -112,    5,    1,    9,    7,    7,    0,    4,    8,    8,    2,    8,    2,    2,    4,    4,    9,    6,    0,    9,    3    },    // 2^-373
    {    -111,    1,    0,    3,    9,    5,    4,    0,    9,    7,    6,    5,    6,    4,    4,    8,    9,    9,    2,    1,    9    },    // 2^-372
    {    -111,    2,    0,    7,    9,    0,    8,    1,    9,    5,    3,    1,    2,    8,    9,    7,    9,    8,    4,    3,    7    },    // 2^-371
    {    -111,    4,    1,    5,    8,    1,    6,    3,    9,    0,    6,    2,    5,    7,    9,    5,    9,    6,    8,    7,    4    },    // 2^-370
    {    -111,    8,    3,    1,    6,    3,    2,    7,    8,    1,    2,    5,    1,    5,    9,    1,    9,    3,    7,    4,    8    },    // 2^-369
    {    -110,    1,    6,    6,    3,    2,    6,    5,    5,    6,    2,    5,    0,    3,    1,    8,    3,    8,    7,    5,    0    },    // 2^-368
    {    -110,    3,    3,    2,    6,    5,    3,    1,    1,    2,    5,    0,    0,    6,    3,    6,    7,    7,    4,    9,    9    },    // 2^-367
    {    -110,    6,    6,    5,    3,    0,    6,    2,    2,    5,    0,    0,    1,    2,    7,    3,    5,    4,    9,    9,    9    },    // 2^-366
    {    -109,    1,    3,    3,    0,    6,    1,    2,    4,    5,    0,    0,    0,    2,    5,    4,    7,    0,    9,    0,    0    },    // 2^-365
    {    -109,    2,    6,    6,    1,    2,    2,    4,    9,    0,    0,    0,    0,    5,    0,    9,    4,    1,    9,    9,    9    },    // 2^-364
    {    -109,    5,    3,    2,    2,    4,    4,    9,    8,    0,    0,    0,    1,    0,    1,    8,    8,    3,    9,    9,    9    },    // 2^-363
    {    -108,    1,    0,    6,    4,    4,    8,    9,    9,    6,    0,    0,    0,    2,    0,    3,    7,    6,    7,    0,    0    },    // 2^-362
    {    -108,    2,    1,    2,    8,    9,    7,    9,    9,    2,    0,    0,    0,    4,    0,    7,    5,    3,    5,    0,    0    },    // 2^-361
    {    -108,    4,    2,    5,    7,    9,    5,    9,    8,    4,    0,    0,    0,    8,    1,    5,    0,    7,    1,    9,    9    },    // 2^-360
    {    -108,    8,    5,    1,    5,    9,    1,    9,    6,    8,    0,    0,    1,    6,    3,    0,    1,    4,    3,    9,    8    },    // 2^-359
    {    -107,    1,    7,    0,    3,    1,    8,    3,    9,    3,    6,    0,    0,    3,    2,    6,    0,    2,    8,    8,    0    },    // 2^-358
    {    -107,    3,    4,    0,    6,    3,    6,    7,    8,    7,    2,    0,    0,    6,    5,    2,    0,    5,    7,    5,    9    },    // 2^-357
    {    -107,    6,    8,    1,    2,    7,    3,    5,    7,    4,    4,    0,    1,    3,    0,    4,    1,    1,    5,    1,    9    },    // 2^-356
    {    -106,    1,    3,    6,    2,    5,    4,    7,    1,    4,    8,    8,    0,    2,    6,    0,    8,    2,    3,    0,    4    },    // 2^-355
    {    -106,    2,    7,    2,    5,    0,    9,    4,    2,    9,    7,    6,    0,    5,    2,    1,    6,    4,    6,    0,    7    },    // 2^-354
    {    -106,    5,    4,    5,    0,    1,    8,    8,    5,    9,    5,    2,    1,    0,    4,    3,    2,    9,    2,    1,    5    },    // 2^-353
    {    -105,    1,    0,    9,    0,    0,    3,    7,    7,    1,    9,    0,    4,    2,    0,    8,    6,    5,    8,    4,    3    },    // 2^-352
    {    -105,    2,    1,    8,    0,    0,    7,    5,    4,    3,    8,    0,    8,    4,    1,    7,    3,    1,    6,    8,    6    },    // 2^-351
    {    -105,    4,    3,    6,    0,    1,    5,    0,    8,    7,    6,    1,    6,    8,    3,    4,    6,    3,    3,    7,    2    },    // 2^-350
    {    -105,    8,    7,    2,    0,    3,    0,    1,    7,    5,    2,    3,    3,    6,    6,    9,    2,    6,    7,    4,    4    },    // 2^-349
    {    -104,    1,    7,    4,    4,    0,    6,    0,    3,    5,    0,    4,    6,    7,    3,    3,    8,    5,    3,    4,    9    },    // 2^-348
    {    -104,    3,    4,    8,    8,    1,    2,    0,    7,    0,    0,    9,    3,    4,    6,    7,    7,    0,    6,    9,    8    },    // 2^-347
    {    -104,    6,    9,    7,    6,    2,    4,    1,    4,    0,    1,    8,    6,    9,    3,    5,    4,    1,    3,    9,    5    },    // 2^-346
    {    -103,    1,    3,    9,    5,    2,    4,    8,    2,    8,    0,    3,    7,    3,    8,    7,    0,    8,    2,    7,    9    },    // 2^-345
    {    -103,    2,    7,    9,    0,    4,    9,    6,    5,    6,    0,    7,    4,    7,    7,    4,    1,    6,    5,    5,    8    },    // 2^-344
    {    -103,    5,    5,    8,    0,    9,    9,    3,    1,    2,    1,    4,    9,    5,    4,    8,    3,    3,    1,    1,    6    },    // 2^-343
    {    -102,    1,    1,    1,    6,    1,    9,    8,    6,    2,    4,    2,    9,    9,    0,    9,    6,    6,    6,    2,    3    },    // 2^-342
    {    -102,    2,    2,    3,    2,    3,    9,    7,    2,    4,    8,    5,    9,    8,    1,    9,    3,    3,    2,    4,    6    },    // 2^-341
    {    -102,    4,    4,    6,    4,    7,    9,    4,    4,    9,    7,    1,    9,    6,    3,    8,    6,    6,    4,    9,    3    },    // 2^-340
    {    -102,    8,    9,    2,    9,    5,    8,    8,    9,    9,    4,    3,    9,    2,    7,    7,    3,    2,    9,    8,    6    },    // 2^-339
    {    -101,    1,    7,    8,    5,    9,    1,    7,    7,    9,    8,    8,    7,    8,    5,    5,    4,    6,    5,    9,    7    },    // 2^-338
    {    -101,    3,    5,    7,    1,    8,    3,    5,    5,    9,    7,    7,    5,    7,    1,    0,    9,    3,    1,    9,    4    },    // 2^-337
    {    -101,    7,    1,    4,    3,    6,    7,    1,    1,    9,    5,    5,    1,    4,    2,    1,    8,    6,    3,    8,    8    },    // 2^-336
    {    -100,    1,    4,    2,    8,    7,    3,    4,    2,    3,    9,    1,    0,    2,    8,    4,    3,    7,    2,    7,    8    },    // 2^-335
    {    -100,    2,    8,    5,    7,    4,    6,    8,    4,    7,    8,    2,    0,    5,    6,    8,    7,    4,    5,    5,    5    },    // 2^-334
    {    -100,    5,    7,    1,    4,    9,    3,    6,    9,    5,    6,    4,    1,    1,    3,    7,    4,    9,    1,    1,    1    },    // 2^-333
    {    -99,    1,    1,    4,    2,    9,    8,    7,    3,    9,    1,    2,    8,    2,    2,    7,    4,    9,    8,    2,    2    },    // 2^-332
    {    -99,    2,    2,    8,    5,    9,    7,    4,    7,    8,    2,    5,    6,    4,    5,    4,    9,    9,    6,    4,    4    },    // 2^-331
    {    -99,    4,    5,    7,    1,    9,    4,    9,    5,    6,    5,    1,    2,    9,    0,    9,    9,    9,    2,    8,    9    },    // 2^-330
    {    -99,    9,    1,    4,    3,    8,    9,    9,    1,    3,    0,    2,    5,    8,    1,    9,    9,    8,    5,    7,    7    },    // 2^-329
    {    -98,    1,    8,    2,    8,    7,    7,    9,    8,    2,    6,    0,    5,    1,    6,    3,    9,    9,    7,    1,    5    },    // 2^-328
    {    -98,    3,    6,    5,    7,    5,    5,    9,    6,    5,    2,    1,    0,    3,    2,    7,    9,    9,    4,    3,    1    },    // 2^-327
    {    -98,    7,    3,    1,    5,    1,    1,    9,    3,    0,    4,    2,    0,    6,    5,    5,    9,    8,    8,    6,    2    },    // 2^-326
    {    -97,    1,    4,    6,    3,    0,    2,    3,    8,    6,    0,    8,    4,    1,    3,    1,    1,    9,    7,    7,    2    },    // 2^-325
    {    -97,    2,    9,    2,    6,    0,    4,    7,    7,    2,    1,    6,    8,    2,    6,    2,    3,    9,    5,    4,    5    },    // 2^-324
    {    -97,    5,    8,    5,    2,    0,    9,    5,    4,    4,    3,    3,    6,    5,    2,    4,    7,    9,    0,    8,    9    },    // 2^-323
    {    -96,    1,    1,    7,    0,    4,    1,    9,    0,    8,    8,    6,    7,    3,    0,    4,    9,    5,    8,    1,    8    },    // 2^-322
    {    -96,    2,    3,    4,    0,    8,    3,    8,    1,    7,    7,    3,    4,    6,    0,    9,    9,    1,    6,    3,    6    },    // 2^-321
    {    -96,    4,    6,    8,    1,    6,    7,    6,    3,    5,    4,    6,    9,    2,    1,    9,    8,    3,    2,    7,    2    },    // 2^-320
    {    -96,    9,    3,    6,    3,    3,    5,    2,    7,    0,    9,    3,    8,    4,    3,    9,    6,    6,    5,    4,    3    },    // 2^-319
    {    -95,    1,    8,    7,    2,    6,    7,    0,    5,    4,    1,    8,    7,    6,    8,    7,    9,    3,    3,    0,    9    },    // 2^-318
    {    -95,    3,    7,    4,    5,    3,    4,    1,    0,    8,    3,    7,    5,    3,    7,    5,    8,    6,    6,    1,    7    },    // 2^-317
    {    -95,    7,    4,    9,    0,    6,    8,    2,    1,    6,    7,    5,    0,    7,    5,    1,    7,    3,    2,    3,    4    },    // 2^-316
    {    -94,    1,    4,    9,    8,    1,    3,    6,    4,    3,    3,    5,    0,    1,    5,    0,    3,    4,    6,    4,    7    },    // 2^-315
    {    -94,    2,    9,    9,    6,    2,    7,    2,    8,    6,    7,    0,    0,    3,    0,    0,    6,    9,    2,    9,    4    },    // 2^-314
    {    -94,    5,    9,    9,    2,    5,    4,    5,    7,    3,    4,    0,    0,    6,    0,    1,    3,    8,    5,    8,    8    },    // 2^-313
    {    -93,    1,    1,    9,    8,    5,    0,    9,    1,    4,    6,    8,    0,    1,    2,    0,    2,    7,    7,    1,    8    },    // 2^-312
    {    -93,    2,    3,    9,    7,    0,    1,    8,    2,    9,    3,    6,    0,    2,    4,    0,    5,    5,    4,    3,    5    },    // 2^-311
    {    -93,    4,    7,    9,    4,    0,    3,    6,    5,    8,    7,    2,    0,    4,    8,    1,    1,    0,    8,    7,    0    },    // 2^-310
    {    -93,    9,    5,    8,    8,    0,    7,    3,    1,    7,    4,    4,    0,    9,    6,    2,    2,    1,    7,    4,    0    },    // 2^-309
    {    -92,    1,    9,    1,    7,    6,    1,    4,    6,    3,    4,    8,    8,    1,    9,    2,    4,    4,    3,    4,    8    },    // 2^-308
    {    -92,    3,    8,    3,    5,    2,    2,    9,    2,    6,    9,    7,    6,    3,    8,    4,    8,    8,    6,    9,    6    },    // 2^-307
    {    -92,    7,    6,    7,    0,    4,    5,    8,    5,    3,    9,    5,    2,    7,    6,    9,    7,    7,    3,    9,    2    },    // 2^-306
    {    -91,    1,    5,    3,    4,    0,    9,    1,    7,    0,    7,    9,    0,    5,    5,    3,    9,    5,    4,    7,    8    },    // 2^-305
    {    -91,    3,    0,    6,    8,    1,    8,    3,    4,    1,    5,    8,    1,    1,    0,    7,    9,    0,    9,    5,    7    },    // 2^-304
    {    -91,    6,    1,    3,    6,    3,    6,    6,    8,    3,    1,    6,    2,    2,    1,    5,    8,    1,    9,    1,    4    },    // 2^-303
    {    -90,    1,    2,    2,    7,    2,    7,    3,    3,    6,    6,    3,    2,    4,    4,    3,    1,    6,    3,    8,    3    },    // 2^-302
    {    -90,    2,    4,    5,    4,    5,    4,    6,    7,    3,    2,    6,    4,    8,    8,    6,    3,    2,    7,    6,    5    },    // 2^-301
    {    -90,    4,    9,    0,    9,    0,    9,    3,    4,    6,    5,    2,    9,    7,    7,    2,    6,    5,    5,    3,    1    },    // 2^-300
    {    -90,    9,    8,    1,    8,    1,    8,    6,    9,    3,    0,    5,    9,    5,    4,    5,    3,    1,    0,    6,    2    },    // 2^-299
    {    -89,    1,    9,    6,    3,    6,    3,    7,    3,    8,    6,    1,    1,    9,    0,    9,    0,    6,    2,    1,    2    },    // 2^-298
    {    -89,    3,    9,    2,    7,    2,    7,    4,    7,    7,    2,    2,    3,    8,    1,    8,    1,    2,    4,    2,    5    },    // 2^-297
    {    -89,    7,    8,    5,    4,    5,    4,    9,    5,    4,    4,    4,    7,    6,    3,    6,    2,    4,    8,    5,    0    },    // 2^-296
    {    -88,    1,    5,    7,    0,    9,    0,    9,    9,    0,    8,    8,    9,    5,    2,    7,    2,    4,    9,    7,    0    },    // 2^-295
    {    -88,    3,    1,    4,    1,    8,    1,    9,    8,    1,    7,    7,    9,    0,    5,    4,    4,    9,    9,    4,    0    },    // 2^-294
    {    -88,    6,    2,    8,    3,    6,    3,    9,    6,    3,    5,    5,    8,    1,    0,    8,    9,    9,    8,    8,    0    },    // 2^-293
    {    -87,    1,    2,    5,    6,    7,    2,    7,    9,    2,    7,    1,    1,    6,    2,    1,    7,    9,    9,    7,    6    },    // 2^-292
    {    -87,    2,    5,    1,    3,    4,    5,    5,    8,    5,    4,    2,    3,    2,    4,    3,    5,    9,    9,    5,    2    },    // 2^-291
    {    -87,    5,    0,    2,    6,    9,    1,    1,    7,    0,    8,    4,    6,    4,    8,    7,    1,    9,    9,    0,    4    },    // 2^-290
    {    -86,    1,    0,    0,    5,    3,    8,    2,    3,    4,    1,    6,    9,    2,    9,    7,    4,    3,    9,    8,    1    },    // 2^-289
    {    -86,    2,    0,    1,    0,    7,    6,    4,    6,    8,    3,    3,    8,    5,    9,    4,    8,    7,    9,    6,    1    },    // 2^-288
    {    -86,    4,    0,    2,    1,    5,    2,    9,    3,    6,    6,    7,    7,    1,    8,    9,    7,    5,    9,    2,    3    },    // 2^-287
    {    -86,    8,    0,    4,    3,    0,    5,    8,    7,    3,    3,    5,    4,    3,    7,    9,    5,    1,    8,    4,    6    },    // 2^-286
    {    -85,    1,    6,    0,    8,    6,    1,    1,    7,    4,    6,    7,    0,    8,    7,    5,    9,    0,    3,    6,    9    },    // 2^-285
    {    -85,    3,    2,    1,    7,    2,    2,    3,    4,    9,    3,    4,    1,    7,    5,    1,    8,    0,    7,    3,    8    },    // 2^-284
    {    -85,    6,    4,    3,    4,    4,    4,    6,    9,    8,    6,    8,    3,    5,    0,    3,    6,    1,    4,    7,    7    },    // 2^-283
    {    -84,    1,    2,    8,    6,    8,    8,    9,    3,    9,    7,    3,    6,    7,    0,    0,    7,    2,    2,    9,    5    },    // 2^-282
    {    -84,    2,    5,    7,    3,    7,    7,    8,    7,    9,    4,    7,    3,    4,    0,    1,    4,    4,    5,    9,    1    },    // 2^-281
    {    -84,    5,    1,    4,    7,    5,    5,    7,    5,    8,    9,    4,    6,    8,    0,    2,    8,    9,    1,    8,    1    },    // 2^-280
    {    -83,    1,    0,    2,    9,    5,    1,    1,    5,    1,    7,    8,    9,    3,    6,    0,    5,    7,    8,    3,    6    },    // 2^-279
    {    -83,    2,    0,    5,    9,    0,    2,    3,    0,    3,    5,    7,    8,    7,    2,    1,    1,    5,    6,    7,    3    },    // 2^-278
    {    -83,    4,    1,    1,    8,    0,    4,    6,    0,    7,    1,    5,    7,    4,    4,    2,    3,    1,    3,    4,    5    },    // 2^-277
    {    -83,    8,    2,    3,    6,    0,    9,    2,    1,    4,    3,    1,    4,    8,    8,    4,    6,    2,    6,    9,    0    },    // 2^-276
    {    -82,    1,    6,    4,    7,    2,    1,    8,    4,    2,    8,    6,    2,    9,    7,    6,    9,    2,    5,    3,    8    },    // 2^-275
    {    -82,    3,    2,    9,    4,    4,    3,    6,    8,    5,    7,    2,    5,    9,    5,    3,    8,    5,    0,    7,    6    },    // 2^-274
    {    -82,    6,    5,    8,    8,    8,    7,    3,    7,    1,    4,    5,    1,    9,    0,    7,    7,    0,    1,    5,    2    },    // 2^-273
    {    -81,    1,    3,    1,    7,    7,    7,    4,    7,    4,    2,    9,    0,    3,    8,    1,    5,    4,    0,    3,    0    },    // 2^-272
    {    -81,    2,    6,    3,    5,    5,    4,    9,    4,    8,    5,    8,    0,    7,    6,    3,    0,    8,    0,    6,    1    },    // 2^-271
    {    -81,    5,    2,    7,    1,    0,    9,    8,    9,    7,    1,    6,    1,    5,    2,    6,    1,    6,    1,    2,    2    },    // 2^-270
    {    -80,    1,    0,    5,    4,    2,    1,    9,    7,    9,    4,    3,    2,    3,    0,    5,    2,    3,    2,    2,    4    },    // 2^-269
    {    -80,    2,    1,    0,    8,    4,    3,    9,    5,    8,    8,    6,    4,    6,    1,    0,    4,    6,    4,    4,    9    },    // 2^-268
    {    -80,    4,    2,    1,    6,    8,    7,    9,    1,    7,    7,    2,    9,    2,    2,    0,    9,    2,    8,    9,    7    },    // 2^-267
    {    -80,    8,    4,    3,    3,    7,    5,    8,    3,    5,    4,    5,    8,    4,    4,    1,    8,    5,    7,    9,    5    },    // 2^-266
    {    -79,    1,    6,    8,    6,    7,    5,    1,    6,    7,    0,    9,    1,    6,    8,    8,    3,    7,    1,    5,    9    },    // 2^-265
    {    -79,    3,    3,    7,    3,    5,    0,    3,    3,    4,    1,    8,    3,    3,    7,    6,    7,    4,    3,    1,    8    },    // 2^-264
    {    -79,    6,    7,    4,    7,    0,    0,    6,    6,    8,    3,    6,    6,    7,    5,    3,    4,    8,    6,    3,    6    },    // 2^-263
    {    -78,    1,    3,    4,    9,    4,    0,    1,    3,    3,    6,    7,    3,    3,    5,    0,    6,    9,    7,    2,    7    },    // 2^-262
    {    -78,    2,    6,    9,    8,    8,    0,    2,    6,    7,    3,    4,    6,    7,    0,    1,    3,    9,    4,    5,    4    },    // 2^-261
    {    -78,    5,    3,    9,    7,    6,    0,    5,    3,    4,    6,    9,    3,    4,    0,    2,    7,    8,    9,    0,    9    },    // 2^-260
    {    -77,    1,    0,    7,    9,    5,    2,    1,    0,    6,    9,    3,    8,    6,    8,    0,    5,    5,    7,    8,    2    },    // 2^-259
    {    -77,    2,    1,    5,    9,    0,    4,    2,    1,    3,    8,    7,    7,    3,    6,    1,    1,    1,    5,    6,    3    },    // 2^-258
    {    -77,    4,    3,    1,    8,    0,    8,    4,    2,    7,    7,    5,    4,    7,    2,    2,    2,    3,    1,    2,    7    },    // 2^-257
    {    -77,    8,    6,    3,    6,    1,    6,    8,    5,    5,    5,    0,    9,    4,    4,    4,    4,    6,    2,    5,    4    },    // 2^-256
    {    -76,    1,    7,    2,    7,    2,    3,    3,    7,    1,    1,    0,    1,    8,    8,    8,    8,    9,    2,    5,    1    },    // 2^-255
    {    -76,    3,    4,    5,    4,    4,    6,    7,    4,    2,    2,    0,    3,    7,    7,    7,    7,    8,    5,    0,    2    },    // 2^-254
    {    -76,    6,    9,    0,    8,    9,    3,    4,    8,    4,    4,    0,    7,    5,    5,    5,    5,    7,    0,    0,    3    },    // 2^-253
    {    -75,    1,    3,    8,    1,    7,    8,    6,    9,    6,    8,    8,    1,    5,    1,    1,    1,    1,    4,    0,    1    },    // 2^-252
    {    -75,    2,    7,    6,    3,    5,    7,    3,    9,    3,    7,    6,    3,    0,    2,    2,    2,    2,    8,    0,    1    },    // 2^-251
    {    -75,    5,    5,    2,    7,    1,    4,    7,    8,    7,    5,    2,    6,    0,    4,    4,    4,    5,    6,    0,    2    },    // 2^-250
    {    -74,    1,    1,    0,    5,    4,    2,    9,    5,    7,    5,    0,    5,    2,    0,    8,    8,    9,    1,    2,    0    },    // 2^-249
    {    -74,    2,    2,    1,    0,    8,    5,    9,    1,    5,    0,    1,    0,    4,    1,    7,    7,    8,    2,    4,    1    },    // 2^-248
    {    -74,    4,    4,    2,    1,    7,    1,    8,    3,    0,    0,    2,    0,    8,    3,    5,    5,    6,    4,    8,    2    },    // 2^-247
    {    -74,    8,    8,    4,    3,    4,    3,    6,    6,    0,    0,    4,    1,    6,    7,    1,    1,    2,    9,    6,    4    },    // 2^-246
    {    -73,    1,    7,    6,    8,    6,    8,    7,    3,    2,    0,    0,    8,    3,    3,    4,    2,    2,    5,    9,    3    },    // 2^-245
    {    -73,    3,    5,    3,    7,    3,    7,    4,    6,    4,    0,    1,    6,    6,    6,    8,    4,    5,    1,    8,    6    },    // 2^-244
    {    -73,    7,    0,    7,    4,    7,    4,    9,    2,    8,    0,    3,    3,    3,    3,    6,    9,    0,    3,    7,    1    },    // 2^-243
    {    -72,    1,    4,    1,    4,    9,    4,    9,    8,    5,    6,    0,    6,    6,    6,    7,    3,    8,    0,    7,    4    },    // 2^-242
    {    -72,    2,    8,    2,    9,    8,    9,    9,    7,    1,    2,    1,    3,    3,    3,    4,    7,    6,    1,    4,    8    },    // 2^-241
    {    -72,    5,    6,    5,    9,    7,    9,    9,    4,    2,    4,    2,    6,    6,    6,    9,    5,    2,    2,    9,    7    },    // 2^-240
    {    -71,    1,    1,    3,    1,    9,    5,    9,    8,    8,    4,    8,    5,    3,    3,    3,    9,    0,    4,    5,    9    },    // 2^-239
    {    -71,    2,    2,    6,    3,    9,    1,    9,    7,    6,    9,    7,    0,    6,    6,    7,    8,    0,    9,    1,    9    },    // 2^-238
    {    -71,    4,    5,    2,    7,    8,    3,    9,    5,    3,    9,    4,    1,    3,    3,    5,    6,    1,    8,    3,    8    },    // 2^-237
    {    -71,    9,    0,    5,    5,    6,    7,    9,    0,    7,    8,    8,    2,    6,    7,    1,    2,    3,    6,    7,    5    },    // 2^-236
    {    -70,    1,    8,    1,    1,    1,    3,    5,    8,    1,    5,    7,    6,    5,    3,    4,    2,    4,    7,    3,    5    },    // 2^-235
    {    -70,    3,    6,    2,    2,    2,    7,    1,    6,    3,    1,    5,    3,    0,    6,    8,    4,    9,    4,    7,    0    },    // 2^-234
    {    -70,    7,    2,    4,    4,    5,    4,    3,    2,    6,    3,    0,    6,    1,    3,    6,    9,    8,    9,    4,    0    },    // 2^-233
    {    -69,    1,    4,    4,    8,    9,    0,    8,    6,    5,    2,    6,    1,    2,    2,    7,    3,    9,    7,    8,    8    },    // 2^-232
    {    -69,    2,    8,    9,    7,    8,    1,    7,    3,    0,    5,    2,    2,    4,    5,    4,    7,    9,    5,    7,    6    },    // 2^-231
    {    -69,    5,    7,    9,    5,    6,    3,    4,    6,    1,    0,    4,    4,    9,    0,    9,    5,    9,    1,    5,    2    },    // 2^-230
    {    -68,    1,    1,    5,    9,    1,    2,    6,    9,    2,    2,    0,    8,    9,    8,    1,    9,    1,    8,    3,    0    },    // 2^-229
    {    -68,    2,    3,    1,    8,    2,    5,    3,    8,    4,    4,    1,    7,    9,    6,    3,    8,    3,    6,    6,    1    },    // 2^-228
    {    -68,    4,    6,    3,    6,    5,    0,    7,    6,    8,    8,    3,    5,    9,    2,    7,    6,    7,    3,    2,    2    },    // 2^-227
    {    -68,    9,    2,    7,    3,    0,    1,    5,    3,    7,    6,    7,    1,    8,    5,    5,    3,    4,    6,    4,    3    },    // 2^-226
    {    -67,    1,    8,    5,    4,    6,    0,    3,    0,    7,    5,    3,    4,    3,    7,    1,    0,    6,    9,    2,    9    },    // 2^-225
    {    -67,    3,    7,    0,    9,    2,    0,    6,    1,    5,    0,    6,    8,    7,    4,    2,    1,    3,    8,    5,    7    },    // 2^-224
    {    -67,    7,    4,    1,    8,    4,    1,    2,    3,    0,    1,    3,    7,    4,    8,    4,    2,    7,    7,    1,    5    },    // 2^-223
    {    -66,    1,    4,    8,    3,    6,    8,    2,    4,    6,    0,    2,    7,    4,    9,    6,    8,    5,    5,    4,    3    },    // 2^-222
    {    -66,    2,    9,    6,    7,    3,    6,    4,    9,    2,    0,    5,    4,    9,    9,    3,    7,    1,    0,    8,    6    },    // 2^-221
    {    -66,    5,    9,    3,    4,    7,    2,    9,    8,    4,    1,    0,    9,    9,    8,    7,    4,    2,    1,    7,    2    },    // 2^-220
    {    -65,    1,    1,    8,    6,    9,    4,    5,    9,    6,    8,    2,    1,    9,    9,    7,    4,    8,    4,    3,    4    },    // 2^-219
    {    -65,    2,    3,    7,    3,    8,    9,    1,    9,    3,    6,    4,    3,    9,    9,    4,    9,    6,    8,    6,    9    },    // 2^-218
    {    -65,    4,    7,    4,    7,    7,    8,    3,    8,    7,    2,    8,    7,    9,    8,    9,    9,    3,    7,    3,    7    },    // 2^-217
    {    -65,    9,    4,    9,    5,    5,    6,    7,    7,    4,    5,    7,    5,    9,    7,    9,    8,    7,    4,    7,    5    },    // 2^-216
    {    -64,    1,    8,    9,    9,    1,    1,    3,    5,    4,    9,    1,    5,    1,    9,    5,    9,    7,    4,    9,    5    },    // 2^-215
    {    -64,    3,    7,    9,    8,    2,    2,    7,    0,    9,    8,    3,    0,    3,    9,    1,    9,    4,    9,    9,    0    },    // 2^-214
    {    -64,    7,    5,    9,    6,    4,    5,    4,    1,    9,    6,    6,    0,    7,    8,    3,    8,    9,    9,    8,    0    },    // 2^-213
    {    -63,    1,    5,    1,    9,    2,    9,    0,    8,    3,    9,    3,    2,    1,    5,    6,    7,    7,    9,    9,    6    },    // 2^-212
    {    -63,    3,    0,    3,    8,    5,    8,    1,    6,    7,    8,    6,    4,    3,    1,    3,    5,    5,    9,    9,    2    },    // 2^-211
    {    -63,    6,    0,    7,    7,    1,    6,    3,    3,    5,    7,    2,    8,    6,    2,    7,    1,    1,    9,    8,    4    },    // 2^-210
    {    -62,    1,    2,    1,    5,    4,    3,    2,    6,    7,    1,    4,    5,    7,    2,    5,    4,    2,    3,    9,    7    },    // 2^-209
    {    -62,    2,    4,    3,    0,    8,    6,    5,    3,    4,    2,    9,    1,    4,    5,    0,    8,    4,    7,    9,    4    },    // 2^-208
    {    -62,    4,    8,    6,    1,    7,    3,    0,    6,    8,    5,    8,    2,    9,    0,    1,    6,    9,    5,    8,    7    },    // 2^-207
    {    -62,    9,    7,    2,    3,    4,    6,    1,    3,    7,    1,    6,    5,    8,    0,    3,    3,    9,    1,    7,    4    },    // 2^-206
    {    -61,    1,    9,    4,    4,    6,    9,    2,    2,    7,    4,    3,    3,    1,    6,    0,    6,    7,    8,    3,    5    },    // 2^-205
    {    -61,    3,    8,    8,    9,    3,    8,    4,    5,    4,    8,    6,    6,    3,    2,    1,    3,    5,    6,    7,    0    },    // 2^-204
    {    -61,    7,    7,    7,    8,    7,    6,    9,    0,    9,    7,    3,    2,    6,    4,    2,    7,    1,    3,    3,    9    },    // 2^-203
    {    -60,    1,    5,    5,    5,    7,    5,    3,    8,    1,    9,    4,    6,    5,    2,    8,    5,    4,    2,    6,    8    },    // 2^-202
    {    -60,    3,    1,    1,    1,    5,    0,    7,    6,    3,    8,    9,    3,    0,    5,    7,    0,    8,    5,    3,    6    },    // 2^-201
    {    -60,    6,    2,    2,    3,    0,    1,    5,    2,    7,    7,    8,    6,    1,    1,    4,    1,    7,    0,    7,    1    },    // 2^-200
    {    -59,    1,    2,    4,    4,    6,    0,    3,    0,    5,    5,    5,    7,    2,    2,    2,    8,    3,    4,    1,    4    },    // 2^-199
    {    -59,    2,    4,    8,    9,    2,    0,    6,    1,    1,    1,    1,    4,    4,    4,    5,    6,    6,    8,    2,    9    },    // 2^-198
    {    -59,    4,    9,    7,    8,    4,    1,    2,    2,    2,    2,    2,    8,    8,    9,    1,    3,    3,    6,    5,    7    },    // 2^-197
    {    -59,    9,    9,    5,    6,    8,    2,    4,    4,    4,    4,    5,    7,    7,    8,    2,    6,    7,    3,    1,    4    },    // 2^-196
    {    -58,    1,    9,    9,    1,    3,    6,    4,    8,    8,    8,    9,    1,    5,    5,    6,    5,    3,    4,    6,    3    },    // 2^-195
    {    -58,    3,    9,    8,    2,    7,    2,    9,    7,    7,    7,    8,    3,    1,    1,    3,    0,    6,    9,    2,    6    },    // 2^-194
    {    -58,    7,    9,    6,    5,    4,    5,    9,    5,    5,    5,    6,    6,    2,    2,    6,    1,    3,    8,    5,    1    },    // 2^-193
    {    -57,    1,    5,    9,    3,    0,    9,    1,    9,    1,    1,    1,    3,    2,    4,    5,    2,    2,    7,    7,    0    },    // 2^-192
    {    -57,    3,    1,    8,    6,    1,    8,    3,    8,    2,    2,    2,    6,    4,    9,    0,    4,    5,    5,    4,    1    },    // 2^-191
    {    -57,    6,    3,    7,    2,    3,    6,    7,    6,    4,    4,    5,    2,    9,    8,    0,    9,    1,    0,    8,    1    },    // 2^-190
    {    -56,    1,    2,    7,    4,    4,    7,    3,    5,    2,    8,    9,    0,    5,    9,    6,    1,    8,    2,    1,    6    },    // 2^-189
    {    -56,    2,    5,    4,    8,    9,    4,    7,    0,    5,    7,    8,    1,    1,    9,    2,    3,    6,    4,    3,    2    },    // 2^-188
    {    -56,    5,    0,    9,    7,    8,    9,    4,    1,    1,    5,    6,    2,    3,    8,    4,    7,    2,    8,    6,    5    },    // 2^-187
    {    -55,    1,    0,    1,    9,    5,    7,    8,    8,    2,    3,    1,    2,    4,    7,    6,    9,    4,    5,    7,    3    },    // 2^-186
    {    -55,    2,    0,    3,    9,    1,    5,    7,    6,    4,    6,    2,    4,    9,    5,    3,    8,    9,    1,    4,    6    },    // 2^-185
    {    -55,    4,    0,    7,    8,    3,    1,    5,    2,    9,    2,    4,    9,    9,    0,    7,    7,    8,    2,    9,    2    },    // 2^-184
    {    -55,    8,    1,    5,    6,    6,    3,    0,    5,    8,    4,    9,    9,    8,    1,    5,    5,    6,    5,    8,    4    },    // 2^-183
    {    -54,    1,    6,    3,    1,    3,    2,    6,    1,    1,    6,    9,    9,    9,    6,    3,    1,    1,    3,    1,    7    },    // 2^-182
    {    -54,    3,    2,    6,    2,    6,    5,    2,    2,    3,    3,    9,    9,    9,    2,    6,    2,    2,    6,    3,    4    },    // 2^-181
    {    -54,    6,    5,    2,    5,    3,    0,    4,    4,    6,    7,    9,    9,    8,    5,    2,    4,    5,    2,    6,    7    },    // 2^-180
    {    -53,    1,    3,    0,    5,    0,    6,    0,    8,    9,    3,    5,    9,    9,    7,    0,    4,    9,    0,    5,    3    },    // 2^-179
    {    -53,    2,    6,    1,    0,    1,    2,    1,    7,    8,    7,    1,    9,    9,    4,    0,    9,    8,    1,    0,    7    },    // 2^-178
    {    -53,    5,    2,    2,    0,    2,    4,    3,    5,    7,    4,    3,    9,    8,    8,    1,    9,    6,    2,    1,    4    },    // 2^-177
    {    -52,    1,    0,    4,    4,    0,    4,    8,    7,    1,    4,    8,    7,    9,    7,    6,    3,    9,    2,    4,    3    },    // 2^-176
    {    -52,    2,    0,    8,    8,    0,    9,    7,    4,    2,    9,    7,    5,    9,    5,    2,    7,    8,    4,    8,    5    },    // 2^-175
    {    -52,    4,    1,    7,    6,    1,    9,    4,    8,    5,    9,    5,    1,    9,    0,    5,    5,    6,    9,    7,    1    },    // 2^-174
    {    -52,    8,    3,    5,    2,    3,    8,    9,    7,    1,    9,    0,    3,    8,    1,    1,    1,    3,    9,    4,    2    },    // 2^-173
    {    -51,    1,    6,    7,    0,    4,    7,    7,    9,    4,    3,    8,    0,    7,    6,    2,    2,    2,    7,    8,    8    },    // 2^-172
    {    -51,    3,    3,    4,    0,    9,    5,    5,    8,    8,    7,    6,    1,    5,    2,    4,    4,    5,    5,    7,    7    },    // 2^-171
    {    -51,    6,    6,    8,    1,    9,    1,    1,    7,    7,    5,    2,    3,    0,    4,    8,    9,    1,    1,    5,    4    },    // 2^-170
    {    -50,    1,    3,    3,    6,    3,    8,    2,    3,    5,    5,    0,    4,    6,    0,    9,    7,    8,    2,    3,    1    },    // 2^-169
    {    -50,    2,    6,    7,    2,    7,    6,    4,    7,    1,    0,    0,    9,    2,    1,    9,    5,    6,    4,    6,    1    },    // 2^-168
    {    -50,    5,    3,    4,    5,    5,    2,    9,    4,    2,    0,    1,    8,    4,    3,    9,    1,    2,    9,    2,    3    },    // 2^-167
    {    -49,    1,    0,    6,    9,    1,    0,    5,    8,    8,    4,    0,    3,    6,    8,    7,    8,    2,    5,    8,    5    },    // 2^-166
    {    -49,    2,    1,    3,    8,    2,    1,    1,    7,    6,    8,    0,    7,    3,    7,    5,    6,    5,    1,    6,    9    },    // 2^-165
    {    -49,    4,    2,    7,    6,    4,    2,    3,    5,    3,    6,    1,    4,    7,    5,    1,    3,    0,    3,    3,    8    },    // 2^-164
    {    -49,    8,    5,    5,    2,    8,    4,    7,    0,    7,    2,    2,    9,    5,    0,    2,    6,    0,    6,    7,    6    },    // 2^-163
    {    -48,    1,    7,    1,    0,    5,    6,    9,    4,    1,    4,    4,    5,    9,    0,    0,    5,    2,    1,    3,    5    },    // 2^-162
    {    -48,    3,    4,    2,    1,    1,    3,    8,    8,    2,    8,    9,    1,    8,    0,    1,    0,    4,    2,    7,    1    },    // 2^-161
    {    -48,    6,    8,    4,    2,    2,    7,    7,    6,    5,    7,    8,    3,    6,    0,    2,    0,    8,    5,    4,    1    },    // 2^-160
    {    -47,    1,    3,    6,    8,    4,    5,    5,    5,    3,    1,    5,    6,    7,    2,    0,    4,    1,    7,    0,    8    },    // 2^-159
    {    -47,    2,    7,    3,    6,    9,    1,    1,    0,    6,    3,    1,    3,    4,    4,    0,    8,    3,    4,    1,    6    },    // 2^-158
    {    -47,    5,    4,    7,    3,    8,    2,    2,    1,    2,    6,    2,    6,    8,    8,    1,    6,    6,    8,    3,    3    },    // 2^-157
    {    -46,    1,    0,    9,    4,    7,    6,    4,    4,    2,    5,    2,    5,    3,    7,    6,    3,    3,    3,    6,    7    },    // 2^-156
    {    -46,    2,    1,    8,    9,    5,    2,    8,    8,    5,    0,    5,    0,    7,    5,    2,    6,    6,    7,    3,    3    },    // 2^-155
    {    -46,    4,    3,    7,    9,    0,    5,    7,    7,    0,    1,    0,    1,    5,    0,    5,    3,    3,    4,    6,    6    },    // 2^-154
    {    -46,    8,    7,    5,    8,    1,    1,    5,    4,    0,    2,    0,    3,    0,    1,    0,    6,    6,    9,    3,    3    },    // 2^-153
    {    -45,    1,    7,    5,    1,    6,    2,    3,    0,    8,    0,    4,    0,    6,    0,    2,    1,    3,    3,    8,    7    },    // 2^-152
    {    -45,    3,    5,    0,    3,    2,    4,    6,    1,    6,    0,    8,    1,    2,    0,    4,    2,    6,    7,    7,    3    },    // 2^-151
    {    -45,    7,    0,    0,    6,    4,    9,    2,    3,    2,    1,    6,    2,    4,    0,    8,    5,    3,    5,    4,    6    },    // 2^-150
    {    -44,    1,    4,    0,    1,    2,    9,    8,    4,    6,    4,    3,    2,    4,    8,    1,    7,    0,    7,    0,    9    },    // 2^-149
    {    -44,    2,    8,    0,    2,    5,    9,    6,    9,    2,    8,    6,    4,    9,    6,    3,    4,    1,    4,    1,    8    },    // 2^-148
    {    -44,    5,    6,    0,    5,    1,    9,    3,    8,    5,    7,    2,    9,    9,    2,    6,    8,    2,    8,    3,    7    },    // 2^-147
    {    -43,    1,    1,    2,    1,    0,    3,    8,    7,    7,    1,    4,    5,    9,    8,    5,    3,    6,    5,    6,    7    },    // 2^-146
    {    -43,    2,    2,    4,    2,    0,    7,    7,    5,    4,    2,    9,    1,    9,    7,    0,    7,    3,    1,    3,    5    },    // 2^-145
    {    -43,    4,    4,    8,    4,    1,    5,    5,    0,    8,    5,    8,    3,    9,    4,    1,    4,    6,    2,    7,    0    },    // 2^-144
    {    -43,    8,    9,    6,    8,    3,    1,    0,    1,    7,    1,    6,    7,    8,    8,    2,    9,    2,    5,    3,    9    },    // 2^-143
    {    -42,    1,    7,    9,    3,    6,    6,    2,    0,    3,    4,    3,    3,    5,    7,    6,    5,    8,    5,    0,    8    },    // 2^-142
    {    -42,    3,    5,    8,    7,    3,    2,    4,    0,    6,    8,    6,    7,    1,    5,    3,    1,    7,    0,    1,    6    },    // 2^-141
    {    -42,    7,    1,    7,    4,    6,    4,    8,    1,    3,    7,    3,    4,    3,    0,    6,    3,    4,    0,    3,    1    },    // 2^-140
    {    -41,    1,    4,    3,    4,    9,    2,    9,    6,    2,    7,    4,    6,    8,    6,    1,    2,    6,    8,    0,    6    },    // 2^-139
    {    -41,    2,    8,    6,    9,    8,    5,    9,    2,    5,    4,    9,    3,    7,    2,    2,    5,    3,    6,    1,    3    },    // 2^-138
    {    -41,    5,    7,    3,    9,    7,    1,    8,    5,    0,    9,    8,    7,    4,    4,    5,    0,    7,    2,    2,    5    },    // 2^-137
    {    -40,    1,    1,    4,    7,    9,    4,    3,    7,    0,    1,    9,    7,    4,    8,    9,    0,    1,    4,    4,    5    },    // 2^-136
    {    -40,    2,    2,    9,    5,    8,    8,    7,    4,    0,    3,    9,    4,    9,    7,    8,    0,    2,    8,    9,    0    },    // 2^-135
    {    -40,    4,    5,    9,    1,    7,    7,    4,    8,    0,    7,    8,    9,    9,    5,    6,    0,    5,    7,    8,    0    },    // 2^-134
    {    -40,    9,    1,    8,    3,    5,    4,    9,    6,    1,    5,    7,    9,    9,    1,    2,    1,    1,    5,    6,    0    },    // 2^-133
    {    -39,    1,    8,    3,    6,    7,    0,    9,    9,    2,    3,    1,    5,    9,    8,    2,    4,    2,    3,    1,    2    },    // 2^-132
    {    -39,    3,    6,    7,    3,    4,    1,    9,    8,    4,    6,    3,    1,    9,    6,    4,    8,    4,    6,    2,    4    },    // 2^-131
    {    -39,    7,    3,    4,    6,    8,    3,    9,    6,    9,    2,    6,    3,    9,    2,    9,    6,    9,    2,    4,    8    },    // 2^-130
    {    -38,    1,    4,    6,    9,    3,    6,    7,    9,    3,    8,    5,    2,    7,    8,    5,    9,    3,    8,    5,    0    },    // 2^-129
    {    -38,    2,    9,    3,    8,    7,    3,    5,    8,    7,    7,    0,    5,    5,    7,    1,    8,    7,    6,    9,    9    },    // 2^-128
    {    -38,    5,    8,    7,    7,    4,    7,    1,    7,    5,    4,    1,    1,    1,    4,    3,    7,    5,    3,    9,    8    },    // 2^-127
    {    -37,    1,    1,    7,    5,    4,    9,    4,    3,    5,    0,    8,    2,    2,    2,    8,    7,    5,    0,    8,    0    },    // 2^-126
    {    -37,    2,    3,    5,    0,    9,    8,    8,    7,    0,    1,    6,    4,    4,    5,    7,    5,    0,    1,    5,    9    },    // 2^-125
    {    -37,    4,    7,    0,    1,    9,    7,    7,    4,    0,    3,    2,    8,    9,    1,    5,    0,    0,    3,    1,    9    },    // 2^-124
    {    -37,    9,    4,    0,    3,    9,    5,    4,    8,    0,    6,    5,    7,    8,    3,    0,    0,    0,    6,    3,    7    },    // 2^-123
    {    -36,    1,    8,    8,    0,    7,    9,    0,    9,    6,    1,    3,    1,    5,    6,    6,    0,    0,    1,    2,    7    },    // 2^-122
    {    -36,    3,    7,    6,    1,    5,    8,    1,    9,    2,    2,    6,    3,    1,    3,    2,    0,    0,    2,    5,    5    },    // 2^-121
    {    -36,    7,    5,    2,    3,    1,    6,    3,    8,    4,    5,    2,    6,    2,    6,    4,    0,    0,    5,    1,    0    },    // 2^-120
    {    -35,    1,    5,    0,    4,    6,    3,    2,    7,    6,    9,    0,    5,    2,    5,    2,    8,    0,    1,    0,    2    },    // 2^-119
    {    -35,    3,    0,    0,    9,    2,    6,    5,    5,    3,    8,    1,    0,    5,    0,    5,    6,    0,    2,    0,    4    },    // 2^-118
    {    -35,    6,    0,    1,    8,    5,    3,    1,    0,    7,    6,    2,    1,    0,    1,    1,    2,    0,    4,    0,    8    },    // 2^-117
    {    -34,    1,    2,    0,    3,    7,    0,    6,    2,    1,    5,    2,    4,    2,    0,    2,    2,    4,    0,    8,    2    },    // 2^-116
    {    -34,    2,    4,    0,    7,    4,    1,    2,    4,    3,    0,    4,    8,    4,    0,    4,    4,    8,    1,    6,    3    },    // 2^-115
    {    -34,    4,    8,    1,    4,    8,    2,    4,    8,    6,    0,    9,    6,    8,    0,    8,    9,    6,    3,    2,    6    },    // 2^-114
    {    -34,    9,    6,    2,    9,    6,    4,    9,    7,    2,    1,    9,    3,    6,    1,    7,    9,    2,    6,    5,    3    },    // 2^-113
    {    -33,    1,    9,    2,    5,    9,    2,    9,    9,    4,    4,    3,    8,    7,    2,    3,    5,    8,    5,    3,    1    },    // 2^-112
    {    -33,    3,    8,    5,    1,    8,    5,    9,    8,    8,    8,    7,    7,    4,    4,    7,    1,    7,    0,    6,    1    },    // 2^-111
    {    -33,    7,    7,    0,    3,    7,    1,    9,    7,    7,    7,    5,    4,    8,    9,    4,    3,    4,    1,    2,    2    },    // 2^-110
    {    -32,    1,    5,    4,    0,    7,    4,    3,    9,    5,    5,    5,    0,    9,    7,    8,    8,    6,    8,    2,    4    },    // 2^-109
    {    -32,    3,    0,    8,    1,    4,    8,    7,    9,    1,    1,    0,    1,    9,    5,    7,    7,    3,    6,    4,    9    },    // 2^-108
    {    -32,    6,    1,    6,    2,    9,    7,    5,    8,    2,    2,    0,    3,    9,    1,    5,    4,    7,    2,    9,    8    },    // 2^-107
    {    -31,    1,    2,    3,    2,    5,    9,    5,    1,    6,    4,    4,    0,    7,    8,    3,    0,    9,    4,    6,    0    },    // 2^-106
    {    -31,    2,    4,    6,    5,    1,    9,    0,    3,    2,    8,    8,    1,    5,    6,    6,    1,    8,    9,    1,    9    },    // 2^-105
    {    -31,    4,    9,    3,    0,    3,    8,    0,    6,    5,    7,    6,    3,    1,    3,    2,    3,    7,    8,    3,    8    },    // 2^-104
    {    -31,    9,    8,    6,    0,    7,    6,    1,    3,    1,    5,    2,    6,    2,    6,    4,    7,    5,    6,    7,    6    },    // 2^-103
    {    -30,    1,    9,    7,    2,    1,    5,    2,    2,    6,    3,    0,    5,    2,    5,    2,    9,    5,    1,    3,    5    },    // 2^-102
    {    -30,    3,    9,    4,    4,    3,    0,    4,    5,    2,    6,    1,    0,    5,    0,    5,    9,    0,    2,    7,    1    },    // 2^-101
    {    -30,    7,    8,    8,    8,    6,    0,    9,    0,    5,    2,    2,    1,    0,    1,    1,    8,    0,    5,    4,    1    },    // 2^-100
    {    -29,    1,    5,    7,    7,    7,    2,    1,    8,    1,    0,    4,    4,    2,    0,    2,    3,    6,    1,    0,    8    },    // 2^-99
    {    -29,    3,    1,    5,    5,    4,    4,    3,    6,    2,    0,    8,    8,    4,    0,    4,    7,    2,    2,    1,    6    },    // 2^-98
    {    -29,    6,    3,    1,    0,    8,    8,    7,    2,    4,    1,    7,    6,    8,    0,    9,    4,    4,    4,    3,    3    },    // 2^-97
    {    -28,    1,    2,    6,    2,    1,    7,    7,    4,    4,    8,    3,    5,    3,    6,    1,    8,    8,    8,    8,    7    },    // 2^-96
    {    -28,    2,    5,    2,    4,    3,    5,    4,    8,    9,    6,    7,    0,    7,    2,    3,    7,    7,    7,    7,    3    },    // 2^-95
    {    -28,    5,    0,    4,    8,    7,    0,    9,    7,    9,    3,    4,    1,    4,    4,    7,    5,    5,    5,    4,    6    },    // 2^-94
    {    -27,    1,    0,    0,    9,    7,    4,    1,    9,    5,    8,    6,    8,    2,    8,    9,    5,    1,    1,    0,    9    },    // 2^-93
    {    -27,    2,    0,    1,    9,    4,    8,    3,    9,    1,    7,    3,    6,    5,    7,    9,    0,    2,    2,    1,    9    },    // 2^-92
    {    -27,    4,    0,    3,    8,    9,    6,    7,    8,    3,    4,    7,    3,    1,    5,    8,    0,    4,    4,    3,    7    },    // 2^-91
    {    -27,    8,    0,    7,    7,    9,    3,    5,    6,    6,    9,    4,    6,    3,    1,    6,    0,    8,    8,    7,    4    },    // 2^-90
    {    -26,    1,    6,    1,    5,    5,    8,    7,    1,    3,    3,    8,    9,    2,    6,    3,    2,    1,    7,    7,    5    },    // 2^-89
    {    -26,    3,    2,    3,    1,    1,    7,    4,    2,    6,    7,    7,    8,    5,    2,    6,    4,    3,    5,    5,    0    },    // 2^-88
    {    -26,    6,    4,    6,    2,    3,    4,    8,    5,    3,    5,    5,    7,    0,    5,    2,    8,    7,    0,    9,    9    },    // 2^-87
    {    -25,    1,    2,    9,    2,    4,    6,    9,    7,    0,    7,    1,    1,    4,    1,    0,    5,    7,    4,    2,    0    },    // 2^-86
    {    -25,    2,    5,    8,    4,    9,    3,    9,    4,    1,    4,    2,    2,    8,    2,    1,    1,    4,    8,    4,    0    },    // 2^-85
    {    -25,    5,    1,    6,    9,    8,    7,    8,    8,    2,    8,    4,    5,    6,    4,    2,    2,    9,    6,    7,    9    },    // 2^-84
    {    -24,    1,    0,    3,    3,    9,    7,    5,    7,    6,    5,    6,    9,    1,    2,    8,    4,    5,    9,    3,    6    },    // 2^-83
    {    -24,    2,    0,    6,    7,    9,    5,    1,    5,    3,    1,    3,    8,    2,    5,    6,    9,    1,    8,    7,    2    },    // 2^-82
    {    -24,    4,    1,    3,    5,    9,    0,    3,    0,    6,    2,    7,    6,    5,    1,    3,    8,    3,    7,    4,    4    },    // 2^-81
    {    -24,    8,    2,    7,    1,    8,    0,    6,    1,    2,    5,    5,    3,    0,    2,    7,    6,    7,    4,    8,    7    },    // 2^-80
    {    -23,    1,    6,    5,    4,    3,    6,    1,    2,    2,    5,    1,    0,    6,    0,    5,    5,    3,    4,    9,    7    },    // 2^-79
    {    -23,    3,    3,    0,    8,    7,    2,    2,    4,    5,    0,    2,    1,    2,    1,    1,    0,    6,    9,    9,    5    },    // 2^-78
    {    -23,    6,    6,    1,    7,    4,    4,    4,    9,    0,    0,    4,    2,    4,    2,    2,    1,    3,    9,    9,    0    },    // 2^-77
    {    -22,    1,    3,    2,    3,    4,    8,    8,    9,    8,    0,    0,    8,    4,    8,    4,    4,    2,    7,    9,    8    },    // 2^-76
    {    -22,    2,    6,    4,    6,    9,    7,    7,    9,    6,    0,    1,    6,    9,    6,    8,    8,    5,    5,    9,    6    },    // 2^-75
    {    -22,    5,    2,    9,    3,    9,    5,    5,    9,    2,    0,    3,    3,    9,    3,    7,    7,    1,    1,    9,    2    },    // 2^-74
    {    -21,    1,    0,    5,    8,    7,    9,    1,    1,    8,    4,    0,    6,    7,    8,    7,    5,    4,    2,    3,    8    },    // 2^-73
    {    -21,    2,    1,    1,    7,    5,    8,    2,    3,    6,    8,    1,    3,    5,    7,    5,    0,    8,    4,    7,    7    },    // 2^-72
    {    -21,    4,    2,    3,    5,    1,    6,    4,    7,    3,    6,    2,    7,    1,    5,    0,    1,    6,    9,    5,    3    },    // 2^-71
    {    -21,    8,    4,    7,    0,    3,    2,    9,    4,    7,    2,    5,    4,    3,    0,    0,    3,    3,    9,    0,    7    },    // 2^-70
    {    -20,    1,    6,    9,    4,    0,    6,    5,    8,    9,    4,    5,    0,    8,    6,    0,    0,    6,    7,    8,    1    },    // 2^-69
    {    -20,    3,    3,    8,    8,    1,    3,    1,    7,    8,    9,    0,    1,    7,    2,    0,    1,    3,    5,    6,    3    },    // 2^-68
    {    -20,    6,    7,    7,    6,    2,    6,    3,    5,    7,    8,    0,    3,    4,    4,    0,    2,    7,    1,    2,    5    },    // 2^-67
    {    -19,    1,    3,    5,    5,    2,    5,    2,    7,    1,    5,    6,    0,    6,    8,    8,    0,    5,    4,    2,    5    },    // 2^-66
    {    -19,    2,    7,    1,    0,    5,    0,    5,    4,    3,    1,    2,    1,    3,    7,    6,    1,    0,    8,    5,    0    },    // 2^-65
    {    -19,    5,    4,    2,    1,    0,    1,    0,    8,    6,    2,    4,    2,    7,    5,    2,    2,    1,    7,    0,    0    },    // 2^-64
    {    -18,    1,    0,    8,    4,    2,    0,    2,    1,    7,    2,    4,    8,    5,    5,    0,    4,    4,    3,    4,    0    },    // 2^-63
    {    -18,    2,    1,    6,    8,    4,    0,    4,    3,    4,    4,    9,    7,    1,    0,    0,    8,    8,    6,    8,    0    },    // 2^-62
    {    -18,    4,    3,    3,    6,    8,    0,    8,    6,    8,    9,    9,    4,    2,    0,    1,    7,    7,    3,    6,    0    },    // 2^-61
    {    -18,    8,    6,    7,    3,    6,    1,    7,    3,    7,    9,    8,    8,    4,    0,    3,    5,    4,    7,    2,    1    },    // 2^-60
    {    -17,    1,    7,    3,    4,    7,    2,    3,    4,    7,    5,    9,    7,    6,    8,    0,    7,    0,    9,    4,    4    },    // 2^-59
    {    -17,    3,    4,    6,    9,    4,    4,    6,    9,    5,    1,    9,    5,    3,    6,    1,    4,    1,    8,    8,    8    },    // 2^-58
    {    -17,    6,    9,    3,    8,    8,    9,    3,    9,    0,    3,    9,    0,    7,    2,    2,    8,    3,    7,    7,    6    },    // 2^-57
    {    -16,    1,    3,    8,    7,    7,    7,    8,    7,    8,    0,    7,    8,    1,    4,    4,    5,    6,    7,    5,    5    },    // 2^-56
    {    -16,    2,    7,    7,    5,    5,    5,    7,    5,    6,    1,    5,    6,    2,    8,    9,    1,    3,    5,    1,    1    },    // 2^-55
    {    -16,    5,    5,    5,    1,    1,    1,    5,    1,    2,    3,    1,    2,    5,    7,    8,    2,    7,    0,    2,    1    },    // 2^-54
    {    -15,    1,    1,    1,    0,    2,    2,    3,    0,    2,    4,    6,    2,    5,    1,    5,    6,    5,    4,    0,    4    },    // 2^-53
    {    -15,    2,    2,    2,    0,    4,    4,    6,    0,    4,    9,    2,    5,    0,    3,    1,    3,    0,    8,    0,    8    },    // 2^-52
    {    -15,    4,    4,    4,    0,    8,    9,    2,    0,    9,    8,    5,    0,    0,    6,    2,    6,    1,    6,    1,    7    },    // 2^-51
    {    -15,    8,    8,    8,    1,    7,    8,    4,    1,    9,    7,    0,    0,    1,    2,    5,    2,    3,    2,    3,    4    },    // 2^-50
    {    -14,    1,    7,    7,    6,    3,    5,    6,    8,    3,    9,    4,    0,    0,    2,    5,    0,    4,    6,    4,    7    },    // 2^-49
    {    -14,    3,    5,    5,    2,    7,    1,    3,    6,    7,    8,    8,    0,    0,    5,    0,    0,    9,    2,    9,    4    },    // 2^-48
    {    -14,    7,    1,    0,    5,    4,    2,    7,    3,    5,    7,    6,    0,    1,    0,    0,    1,    8,    5,    8,    7    },    // 2^-47
    {    -13,    1,    4,    2,    1,    0,    8,    5,    4,    7,    1,    5,    2,    0,    2,    0,    0,    3,    7,    1,    7    },    // 2^-46
    {    -13,    2,    8,    4,    2,    1,    7,    0,    9,    4,    3,    0,    4,    0,    4,    0,    0,    7,    4,    3,    5    },    // 2^-45
    {    -13,    5,    6,    8,    4,    3,    4,    1,    8,    8,    6,    0,    8,    0,    8,    0,    1,    4,    8,    7,    0    },    // 2^-44
    {    -12,    1,    1,    3,    6,    8,    6,    8,    3,    7,    7,    2,    1,    6,    1,    6,    0,    2,    9,    7,    4    },    // 2^-43
    {    -12,    2,    2,    7,    3,    7,    3,    6,    7,    5,    4,    4,    3,    2,    3,    2,    0,    5,    9,    4,    8    },    // 2^-42
    {    -12,    4,    5,    4,    7,    4,    7,    3,    5,    0,    8,    8,    6,    4,    6,    4,    1,    1,    8,    9,    6    },    // 2^-41
    {    -12,    9,    0,    9,    4,    9,    4,    7,    0,    1,    7,    7,    2,    9,    2,    8,    2,    3,    7,    9,    2    },    // 2^-40
    {    -11,    1,    8,    1,    8,    9,    8,    9,    4,    0,    3,    5,    4,    5,    8,    5,    6,    4,    7,    5,    8    },    // 2^-39
    {    -11,    3,    6,    3,    7,    9,    7,    8,    8,    0,    7,    0,    9,    1,    7,    1,    2,    9,    5,    1,    7    },    // 2^-38
    {    -11,    7,    2,    7,    5,    9,    5,    7,    6,    1,    4,    1,    8,    3,    4,    2,    5,    9,    0,    3,    3    },    // 2^-37
    {    -10,    1,    4,    5,    5,    1,    9,    1,    5,    2,    2,    8,    3,    6,    6,    8,    5,    1,    8,    0,    7    },    // 2^-36
    {    -10,    2,    9,    1,    0,    3,    8,    3,    0,    4,    5,    6,    7,    3,    3,    7,    0,    3,    6,    1,    3    },    // 2^-35
    {    -10,    5,    8,    2,    0,    7,    6,    6,    0,    9,    1,    3,    4,    6,    7,    4,    0,    7,    2,    2,    7    },    // 2^-34
    {    -9,    1,    1,    6,    4,    1,    5,    3,    2,    1,    8,    2,    6,    9,    3,    4,    8,    1,    4,    4,    5    },    // 2^-33
    {    -9,    2,    3,    2,    8,    3,    0,    6,    4,    3,    6,    5,    3,    8,    6,    9,    6,    2,    8,    9,    1    },    // 2^-32
    {    -9,    4,    6,    5,    6,    6,    1,    2,    8,    7,    3,    0,    7,    7,    3,    9,    2,    5,    7,    8,    1    },    // 2^-31
    {    -9,    9,    3,    1,    3,    2,    2,    5,    7,    4,    6,    1,    5,    4,    7,    8,    5,    1,    5,    6,    3    },    // 2^-30
    {    -8,    1,    8,    6,    2,    6,    4,    5,    1,    4,    9,    2,    3,    0,    9,    5,    7,    0,    3,    1,    3    },    // 2^-29
    {    -8,    3,    7,    2,    5,    2,    9,    0,    2,    9,    8,    4,    6,    1,    9,    1,    4,    0,    6,    2,    5    },    // 2^-28
    {    -8,    7,    4,    5,    0,    5,    8,    0,    5,    9,    6,    9,    2,    3,    8,    2,    8,    1,    2,    5,    0    },    // 2^-27
    {    -7,    1,    4,    9,    0,    1,    1,    6,    1,    1,    9,    3,    8,    4,    7,    6,    5,    6,    2,    5,    0    },    // 2^-26
    {    -7,    2,    9,    8,    0,    2,    3,    2,    2,    3,    8,    7,    6,    9,    5,    3,    1,    2,    5,    0,    0    },    // 2^-25
    {    -7,    5,    9,    6,    0,    4,    6,    4,    4,    7,    7,    5,    3,    9,    0,    6,    2,    5,    0,    0,    0    },    // 2^-24
    {    -6,    1,    1,    9,    2,    0,    9,    2,    8,    9,    5,    5,    0,    7,    8,    1,    2,    5,    0,    0,    0    },    // 2^-23
    {    -6,    2,    3,    8,    4,    1,    8,    5,    7,    9,    1,    0,    1,    5,    6,    2,    5,    0,    0,    0,    0    },    // 2^-22
    {    -6,    4,    7,    6,    8,    3,    7,    1,    5,    8,    2,    0,    3,    1,    2,    5,    0,    0,    0,    0,    0    },    // 2^-21
    {    -6,    9,    5,    3,    6,    7,    4,    3,    1,    6,    4,    0,    6,    2,    5,    0,    0,    0,    0,    0,    0    },    // 2^-20
    {    -5,    1,    9,    0,    7,    3,    4,    8,    6,    3,    2,    8,    1,    2,    5,    0,    0,    0,    0,    0,    0    },    // 2^-19
    {    -5,    3,    8,    1,    4,    6,    9,    7,    2,    6,    5,    6,    2,    5,    0,    0,    0,    0,    0,    0,    0    },    // 2^-18
    {    -5,    7,    6,    2,    9,    3,    9,    4,    5,    3,    1,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-17
    {    -4,    1,    5,    2,    5,    8,    7,    8,    9,    0,    6,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-16
    {    -4,    3,    0,    5,    1,    7,    5,    7,    8,    1,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-15
    {    -4,    6,    1,    0,    3,    5,    1,    5,    6,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-14
    {    -3,    1,    2,    2,    0,    7,    0,    3,    1,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-13
    {    -3,    2,    4,    4,    1,    4,    0,    6,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-12
    {    -3,    4,    8,    8,    2,    8,    1,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-11
    {    -3,    9,    7,    6,    5,    6,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-10
    {    -2,    1,    9,    5,    3,    1,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-9
    {    -2,    3,    9,    0,    6,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-8
    {    -2,    7,    8,    1,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-7
    {    -1,    1,    5,    6,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-6
    {    -1,    3,    1,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-5
    {    -1,    6,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-4
    {    0,    1,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-3
    {    0,    2,    5,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-2
    {    0,    5,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^-1
    {    1,    1,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^0
    {    1,    2,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^1
    {    1,    4,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^2
    {    1,    8,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^3
    {    2,    1,    6,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^4
    {    2,    3,    2,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^5
    {    2,    6,    4,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^6
    {    3,    1,    2,    8,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^7
    {    3,    2,    5,    6,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^8
    {    3,    5,    1,    2,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^9
    {    4,    1,    0,    2,    4,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^10
    {    4,    2,    0,    4,    8,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^11
    {    4,    4,    0,    9,    6,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^12
    {    4,    8,    1,    9,    2,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^13
    {    5,    1,    6,    3,    8,    4,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^14
    {    5,    3,    2,    7,    6,    8,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^15
    {    5,    6,    5,    5,    3,    6,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^16
    {    6,    1,    3,    1,    0,    7,    2,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^17
    {    6,    2,    6,    2,    1,    4,    4,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^18
    {    6,    5,    2,    4,    2,    8,    8,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^19
    {    7,    1,    0,    4,    8,    5,    7,    6,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^20
    {    7,    2,    0,    9,    7,    1,    5,    2,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^21
    {    7,    4,    1,    9,    4,    3,    0,    4,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^22
    {    7,    8,    3,    8,    8,    6,    0,    8,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^23
    {    8,    1,    6,    7,    7,    7,    2,    1,    6,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^24
    {    8,    3,    3,    5,    5,    4,    4,    3,    2,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^25
    {    8,    6,    7,    1,    0,    8,    8,    6,    4,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^26
    {    9,    1,    3,    4,    2,    1,    7,    7,    2,    8,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^27
    {    9,    2,    6,    8,    4,    3,    5,    4,    5,    6,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^28
    {    9,    5,    3,    6,    8,    7,    0,    9,    1,    2,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^29
    {    10,    1,    0,    7,    3,    7,    4,    1,    8,    2,    4,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^30
    {    10,    2,    1,    4,    7,    4,    8,    3,    6,    4,    8,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^31
    {    10,    4,    2,    9,    4,    9,    6,    7,    2,    9,    6,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^32
    {    10,    8,    5,    8,    9,    9,    3,    4,    5,    9,    2,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^33
    {    11,    1,    7,    1,    7,    9,    8,    6,    9,    1,    8,    4,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^34
    {    11,    3,    4,    3,    5,    9,    7,    3,    8,    3,    6,    8,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^35
    {    11,    6,    8,    7,    1,    9,    4,    7,    6,    7,    3,    6,    0,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^36
    {    12,    1,    3,    7,    4,    3,    8,    9,    5,    3,    4,    7,    2,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^37
    {    12,    2,    7,    4,    8,    7,    7,    9,    0,    6,    9,    4,    4,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^38
    {    12,    5,    4,    9,    7,    5,    5,    8,    1,    3,    8,    8,    8,    0,    0,    0,    0,    0,    0,    0,    0    },    // 2^39
    {    13,    1,    0,    9,    9,    5,    1,    1,    6,    2,    7,    7,    7,    6,    0,    0,    0,    0,    0,    0,    0    },    // 2^40
    {    13,    2,    1,    9,    9,    0,    2,    3,    2,    5,    5,    5,    5,    2,    0,    0,    0,    0,    0,    0,    0    },    // 2^41
    {    13,    4,    3,    9,    8,    0,    4,    6,    5,    1,    1,    1,    0,    4,    0,    0,    0,    0,    0,    0,    0    },    // 2^42
    {    13,    8,    7,    9,    6,    0,    9,    3,    0,    2,    2,    2,    0,    8,    0,    0,    0,    0,    0,    0,    0    },    // 2^43
    {    14,    1,    7,    5,    9,    2,    1,    8,    6,    0,    4,    4,    4,    1,    6,    0,    0,    0,    0,    0,    0    },    // 2^44
    {    14,    3,    5,    1,    8,    4,    3,    7,    2,    0,    8,    8,    8,    3,    2,    0,    0,    0,    0,    0,    0    },    // 2^45
    {    14,    7,    0,    3,    6,    8,    7,    4,    4,    1,    7,    7,    6,    6,    4,    0,    0,    0,    0,    0,    0    },    // 2^46
    {    15,    1,    4,    0,    7,    3,    7,    4,    8,    8,    3,    5,    5,    3,    2,    8,    0,    0,    0,    0,    0    },    // 2^47
    {    15,    2,    8,    1,    4,    7,    4,    9,    7,    6,    7,    1,    0,    6,    5,    6,    0,    0,    0,    0,    0    },    // 2^48
    {    15,    5,    6,    2,    9,    4,    9,    9,    5,    3,    4,    2,    1,    3,    1,    2,    0,    0,    0,    0,    0    },    // 2^49
    {    16,    1,    1,    2,    5,    8,    9,    9,    9,    0,    6,    8,    4,    2,    6,    2,    4,    0,    0,    0,    0    },    // 2^50
    {    16,    2,    2,    5,    1,    7,    9,    9,    8,    1,    3,    6,    8,    5,    2,    4,    8,    0,    0,    0,    0    },    // 2^51
    {    16,    4,    5,    0,    3,    5,    9,    9,    6,    2,    7,    3,    7,    0,    4,    9,    6,    0,    0,    0,    0    },    // 2^52
    {    16,    9,    0,    0,    7,    1,    9,    9,    2,    5,    4,    7,    4,    0,    9,    9,    2,    0,    0,    0,    0    },    // 2^53
    {    17,    1,    8,    0,    1,    4,    3,    9,    8,    5,    0,    9,    4,    8,    1,    9,    8,    4,    0,    0,    0    },    // 2^54
    {    17,    3,    6,    0,    2,    8,    7,    9,    7,    0,    1,    8,    9,    6,    3,    9,    6,    8,    0,    0,    0    },    // 2^55
    {    17,    7,    2,    0,    5,    7,    5,    9,    4,    0,    3,    7,    9,    2,    7,    9,    3,    6,    0,    0,    0    },    // 2^56
    {    18,    1,    4,    4,    1,    1,    5,    1,    8,    8,    0,    7,    5,    8,    5,    5,    8,    7,    2,    0,    0    },    // 2^57
    {    18,    2,    8,    8,    2,    3,    0,    3,    7,    6,    1,    5,    1,    7,    1,    1,    7,    4,    4,    0,    0    },    // 2^58
    {    18,    5,    7,    6,    4,    6,    0,    7,    5,    2,    3,    0,    3,    4,    2,    3,    4,    8,    8,    0,    0    },    // 2^59
    {    19,    1,    1,    5,    2,    9,    2,    1,    5,    0,    4,    6,    0,    6,    8,    4,    6,    9,    7,    6,    0    },    // 2^60
    {    19,    2,    3,    0,    5,    8,    4,    3,    0,    0,    9,    2,    1,    3,    6,    9,    3,    9,    5,    2,    0    },    // 2^61
    {    19,    4,    6,    1,    1,    6,    8,    6,    0,    1,    8,    4,    2,    7,    3,    8,    7,    9,    0,    4,    0    },    // 2^62
    {    19,    9,    2,    2,    3,    3,    7,    2,    0,    3,    6,    8,    5,    4,    7,    7,    5,    8,    0,    8,    0    },    // 2^63
    {    20,    1,    8,    4,    4,    6,    7,    4,    4,    0,    7,    3,    7,    0,    9,    5,    5,    1,    6,    1,    6    },    // 2^64
    {    20,    3,    6,    8,    9,    3,    4,    8,    8,    1,    4,    7,    4,    1,    9,    1,    0,    3,    2,    3,    2    },    // 2^65
    {    20,    7,    3,    7,    8,    6,    9,    7,    6,    2,    9,    4,    8,    3,    8,    2,    0,    6,    4,    6,    4    },    // 2^66
    {    21,    1,    4,    7,    5,    7,    3,    9,    5,    2,    5,    8,    9,    6,    7,    6,    4,    1,    2,    9,    3    },    // 2^67
    {    21,    2,    9,    5,    1,    4,    7,    9,    0,    5,    1,    7,    9,    3,    5,    2,    8,    2,    5,    8,    6    },    // 2^68
    {    21,    5,    9,    0,    2,    9,    5,    8,    1,    0,    3,    5,    8,    7,    0,    5,    6,    5,    1,    7,    1    },    // 2^69
    {    22,    1,    1,    8,    0,    5,    9,    1,    6,    2,    0,    7,    1,    7,    4,    1,    1,    3,    0,    3,    4    },    // 2^70
    {    22,    2,    3,    6,    1,    1,    8,    3,    2,    4,    1,    4,    3,    4,    8,    2,    2,    6,    0,    6,    8    },    // 2^71
    {    22,    4,    7,    2,    2,    3,    6,    6,    4,    8,    2,    8,    6,    9,    6,    4,    5,    2,    1,    3,    7    },    // 2^72
    {    22,    9,    4,    4,    4,    7,    3,    2,    9,    6,    5,    7,    3,    9,    2,    9,    0,    4,    2,    7,    4    },    // 2^73
    {    23,    1,    8,    8,    8,    9,    4,    6,    5,    9,    3,    1,    4,    7,    8,    5,    8,    0,    8,    5,    5    },    // 2^74
    {    23,    3,    7,    7,    7,    8,    9,    3,    1,    8,    6,    2,    9,    5,    7,    1,    6,    1,    7,    1,    0    },    // 2^75
    {    23,    7,    5,    5,    5,    7,    8,    6,    3,    7,    2,    5,    9,    1,    4,    3,    2,    3,    4,    1,    9    },    // 2^76
    {    24,    1,    5,    1,    1,    1,    5,    7,    2,    7,    4,    5,    1,    8,    2,    8,    6,    4,    6,    8,    4    },    // 2^77
    {    24,    3,    0,    2,    2,    3,    1,    4,    5,    4,    9,    0,    3,    6,    5,    7,    2,    9,    3,    6,    8    },    // 2^78
    {    24,    6,    0,    4,    4,    6,    2,    9,    0,    9,    8,    0,    7,    3,    1,    4,    5,    8,    7,    3,    5    },    // 2^79
    {    25,    1,    2,    0,    8,    9,    2,    5,    8,    1,    9,    6,    1,    4,    6,    2,    9,    1,    7,    4,    7    },    // 2^80
    {    25,    2,    4,    1,    7,    8,    5,    1,    6,    3,    9,    2,    2,    9,    2,    5,    8,    3,    4,    9,    4    },    // 2^81
    {    25,    4,    8,    3,    5,    7,    0,    3,    2,    7,    8,    4,    5,    8,    5,    1,    6,    6,    9,    8,    8    },    // 2^82
    {    25,    9,    6,    7,    1,    4,    0,    6,    5,    5,    6,    9,    1,    7,    0,    3,    3,    3,    9,    7,    6    },    // 2^83
    {    26,    1,    9,    3,    4,    2,    8,    1,    3,    1,    1,    3,    8,    3,    4,    0,    6,    6,    7,    9,    5    },    // 2^84
    {    26,    3,    8,    6,    8,    5,    6,    2,    6,    2,    2,    7,    6,    6,    8,    1,    3,    3,    5,    9,    1    },    // 2^85
    {    26,    7,    7,    3,    7,    1,    2,    5,    2,    4,    5,    5,    3,    3,    6,    2,    6,    7,    1,    8,    1    },    // 2^86
    {    27,    1,    5,    4,    7,    4,    2,    5,    0,    4,    9,    1,    0,    6,    7,    2,    5,    3,    4,    3,    6    },    // 2^87
    {    27,    3,    0,    9,    4,    8,    5,    0,    0,    9,    8,    2,    1,    3,    4,    5,    0,    6,    8,    7,    2    },    // 2^88
    {    27,    6,    1,    8,    9,    7,    0,    0,    1,    9,    6,    4,    2,    6,    9,    0,    1,    3,    7,    4,    5    },    // 2^89
    {    28,    1,    2,    3,    7,    9,    4,    0,    0,    3,    9,    2,    8,    5,    3,    8,    0,    2,    7,    4,    9    },    // 2^90
    {    28,    2,    4,    7,    5,    8,    8,    0,    0,    7,    8,    5,    7,    0,    7,    6,    0,    5,    4,    9,    8    },    // 2^91
    {    28,    4,    9,    5,    1,    7,    6,    0,    1,    5,    7,    1,    4,    1,    5,    2,    1,    0,    9,    9,    6    },    // 2^92
    {    28,    9,    9,    0,    3,    5,    2,    0,    3,    1,    4,    2,    8,    3,    0,    4,    2,    1,    9,    9,    2    },    // 2^93
    {    29,    1,    9,    8,    0,    7,    0,    4,    0,    6,    2,    8,    5,    6,    6,    0,    8,    4,    3,    9,    8    },    // 2^94
    {    29,    3,    9,    6,    1,    4,    0,    8,    1,    2,    5,    7,    1,    3,    2,    1,    6,    8,    7,    9,    7    },    // 2^95
    {    29,    7,    9,    2,    2,    8,    1,    6,    2,    5,    1,    4,    2,    6,    4,    3,    3,    7,    5,    9,    4    },    // 2^96
    {    30,    1,    5,    8,    4,    5,    6,    3,    2,    5,    0,    2,    8,    5,    2,    8,    6,    7,    5,    1,    9    },    // 2^97
    {    30,    3,    1,    6,    9,    1,    2,    6,    5,    0,    0,    5,    7,    0,    5,    7,    3,    5,    0,    3,    7    },    // 2^98
    {    30,    6,    3,    3,    8,    2,    5,    3,    0,    0,    1,    1,    4,    1,    1,    4,    7,    0,    0,    7,    5    },    // 2^99
    {    31,    1,    2,    6,    7,    6,    5,    0,    6,    0,    0,    2,    2,    8,    2,    2,    9,    4,    0,    1,    5    },    // 2^100
    {    31,    2,    5,    3,    5,    3,    0,    1,    2,    0,    0,    4,    5,    6,    4,    5,    8,    8,    0,    3,    0    },    // 2^101
    {    31,    5,    0,    7,    0,    6,    0,    2,    4,    0,    0,    9,    1,    2,    9,    1,    7,    6,    0,    6,    0    },    // 2^102
    {    32,    1,    0,    1,    4,    1,    2,    0,    4,    8,    0,    1,    8,    2,    5,    8,    3,    5,    2,    1,    2    },    // 2^103
    {    32,    2,    0,    2,    8,    2,    4,    0,    9,    6,    0,    3,    6,    5,    1,    6,    7,    0,    4,    2,    4    },    // 2^104
    {    32,    4,    0,    5,    6,    4,    8,    1,    9,    2,    0,    7,    3,    0,    3,    3,    4,    0,    8,    4,    8    },    // 2^105
    {    32,    8,    1,    1,    2,    9,    6,    3,    8,    4,    1,    4,    6,    0,    6,    6,    8,    1,    6,    9,    6    },    // 2^106
    {    33,    1,    6,    2,    2,    5,    9,    2,    7,    6,    8,    2,    9,    2,    1,    3,    3,    6,    3,    3,    9    },    // 2^107
    {    33,    3,    2,    4,    5,    1,    8,    5,    5,    3,    6,    5,    8,    4,    2,    6,    7,    2,    6,    7,    8    },    // 2^108
    {    33,    6,    4,    9,    0,    3,    7,    1,    0,    7,    3,    1,    6,    8,    5,    3,    4,    5,    3,    5,    7    },    // 2^109
    {    34,    1,    2,    9,    8,    0,    7,    4,    2,    1,    4,    6,    3,    3,    7,    0,    6,    9,    0,    7,    1    },    // 2^110
    {    34,    2,    5,    9,    6,    1,    4,    8,    4,    2,    9,    2,    6,    7,    4,    1,    3,    8,    1,    4,    3    },    // 2^111
    {    34,    5,    1,    9,    2,    2,    9,    6,    8,    5,    8,    5,    3,    4,    8,    2,    7,    6,    2,    8,    5    },    // 2^112
    {    35,    1,    0,    3,    8,    4,    5,    9,    3,    7,    1,    7,    0,    6,    9,    6,    5,    5,    2,    5,    7    },    // 2^113
    {    35,    2,    0,    7,    6,    9,    1,    8,    7,    4,    3,    4,    1,    3,    9,    3,    1,    0,    5,    1,    4    },    // 2^114
    {    35,    4,    1,    5,    3,    8,    3,    7,    4,    8,    6,    8,    2,    7,    8,    6,    2,    1,    0,    2,    8    },    // 2^115
    {    35,    8,    3,    0,    7,    6,    7,    4,    9,    7,    3,    6,    5,    5,    7,    2,    4,    2,    0,    5,    6    },    // 2^116
    {    36,    1,    6,    6,    1,    5,    3,    4,    9,    9,    4,    7,    3,    1,    1,    4,    4,    8,    4,    1,    1    },    // 2^117
    {    36,    3,    3,    2,    3,    0,    6,    9,    9,    8,    9,    4,    6,    2,    2,    8,    9,    6,    8,    2,    3    },    // 2^118
    {    36,    6,    6,    4,    6,    1,    3,    9,    9,    7,    8,    9,    2,    4,    5,    7,    9,    3,    6,    4,    5    },    // 2^119
    {    37,    1,    3,    2,    9,    2,    2,    7,    9,    9,    5,    7,    8,    4,    9,    1,    5,    8,    7,    2,    9    },    // 2^120
    {    37,    2,    6,    5,    8,    4,    5,    5,    9,    9,    1,    5,    6,    9,    8,    3,    1,    7,    4,    5,    8    },    // 2^121
    {    37,    5,    3,    1,    6,    9,    1,    1,    9,    8,    3,    1,    3,    9,    6,    6,    3,    4,    9,    1,    6    },    // 2^122
    {    38,    1,    0,    6,    3,    3,    8,    2,    3,    9,    6,    6,    2,    7,    9,    3,    2,    6,    9,    8,    3    },    // 2^123
    {    38,    2,    1,    2,    6,    7,    6,    4,    7,    9,    3,    2,    5,    5,    8,    6,    5,    3,    9,    6,    6    },    // 2^124
    {    38,    4,    2,    5,    3,    5,    2,    9,    5,    8,    6,    5,    1,    1,    7,    3,    0,    7,    9,    3,    3    },    // 2^125
    {    38,    8,    5,    0,    7,    0,    5,    9,    1,    7,    3,    0,    2,    3,    4,    6,    1,    5,    8,    6,    6    },    // 2^126
    {    39,    1,    7,    0,    1,    4,    1,    1,    8,    3,    4,    6,    0,    4,    6,    9,    2,    3,    1,    7,    3    },    // 2^127
    {    39,    3,    4,    0,    2,    8,    2,    3,    6,    6,    9,    2,    0,    9,    3,    8,    4,    6,    3,    4,    6    },    // 2^128
    {    39,    6,    8,    0,    5,    6,    4,    7,    3,    3,    8,    4,    1,    8,    7,    6,    9,    2,    6,    9,    3    },    // 2^129
    {    40,    1,    3,    6,    1,    1,    2,    9,    4,    6,    7,    6,    8,    3,    7,    5,    3,    8,    5,    3,    9    },    // 2^130
    {    40,    2,    7,    2,    2,    2,    5,    8,    9,    3,    5,    3,    6,    7,    5,    0,    7,    7,    0,    7,    7    },    // 2^131
    {    40,    5,    4,    4,    4,    5,    1,    7,    8,    7,    0,    7,    3,    5,    0,    1,    5,    4,    1,    5,    4    },    // 2^132
    {    41,    1,    0,    8,    8,    9,    0,    3,    5,    7,    4,    1,    4,    7,    0,    0,    3,    0,    8,    3,    1    },    // 2^133
    {    41,    2,    1,    7,    7,    8,    0,    7,    1,    4,    8,    2,    9,    4,    0,    0,    6,    1,    6,    6,    2    },    // 2^134
    {    41,    4,    3,    5,    5,    6,    1,    4,    2,    9,    6,    5,    8,    8,    0,    1,    2,    3,    3,    2,    3    },    // 2^135
    {    41,    8,    7,    1,    1,    2,    2,    8,    5,    9,    3,    1,    7,    6,    0,    2,    4,    6,    6,    4,    7    },    // 2^136
    {    42,    1,    7,    4,    2,    2,    4,    5,    7,    1,    8,    6,    3,    5,    2,    0,    4,    9,    3,    2,    9    },    // 2^137
    {    42,    3,    4,    8,    4,    4,    9,    1,    4,    3,    7,    2,    7,    0,    4,    0,    9,    8,    6,    5,    9    },    // 2^138
    {    42,    6,    9,    6,    8,    9,    8,    2,    8,    7,    4,    5,    4,    0,    8,    1,    9,    7,    3,    1,    7    },    // 2^139
    {    43,    1,    3,    9,    3,    7,    9,    6,    5,    7,    4,    9,    0,    8,    1,    6,    3,    9,    4,    6,    3    },    // 2^140
    {    43,    2,    7,    8,    7,    5,    9,    3,    1,    4,    9,    8,    1,    6,    3,    2,    7,    8,    9,    2,    7    },    // 2^141
    {    43,    5,    5,    7,    5,    1,    8,    6,    2,    9,    9,    6,    3,    2,    6,    5,    5,    7,    8,    5,    4    },    // 2^142
    {    44,    1,    1,    1,    5,    0,    3,    7,    2,    5,    9,    9,    2,    6,    5,    3,    1,    1,    5,    7,    1    },    // 2^143
    {    44,    2,    2,    3,    0,    0,    7,    4,    5,    1,    9,    8,    5,    3,    0,    6,    2,    3,    1,    4,    2    },    // 2^144
    {    44,    4,    4,    6,    0,    1,    4,    9,    0,    3,    9,    7,    0,    6,    1,    2,    4,    6,    2,    8,    3    },    // 2^145
    {    44,    8,    9,    2,    0,    2,    9,    8,    0,    7,    9,    4,    1,    2,    2,    4,    9,    2,    5,    6,    6    },    // 2^146
    {    45,    1,    7,    8,    4,    0,    5,    9,    6,    1,    5,    8,    8,    2,    4,    4,    9,    8,    5,    1,    3    },    // 2^147
    {    45,    3,    5,    6,    8,    1,    1,    9,    2,    3,    1,    7,    6,    4,    8,    9,    9,    7,    0,    2,    6    },    // 2^148
    {    45,    7,    1,    3,    6,    2,    3,    8,    4,    6,    3,    5,    2,    9,    7,    9,    9,    4,    0,    5,    3    },    // 2^149
    {    46,    1,    4,    2,    7,    2,    4,    7,    6,    9,    2,    7,    0,    5,    9,    5,    9,    8,    8,    1,    1    },    // 2^150
    {    46,    2,    8,    5,    4,    4,    9,    5,    3,    8,    5,    4,    1,    1,    9,    1,    9,    7,    6,    2,    1    },    // 2^151
    {    46,    5,    7,    0,    8,    9,    9,    0,    7,    7,    0,    8,    2,    3,    8,    3,    9,    5,    2,    4,    2    },    // 2^152
    {    47,    1,    1,    4,    1,    7,    9,    8,    1,    5,    4,    1,    6,    4,    7,    6,    7,    9,    0,    4,    8    },    // 2^153
    {    47,    2,    2,    8,    3,    5,    9,    6,    3,    0,    8,    3,    2,    9,    5,    3,    5,    8,    0,    9,    7    },    // 2^154
    {    47,    4,    5,    6,    7,    1,    9,    2,    6,    1,    6,    6,    5,    9,    0,    7,    1,    6,    1,    9,    4    },    // 2^155
    {    47,    9,    1,    3,    4,    3,    8,    5,    2,    3,    3,    3,    1,    8,    1,    4,    3,    2,    3,    8,    8    },    // 2^156
    {    48,    1,    8,    2,    6,    8,    7,    7,    0,    4,    6,    6,    6,    3,    6,    2,    8,    6,    4,    7,    8    },    // 2^157
    {    48,    3,    6,    5,    3,    7,    5,    4,    0,    9,    3,    3,    2,    7,    2,    5,    7,    2,    9,    5,    5    },    // 2^158
    {    48,    7,    3,    0,    7,    5,    0,    8,    1,    8,    6,    6,    5,    4,    5,    1,    4,    5,    9,    1,    0    },    // 2^159
    {    49,    1,    4,    6,    1,    5,    0,    1,    6,    3,    7,    3,    3,    0,    9,    0,    2,    9,    1,    8,    2    },    // 2^160
    {    49,    2,    9,    2,    3,    0,    0,    3,    2,    7,    4,    6,    6,    1,    8,    0,    5,    8,    3,    6,    4    },    // 2^161
    {    49,    5,    8,    4,    6,    0,    0,    6,    5,    4,    9,    3,    2,    3,    6,    1,    1,    6,    7,    2,    8    },    // 2^162
    {    50,    1,    1,    6,    9,    2,    0,    1,    3,    0,    9,    8,    6,    4,    7,    2,    2,    3,    3,    4,    6    },    // 2^163
    {    50,    2,    3,    3,    8,    4,    0,    2,    6,    1,    9,    7,    2,    9,    4,    4,    4,    6,    6,    9,    1    },    // 2^164
    {    50,    4,    6,    7,    6,    8,    0,    5,    2,    3,    9,    4,    5,    8,    8,    8,    9,    3,    3,    8,    3    },    // 2^165
    {    50,    9,    3,    5,    3,    6,    1,    0,    4,    7,    8,    9,    1,    7,    7,    7,    8,    6,    7,    6,    5    },    // 2^166
    {    51,    1,    8,    7,    0,    7,    2,    2,    0,    9,    5,    7,    8,    3,    5,    5,    5,    7,    3,    5,    3    },    // 2^167
    {    51,    3,    7,    4,    1,    4,    4,    4,    1,    9,    1,    5,    6,    7,    1,    1,    1,    4,    7,    0,    6    },    // 2^168
    {    51,    7,    4,    8,    2,    8,    8,    8,    3,    8,    3,    1,    3,    4,    2,    2,    2,    9,    4,    1,    2    },    // 2^169
    {    52,    1,    4,    9,    6,    5,    7,    7,    6,    7,    6,    6,    2,    6,    8,    4,    4,    5,    8,    8,    2    },    // 2^170
    {    52,    2,    9,    9,    3,    1,    5,    5,    3,    5,    3,    2,    5,    3,    6,    8,    9,    1,    7,    6,    5    },    // 2^171
    {    52,    5,    9,    8,    6,    3,    1,    0,    7,    0,    6,    5,    0,    7,    3,    7,    8,    3,    5,    3,    0    },    // 2^172
    {    53,    1,    1,    9,    7,    2,    6,    2,    1,    4,    1,    3,    0,    1,    4,    7,    5,    6,    7,    0,    6    },    // 2^173
    {    53,    2,    3,    9,    4,    5,    2,    4,    2,    8,    2,    6,    0,    2,    9,    5,    1,    3,    4,    1,    2    },    // 2^174
    {    53,    4,    7,    8,    9,    0,    4,    8,    5,    6,    5,    2,    0,    5,    9,    0,    2,    6,    8,    2,    4    },    // 2^175
    {    53,    9,    5,    7,    8,    0,    9,    7,    1,    3,    0,    4,    1,    1,    8,    0,    5,    3,    6,    4,    7    },    // 2^176
    {    54,    1,    9,    1,    5,    6,    1,    9,    4,    2,    6,    0,    8,    2,    3,    6,    1,    0,    7,    2,    9    },    // 2^177
    {    54,    3,    8,    3,    1,    2,    3,    8,    8,    5,    2,    1,    6,    4,    7,    2,    2,    1,    4,    5,    9    },    // 2^178
    {    54,    7,    6,    6,    2,    4,    7,    7,    7,    0,    4,    3,    2,    9,    4,    4,    4,    2,    9,    1,    8    },    // 2^179
    {    55,    1,    5,    3,    2,    4,    9,    5,    5,    4,    0,    8,    6,    5,    8,    8,    8,    8,    5,    8,    4    },    // 2^180
    {    55,    3,    0,    6,    4,    9,    9,    1,    0,    8,    1,    7,    3,    1,    7,    7,    7,    7,    1,    6,    7    },    // 2^181
    {    55,    6,    1,    2,    9,    9,    8,    2,    1,    6,    3,    4,    6,    3,    5,    5,    5,    4,    3,    3,    4    },    // 2^182
    {    56,    1,    2,    2,    5,    9,    9,    6,    4,    3,    2,    6,    9,    2,    7,    1,    1,    0,    8,    6,    7    },    // 2^183
    {    56,    2,    4,    5,    1,    9,    9,    2,    8,    6,    5,    3,    8,    5,    4,    2,    2,    1,    7,    3,    4    },    // 2^184
    {    56,    4,    9,    0,    3,    9,    8,    5,    7,    3,    0,    7,    7,    0,    8,    4,    4,    3,    4,    6,    7    },    // 2^185
    {    56,    9,    8,    0,    7,    9,    7,    1,    4,    6,    1,    5,    4,    1,    6,    8,    8,    6,    9,    3,    5    },    // 2^186
    {    57,    1,    9,    6,    1,    5,    9,    4,    2,    9,    2,    3,    0,    8,    3,    3,    7,    7,    3,    8,    7    },    // 2^187
    {    57,    3,    9,    2,    3,    1,    8,    8,    5,    8,    4,    6,    1,    6,    6,    7,    5,    4,    7,    7,    4    },    // 2^188
    {    57,    7,    8,    4,    6,    3,    7,    7,    1,    6,    9,    2,    3,    3,    3,    5,    0,    9,    5,    4,    8    },    // 2^189
    {    58,    1,    5,    6,    9,    2,    7,    5,    4,    3,    3,    8,    4,    6,    6,    7,    0,    1,    9,    1,    0    },    // 2^190
    {    58,    3,    1,    3,    8,    5,    5,    0,    8,    6,    7,    6,    9,    3,    3,    4,    0,    3,    8,    1,    9    },    // 2^191
    {    58,    6,    2,    7,    7,    1,    0,    1,    7,    3,    5,    3,    8,    6,    6,    8,    0,    7,    6,    3,    8    },    // 2^192
    {    59,    1,    2,    5,    5,    4,    2,    0,    3,    4,    7,    0,    7,    7,    3,    3,    6,    1,    5,    2,    8    },    // 2^193
    {    59,    2,    5,    1,    0,    8,    4,    0,    6,    9,    4,    1,    5,    4,    6,    7,    2,    3,    0,    5,    5    },    // 2^194
    {    59,    5,    0,    2,    1,    6,    8,    1,    3,    8,    8,    3,    0,    9,    3,    4,    4,    6,    1,    1,    1    },    // 2^195
    {    60,    1,    0,    0,    4,    3,    3,    6,    2,    7,    7,    6,    6,    1,    8,    6,    8,    9,    2,    2,    2    },    // 2^196
    {    60,    2,    0,    0,    8,    6,    7,    2,    5,    5,    5,    3,    2,    3,    7,    3,    7,    8,    4,    4,    4    },    // 2^197
    {    60,    4,    0,    1,    7,    3,    4,    5,    1,    1,    0,    6,    4,    7,    4,    7,    5,    6,    8,    8,    9    },    // 2^198
    {    60,    8,    0,    3,    4,    6,    9,    0,    2,    2,    1,    2,    9,    4,    9,    5,    1,    3,    7,    7,    7    },    // 2^199
    {    61,    1,    6,    0,    6,    9,    3,    8,    0,    4,    4,    2,    5,    8,    9,    9,    0,    2,    7,    5,    5    },    // 2^200
    {    61,    3,    2,    1,    3,    8,    7,    6,    0,    8,    8,    5,    1,    7,    9,    8,    0,    5,    5,    1,    1    },    // 2^201
    {    61,    6,    4,    2,    7,    7,    5,    2,    1,    7,    7,    0,    3,    5,    9,    6,    1,    1,    0,    2,    2    },    // 2^202
    {    62,    1,    2,    8,    5,    5,    5,    0,    4,    3,    5,    4,    0,    7,    1,    9,    2,    2,    2,    0,    4    },    // 2^203
    {    62,    2,    5,    7,    1,    1,    0,    0,    8,    7,    0,    8,    1,    4,    3,    8,    4,    4,    4,    0,    9    },    // 2^204
    {    62,    5,    1,    4,    2,    2,    0,    1,    7,    4,    1,    6,    2,    8,    7,    6,    8,    8,    8,    1,    7    },    // 2^205
    {    63,    1,    0,    2,    8,    4,    4,    0,    3,    4,    8,    3,    2,    5,    7,    5,    3,    7,    7,    6,    3    },    // 2^206
    {    63,    2,    0,    5,    6,    8,    8,    0,    6,    9,    6,    6,    5,    1,    5,    0,    7,    5,    5,    2,    7    },    // 2^207
    {    63,    4,    1,    1,    3,    7,    6,    1,    3,    9,    3,    3,    0,    3,    0,    1,    5,    1,    0,    5,    4    },    // 2^208
    {    63,    8,    2,    2,    7,    5,    2,    2,    7,    8,    6,    6,    0,    6,    0,    3,    0,    2,    1,    0,    8    },    // 2^209
    {    64,    1,    6,    4,    5,    5,    0,    4,    5,    5,    7,    3,    2,    1,    2,    0,    6,    0,    4,    2,    2    },    // 2^210
    {    64,    3,    2,    9,    1,    0,    0,    9,    1,    1,    4,    6,    4,    2,    4,    1,    2,    0,    8,    4,    3    },    // 2^211
    {    64,    6,    5,    8,    2,    0,    1,    8,    2,    2,    9,    2,    8,    4,    8,    2,    4,    1,    6,    8,    6    },    // 2^212
    {    65,    1,    3,    1,    6,    4,    0,    3,    6,    4,    5,    8,    5,    6,    9,    6,    4,    8,    3,    3,    7    },    // 2^213
    {    65,    2,    6,    3,    2,    8,    0,    7,    2,    9,    1,    7,    1,    3,    9,    2,    9,    6,    6,    7,    4    },    // 2^214
    {    65,    5,    2,    6,    5,    6,    1,    4,    5,    8,    3,    4,    2,    7,    8,    5,    9,    3,    3,    4,    9    },    // 2^215
    {    66,    1,    0,    5,    3,    1,    2,    2,    9,    1,    6,    6,    8,    5,    5,    7,    1,    8,    6,    7,    0    },    // 2^216
    {    66,    2,    1,    0,    6,    2,    4,    5,    8,    3,    3,    3,    7,    1,    1,    4,    3,    7,    3,    4,    0    },    // 2^217
    {    66,    4,    2,    1,    2,    4,    9,    1,    6,    6,    6,    7,    4,    2,    2,    8,    7,    4,    6,    7,    9    },    // 2^218
    {    66,    8,    4,    2,    4,    9,    8,    3,    3,    3,    3,    4,    8,    4,    5,    7,    4,    9,    3,    5,    8    },    // 2^219
    {    67,    1,    6,    8,    4,    9,    9,    6,    6,    6,    6,    6,    9,    6,    9,    1,    4,    9,    8,    7,    2    },    // 2^220
    {    67,    3,    3,    6,    9,    9,    9,    3,    3,    3,    3,    3,    9,    3,    8,    2,    9,    9,    7,    4,    3    },    // 2^221
    {    67,    6,    7,    3,    9,    9,    8,    6,    6,    6,    6,    7,    8,    7,    6,    5,    9,    9,    4,    8,    7    },    // 2^222
    {    68,    1,    3,    4,    7,    9,    9,    7,    3,    3,    3,    3,    5,    7,    5,    3,    1,    9,    8,    9,    7    },    // 2^223
    {    68,    2,    6,    9,    5,    9,    9,    4,    6,    6,    6,    7,    1,    5,    0,    6,    3,    9,    7,    9,    5    },    // 2^224
    {    68,    5,    3,    9,    1,    9,    8,    9,    3,    3,    3,    4,    3,    0,    1,    2,    7,    9,    5,    8,    9    },    // 2^225
    {    69,    1,    0,    7,    8,    3,    9,    7,    8,    6,    6,    6,    8,    6,    0,    2,    5,    5,    9,    1,    8    },    // 2^226
    {    69,    2,    1,    5,    6,    7,    9,    5,    7,    3,    3,    3,    7,    2,    0,    5,    1,    1,    8,    3,    6    },    // 2^227
    {    69,    4,    3,    1,    3,    5,    9,    1,    4,    6,    6,    7,    4,    4,    1,    0,    2,    3,    6,    7,    1    },    // 2^228
    {    69,    8,    6,    2,    7,    1,    8,    2,    9,    3,    3,    4,    8,    8,    2,    0,    4,    7,    3,    4,    3    },    // 2^229
    {    70,    1,    7,    2,    5,    4,    3,    6,    5,    8,    6,    6,    9,    7,    6,    4,    0,    9,    4,    6,    9    },    // 2^230
    {    70,    3,    4,    5,    0,    8,    7,    3,    1,    7,    3,    3,    9,    5,    2,    8,    1,    8,    9,    3,    7    },    // 2^231
    {    70,    6,    9,    0,    1,    7,    4,    6,    3,    4,    6,    7,    9,    0,    5,    6,    3,    7,    8,    7,    4    },    // 2^232
    {    71,    1,    3,    8,    0,    3,    4,    9,    2,    6,    9,    3,    5,    8,    1,    1,    2,    7,    5,    7,    5    },    // 2^233
    {    71,    2,    7,    6,    0,    6,    9,    8,    5,    3,    8,    7,    1,    6,    2,    2,    5,    5,    1,    5,    0    },    // 2^234
    {    71,    5,    5,    2,    1,    3,    9,    7,    0,    7,    7,    4,    3,    2,    4,    5,    1,    0,    2,    9,    9    },    // 2^235
    {    72,    1,    1,    0,    4,    2,    7,    9,    4,    1,    5,    4,    8,    6,    4,    9,    0,    2,    0,    6,    0    },    // 2^236
    {    72,    2,    2,    0,    8,    5,    5,    8,    8,    3,    0,    9,    7,    2,    9,    8,    0,    4,    1,    2,    0    },    // 2^237
    {    72,    4,    4,    1,    7,    1,    1,    7,    6,    6,    1,    9,    4,    5,    9,    6,    0,    8,    2,    4,    0    },    // 2^238
    {    72,    8,    8,    3,    4,    2,    3,    5,    3,    2,    3,    8,    9,    1,    9,    2,    1,    6,    4,    7,    9    },    // 2^239
    {    73,    1,    7,    6,    6,    8,    4,    7,    0,    6,    4,    7,    7,    8,    3,    8,    4,    3,    2,    9,    6    },    // 2^240
    {    73,    3,    5,    3,    3,    6,    9,    4,    1,    2,    9,    5,    5,    6,    7,    6,    8,    6,    5,    9,    2    },    // 2^241
    {    73,    7,    0,    6,    7,    3,    8,    8,    2,    5,    9,    1,    1,    3,    5,    3,    7,    3,    1,    8,    3    },    // 2^242
    {    74,    1,    4,    1,    3,    4,    7,    7,    6,    5,    1,    8,    2,    2,    7,    0,    7,    4,    6,    3,    7    },    // 2^243
    {    74,    2,    8,    2,    6,    9,    5,    5,    3,    0,    3,    6,    4,    5,    4,    1,    4,    9,    2,    7,    3    },    // 2^244
    {    74,    5,    6,    5,    3,    9,    1,    0,    6,    0,    7,    2,    9,    0,    8,    2,    9,    8,    5,    4,    7    },    // 2^245
    {    75,    1,    1,    3,    0,    7,    8,    2,    1,    2,    1,    4,    5,    8,    1,    6,    5,    9,    7,    0,    9    },    // 2^246
    {    75,    2,    2,    6,    1,    5,    6,    4,    2,    4,    2,    9,    1,    6,    3,    3,    1,    9,    4,    1,    9    },    // 2^247
    {    75,    4,    5,    2,    3,    1,    2,    8,    4,    8,    5,    8,    3,    2,    6,    6,    3,    8,    8,    3,    7    },    // 2^248
    {    75,    9,    0,    4,    6,    2,    5,    6,    9,    7,    1,    6,    6,    5,    3,    2,    7,    7,    6,    7,    5    },    // 2^249
    {    76,    1,    8,    0,    9,    2,    5,    1,    3,    9,    4,    3,    3,    3,    0,    6,    5,    5,    5,    3,    5    },    // 2^250
    {    76,    3,    6,    1,    8,    5,    0,    2,    7,    8,    8,    6,    6,    6,    1,    3,    1,    1,    0,    7,    0    },    // 2^251
    {    76,    7,    2,    3,    7,    0,    0,    5,    5,    7,    7,    3,    3,    2,    2,    6,    2,    2,    1,    4,    0    },    // 2^252
    {    77,    1,    4,    4,    7,    4,    0,    1,    1,    1,    5,    4,    6,    6,    4,    5,    2,    4,    4,    2,    8    },    // 2^253
    {    77,    2,    8,    9,    4,    8,    0,    2,    2,    3,    0,    9,    3,    2,    9,    0,    4,    8,    8,    5,    6    },    // 2^254
    {    77,    5,    7,    8,    9,    6,    0,    4,    4,    6,    1,    8,    6,    5,    8,    0,    9,    7,    7,    1,    2    },    // 2^255
    {    78,    1,    1,    5,    7,    9,    2,    0,    8,    9,    2,    3,    7,    3,    1,    6,    1,    9,    5,    4,    2    },    // 2^256
    {    78,    2,    3,    1,    5,    8,    4,    1,    7,    8,    4,    7,    4,    6,    3,    2,    3,    9,    0,    8,    5    },    // 2^257
    {    78,    4,    6,    3,    1,    6,    8,    3,    5,    6,    9,    4,    9,    2,    6,    4,    7,    8,    1,    6,    9    },    // 2^258
    {    78,    9,    2,    6,    3,    3,    6,    7,    1,    3,    8,    9,    8,    5,    2,    9,    5,    6,    3,    3,    9    },    // 2^259
    {    79,    1,    8,    5,    2,    6,    7,    3,    4,    2,    7,    7,    9,    7,    0,    5,    9,    1,    2,    6,    8    },    // 2^260
    {    79,    3,    7,    0,    5,    3,    4,    6,    8,    5,    5,    5,    9,    4,    1,    1,    8,    2,    5,    3,    6    },    // 2^261
    {    79,    7,    4,    1,    0,    6,    9,    3,    7,    1,    1,    1,    8,    8,    2,    3,    6,    5,    0,    7,    1    },    // 2^262
    {    80,    1,    4,    8,    2,    1,    3,    8,    7,    4,    2,    2,    3,    7,    6,    4,    7,    3,    0,    1,    4    },    // 2^263
    {    80,    2,    9,    6,    4,    2,    7,    7,    4,    8,    4,    4,    7,    5,    2,    9,    4,    6,    0,    2,    8    },    // 2^264
    {    80,    5,    9,    2,    8,    5,    5,    4,    9,    6,    8,    9,    5,    0,    5,    8,    9,    2,    0,    5,    7    },    // 2^265
    {    81,    1,    1,    8,    5,    7,    1,    0,    9,    9,    3,    7,    9,    0,    1,    1,    7,    8,    4,    1,    1    },    // 2^266
    {    81,    2,    3,    7,    1,    4,    2,    1,    9,    8,    7,    5,    8,    0,    2,    3,    5,    6,    8,    2,    3    },    // 2^267
    {    81,    4,    7,    4,    2,    8,    4,    3,    9,    7,    5,    1,    6,    0,    4,    7,    1,    3,    6,    4,    5    },    // 2^268
    {    81,    9,    4,    8,    5,    6,    8,    7,    9,    5,    0,    3,    2,    0,    9,    4,    2,    7,    2,    9,    1    },    // 2^269
    {    82,    1,    8,    9,    7,    1,    3,    7,    5,    9,    0,    0,    6,    4,    1,    8,    8,    5,    4,    5,    8    },    // 2^270
    {    82,    3,    7,    9,    4,    2,    7,    5,    1,    8,    0,    1,    2,    8,    3,    7,    7,    0,    9,    1,    6    },    // 2^271
    {    82,    7,    5,    8,    8,    5,    5,    0,    3,    6,    0,    2,    5,    6,    7,    5,    4,    1,    8,    3,    3    },    // 2^272
    {    83,    1,    5,    1,    7,    7,    1,    0,    0,    7,    2,    0,    5,    1,    3,    5,    0,    8,    3,    6,    7    },    // 2^273
    {    83,    3,    0,    3,    5,    4,    2,    0,    1,    4,    4,    1,    0,    2,    7,    0,    1,    6,    7,    3,    3    },    // 2^274
    {    83,    6,    0,    7,    0,    8,    4,    0,    2,    8,    8,    2,    0,    5,    4,    0,    3,    3,    4,    6,    6    },    // 2^275
    {    84,    1,    2,    1,    4,    1,    6,    8,    0,    5,    7,    6,    4,    1,    0,    8,    0,    6,    6,    9,    3    },    // 2^276
    {    84,    2,    4,    2,    8,    3,    3,    6,    1,    1,    5,    2,    8,    2,    1,    6,    1,    3,    3,    8,    6    },    // 2^277
    {    84,    4,    8,    5,    6,    6,    7,    2,    2,    3,    0,    5,    6,    4,    3,    2,    2,    6,    7,    7,    3    },    // 2^278
    {    84,    9,    7,    1,    3,    3,    4,    4,    4,    6,    1,    1,    2,    8,    6,    4,    5,    3,    5,    4,    6    },    // 2^279
    {    85,    1,    9,    4,    2,    6,    6,    8,    8,    9,    2,    2,    2,    5,    7,    2,    9,    0,    7,    0,    9    },    // 2^280
    {    85,    3,    8,    8,    5,    3,    3,    7,    7,    8,    4,    4,    5,    1,    4,    5,    8,    1,    4,    1,    8    },    // 2^281
    {    85,    7,    7,    7,    0,    6,    7,    5,    5,    6,    8,    9,    0,    2,    9,    1,    6,    2,    8,    3,    7    },    // 2^282
    {    86,    1,    5,    5,    4,    1,    3,    5,    1,    1,    3,    7,    8,    0,    5,    8,    3,    2,    5,    6,    7    },    // 2^283
    {    86,    3,    1,    0,    8,    2,    7,    0,    2,    2,    7,    5,    6,    1,    1,    6,    6,    5,    1,    3,    5    },    // 2^284
    {    86,    6,    2,    1,    6,    5,    4,    0,    4,    5,    5,    1,    2,    2,    3,    3,    3,    0,    2,    6,    9    },    // 2^285
    {    87,    1,    2,    4,    3,    3,    0,    8,    0,    9,    1,    0,    2,    4,    4,    6,    6,    6,    0,    5,    4    },    // 2^286
    {    87,    2,    4,    8,    6,    6,    1,    6,    1,    8,    2,    0,    4,    8,    9,    3,    3,    2,    1,    0,    8    },    // 2^287
    {    87,    4,    9,    7,    3,    2,    3,    2,    3,    6,    4,    0,    9,    7,    8,    6,    6,    4,    2,    1,    6    },    // 2^288
    {    87,    9,    9,    4,    6,    4,    6,    4,    7,    2,    8,    1,    9,    5,    7,    3,    2,    8,    4,    3,    1    },    // 2^289
    {    88,    1,    9,    8,    9,    2,    9,    2,    9,    4,    5,    6,    3,    9,    1,    4,    6,    5,    6,    8,    6    },    // 2^290
    {    88,    3,    9,    7,    8,    5,    8,    5,    8,    9,    1,    2,    7,    8,    2,    9,    3,    1,    3,    7,    2    },    // 2^291
    {    88,    7,    9,    5,    7,    1,    7,    1,    7,    8,    2,    5,    5,    6,    5,    8,    6,    2,    7,    4,    5    },    // 2^292
    {    89,    1,    5,    9,    1,    4,    3,    4,    3,    5,    6,    5,    1,    1,    3,    1,    7,    2,    5,    4,    9    },    // 2^293
    {    89,    3,    1,    8,    2,    8,    6,    8,    7,    1,    3,    0,    2,    2,    6,    3,    4,    5,    0,    9,    8    },    // 2^294
    {    89,    6,    3,    6,    5,    7,    3,    7,    4,    2,    6,    0,    4,    5,    2,    6,    9,    0,    1,    9,    6    },    // 2^295
    {    90,    1,    2,    7,    3,    1,    4,    7,    4,    8,    5,    2,    0,    9,    0,    5,    3,    8,    0,    3,    9    },    // 2^296
    {    90,    2,    5,    4,    6,    2,    9,    4,    9,    7,    0,    4,    1,    8,    1,    0,    7,    6,    0,    7,    8    },    // 2^297
    {    90,    5,    0,    9,    2,    5,    8,    9,    9,    4,    0,    8,    3,    6,    2,    1,    5,    2,    1,    5,    7    },    // 2^298
    {    91,    1,    0,    1,    8,    5,    1,    7,    9,    8,    8,    1,    6,    7,    2,    4,    3,    0,    4,    3,    1    },    // 2^299
    {    91,    2,    0,    3,    7,    0,    3,    5,    9,    7,    6,    3,    3,    4,    4,    8,    6,    0,    8,    6,    3    },    // 2^300
    {    91,    4,    0,    7,    4,    0,    7,    1,    9,    5,    2,    6,    6,    8,    9,    7,    2,    1,    7,    2,    5    },    // 2^301
    {    91,    8,    1,    4,    8,    1,    4,    3,    9,    0,    5,    3,    3,    7,    9,    4,    4,    3,    4,    5,    1    },    // 2^302
    {    92,    1,    6,    2,    9,    6,    2,    8,    7,    8,    1,    0,    6,    7,    5,    8,    8,    8,    6,    9,    0    },    // 2^303
    {    92,    3,    2,    5,    9,    2,    5,    7,    5,    6,    2,    1,    3,    5,    1,    7,    7,    7,    3,    8,    0    },    // 2^304
    {    92,    6,    5,    1,    8,    5,    1,    5,    1,    2,    4,    2,    7,    0,    3,    5,    5,    4,    7,    6,    1    },    // 2^305
    {    93,    1,    3,    0,    3,    7,    0,    3,    0,    2,    4,    8,    5,    4,    0,    7,    1,    0,    9,    5,    2    },    // 2^306
    {    93,    2,    6,    0,    7,    4,    0,    6,    0,    4,    9,    7,    0,    8,    1,    4,    2,    1,    9,    0,    4    },    // 2^307
    {    93,    5,    2,    1,    4,    8,    1,    2,    0,    9,    9,    4,    1,    6,    2,    8,    4,    3,    8,    0,    8    },    // 2^308
    {    94,    1,    0,    4,    2,    9,    6,    2,    4,    1,    9,    8,    8,    3,    2,    5,    6,    8,    7,    6,    2    },    // 2^309
    {    94,    2,    0,    8,    5,    9,    2,    4,    8,    3,    9,    7,    6,    6,    5,    1,    3,    7,    5,    2,    3    },    // 2^310
    {    94,    4,    1,    7,    1,    8,    4,    9,    6,    7,    9,    5,    3,    3,    0,    2,    7,    5,    0,    4,    7    },    // 2^311
    {    94,    8,    3,    4,    3,    6,    9,    9,    3,    5,    9,    0,    6,    6,    0,    5,    5,    0,    0,    9,    4    },    // 2^312
    {    95,    1,    6,    6,    8,    7,    3,    9,    8,    7,    1,    8,    1,    3,    2,    1,    1,    0,    0,    1,    9    },    // 2^313
    {    95,    3,    3,    3,    7,    4,    7,    9,    7,    4,    3,    6,    2,    6,    4,    2,    2,    0,    0,    3,    7    },    // 2^314
    {    95,    6,    6,    7,    4,    9,    5,    9,    4,    8,    7,    2,    5,    2,    8,    4,    4,    0,    0,    7,    5    },    // 2^315
    {    96,    1,    3,    3,    4,    9,    9,    1,    8,    9,    7,    4,    5,    0,    5,    6,    8,    8,    0,    1,    5    },    // 2^316
    {    96,    2,    6,    6,    9,    9,    8,    3,    7,    9,    4,    9,    0,    1,    1,    3,    7,    6,    0,    3,    0    },    // 2^317
    {    96,    5,    3,    3,    9,    9,    6,    7,    5,    8,    9,    8,    0,    2,    2,    7,    5,    2,    0,    6,    0    },    // 2^318
    {    97,    1,    0,    6,    7,    9,    9,    3,    5,    1,    7,    9,    6,    0,    4,    5,    5,    0,    4,    1,    2    },    // 2^319
    {    97,    2,    1,    3,    5,    9,    8,    7,    0,    3,    5,    9,    2,    0,    9,    1,    0,    0,    8,    2,    4    },    // 2^320
    {    97,    4,    2,    7,    1,    9,    7,    4,    0,    7,    1,    8,    4,    1,    8,    2,    0,    1,    6,    4,    8    },    // 2^321
    {    97,    8,    5,    4,    3,    9,    4,    8,    1,    4,    3,    6,    8,    3,    6,    4,    0,    3,    2,    9,    6    },    // 2^322
    {    98,    1,    7,    0,    8,    7,    8,    9,    6,    2,    8,    7,    3,    6,    7,    2,    8,    0,    6,    5,    9    },    // 2^323
    {    98,    3,    4,    1,    7,    5,    7,    9,    2,    5,    7,    4,    7,    3,    4,    5,    6,    1,    3,    1,    8    },    // 2^324
    {    98,    6,    8,    3,    5,    1,    5,    8,    5,    1,    4,    9,    4,    6,    9,    1,    2,    2,    6,    3,    7    },    // 2^325
    {    99,    1,    3,    6,    7,    0,    3,    1,    7,    0,    2,    9,    8,    9,    3,    8,    2,    4,    5,    2,    7    },    // 2^326
    {    99,    2,    7,    3,    4,    0,    6,    3,    4,    0,    5,    9,    7,    8,    7,    6,    4,    9,    0,    5,    5    },    // 2^327
    {    99,    5,    4,    6,    8,    1,    2,    6,    8,    1,    1,    9,    5,    7,    5,    2,    9,    8,    1,    0,    9    },    // 2^328
    {    100,    1,    0,    9,    3,    6,    2,    5,    3,    6,    2,    3,    9,    1,    5,    0,    5,    9,    6,    2,    2    },    // 2^329
    {    100,    2,    1,    8,    7,    2,    5,    0,    7,    2,    4,    7,    8,    3,    0,    1,    1,    9,    2,    4,    4    },    // 2^330
    {    100,    4,    3,    7,    4,    5,    0,    1,    4,    4,    9,    5,    6,    6,    0,    2,    3,    8,    4,    8,    7    },    // 2^331
    {    100,    8,    7,    4,    9,    0,    0,    2,    8,    9,    9,    1,    3,    2,    0,    4,    7,    6,    9,    7,    5    },    // 2^332
    {    101,    1,    7,    4,    9,    8,    0,    0,    5,    7,    9,    8,    2,    6,    4,    0,    9,    5,    3,    9,    5    },    // 2^333
    {    101,    3,    4,    9,    9,    6,    0,    1,    1,    5,    9,    6,    5,    2,    8,    1,    9,    0,    7,    9,    0    },    // 2^334
    {    101,    6,    9,    9,    9,    2,    0,    2,    3,    1,    9,    3,    0,    5,    6,    3,    8,    1,    5,    8,    0    },    // 2^335
    {    102,    1,    3,    9,    9,    8,    4,    0,    4,    6,    3,    8,    6,    1,    1,    2,    7,    6,    3,    1,    6    },    // 2^336
    {    102,    2,    7,    9,    9,    6,    8,    0,    9,    2,    7,    7,    2,    2,    2,    5,    5,    2,    6,    3,    2    },    // 2^337
    {    102,    5,    5,    9,    9,    3,    6,    1,    8,    5,    5,    4,    4,    4,    5,    1,    0,    5,    2,    6,    4    },    // 2^338
    {    103,    1,    1,    1,    9,    8,    7,    2,    3,    7,    1,    0,    8,    8,    9,    0,    2,    1,    0,    5,    3    },    // 2^339
    {    103,    2,    2,    3,    9,    7,    4,    4,    7,    4,    2,    1,    7,    7,    8,    0,    4,    2,    1,    0,    6    },    // 2^340
    {    103,    4,    4,    7,    9,    4,    8,    9,    4,    8,    4,    3,    5,    5,    6,    0,    8,    4,    2,    1,    1    },    // 2^341
    {    103,    8,    9,    5,    8,    9,    7,    8,    9,    6,    8,    7,    1,    1,    2,    1,    6,    8,    4,    2,    2    },    // 2^342
    {    104,    1,    7,    9,    1,    7,    9,    5,    7,    9,    3,    7,    4,    2,    2,    4,    3,    3,    6,    8,    4    },    // 2^343
    {    104,    3,    5,    8,    3,    5,    9,    1,    5,    8,    7,    4,    8,    4,    4,    8,    6,    7,    3,    6,    9    },    // 2^344
    {    104,    7,    1,    6,    7,    1,    8,    3,    1,    7,    4,    9,    6,    8,    9,    7,    3,    4,    7,    3,    8    },    // 2^345
    {    105,    1,    4,    3,    3,    4,    3,    6,    6,    3,    4,    9,    9,    3,    7,    9,    4,    6,    9,    4,    8    },    // 2^346
    {    105,    2,    8,    6,    6,    8,    7,    3,    2,    6,    9,    9,    8,    7,    5,    8,    9,    3,    8,    9,    5    },    // 2^347
    {    105,    5,    7,    3,    3,    7,    4,    6,    5,    3,    9,    9,    7,    5,    1,    7,    8,    7,    7,    9,    0    },    // 2^348
    {    106,    1,    1,    4,    6,    7,    4,    9,    3,    0,    7,    9,    9,    5,    0,    3,    5,    7,    5,    5,    8    },    // 2^349
    {    106,    2,    2,    9,    3,    4,    9,    8,    6,    1,    5,    9,    9,    0,    0,    7,    1,    5,    1,    1,    6    },    // 2^350
    {    106,    4,    5,    8,    6,    9,    9,    7,    2,    3,    1,    9,    8,    0,    1,    4,    3,    0,    2,    3,    2    },    // 2^351
    {    106,    9,    1,    7,    3,    9,    9,    4,    4,    6,    3,    9,    6,    0,    2,    8,    6,    0,    4,    6,    4    },    // 2^352
    {    107,    1,    8,    3,    4,    7,    9,    8,    8,    9,    2,    7,    9,    2,    0,    5,    7,    2,    0,    9,    3    },    // 2^353
    {    107,    3,    6,    6,    9,    5,    9,    7,    7,    8,    5,    5,    8,    4,    1,    1,    4,    4,    1,    8,    6    },    // 2^354
    {    107,    7,    3,    3,    9,    1,    9,    5,    5,    7,    1,    1,    6,    8,    2,    2,    8,    8,    3,    7,    2    },    // 2^355
    {    108,    1,    4,    6,    7,    8,    3,    9,    1,    1,    4,    2,    3,    3,    6,    4,    5,    7,    6,    7,    4    },    // 2^356
    {    108,    2,    9,    3,    5,    6,    7,    8,    2,    2,    8,    4,    6,    7,    2,    9,    1,    5,    3,    4,    9    },    // 2^357
    {    108,    5,    8,    7,    1,    3,    5,    6,    4,    5,    6,    9,    3,    4,    5,    8,    3,    0,    6,    9,    7    },    // 2^358
    {    109,    1,    1,    7,    4,    2,    7,    1,    2,    9,    1,    3,    8,    6,    9,    1,    6,    6,    1,    3,    9    },    // 2^359
    {    109,    2,    3,    4,    8,    5,    4,    2,    5,    8,    2,    7,    7,    3,    8,    3,    3,    2,    2,    7,    9    },    // 2^360
    {    109,    4,    6,    9,    7,    0,    8,    5,    1,    6,    5,    5,    4,    7,    6,    6,    6,    4,    5,    5,    8    },    // 2^361
    {    109,    9,    3,    9,    4,    1,    7,    0,    3,    3,    1,    0,    9,    5,    3,    3,    2,    9,    1,    1,    6    },    // 2^362
    {    110,    1,    8,    7,    8,    8,    3,    4,    0,    6,    6,    2,    1,    9,    0,    6,    6,    5,    8,    2,    3    },    // 2^363
    {    110,    3,    7,    5,    7,    6,    6,    8,    1,    3,    2,    4,    3,    8,    1,    3,    3,    1,    6,    4,    6    },    // 2^364
    {    110,    7,    5,    1,    5,    3,    3,    6,    2,    6,    4,    8,    7,    6,    2,    6,    6,    3,    2,    9,    2    },    // 2^365
    {    111,    1,    5,    0,    3,    0,    6,    7,    2,    5,    2,    9,    7,    5,    2,    5,    3,    2,    6,    5,    8    },    // 2^366
    {    111,    3,    0,    0,    6,    1,    3,    4,    5,    0,    5,    9,    5,    0,    5,    0,    6,    5,    3,    1,    7    },    // 2^367
    {    111,    6,    0,    1,    2,    2,    6,    9,    0,    1,    1,    9,    0,    1,    0,    1,    3,    0,    6,    3,    4    },    // 2^368
    {    112,    1,    2,    0,    2,    4,    5,    3,    8,    0,    2,    3,    8,    0,    2,    0,    2,    6,    1,    2,    7    },    // 2^369
    {    112,    2,    4,    0,    4,    9,    0,    7,    6,    0,    4,    7,    6,    0,    4,    0,    5,    2,    2,    5,    4    },    // 2^370
    {    112,    4,    8,    0,    9,    8,    1,    5,    2,    0,    9,    5,    2,    0,    8,    1,    0,    4,    5,    0,    7    },    // 2^371
    {    112,    9,    6,    1,    9,    6,    3,    0,    4,    1,    9,    0,    4,    1,    6,    2,    0,    9,    0,    1,    4    },    // 2^372
    {    113,    1,    9,    2,    3,    9,    2,    6,    0,    8,    3,    8,    0,    8,    3,    2,    4,    1,    8,    0,    3    },    // 2^373
    {    113,    3,    8,    4,    7,    8,    5,    2,    1,    6,    7,    6,    1,    6,    6,    4,    8,    3,    6,    0,    6    },    // 2^374
    {    113,    7,    6,    9,    5,    7,    0,    4,    3,    3,    5,    2,    3,    3,    2,    9,    6,    7,    2,    1,    1    },    // 2^375
    {    114,    1,    5,    3,    9,    1,    4,    0,    8,    6,    7,    0,    4,    6,    6,    5,    9,    3,    4,    4,    2    },    // 2^376
    {    114,    3,    0,    7,    8,    2,    8,    1,    7,    3,    4,    0,    9,    3,    3,    1,    8,    6,    8,    8,    5    },    // 2^377
    {    114,    6,    1,    5,    6,    5,    6,    3,    4,    6,    8,    1,    8,    6,    6,    3,    7,    3,    7,    6,    9    },    // 2^378
    {    115,    1,    2,    3,    1,    3,    1,    2,    6,    9,    3,    6,    3,    7,    3,    2,    7,    4,    7,    5,    4    },    // 2^379
    {    115,    2,    4,    6,    2,    6,    2,    5,    3,    8,    7,    2,    7,    4,    6,    5,    4,    9,    5,    0,    8    },    // 2^380
    {    115,    4,    9,    2,    5,    2,    5,    0,    7,    7,    4,    5,    4,    9,    3,    0,    9,    9,    0,    1,    5    },    // 2^381
    {    115,    9,    8,    5,    0,    5,    0,    1,    5,    4,    9,    0,    9,    8,    6,    1,    9,    8,    0,    3,    1    },    // 2^382
    {    116,    1,    9,    7,    0,    1,    0,    0,    3,    0,    9,    8,    1,    9,    7,    2,    3,    9,    6,    0,    6    },    // 2^383
    {    116,    3,    9,    4,    0,    2,    0,    0,    6,    1,    9,    6,    3,    9,    4,    4,    7,    9,    2,    1,    2    },    // 2^384
    {    116,    7,    8,    8,    0,    4,    0,    1,    2,    3,    9,    2,    7,    8,    8,    9,    5,    8,    4,    2,    5    },    // 2^385
    {    117,    1,    5,    7,    6,    0,    8,    0,    2,    4,    7,    8,    5,    5,    7,    7,    9,    1,    6,    8,    5    },    // 2^386
    {    117,    3,    1,    5,    2,    1,    6,    0,    4,    9,    5,    7,    1,    1,    5,    5,    8,    3,    3,    7,    0    },    // 2^387
    {    117,    6,    3,    0,    4,    3,    2,    0,    9,    9,    1,    4,    2,    3,    1,    1,    6,    6,    7,    4,    0    },    // 2^388
    {    118,    1,    2,    6,    0,    8,    6,    4,    1,    9,    8,    2,    8,    4,    6,    2,    3,    3,    3,    4,    8    },    // 2^389
    {    118,    2,    5,    2,    1,    7,    2,    8,    3,    9,    6,    5,    6,    9,    2,    4,    6,    6,    6,    9,    6    },    // 2^390
    {    118,    5,    0,    4,    3,    4,    5,    6,    7,    9,    3,    1,    3,    8,    4,    9,    3,    3,    3,    9,    2    },    // 2^391
    {    119,    1,    0,    0,    8,    6,    9,    1,    3,    5,    8,    6,    2,    7,    6,    9,    8,    6,    6,    7,    8    },    // 2^392
    {    119,    2,    0,    1,    7,    3,    8,    2,    7,    1,    7,    2,    5,    5,    3,    9,    7,    3,    3,    5,    7    },    // 2^393
    {    119,    4,    0,    3,    4,    7,    6,    5,    4,    3,    4,    5,    1,    0,    7,    9,    4,    6,    7,    1,    3    },    // 2^394
    {    119,    8,    0,    6,    9,    5,    3,    0,    8,    6,    9,    0,    2,    1,    5,    8,    9,    3,    4,    2,    7    },    // 2^395
    {    120,    1,    6,    1,    3,    9,    0,    6,    1,    7,    3,    8,    0,    4,    3,    1,    7,    8,    6,    8,    5    },    // 2^396
    {    120,    3,    2,    2,    7,    8,    1,    2,    3,    4,    7,    6,    0,    8,    6,    3,    5,    7,    3,    7,    1    },    // 2^397
    {    120,    6,    4,    5,    5,    6,    2,    4,    6,    9,    5,    2,    1,    7,    2,    7,    1,    4,    7,    4,    1    },    // 2^398
    {    121,    1,    2,    9,    1,    1,    2,    4,    9,    3,    9,    0,    4,    3,    4,    5,    4,    2,    9,    4,    8    },    // 2^399
    {    121,    2,    5,    8,    2,    2,    4,    9,    8,    7,    8,    0,    8,    6,    9,    0,    8,    5,    8,    9,    7    },    // 2^400
    {    121,    5,    1,    6,    4,    4,    9,    9,    7,    5,    6,    1,    7,    3,    8,    1,    7,    1,    7,    9,    3    },    // 2^401
    {    122,    1,    0,    3,    2,    8,    9,    9,    9,    5,    1,    2,    3,    4,    7,    6,    3,    4,    3,    5,    9    },    // 2^402
    {    122,    2,    0,    6,    5,    7,    9,    9,    9,    0,    2,    4,    6,    9,    5,    2,    6,    8,    7,    1,    7    },    // 2^403
    {    122,    4,    1,    3,    1,    5,    9,    9,    8,    0,    4,    9,    3,    9,    0,    5,    3,    7,    4,    3,    4    },    // 2^404
    {    122,    8,    2,    6,    3,    1,    9,    9,    6,    0,    9,    8,    7,    8,    1,    0,    7,    4,    8,    6,    9    },    // 2^405
    {    123,    1,    6,    5,    2,    6,    3,    9,    9,    2,    1,    9,    7,    5,    6,    2,    1,    4,    9,    7,    4    },    // 2^406
    {    123,    3,    3,    0,    5,    2,    7,    9,    8,    4,    3,    9,    5,    1,    2,    4,    2,    9,    9,    4,    8    },    // 2^407
    {    123,    6,    6,    1,    0,    5,    5,    9,    6,    8,    7,    9,    0,    2,    4,    8,    5,    9,    8,    9,    5    },    // 2^408
    {    124,    1,    3,    2,    2,    1,    1,    1,    9,    3,    7,    5,    8,    0,    4,    9,    7,    1,    9,    7,    9    },    // 2^409
    {    124,    2,    6,    4,    4,    2,    2,    3,    8,    7,    5,    1,    6,    0,    9,    9,    4,    3,    9,    5,    8    },    // 2^410
    {    124,    5,    2,    8,    8,    4,    4,    7,    7,    5,    0,    3,    2,    1,    9,    8,    8,    7,    9,    1,    6    },    // 2^411
    {    125,    1,    0,    5,    7,    6,    8,    9,    5,    5,    0,    0,    6,    4,    3,    9,    7,    7,    5,    8,    3    },    // 2^412
    {    125,    2,    1,    1,    5,    3,    7,    9,    1,    0,    0,    1,    2,    8,    7,    9,    5,    5,    1,    6,    6    },    // 2^413
    {    125,    4,    2,    3,    0,    7,    5,    8,    2,    0,    0,    2,    5,    7,    5,    9,    1,    0,    3,    3,    3    },    // 2^414
    {    125,    8,    4,    6,    1,    5,    1,    6,    4,    0,    0,    5,    1,    5,    1,    8,    2,    0,    6,    6,    6    },    // 2^415
    {    126,    1,    6,    9,    2,    3,    0,    3,    2,    8,    0,    1,    0,    3,    0,    3,    6,    4,    1,    3,    3    },    // 2^416
    {    126,    3,    3,    8,    4,    6,    0,    6,    5,    6,    0,    2,    0,    6,    0,    7,    2,    8,    2,    6,    6    },    // 2^417
    {    126,    6,    7,    6,    9,    2,    1,    3,    1,    2,    0,    4,    1,    2,    1,    4,    5,    6,    5,    3,    3    },    // 2^418
    {    127,    1,    3,    5,    3,    8,    4,    2,    6,    2,    4,    0,    8,    2,    4,    2,    9,    1,    3,    0,    7    },    // 2^419
    {    127,    2,    7,    0,    7,    6,    8,    5,    2,    4,    8,    1,    6,    4,    8,    5,    8,    2,    6,    1,    3    },    // 2^420
    {    127,    5,    4,    1,    5,    3,    7,    0,    4,    9,    6,    3,    2,    9,    7,    1,    6,    5,    2,    2,    6    },    // 2^421
    {    128,    1,    0,    8,    3,    0,    7,    4,    0,    9,    9,    2,    6,    5,    9,    4,    3,    3,    0,    4,    5    },    // 2^422
    {    128,    2,    1,    6,    6,    1,    4,    8,    1,    9,    8,    5,    3,    1,    8,    8,    6,    6,    0,    9,    0    },    // 2^423
    {    128,    4,    3,    3,    2,    2,    9,    6,    3,    9,    7,    0,    6,    3,    7,    7,    3,    2,    1,    8,    1    },    // 2^424
    {    128,    8,    6,    6,    4,    5,    9,    2,    7,    9,    4,    1,    2,    7,    5,    4,    6,    4,    3,    6,    2    },    // 2^425
    {    129,    1,    7,    3,    2,    9,    1,    8,    5,    5,    8,    8,    2,    5,    5,    0,    9,    2,    8,    7,    2    },    // 2^426
    {    129,    3,    4,    6,    5,    8,    3,    7,    1,    1,    7,    6,    5,    1,    0,    1,    8,    5,    7,    4,    5    },    // 2^427
    {    129,    6,    9,    3,    1,    6,    7,    4,    2,    3,    5,    3,    0,    2,    0,    3,    7,    1,    4,    8,    9    },    // 2^428
    {    130,    1,    3,    8,    6,    3,    3,    4,    8,    4,    7,    0,    6,    0,    4,    0,    7,    4,    2,    9,    8    },    // 2^429
    {    130,    2,    7,    7,    2,    6,    6,    9,    6,    9,    4,    1,    2,    0,    8,    1,    4,    8,    5,    9,    6    },    // 2^430
    {    130,    5,    5,    4,    5,    3,    3,    9,    3,    8,    8,    2,    4,    1,    6,    2,    9,    7,    1,    9,    2    },    // 2^431
    {    131,    1,    1,    0,    9,    0,    6,    7,    8,    7,    7,    6,    4,    8,    3,    2,    5,    9,    4,    3,    8    },    // 2^432
    {    131,    2,    2,    1,    8,    1,    3,    5,    7,    5,    5,    2,    9,    6,    6,    5,    1,    8,    8,    7,    7    },    // 2^433
    {    131,    4,    4,    3,    6,    2,    7,    1,    5,    1,    0,    5,    9,    3,    3,    0,    3,    7,    7,    5,    3    },    // 2^434
    {    131,    8,    8,    7,    2,    5,    4,    3,    0,    2,    1,    1,    8,    6,    6,    0,    7,    5,    5,    0,    7    },    // 2^435
    {    132,    1,    7,    7,    4,    5,    0,    8,    6,    0,    4,    2,    3,    7,    3,    2,    1,    5,    1,    0,    1    },    // 2^436
    {    132,    3,    5,    4,    9,    0,    1,    7,    2,    0,    8,    4,    7,    4,    6,    4,    3,    0,    2,    0,    3    },    // 2^437
    {    132,    7,    0,    9,    8,    0,    3,    4,    4,    1,    6,    9,    4,    9,    2,    8,    6,    0,    4,    0,    5    },    // 2^438
    {    133,    1,    4,    1,    9,    6,    0,    6,    8,    8,    3,    3,    8,    9,    8,    5,    7,    2,    0,    8,    1    },    // 2^439
    {    133,    2,    8,    3,    9,    2,    1,    3,    7,    6,    6,    7,    7,    9,    7,    1,    4,    4,    1,    6,    2    },    // 2^440
    {    133,    5,    6,    7,    8,    4,    2,    7,    5,    3,    3,    5,    5,    9,    4,    2,    8,    8,    3,    2,    4    },    // 2^441
    {    134,    1,    1,    3,    5,    6,    8,    5,    5,    0,    6,    7,    1,    1,    8,    8,    5,    7,    6,    6,    5    },    // 2^442
    {    134,    2,    2,    7,    1,    3,    7,    1,    0,    1,    3,    4,    2,    3,    7,    7,    1,    5,    3,    3,    0    },    // 2^443
    {    134,    4,    5,    4,    2,    7,    4,    2,    0,    2,    6,    8,    4,    7,    5,    4,    3,    0,    6,    5,    9    },    // 2^444
    {    134,    9,    0,    8,    5,    4,    8,    4,    0,    5,    3,    6,    9,    5,    0,    8,    6,    1,    3,    1,    9    },    // 2^445
    {    135,    1,    8,    1,    7,    0,    9,    6,    8,    1,    0,    7,    3,    9,    0,    1,    7,    2,    2,    6,    4    },    // 2^446
    {    135,    3,    6,    3,    4,    1,    9,    3,    6,    2,    1,    4,    7,    8,    0,    3,    4,    4,    5,    2,    7    },    // 2^447
    {    135,    7,    2,    6,    8,    3,    8,    7,    2,    4,    2,    9,    5,    6,    0,    6,    8,    9,    0,    5,    5    },    // 2^448
    {    136,    1,    4,    5,    3,    6,    7,    7,    4,    4,    8,    5,    9,    1,    2,    1,    3,    7,    8,    1,    1    },    // 2^449
    {    136,    2,    9,    0,    7,    3,    5,    4,    8,    9,    7,    1,    8,    2,    4,    2,    7,    5,    6,    2,    2    },    // 2^450
    {    136,    5,    8,    1,    4,    7,    0,    9,    7,    9,    4,    3,    6,    4,    8,    5,    5,    1,    2,    4,    4    },    // 2^451
    {    137,    1,    1,    6,    2,    9,    4,    1,    9,    5,    8,    8,    7,    2,    9,    7,    1,    0,    2,    4,    9    },    // 2^452
    {    137,    2,    3,    2,    5,    8,    8,    3,    9,    1,    7,    7,    4,    5,    9,    4,    2,    0,    4,    9,    8    },    // 2^453
    {    137,    4,    6,    5,    1,    7,    6,    7,    8,    3,    5,    4,    9,    1,    8,    8,    4,    0,    9,    9,    5    },    // 2^454
    {    137,    9,    3,    0,    3,    5,    3,    5,    6,    7,    0,    9,    8,    3,    7,    6,    8,    1,    9,    9,    0    },    // 2^455
    {    138,    1,    8,    6,    0,    7,    0,    7,    1,    3,    4,    1,    9,    6,    7,    5,    3,    6,    3,    9,    8    },    // 2^456
    {    138,    3,    7,    2,    1,    4,    1,    4,    2,    6,    8,    3,    9,    3,    5,    0,    7,    2,    7,    9,    6    },    // 2^457
    {    138,    7,    4,    4,    2,    8,    2,    8,    5,    3,    6,    7,    8,    7,    0,    1,    4,    5,    5,    9,    2    },    // 2^458
    {    139,    1,    4,    8,    8,    5,    6,    5,    7,    0,    7,    3,    5,    7,    4,    0,    2,    9,    1,    1,    8    },    // 2^459
    {    139,    2,    9,    7,    7,    1,    3,    1,    4,    1,    4,    7,    1,    4,    8,    0,    5,    8,    2,    3,    7    },    // 2^460
    {    139,    5,    9,    5,    4,    2,    6,    2,    8,    2,    9,    4,    2,    9,    6,    1,    1,    6,    4,    7,    4    },    // 2^461
    {    140,    1,    1,    9,    0,    8,    5,    2,    5,    6,    5,    8,    8,    5,    9,    2,    2,    3,    2,    9,    5    },    // 2^462
    {    140,    2,    3,    8,    1,    7,    0,    5,    1,    3,    1,    7,    7,    1,    8,    4,    4,    6,    5,    9,    0    },    // 2^463
    {    140,    4,    7,    6,    3,    4,    1,    0,    2,    6,    3,    5,    4,    3,    6,    8,    9,    3,    1,    7,    9    },    // 2^464
    {    140,    9,    5,    2,    6,    8,    2,    0,    5,    2,    7,    0,    8,    7,    3,    7,    8,    6,    3,    5,    8    },    // 2^465
    {    141,    1,    9,    0,    5,    3,    6,    4,    1,    0,    5,    4,    1,    7,    4,    7,    5,    7,    2,    7,    2    },    // 2^466
    {    141,    3,    8,    1,    0,    7,    2,    8,    2,    1,    0,    8,    3,    4,    9,    5,    1,    4,    5,    4,    3    },    // 2^467
    {    141,    7,    6,    2,    1,    4,    5,    6,    4,    2,    1,    6,    6,    9,    9,    0,    2,    9,    0,    8,    6    },    // 2^468
    {    142,    1,    5,    2,    4,    2,    9,    1,    2,    8,    4,    3,    3,    3,    9,    8,    0,    5,    8,    1,    7    },    // 2^469
    {    142,    3,    0,    4,    8,    5,    8,    2,    5,    6,    8,    6,    6,    7,    9,    6,    1,    1,    6,    3,    5    },    // 2^470
    {    142,    6,    0,    9,    7,    1,    6,    5,    1,    3,    7,    3,    3,    5,    9,    2,    2,    3,    2,    6,    9    },    // 2^471
    {    143,    1,    2,    1,    9,    4,    3,    3,    0,    2,    7,    4,    6,    7,    1,    8,    4,    4,    6,    5,    4    },    // 2^472
    {    143,    2,    4,    3,    8,    8,    6,    6,    0,    5,    4,    9,    3,    4,    3,    6,    8,    9,    3,    0,    8    },    // 2^473
    {    143,    4,    8,    7,    7,    7,    3,    2,    1,    0,    9,    8,    6,    8,    7,    3,    7,    8,    6,    1,    5    },    // 2^474
    {    143,    9,    7,    5,    5,    4,    6,    4,    2,    1,    9,    7,    3,    7,    4,    7,    5,    7,    2,    3,    1    },    // 2^475
    {    144,    1,    9,    5,    1,    0,    9,    2,    8,    4,    3,    9,    4,    7,    4,    9,    5,    1,    4,    4,    6    },    // 2^476
    {    144,    3,    9,    0,    2,    1,    8,    5,    6,    8,    7,    8,    9,    4,    9,    9,    0,    2,    8,    9,    2    },    // 2^477
    {    144,    7,    8,    0,    4,    3,    7,    1,    3,    7,    5,    7,    8,    9,    9,    8,    0,    5,    7,    8,    5    },    // 2^478
    {    145,    1,    5,    6,    0,    8,    7,    4,    2,    7,    5,    1,    5,    7,    9,    9,    6,    1,    1,    5,    7    },    // 2^479
    {    145,    3,    1,    2,    1,    7,    4,    8,    5,    5,    0,    3,    1,    5,    9,    9,    2,    2,    3,    1,    4    },    // 2^480
    {    145,    6,    2,    4,    3,    4,    9,    7,    1,    0,    0,    6,    3,    1,    9,    8,    4,    4,    6,    2,    8    },    // 2^481
    {    146,    1,    2,    4,    8,    6,    9,    9,    4,    2,    0,    1,    2,    6,    3,    9,    6,    8,    9,    2,    6    },    // 2^482
    {    146,    2,    4,    9,    7,    3,    9,    8,    8,    4,    0,    2,    5,    2,    7,    9,    3,    7,    8,    5,    1    },    // 2^483
    {    146,    4,    9,    9,    4,    7,    9,    7,    6,    8,    0,    5,    0,    5,    5,    8,    7,    5,    7,    0,    2    },    // 2^484
    {    146,    9,    9,    8,    9,    5,    9,    5,    3,    6,    1,    0,    1,    1,    1,    7,    5,    1,    4,    0,    4    },    // 2^485
    {    147,    1,    9,    9,    7,    9,    1,    9,    0,    7,    2,    2,    0,    2,    2,    3,    5,    0,    2,    8,    1    },    // 2^486
    {    147,    3,    9,    9,    5,    8,    3,    8,    1,    4,    4,    4,    0,    4,    4,    7,    0,    0,    5,    6,    2    },    // 2^487
    {    147,    7,    9,    9,    1,    6,    7,    6,    2,    8,    8,    8,    0,    8,    9,    4,    0,    1,    1,    2,    3    },    // 2^488
    {    148,    1,    5,    9,    8,    3,    3,    5,    2,    5,    7,    7,    6,    1,    7,    8,    8,    0,    2,    2,    5    },    // 2^489
    {    148,    3,    1,    9,    6,    6,    7,    0,    5,    1,    5,    5,    2,    3,    5,    7,    6,    0,    4,    4,    9    },    // 2^490
    {    148,    6,    3,    9,    3,    3,    4,    1,    0,    3,    1,    0,    4,    7,    1,    5,    2,    0,    8,    9,    9    },    // 2^491
    {    149,    1,    2,    7,    8,    6,    6,    8,    2,    0,    6,    2,    0,    9,    4,    3,    0,    4,    1,    8,    0    },    // 2^492
    {    149,    2,    5,    5,    7,    3,    3,    6,    4,    1,    2,    4,    1,    8,    8,    6,    0,    8,    3,    5,    9    },    // 2^493
    {    149,    5,    1,    1,    4,    6,    7,    2,    8,    2,    4,    8,    3,    7,    7,    2,    1,    6,    7,    1,    9    },    // 2^494
    {    150,    1,    0,    2,    2,    9,    3,    4,    5,    6,    4,    9,    6,    7,    5,    4,    4,    3,    3,    4,    4    },    // 2^495
    {    150,    2,    0,    4,    5,    8,    6,    9,    1,    2,    9,    9,    3,    5,    0,    8,    8,    6,    6,    8,    8    },    // 2^496
    {    150,    4,    0,    9,    1,    7,    3,    8,    2,    5,    9,    8,    7,    0,    1,    7,    7,    3,    3,    7,    5    },    // 2^497
    {    150,    8,    1,    8,    3,    4,    7,    6,    5,    1,    9,    7,    4,    0,    3,    5,    4,    6,    7,    5,    0    },    // 2^498
    {    151,    1,    6,    3,    6,    6,    9,    5,    3,    0,    3,    9,    4,    8,    0,    7,    0,    9,    3,    5,    0    },    // 2^499
    {    151,    3,    2,    7,    3,    3,    9,    0,    6,    0,    7,    8,    9,    6,    1,    4,    1,    8,    7,    0,    0    },    // 2^500
    {    151,    6,    5,    4,    6,    7,    8,    1,    2,    1,    5,    7,    9,    2,    2,    8,    3,    7,    4,    0,    0    },    // 2^501
    {    152,    1,    3,    0,    9,    3,    5,    6,    2,    4,    3,    1,    5,    8,    4,    5,    6,    7,    4,    8,    0    },    // 2^502
    {    152,    2,    6,    1,    8,    7,    1,    2,    4,    8,    6,    3,    1,    6,    9,    1,    3,    4,    9,    6,    0    },    // 2^503
    {    152,    5,    2,    3,    7,    4,    2,    4,    9,    7,    2,    6,    3,    3,    8,    2,    6,    9,    9,    2,    0    },    // 2^504
    {    153,    1,    0,    4,    7,    4,    8,    4,    9,    9,    4,    5,    2,    6,    7,    6,    5,    3,    9,    8,    4    },    // 2^505
    {    153,    2,    0,    9,    4,    9,    6,    9,    9,    8,    9,    0,    5,    3,    5,    3,    0,    7,    9,    6,    8    },    // 2^506
    {    153,    4,    1,    8,    9,    9,    3,    9,    9,    7,    8,    1,    0,    7,    0,    6,    1,    5,    9,    3,    6    },    // 2^507
    {    153,    8,    3,    7,    9,    8,    7,    9,    9,    5,    6,    2,    1,    4,    1,    2,    3,    1,    8,    7,    2    },    // 2^508
    {    154,    1,    6,    7,    5,    9,    7,    5,    9,    9,    1,    2,    4,    2,    8,    2,    4,    6,    3,    7,    4    },    // 2^509
    {    154,    3,    3,    5,    1,    9,    5,    1,    9,    8,    2,    4,    8,    5,    6,    4,    9,    2,    7,    4,    9    },    // 2^510
    {    154,    6,    7,    0,    3,    9,    0,    3,    9,    6,    4,    9,    7,    1,    2,    9,    8,    5,    4,    9,    8    },    // 2^511
    {    155,    1,    3,    4,    0,    7,    8,    0,    7,    9,    2,    9,    9,    4,    2,    5,    9,    7,    0,    0,    0    },    // 2^512
    {    155,    2,    6,    8,    1,    5,    6,    1,    5,    8,    5,    9,    8,    8,    5,    1,    9,    4,    1,    9,    9    },    // 2^513
    {    155,    5,    3,    6,    3,    1,    2,    3,    1,    7,    1,    9,    7,    7,    0,    3,    8,    8,    3,    9,    8    },    // 2^514
    {    156,    1,    0,    7,    2,    6,    2,    4,    6,    3,    4,    3,    9,    5,    4,    0,    7,    7,    6,    8,    0    },    // 2^515
    {    156,    2,    1,    4,    5,    2,    4,    9,    2,    6,    8,    7,    9,    0,    8,    1,    5,    5,    3,    5,    9    },    // 2^516
    {    156,    4,    2,    9,    0,    4,    9,    8,    5,    3,    7,    5,    8,    1,    6,    3,    1,    0,    7,    1,    9    },    // 2^517
    {    156,    8,    5,    8,    0,    9,    9,    7,    0,    7,    5,    1,    6,    3,    2,    6,    2,    1,    4,    3,    7    },    // 2^518
    {    157,    1,    7,    1,    6,    1,    9,    9,    4,    1,    5,    0,    3,    2,    6,    5,    2,    4,    2,    8,    7    },    // 2^519
    {    157,    3,    4,    3,    2,    3,    9,    8,    8,    3,    0,    0,    6,    5,    3,    0,    4,    8,    5,    7,    5    },    // 2^520
    {    157,    6,    8,    6,    4,    7,    9,    7,    6,    6,    0,    1,    3,    0,    6,    0,    9,    7,    1,    5,    0    },    // 2^521
    {    158,    1,    3,    7,    2,    9,    5,    9,    5,    3,    2,    0,    2,    6,    1,    2,    1,    9,    4,    3,    0    },    // 2^522
    {    158,    2,    7,    4,    5,    9,    1,    9,    0,    6,    4,    0,    5,    2,    2,    4,    3,    8,    8,    6,    0    },    // 2^523
    {    158,    5,    4,    9,    1,    8,    3,    8,    1,    2,    8,    1,    0,    4,    4,    8,    7,    7,    7,    2,    0    },    // 2^524
    {    159,    1,    0,    9,    8,    3,    6,    7,    6,    2,    5,    6,    2,    0,    8,    9,    7,    5,    5,    4,    4    },    // 2^525
    {    159,    2,    1,    9,    6,    7,    3,    5,    2,    5,    1,    2,    4,    1,    7,    9,    5,    1,    0,    8,    8    },    // 2^526
    {    159,    4,    3,    9,    3,    4,    7,    0,    5,    0,    2,    4,    8,    3,    5,    9,    0,    2,    1,    7,    6    },    // 2^527
    {    159,    8,    7,    8,    6,    9,    4,    1,    0,    0,    4,    9,    6,    7,    1,    8,    0,    4,    3,    5,    2    },    // 2^528
    {    160,    1,    7,    5,    7,    3,    8,    8,    2,    0,    0,    9,    9,    3,    4,    3,    6,    0,    8,    7,    0    },    // 2^529
    {    160,    3,    5,    1,    4,    7,    7,    6,    4,    0,    1,    9,    8,    6,    8,    7,    2,    1,    7,    4,    1    },    // 2^530
    {    160,    7,    0,    2,    9,    5,    5,    2,    8,    0,    3,    9,    7,    3,    7,    4,    4,    3,    4,    8,    1    },    // 2^531
    {    161,    1,    4,    0,    5,    9,    1,    0,    5,    6,    0,    7,    9,    4,    7,    4,    8,    8,    6,    9,    6    },    // 2^532
    {    161,    2,    8,    1,    1,    8,    2,    1,    1,    2,    1,    5,    8,    9,    4,    9,    7,    7,    3,    9,    3    },    // 2^533
    {    161,    5,    6,    2,    3,    6,    4,    2,    2,    4,    3,    1,    7,    8,    9,    9,    5,    4,    7,    8,    5    },    // 2^534
    {    162,    1,    1,    2,    4,    7,    2,    8,    4,    4,    8,    6,    3,    5,    7,    9,    9,    0,    9,    5,    7    },    // 2^535
    {    162,    2,    2,    4,    9,    4,    5,    6,    8,    9,    7,    2,    7,    1,    5,    9,    8,    1,    9,    1,    4    },    // 2^536
    {    162,    4,    4,    9,    8,    9,    1,    3,    7,    9,    4,    5,    4,    3,    1,    9,    6,    3,    8,    2,    8    },    // 2^537
    {    162,    8,    9,    9,    7,    8,    2,    7,    5,    8,    9,    0,    8,    6,    3,    9,    2,    7,    6,    5,    6    },    // 2^538
    {    163,    1,    7,    9,    9,    5,    6,    5,    5,    1,    7,    8,    1,    7,    2,    7,    8,    5,    5,    3,    1    },    // 2^539
    {    163,    3,    5,    9,    9,    1,    3,    1,    0,    3,    5,    6,    3,    4,    5,    5,    7,    1,    0,    6,    2    },    // 2^540
    {    163,    7,    1,    9,    8,    2,    6,    2,    0,    7,    1,    2,    6,    9,    1,    1,    4,    2,    1,    2,    5    },    // 2^541
    {    164,    1,    4,    3,    9,    6,    5,    2,    4,    1,    4,    2,    5,    3,    8,    2,    2,    8,    4,    2,    5    },    // 2^542
    {    164,    2,    8,    7,    9,    3,    0,    4,    8,    2,    8,    5,    0,    7,    6,    4,    5,    6,    8,    5,    0    },    // 2^543
    {    164,    5,    7,    5,    8,    6,    0,    9,    6,    5,    7,    0,    1,    5,    2,    9,    1,    3,    6,    0,    0    },    // 2^544
    {    165,    1,    1,    5,    1,    7,    2,    1,    9,    3,    1,    4,    0,    3,    0,    5,    8,    2,    7,    4,    0    },    // 2^545
    {    165,    2,    3,    0,    3,    4,    4,    3,    8,    6,    2,    8,    0,    6,    1,    1,    6,    5,    4,    8,    0    },    // 2^546
    {    165,    4,    6,    0,    6,    8,    8,    7,    7,    2,    5,    6,    1,    2,    2,    3,    3,    0,    9,    6,    0    },    // 2^547
    {    165,    9,    2,    1,    3,    7,    7,    5,    4,    5,    1,    2,    2,    4,    4,    6,    6,    1,    9,    2,    0    },    // 2^548
    {    166,    1,    8,    4,    2,    7,    5,    5,    0,    9,    0,    2,    4,    4,    8,    9,    3,    2,    3,    8,    4    },    // 2^549
    {    166,    3,    6,    8,    5,    5,    1,    0,    1,    8,    0,    4,    8,    9,    7,    8,    6,    4,    7,    6,    8    },    // 2^550
    {    166,    7,    3,    7,    1,    0,    2,    0,    3,    6,    0,    9,    7,    9,    5,    7,    2,    9,    5,    3,    6    },    // 2^551
    {    167,    1,    4,    7,    4,    2,    0,    4,    0,    7,    2,    1,    9,    5,    9,    1,    4,    5,    9,    0,    7    },    // 2^552
    {    167,    2,    9,    4,    8,    4,    0,    8,    1,    4,    4,    3,    9,    1,    8,    2,    9,    1,    8,    1,    4    },    // 2^553
    {    167,    5,    8,    9,    6,    8,    1,    6,    2,    8,    8,    7,    8,    3,    6,    5,    8,    3,    6,    2,    9    },    // 2^554
    {    168,    1,    1,    7,    9,    3,    6,    3,    2,    5,    7,    7,    5,    6,    7,    3,    1,    6,    7,    2,    6    },    // 2^555
    {    168,    2,    3,    5,    8,    7,    2,    6,    5,    1,    5,    5,    1,    3,    4,    6,    3,    3,    4,    5,    2    },    // 2^556
    {    168,    4,    7,    1,    7,    4,    5,    3,    0,    3,    1,    0,    2,    6,    9,    2,    6,    6,    9,    0,    3    },    // 2^557
    {    168,    9,    4,    3,    4,    9,    0,    6,    0,    6,    2,    0,    5,    3,    8,    5,    3,    3,    8,    0,    6    },    // 2^558
    {    169,    1,    8,    8,    6,    9,    8,    1,    2,    1,    2,    4,    1,    0,    7,    7,    0,    6,    7,    6,    1    },    // 2^559
    {    169,    3,    7,    7,    3,    9,    6,    2,    4,    2,    4,    8,    2,    1,    5,    4,    1,    3,    5,    2,    2    },    // 2^560
    {    169,    7,    5,    4,    7,    9,    2,    4,    8,    4,    9,    6,    4,    3,    0,    8,    2,    7,    0,    4,    5    },    // 2^561
    {    170,    1,    5,    0,    9,    5,    8,    4,    9,    6,    9,    9,    2,    8,    6,    1,    6,    5,    4,    0,    9    },    // 2^562
    {    170,    3,    0,    1,    9,    1,    6,    9,    9,    3,    9,    8,    5,    7,    2,    3,    3,    0,    8,    1,    8    },    // 2^563
    {    170,    6,    0,    3,    8,    3,    3,    9,    8,    7,    9,    7,    1,    4,    4,    6,    6,    1,    6,    3,    6    },    // 2^564
    {    171,    1,    2,    0,    7,    6,    6,    7,    9,    7,    5,    9,    4,    2,    8,    9,    3,    2,    3,    2,    7    },    // 2^565
    {    171,    2,    4,    1,    5,    3,    3,    5,    9,    5,    1,    8,    8,    5,    7,    8,    6,    4,    6,    5,    4    },    // 2^566
    {    171,    4,    8,    3,    0,    6,    7,    1,    9,    0,    3,    7,    7,    1,    5,    7,    2,    9,    3,    0,    9    },    // 2^567
    {    171,    9,    6,    6,    1,    3,    4,    3,    8,    0,    7,    5,    4,    3,    1,    4,    5,    8,    6,    1,    7    },    // 2^568
    {    172,    1,    9,    3,    2,    2,    6,    8,    7,    6,    1,    5,    0,    8,    6,    2,    9,    1,    7,    2,    3    },    // 2^569
    {    172,    3,    8,    6,    4,    5,    3,    7,    5,    2,    3,    0,    1,    7,    2,    5,    8,    3,    4,    4,    7    },    // 2^570
    {    172,    7,    7,    2,    9,    0,    7,    5,    0,    4,    6,    0,    3,    4,    5,    1,    6,    6,    8,    9,    4    },    // 2^571
    {    173,    1,    5,    4,    5,    8,    1,    5,    0,    0,    9,    2,    0,    6,    9,    0,    3,    3,    3,    7,    9    },    // 2^572
    {    173,    3,    0,    9,    1,    6,    3,    0,    0,    1,    8,    4,    1,    3,    8,    0,    6,    6,    7,    5,    8    },    // 2^573
    {    173,    6,    1,    8,    3,    2,    6,    0,    0,    3,    6,    8,    2,    7,    6,    1,    3,    3,    5,    1,    5    },    // 2^574
    {    174,    1,    2,    3,    6,    6,    5,    2,    0,    0,    7,    3,    6,    5,    5,    2,    2,    6,    7,    0,    3    },    // 2^575
    {    174,    2,    4,    7,    3,    3,    0,    4,    0,    1,    4,    7,    3,    1,    0,    4,    5,    3,    4,    0,    6    },    // 2^576
    {    174,    4,    9,    4,    6,    6,    0,    8,    0,    2,    9,    4,    6,    2,    0,    9,    0,    6,    8,    1,    2    },    // 2^577
    {    174,    9,    8,    9,    3,    2,    1,    6,    0,    5,    8,    9,    2,    4,    1,    8,    1,    3,    6,    2,    4    },    // 2^578
    {    175,    1,    9,    7,    8,    6,    4,    3,    2,    1,    1,    7,    8,    4,    8,    3,    6,    2,    7,    2,    5    },    // 2^579
    {    175,    3,    9,    5,    7,    2,    8,    6,    4,    2,    3,    5,    6,    9,    6,    7,    2,    5,    4,    5,    0    },    // 2^580
    {    175,    7,    9,    1,    4,    5,    7,    2,    8,    4,    7,    1,    3,    9,    3,    4,    5,    0,    8,    9,    9    },    // 2^581
    {    176,    1,    5,    8,    2,    9,    1,    4,    5,    6,    9,    4,    2,    7,    8,    6,    9,    0,    1,    8,    0    },    // 2^582
    {    176,    3,    1,    6,    5,    8,    2,    9,    1,    3,    8,    8,    5,    5,    7,    3,    8,    0,    3,    6,    0    },    // 2^583
    {    176,    6,    3,    3,    1,    6,    5,    8,    2,    7,    7,    7,    1,    1,    4,    7,    6,    0,    7,    1,    9    },    // 2^584
    {    177,    1,    2,    6,    6,    3,    3,    1,    6,    5,    5,    5,    4,    2,    2,    9,    5,    2,    1,    4,    4    },    // 2^585
    {    177,    2,    5,    3,    2,    6,    6,    3,    3,    1,    1,    0,    8,    4,    5,    9,    0,    4,    2,    8,    8    },    // 2^586
    {    177,    5,    0,    6,    5,    3,    2,    6,    6,    2,    2,    1,    6,    9,    1,    8,    0,    8,    5,    7,    6    },    // 2^587
    {    178,    1,    0,    1,    3,    0,    6,    5,    3,    2,    4,    4,    3,    3,    8,    3,    6,    1,    7,    1,    5    },    // 2^588
    {    178,    2,    0,    2,    6,    1,    3,    0,    6,    4,    8,    8,    6,    7,    6,    7,    2,    3,    4,    3,    0    },    // 2^589
    {    178,    4,    0,    5,    2,    2,    6,    1,    2,    9,    7,    7,    3,    5,    3,    4,    4,    6,    8,    6,    0    },    // 2^590
    {    178,    8,    1,    0,    4,    5,    2,    2,    5,    9,    5,    4,    7,    0,    6,    8,    9,    3,    7,    2,    1    },    // 2^591
    {    179,    1,    6,    2,    0,    9,    0,    4,    5,    1,    9,    0,    9,    4,    1,    3,    7,    8,    7,    4,    4    },    // 2^592
    {    179,    3,    2,    4,    1,    8,    0,    9,    0,    3,    8,    1,    8,    8,    2,    7,    5,    7,    4,    8,    8    },    // 2^593
    {    179,    6,    4,    8,    3,    6,    1,    8,    0,    7,    6,    3,    7,    6,    5,    5,    1,    4,    9,    7,    7    },    // 2^594
    {    180,    1,    2,    9,    6,    7,    2,    3,    6,    1,    5,    2,    7,    5,    3,    1,    0,    2,    9,    9,    5    },    // 2^595
    {    180,    2,    5,    9,    3,    4,    4,    7,    2,    3,    0,    5,    5,    0,    6,    2,    0,    5,    9,    9,    1    },    // 2^596
    {    180,    5,    1,    8,    6,    8,    9,    4,    4,    6,    1,    1,    0,    1,    2,    4,    1,    1,    9,    8,    1    },    // 2^597
    {    181,    1,    0,    3,    7,    3,    7,    8,    8,    9,    2,    2,    2,    0,    2,    4,    8,    2,    3,    9,    6    },    // 2^598
    {    181,    2,    0,    7,    4,    7,    5,    7,    7,    8,    4,    4,    4,    0,    4,    9,    6,    4,    7,    9,    3    },    // 2^599
    {    181,    4,    1,    4,    9,    5,    1,    5,    5,    6,    8,    8,    8,    0,    9,    9,    2,    9,    5,    8,    5    },    // 2^600
    {    181,    8,    2,    9,    9,    0,    3,    1,    1,    3,    7,    7,    6,    1,    9,    8,    5,    9,    1,    7,    0    },    // 2^601
    {    182,    1,    6,    5,    9,    8,    0,    6,    2,    2,    7,    5,    5,    2,    3,    9,    7,    1,    8,    3,    4    },    // 2^602
    {    182,    3,    3,    1,    9,    6,    1,    2,    4,    5,    5,    1,    0,    4,    7,    9,    4,    3,    6,    6,    8    },    // 2^603
    {    182,    6,    6,    3,    9,    2,    2,    4,    9,    1,    0,    2,    0,    9,    5,    8,    8,    7,    3,    3,    6    },    // 2^604
    {    183,    1,    3,    2,    7,    8,    4,    4,    9,    8,    2,    0,    4,    1,    9,    1,    7,    7,    4,    6,    7    },    // 2^605
    {    183,    2,    6,    5,    5,    6,    8,    9,    9,    6,    4,    0,    8,    3,    8,    3,    5,    4,    9,    3,    4    },    // 2^606
    {    183,    5,    3,    1,    1,    3,    7,    9,    9,    2,    8,    1,    6,    7,    6,    7,    0,    9,    8,    6,    9    },    // 2^607
    {    184,    1,    0,    6,    2,    2,    7,    5,    9,    8,    5,    6,    3,    3,    5,    3,    4,    1,    9,    7,    4    },    // 2^608
    {    184,    2,    1,    2,    4,    5,    5,    1,    9,    7,    1,    2,    6,    7,    0,    6,    8,    3,    9,    4,    8    },    // 2^609
    {    184,    4,    2,    4,    9,    1,    0,    3,    9,    4,    2,    5,    3,    4,    1,    3,    6,    7,    8,    9,    5    },    // 2^610
    {    184,    8,    4,    9,    8,    2,    0,    7,    8,    8,    5,    0,    6,    8,    2,    7,    3,    5,    7,    9,    0    },    // 2^611
    {    185,    1,    6,    9,    9,    6,    4,    1,    5,    7,    7,    0,    1,    3,    6,    5,    4,    7,    1,    5,    8    },    // 2^612
    {    185,    3,    3,    9,    9,    2,    8,    3,    1,    5,    4,    0,    2,    7,    3,    0,    9,    4,    3,    1,    6    },    // 2^613
    {    185,    6,    7,    9,    8,    5,    6,    6,    3,    0,    8,    0,    5,    4,    6,    1,    8,    8,    6,    3,    2    },    // 2^614
    {    186,    1,    3,    5,    9,    7,    1,    3,    2,    6,    1,    6,    1,    0,    9,    2,    3,    7,    7,    2,    6    },    // 2^615
    {    186,    2,    7,    1,    9,    4,    2,    6,    5,    2,    3,    2,    2,    1,    8,    4,    7,    5,    4,    5,    3    },    // 2^616
    {    186,    5,    4,    3,    8,    8,    5,    3,    0,    4,    6,    4,    4,    3,    6,    9,    5,    0,    9,    0,    6    },    // 2^617
    {    187,    1,    0,    8,    7,    7,    7,    0,    6,    0,    9,    2,    8,    8,    7,    3,    9,    0,    1,    8,    1    },    // 2^618
    {    187,    2,    1,    7,    5,    5,    4,    1,    2,    1,    8,    5,    7,    7,    4,    7,    8,    0,    3,    6,    2    },    // 2^619
    {    187,    4,    3,    5,    1,    0,    8,    2,    4,    3,    7,    1,    5,    4,    9,    5,    6,    0,    7,    2,    5    },    // 2^620
    {    187,    8,    7,    0,    2,    1,    6,    4,    8,    7,    4,    3,    0,    9,    9,    1,    2,    1,    4,    4,    9    },    // 2^621
    {    188,    1,    7,    4,    0,    4,    3,    2,    9,    7,    4,    8,    6,    1,    9,    8,    2,    4,    2,    9,    0    },    // 2^622
    {    188,    3,    4,    8,    0,    8,    6,    5,    9,    4,    9,    7,    2,    3,    9,    6,    4,    8,    5,    8,    0    },    // 2^623
    {    188,    6,    9,    6,    1,    7,    3,    1,    8,    9,    9,    4,    4,    7,    9,    2,    9,    7,    1,    5,    9    },    // 2^624
    {    189,    1,    3,    9,    2,    3,    4,    6,    3,    7,    9,    8,    8,    9,    5,    8,    5,    9,    4,    3,    2    },    // 2^625
    {    189,    2,    7,    8,    4,    6,    9,    2,    7,    5,    9,    7,    7,    9,    1,    7,    1,    8,    8,    6,    4    },    // 2^626
    {    189,    5,    5,    6,    9,    3,    8,    5,    5,    1,    9,    5,    5,    8,    3,    4,    3,    7,    7,    2,    8    },    // 2^627
    {    190,    1,    1,    1,    3,    8,    7,    7,    1,    0,    3,    9,    1,    1,    6,    6,    8,    7,    5,    4,    6    },    // 2^628
    {    190,    2,    2,    2,    7,    7,    5,    4,    2,    0,    7,    8,    2,    3,    3,    3,    7,    5,    0,    9,    1    },    // 2^629
    {    190,    4,    4,    5,    5,    5,    0,    8,    4,    1,    5,    6,    4,    6,    6,    7,    5,    0,    1,    8,    2    },    // 2^630
    {    190,    8,    9,    1,    1,    0,    1,    6,    8,    3,    1,    2,    9,    3,    3,    5,    0,    0,    3,    6,    4    },    // 2^631
    {    191,    1,    7,    8,    2,    2,    0,    3,    3,    6,    6,    2,    5,    8,    6,    7,    0,    0,    0,    7,    3    },    // 2^632
    {    191,    3,    5,    6,    4,    4,    0,    6,    7,    3,    2,    5,    1,    7,    3,    4,    0,    0,    1,    4,    6    },    // 2^633
    {    191,    7,    1,    2,    8,    8,    1,    3,    4,    6,    5,    0,    3,    4,    6,    8,    0,    0,    2,    9,    1    },    // 2^634
    {    192,    1,    4,    2,    5,    7,    6,    2,    6,    9,    3,    0,    0,    6,    9,    3,    6,    0,    0,    5,    8    },    // 2^635
    {    192,    2,    8,    5,    1,    5,    2,    5,    3,    8,    6,    0,    1,    3,    8,    7,    2,    0,    1,    1,    7    },    // 2^636
    {    192,    5,    7,    0,    3,    0,    5,    0,    7,    7,    2,    0,    2,    7,    7,    4,    4,    0,    2,    3,    3    },    // 2^637
    {    193,    1,    1,    4,    0,    6,    1,    0,    1,    5,    4,    4,    0,    5,    5,    4,    8,    8,    0,    4,    7    },    // 2^638
    {    193,    2,    2,    8,    1,    2,    2,    0,    3,    0,    8,    8,    1,    1,    0,    9,    7,    6,    0,    9,    3    },    // 2^639
    {    193,    4,    5,    6,    2,    4,    4,    0,    6,    1,    7,    6,    2,    2,    1,    9,    5,    2,    1,    8,    6    },    // 2^640
    {    193,    9,    1,    2,    4,    8,    8,    1,    2,    3,    5,    2,    4,    4,    3,    9,    0,    4,    3,    7,    3    },    // 2^641
    {    194,    1,    8,    2,    4,    9,    7,    6,    2,    4,    7,    0,    4,    8,    8,    7,    8,    0,    8,    7,    5    },    // 2^642
    {    194,    3,    6,    4,    9,    9,    5,    2,    4,    9,    4,    0,    9,    7,    7,    5,    6,    1,    7,    4,    9    },    // 2^643
    {    194,    7,    2,    9,    9,    9,    0,    4,    9,    8,    8,    1,    9,    5,    5,    1,    2,    3,    4,    9,    8    },    // 2^644
    {    195,    1,    4,    5,    9,    9,    8,    0,    9,    9,    7,    6,    3,    9,    1,    0,    2,    4,    6,    0,    0    },    // 2^645
    {    195,    2,    9,    1,    9,    9,    6,    1,    9,    9,    5,    2,    7,    8,    2,    0,    4,    9,    3,    9,    9    },    // 2^646
    {    195,    5,    8,    3,    9,    9,    2,    3,    9,    9,    0,    5,    5,    6,    4,    0,    9,    8,    7,    9,    9    },    // 2^647
    {    196,    1,    1,    6,    7,    9,    8,    4,    7,    9,    8,    1,    1,    1,    2,    8,    1,    9,    7,    6,    0    },    // 2^648
    {    196,    2,    3,    3,    5,    9,    6,    9,    5,    9,    6,    2,    2,    2,    5,    6,    3,    9,    5,    1,    9    },    // 2^649
    {    196,    4,    6,    7,    1,    9,    3,    9,    1,    9,    2,    4,    4,    5,    1,    2,    7,    9,    0,    3,    9    },    // 2^650
    {    196,    9,    3,    4,    3,    8,    7,    8,    3,    8,    4,    8,    9,    0,    2,    5,    5,    8,    0,    7,    8    },    // 2^651
    {    197,    1,    8,    6,    8,    7,    7,    5,    6,    7,    6,    9,    7,    8,    0,    5,    1,    1,    6,    1,    6    },    // 2^652
    {    197,    3,    7,    3,    7,    5,    5,    1,    3,    5,    3,    9,    5,    6,    1,    0,    2,    3,    2,    3,    1    },    // 2^653
    {    197,    7,    4,    7,    5,    1,    0,    2,    7,    0,    7,    9,    1,    2,    2,    0,    4,    6,    4,    6,    2    },    // 2^654
    {    198,    1,    4,    9,    5,    0,    2,    0,    5,    4,    1,    5,    8,    2,    4,    4,    0,    9,    2,    9,    2    },    // 2^655
    {    198,    2,    9,    9,    0,    0,    4,    1,    0,    8,    3,    1,    6,    4,    8,    8,    1,    8,    5,    8,    5    },    // 2^656
    {    198,    5,    9,    8,    0,    0,    8,    2,    1,    6,    6,    3,    2,    9,    7,    6,    3,    7,    1,    7,    0    },    // 2^657
    {    199,    1,    1,    9,    6,    0,    1,    6,    4,    3,    3,    2,    6,    5,    9,    5,    2,    7,    4,    3,    4    },    // 2^658
    {    199,    2,    3,    9,    2,    0,    3,    2,    8,    6,    6,    5,    3,    1,    9,    0,    5,    4,    8,    6,    8    },    // 2^659
    {    199,    4,    7,    8,    4,    0,    6,    5,    7,    3,    3,    0,    6,    3,    8,    1,    0,    9,    7,    3,    6    },    // 2^660
    {    199,    9,    5,    6,    8,    1,    3,    1,    4,    6,    6,    1,    2,    7,    6,    2,    1,    9,    4,    7,    2    },    // 2^661
    {    200,    1,    9,    1,    3,    6,    2,    6,    2,    9,    3,    2,    2,    5,    5,    2,    4,    3,    8,    9,    4    },    // 2^662
    {    200,    3,    8,    2,    7,    2,    5,    2,    5,    8,    6,    4,    5,    1,    0,    4,    8,    7,    7,    8,    9    },    // 2^663
    {    200,    7,    6,    5,    4,    5,    0,    5,    1,    7,    2,    9,    0,    2,    0,    9,    7,    5,    5,    7,    7    },    // 2^664
    {    201,    1,    5,    3,    0,    9,    0,    1,    0,    3,    4,    5,    8,    0,    4,    1,    9,    5,    1,    1,    5    },    // 2^665
    {    201,    3,    0,    6,    1,    8,    0,    2,    0,    6,    9,    1,    6,    0,    8,    3,    9,    0,    2,    3,    1    },    // 2^666
    {    201,    6,    1,    2,    3,    6,    0,    4,    1,    3,    8,    3,    2,    1,    6,    7,    8,    0,    4,    6,    2    },    // 2^667
    {    202,    1,    2,    2,    4,    7,    2,    0,    8,    2,    7,    6,    6,    4,    3,    3,    5,    6,    0,    9,    2    },    // 2^668
    {    202,    2,    4,    4,    9,    4,    4,    1,    6,    5,    5,    3,    2,    8,    6,    7,    1,    2,    1,    8,    5    },    // 2^669
    {    202,    4,    8,    9,    8,    8,    8,    3,    3,    1,    0,    6,    5,    7,    3,    4,    2,    4,    3,    6,    9    },    // 2^670
    {    202,    9,    7,    9,    7,    7,    6,    6,    6,    2,    1,    3,    1,    4,    6,    8,    4,    8,    7,    3,    9    },    // 2^671
    {    203,    1,    9,    5,    9,    5,    5,    3,    3,    2,    4,    2,    6,    2,    9,    3,    6,    9,    7,    4,    8    },    // 2^672
    {    203,    3,    9,    1,    9,    1,    0,    6,    6,    4,    8,    5,    2,    5,    8,    7,    3,    9,    4,    9,    6    },    // 2^673
    {    203,    7,    8,    3,    8,    2,    1,    3,    2,    9,    7,    0,    5,    1,    7,    4,    7,    8,    9,    9,    1    },    // 2^674
    {    204,    1,    5,    6,    7,    6,    4,    2,    6,    5,    9,    4,    1,    0,    3,    4,    9,    5,    7,    9,    8    },    // 2^675
    {    204,    3,    1,    3,    5,    2,    8,    5,    3,    1,    8,    8,    2,    0,    6,    9,    9,    1,    5,    9,    6    },    // 2^676
    {    204,    6,    2,    7,    0,    5,    7,    0,    6,    3,    7,    6,    4,    1,    3,    9,    8,    3,    1,    9,    3    },    // 2^677
    {    205,    1,    2,    5,    4,    1,    1,    4,    1,    2,    7,    5,    2,    8,    2,    7,    9,    6,    6,    3,    9    },    // 2^678
    {    205,    2,    5,    0,    8,    2,    2,    8,    2,    5,    5,    0,    5,    6,    5,    5,    9,    3,    2,    7,    7    },    // 2^679
    {    205,    5,    0,    1,    6,    4,    5,    6,    5,    1,    0,    1,    1,    3,    1,    1,    8,    6,    5,    5,    4    },    // 2^680
    {    206,    1,    0,    0,    3,    2,    9,    1,    3,    0,    2,    0,    2,    2,    6,    2,    3,    7,    3,    1,    1    },    // 2^681
    {    206,    2,    0,    0,    6,    5,    8,    2,    6,    0,    4,    0,    4,    5,    2,    4,    7,    4,    6,    2,    2    },    // 2^682
    {    206,    4,    0,    1,    3,    1,    6,    5,    2,    0,    8,    0,    9,    0,    4,    9,    4,    9,    2,    4,    3    },    // 2^683
    {    206,    8,    0,    2,    6,    3,    3,    0,    4,    1,    6,    1,    8,    0,    9,    8,    9,    8,    4,    8,    7    },    // 2^684
    {    207,    1,    6,    0,    5,    2,    6,    6,    0,    8,    3,    2,    3,    6,    1,    9,    7,    9,    6,    9,    7    },    // 2^685
    {    207,    3,    2,    1,    0,    5,    3,    2,    1,    6,    6,    4,    7,    2,    3,    9,    5,    9,    3,    9,    5    },    // 2^686
    {    207,    6,    4,    2,    1,    0,    6,    4,    3,    3,    2,    9,    4,    4,    7,    9,    1,    8,    7,    9,    0    },    // 2^687
    {    208,    1,    2,    8,    4,    2,    1,    2,    8,    6,    6,    5,    8,    8,    9,    5,    8,    3,    7,    5,    8    },    // 2^688
    {    208,    2,    5,    6,    8,    4,    2,    5,    7,    3,    3,    1,    7,    7,    9,    1,    6,    7,    5,    1,    6    },    // 2^689
    {    208,    5,    1,    3,    6,    8,    5,    1,    4,    6,    6,    3,    5,    5,    8,    3,    3,    5,    0,    3,    2    },    // 2^690
    {    209,    1,    0,    2,    7,    3,    7,    0,    2,    9,    3,    2,    7,    1,    1,    6,    6,    7,    0,    0,    6    },    // 2^691
    {    209,    2,    0,    5,    4,    7,    4,    0,    5,    8,    6,    5,    4,    2,    3,    3,    3,    4,    0,    1,    3    },    // 2^692
    {    209,    4,    1,    0,    9,    4,    8,    1,    1,    7,    3,    0,    8,    4,    6,    6,    6,    8,    0,    2,    5    },    // 2^693
    {    209,    8,    2,    1,    8,    9,    6,    2,    3,    4,    6,    1,    6,    9,    3,    3,    3,    6,    0,    5,    1    },    // 2^694
    {    210,    1,    6,    4,    3,    7,    9,    2,    4,    6,    9,    2,    3,    3,    8,    6,    6,    7,    2,    1,    0    },    // 2^695
    {    210,    3,    2,    8,    7,    5,    8,    4,    9,    3,    8,    4,    6,    7,    7,    3,    3,    4,    4,    2,    0    },    // 2^696
    {    210,    6,    5,    7,    5,    1,    6,    9,    8,    7,    6,    9,    3,    5,    4,    6,    6,    8,    8,    4,    1    },    // 2^697
    {    211,    1,    3,    1,    5,    0,    3,    3,    9,    7,    5,    3,    8,    7,    0,    9,    3,    3,    7,    6,    8    },    // 2^698
    {    211,    2,    6,    3,    0,    0,    6,    7,    9,    5,    0,    7,    7,    4,    1,    8,    6,    7,    5,    3,    6    },    // 2^699
    {    211,    5,    2,    6,    0,    1,    3,    5,    9,    0,    1,    5,    4,    8,    3,    7,    3,    5,    0,    7,    2    },    // 2^700
    {    212,    1,    0,    5,    2,    0,    2,    7,    1,    8,    0,    3,    0,    9,    6,    7,    4,    7,    0,    1,    4    },    // 2^701
    {    212,    2,    1,    0,    4,    0,    5,    4,    3,    6,    0,    6,    1,    9,    3,    4,    9,    4,    0,    2,    9    },    // 2^702
    {    212,    4,    2,    0,    8,    1,    0,    8,    7,    2,    1,    2,    3,    8,    6,    9,    8,    8,    0,    5,    8    },    // 2^703
    {    212,    8,    4,    1,    6,    2,    1,    7,    4,    4,    2,    4,    7,    7,    3,    9,    7,    6,    1,    1,    6    },    // 2^704
    {    213,    1,    6,    8,    3,    2,    4,    3,    4,    8,    8,    4,    9,    5,    4,    7,    9,    5,    2,    2,    3    },    // 2^705
    {    213,    3,    3,    6,    6,    4,    8,    6,    9,    7,    6,    9,    9,    0,    9,    5,    9,    0,    4,    4,    6    },    // 2^706
    {    213,    6,    7,    3,    2,    9,    7,    3,    9,    5,    3,    9,    8,    1,    9,    1,    8,    0,    8,    9,    3    },    // 2^707
    {    214,    1,    3,    4,    6,    5,    9,    4,    7,    9,    0,    7,    9,    6,    3,    8,    3,    6,    1,    7,    9    },    // 2^708
    {    214,    2,    6,    9,    3,    1,    8,    9,    5,    8,    1,    5,    9,    2,    7,    6,    7,    2,    3,    5,    7    },    // 2^709
    {    214,    5,    3,    8,    6,    3,    7,    9,    1,    6,    3,    1,    8,    5,    5,    3,    4,    4,    7,    1,    4    },    // 2^710
    {    215,    1,    0,    7,    7,    2,    7,    5,    8,    3,    2,    6,    3,    7,    1,    0,    6,    8,    9,    4,    3    },    // 2^711
    {    215,    2,    1,    5,    4,    5,    5,    1,    6,    6,    5,    2,    7,    4,    2,    1,    3,    7,    8,    8,    6    },    // 2^712
    {    215,    4,    3,    0,    9,    1,    0,    3,    3,    3,    0,    5,    4,    8,    4,    2,    7,    5,    7,    7,    1    },    // 2^713
    {    215,    8,    6,    1,    8,    2,    0,    6,    6,    6,    1,    0,    9,    6,    8,    5,    5,    1,    5,    4,    3    },    // 2^714
    {    216,    1,    7,    2,    3,    6,    4,    1,    3,    3,    2,    2,    1,    9,    3,    7,    1,    0,    3,    0,    9    },    // 2^715
    {    216,    3,    4,    4,    7,    2,    8,    2,    6,    6,    4,    4,    3,    8,    7,    4,    2,    0,    6,    1,    7    },    // 2^716
    {    216,    6,    8,    9,    4,    5,    6,    5,    3,    2,    8,    8,    7,    7,    4,    8,    4,    1,    2,    3,    4    },    // 2^717
    {    217,    1,    3,    7,    8,    9,    1,    3,    0,    6,    5,    7,    7,    5,    4,    9,    6,    8,    2,    4,    7    },    // 2^718
    {    217,    2,    7,    5,    7,    8,    2,    6,    1,    3,    1,    5,    5,    0,    9,    9,    3,    6,    4,    9,    4    },    // 2^719
    {    217,    5,    5,    1,    5,    6,    5,    2,    2,    6,    3,    1,    0,    1,    9,    8,    7,    2,    9,    8,    7    },    // 2^720
    {    218,    1,    1,    0,    3,    1,    3,    0,    4,    5,    2,    6,    2,    0,    3,    9,    7,    4,    5,    9,    7    },    // 2^721
    {    218,    2,    2,    0,    6,    2,    6,    0,    9,    0,    5,    2,    4,    0,    7,    9,    4,    9,    1,    9,    5    },    // 2^722
    {    218,    4,    4,    1,    2,    5,    2,    1,    8,    1,    0,    4,    8,    1,    5,    8,    9,    8,    3,    9,    0    },    // 2^723
    {    218,    8,    8,    2,    5,    0,    4,    3,    6,    2,    0,    9,    6,    3,    1,    7,    9,    6,    7,    8,    0    },    // 2^724
    {    219,    1,    7,    6,    5,    0,    0,    8,    7,    2,    4,    1,    9,    2,    6,    3,    5,    9,    3,    5,    6    },    // 2^725
    {    219,    3,    5,    3,    0,    0,    1,    7,    4,    4,    8,    3,    8,    5,    2,    7,    1,    8,    7,    1,    2    },    // 2^726
    {    219,    7,    0,    6,    0,    0,    3,    4,    8,    9,    6,    7,    7,    0,    5,    4,    3,    7,    4,    2,    4    },    // 2^727
    {    220,    1,    4,    1,    2,    0,    0,    6,    9,    7,    9,    3,    5,    4,    1,    0,    8,    7,    4,    8,    5    },    // 2^728
    {    220,    2,    8,    2,    4,    0,    1,    3,    9,    5,    8,    7,    0,    8,    2,    1,    7,    4,    9,    6,    9    },    // 2^729
    {    220,    5,    6,    4,    8,    0,    2,    7,    9,    1,    7,    4,    1,    6,    4,    3,    4,    9,    9,    3,    9    },    // 2^730
    {    221,    1,    1,    2,    9,    6,    0,    5,    5,    8,    3,    4,    8,    3,    2,    8,    6,    9,    9,    8,    8    },    // 2^731
    {    221,    2,    2,    5,    9,    2,    1,    1,    1,    6,    6,    9,    6,    6,    5,    7,    3,    9,    9,    7,    6    },    // 2^732
    {    221,    4,    5,    1,    8,    4,    2,    2,    3,    3,    3,    9,    3,    3,    1,    4,    7,    9,    9,    5,    1    },    // 2^733
    {    221,    9,    0,    3,    6,    8,    4,    4,    6,    6,    7,    8,    6,    6,    2,    9,    5,    9,    9,    0,    2    },    // 2^734
    {    222,    1,    8,    0,    7,    3,    6,    8,    9,    3,    3,    5,    7,    3,    2,    5,    9,    1,    9,    8,    0    },    // 2^735
    {    222,    3,    6,    1,    4,    7,    3,    7,    8,    6,    7,    1,    4,    6,    5,    1,    8,    3,    9,    6,    1    },    // 2^736
    {    222,    7,    2,    2,    9,    4,    7,    5,    7,    3,    4,    2,    9,    3,    0,    3,    6,    7,    9,    2,    2    },    // 2^737
    {    223,    1,    4,    4,    5,    8,    9,    5,    1,    4,    6,    8,    5,    8,    6,    0,    7,    3,    5,    8,    4    },    // 2^738
    {    223,    2,    8,    9,    1,    7,    9,    0,    2,    9,    3,    7,    1,    7,    2,    1,    4,    7,    1,    6,    9    },    // 2^739
    {    223,    5,    7,    8,    3,    5,    8,    0,    5,    8,    7,    4,    3,    4,    4,    2,    9,    4,    3,    3,    8    },    // 2^740
    {    224,    1,    1,    5,    6,    7,    1,    6,    1,    1,    7,    4,    8,    6,    8,    8,    5,    8,    8,    6,    8    },    // 2^741
    {    224,    2,    3,    1,    3,    4,    3,    2,    2,    3,    4,    9,    7,    3,    7,    7,    1,    7,    7,    3,    5    },    // 2^742
    {    224,    4,    6,    2,    6,    8,    6,    4,    4,    6,    9,    9,    4,    7,    5,    4,    3,    5,    4,    7,    0    },    // 2^743
    {    224,    9,    2,    5,    3,    7,    2,    8,    9,    3,    9,    8,    9,    5,    0,    8,    7,    0,    9,    4,    0    },    // 2^744
    {    225,    1,    8,    5,    0,    7,    4,    5,    7,    8,    7,    9,    7,    9,    0,    1,    7,    4,    1,    8,    8    },    // 2^745
    {    225,    3,    7,    0,    1,    4,    9,    1,    5,    7,    5,    9,    5,    8,    0,    3,    4,    8,    3,    7,    6    },    // 2^746
    {    225,    7,    4,    0,    2,    9,    8,    3,    1,    5,    1,    9,    1,    6,    0,    6,    9,    6,    7,    5,    2    },    // 2^747
    {    226,    1,    4,    8,    0,    5,    9,    6,    6,    3,    0,    3,    8,    3,    2,    1,    3,    9,    3,    5,    0    },    // 2^748
    {    226,    2,    9,    6,    1,    1,    9,    3,    2,    6,    0,    7,    6,    6,    4,    2,    7,    8,    7,    0,    1    },    // 2^749
    {    226,    5,    9,    2,    2,    3,    8,    6,    5,    2,    1,    5,    3,    2,    8,    5,    5,    7,    4,    0,    2    },    // 2^750
    {    227,    1,    1,    8,    4,    4,    7,    7,    3,    0,    4,    3,    0,    6,    5,    7,    1,    1,    4,    8,    0    },    // 2^751
    {    227,    2,    3,    6,    8,    9,    5,    4,    6,    0,    8,    6,    1,    3,    1,    4,    2,    2,    9,    6,    1    },    // 2^752
    {    227,    4,    7,    3,    7,    9,    0,    9,    2,    1,    7,    2,    2,    6,    2,    8,    4,    5,    9,    2,    1    },    // 2^753
    {    227,    9,    4,    7,    5,    8,    1,    8,    4,    3,    4,    4,    5,    2,    5,    6,    9,    1,    8,    4,    3    },    // 2^754
    {    228,    1,    8,    9,    5,    1,    6,    3,    6,    8,    6,    8,    9,    0,    5,    1,    3,    8,    3,    6,    9    },    // 2^755
    {    228,    3,    7,    9,    0,    3,    2,    7,    3,    7,    3,    7,    8,    1,    0,    2,    7,    6,    7,    3,    7    },    // 2^756
    {    228,    7,    5,    8,    0,    6,    5,    4,    7,    4,    7,    5,    6,    2,    0,    5,    5,    3,    4,    7,    4    },    // 2^757
    {    229,    1,    5,    1,    6,    1,    3,    0,    9,    4,    9,    5,    1,    2,    4,    1,    1,    0,    6,    9,    5    },    // 2^758
    {    229,    3,    0,    3,    2,    2,    6,    1,    8,    9,    9,    0,    2,    4,    8,    2,    2,    1,    3,    9,    0    },    // 2^759
    {    229,    6,    0,    6,    4,    5,    2,    3,    7,    9,    8,    0,    4,    9,    6,    4,    4,    2,    7,    7,    9    },    // 2^760
    {    230,    1,    2,    1,    2,    9,    0,    4,    7,    5,    9,    6,    0,    9,    9,    2,    8,    8,    5,    5,    6    },    // 2^761
    {    230,    2,    4,    2,    5,    8,    0,    9,    5,    1,    9,    2,    1,    9,    8,    5,    7,    7,    1,    1,    2    },    // 2^762
    {    230,    4,    8,    5,    1,    6,    1,    9,    0,    3,    8,    4,    3,    9,    7,    1,    5,    4,    2,    2,    3    },    // 2^763
    {    230,    9,    7,    0,    3,    2,    3,    8,    0,    7,    6,    8,    7,    9,    4,    3,    0,    8,    4,    4,    7    },    // 2^764
    {    231,    1,    9,    4,    0,    6,    4,    7,    6,    1,    5,    3,    7,    5,    8,    8,    6,    1,    6,    8,    9    },    // 2^765
    {    231,    3,    8,    8,    1,    2,    9,    5,    2,    3,    0,    7,    5,    1,    7,    7,    2,    3,    3,    7,    9    },    // 2^766
    {    231,    7,    7,    6,    2,    5,    9,    0,    4,    6,    1,    5,    0,    3,    5,    4,    4,    6,    7,    5,    7    },    // 2^767
    {    232,    1,    5,    5,    2,    5,    1,    8,    0,    9,    2,    3,    0,    0,    7,    0,    8,    9,    3,    5,    1    },    // 2^768
    {    232,    3,    1,    0,    5,    0,    3,    6,    1,    8,    4,    6,    0,    1,    4,    1,    7,    8,    7,    0,    3    },    // 2^769
    {    232,    6,    2,    1,    0,    0,    7,    2,    3,    6,    9,    2,    0,    2,    8,    3,    5,    7,    4,    0,    6    },    // 2^770
    {    233,    1,    2,    4,    2,    0,    1,    4,    4,    7,    3,    8,    4,    0,    5,    6,    7,    1,    4,    8,    1    },    // 2^771
    {    233,    2,    4,    8,    4,    0,    2,    8,    9,    4,    7,    6,    8,    1,    1,    3,    4,    2,    9,    6,    2    },    // 2^772
    {    233,    4,    9,    6,    8,    0,    5,    7,    8,    9,    5,    3,    6,    2,    2,    6,    8,    5,    9,    2,    5    },    // 2^773
    {    233,    9,    9,    3,    6,    1,    1,    5,    7,    9,    0,    7,    2,    4,    5,    3,    7,    1,    8,    5,    0    },    // 2^774
    {    234,    1,    9,    8,    7,    2,    2,    3,    1,    5,    8,    1,    4,    4,    9,    0,    7,    4,    3,    7,    0    },    // 2^775
    {    234,    3,    9,    7,    4,    4,    4,    6,    3,    1,    6,    2,    8,    9,    8,    1,    4,    8,    7,    4,    0    },    // 2^776
    {    234,    7,    9,    4,    8,    8,    9,    2,    6,    3,    2,    5,    7,    9,    6,    2,    9,    7,    4,    8,    0    },    // 2^777
    {    235,    1,    5,    8,    9,    7,    7,    8,    5,    2,    6,    5,    1,    5,    9,    2,    5,    9,    4,    9,    6    },    // 2^778
    {    235,    3,    1,    7,    9,    5,    5,    7,    0,    5,    3,    0,    3,    1,    8,    5,    1,    8,    9,    9,    2    },    // 2^779
    {    235,    6,    3,    5,    9,    1,    1,    4,    1,    0,    6,    0,    6,    3,    7,    0,    3,    7,    9,    8,    4    },    // 2^780
    {    236,    1,    2,    7,    1,    8,    2,    2,    8,    2,    1,    2,    1,    2,    7,    4,    0,    7,    5,    9,    7    },    // 2^781
    {    236,    2,    5,    4,    3,    6,    4,    5,    6,    4,    2,    4,    2,    5,    4,    8,    1,    5,    1,    9,    3    },    // 2^782
    {    236,    5,    0,    8,    7,    2,    9,    1,    2,    8,    4,    8,    5,    0,    9,    6,    3,    0,    3,    8,    7    },    // 2^783
    {    237,    1,    0,    1,    7,    4,    5,    8,    2,    5,    6,    9,    7,    0,    1,    9,    2,    6,    0,    7,    7    },    // 2^784
    {    237,    2,    0,    3,    4,    9,    1,    6,    5,    1,    3,    9,    4,    0,    3,    8,    5,    2,    1,    5,    5    },    // 2^785
    {    237,    4,    0,    6,    9,    8,    3,    3,    0,    2,    7,    8,    8,    0,    7,    7,    0,    4,    3,    1,    0    },    // 2^786
    {    237,    8,    1,    3,    9,    6,    6,    6,    0,    5,    5,    7,    6,    1,    5,    4,    0,    8,    6,    1,    9    },    // 2^787
    {    238,    1,    6,    2,    7,    9,    3,    3,    2,    1,    1,    1,    5,    2,    3,    0,    8,    1,    7,    2,    4    },    // 2^788
    {    238,    3,    2,    5,    5,    8,    6,    6,    4,    2,    2,    3,    0,    4,    6,    1,    6,    3,    4,    4,    8    },    // 2^789
    {    238,    6,    5,    1,    1,    7,    3,    2,    8,    4,    4,    6,    0,    9,    2,    3,    2,    6,    8,    9,    5    },    // 2^790
    {    239,    1,    3,    0,    2,    3,    4,    6,    5,    6,    8,    9,    2,    1,    8,    4,    6,    5,    3,    7,    9    },    // 2^791
    {    239,    2,    6,    0,    4,    6,    9,    3,    1,    3,    7,    8,    4,    3,    6,    9,    3,    0,    7,    5,    8    },    // 2^792
    {    239,    5,    2,    0,    9,    3,    8,    6,    2,    7,    5,    6,    8,    7,    3,    8,    6,    1,    5,    1,    6    },    // 2^793
    {    240,    1,    0,    4,    1,    8,    7,    7,    2,    5,    5,    1,    3,    7,    4,    7,    7,    2,    3,    0,    3    },    // 2^794
    {    240,    2,    0,    8,    3,    7,    5,    4,    5,    1,    0,    2,    7,    4,    9,    5,    4,    4,    6,    0,    6    },    // 2^795
    {    240,    4,    1,    6,    7,    5,    0,    9,    0,    2,    0,    5,    4,    9,    9,    0,    8,    9,    2,    1,    3    },    // 2^796
    {    240,    8,    3,    3,    5,    0,    1,    8,    0,    4,    1,    0,    9,    9,    8,    1,    7,    8,    4,    2,    6    },    // 2^797
    {    241,    1,    6,    6,    7,    0,    0,    3,    6,    0,    8,    2,    1,    9,    9,    6,    3,    5,    6,    8,    5    },    // 2^798
    {    241,    3,    3,    3,    4,    0,    0,    7,    2,    1,    6,    4,    3,    9,    9,    2,    7,    1,    3,    7,    0    },    // 2^799
    {    241,    6,    6,    6,    8,    0,    1,    4,    4,    3,    2,    8,    7,    9,    8,    5,    4,    2,    7,    4,    1    },    // 2^800
    {    242,    1,    3,    3,    3,    6,    0,    2,    8,    8,    6,    5,    7,    5,    9,    7,    0,    8,    5,    4,    8    },    // 2^801
    {    242,    2,    6,    6,    7,    2,    0,    5,    7,    7,    3,    1,    5,    1,    9,    4,    1,    7,    0,    9,    6    },    // 2^802
    {    242,    5,    3,    3,    4,    4,    1,    1,    5,    4,    6,    3,    0,    3,    8,    8,    3,    4,    1,    9,    3    },    // 2^803
    {    243,    1,    0,    6,    6,    8,    8,    2,    3,    0,    9,    2,    6,    0,    7,    7,    6,    6,    8,    3,    9    },    // 2^804
    {    243,    2,    1,    3,    3,    7,    6,    4,    6,    1,    8,    5,    2,    1,    5,    5,    3,    3,    6,    7,    7    },    // 2^805
    {    243,    4,    2,    6,    7,    5,    2,    9,    2,    3,    7,    0,    4,    3,    1,    0,    6,    7,    3,    5,    4    },    // 2^806
    {    243,    8,    5,    3,    5,    0,    5,    8,    4,    7,    4,    0,    8,    6,    2,    1,    3,    4,    7,    0,    8    },    // 2^807
    {    244,    1,    7,    0,    7,    0,    1,    1,    6,    9,    4,    8,    1,    7,    2,    4,    2,    6,    9,    4,    2    },    // 2^808
    {    244,    3,    4,    1,    4,    0,    2,    3,    3,    8,    9,    6,    3,    4,    4,    8,    5,    3,    8,    8,    3    },    // 2^809
    {    244,    6,    8,    2,    8,    0,    4,    6,    7,    7,    9,    2,    6,    8,    9,    7,    0,    7,    7,    6,    7    },    // 2^810
    {    245,    1,    3,    6,    5,    6,    0,    9,    3,    5,    5,    8,    5,    3,    7,    9,    4,    1,    5,    5,    3    },    // 2^811
    {    245,    2,    7,    3,    1,    2,    1,    8,    7,    1,    1,    7,    0,    7,    5,    8,    8,    3,    1,    0,    7    },    // 2^812
    {    245,    5,    4,    6,    2,    4,    3,    7,    4,    2,    3,    4,    1,    5,    1,    7,    6,    6,    2,    1,    3    },    // 2^813
    {    246,    1,    0,    9,    2,    4,    8,    7,    4,    8,    4,    6,    8,    3,    0,    3,    5,    3,    2,    4,    3    },    // 2^814
    {    246,    2,    1,    8,    4,    9,    7,    4,    9,    6,    9,    3,    6,    6,    0,    7,    0,    6,    4,    8,    5    },    // 2^815
    {    246,    4,    3,    6,    9,    9,    4,    9,    9,    3,    8,    7,    3,    2,    1,    4,    1,    2,    9,    7,    1    },    // 2^816
    {    246,    8,    7,    3,    9,    8,    9,    9,    8,    7,    7,    4,    6,    4,    2,    8,    2,    5,    9,    4,    1    },    // 2^817
    {    247,    1,    7,    4,    7,    9,    7,    9,    9,    7,    5,    4,    9,    2,    8,    5,    6,    5,    1,    8,    8    },    // 2^818
    {    247,    3,    4,    9,    5,    9,    5,    9,    9,    5,    0,    9,    8,    5,    7,    1,    3,    0,    3,    7,    6    },    // 2^819
    {    247,    6,    9,    9,    1,    9,    1,    9,    9,    0,    1,    9,    7,    1,    4,    2,    6,    0,    7,    5,    3    },    // 2^820
    {    248,    1,    3,    9,    8,    3,    8,    3,    9,    8,    0,    3,    9,    4,    2,    8,    5,    2,    1,    5,    1    },    // 2^821
    {    248,    2,    7,    9,    6,    7,    6,    7,    9,    6,    0,    7,    8,    8,    5,    7,    0,    4,    3,    0,    1    },    // 2^822
    {    248,    5,    5,    9,    3,    5,    3,    5,    9,    2,    1,    5,    7,    7,    1,    4,    0,    8,    6,    0,    2    },    // 2^823
    {    249,    1,    1,    1,    8,    7,    0,    7,    1,    8,    4,    3,    1,    5,    4,    2,    8,    1,    7,    2,    0    },    // 2^824
    {    249,    2,    2,    3,    7,    4,    1,    4,    3,    6,    8,    6,    3,    0,    8,    5,    6,    3,    4,    4,    1    },    // 2^825
    {    249,    4,    4,    7,    4,    8,    2,    8,    7,    3,    7,    2,    6,    1,    7,    1,    2,    6,    8,    8,    2    },    // 2^826
    {    249,    8,    9,    4,    9,    6,    5,    7,    4,    7,    4,    5,    2,    3,    4,    2,    5,    3,    7,    6,    4    },    // 2^827
    {    250,    1,    7,    8,    9,    9,    3,    1,    4,    9,    4,    9,    0,    4,    6,    8,    5,    0,    7,    5,    3    },    // 2^828
    {    250,    3,    5,    7,    9,    8,    6,    2,    9,    8,    9,    8,    0,    9,    3,    7,    0,    1,    5,    0,    6    },    // 2^829
    {    250,    7,    1,    5,    9,    7,    2,    5,    9,    7,    9,    6,    1,    8,    7,    4,    0,    3,    0,    1,    1    },    // 2^830
    {    251,    1,    4,    3,    1,    9,    4,    5,    1,    9,    5,    9,    2,    3,    7,    4,    8,    0,    6,    0,    2    },    // 2^831
    {    251,    2,    8,    6,    3,    8,    9,    0,    3,    9,    1,    8,    4,    7,    4,    9,    6,    1,    2,    0,    4    },    // 2^832
    {    251,    5,    7,    2,    7,    7,    8,    0,    7,    8,    3,    6,    9,    4,    9,    9,    2,    2,    4,    0,    9    },    // 2^833
    {    252,    1,    1,    4,    5,    5,    5,    6,    1,    5,    6,    7,    3,    8,    9,    9,    8,    4,    4,    8,    2    },    // 2^834
    {    252,    2,    2,    9,    1,    1,    1,    2,    3,    1,    3,    4,    7,    7,    9,    9,    6,    8,    9,    6,    4    },    // 2^835
    {    252,    4,    5,    8,    2,    2,    2,    4,    6,    2,    6,    9,    5,    5,    9,    9,    3,    7,    9,    2,    7    },    // 2^836
    {    252,    9,    1,    6,    4,    4,    4,    9,    2,    5,    3,    9,    1,    1,    9,    8,    7,    5,    8,    5,    4    },    // 2^837
    {    253,    1,    8,    3,    2,    8,    8,    9,    8,    5,    0,    7,    8,    2,    3,    9,    7,    5,    1,    7,    1    },    // 2^838
    {    253,    3,    6,    6,    5,    7,    7,    9,    7,    0,    1,    5,    6,    4,    7,    9,    5,    0,    3,    4,    2    },    // 2^839
    {    253,    7,    3,    3,    1,    5,    5,    9,    4,    0,    3,    1,    2,    9,    5,    9,    0,    0,    6,    8,    3    },    // 2^840
    {    254,    1,    4,    6,    6,    3,    1,    1,    8,    8,    0,    6,    2,    5,    9,    1,    8,    0,    1,    3,    7    },    // 2^841
    {    254,    2,    9,    3,    2,    6,    2,    3,    7,    6,    1,    2,    5,    1,    8,    3,    6,    0,    2,    7,    3    },    // 2^842
    {    254,    5,    8,    6,    5,    2,    4,    7,    5,    2,    2,    5,    0,    3,    6,    7,    2,    0,    5,    4,    7    },    // 2^843
    {    255,    1,    1,    7,    3,    0,    4,    9,    5,    0,    4,    5,    0,    0,    7,    3,    4,    4,    1,    0,    9    },    // 2^844
    {    255,    2,    3,    4,    6,    0,    9,    9,    0,    0,    9,    0,    0,    1,    4,    6,    8,    8,    2,    1,    9    },    // 2^845
    {    255,    4,    6,    9,    2,    1,    9,    8,    0,    1,    8,    0,    0,    2,    9,    3,    7,    6,    4,    3,    7    },    // 2^846
    {    255,    9,    3,    8,    4,    3,    9,    6,    0,    3,    6,    0,    0,    5,    8,    7,    5,    2,    8,    7,    5    },    // 2^847
    {    256,    1,    8,    7,    6,    8,    7,    9,    2,    0,    7,    2,    0,    1,    1,    7,    5,    0,    5,    7,    5    },    // 2^848
    {    256,    3,    7,    5,    3,    7,    5,    8,    4,    1,    4,    4,    0,    2,    3,    5,    0,    1,    1,    5,    0    },    // 2^849
    {    256,    7,    5,    0,    7,    5,    1,    6,    8,    2,    8,    8,    0,    4,    7,    0,    0,    2,    2,    0,    0    },    // 2^850
    {    257,    1,    5,    0,    1,    5,    0,    3,    3,    6,    5,    7,    6,    0,    9,    4,    0,    0,    4,    6,    0    },    // 2^851
    {    257,    3,    0,    0,    3,    0,    0,    6,    7,    3,    1,    5,    2,    1,    8,    8,    0,    0,    9,    2,    0    },    // 2^852
    {    257,    6,    0,    0,    6,    0,    1,    3,    4,    6,    3,    0,    4,    3,    7,    6,    0,    1,    8,    4,    0    },    // 2^853
    {    258,    1,    2,    0,    1,    2,    0,    2,    6,    9,    2,    6,    0,    8,    7,    5,    2,    0,    3,    6,    8    },    // 2^854
    {    258,    2,    4,    0,    2,    4,    0,    5,    3,    8,    5,    2,    1,    7,    5,    0,    4,    0,    7,    3,    6    },    // 2^855
    {    258,    4,    8,    0,    4,    8,    1,    0,    7,    7,    0,    4,    3,    5,    0,    0,    8,    1,    4,    7,    2    },    // 2^856
    {    258,    9,    6,    0,    9,    6,    2,    1,    5,    4,    0,    8,    7,    0,    0,    1,    6,    2,    9,    4,    4    },    // 2^857
    {    259,    1,    9,    2,    1,    9,    2,    4,    3,    0,    8,    1,    7,    4,    0,    0,    3,    2,    5,    8,    9    },    // 2^858
    {    259,    3,    8,    4,    3,    8,    4,    8,    6,    1,    6,    3,    4,    8,    0,    0,    6,    5,    1,    7,    7    },    // 2^859
    {    259,    7,    6,    8,    7,    6,    9,    7,    2,    3,    2,    6,    9,    6,    0,    1,    3,    0,    3,    5,    5    },    // 2^860
    {    260,    1,    5,    3,    7,    5,    3,    9,    4,    4,    6,    5,    3,    9,    2,    0,    2,    6,    0,    7,    1    },    // 2^861
    {    260,    3,    0,    7,    5,    0,    7,    8,    8,    9,    3,    0,    7,    8,    4,    0,    5,    2,    1,    4,    2    },    // 2^862
    {    260,    6,    1,    5,    0,    1,    5,    7,    7,    8,    6,    1,    5,    6,    8,    1,    0,    4,    2,    8,    4    },    // 2^863
    {    261,    1,    2,    3,    0,    0,    3,    1,    5,    5,    7,    2,    3,    1,    3,    6,    2,    0,    8,    5,    7    },    // 2^864
    {    261,    2,    4,    6,    0,    0,    6,    3,    1,    1,    4,    4,    6,    2,    7,    2,    4,    1,    7,    1,    4    },    // 2^865
    {    261,    4,    9,    2,    0,    1,    2,    6,    2,    2,    8,    9,    2,    5,    4,    4,    8,    3,    4,    2,    7    },    // 2^866
    {    261,    9,    8,    4,    0,    2,    5,    2,    4,    5,    7,    8,    5,    0,    8,    9,    6,    6,    8,    5,    4    },    // 2^867
    {    262,    1,    9,    6,    8,    0,    5,    0,    4,    9,    1,    5,    7,    0,    1,    7,    9,    3,    3,    7,    1    },    // 2^868
    {    262,    3,    9,    3,    6,    1,    0,    0,    9,    8,    3,    1,    4,    0,    3,    5,    8,    6,    7,    4,    2    },    // 2^869
    {    262,    7,    8,    7,    2,    2,    0,    1,    9,    6,    6,    2,    8,    0,    7,    1,    7,    3,    4,    8,    3    },    // 2^870
    {    263,    1,    5,    7,    4,    4,    4,    0,    3,    9,    3,    2,    5,    6,    1,    4,    3,    4,    6,    9,    7    },    // 2^871
    {    263,    3,    1,    4,    8,    8,    8,    0,    7,    8,    6,    5,    1,    2,    2,    8,    6,    9,    3,    9,    3    },    // 2^872
    {    263,    6,    2,    9,    7,    7,    6,    1,    5,    7,    3,    0,    2,    4,    5,    7,    3,    8,    7,    8,    7    },    // 2^873
    {    264,    1,    2,    5,    9,    5,    5,    2,    3,    1,    4,    6,    0,    4,    9,    1,    4,    7,    7,    5,    7    },    // 2^874
    {    264,    2,    5,    1,    9,    1,    0,    4,    6,    2,    9,    2,    0,    9,    8,    2,    9,    5,    5,    1,    5    },    // 2^875
    {    264,    5,    0,    3,    8,    2,    0,    9,    2,    5,    8,    4,    1,    9,    6,    5,    9,    1,    0,    2,    9    },    // 2^876
    {    265,    1,    0,    0,    7,    6,    4,    1,    8,    5,    1,    6,    8,    3,    9,    3,    1,    8,    2,    0,    6    },    // 2^877
    {    265,    2,    0,    1,    5,    2,    8,    3,    7,    0,    3,    3,    6,    7,    8,    6,    3,    6,    4,    1,    2    },    // 2^878
    {    265,    4,    0,    3,    0,    5,    6,    7,    4,    0,    6,    7,    3,    5,    7,    2,    7,    2,    8,    2,    4    },    // 2^879
    {    265,    8,    0,    6,    1,    1,    3,    4,    8,    1,    3,    4,    7,    1,    4,    5,    4,    5,    6,    4,    7    },    // 2^880
    {    266,    1,    6,    1,    2,    2,    2,    6,    9,    6,    2,    6,    9,    4,    2,    9,    0,    9,    1,    2,    9    },    // 2^881
    {    266,    3,    2,    2,    4,    4,    5,    3,    9,    2,    5,    3,    8,    8,    5,    8,    1,    8,    2,    5,    9    },    // 2^882
    {    266,    6,    4,    4,    8,    9,    0,    7,    8,    5,    0,    7,    7,    7,    1,    6,    3,    6,    5,    1,    8    },    // 2^883
    {    267,    1,    2,    8,    9,    7,    8,    1,    5,    7,    0,    1,    5,    5,    4,    3,    2,    7,    3,    0,    4    },    // 2^884
    {    267,    2,    5,    7,    9,    5,    6,    3,    1,    4,    0,    3,    1,    0,    8,    6,    5,    4,    6,    0,    7    },    // 2^885
    {    267,    5,    1,    5,    9,    1,    2,    6,    2,    8,    0,    6,    2,    1,    7,    3,    0,    9,    2,    1,    4    },    // 2^886
    {    268,    1,    0,    3,    1,    8,    2,    5,    2,    5,    6,    1,    2,    4,    3,    4,    6,    1,    8,    4,    3    },    // 2^887
    {    268,    2,    0,    6,    3,    6,    5,    0,    5,    1,    2,    2,    4,    8,    6,    9,    2,    3,    6,    8,    6    },    // 2^888
    {    268,    4,    1,    2,    7,    3,    0,    1,    0,    2,    4,    4,    9,    7,    3,    8,    4,    7,    3,    7,    1    },    // 2^889
    {    268,    8,    2,    5,    4,    6,    0,    2,    0,    4,    8,    9,    9,    4,    7,    6,    9,    4,    7,    4,    3    },    // 2^890
    {    269,    1,    6,    5,    0,    9,    2,    0,    4,    0,    9,    7,    9,    8,    9,    5,    3,    8,    9,    4,    9    },    // 2^891
    {    269,    3,    3,    0,    1,    8,    4,    0,    8,    1,    9,    5,    9,    7,    9,    0,    7,    7,    8,    9,    7    },    // 2^892
    {    269,    6,    6,    0,    3,    6,    8,    1,    6,    3,    9,    1,    9,    5,    8,    1,    5,    5,    7,    9,    4    },    // 2^893
    {    270,    1,    3,    2,    0,    7,    3,    6,    3,    2,    7,    8,    3,    9,    1,    6,    3,    1,    1,    5,    9    },    // 2^894
    {    270,    2,    6,    4,    1,    4,    7,    2,    6,    5,    5,    6,    7,    8,    3,    2,    6,    2,    3,    1,    8    },    // 2^895
    {    270,    5,    2,    8,    2,    9,    4,    5,    3,    1,    1,    3,    5,    6,    6,    5,    2,    4,    6,    3,    5    },    // 2^896
    {    271,    1,    0,    5,    6,    5,    8,    9,    0,    6,    2,    2,    7,    1,    3,    3,    0,    4,    9,    2,    7    },    // 2^897
    {    271,    2,    1,    1,    3,    1,    7,    8,    1,    2,    4,    5,    4,    2,    6,    6,    0,    9,    8,    5,    4    },    // 2^898
    {    271,    4,    2,    2,    6,    3,    5,    6,    2,    4,    9,    0,    8,    5,    3,    2,    1,    9,    7,    0,    8    },    // 2^899
    {    271,    8,    4,    5,    2,    7,    1,    2,    4,    9,    8,    1,    7,    0,    6,    4,    3,    9,    4,    1,    6    },    // 2^900
    {    272,    1,    6,    9,    0,    5,    4,    2,    4,    9,    9,    6,    3,    4,    1,    2,    8,    7,    8,    8,    3    },    // 2^901
    {    272,    3,    3,    8,    1,    0,    8,    4,    9,    9,    9,    2,    6,    8,    2,    5,    7,    5,    7,    6,    7    },    // 2^902
    {    272,    6,    7,    6,    2,    1,    6,    9,    9,    9,    8,    5,    3,    6,    5,    1,    5,    1,    5,    3,    3    },    // 2^903
    {    273,    1,    3,    5,    2,    4,    3,    3,    9,    9,    9,    7,    0,    7,    3,    0,    3,    0,    3,    0,    7    },    // 2^904
    {    273,    2,    7,    0,    4,    8,    6,    7,    9,    9,    9,    4,    1,    4,    6,    0,    6,    0,    6,    1,    3    },    // 2^905
    {    273,    5,    4,    0,    9,    7,    3,    5,    9,    9,    8,    8,    2,    9,    2,    1,    2,    1,    2,    2,    6    },    // 2^906
    {    274,    1,    0,    8,    1,    9,    4,    7,    1,    9,    9,    7,    6,    5,    8,    4,    2,    4,    2,    4,    5    },    // 2^907
    {    274,    2,    1,    6,    3,    8,    9,    4,    3,    9,    9,    5,    3,    1,    6,    8,    4,    8,    4,    9,    1    },    // 2^908
    {    274,    4,    3,    2,    7,    7,    8,    8,    7,    9,    9,    0,    6,    3,    3,    6,    9,    6,    9,    8,    1    },    // 2^909
    {    274,    8,    6,    5,    5,    5,    7,    7,    5,    9,    8,    1,    2,    6,    7,    3,    9,    3,    9,    6,    2    },    // 2^910
    {    275,    1,    7,    3,    1,    1,    1,    5,    5,    1,    9,    6,    2,    5,    3,    4,    7,    8,    7,    9,    2    },    // 2^911
    {    275,    3,    4,    6,    2,    2,    3,    1,    0,    3,    9,    2,    5,    0,    6,    9,    5,    7,    5,    8,    5    },    // 2^912
    {    275,    6,    9,    2,    4,    4,    6,    2,    0,    7,    8,    5,    0,    1,    3,    9,    1,    5,    1,    7,    0    },    // 2^913
    {    276,    1,    3,    8,    4,    8,    9,    2,    4,    1,    5,    7,    0,    0,    2,    7,    8,    3,    0,    3,    4    },    // 2^914
    {    276,    2,    7,    6,    9,    7,    8,    4,    8,    3,    1,    4,    0,    0,    5,    5,    6,    6,    0,    6,    8    },    // 2^915
    {    276,    5,    5,    3,    9,    5,    6,    9,    6,    6,    2,    8,    0,    1,    1,    1,    3,    2,    1,    3,    6    },    // 2^916
    {    277,    1,    1,    0,    7,    9,    1,    3,    9,    3,    2,    5,    6,    0,    2,    2,    2,    6,    4,    2,    7    },    // 2^917
    {    277,    2,    2,    1,    5,    8,    2,    7,    8,    6,    5,    1,    2,    0,    4,    4,    5,    2,    8,    5,    4    },    // 2^918
    {    277,    4,    4,    3,    1,    6,    5,    5,    7,    3,    0,    2,    4,    0,    8,    9,    0,    5,    7,    0,    9    },    // 2^919
    {    277,    8,    8,    6,    3,    3,    1,    1,    4,    6,    0,    4,    8,    1,    7,    8,    1,    1,    4,    1,    7    },    // 2^920
    {    278,    1,    7,    7,    2,    6,    6,    2,    2,    9,    2,    0,    9,    6,    3,    5,    6,    2,    2,    8,    3    },    // 2^921
    {    278,    3,    5,    4,    5,    3,    2,    4,    5,    8,    4,    1,    9,    2,    7,    1,    2,    4,    5,    6,    7    },    // 2^922
    {    278,    7,    0,    9,    0,    6,    4,    9,    1,    6,    8,    3,    8,    5,    4,    2,    4,    9,    1,    3,    4    },    // 2^923
    {    279,    1,    4,    1,    8,    1,    2,    9,    8,    3,    3,    6,    7,    7,    0,    8,    4,    9,    8,    2,    7    },    // 2^924
    {    279,    2,    8,    3,    6,    2,    5,    9,    6,    6,    7,    3,    5,    4,    1,    6,    9,    9,    6,    5,    4    },    // 2^925
    {    279,    5,    6,    7,    2,    5,    1,    9,    3,    3,    4,    7,    0,    8,    3,    3,    9,    9,    3,    0,    7    },    // 2^926
    {    280,    1,    1,    3,    4,    5,    0,    3,    8,    6,    6,    9,    4,    1,    6,    6,    7,    9,    8,    6,    1    },    // 2^927
    {    280,    2,    2,    6,    9,    0,    0,    7,    7,    3,    3,    8,    8,    3,    3,    3,    5,    9,    7,    2,    3    },    // 2^928
    {    280,    4,    5,    3,    8,    0,    1,    5,    4,    6,    7,    7,    6,    6,    6,    7,    1,    9,    4,    4,    6    },    // 2^929
    {    280,    9,    0,    7,    6,    0,    3,    0,    9,    3,    5,    5,    3,    3,    3,    4,    3,    8,    8,    9,    1    },    // 2^930
    {    281,    1,    8,    1,    5,    2,    0,    6,    1,    8,    7,    1,    0,    6,    6,    6,    8,    7,    7,    7,    8    },    // 2^931
    {    281,    3,    6,    3,    0,    4,    1,    2,    3,    7,    4,    2,    1,    3,    3,    3,    7,    5,    5,    5,    7    },    // 2^932
    {    281,    7,    2,    6,    0,    8,    2,    4,    7,    4,    8,    4,    2,    6,    6,    7,    5,    1,    1,    1,    3    },    // 2^933
    {    282,    1,    4,    5,    2,    1,    6,    4,    9,    4,    9,    6,    8,    5,    3,    3,    5,    0,    2,    2,    3    },    // 2^934
    {    282,    2,    9,    0,    4,    3,    2,    9,    8,    9,    9,    3,    7,    0,    6,    7,    0,    0,    4,    4,    5    },    // 2^935
    {    282,    5,    8,    0,    8,    6,    5,    9,    7,    9,    8,    7,    4,    1,    3,    4,    0,    0,    8,    9,    1    },    // 2^936
    {    283,    1,    1,    6,    1,    7,    3,    1,    9,    5,    9,    7,    4,    8,    2,    6,    8,    0,    1,    7,    8    },    // 2^937
    {    283,    2,    3,    2,    3,    4,    6,    3,    9,    1,    9,    4,    9,    6,    5,    3,    6,    0,    3,    5,    6    },    // 2^938
    {    283,    4,    6,    4,    6,    9,    2,    7,    8,    3,    8,    9,    9,    3,    0,    7,    2,    0,    7,    1,    2    },    // 2^939
    {    283,    9,    2,    9,    3,    8,    5,    5,    6,    7,    7,    9,    8,    6,    1,    4,    4,    1,    4,    2,    5    },    // 2^940
    {    284,    1,    8,    5,    8,    7,    7,    1,    1,    3,    5,    5,    9,    7,    2,    2,    8,    8,    2,    8,    5    },    // 2^941
    {    284,    3,    7,    1,    7,    5,    4,    2,    2,    7,    1,    1,    9,    4,    4,    5,    7,    6,    5,    7,    0    },    // 2^942
    {    284,    7,    4,    3,    5,    0,    8,    4,    5,    4,    2,    3,    8,    8,    9,    1,    5,    3,    1,    4,    0    },    // 2^943
    {    285,    1,    4,    8,    7,    0,    1,    6,    9,    0,    8,    4,    7,    7,    7,    8,    3,    0,    6,    2,    8    },    // 2^944
    {    285,    2,    9,    7,    4,    0,    3,    3,    8,    1,    6,    9,    5,    5,    5,    6,    6,    1,    2,    5,    6    },    // 2^945
    {    285,    5,    9,    4,    8,    0,    6,    7,    6,    3,    3,    9,    1,    1,    1,    3,    2,    2,    5,    1,    2    },    // 2^946
    {    286,    1,    1,    8,    9,    6,    1,    3,    5,    2,    6,    7,    8,    2,    2,    2,    6,    4,    5,    0,    2    },    // 2^947
    {    286,    2,    3,    7,    9,    2,    2,    7,    0,    5,    3,    5,    6,    4,    4,    5,    2,    9,    0,    0,    5    },    // 2^948
    {    286,    4,    7,    5,    8,    4,    5,    4,    1,    0,    7,    1,    2,    8,    9,    0,    5,    8,    0,    1,    0    },    // 2^949
    {    286,    9,    5,    1,    6,    9,    0,    8,    2,    1,    4,    2,    5,    7,    8,    1,    1,    6,    0,    1,    9    },    // 2^950
    {    287,    1,    9,    0,    3,    3,    8,    1,    6,    4,    2,    8,    5,    1,    5,    6,    2,    3,    2,    0,    4    },    // 2^951
    {    287,    3,    8,    0,    6,    7,    6,    3,    2,    8,    5,    7,    0,    3,    1,    2,    4,    6,    4,    0,    8    },    // 2^952
    {    287,    7,    6,    1,    3,    5,    2,    6,    5,    7,    1,    4,    0,    6,    2,    4,    9,    2,    8,    1,    5    },    // 2^953
    {    288,    1,    5,    2,    2,    7,    0,    5,    3,    1,    4,    2,    8,    1,    2,    4,    9,    8,    5,    6,    3    },    // 2^954
    {    288,    3,    0,    4,    5,    4,    1,    0,    6,    2,    8,    5,    6,    2,    4,    9,    9,    7,    1,    2,    6    },    // 2^955
    {    288,    6,    0,    9,    0,    8,    2,    1,    2,    5,    7,    1,    2,    4,    9,    9,    9,    4,    2,    5,    2    },    // 2^956
    {    289,    1,    2,    1,    8,    1,    6,    4,    2,    5,    1,    4,    2,    4,    9,    9,    9,    8,    8,    5,    0    },    // 2^957
    {    289,    2,    4,    3,    6,    3,    2,    8,    5,    0,    2,    8,    4,    9,    9,    9,    9,    7,    7,    0,    1    },    // 2^958
    {    289,    4,    8,    7,    2,    6,    5,    7,    0,    0,    5,    6,    9,    9,    9,    9,    9,    5,    4,    0,    2    },    // 2^959
    {    289,    9,    7,    4,    5,    3,    1,    4,    0,    1,    1,    3,    9,    9,    9,    9,    9,    0,    8,    0,    4    },    // 2^960
    {    290,    1,    9,    4,    9,    0,    6,    2,    8,    0,    2,    2,    7,    9,    9,    9,    9,    8,    1,    6,    1    },    // 2^961
    {    290,    3,    8,    9,    8,    1,    2,    5,    6,    0,    4,    5,    5,    9,    9,    9,    9,    6,    3,    2,    1    },    // 2^962
    {    290,    7,    7,    9,    6,    2,    5,    1,    2,    0,    9,    1,    1,    9,    9,    9,    9,    2,    6,    4,    3    },    // 2^963
    {    291,    1,    5,    5,    9,    2,    5,    0,    2,    4,    1,    8,    2,    3,    9,    9,    9,    8,    5,    2,    9    },    // 2^964
    {    291,    3,    1,    1,    8,    5,    0,    0,    4,    8,    3,    6,    4,    7,    9,    9,    9,    7,    0,    5,    7    },    // 2^965
    {    291,    6,    2,    3,    7,    0,    0,    0,    9,    6,    7,    2,    9,    5,    9,    9,    9,    4,    1,    1,    4    },    // 2^966
    {    292,    1,    2,    4,    7,    4,    0,    0,    1,    9,    3,    4,    5,    9,    1,    9,    9,    8,    8,    2,    3    },    // 2^967
    {    292,    2,    4,    9,    4,    8,    0,    0,    3,    8,    6,    9,    1,    8,    3,    9,    9,    7,    6,    4,    6    },    // 2^968
    {    292,    4,    9,    8,    9,    6,    0,    0,    7,    7,    3,    8,    3,    6,    7,    9,    9,    5,    2,    9,    1    },    // 2^969
    {    292,    9,    9,    7,    9,    2,    0,    1,    5,    4,    7,    6,    7,    3,    5,    9,    9,    0,    5,    8,    3    },    // 2^970
    {    293,    1,    9,    9,    5,    8,    4,    0,    3,    0,    9,    5,    3,    4,    7,    1,    9,    8,    1,    1,    7    },    // 2^971
    {    293,    3,    9,    9,    1,    6,    8,    0,    6,    1,    9,    0,    6,    9,    4,    3,    9,    6,    2,    3,    3    },    // 2^972
    {    293,    7,    9,    8,    3,    3,    6,    1,    2,    3,    8,    1,    3,    8,    8,    7,    9,    2,    4,    6,    6    },    // 2^973
    {    294,    1,    5,    9,    6,    6,    7,    2,    2,    4,    7,    6,    2,    7,    7,    7,    5,    8,    4,    9,    3    },    // 2^974
    {    294,    3,    1,    9,    3,    3,    4,    4,    4,    9,    5,    2,    5,    5,    5,    5,    1,    6,    9,    8,    7    },    // 2^975
    {    294,    6,    3,    8,    6,    6,    8,    8,    9,    9,    0,    5,    1,    1,    1,    0,    3,    3,    9,    7,    3    },    // 2^976
    {    295,    1,    2,    7,    7,    3,    3,    7,    7,    9,    8,    1,    0,    2,    2,    2,    0,    6,    7,    9,    5    },    // 2^977
    {    295,    2,    5,    5,    4,    6,    7,    5,    5,    9,    6,    2,    0,    4,    4,    4,    1,    3,    5,    8,    9    },    // 2^978
    {    295,    5,    1,    0,    9,    3,    5,    1,    1,    9,    2,    4,    0,    8,    8,    8,    2,    7,    1,    7,    8    },    // 2^979
    {    296,    1,    0,    2,    1,    8,    7,    0,    2,    3,    8,    4,    8,    1,    7,    7,    6,    5,    4,    3,    6    },    // 2^980
    {    296,    2,    0,    4,    3,    7,    4,    0,    4,    7,    6,    9,    6,    3,    5,    5,    3,    0,    8,    7,    1    },    // 2^981
    {    296,    4,    0,    8,    7,    4,    8,    0,    9,    5,    3,    9,    2,    7,    1,    0,    6,    1,    7,    4,    3    },    // 2^982
    {    296,    8,    1,    7,    4,    9,    6,    1,    9,    0,    7,    8,    5,    4,    2,    1,    2,    3,    4,    8,    5    },    // 2^983
    {    297,    1,    6,    3,    4,    9,    9,    2,    3,    8,    1,    5,    7,    0,    8,    4,    2,    4,    6,    9,    7    },    // 2^984
    {    297,    3,    2,    6,    9,    9,    8,    4,    7,    6,    3,    1,    4,    1,    6,    8,    4,    9,    3,    9,    4    },    // 2^985
    {    297,    6,    5,    3,    9,    9,    6,    9,    5,    2,    6,    2,    8,    3,    3,    6,    9,    8,    7,    8,    8    },    // 2^986
    {    298,    1,    3,    0,    7,    9,    9,    3,    9,    0,    5,    2,    5,    6,    6,    7,    3,    9,    7,    5,    8    },    // 2^987
    {    298,    2,    6,    1,    5,    9,    8,    7,    8,    1,    0,    5,    1,    3,    3,    4,    7,    9,    5,    1,    5    },    // 2^988
    {    298,    5,    2,    3,    1,    9,    7,    5,    6,    2,    1,    0,    2,    6,    6,    9,    5,    9,    0,    3,    1    },    // 2^989
    {    299,    1,    0,    4,    6,    3,    9,    5,    1,    2,    4,    2,    0,    5,    3,    3,    9,    1,    8,    0,    6    },    // 2^990
    {    299,    2,    0,    9,    2,    7,    9,    0,    2,    4,    8,    4,    1,    0,    6,    7,    8,    3,    6,    1,    2    },    // 2^991
    {    299,    4,    1,    8,    5,    5,    8,    0,    4,    9,    6,    8,    2,    1,    3,    5,    6,    7,    2,    2,    5    },    // 2^992
    {    299,    8,    3,    7,    1,    1,    6,    0,    9,    9,    3,    6,    4,    2,    7,    1,    3,    4,    4,    4,    9    },    // 2^993
    {    300,    1,    6,    7,    4,    2,    3,    2,    1,    9,    8,    7,    2,    8,    5,    4,    2,    6,    8,    9,    0    },    // 2^994
    {    300,    3,    3,    4,    8,    4,    6,    4,    3,    9,    7,    4,    5,    7,    0,    8,    5,    3,    7,    8,    0    },    // 2^995
    {    300,    6,    6,    9,    6,    9,    2,    8,    7,    9,    4,    9,    1,    4,    1,    7,    0,    7,    5,    5,    9    },    // 2^996
    {    301,    1,    3,    3,    9,    3,    8,    5,    7,    5,    8,    9,    8,    2,    8,    3,    4,    1,    5,    1,    2    },    // 2^997
    {    301,    2,    6,    7,    8,    7,    7,    1,    5,    1,    7,    9,    6,    5,    6,    6,    8,    3,    0,    2,    4    },    // 2^998
    {    301,    5,    3,    5,    7,    5,    4,    3,    0,    3,    5,    9,    3,    1,    3,    3,    6,    6,    0,    4,    7    },    // 2^999
    {    302,    1,    0,    7,    1,    5,    0,    8,    6,    0,    7,    1,    8,    6,    2,    6,    7,    3,    2,    0,    9    },    // 2^1000
    {    302,    2,    1,    4,    3,    0,    1,    7,    2,    1,    4,    3,    7,    2,    5,    3,    4,    6,    4,    1,    9    },    // 2^1001
    {    302,    4,    2,    8,    6,    0,    3,    4,    4,    2,    8,    7,    4,    5,    0,    6,    9,    2,    8,    3,    8    },    // 2^1002
    {    302,    8,    5,    7,    2,    0,    6,    8,    8,    5,    7,    4,    9,    0,    1,    3,    8,    5,    6,    7,    6    },    // 2^1003
    {    303,    1,    7,    1,    4,    4,    1,    3,    7,    7,    1,    4,    9,    8,    0,    2,    7,    7,    1,    3,    5    },    // 2^1004
    {    303,    3,    4,    2,    8,    8,    2,    7,    5,    4,    2,    9,    9,    6,    0,    5,    5,    4,    2,    7,    0    },    // 2^1005
    {    303,    6,    8,    5,    7,    6,    5,    5,    0,    8,    5,    9,    9,    2,    1,    1,    0,    8,    5,    4,    1    },    // 2^1006
    {    304,    1,    3,    7,    1,    5,    3,    1,    0,    1,    7,    1,    9,    8,    4,    2,    2,    1,    7,    0,    8    },    // 2^1007
    {    304,    2,    7,    4,    3,    0,    6,    2,    0,    3,    4,    3,    9,    6,    8,    4,    4,    3,    4,    1,    6    },    // 2^1008
    {    304,    5,    4,    8,    6,    1,    2,    4,    0,    6,    8,    7,    9,    3,    6,    8,    8,    6,    8,    3,    3    },    // 2^1009
    {    305,    1,    0,    9,    7,    2,    2,    4,    8,    1,    3,    7,    5,    8,    7,    3,    7,    7,    3,    6,    7    },    // 2^1010
    {    305,    2,    1,    9,    4,    4,    4,    9,    6,    2,    7,    5,    1,    7,    4,    7,    5,    4,    7,    3,    3    },    // 2^1011
    {    305,    4,    3,    8,    8,    8,    9,    9,    2,    5,    5,    0,    3,    4,    9,    5,    0,    9,    4,    6,    6    },    // 2^1012
    {    305,    8,    7,    7,    7,    7,    9,    8,    5,    1,    0,    0,    6,    9,    9,    0,    1,    8,    9,    3,    2    },    // 2^1013
    {    306,    1,    7,    5,    5,    5,    5,    9,    7,    0,    2,    0,    1,    3,    9,    8,    0,    3,    7,    8,    6    },    // 2^1014
    {    306,    3,    5,    1,    1,    1,    1,    9,    4,    0,    4,    0,    2,    7,    9,    6,    0,    7,    5,    7,    3    },    // 2^1015
    {    306,    7,    0,    2,    2,    2,    3,    8,    8,    0,    8,    0,    5,    5,    9,    2,    1,    5,    1,    4,    6    },    // 2^1016
    {    307,    1,    4,    0,    4,    4,    4,    7,    7,    6,    1,    6,    1,    1,    1,    8,    4,    3,    0,    2,    9    },    // 2^1017
    {    307,    2,    8,    0,    8,    8,    9,    5,    5,    2,    3,    2,    2,    2,    3,    6,    8,    6,    0,    5,    8    },    // 2^1018
    {    307,    5,    6,    1,    7,    7,    9,    1,    0,    4,    6,    4,    4,    4,    7,    3,    7,    2,    1,    1,    7    },    // 2^1019
    {    308,    1,    1,    2,    3,    5,    5,    8,    2,    0,    9,    2,    8,    8,    9,    4,    7,    4,    4,    2,    3    },    // 2^1020
    {    308,    2,    2,    4,    7,    1,    1,    6,    4,    1,    8,    5,    7,    7,    8,    9,    4,    8,    8,    4,    7    },    // 2^1021
    {    308,    4,    4,    9,    4,    2,    3,    2,    8,    3,    7,    1,    5,    5,    7,    8,    9,    7,    6,    9,    3    },    // 2^1022
    {    308,    8,    9,    8,    8,    4,    6,    5,    6,    7,    4,    3,    1,    1,    5,    7,    9,    5,    3,    8,    6    }    // 2^1023
};

} // namespace detail
} // namespace ut
} // namespace nw

#endif //  NW_UT_FORMAT_PARSER_16_H_
