﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_BIT_FLAG_H_
#define NW_UT_BIT_FLAG_H_

#include <nw/types.h>
#include <nw/assert.h>

namespace nw {
namespace ut {

//---------------------------------------------------------------------------
//! @brief ビットフラグを扱うためのクラスです。
//!
//! @details
//! T は u8/u16/u32 のみ有効です。実際に使用する際は、
//! BitFlag8,BitFlag16,BitFlag32の何れかをご使用ください。
//!
//! @since 2009/09/18 初版。
//---------------------------------------------------------------------------
template <typename T>
class BitFlag
{
    NW_STATIC_ASSERT( sizeof(T) <= 4 );
public:
    //! @brief デフォルトコンストラクタ
    //!
    //! @details
    //! 0で初期化されます。
    //!
    //! @since 2010/03/26 初版。
    //!
    BitFlag() : m_Bits(0) {}

    //! @brief 引数で指定された値で初期化するコンストラクタです。
    //!
    //! @param bits 初期状態の値
    //!
    //! @since 2010/03/26 初版。
    //!
    explicit BitFlag( T bits ) : m_Bits(bits) {}

    //! @brief Tへのキャスト
    //!
    //! @since 2010/03/26 初版。
    //!
    operator T() { return m_Bits; }

    //! @brief 全てのbitが0か否かを返します。
    //!
    //! @return 全てのbitが0ならtrue、そうでなければfalse
    //!
    //! @since 2010/03/26 初版。
    //!
    bool IsZero() const { return m_Bits == 0; }

    //! @brief bitを全て0にします。
    //!
    //! @since 2010/03/26 初版。
    //!
    void SetAllZero()
    {
        m_Bits =    static_cast<T>(0);
    }

    //! @brief bitを全て1にします。
    //!
    //! @since 2010/03/26 初版。
    //!
    void SetAllOne()
    {
        m_Bits =    static_cast<T>(-1);
    }

    //! @brief 直接bitを設定します。
    //!
    //! @param bits 設定したい値
    //!
    //! @since 2010/03/26 初版。
    //!
    void SetDirect( T bits ) { m_Bits = bits; }

    //! @brief 直接bitを取得します。
    //!
    //! @return 内部的に保持しているbit値
    //!
    //! @since 2010/03/26 初版。
    //!
    T GetDirect() const { return m_Bits; }

    //! @brief ビットフラグを格納している内部変数のアドレスを直接取得します。
    //!
    //! @details
    //! アドレスを取得した後にBitFlag自体があったメモリ領域が解放された
    //! 場合、不正アクセスの危険がありますので、注意してご使用ください。
    //!
    //! @return ビットフラグを格納している内部変数のアドレス
    //!
    //! @since 2010/03/26 初版。
    //!
    T* GetPtr() { return &m_Bits; }

    //! @brief ビットフラグを格納している内部変数のconstアドレスを直接取得します。
    //!
    //! @details
    //! アドレスを取得した後にBitFlag自体があったメモリ領域が解放された
    //! 場合、不正アクセスの危険がありますので、注意してご使用ください。
    //!
    //! @return ビットフラグを格納している内部変数のアドレス
    //!
    //! @since 2010/03/26 初版。
    //!
    const T* GetPtr() const { return &m_Bits; }

    //! @brief フラグ変数の大きさを返します。(バイト)
    //!
    //! @return この変数の大きさ(バイト)
    //!
    //! @since 2010/03/26 初版。
    //!
    size_t GetByteSize() const { return sizeof(T); }

    //----------------------------------------
    //! @name mask による操作
    //@{
    //! @brief maskで指定されたbitを1にします。
    //!
    //! @param mask 1にするbitを1にした値
    //!
    //! @since 2010/03/26 初版。
    //!
    void SetMaskOn( T mask ) { m_Bits |= mask;  }

    //! @brief maskで指定されたbitを0にします。
    //!
    //! @param mask 0にするbitを1にした値
    //!
    //! @since 2010/03/26 初版。
    //!
    void SetMaskOff( T mask ) { m_Bits &= ~mask; }

    //! @brief maskで指定されたbitを反転します。
    //!
    //! @param mask 反転するbitを1にした値
    //!
    //! @since 2010/03/26 初版。
    //!
    void ToggleMask( T mask ) { m_Bits ^= mask; }

    //! @brief maskで指定されたbitをbの値にします。
    //!
    //! @param mask bの値にするbitを1にした値
    //! @param b    セットしたい値
    //!
    //! @since 2010/03/26 初版。
    //!
    void ChangeMask( T mask, bool b )
    {
        if ( b ) { SetMaskOn(mask); }   else { SetMaskOff(mask);    }
    }

    //! @brief  与えられたマスクの何れかのbitが立っているか調べます。
    //!
    //! @param mask 調べるビットマスク
    //! @return     maskで与えられたビットのうち、何れかが1ならtrueを返します。
    //!
    //! @since 2010/03/26 初版。
    //!
    bool IsMaskOn( T mask ) const { return static_cast<bool>((m_Bits & mask) != 0); }

    //! @brief  与えられたマスクのbitの何れかが立っていないかを調べます。
    //!
    //! @param mask 調べるビットマスク
    //! @return     maskで与えられたビットのうち、全てが0ならtrueを返します。
    //!
    //! @since 2010/03/26 初版。
    //!
    bool IsMaskOff( T mask ) const { return ! IsMaskOn(mask); }

    //! @brief  与えられたビットマスクとのビット積を返します。
    //!
    //! @param mask ビット積を取得したいビットマスク
    //! @return ビット積
    //!
    //! @since 2010/03/26 初版。
    //!
    T GetAndMask( T mask ) const { return static_cast<T>(m_Bits & mask); }

    //! @brief  下位から数えて、bit目のbitが1となるマスクを作成します。
    //!
    //! @param bit  1とするbit(下位から数えて)
    //! @return マスク
    //!
    //! @since 2010/03/26 初版。
    //!
    static T MakeMask( int bit )
    {
        NW_ASSERT(static_cast<u32>(bit) < sizeof(T)*8);
        return static_cast<T>(1 << bit);
    }
    //@}
    //----------------------------------------
    //! @name bit による操作
    //@{

    //! @brief  下位から数えて、bit目のbitを1にする。
    //!
    //! @param bit  1とするbit(下位から数えて)
    //!
    //! @since 2010/03/26 初版。
    //!
    void SetBitOn( int bit ) { SetMaskOn( MakeMask( bit ) ); }

    //! @brief  下位から数えて、bit目のbitを0にする。
    //!
    //! @param bit  0とするbit(下位から数えて)
    //!
    //! @since 2010/03/26 初版。
    //!
    void SetBitOff( int bit ) { SetMaskOff( MakeMask( bit ) ); }

    //! @brief  下位から数えて、bit目のbitを反転する。
    //!
    //! @param bit  反転するbit(下位から数えて)
    //!
    //! @since 2010/03/26 初版。
    //!
    void ToggleBit( int bit ) { ToggleMask( MakeMask( bit ) ); }

    //! @brief  下位から数えて、bit目のbitを指定した値にする。
    //!
    //! @param bit  対象とするbit(下位から数えて)
    //! @param b    セットする値
    //!
    //! @since 2010/03/26 初版。
    //!
    void ChangeBit( int bit, bool b ) { ChangeMask( MakeMask( bit ), b ); }

    //! @brief  下位から数えて、bit目のbitが1ならtrueを返す。
    //!
    //! @param bit  調べるbit(下位から数えて)
    //! @return     1ならtrue
    //!
    //! @since 2010/03/26 初版。
    //!
    bool IsBitOn( int bit ) const { return IsMaskOn( MakeMask( bit ) ); }

    //! @brief  下位から数えて、bit目のbitが0ならtrueを返す。
    //!
    //! @param bit  調べるbit(下位から数えて)
    //! @return     0ならtrue
    //!
    //! @since 2010/03/26 初版。
    //!
    bool IsBitOff( int bit ) const { return IsMaskOff( MakeMask( bit ) ); }
    //@}

protected:
    T   m_Bits;
};

/// 8bitまでのフラグを扱うことのできるBitFlagです。
typedef BitFlag<u8>     BitFlag8;
/// 16bitまでのフラグを扱うことのできるBitFlagです。
typedef BitFlag<u16>    BitFlag16;
/// 32bitまでのフラグを扱うことのできるBitFlagです。
typedef BitFlag<u32>    BitFlag32;

} /* namespace ut */
} /* namespace nw */

#endif /* NW_UT_BIT_FLAG_H_ */
