﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_THREAD_H_
#define NW_UT_THREAD_H_

#include <nw/config.h>

#if defined(NW_PLATFORM_CAFE)
    #include <cafe/os.h>
#elif defined(NW_PLATFORM_ANDROID)
    #include <semaphore.h>
#elif defined(NW_PLATFORM_IOS)
    #include <dispatch/dispatch.h>
#endif

#if defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
    #include <pthread.h>
#endif

namespace nw {
namespace ut {

//---------------------------------------------------------------------------
//! @brief        スレッドハンドラクラスです。
//---------------------------------------------------------------------------
class ThreadHandler
{
public:
    virtual ~ThreadHandler() {}

    virtual void ThreadHandlerProc() = 0;
};

//---------------------------------------------------------------------------
//! @brief        スレッドを扱うクラスです。
//---------------------------------------------------------------------------
class Thread
{
#if defined( NW_PLATFORM_CAFE )
    static const u32 OSTHREAD_ALIGNMENT = 8;
#endif

public:
    //! @brief 生成パラメータです。
    struct CreateArg
    {
        s32   priority;         //!< スレッド優先度です。
        void* stackBase;        //!< スタックポインタです。
        u32   stackSize;        //!< スタックサイズです。
        u32   attribute;        //!< 属性です。
#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK ) || defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        u32   creationFlags;    //!< 作成オプションです。(Windows 上で動作する場合にのみ機能します。)
#elif defined( NW_PLATFORM_CAFE )
        const char* nameString; //!< スレッドに付ける名前です。
#endif

        //! @brief コンストラクタです。
        CreateArg() :
            priority( 0 ),
            stackBase( NULL ),
            stackSize( 0 ),
            attribute( 0 ),
#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK ) || defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
            // TODO: NintendoSdk 対応後、このコメントを削除してください。
            creationFlags( 0 )
#elif defined( NW_PLATFORM_CAFE )
            nameString( NULL )
#endif
        {}
    };

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    handler   スレッドハンドラへのポインタです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit Thread( ThreadHandler* handler );

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    ~Thread();

    //---------------------------------------------------------------------------
    //! @brief        スレッドを生成します。
    //!
    //! @param[in]    arg      生成パラメータです。
    //! @return       スレッドの生成に成功すれば true を返します。
    //---------------------------------------------------------------------------
    bool Create( const CreateArg& arg );

    //---------------------------------------------------------------------------
    //! @brief        スレッドを破棄します。
    //---------------------------------------------------------------------------
    void Destroy();

    //---------------------------------------------------------------------------
    //! @brief        スレッド優先度を設定します。
    //!
    //! @param[in]    priority  スレッド優先度です。
    //---------------------------------------------------------------------------
    void SetPriority( s32 priority );

    //---------------------------------------------------------------------------
    //! @brief        スレッド優先度を取得します。
    //!
    //! @return       スレッド優先度を返します。
    //---------------------------------------------------------------------------
    s32 GetPriority() const;

    //---------------------------------------------------------------------------
    //! @brief        スレッドが終了するまで待ちます。
    //---------------------------------------------------------------------------
    void Join();

    //---------------------------------------------------------------------------
    //! @brief        スレッドの実行を再開します。
    //---------------------------------------------------------------------------
    void Resume();

    //---------------------------------------------------------------------------
    //! @brief        スレッドを終了させます。
    //---------------------------------------------------------------------------
    void Exit();

    //---------------------------------------------------------------------------
    //! @brief        スレッドをスリープさせます。
    //---------------------------------------------------------------------------
    void Sleep();

    //---------------------------------------------------------------------------
    //! @brief        スリープさせたスレッドを起こします。
    //---------------------------------------------------------------------------
    void Wakeup();

    //---------------------------------------------------------------------------
    //! @brief        スレッドが停止状態かを取得します。
    //!
    //! @return       停止状態の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsTerminated();


private:
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    static unsigned int __stdcall ThreadFunc(void* arg);
    void* m_Thread;
    unsigned int m_ThreadID;
#elif defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
    static void* ThreadFunc(void* arg);
    pthread_t m_Thread;
#else
    static int ThreadFunc(int inrArg, void* ptrArg);
    u8 m_ThreadBuffer[sizeof(OSThread)+OSTHREAD_ALIGNMENT];
    OSThread* m_Thread;
    OSThreadQueue m_ThreadQueue;
#endif

#if defined(NW_PLATFORM_ANDROID)
    sem_t m_ResumeSemaphore;
#elif defined(NW_PLATFORM_IOS)
    dispatch_semaphore_t m_ResumeSemaphore;
#endif

    CreateArg m_Arg;
    ThreadHandler* m_pThreadHandler;
};

} // namespace ut
} // namespace nw

#endif // NW_UT_THREAD_H_

