﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_SPIN_LOCK_H_
#define NW_UT_SPIN_LOCK_H_

#include <nw/types.h>

#if defined(NW_PLATFORM_CAFE)
    #include <cafe/os/OSSync.h>
#endif

namespace nw {
namespace ut {

//---------------------------------------------------------------------------
//! @brief    SpinLock を使用したロックオブジェクトの具象クラスです。
//---------------------------------------------------------------------------
class SpinLock
{
public:
    /* ctor */ SpinLock() : m_IsIntialized(false) {}

    /* dtor */ ~SpinLock() {}

#if defined(NW_PLATFORM_CAFE)
    void Initialize()
    {
        if (!m_IsIntialized)
        {
            m_IsIntialized = true;
            OSInitSpinLock( &m_SpinLock );
        }
    }

    void Finalize()
    {
        m_IsIntialized = false;
    }

    void Lock()
    {
        NW_ASSERT(m_IsIntialized);
        OSAcquireSpinLock( &m_SpinLock );
    }

    bool TryLock()
    {
        NW_ASSERT(m_IsIntialized);
        return (OSTryAcquireSpinLock( &m_SpinLock ) != FALSE);
    }

    void Unlock()
    {
        NW_ASSERT(m_IsIntialized);
        OSReleaseSpinLock( &m_SpinLock );
    }
#else

    void Initialize();
    void Finalize();
    void Lock();
    bool TryLock();
    void Unlock();

#endif


private:
  #if defined(NW_PLATFORM_CAFE)
    OSSpinLock m_SpinLock;
  #elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    static const u32 CRITICAL_SECTION_SIZE = 24;

    u32 m_CriticalSection[ CRITICAL_SECTION_SIZE / sizeof(u32) ];
  #endif

    bool    m_IsIntialized;
};

} // namespace ut
} // namespace nw

#endif // NW_UT_SPIN_LOCK_H_

