﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SPY_FND_MEMORY_H_
#define NW_SND_SPY_FND_MEMORY_H_

#include <nw/snd/spy/fnd/basis/sndspyfnd_Inlines.h>
#include <nw/snd/spy/fnd/basis/sndspyfnd_Config.h>

#if defined(NW_PLATFORM_CAFE)
#include <cafe/mem.h>
#elif defined(NW_PLATFORM_CTR)
#include <nn/fnd.h>
#endif

namespace nw {
namespace snd {
namespace spy {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
//! @brief  メモリ特性です。
//---------------------------------------------------------------------------
class MemoryTraits
{
public:
    static const s32 DEFAULT_ALIGNMENT = 4; //!< デフォルトのアライメント
};

//---------------------------------------------------------------------------
//! @brief  メモリアロケータのインターフェイスです。
//---------------------------------------------------------------------------
class IAllocator
{
public:
    virtual ~IAllocator() { }

public:
    virtual void* Alloc(size_t size, s32 alignment = MemoryTraits::DEFAULT_ALIGNMENT) = 0;
    virtual void Free(void* ptr) = 0;
};

//---------------------------------------------------------------------------
//! @brief  前から順にスタック方式でメモリを割り当てるヒープクラスです。
//---------------------------------------------------------------------------
class FrameHeap
{
    NW_DISALLOW_COPY_AND_ASSIGN(FrameHeap);

public: // コンストラクタ
    FrameHeap() :
        m_Buffer(NULL),
        m_BufferLength(0),
        m_UsedLength(0)
    {
    }

    virtual ~FrameHeap()
    {
        Finalize();
    }

public: // メソッド
    void Initialize(void* buffer, size_t length);
    void Finalize();

    bool IsInitialized() const
    {
        return m_Buffer != NULL;
    }

    size_t GetUsedLength() const { return m_UsedLength; }

    size_t GetFreeLength(s32 alignment = MemoryTraits::DEFAULT_ALIGNMENT) const;

    virtual void* Alloc(size_t size, s32 alignment = MemoryTraits::DEFAULT_ALIGNMENT);

    void Clear()
    {
        m_UsedLength = 0;
    }

private: // メンバ変数
    void*  m_Buffer;         //!< バッファです。
    size_t m_BufferLength;   //!< バッファの長さです。
    size_t m_UsedLength;     //!< 使用中のメモリサイズです。
};

//---------------------------------------------------------------------------
//! @brief  固定長ユニット単位でメモリを割り当てるヒープクラスです。
//---------------------------------------------------------------------------
class UnitHeap : public IAllocator
{
    NW_DISALLOW_COPY_AND_ASSIGN(UnitHeap);

public: // コンストラクタ
    UnitHeap() :
#if (defined(NW_DEBUG) || defined(NW_DEVELOP))
        m_AllocatedUnitCount(0),
#endif
        m_Buffer(NULL),
        m_BufferLength(0),
        m_UnitLength(0),
        m_FirstFreeUnit(NULL)
    {
    }

    virtual ~UnitHeap()
    {
        Finalize();
    }

public: // メソッド
    void Initialize(
        void* buffer,
        size_t length,
        size_t unitLength,
        u32 unitCount,
        u32 unitAlignment = MemoryTraits::DEFAULT_ALIGNMENT);

    void Finalize();

    bool IsInitialized() const
    {
        return m_Buffer != NULL;
    }

    size_t GetBufferLength() const
    {
        return m_BufferLength;
    }

    size_t GetUsedLength() const { return m_AllocatedUnitCount * m_UnitLength; }

    virtual void* Alloc(size_t size = 0, s32 alignment = MemoryTraits::DEFAULT_ALIGNMENT);
    virtual void Free(void* ptr);

private: // メソッド
    void* GetNextFreeUnit(void* unit) const;

private: // メンバ変数
    void*  m_Buffer;         //!< バッファです。
    size_t m_BufferLength;   //!< バッファの長さです。
    size_t m_UnitLength;     //!< ユニットサイズです。
    void*  m_FirstFreeUnit;  //!< 最初の空きユニットです。
    u32    m_AllocatedUnitCount;   //!< 割り当て済みユニット数です。
};

#if defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
#else

//---------------------------------------------------------------------------
//! @brief  拡張ヒープマネージャを利用したヒープクラスです。
//---------------------------------------------------------------------------
class ExpandedHeap : public IAllocator
{
    NW_DISALLOW_COPY_AND_ASSIGN(ExpandedHeap);

public: // コンストラクタ
    ExpandedHeap() : m_HeapHandle(NULL) { }

    virtual ~ExpandedHeap()
    {
        Finalize();
    }

public: // メソッド
    bool Initialize(void* buffer, size_t length);

    void Finalize();

    bool IsInitialized() const
    {
        return m_HeapHandle != NULL;
    }

    virtual void* Alloc(size_t size, s32 alignment = MemoryTraits::DEFAULT_ALIGNMENT);
    virtual void Free(void* ptr);

private: // メンバ変数
#if defined( NW_PLATFORM_CAFE )
    MEMHeapHandle m_HeapHandle;
#elif defined( NW_PLATFORM_CTR )
    nn::fnd::ExpHeap m_Heap;
    nn::fnd::ExpHeap* m_HeapHandle;
#endif
};

#endif

} // namespace nw::snd::spy::internal::fnd
} // namespace nw::snd::spy::internal
} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_FND_MEMORY_H_
