﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_VOICE_H_
#define NW_SND_VOICE_H_

#include <nw/types.h>

#if defined( NW_PLATFORM_CAFE )
  #include <cafe/os.h>
  #include <cafe/ax.h>
#else
  #include <winext/cafe/os.h>
  #include <winext/cafe/ax.h>
#endif

#include <nw/snd/snd_Config.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_Adpcm.h>
#include <nw/snd/snd_AxVoice.h>
#include <nw/snd/snd_NwVoice.h>

/*! @file
    @brief      Voice に関する関数、およびクラス定義
*/

namespace nw  { namespace snd {

namespace internal
{

class NwVoiceManager;

/*!
    @brief      Voice を操作するクラスです。
 */
class Voice
{
    friend class AxVoice;
    friend class NwVoice;

private:
    NW_DISALLOW_COPY_AND_ASSIGN(Voice);

public:
    static const u32 PRIORITY_MIN = 0;
    static const u32 PRIORITY_MAX = 255;
    static const u32 PRIORITY_NODROP = PRIORITY_MAX;

    Voice();
    ~Voice();

    static void detail_SetNwVoiceManager( NwVoiceManager* voiceManager );

    bool AllocVoice( u32 priority, VoiceRendererType type = VOICE_RENDERER_SDK );
    void Free();

    bool IsAvailable() const;

    /*!
        @brief      ボイスの状態を設定します。
        @param[in]  state       状態
     */
    void SetState(VoiceState state);

    /*!
        @name       バッファ追加
        @{
     */
    /*!
        @brief      ボイスに対してサンプルデータ情報を追加します。
        @param[in]  waveBuffer  バッファ情報構造体のポインタ
     */
    void AppendWaveBuffer( WaveBuffer* waveBuffer );
    /*!
        @}
     */

    /*!
        @name       パラメータ設定
        @{
     */
    /*
        @brief      ボイスのチャンネル数を設定します。
        @param[in]  channelCount    チャンネル数
     */
//    void SetChannelCount(s32 channelCount);

    /*!
        @brief      ボイスにリンクされるサンプルの形式を設定します。
        @param[in]  format      サンプルの形式
     */
    void SetSampleFormat(SampleFormat format) { m_SampleFormat = format; }

    /*
        @brief      再生開始時にボリューム 0 からの短いフェードインを用いるかどうかを指定します。
        @param[in]  flag        On/Off フラグ
     */
//    void SetStartFrameFadeInFlag(bool flag);

    /*!
        @brief      ボイスのサンプリングレートを設定します。
        @param[in]  sampleRate  サンプリングレート
     */
    void SetSampleRate(u32 sampleRate) { m_SampleRate = sampleRate; }

    /*!
        @brief      Adpcm の係数を設定します。
        @param[in]  param Adpcm 係数構造体への参照
     */
    void SetAdpcmParam(const AdpcmParam& param) { m_AdpcmParam = param; }

    /*!
        @brief      ボイスの優先度を設定します。
        @param[in]  priority    優先度
     */
    void SetPriority(u32 priority);

    /*!
        @brief      ボイスのボリュームを設定します。
        @param[in]  volume      ボリューム値
     */
    void SetVolume(f32 volume) { m_VoiceParam.m_Volume = volume; }

    /*!
        @brief      ボイスのピッチを設定します。
        @param[in]  pitch       サンプリングレートに対する再生速度比
     */
    void SetPitch(f32 pitch) { m_VoiceParam.m_Pitch = pitch; }

    /*!
        @brief      ボイスの各チャンネルのゲインを設定します。
        @param[in]  mix    ゲイン構造体への参照
     */
    void SetTvMix( const OutputMix& mix ) { m_VoiceParam.m_TvMix = mix; }
    void SetDrcMix( u32 drcIndex, const OutputMix& mix )
    {
        NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
        m_VoiceParam.m_DrcMix[drcIndex] = mix;
    }
    void SetRemoteMix( const RemoteOutputMix& mix );

    void SetFrontBypass( bool isFrontBypass ) { m_VoiceParam.m_DrcFrontBypass = isFrontBypass; }

    void SetInterpolationType( u8 interpolationType ) { m_VoiceParam.m_InterpolationType = interpolationType; }

    void SetMonoFilter( bool enable, u16 cutoff = 0 );
    void SetBiquadFilter( bool enable, const BiquadFilterCoefficients* coef = NULL );
    void SetRemoteFilter( bool enable, u8 filter = 0 );

    void UpdateParam();

    // #ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    void UpdateVoiceStatus();
    // #endif

    /*!
        @name       再生状況取得
        @{
     */
    /*!
        @brief      ボイスの状態を取得します。
        @return     状態を返します。
     */
    VoiceState GetState() const { return m_State; }

    /*!
        @brief      使用中のバッファ内での再生位置を取得します。
        @return     再生位置をサンプル数で返します。
     */
    u32 GetPlayPosition() const;

    /*!
        @brief      ボイスの優先度を取得します。
        @return     優先度を返します。
     */
    s32 GetPriority() const { return m_Priority; }

    f32 GetVolume() const { return m_VoiceParam.m_Volume; }
    f32 GetPitch() const { return m_VoiceParam.m_Pitch; }

    const OutputMix& GetTvMix() const { return m_VoiceParam.m_TvMix; }
    const OutputMix& GetDrcMix( u32 drcIndex ) const
    {
        NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
        return m_VoiceParam.m_DrcMix[drcIndex];
    }
    const RemoteOutputMix& GetRemoteMix() const { return m_VoiceParam.m_RemoteMix; }

    /*!
        @brief      ボイスの再生状態を取得します。
        @return     ボイスの再生状態を返します。
     */
//    bool IsPlaying() const;
    /*!
        @}
     */

    /*!
        @}
     */

private:
    void Initialize( u32 priority );
    void FreeAllWaveBuffer();

    // #ifndef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    static void AxVoiceDisposeCallback( AxVoice* axVoice, void* arg );
    static void NwVoiceDisposeCallback( NwVoice* nwVoice, void* arg );
    // #endif

    //-------------------------
    // プライオリティ
    //-------------------------
    u32                 m_Priority;

    //-------------------------
    // パラメータ
    //-------------------------
    VoiceState m_State;
    VoiceParam m_VoiceParam;

    SampleFormat        m_SampleFormat;
    u32                 m_SampleRate;
    AdpcmParam          m_AdpcmParam;

    //-------------------------
    // ボイス実装への参照
    //-------------------------
    // #ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    u32 m_VoiceId;
    bool m_AxVoiceFlag;
    u32 m_PlayPosition;
    u32 m_VoiceInfoEnableFlag;
    u32 m_CommandTag;
    WaveBuffer*         m_WaveBufferListBegin;
    WaveBuffer*         m_WaveBufferListEnd;
    // #else
    AxVoice* m_pAxVoice;
    NwVoice* m_pNwVoice;
    // #endif

    static NwVoiceManager* s_pNwVoiceManager;
}; // class Voice


class VirtualVoiceManager
{
public:
    static const u32 INVALID_VOICE_ID = 0xffffffff;

    static const int VIRTUAL_VOICE_COUNT = 256; // NOTE: 32の倍数

    static VirtualVoiceManager& GetInstance();

    u32 AllocVirtualVoice();
    void FreeVirtualVoice(u32 id);

    void SetAxVoice( u32 voiceId, AxVoice* axVoice ) { m_AxVoiceTable[voiceId] = axVoice; }
    void SetNwVoice( u32 voiceId, NwVoice* nwVoice ) { m_NwVoiceTable[voiceId] = nwVoice; }

    AxVoice* GetAxVoice( u32 voiceId ) { return m_AxVoiceTable[voiceId]; }
    NwVoice* GetNwVoice( u32 voiceId ) { return m_NwVoiceTable[voiceId]; }

    VoiceInfo* GetVoiceInfo( u32 voiceId ) { return &m_VoiceInfoTable[m_VoiceInfoTableRead][voiceId]; }
    const VoiceInfo* GetVoiceInfo( u32 voiceId ) const { return &m_VoiceInfoTable[m_VoiceInfoTableRead][voiceId]; }

    void UpdateVoiceInfo();

private:
    VirtualVoiceManager()
    {
        Initialize();
    }

    void Initialize();

    static const int VIRTUAL_VOICE_ELEMENT_COUNT = VIRTUAL_VOICE_COUNT / 32;
    u32 sVirtualVoiceAllocationTable[VIRTUAL_VOICE_ELEMENT_COUNT];

    u32 m_VoiceInfoTableRead;
    AxVoice* m_AxVoiceTable[ VIRTUAL_VOICE_COUNT ];
    NwVoice* m_NwVoiceTable[ VIRTUAL_VOICE_COUNT ];
    VoiceInfo m_VoiceInfoTable[2][ VIRTUAL_VOICE_COUNT ];
};

}}} // namespace nw::snd

#endif //NW_SND_VOICE_H_
