﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SOUND_ARCHIVE_PARAMETERS_HOOK_H_
#define NW_SND_SOUND_ARCHIVE_PARAMETERS_HOOK_H_

#include <nw/snd/snd_SoundArchive.h>

namespace nw {
namespace snd {
namespace internal {

//! @brief  サウンドアーカイブパラメータへのアクセスをフックします。
class SoundArchiveParametersHook
{
protected:
    SoundArchiveParametersHook() : m_IsEnabled(true) { }
    virtual ~SoundArchiveParametersHook() { }

public:
    //---------------------------------------------------------------------------
    //! @brief         フックが有効かどうかを調べます。
    //!
    //! @return        有効な場合は true、無効な場合は false を返します。
    //---------------------------------------------------------------------------
    bool GetIsEnable() const { return m_IsEnabled; }

    //---------------------------------------------------------------------------
    //! @brief         フックの有効/無効を設定します。
    //!
    //! @param[in]     value     有効性を指定します。
    //---------------------------------------------------------------------------
    void SetIsEnable(bool value) { m_IsEnabled = value; }

    //---------------------------------------------------------------------------
    //! @brief         指定アイテムラベルがフック対象かどうかを調べます。
    //!
    //! @param[in]     itemLabel     アイテムラベルを指定します。
    //!
    //! @return        対象の場合は true、対象でない場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsTargetItem(const char* itemLabel)
    {
        return m_IsEnabled ? IsTargetItemImpl(itemLabel) : false;
    }

    const char* GetItemLabel(SoundArchive::ItemId id) const
    {
        if(m_IsEnabled)
        {
            return GetItemLabelImpl(id);
        }

        return NULL;
    }

    SoundArchive::ItemId GetItemId(const char* itemLabel) const
    {
        if(m_IsEnabled)
        {
            return GetItemIdImpl(itemLabel);
        }

        return SoundArchive::INVALID_ID;
    }

    SoundArchive::SoundType GetSoundType(const char* itemLabel)
    {
        if(m_IsEnabled)
        {
            return GetSoundTypeImpl(itemLabel);
        }

        return SoundArchive::SOUND_TYPE_INVALID;
    }

    bool ReadSoundInfo(SoundArchive::ItemId soundId, SoundArchive::SoundInfo* info) const
    {
        if(m_IsEnabled)
        {
            return ReadSoundInfoImpl(soundId, info);
        }

        return false;
    }

    bool ReadBankInfo(SoundArchive::ItemId bankId, SoundArchive::BankInfo* info) const
    {
        if(m_IsEnabled)
        {
            return ReadBankInfoImpl(bankId, info);
        }

        return false;
    }

    bool ReadPlayerInfo(SoundArchive::ItemId playerId, SoundArchive::PlayerInfo* info) const
    {
        if(m_IsEnabled)
        {
            return ReadPlayerInfoImpl(playerId, info);
        }

        return false;
    }

    bool ReadSoundGroupInfo(
        SoundArchive::ItemId soundGroupId,
        SoundArchive::SoundGroupInfo* info) const
    {
        if(m_IsEnabled)
        {
            return ReadSoundGroupInfoImpl(soundGroupId, info);
        }

        return false;
    }

    bool ReadGroupInfo(SoundArchive::ItemId groupId, SoundArchive::GroupInfo* info) const
    {
        if(m_IsEnabled)
        {
            return ReadGroupInfoImpl(groupId, info);
        }

        return false;
    }

    bool ReadWaveArchiveInfo(
        SoundArchive::ItemId warcId,
        SoundArchive::WaveArchiveInfo* info) const
    {
        if(m_IsEnabled)
        {
            return ReadWaveArchiveInfoImpl(warcId, info);
        }

        return false;
    }

    void ReadFileInfo(
        SoundArchive::FileId id,
        SoundArchive::FileInfo* info,
        int index=0) const
    {
        if(m_IsEnabled)
        {
            ReadFileInfoImpl(id, info, index);
        }
    }

    bool ReadSoundArchivePlayerInfo(SoundArchive::SoundArchivePlayerInfo* info) const
    {
        if(m_IsEnabled)
        {
            return ReadSoundArchivePlayerInfoImpl(info);
        }

        return false;
    }

    bool ReadSound3DInfo(
        SoundArchive::ItemId soundId,
        SoundArchive::Sound3DInfo* info) const
    {
        if(m_IsEnabled)
        {
            return ReadSound3DInfoImpl(soundId, info);
        }

        return false;
    }

    bool ReadSequenceSoundInfo(
        SoundArchive::ItemId soundId,
        SoundArchive::SequenceSoundInfo* info) const
    {
        if(m_IsEnabled)
        {
            return ReadSequenceSoundInfoImpl(soundId, info);
        }

        return false;
    }

    bool ReadStreamSoundInfo(
        SoundArchive::ItemId soundId,
        SoundArchive::StreamSoundInfo* info) const
    {
        if(m_IsEnabled)
        {
            return ReadStreamSoundInfoImpl(soundId, info);
        }

        return false;
    }

    bool ReadStreamSoundInfo2(
        SoundArchive::ItemId soundId,
        SoundArchive::StreamSoundInfo2* info ) const
    {
        if(m_IsEnabled)
        {
            return ReadStreamSoundInfo2Impl(soundId, info);
        }

        return false;
    }

    bool ReadWaveSoundInfo(
        SoundArchive::ItemId soundId,
        SoundArchive::WaveSoundInfo* info) const
    {
        if(m_IsEnabled)
        {
            return ReadWaveSoundInfoImpl(soundId, info);
        }

        return false;
    }

    bool ReadSoundUserParam(
        SoundArchive::ItemId soundId,
        int index,
        u32& value) const
    {
        return m_IsEnabled ? ReadSoundUserParamImpl(soundId, index, value) : false;
    }

    u32 GetSoundUserParam(
        SoundArchive::ItemId soundId,
        u32 userParam) const
    {
        return m_IsEnabled ? GetSoundUserParamImpl(soundId, userParam) : userParam;
    }

protected:
    //---------------------------------------------------------------------------
    //! @brief         指定アイテムラベルがフック対象かどうかを調べます。
    //!
    //! @param[in]     itemLabel     アイテムラベルを指定します。
    //!
    //! @return        対象の場合は true、対象でない場合は false を返します。
    //---------------------------------------------------------------------------
    virtual bool IsTargetItemImpl(const char* itemLabel)
    {
        (void)itemLabel;
        return false;
    }

    virtual const char* GetItemLabelImpl(SoundArchive::ItemId id) const
    {
        (void)id;
        return NULL;
    }

    virtual SoundArchive::ItemId GetItemIdImpl(const char* itemLabel) const
    {
        (void)itemLabel;
        return SoundArchive::INVALID_ID;
    }

    virtual SoundArchive::SoundType GetSoundTypeImpl(const char* itemLabel)
    {
        (void)itemLabel;
        return SoundArchive::SOUND_TYPE_INVALID;
    }

    virtual bool ReadSoundInfoImpl(SoundArchive::ItemId soundId, SoundArchive::SoundInfo* info) const
    {
        (void)soundId; (void)info;
        return false;
    }

    virtual bool ReadBankInfoImpl(SoundArchive::ItemId bankId, SoundArchive::BankInfo* info) const
    {
        (void)bankId; (void)info;
        return false;
    }

    virtual bool ReadPlayerInfoImpl(SoundArchive::ItemId playerId, SoundArchive::PlayerInfo* info) const
    {
        (void)playerId; (void)info;
        return false;
    }

    virtual bool ReadSoundGroupInfoImpl(
        SoundArchive::ItemId soundGroupId,
        SoundArchive::SoundGroupInfo* info) const
    {
        (void)soundGroupId; (void)info;
        return false;
    }

    virtual bool ReadGroupInfoImpl(SoundArchive::ItemId groupId, SoundArchive::GroupInfo* info) const
    {
        (void)groupId; (void)info;
        return false;
    }

    virtual bool ReadWaveArchiveInfoImpl(
        SoundArchive::ItemId warcId,
        SoundArchive::WaveArchiveInfo* info) const
    {
        (void)warcId; (void)info;
        return false;
    }

    virtual void ReadFileInfoImpl(
        SoundArchive::FileId id,
        SoundArchive::FileInfo* info,
        int index=0) const
    {
        (void)id; (void)info; (void)index;
    }

    virtual bool ReadSoundArchivePlayerInfoImpl(SoundArchive::SoundArchivePlayerInfo* info) const
    {
        (void)info;
        return false;
    }

    virtual bool ReadSound3DInfoImpl(
        SoundArchive::ItemId soundId,
        SoundArchive::Sound3DInfo* info) const
    {
        (void)soundId; (void)info;
        return false;
    }

    virtual bool ReadSequenceSoundInfoImpl(
        SoundArchive::ItemId soundId,
        SoundArchive::SequenceSoundInfo* info) const
    {
        (void)soundId; (void)info;
        return false;
    }

    virtual bool ReadStreamSoundInfoImpl(
        SoundArchive::ItemId soundId,
        SoundArchive::StreamSoundInfo* info) const
    {
        (void)soundId; (void)info;
        return false;
    }

    virtual bool ReadStreamSoundInfo2Impl(
        SoundArchive::ItemId soundId,
        SoundArchive::StreamSoundInfo2* info) const
    {
        (void)soundId; (void)info;
        return false;
    }

    virtual bool ReadWaveSoundInfoImpl(
        SoundArchive::ItemId soundId,
        SoundArchive::WaveSoundInfo* info) const
    {
        (void)soundId; (void)info;
        return false;
    }

    virtual bool ReadSoundUserParamImpl(
        SoundArchive::ItemId soundId,
        int index,
        u32& value) const
    {
        (void)soundId; (void)index; (void)value;
        return false;
    }

    virtual u32 GetSoundUserParamImpl(
        SoundArchive::ItemId soundId,
        u32 userParam) const
    {
        (void)soundId; (void)userParam;
        return userParam;
    }


private:
    bool m_IsEnabled;
    u8   m_Padding[3];
};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_SOUND_ARCHIVE_PARAMETERS_HOOK_H_ */

