﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SEQUENCE_TRACK_H_
#define NW_SND_SEQUENCE_TRACK_H_

#include <nw/snd/snd_Channel.h>
#include <nw/snd/snd_CurveLfo.h>
#include <nw/snd/snd_MoveValue.h>
#include <nw/snd/snd_Global.h>      // SeqMute

namespace nw {
namespace snd {
namespace internal {
namespace driver {

/* ========================================================================
        typename declaration
   ======================================================================== */

class SequenceSoundPlayer;

/* ========================================================================
        class definition
   ======================================================================== */

class SequenceTrack
{
    /* ------------------------------------------------------------------------
            type definition
       ------------------------------------------------------------------------ */
public:
    static const int CALL_STACK_DEPTH = 10;

    struct ParserTrackParam
    {
        const u8* baseAddr;
        const u8* currentAddr;

        bool cmpFlag;
        bool noteWaitFlag;
        bool tieFlag;
        bool monophonicFlag;

        struct CallStack
        {
            u8 loopFlag;
            u8 loopCount;
            u8 padding[2];
            const u8* address;
        };
        CallStack callStack[ CALL_STACK_DEPTH ];
        u8 callStackDepth;
        bool frontBypassFlag;
        bool muteFlag;
        bool silenceFlag;

        s32 wait;

        bool noteFinishWait;
        bool portaFlag;
        bool damperFlag;
        u8  bankIndex;      // 0-3 のシーケンス内バンクインデックスと見る

        int prgNo;

        f32 sweepPitch;

        MoveValue<u8,s16> volume;
        MoveValue<u8,s16> volume2;
        MoveValue<s8,s16> pan;
        MoveValue<s8,s16> surroundPan;
        MoveValue<s8,s16> pitchBend;

        CurveLfoParam lfoParam[Channel::MOD_COUNT];
        u8 lfoTarget[Channel::MOD_COUNT]; // enum Channel::LfoTarget;

        u8 velocityRange;
        u8 bendRange;
        s8 initPan;
        u8 padding;

        s8 transpose;
        u8 priority;
        u8 portaKey;
        u8 portaTime;

        u8 attack;
        u8 decay;
        u8 sustain;
        u8 release;

        s16 envHold;
        s8 biquadType;
        u8 mainSend;

        u8 fxSend[ AUX_BUS_NUM ];
        u8 padding2;

        f32 lpfFreq;
        f32 biquadValue;
        s32 outputLine;
    };

    enum ParseResult
    {
        PARSE_RESULT_CONTINUE,
        PARSE_RESULT_FINISH
    };

    /* ------------------------------------------------------------------------
            constant variable
       ------------------------------------------------------------------------ */
public:
    static const int DEFAULT_PRIORITY   = 64;
    static const int DEFAULT_BENDRANGE  = 2;
    static const int DEFAULT_PORTA_KEY  = 60; /* cn4 */
    static const int INVALID_ENVELOPE   = 0xff;
    static const int MAX_ENVELOPE_VALUE = 0x7f;
    static const int PARSER_PARAM_SIZE  = 32;
    static const int TRACK_VARIABLE_NUM = 16;

    static const int PAUSE_RELEASE_VALUE    = 127;
    static const int MUTE_RELEASE_VALUE     = 127;

    /* ------------------------------------------------------------------------
            static member
       ------------------------------------------------------------------------ */
public:
    static void ChannelCallbackFunc(
        Channel* dropChannel,
        Channel::ChannelCallbackStatus status,
        void* userData
    );

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */

public:
    // 初期化・終了処理
    SequenceTrack();
    virtual ~SequenceTrack();

    void InitParam();
    void SetSeqData( const void* seqBase, s32 seqOffset );
    void Open();
    void Close();
    bool IsOpened() const { return m_OpenFlag; }

    // 更新処理
    int ParseNextTick( bool doNoteOn );
    void UpdateChannelLength();
    void UpdateChannelParam();

    // チャンネル操作
    Channel* NoteOn(
        int key,
        int velocity,
        s32 length,
        bool tieFlag
    );
    void StopAllChannel();
    void ReleaseAllChannel( int release );
    void FreeAllChannel();
    void PauseAllChannel( bool flag );

    // 情報取得
    int GetChannelCount() const;
    const ParserTrackParam& GetParserTrackParam() const { return m_ParserTrackParam; }
    ParserTrackParam& GetParserTrackParam() { return m_ParserTrackParam; }

    // パラメータ設定
    void SetMute( SeqMute mute );
    void SetSilence( bool silenceFlag, int fadeTimes );
    void SetVolume( f32 volume ) { m_ExtVolume = volume; }
    void SetPitch( f32 pitch ) { m_ExtPitch = pitch; }
    void SetPanRange( f32 panRange ) { m_PanRange = panRange; }
    void SetLpfFreq( f32 lpfFreq ) { m_ParserTrackParam.lpfFreq = lpfFreq; }
    void SetBiquadFilter( int type, f32 value );
    void SetBankIndex( int bankIndex );
    void SetTranspose( s8 transpose );
    void SetVelocityRange( u8 range );
    void SetOutputLine( s32 outputLine );

    void SetTvVolume( f32 volume ) { m_TvParam.volume = volume; }
    void SetTvMixParameter( u32 srcChNo, s32 mixChNo, f32 param ) { m_TvParam.mixParameter[srcChNo].ch[mixChNo] = param; }
    void SetTvPan( f32 pan ) { m_TvParam.pan = pan; }
    void SetTvSurroundPan( f32 span ) { m_TvParam.span = span; }
    void SetTvMainSend( f32 send ) { m_TvParam.mainSend = send; }
    void SetTvFxSend( AuxBus bus, f32 send ) { m_TvParam.fxSend[bus] = send; }

    void SetDrcVolume( u32 drcIndex, f32 volume ) { m_DrcParam[drcIndex].volume = volume; }
    void SetDrcMixParameter( u32 drcIndex, u32 srcChNo, s32 mixChNo, f32 param ) { m_DrcParam[drcIndex].mixParameter[srcChNo].ch[mixChNo] = param; }
    void SetDrcPan( u32 drcIndex, f32 pan ) { m_DrcParam[drcIndex].pan = pan; }
    void SetDrcSurroundPan( u32 drcIndex, f32 span ) { m_DrcParam[drcIndex].span = span; }
    void SetDrcMainSend( u32 drcIndex, f32 send ) { m_DrcParam[drcIndex].mainSend = send; }
    void SetDrcFxSend( u32 drcIndex, AuxBus bus, f32 send ) { m_DrcParam[drcIndex].fxSend[bus] = send; }

    // パラメータ取得
    f32 GetVolume() const { return m_ExtVolume; }
    f32 GetPitch() const { return m_ExtPitch; }
    f32 GetPanRange() const { return m_PanRange; }
    f32 GetLpfFreq() const { return m_ParserTrackParam.lpfFreq; }
    int GetBiquadType() const { return m_ParserTrackParam.biquadType; }
    f32 GetBiquadValue() const { return m_ParserTrackParam.biquadValue; }

    // シーケンス処理
    s16 GetTrackVariable( int varNo ) const;
    void SetTrackVariable( int varNo, s16 var );
    vs16* GetVariablePtr( int varNo );

    void SetSequenceSoundPlayer( SequenceSoundPlayer* player ) { m_pSequenceSoundPlayer = player; }
    const SequenceSoundPlayer* GetSequenceSoundPlayer() const { return m_pSequenceSoundPlayer; }
    SequenceSoundPlayer* GetSequenceSoundPlayer() { return m_pSequenceSoundPlayer; }

    void SetPlayerTrackNo( int playerTrackNo );
    u8 GetPlayerTrackNo() const { return m_PlayerTrackNo; }

    void UpdateChannelRelease( Channel* channel );

protected:
    virtual ParseResult Parse( bool doNoteOn ) = 0;

private:
    Channel* GetLastChannel() const { return m_pChannelList; }
    void AddChannel( Channel* channel );

    u8 m_PlayerTrackNo;
    bool m_OpenFlag;
    f32 m_ExtVolume;
    f32 m_ExtPitch;
    f32 m_PanRange;

    OutputParam m_TvParam;
    OutputParam m_DrcParam[DRC_OUT_COUNT];

    ParserTrackParam m_ParserTrackParam;
    vs16 m_TrackVariable[ TRACK_VARIABLE_NUM ];

    SequenceSoundPlayer* m_pSequenceSoundPlayer;
    Channel* m_pChannelList;
};

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SEQUENCE_TRACK_H_ */

