﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_PROFILE_READER_H_
#define NW_SND_PROFILE_READER_H_

#include <nw/snd/snd_Global.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief  プロファイルデータを取得するためのクラスです。
//!
#if defined(NW_PLATFORM_CAFE)
//!         本クラスのオブジェクトを生成し、@ref SoundSystem::RegisterProfileReader 関数で登録
//!         すると、本クラスのオブジェクトのバッファにプロファイルデータが記録されるようになります。
//!         この記録に使用するバッファは各オブジェクトごとに確保されます。
//!
//!         @ref ProfileReader::Read 関数を使用することで、
//!         本クラスのオブジェクトのバッファの内容を読みとることができます。
//!
//!         ProfileReader の登録数が増えるほど処理負荷が増える点に注意してください。
//!
//!         詳しい使用方法は、Demo/snd/profile デモをご覧ください。
//!
//!         この API は Cafe プラットフォームでのみ利用可能です。
#endif
//!
//! @see SoundProfile 構造体
#if defined(NW_PLATFORM_CAFE)
//! @see SoundSystem::RegisterProfileReader
#endif
//!
//! @date 2015/02/23 Cafe プラットフォームでのみ利用可能な旨追記
//! @date 2012/02/09 初版
//---------------------------------------------------------------------------
class ProfileReader
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @date 2012/02/01 初版
    //---------------------------------------------------------------------------
    ProfileReader();

    //---------------------------------------------------------------------------
    //! @brief  プロファイルデータを取得します。
    //!
    //!         プロファイルデータは、1 オーディオフレーム (= 3ms) に
    //!         1 SoundProfile 消費します。
    //!
    //!         1/60 秒周期で本関数を呼び出す場合は、最低 6 個分
    //!         (1/60秒 = 16.67ms = オーディオフレーム換算で 5～6 フレームに相当)、
    //!         フレーム落ちなどを考慮しても 3 倍の 17 個分程度の SoundProfile
    //!         構造体を profile に渡せば十分です。
    //!
    //!         つまり、T ms 周期で本関数を呼び出す場合の SoundProfile
    //!         の最低必要数は下記の式で求められます。
    //!
    //! @code
    //!  (最低必要数) = (T / 3) + 1;
    //! @endcode
    //!
    //!         また、サウンドライブラリ内部では、32 個の SoundProfile を持っています。
    //!
    //! @param[out] profile   読み取ったプロファイルデータの格納先を指定します。
    //! @param[in] maxCount   オーディオフレーム単位で最大で何フレーム遡ったプロファイルデータを取得するのかを指定します。32 まで指定できます。
    //!
    //! @return 取得したプロファイルデータの個数を返します。
    //!
    //! @see SoundProfile 構造体
    //!
    //! @date 2012/03/08 profile に渡す SoundProfile 必要数について説明を追加
    //! @date 2012/02/01 初版
    //---------------------------------------------------------------------------
    int Read( SoundProfile* profile, int maxCount );

    //! @briefprivate
    //! @param src :private
    void Record( const SoundProfile& src );

public:
    //! @briefprivate
    nw::ut::LinkListNode m_Link;

private:
    SoundProfile m_ProfileBuffer[internal::MAX_PROFILE_COUNT];
    int m_ProfileBufferRead;
    int m_ProfileBufferWrite;
};

//! @briefprivate
typedef nw::ut::LinkList< ProfileReader, offsetof(ProfileReader, m_Link) > ProfileReaderList;

} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_PROFILE_READER_H_ */

