﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_INSTANCE_POOL_H_
#define NW_SND_INSTANCE_POOL_H_

#include <new>
#include <nw/ut/os/ut_CriticalSection.h>

#include <nw/config.h>
#if defined( NW_PLATFORM_CTR )
    #include <nn/types.h>
#elif defined( NW_PLATFORM_CAFE )
    #include <nw/types.h>
#endif

namespace nw {
namespace snd {
namespace internal {

/* ========================================================================
        PoolImpl class
   ======================================================================== */

class PoolImpl
{
public:
    PoolImpl() : m_pNext(NULL), m_pBuffer(NULL), m_BufferSize(0) {}
protected:
    u32 CreateImpl( void* buffer, size_t size, u32 objSize );
    void DestroyImpl();
    int CountImpl() const;
    void* AllocImpl();
    void FreeImpl( void* ptr );

private:
    PoolImpl* m_pNext;
    void* m_pBuffer;
    size_t m_BufferSize;
};

template< typename T >
class MemoryPool : private internal::PoolImpl
{
public:
    /*---------------------------------------------------------------------------*
      Name:         Create

      Description:  指定したメモリ領域をプールに割り当てます

      Arguments:    buffer - メモリ領域の開始アドレス
                    size   - メモリ領域のサイズ

      Returns:      作成したプール中のインスタンス数
     *---------------------------------------------------------------------------*/
    u32 Create( void* buffer, size_t size )
    {
        u32 objSize =
            ( sizeof( T ) > sizeof( MemoryPool<T>* ) ) ?
            sizeof( T ):
            sizeof( MemoryPool<T>* )
            ;
        return CreateImpl( buffer, size, objSize );
    }

    /*---------------------------------------------------------------------------*
      Name:         Destroy

      Description:  指定したメモリ領域をプールから解放します
                    解放するためにはメモリ領域が Free されてプールに格納されている
                    必要があります。

      Arguments:    buffer - メモリ領域の開始アドレス
                    size   - メモリ領域のサイズ

      Returns:      無し
     *---------------------------------------------------------------------------*/
    void Destroy() { DestroyImpl(); }

    /*---------------------------------------------------------------------------*
      Name:         Count

      Description:  プール中のインスタンス数を取得します。

      Arguments:    無し

      Returns:      インスタンス数
     *---------------------------------------------------------------------------*/
    int Count() const { return CountImpl(); }

    /*---------------------------------------------------------------------------*
      Name:         Alloc

      Description:  インスタンスを確保する。

      Arguments:    無し

      Returns:      確保したインスタンス、確保できなければNULL
     *---------------------------------------------------------------------------*/
    T* Alloc() { return static_cast<T*>( AllocImpl() ); }

    /*---------------------------------------------------------------------------*
      Name:         Free

      Description:  インスタンスをプールに解放する

      Arguments:    obj - インスタンス

      Returns:      無し
     *---------------------------------------------------------------------------*/
    void Free( T* obj ) { FreeImpl( obj ); }
};


/* ========================================================================
        InstancePool class
   ======================================================================== */

template< typename T >
class InstancePool : private internal::PoolImpl
{
public:
    /*---------------------------------------------------------------------------*
      Name:         Create

      Description:  指定したメモリ領域をプールに割り当てます

      Arguments:    buffer - メモリ領域の開始アドレス
                    size   - メモリ領域のサイズ

      Returns:      作成したプール中のインスタンス数
     *---------------------------------------------------------------------------*/
    u32 Create( void* buffer, unsigned long size )
    {
        u32 objSize =
            ( sizeof( T ) > sizeof( InstancePool<T>* ) ) ?
            sizeof( T ):
            sizeof( InstancePool<T>* )
            ;
        return CreateImpl( buffer, size, objSize );
    }

    /*---------------------------------------------------------------------------*
      Name:         Destroy

      Description:  指定したメモリ領域をプールから解放します
                    解放するためにはメモリ領域が Free されてプールに格納されている
                    必要があります。

      Arguments:    buffer - メモリ領域の開始アドレス
                    size   - メモリ領域のサイズ

      Returns:      無し
     *---------------------------------------------------------------------------*/
    void Destroy() { DestroyImpl(); }

    /*---------------------------------------------------------------------------*
      Name:         Count

      Description:  プール中のインスタンス数を取得します。

      Arguments:    無し

      Returns:      インスタンス数
     *---------------------------------------------------------------------------*/
    int Count() const { return CountImpl(); }

    /*---------------------------------------------------------------------------*
      Name:         Alloc

      Description:  インスタンスを確保する。

      Arguments:    無し

      Returns:      確保したインスタンス、確保できなければNULL
     *---------------------------------------------------------------------------*/
    T* Alloc()
    {
        void *ptr = AllocImpl();
        if ( ptr == NULL ) return NULL;
        return new( ptr ) T(); // コンストラクタ呼びだし
    }

    /*---------------------------------------------------------------------------*
      Name:         Free

      Description:  インスタンスをプールに解放する

      Arguments:    obj - インスタンス

      Returns:      無し
     *---------------------------------------------------------------------------*/
    void Free( T* obj )
    {
        if ( obj == NULL ) return;
        obj->~T(); // デストラクタ呼びだし
        FreeImpl( obj );
    }
};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_INSTANCE_POOL_H_ */

