﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_FxReverbStd.h
 *
 * @file snd_FxReverbStd.h
 */


#ifndef NW_SND_FX_REVERB_STD_H_
#define NW_SND_FX_REVERB_STD_H_

#include <nw/types.h>

#include <nw/snd/snd_FxParam.h>
#include <nw/snd/snd_FxBase.h>
#include <nw/snd/snd_AxfxImpl.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief  標準リバーブエフェクトです。
//!
//!         FxReverbStd は SDK の AXFXReverbStdExp エフェクトをラッピングしたクラスです。
//!
//! @see FxBase クラス
//!
//! @date 2011/08/11 説明を追加
//! @date 2011/07/04 初版
//---------------------------------------------------------------------------
class FxReverbStd : public FxBase
{
    /* ------------------------------------------------------------------------
            constant definition
       ------------------------------------------------------------------------ */
public:
    static const f32 PRE_DELAY_TIME_MAX_MIN; //!< ReverbStdParam::preDelayTimeMax の最小値 (0.0) です。
    static const f32 PRE_DELAY_TIME_MIN;     //!< ReverbStdParam::preDelayTime の最小値 (0.0) です。
    static const f32 FUSED_TIME_MIN;         //!< ReverbStdParam::fusedTime の最小値 (0.0) です。
    static const f32 COLORATION_MIN;         //!< ReverbStdParam::coloration の最小値 (0.0) です。
    static const f32 COLORATION_MAX;         //!< ReverbStdParam::coloration の最大値 (1.0) です。
    static const f32 DAMPING_MIN;            //!< ReverbStdParam::damping の最小値 (0.0) です。
    static const f32 DAMPING_MAX;            //!< ReverbStdParam::damping の最大値 (1.0) です。
    static const f32 EARLY_GAIN_MIN;         //!< ReverbStdParam::earlyGain の最小値 (0.0) です。
    static const f32 EARLY_GAIN_MAX;         //!< ReverbStdParam::earlyGain の最大値 (1.0) です。
    static const f32 FUSED_GAIN_MIN;         //!< ReverbStdParam::fusedGain の最小値 (0.0) です。
    static const f32 FUSED_GAIN_MAX;         //!< ReverbStdParam::fusedGain の最大値 (1.0) です。
    static const f32 OUT_GAIN_MIN;           //!< ReverbStdParam::outGain の最小値 (0.0) です。
    static const f32 OUT_GAIN_MAX;           //!< ReverbStdParam::outGain の最大値 (1.0) です。

    /* ------------------------------------------------------------------------
            typename definition
       ------------------------------------------------------------------------ */
public:
    //---------------------------------------------------------------------------
    //! @brief  標準リバーブエフェクトパラメータの構造体です。
    //!
    //! @see SetParam
    //! @see GetParam
    //!
    //! @date 2011/08/18 初版
    //---------------------------------------------------------------------------
    struct ReverbStdParam
    {
        //---------------------------------------------------------------------------
        //! @brief    反響を開始するまでの時間です。
        //!           値の範囲は 0.0～preDelayTimeMax (sec 単位) です。
        //!           初期値は 0.2 です。
        //---------------------------------------------------------------------------
        f32 preDelayTime;
        //---------------------------------------------------------------------------
        //! @brief    反響が減衰するまでの時間です。
        //!           値の範囲は 0.0～ (sec 単位) です。
        //!           初期値は 3.0 です。
        //---------------------------------------------------------------------------
        f32 fusedTime;
        //---------------------------------------------------------------------------
        //! @brief    アルゴリズムの全パスフィルタの係数を調節し、ルームの壁の音響特性を
        //!           シミュレートします。 値が小さくなるほど残響音の密度が粗くなります。
        //!           値の範囲は 0.0～1.0 です。
        //!           初期値は 0.6 です。
        //---------------------------------------------------------------------------
        f32 coloration;
        //---------------------------------------------------------------------------
        //! @brief    反響音の高周波減衰を調整します。 値を小さくすると低周波成分が主流
        //!           となり、大きくすると高周波成分が減衰せずに残るようになります。
        //!           値の範囲は 0.0～1.0 です。
        //!           初期値は 0.4 です。
        //---------------------------------------------------------------------------
        f32 damping;
        //---------------------------------------------------------------------------
        //! @brief    出力ゲインです。
        //!           値の範囲は 0.0～1.0 です。
        //!           初期値は 1.0 です。
        //---------------------------------------------------------------------------
        f32 outGain;

        //---------------------------------------------------------------------------
        //! @brief    初期反射音のモードです。
        //!           初期値は FX_REVERB_EARLY_REFLECTION_30MS です。
        //---------------------------------------------------------------------------
        FxReverbEarlyReflectionMode earlyMode;
        //---------------------------------------------------------------------------
        //! @brief    反響を開始するまでの時間に設定できる最大値です。
        //!           この値を大きくするとエフェクトが必要とするメモリサイズも増加します。
        //!           初期値は 0.02 です。
        //---------------------------------------------------------------------------
        f32 preDelayTimeMax;
        //---------------------------------------------------------------------------
        //! @brief    残響音のモードです。
        //!           初期値は FX_REVERB_FUSED_OLD_AXFX です。
        //---------------------------------------------------------------------------
        FxReverbFusedMode fusedMode;
        //---------------------------------------------------------------------------
        //! @brief    初期反射音の出力ゲインです。
        //!           値の範囲は 0.0～1.0 です。
        //!           初期値は 0.0 です。
        //---------------------------------------------------------------------------
        f32 earlyGain;
        //---------------------------------------------------------------------------
        //! @brief    残響音の出力ゲインです。
        //!           値の範囲は 0.0～1.0 です。
        //!           初期値は 1.0 です。
        //---------------------------------------------------------------------------
        f32 fusedGain;

        //---------------------------------------------------------------------------
        //! @brief    コンストラクタです。
        //! @date 2011/08/18 初版
        //---------------------------------------------------------------------------
        ReverbStdParam()
        : preDelayTime( 0.02f ),
          fusedTime( 3.0f ),
          coloration( 0.6f ),
          damping( 0.4f ),
          outGain( 1.0f ),
          earlyMode( FX_REVERB_EARLY_REFLECTION_30MS ),
          preDelayTimeMax( 0.02f ),
          fusedMode( FX_REVERB_FUSED_OLD_AXFX ),
          earlyGain( 0.0f ),
          fusedGain( 1.0f )
        {}
    };

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:
    //! @name コンストラクタ/デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //!         エフェクトパラメータは、下記のように初期化されます。
    //!
    //!         - preDelayTime = 0.02
    //!         - fusedTime = 3.0
    //!         - coloration = 0.6
    //!         - damping = 0.4
    //!         - outGain = 1.0
    //!         - earlyMode = FX_REVERB_EARLY_REFLECTION_30MS
    //!         - preDelayTimeMax = 0.02
    //!         - fusedMode = FX_REVERB_FUSED_OLD_AXFX
    //!         - earlyGain = 0.0
    //!         - fusedGain = 1.0
    //!
    //! @date 2011/08/11 初版
    //---------------------------------------------------------------------------
    FxReverbStd();

    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //!
    //! @date 2011/08/11 初版
    //---------------------------------------------------------------------------
    virtual ~FxReverbStd() { Finalize(); ReleaseWorkBuffer(); }

    //@}

    //! @name メモリ割り当て
    //@{

    //---------------------------------------------------------------------------
    //! @brief    FxReverbStd が必要とするメモリのサイズを取得します。
    //!
    //!           この関数を呼び出す前に @ref SetParam を呼びだして
    //!           エフェクトパラメータを設定してください。呼び出さない場合は、
    //!           初期設定の値が使用されます。
    //!
    //!           エフェクトを使用するためには、取得したメモリサイズのメモリを確保
    //!           した後に @ref AssignWorkBuffer を呼びだし、エフェクトにメモリ領域
    //!           を割り当ててください。
    //!
    //! @return   FxReverbStd が必要とするメモリのサイズを返します。
    //!
    //! @see  SetParam
    //! @see  AssignWorkBuffer
    //!
    //! @date 2011/08/11 初版
    //---------------------------------------------------------------------------
    u32 GetRequiredMemSize();

    //---------------------------------------------------------------------------
    //! @brief    エフェクトで使用するバッファ領域を割り当てます。
    //!
    //!           バッファ領域の割り当てはエフェクトを使用する前に行う必要があります。
    //!
    //!           エフェクトが必要とするバッファサイズはパラメータの値によって変化しますので、
    //!           @ref GetRequiredMemSize を呼び出してバッファサイズを取得してください。
    //!
    //! @param[in] buffer   割り当てるバッファの先頭アドレス。
    //! @param[in] size     割り当てるバッファのサイズ。
    //!
    //! @return   バッファ領域の割り当てに成功した場合は true を、失敗した場合は false を返します。
    //!
    //! @see  GetRequiredMemSize
    //!
    //! @date 2011/08/11 初版
    //---------------------------------------------------------------------------
    virtual bool AssignWorkBuffer( void* buffer, u32 size );

    //---------------------------------------------------------------------------
    //! @brief  エフェクトで使用していたメモリを解放します。
    //!
    //!         @ref AssignWorkBuffer で割り当てたバッファ領域を解放します。 この関数で
    //!         解放を行う前に、@ref Finalize を呼び出してエフェクト処理を終了させる
    //!         必要があります。
    //!
    //! @see  AssignWorkBuffer
    //! @see  Finalize
    //!
    //! @date 2011/08/11 初版
    //---------------------------------------------------------------------------
    virtual void ReleaseWorkBuffer();

    //@}

    //! @name エフェクト
    //@{

    //---------------------------------------------------------------------------
    //! @brief    エフェクトの開始処理を行います。
    //!
    //!           @ref SoundSystem クラスを通じてエフェクトを使用する場合、
    //!           @ref SoundSystem クラスからこの関数が呼び出されます。 ユーザがこの関数を
    //!           呼び出す必要はありません。
    //!
    //!           この関数を呼び出す前に、@ref AssignWorkBuffer を呼びだし、エフェクトで
    //!           使用するワークメモリを割り当てる必要があります。
    //!
    //! @return   エフェクトの開始に成功したら true を、失敗したら false を返します。
    //!
    //! @see  SoundSystem クラス
    //! @see  AssignWorkBuffer
    //!
    //! @date 2011/08/11 初版
    //---------------------------------------------------------------------------
    virtual bool Initialize();

    //---------------------------------------------------------------------------
    //! @brief    エフェクトの終了処理を行います。
    //!
    //!           @ref SoundSystem クラスを通じてエフェクトを使用する場合、
    //!           @ref SoundSystem クラスからこの関数が呼び出されます。 ユーザがこの関数を
    //!           呼び出す必要はありません。
    //!
    //! @see  SoundSystem クラス
    //!
    //! @date 2011/08/11 初版
    //---------------------------------------------------------------------------
    virtual void Finalize();

    //---------------------------------------------------------------------------
    //! @brief    エフェクトパラメータを設定します。
    //!
    //!           パラメータの設定は、エフェクト処理で使用することに加え、エフェクト処理に
    //!           必要なメモリサイズを計算することにも使用します。
    //!
    //!           エフェクトパラメータの設定はエフェクト動作中でも行うことができます。
    //!           しかし、エフェクト処理に必要なメモリが不足するようなパラメータを新たに
    //!           設定した場合、パラメータの設定は失敗します。
    //!           後からメモリの使用量が大きくなるようなパラメータ変更を行いたい場合は、
    //!           あらかじめ大きめのメモリサイズを設定しておく必要があります。
    //!
    //! @param[in] param     エフェクトパラメータ。
    //!
    //! @return   エフェクト動作中は、パラメータ変更に成功したら true を、失敗したら false
    //!           を返します。 エフェクト処理を開始する前は常に true を返します。
    //!
    //! @see  GetRequiredMemSize
    //! @see  GetParam
    //!
    //! @date 2011/08/11 初版
    //---------------------------------------------------------------------------
    bool SetParam( const ReverbStdParam& param );

    //---------------------------------------------------------------------------
    //! @brief    エフェクトのパラメータを取得します。
    //!
    //! @return   現在のエフェクトパラメータの値です。
    //!
    //! @see  SetParam
    //!
    //! @date 2011/08/11 初版
    //---------------------------------------------------------------------------
    const ReverbStdParam& GetParam() const;

    // エフェクトコールバック
    virtual void UpdateBuffer(
        int numChannels,
        void* buffer[],
        unsigned long bufferSize,
        SampleFormat format,
        f32 sampleRate,
        OutputMode mode
    );

    //@}

private:
#if defined( NW_PLATFORM_WIN32 )
    typedef nw::internal::winext::AXFX_REVERBSTD_EXP AXFX_REVERBSTD_EXP;
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
    typedef nw::internal::winext::AXFX_REVERBSTD_EXP AXFX_REVERBSTD_EXP;
#elif defined( NW_USE_NINTENDO_SDK )
    // TODO: nn_audio
    typedef nw::internal::winext::AXFX_REVERBSTD_EXP AXFX_REVERBSTD_EXP;
#endif

    bool m_IsActive;
    internal::AxfxImpl m_Impl;
    ReverbStdParam m_Param;
    AXFX_REVERBSTD_EXP m_AxfxParam;
};

} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_FX_REVERB_STD_H_ */
