﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_FxDelay.h
 *
 * @file snd_FxDelay.h
 */

#ifndef NW_SND_FX_DELAY_H_
#define NW_SND_FX_DELAY_H_

#include <nw/types.h>

#include <nw/snd/snd_FxParam.h>
#include <nw/snd/snd_FxBase.h>
#include <nw/snd/snd_AxfxImpl.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief  ディレイエフェクトです。
//!
//!         FxDelay は SDK の AXFXDelayExp エフェクトをラッピングしたクラスです。
//!
//! @see FxBase クラス
//!
//! @date 2011/08/18 説明を追加
//! @date 2011/07/04 初版
//---------------------------------------------------------------------------
class FxDelay : public FxBase
{
    /* ------------------------------------------------------------------------
            constant definition
       ------------------------------------------------------------------------ */
public:
    static const f32 MAX_DELAY_MIN; //!< DelayParam::maxDelay の最小値 (0.04) です。
    static const f32 DELAY_MIN;     //!< DelayParam::delay の最小値 (0.04) です。
    static const f32 FEEDBACK_MIN;  //!< DelayParam::feedback の最小値 (0.0) です。
    static const f32 FEEDBACK_MAX;  //!< DelayParam::feedback の最大値 (0.99) です。
    static const f32 LPF_MIN;       //!< DelayParam::lpf の最小値 (0.0) です。
    static const f32 LPF_MAX;       //!< DelayParam::lpf の最大値 (1.0) です。
    static const f32 OUT_GAIN_MIN;  //!< DelayParam::outGain の最小値 (0.0) です。
    static const f32 OUT_GAIN_MAX;  //!< DelayParam::outGain の最小値 (1.0) です。

    /* ------------------------------------------------------------------------
            typename definition
       ------------------------------------------------------------------------ */
public:

    //---------------------------------------------------------------------------
    //! @brief  ディレイエフェクトパラメータの構造体です。
    //!
    //! @see SetParam
    //! @see GetParam
    //!
    //! @date 2011/08/18 初版
    //---------------------------------------------------------------------------
    struct DelayParam
    {
        //---------------------------------------------------------------------------
        //! @brief    ディレイタイム（遅延時間）です。
        //!           値の範囲は 0.04～maxDelay (msec 単位) です。
        //!           初期値は 160.0 です。
        //---------------------------------------------------------------------------
        f32 delay;
        //---------------------------------------------------------------------------
        //! @brief    ディレイ音のフィードバックゲインです。
        //!           値の範囲は 0.0～0.99 です。0.0 でフィードバックがかかりません。
        //!           初期値は 0.4 です。
        //---------------------------------------------------------------------------
        f32 feedback;
        //---------------------------------------------------------------------------
        //! @brief    出力ゲインです。
        //!           値の範囲は 0.0～1.0 です。
        //!           初期値は 1.0 です。
        //---------------------------------------------------------------------------
        f32 outGain;
        //---------------------------------------------------------------------------
        //! @brief    ディレイタイムに設定できる最大値です。
        //!           この値を大きくするとエフェクトが必要とするメモリサイズも増加します。
        //!           値の範囲は 0.04～ です。
        //!           初期値は 160.0 です。
        //---------------------------------------------------------------------------
        f32 maxDelay;
        //---------------------------------------------------------------------------
        //! @brief    ディレイ音にかけるローパスフィルタの値です。
        //!           値の範囲は 0.0～1.0 です。
        //!           初期値は 1.0 です。
        //---------------------------------------------------------------------------
        f32 lpf;

        //---------------------------------------------------------------------------
        //! @brief    コンストラクタです。
        //! @date 2011/08/18 初版
        //---------------------------------------------------------------------------
        DelayParam()
        : delay( 160.0f ),
          feedback( 0.4f ),
          outGain( 1.0f ),
          maxDelay( 160.0f ),
          lpf( 1.0f )
        {}
    };

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:
    //! @name コンストラクタ/デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //!         エフェクトパラメータは、下記のように初期化されます。
    //!
    //!         - delay = 160.0
    //!         - feedback = 0.4
    //!         - outGain = 1.0
    //!         - maxDelay = 160.0
    //!         - lpf = 1.0
    //!
    //! @date 2011/08/18 初版
    //---------------------------------------------------------------------------
    FxDelay();

    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //!
    //! @date 2011/08/18 初版
    //---------------------------------------------------------------------------
    virtual ~FxDelay() { Finalize(); ReleaseWorkBuffer(); }

    //@}

    //! @name メモリ割り当て
    //@{

    //---------------------------------------------------------------------------
    //! @brief    FxDelay が必要とするメモリのサイズを取得します。
    //!
    //!           FxDelay が現在のパラメータ設定でエフェクト処理を行うために必要とする
    //!           メモリのサイズを取得します。
    //!
    //!           この関数を呼び出す前に @ref SetParam を呼びだしてエフェクトパラメータを
    //!           設定してください。呼び出さない場合は、初期設定の値が使用されます。
    //!
    //!           エフェクトを使用するためには、取得したメモリサイズのメモリを確保
    //!           した後に @ref AssignWorkBuffer を呼びだし、エフェクトにメモリ領域を
    //!           割り当ててください。
    //!
    //! @return   FxDelay が必要とするメモリのサイズを返します。
    //!
    //! @see  SetParam
    //! @see  AssignWorkBuffer
    //!
    //! @date 2011/08/18 初版
    //---------------------------------------------------------------------------
    u32 GetRequiredMemSize();

    //---------------------------------------------------------------------------
    //! @brief    エフェクトで使用するバッファ領域を割り当てます。
    //!
    //!           バッファ領域の割り当てはエフェクトを使用する前に行う必要があります。
    //!
    //!           エフェクトが必要とするバッファサイズはパラメータの値によって変化しますので、
    //!           @ref GetRequiredMemSize を呼び出してバッファサイズを取得してください。
    //!
    //! @param[in] buffer   割り当てるバッファの先頭アドレス。
    //! @param[in] size     割り当てるバッファのサイズ。
    //!
    //! @return   バッファ領域の割り当てに成功した場合は true を、失敗した場合は false を返します。
    //!
    //! @see  GetRequiredMemSize
    //!
    //! @date 2011/08/18 初版
    //---------------------------------------------------------------------------
    virtual bool AssignWorkBuffer( void* buffer, u32 size );

    //---------------------------------------------------------------------------
    //! @brief  エフェクトで使用していたメモリを解放します。
    //!
    //!         @ref AssignWorkBuffer で割り当てたバッファ領域を解放します。 この関数で
    //!         解放を行う前に、@ref Finalize を呼び出してエフェクト処理を終了させる
    //!         必要があります。
    //!
    //! @see  AssignWorkBuffer
    //! @see  Finalize
    //!
    //! @date 2011/08/18 初版
    //---------------------------------------------------------------------------
    virtual void ReleaseWorkBuffer();

    //@}

    //! @name エフェクト
    //@{

    //---------------------------------------------------------------------------
    //! @brief    エフェクトの開始処理を行います。
    //!
    //!           @ref SoundSystem クラスを通じてエフェクトを使用する場合、
    //!           @ref SoundSystem クラスからこの関数が呼び出されます。 ユーザがこの関数を
    //!           呼び出す必要はありません。
    //!
    //!           この関数を呼び出す前に、@ref AssignWorkBuffer を呼びだし、エフェクトで
    //!           使用するワークメモリを割り当てる必要があります。
    //!
    //! @return   エフェクトの開始に成功したら true を、失敗したら false を返します。
    //!
    //! @see  SoundSystem クラス
    //! @see  AssignWorkBuffer
    //!
    //! @date 2011/08/18 初版
    //---------------------------------------------------------------------------
    virtual bool Initialize();

    //---------------------------------------------------------------------------
    //! @brief    エフェクトの終了処理を行います。
    //!
    //!           @ref SoundSystem クラスを通じてエフェクトを使用する場合、
    //!           @ref SoundSystem クラスからこの関数が呼び出されます。 ユーザがこの関数を
    //!           呼び出す必要はありません。
    //!
    //! @see  SoundSystem クラス
    //!
    //! @date 2011/08/18 初版
    //---------------------------------------------------------------------------
    virtual void Finalize();

    //---------------------------------------------------------------------------
    //! @brief    エフェクトパラメータを設定します。
    //!
    //!           パラメータの設定は、エフェクト処理で使用することに加え、エフェクト処理に
    //!           必要なメモリサイズを計算することにも使用します。
    //!
    //!           エフェクトパラメータの設定はエフェクト動作中でも行うことができます。
    //!           しかし、エフェクト処理に必要なメモリが不足するようなパラメータを新たに
    //!           設定した場合、パラメータの設定は失敗します。
    //!           後からメモリの使用量が大きくなるようなパラメータ変更を行いたい場合は、
    //!           あらかじめ大きめのメモリサイズを設定しておく必要があります。
    //!
    //! @param[in] param     エフェクトパラメータ。
    //!
    //! @return   エフェクト動作中は、パラメータ変更に成功したら true を、失敗したら false
    //!           を返します。 エフェクト処理を開始する前は常に true を返します。
    //!
    //! @see  GetRequiredMemSize
    //! @see  GetParam
    //!
    //! @date 2011/08/18 初版
    //---------------------------------------------------------------------------
    bool SetParam( const DelayParam& param );

    //---------------------------------------------------------------------------
    //! @brief    エフェクトのパラメータを取得します。
    //!
    //! @return   現在のエフェクトパラメータの値です。
    //!
    //! @see  SetParam
    //!
    //! @date 2011/08/18 初版
    //---------------------------------------------------------------------------
    const DelayParam& GetParam() const;

    // エフェクトコールバック
    virtual void UpdateBuffer(
        int numChannels,
        void* buffer[],
        unsigned long bufferSize,
        SampleFormat format,
        f32 sampleRate,
        OutputMode mode
    );

    //@}

  private:
#if defined( NW_PLATFORM_WIN32 )
    typedef nw::internal::winext::AXFX_DELAY_EXP AXFX_DELAY_EXP;
#elif defined( NW_PLATFORM_ANDROID ) || defined( NW_PLATFORM_IOS )
    typedef nw::internal::winext::AXFX_DELAY_EXP AXFX_DELAY_EXP;
#elif defined( NW_USE_NINTENDO_SDK )
    // TODO: nn_audio
    typedef nw::internal::winext::AXFX_DELAY_EXP AXFX_DELAY_EXP;
#endif

    bool m_IsActive;
    internal::AxfxImpl m_Impl;
    DelayParam m_Param;
    AXFX_DELAY_EXP m_AxfxParam;
};

} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_FX_DELAY_H_ */
