﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_COMMAND_MANAGER_H_
#define NW_SND_COMMAND_MANAGER_H_

#if NW_USE_NINTENDO_SDK
#include <atomic>
#endif

#include <nw/snd/snd_Command.h>
#include <nw/snd/snd_Config.h>
#include <nw/ut.h>

#if defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
#include <atomic>
#endif

namespace nw {
namespace snd {
namespace internal {

class CommandBuffer
{
public:
    CommandBuffer();
    ~CommandBuffer();

    void Initialize( void* commandBuffer, u32 commandBufferSize );
    void Finalize();

    void* AllocMemory( u32 size );
    void FreeMemory( Command* lastCommand );

    u32 GetAllocatableCommandSize() const;

private:
    u32* m_CommandMemoryArea;
    u32 m_CommandMemoryAreaSize;

    u32 m_CommandMemoryAreaBegin;
    u32 m_CommandMemoryAreaEnd;

    // m_CommandMemoryAreaBegin == m_CommandMemoryAreaBegin の時、
    // エリアサイズがゼロなのか、最大なのかを判定するフラグ
    bool m_CommandMemoryAreaZeroFlag;
};

class CommandManager
{
public:
    // コマンド処理の実装関数
    typedef void (*ProcessCommandListFunc)( Command* commandList );

    // コマンド処理をリクエストする関数
    typedef void (*RequestProcessCommandFunc)();

    CommandManager();
    ~CommandManager();

    bool IsAvailable() const { return m_Available; }

    void Initialize( void* commandBuffer, u32 commandBufferSize, ProcessCommandListFunc func );
    void Finalize();

    void SetRequestProcessCommandFunc( RequestProcessCommandFunc func ) { m_pRequestProcessCommandFunc = func; }

    template< typename CommandType >
    CommandType* AllocCommand()
    {
        return reinterpret_cast<CommandType*>( AllocMemory( sizeof(CommandType) ) );
    }

    u32 PushCommand( Command* command );

    u32 FlushCommand( bool forceFlag );
    void RecvCommandReply();

    void WaitCommandReply( u32 tag );
    bool IsFinishCommand( u32 tag ) const;

    u32 GetAllocatableCommandSize() const { return m_CommandBuffer.GetAllocatableCommandSize(); }
    u32 GetCommandListCount() const
    {
#ifdef NW_PLATFORM_CAFE
        return m_CommandListCount.u.u32;
#else
        return static_cast<const u32>(m_CommandListCount);
#endif
    }

    bool ProcessCommand();

private:
    static const int SEND_COMMAND_QUEUE_SIZE = 32;
    static const int RECV_COMMAND_QUEUE_SIZE = SEND_COMMAND_QUEUE_SIZE + 1;

    static const u32 INVALID_COMMAND = 0xffffffff;

    void* AllocMemory( u32 size );
    void* TryAllocMemory( u32 size );
    void FinalizeCommandList( Command* command );

    bool m_Available;
    ProcessCommandListFunc m_pProcessCommandListFunc;
    RequestProcessCommandFunc m_pRequestProcessCommandFunc;

    ut::MessageQueue   m_SendCommandQueue;
    ut::MessageQueue::BufferType m_SendCommandQueueBuffer[ SEND_COMMAND_QUEUE_SIZE ];
    ut::MessageQueue   m_RecvCommandQueue;
    ut::MessageQueue::BufferType m_RecvCommandQueueBuffer[ RECV_COMMAND_QUEUE_SIZE ];

    Command* m_CommandListBegin;
    Command* m_CommandListEnd;

#ifdef NW_PLATFORM_CAFE
    OSAtomicVar m_CommandListCount;
#elif defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
    std::atomic_long m_CommandListCount;
#elif NW_USE_NINTENDO_SDK
    std::atomic<int> m_CommandListCount;
#else
    long m_CommandListCount;
#endif

    u32 m_CommandTag;
    u32 m_FinishCommandTag;

    CommandBuffer m_CommandBuffer;
};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_COMMAND_MANAGER_H_ */

