﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_BIQUAD_FILTER_CALLBACK_H_
#define NW_SND_BIQUAD_FILTER_CALLBACK_H_

#include <nw/types.h>
#include <nw/snd/snd_Voice.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief  biquad フィルタを表す抽象クラスです。
//!
//!         BiquadFilterCallback クラスから継承し、
//!         ユーザー側で biquad フィルタを実装することができます。
//!
//!         BiquadFilterCallback の使用方法については、
//!         @ref SoundSystem::SetBiquadFilterCallback 関数を参照してください。
//!
//! @see SoundSystem::SetBiquadFilterCallback
//!
//! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class BiquadFilterCallback
{
public:
    //---------------------------------------------------------------------------
    //! @brief  biquad フィルタの計算に使用する係数の構造体です。
    //!
    //! @see GetCoefficients
    //!
    //! @date 2011/07/04 初版
    //---------------------------------------------------------------------------
    typedef BiquadFilterCoefficients Coefficients;

    //---------------------------------------------------------------------------
    //! @brief  デストラクタです。
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    virtual ~BiquadFilterCallback() {}

    //---------------------------------------------------------------------------
    //! @brief  biquad フィルタの係数を返します。
    //!
    //!         nw::snd ライブラリ内部の biquad フィルタ係数が必要なタイミングで呼び出されます。
    //!         BiquadFilterCallback を継承したクラスで、
    //!         この関数にフィルタ係数を実装する必要があります。
    //!
    //!         type は、コールバッククラスを @ref SoundSystem::SetBiquadFilterCallback
    //!         で登録したときのタイプ番号です。
    //!         @ref BiquadFilterType に記述されている BIQUAD_FILTER_TYPE_USER_MIN ～
    //!         BIQUAD_FILTER_TYPE_USER_MAX までの値が入ります。
    //!
    //!         value はフィルタのかかり具合を表す 0.0f ～ 1.0f の値です。
    //!         この値に応じたフィルタ係数を coef に格納するように実装します。
    //!
    //! @param[in] type     フィルタの種類です
    //!                     (BIQUAD_FILTER_TYPE_USER_MIN ～BIQUAD_FILTER_TYPE_USER_MAX)。
    //! @param[in] value    フィルタのかかり具合です (0.0f ～ 1.0f)
    //! @param[out] coef    計算したフィルタ係数を格納するための構造体です。
    //!
    //! @see SoundSystem::SetBiquadFilterCallback
    //! @see BiquadFilterType
    //!
    //! @date 2011/07/04 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    virtual void GetCoefficients( int type, f32 value, Coefficients* coef ) const = 0;
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_BIQUAD_FILTER_CALLBACK_H_ */

