﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_FND_FILE_STREAM_IMPL_H_
#define NW_SND_FND_FILE_STREAM_IMPL_H_

#include <limits.h>
#include <nw/ut/ut_Inlines.h>
#include <nw/snd/fnd/basis/sndfnd_Config.h>
#include <nw/snd/fnd/io/sndfnd_StreamCache.h>

#include "nw/snd/fnd/io/sndfnd_FileStream.h"

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
//! @brief  ファイルストリームです。
//---------------------------------------------------------------------------
class FileStreamImpl : public FileStream
{
    NW_DISALLOW_COPY_AND_ASSIGN(FileStreamImpl);

private: // クラスの定義
    class DirectStream : public Stream
    {
    public:
        DirectStream() : m_Owner(NULL) {}

        void Initialize(FileStreamImpl& owner)
        {
            m_Owner = &owner;
        }

        virtual void Close() {}
        virtual bool IsOpened() const { return m_Owner->IsOpened(); }
        virtual bool CanRead() const { return m_Owner->CanRead(); }
        virtual bool CanWrite() const { return m_Owner->CanWrite(); }
        virtual bool CanSeek() const { return m_Owner->CanSeek(); }
        virtual u32 GetSize() const { return m_Owner->GetSize(); }
        virtual u32 Read(void* buf, u32 length, FndResult* result = NULL)
        {
            return m_Owner->ReadDirect(buf, length, result);
        }
        virtual u32 Write(const void* buf, u32 length, FndResult* result = NULL)
        {
            return m_Owner->WriteDirect(buf, length, result);
        }
        virtual FndResult Seek(s32 offset, SeekOrigin origin)
        {
            return m_Owner->SeekDirect(offset, origin);
        }
        virtual u32 GetCurrentPosition() const { return m_Owner->m_CurrentPosition; }

    private:
        FileStreamImpl* m_Owner;
    };

public: // コンストラクタ
    FileStreamImpl();

#if defined(NW_PLATFORM_CAFE)
    explicit FileStreamImpl(FSClient* fsClient, FSCmdBlock* fsCmdBlock, FSFileHandle fsFileHandle);
#elif defined(NW_PLATFORM_WIN32) || defined(NW_PLATFORM_ANDROID)
    explicit FileStreamImpl(void* fileHandle);
#elif defined(NW_USE_NINTENDO_SDK)
    explicit FileStreamImpl(void* fileHandle);
#endif

    virtual ~FileStreamImpl() { }

public: // メソッド
#if defined(NW_PLATFORM_CAFE)
    virtual void Open(FSClient* fsClient, FSCmdBlock* fsCmdBlock, FSFileHandle fsFileHandle);
#elif defined(NW_PLATFORM_CTR)
    virtual nn::Result Open(const char* filePath, bit32 openMode);
#elif defined(NW_PLATFORM_WIN32) || defined(NW_PLATFORM_ANDROID)
    virtual void Open(void* fileHandle);
#elif defined(NW_USE_NINTENDO_SDK)
    virtual nn::Result Open(const char* filePath, nn::fs::OpenMode openMode);
#endif

    virtual void Close();

    virtual bool IsOpened() const;

    virtual bool CanRead() const;

    virtual bool CanWrite() const;

    virtual bool CanSeek() const;

    virtual u32 GetSize() const;

    virtual u32 Read(void* buf, u32 length, FndResult* result = NULL);

    virtual u32 Write(const void* buf, u32 length, FndResult* result = NULL);

    virtual FndResult Seek(s32 offset, SeekOrigin origin);

    virtual u32 GetCurrentPosition() const
    {
        return IsCacheEnabled() ? m_StreamCache.GetCurrentPosition() : m_CurrentPosition;
    }

    virtual void EnableCache(void* buffer, u32 length);

    virtual void DisableCache();

    virtual bool IsCacheEnabled() const
    {
        return m_StreamCache.IsInitialized();
    }

private: // メソッド
    u32 ReadDirect(void* buf, u32 length, FndResult* result = NULL);

    u32 WriteDirect(const void* buf, u32 length, FndResult* result = NULL);

    FndResult SeekDirect(s32 offset, SeekOrigin origin);

    void ValidateAlignment(const void* buf) const;

private: // メンバ変数
#if defined(NW_PLATFORM_CAFE)
    FSClient*    m_FsClient;
    FSCmdBlock*  m_FsCmdBlock;
    FSFileHandle m_FsFileHandle;
#elif defined(NW_PLATFORM_CTR)
    nn::fs::FileStream m_FileStream;
    bool               m_IsOpened;
    u8                 m_Padding1;
    u16                m_Padding2;
#elif defined(NW_PLATFORM_WIN32) || defined(NW_PLATFORM_ANDROID)
    void* m_FileHandle;
#elif defined(NW_USE_NINTENDO_SDK)
    nn::fs::FileHandle m_FileHandle;
    bool               m_IsOpened;
    u8                 m_Padding1;
    u16                m_Padding2;
#endif

    mutable u32 m_FileSize;
    u32         m_CurrentPosition;

    StreamCache  m_StreamCache;
    DirectStream m_DirectStream;    //!< StreamCache からの直接入出力するためのアダプタ
};

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_FND_FILE_STREAM_IMPL_H_
