﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_FND_AIFF_BINARY_H_
#define NW_SND_FND_AIFF_BINARY_H_

#include <float.h>
#include <math.h>
#include <nw/snd/fnd/binary/sndfnd_PrimitiveTypes.h>
#include <nw/snd/fnd/binary/sndfnd_Binary.h>
#include <nw/snd/fnd/binary/sndfnd_IffBinary.h>

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

#pragma pack(push, 1)

//---------------------------------------------------------------------------
//! @brief  IEEE Standard 754 floating point number（ビッグエンディアン版）
//---------------------------------------------------------------------------
struct IEEE754BE
{
    IEEE754BE() :
        head(0),
        hiMant(0),
        loMant(0)
    { }

    operator f64() const
    {
        f64 result;
        int expon;

        expon = head & 0x7fff;

        if (expon == 0 && hiMant == 0 && loMant == 0)
        {
            result = 0;
        }
        else
        {
            if (expon == 0x7FFF)
            {    /* Infinity or NaN */
                result = DBL_MAX;
            }
            else
            {
                expon -= 0x3fff;

                result = hiMant * pow(2.0, (expon -= 31));
                result += loMant * pow(2.0, (expon -= 32));
            }
        }

        return (head & 0x8000) != 0 ? -result : result;
    }

    PcBeBinU16 head;
    PcBeBinU32 hiMant;
    PcBeBinU32 loMant;
};

//---------------------------------------------------------------------------
//! @brief  Aiff COMM チャンク
//---------------------------------------------------------------------------
struct AiffCommChunk
{
    static const u32 ValidID = NW_SND_BINARY_MAKE_SIGNATURE('C', 'O', 'M', 'M');

    AiffCommChunk() :
        header(ValidID),
        numChannels(0),
        numSampleFrames(0),
        sampleSize(0)
    { }

    static u32 GetValidBodySize()
    {
        return sizeof(AiffCommChunk) - sizeof(ChunkHeaderBE);
    }

    bool IsValid()
    {
        return header.id == ValidID;
    }

    ChunkHeaderBE header;
    PcBeBinU16    numChannels;
    PcBeBinU32    numSampleFrames;
    PcBeBinU16    sampleSize;
    IEEE754BE     sampleRate;
};

//---------------------------------------------------------------------------
//! @brief  Aiff SSND チャンク
//---------------------------------------------------------------------------
struct AiffSoundDataChunk
{
    static const u32 ValidID = NW_SND_BINARY_MAKE_SIGNATURE('S', 'S', 'N', 'D');

    AiffSoundDataChunk() :
        header(ValidID),
        offset(0),
        blockSize(0)
    { }

    static u32 GetValidBodySize()
    {
        return sizeof(AiffSoundDataChunk) - sizeof(ChunkHeaderBE);
    }

    bool IsValid()
    {
        return header.id == ValidID;
    }

    ChunkHeaderBE header;
    PcBeBinU32    offset;
    PcBeBinU32    blockSize;
};

//---------------------------------------------------------------------------
//! @brief  Aiff FORM チャンクヘッダ
//---------------------------------------------------------------------------
struct AiffFormChunkHeader
{
    static const u32 ValidID = NW_SND_BINARY_MAKE_SIGNATURE('F', 'O', 'R', 'M');
    static const u32 ValidFormatType = NW_SND_BINARY_MAKE_SIGNATURE('A', 'I', 'F', 'F');

    AiffFormChunkHeader() :
        header(ValidID)
    { }

    bool IsValid()
    {
        return header.id == ValidID && formType == ValidFormatType;
    }

    ChunkHeaderBE header;
    u32           formType;
};

#pragma pack(pop)

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_FND_AIFF_BINARY_H_
