﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_FND_TIME_H_
#define NW_SND_FND_TIME_H_

#include <nw/snd/fnd/basis/sndfnd_Config.h>

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
//! @brief  時間間隔を管理するクラスです。
//---------------------------------------------------------------------------
class TimeSpan
{
public:
    //---------------------------------------------------------------------------
#if defined(NW_PLATFORM_CAFE)
    //---------------------------------------------------------------------------
    typedef OSTime TickType;
    //---------------------------------------------------------------------------
#elif defined(NW_PLATFORM_CTR)
    //---------------------------------------------------------------------------
    typedef s64 TickType;
    //---------------------------------------------------------------------------
#elif defined(NW_PLATFORM_WIN32)
    //---------------------------------------------------------------------------
    typedef u32 TickType;
    //---------------------------------------------------------------------------
#elif defined(NW_PLATFORM_ANDROID)
    //---------------------------------------------------------------------------
    typedef u32 TickType;
#elif defined(NW_USE_NINTENDO_SDK)
    //---------------------------------------------------------------------------
    typedef s64 TickType;
    //---------------------------------------------------------------------------
#endif
    //---------------------------------------------------------------------------

public:
    TimeSpan() : m_TickSpan(0) { }
    explicit TimeSpan(TickType tickSpan) : m_TickSpan(tickSpan) { }

public:
    static TimeSpan FromNanoSeconds(s64 value);
    static TimeSpan FromMicroSeconds(s64 value);
    static TimeSpan FromMilliSeconds(s64 value);

    TickType GetTick() const { return m_TickSpan; }

    s64 ToNanoSeconds() const;
    s64 ToMicroSeconds() const;
    s64 ToMilliSeconds() const;

    TimeSpan& operator =(const TimeSpan& value) { m_TickSpan = value.m_TickSpan; return *this; }
    TimeSpan& operator =(TickType value) { m_TickSpan = value; return *this; }

    operator TickType() const { return m_TickSpan; }

    TimeSpan operator +(const TimeSpan& right) const { return TimeSpan(m_TickSpan + right.m_TickSpan); }
    TimeSpan operator -(const TimeSpan& right) const { return TimeSpan(m_TickSpan - right.m_TickSpan); }
    TimeSpan operator *(const TimeSpan& right) const { return TimeSpan(m_TickSpan * right.m_TickSpan); }
    TimeSpan operator /(const TimeSpan& right) const { return TimeSpan(m_TickSpan / right.m_TickSpan); }

    TimeSpan& operator +=(const TimeSpan& rhs) { m_TickSpan += rhs.m_TickSpan; return *this; }
    TimeSpan& operator -=(const TimeSpan& rhs) { m_TickSpan -= rhs.m_TickSpan; return *this; }
    TimeSpan& operator *=(const TimeSpan& rhs) { m_TickSpan *= rhs.m_TickSpan; return *this; }
    TimeSpan& operator /=(const TimeSpan& rhs) { m_TickSpan /= rhs.m_TickSpan; return *this; }

    bool operator ==(const TimeSpan& rhs) const { return (rhs.m_TickSpan == m_TickSpan); }
    bool operator !=(const TimeSpan& rhs) const { return !(*this == rhs); }

    bool operator < (const TimeSpan& right) const { return m_TickSpan < right.m_TickSpan; }
    bool operator > (const TimeSpan& right) const { return m_TickSpan > right.m_TickSpan; }
    bool operator <=(const TimeSpan& rhs)   const { return m_TickSpan <= rhs.m_TickSpan; }
    bool operator >=(const TimeSpan& rhs)   const { return m_TickSpan >= rhs.m_TickSpan; }

private:
    TickType m_TickSpan;
};

//---------------------------------------------------------------------------
//! @brief  時間を管理するクラスです。
//---------------------------------------------------------------------------
class Time
{
private:
    Time() { }

public:
    static TimeSpan Current();
};

//---------------------------------------------------------------------------
//! @brief  経過時間を計測するストップウォッチクラスです。
//---------------------------------------------------------------------------
class StopWatch
{
public:
    StopWatch() : m_IsStarted(false) { }

public:
    void Start()
    {
        if(!m_IsStarted)
        {
            m_IsStarted = true;
            Reset();
        }
    }

    void Stop()
    {
        if(m_IsStarted)
        {
            m_IsStarted = false;
            m_StopTime = Time::Current();
        }
    }

    bool IsStarted() const
    {
        return m_IsStarted;
    }

    TimeSpan GetElapsedTime() const
    {
        return m_IsStarted ? Time::Current() - m_StartTime : m_StopTime - m_StartTime;
    }

    void Reset()
    {
        m_StartTime = Time::Current();
        m_StopTime  = m_StartTime;
    }

private:
    TimeSpan m_StartTime;
    TimeSpan m_StopTime;
    bool     m_IsStarted;
    u8       m_Padding1;
    u16      m_Padding2;
    u32      m_Padding3;
};

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_FND_TIME_SPAN_H_
